/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) ISO14443-3A Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7230 $
 * $Date: 2025-03-14 10:27:23 +0530 (Fri, 14 Mar 2025) $
 */

#include <ph_Status.h>
#include <phhalHw.h>
#include <phpalI14443p3a.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHPAL_I14443P3A_SAM_X

#include "phpalI14443p3a_Sam_X.h"
#include "../../../phhalHw/src/Sam/Commands/16_ISO14443_3/phhalHw_Sam_Cmd_ISO14443_3.h"

/** ReqA Command code */
#define I14443P3A_SAM_REQUEST_CMD       0x26

/** WupA Command code */
#define I14443P3A_SAM_WAKEUP_CMD        0x52

phStatus_t phpalI14443p3a_Sam_X_Init(phpalI14443p3a_Sam_X_DataParams_t * pDataParams, uint16_t wSizeOfDataParams,
    phhalHw_Sam_DataParams_t * pHalDataParams)
{
    if(sizeof(phpalI14443p3a_Sam_X_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_ISO14443P3A);
    }
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_PAL_ISO14443P3A);
    PH_ASSERT_NULL_PARAM(pHalDataParams, PH_COMP_PAL_ISO14443P3A);

    /* init private data */
    pDataParams->wId = PH_COMP_PAL_ISO14443P3A | PHPAL_I14443P3A_SAM_X_ID;
    pDataParams->pHalDataParams = pHalDataParams;
    pDataParams->bUidLength = 0;
    pDataParams->bUidComplete = 0;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P3A);
}

phStatus_t phpalI14443p3a_Sam_X_RequestA(phpalI14443p3a_Sam_X_DataParams_t * pDataParams, uint8_t * pAtqa)
{
    phStatus_t PH_MEMLOC_REM wStatus;

    /* Set RxDeafBits */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SetConfig(
        pDataParams->pHalDataParams,
        PHHAL_HW_CONFIG_RXDEAFBITS,
        8));

    /* Perform RequestA_Wakeup command */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_X_ISO14443_3_RequestA_Wakeup(
        pDataParams->pHalDataParams,
        I14443P3A_SAM_REQUEST_CMD,
        pAtqa));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P3A);
}

phStatus_t phpalI14443p3a_Sam_X_WakeUpA(phpalI14443p3a_Sam_X_DataParams_t * pDataParams, uint8_t * pAtqa)
{
    phStatus_t PH_MEMLOC_REM wStatus;

    /* Set RxDeafBits */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SetConfig(
        pDataParams->pHalDataParams,
        PHHAL_HW_CONFIG_RXDEAFBITS,
        9));

    /* Perform RequestA_Wakeup command */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_X_ISO14443_3_RequestA_Wakeup(
        pDataParams->pHalDataParams,
        I14443P3A_SAM_WAKEUP_CMD,
        pAtqa));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P3A);
}

phStatus_t phpalI14443p3a_Sam_X_HaltA(phpalI14443p3a_Sam_X_DataParams_t * pDataParams)
{
    phStatus_t PH_MEMLOC_REM wStatus;

    /* Preform HaltA command */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_X_ISO14443_3_HaltA(pDataParams->pHalDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P3A);
}

phStatus_t phpalI14443p3a_Sam_X_Anticollision(phpalI14443p3a_Sam_X_DataParams_t * pDataParams, uint8_t bCascadeLevel,
    uint8_t * pUidIn, uint8_t bNvbUidIn, uint8_t * pUidOut, uint8_t * pNvbUidOut)
{
    /* satisfy compiler */
    if(pDataParams || bCascadeLevel || pUidIn || bNvbUidIn || pUidOut || pNvbUidOut);

    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_PAL_ISO14443P3A);
}

phStatus_t phpalI14443p3a_Sam_X_Select(phpalI14443p3a_Sam_X_DataParams_t * pDataParams, uint8_t bCascadeLevel,
    uint8_t * pUidIn, uint8_t * pSak)
{
    /* satisfy compiler */
    if(pDataParams || bCascadeLevel || pUidIn || pSak);

    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_PAL_ISO14443P3A);
}

phStatus_t phpalI14443p3a_Sam_X_ActivateCard(phpalI14443p3a_Sam_X_DataParams_t * pDataParams, uint8_t * pUidIn,
    uint8_t bLenUidIn, uint8_t * pUidOut, uint8_t * pLenUidOut, uint8_t * pSak, uint8_t * pMoreCardsAvailable)
{
    phStatus_t  PH_MEMLOC_REM wStatus;
    uint8_t *   PH_MEMLOC_REM pRxBuffer;
    uint16_t    PH_MEMLOC_REM wRxLength;

    /* Parameter check */
    if((bLenUidIn != 0U) &&
        (bLenUidIn != 4U) &&
        (bLenUidIn != 7U) &&
        (bLenUidIn != 10U))
    {
        /* Given UID length is invalid, return error */
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P3A);
    }

    /* Reset UID complete flag */
    pDataParams->bUidComplete = 0;

    /* No possibility to check that */
    *pMoreCardsAvailable = 0;

    if(bLenUidIn != 0)
    {
        /* Call activate Wakeup command */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_X_ISO14443_3_ActivateWakeUp(
            pDataParams->pHalDataParams,
            pUidIn,
            bLenUidIn));

        /* Store UID */
        pDataParams->bUidLength = bLenUidIn;
        memcpy(pDataParams->abUid, pUidIn, pDataParams->bUidLength);

        /* Return UID */
        *pLenUidOut = pDataParams->bUidLength;
        memcpy(pUidOut, pDataParams->abUid, pDataParams->bUidLength);
    }
    else
    {
        /* Preform Activate Card command */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_X_ISO14443_3_ActivateIdle(
            pDataParams->pHalDataParams,
            PHHAL_HW_SAM_CMD_ISO14443_3_ACTIVATEIDLE_DEFAULT,
            1,
            0,
            NULL,
            NULL,
            &pRxBuffer,
            &wRxLength));

        /* Retrieve SAK byte */
        *pSak = pRxBuffer[2];

        /* Retrieve UID */
        pDataParams->bUidLength = pRxBuffer[3];
        memcpy(pDataParams->abUid, &pRxBuffer[4], pDataParams->bUidLength);

        /* Return UID */
        *pLenUidOut = pDataParams->bUidLength;
        memcpy(pUidOut, pDataParams->abUid, pDataParams->bUidLength);

        /* set default card timeout */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SetConfig(
            pDataParams->pHalDataParams,
            PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS,
            PHPAL_I14443P3A_TIMEOUT_DEFAULT_MS));

        /* enable TxCrc */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SetConfig(
            pDataParams->pHalDataParams,
            PHHAL_HW_CONFIG_TXCRC,
            PH_ON));

        /* enable RxCrc */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SetConfig(
            pDataParams->pHalDataParams,
            PHHAL_HW_CONFIG_RXCRC,
            PH_ON));
    }

    /* Set UID complete flag */
    pDataParams->bUidComplete = 1;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P3A);
}

phStatus_t phpalI14443p3a_Sam_X_Exchange(phpalI14443p3a_Sam_X_DataParams_t * pDataParams, uint16_t wOption,
    uint8_t * pTxBuffer, uint16_t wTxLength, uint8_t ** ppResponse, uint16_t * pRespLen)
{
    /* Preform Exchange command */
    return phhalHw_Exchange(
        pDataParams->pHalDataParams,
        wOption,
        pTxBuffer,
        wTxLength,
        ppResponse,
        pRespLen);
}

phStatus_t phpalI14443p3a_Sam_X_GetSerialNo(phpalI14443p3a_Sam_X_DataParams_t * pDataParams, uint8_t * pResponse,
    uint8_t * pRespLen)
{
    /* Check if UID is valid */
    if(!pDataParams->bUidComplete)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_PAL_ISO14443P3A);
    }

    memcpy(pResponse, pDataParams->abUid, pDataParams->bUidLength);
    *pRespLen = pDataParams->bUidLength;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P3A);
}

#endif /* NXPBUILD__PHPAL_I14443P3A_SAM_X */
