/*
 * Copyright 2022 - 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/*
 * Transparent specific HAL-Component of Reader Library Framework.
 * $RCSfile $
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifdef _WIN32

#include <ph_Status.h>
#include <phbalReg.h>
#include <phhalHwContact.h>
#include <ph_RefDefs.h>
#include <assert.h>
#include <phTools.h>
#include <phToolsAtrParser.h>

#ifdef NXPBUILD__PHHAL_HW_CONTACT_TRANSPARENT

#include "phhalHwContact_Transparent.h"
#include "phhalHwContact_Transparent_Int.h"

phStatus_t phhalHwContact_Transparent_Init(
    phhalHwContact_Transparent_DataParams_t * pDataParams,
    uint16_t wSizeOfDataParams,
    void    * pBalDataParams,
    uint8_t * pTxBuffer,
    uint16_t  wTxBufSize,
    uint8_t * pRxBuffer,
    uint16_t  wRxBufSize
    )
{
    if (sizeof(phhalHwContact_Transparent_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_HAL);
    }

    if (wRxBufSize == 0) /* wTxBufSize == 0 is allowed */
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pBalDataParams);
    PH_ASSERT_NULL (pTxBuffer);
    PH_ASSERT_NULL (pRxBuffer);

    /* Init private data */
    pDataParams->wId                = PH_COMP_HAL | PHHAL_HW_CONTACT_TRANSPARENT_ID;
    pDataParams->pBalDataParams     = pBalDataParams;
    pDataParams->pTxBuffer          = pTxBuffer;
    pDataParams->wTxBufSize         = wTxBufSize;
    pDataParams->wTxBufLen          = 0;
    pDataParams->pRxBuffer          = pRxBuffer;
    pDataParams->wRxBufSize         = wRxBufSize;
    pDataParams->wRxBufLen          = 0;
    pDataParams->dwATRLength        = 0;
    pDataParams->bProtocolType      = PHHAL_HW_CONTACT_PROTOCOLTYPE_T0;
    pDataParams->wRxBufStartPos = 0;
    pDataParams->dwNumExpectedBytes = 0;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Transparent_SetConfig32(
    phhalHwContact_Transparent_DataParams_t * pDataParams,
    uint16_t  wConfig,
    uint32_t  dwValue
    )
{
    phStatus_t status = PH_ERR_SUCCESS;
    uint16_t tmpValue = 0;
    uint64_t val = 0;

    switch (wConfig)
    {
    case PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE:
        if ((dwValue & PHHAL_HW_CONTACT_MODE_MASK_TX) != PHHAL_HW_CONTACT_MODE_DISABLED &&
            (dwValue & PHHAL_HW_CONTACT_MODE_MASK_TX) != PHHAL_HW_CONTACT_MODE_TX_LRC &&
            (dwValue & PHHAL_HW_CONTACT_MODE_MASK_TX) != PHHAL_HW_CONTACT_MODE_TX_CRC)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        if ((dwValue & PHHAL_HW_CONTACT_MODE_MASK_RX) != PHHAL_HW_CONTACT_MODE_DISABLED &&
            (dwValue & PHHAL_HW_CONTACT_MODE_MASK_RX) != PHHAL_HW_CONTACT_MODE_RX_LRC &&
            (dwValue & PHHAL_HW_CONTACT_MODE_MASK_RX) != PHHAL_HW_CONTACT_MODE_RX_CRC)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }

        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = dwValue;
        break;

    case PHHAL_HW_CONTACT_CONFIG_PROTOCOLTYPE:
    case PHHAL_HW_CONTACT_CONFIG_MAX_CLK_ATR:
        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = dwValue;
        break;

    case PHHAL_HW_CONTACT_CONFIG_FD:
        /* Input binary coding as described in ISO */
        status = phhalHwContact_Transparent_Int_SetBitrateFromFD(pDataParams, (uint8_t) dwValue);
        break;

    case PHHAL_HW_CONTACT_CONFIG_PARITY_ERROR_SIGNAL:
        status = phhalHwContact_Transparent_Int_SetParityErrorSignal(pDataParams, dwValue);
        if(status == PH_ERR_SUCCESS)
        {
            pDataParams->wCfgShadow[wConfig] = dwValue;
        }
        break;

    case PHHAL_HW_CONTACT_CONFIG_BWT_CLK:
        val = dwValue;
        tmpValue = (uint16_t)(val * 1000 / PHHAL_HW_CONTACT_TRANSPARENT_DEFAULT_CLK);
        status = phbalReg_SetConfig(pDataParams->pBalDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, tmpValue);
        if(status == PH_ERR_SUCCESS)
        {
            pDataParams->wCfgShadow[wConfig] = dwValue;
        }
        break;

    case PHHAL_HW_CONTACT_CONFIG_WT_CLK:
        val = dwValue;
        tmpValue = (uint16_t)(val * 1000 / PHHAL_HW_CONTACT_TRANSPARENT_DEFAULT_CLK);
        status = phbalReg_SetConfig(pDataParams->pBalDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, tmpValue);
        status = phbalReg_SetConfig(pDataParams->pBalDataParams, PHBAL_REG_SERIALWIN_CONFIG_TIMEOUT_MULTIPLIER_MS, tmpValue);
        if(status == PH_ERR_SUCCESS)
        {
            pDataParams->wCfgShadow[wConfig] = dwValue;
        }
        break;

    case PHHAL_HW_CONTACT_CONFIG_CWT_CLK:
        val = dwValue;
        tmpValue = (uint16_t)(val * 1000 / PHHAL_HW_CONTACT_TRANSPARENT_DEFAULT_CLK);
        status = phbalReg_SetConfig(pDataParams->pBalDataParams, PHBAL_REG_SERIALWIN_CONFIG_TIMEOUT_MULTIPLIER_MS, tmpValue);
        if(status == PH_ERR_SUCCESS)
        {
            pDataParams->wCfgShadow[wConfig] = dwValue;
        }
        break;

    case PHHAL_HW_CONTACT_CONFIG_NUM_EXPECTED_BYTES:
        pDataParams->dwNumExpectedBytes = dwValue;
        break;

    case PHHAL_HW_CONTACT_CONFIG_RXBUFFER_STARTPOS:
        /* Boundary check */
        if (dwValue > 0xFFFF || dwValue >= pDataParams->wRxBufSize)
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }
        /* Set start position */
        pDataParams->wRxBufStartPos = (uint16_t)dwValue;
        pDataParams->wRxBufLen = (uint16_t)dwValue;
        break;

    /* Timing Parameters VCC and CLK are not supported for this reader */
    case PHHAL_HW_CONTACT_CONFIG_BGT_CLK:
    case PHHAL_HW_CONTACT_CONFIG_CGT_CLK:
    case PHHAL_HW_CONTACT_CONFIG_GT_CLK:
    case PHHAL_HW_CONTACT_CONFIG_VCC:
    case PHHAL_HW_CONTACT_CONFIG_CLOCK_FREQUENCY_HZ:
    case PHHAL_HW_CONTACT_CONFIG_EGT_CLK:
    case PHHAL_HW_CONTACT_CONFIG_RGT_CLK:
    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    }

    return PH_ADD_COMPCODE(status, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Transparent_GetConfig32(
    phhalHwContact_Transparent_DataParams_t * pDataParams,
    uint16_t   wConfig,
    uint32_t * pValue
    )
{
    phStatus_t status = PH_ERR_SUCCESS;

    switch (wConfig)
    {
    case PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE:
    case PHHAL_HW_CONTACT_CONFIG_PROTOCOLTYPE:
    case PHHAL_HW_CONTACT_CONFIG_PARITY_ERROR_SIGNAL:
    case PHHAL_HW_CONTACT_CONFIG_BWT_CLK:
    case PHHAL_HW_CONTACT_CONFIG_WT_CLK:
    case PHHAL_HW_CONTACT_CONFIG_MAX_CLK_ATR:
    case PHHAL_HW_CONTACT_CONFIG_CWT_CLK:
        /* Read config from shadow */
        *pValue = pDataParams->wCfgShadow[wConfig];
        break;

    case PHHAL_HW_CONTACT_CONFIG_FD:
        /* Input binary coding as described in ISO */
        *pValue = 0;
        status = phhalHwContact_Transparent_Int_GetBitrateToFD(pDataParams, (uint8_t*)pValue);
        break;

    case PHHAL_HW_CONTACT_CONFIG_NUM_EXPECTED_BYTES:
        *pValue = pDataParams->dwNumExpectedBytes;
        break;

    case PHHAL_HW_CONTACT_CONFIG_RXBUFFER_STARTPOS:
        *pValue = (uint32_t)pDataParams->wRxBufStartPos;
        break;

    case PHHAL_HW_CONTACT_CONFIG_RXBUFFER_BUFSIZE:
        *pValue = pDataParams->wRxBufSize - PHHAL_HW_CONTACT_MAX_CRC_BUFFER_LEN;
        break;

    case PHHAL_HW_CONTACT_CONFIG_COMMUNICATION_CHANNEL:
        *pValue = PHHAL_HW_CONTACT_COMMUNICATIONCHANNEL_STANDARD;
        break;

    case PHHAL_HW_CONTACT_CONFIG_CGT_CLK:
    case PHHAL_HW_CONTACT_CONFIG_GT_CLK:
    case PHHAL_HW_CONTACT_CONFIG_BGT_CLK:
    case PHHAL_HW_CONTACT_CONFIG_VCC:
    case PHHAL_HW_CONTACT_CONFIG_CLOCK_FREQUENCY_HZ:
    case PHHAL_HW_CONTACT_CONFIG_EGT_CLK:
    case PHHAL_HW_CONTACT_CONFIG_RGT_CLK:
    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    }

    return PH_ADD_COMPCODE(status, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Transparent_Wait(
    phhalHwContact_Transparent_DataParams_t * pDataParams,
    uint8_t   bUnit,
    uint16_t  wTimeout
    )
{
    if (bUnit == PHHAL_HW_CONTACT_TIME_MICROSECONDS)
    {
        phTools_Sleep((wTimeout + 999) / 1000);
    }
    else if (bUnit == PHHAL_HW_CONTACT_TIME_MILLISECONDS)
    {
        phTools_Sleep(wTimeout);
    }

    /* just to satisfy compiler */
    if(pDataParams);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Transparent_Exchange(
    phhalHwContact_Transparent_DataParams_t * pDataParams,
    uint16_t    wOption,
    uint8_t   * pTxBuffer,
    uint16_t    wTxLength,
    uint8_t  ** ppRxBuffer,
    uint16_t  * pRxLength
    )
{
    phStatus_t PH_MEMLOC_REM status;
    uint8_t    PH_MEMLOC_REM pCrc[2];
    uint8_t    PH_MEMLOC_REM pLrc[1];
    uint8_t    PH_MEMLOC_REM bLrc;
    uint16_t   PH_MEMLOC_REM wCrc;
    uint16_t   PH_MEMLOC_REM wCrcLength;
    uint8_t *  PH_MEMLOC_REM pCrcData;

    /* Check if caller has provided valid RxBuffer */
    if (ppRxBuffer == NULL)
    {
        ppRxBuffer = &pDataParams->pRxBuffer;
    }
    if (pRxLength == NULL)
    {
        pRxLength = &pDataParams->wRxBufLen;
    }

    /* Reset received length */
    *pRxLength = 0;

    /* copy data */
    memcpy(&pDataParams->pTxBuffer[pDataParams->wTxBufLen], pTxBuffer, wTxLength); /* PRQA S 3200 */
    pDataParams->wTxBufLen = pDataParams->wTxBufLen + wTxLength;

    /* do not perform real exchange, just fill the global TxBuffer */
    if (wOption & PH_EXCHANGE_BUFFERED_BIT)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    }

    if ((pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE] & PHHAL_HW_CONTACT_MODE_MASK_TX) == PHHAL_HW_CONTACT_MODE_TX_CRC)
    {
        PH_CHECK_SUCCESS_FCT(status, phTools_CalculateCrc16(
            PH_TOOLS_CRC_OPTION_OUPUT_INVERTED,
            PH_TOOLS_CRC16_PRESET_CONTACTBASED,
            PH_TOOLS_CRC16_POLY_CONTACTBASED,
            pTxBuffer,
            wTxLength,
            &wCrc));

        if ((pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE] & PHHAL_HW_CONTACT_MODE_TX_CRC_SWAPPED) == PHHAL_HW_CONTACT_MODE_TX_CRC_SWAPPED)
        {
            pCrc[0] = (uint8_t)(wCrc >> 8);
            pCrc[1] = (uint8_t)(wCrc);
        }
        else
        {
            pCrc[0] = (uint8_t)(wCrc);
            pCrc[1] = (uint8_t)(wCrc >> 8);
        }

        memcpy(&pDataParams->pTxBuffer[pDataParams->wTxBufLen], pCrc, 2); /* PRQA S 3200 */
        pDataParams->wTxBufLen += 2;
    }
    else if ((pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE] & PHHAL_HW_CONTACT_MODE_MASK_TX) == PHHAL_HW_CONTACT_MODE_TX_LRC)
    {
        PH_CHECK_SUCCESS_FCT(status, phTools_CalculateLrc(pTxBuffer, wTxLength, pLrc));

        memcpy(&pDataParams->pTxBuffer[pDataParams->wTxBufLen], pLrc, 1); /* PRQA S 3200 */
        pDataParams->wTxBufLen += 1;
    }

    if(pDataParams->dwNumExpectedBytes != 0)
    {
        /* Do the exchange with BAL Layer */
        status = phbalReg_Exchange(
            pDataParams->pBalDataParams,
            PH_EXCHANGE_DEFAULT,
            pDataParams->pTxBuffer,
            pDataParams->wTxBufLen,
            (uint16_t)pDataParams->dwNumExpectedBytes,
            pDataParams->pRxBuffer,
            &pDataParams->wRxBufLen);

        /* BAL handles this case as buffer overflow because RxBuffer is full and their are still bytes remaining */
        if((status & PH_ERR_MASK) == PH_ERR_BUFFER_OVERFLOW)
        {
            status = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_I7816P4);
            pDataParams->wRxBufLen = (uint16_t)pDataParams->dwNumExpectedBytes;
        }
    }
    else
    {
        /* Do the exchange with BAL Layer */
        status = phbalReg_Exchange(
            pDataParams->pBalDataParams,
            PH_EXCHANGE_DEFAULT,
            pDataParams->pTxBuffer,
            pDataParams->wTxBufLen,
            pDataParams->wRxBufSize,
            pDataParams->pRxBuffer,
            &pDataParams->wRxBufLen);
    }

    /* Clear TxBuffer Data */
    pDataParams->wTxBufLen = 0;

    /* get data for CRC check */
    if ((pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE] & PHHAL_HW_CONTACT_MODE_FULL_RX_BUFFER) == PHHAL_HW_CONTACT_MODE_FULL_RX_BUFFER)
    {
        wCrcLength = pDataParams->wRxBufLen;
        pCrcData = pDataParams->pRxBuffer;
    }
    else
    {
        wCrcLength = pDataParams->wRxBufLen - pDataParams->wRxBufStartPos;
        pCrcData = &pDataParams->pRxBuffer[pDataParams->wRxBufStartPos];
    }

    /* Check and remove CRC in software if required */
    if ((pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE] & PHHAL_HW_CONTACT_MODE_MASK_RX) == PHHAL_HW_CONTACT_MODE_RX_CRC)
    {
        if (wCrcLength < 2)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTEGRITY_ERROR, PH_COMP_HAL);
        }

        PH_CHECK_SUCCESS_FCT(status, phTools_CalculateCrc16(
            PH_TOOLS_CRC_OPTION_OUPUT_INVERTED,
            PH_TOOLS_CRC16_PRESET_CONTACTBASED,
            PH_TOOLS_CRC16_POLY_CONTACTBASED,
            pCrcData,
            wCrcLength - 2,
            &wCrc));

        if ((pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE] & PHHAL_HW_CONTACT_MODE_RX_CRC_SWAPPED) == PHHAL_HW_CONTACT_MODE_RX_CRC_SWAPPED)
        {
            pCrc[0] = (uint8_t)(wCrc >> 8);
            pCrc[1] = (uint8_t)(wCrc);
        }
        else
        {
            pCrc[0] = (uint8_t)(wCrc);
            pCrc[1] = (uint8_t)(wCrc >> 8);
        }

        /* CRC over all data must be zero */
        if (pCrc[0] != pCrcData[wCrcLength-2] || pCrc[1] != pCrcData[wCrcLength-1])
        {
            return PH_ADD_COMPCODE(PH_ERR_INTEGRITY_ERROR, PH_COMP_HAL);
        }

        pDataParams->wRxBufLen -= 2;
    }
    else if ((pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE] & PHHAL_HW_CONTACT_MODE_MASK_RX) == PHHAL_HW_CONTACT_MODE_RX_LRC)
    {
        if (wCrcLength < 1)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTEGRITY_ERROR, PH_COMP_HAL);
        }

        PH_CHECK_SUCCESS_FCT(status, phTools_CalculateLrc(
            pCrcData,
            wCrcLength,
            &bLrc));

        /* LRC over all data must be zero */
        if (bLrc != 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTEGRITY_ERROR, PH_COMP_HAL);
        }

        pDataParams->wRxBufLen -= 1;
    }

    *ppRxBuffer = pDataParams->pRxBuffer;
    *pRxLength = pDataParams->wRxBufLen;

    return PH_ADD_COMPCODE(status, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Transparent_ApplyProtocolSettings(
    phhalHwContact_Transparent_DataParams_t * pDataParams,
    uint8_t * pAtr,
    uint16_t dwAtrLength,
    uint8_t bProtocolType
    )
{
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint32_t    PH_MEMLOC_REM wConfig = 0;
    uint8_t     PH_MEMLOC_REM bConfig = 0;
    uint8_t     PH_MEMLOC_REM bSupported;
    uint32_t    PH_MEMLOC_REM wFDIndexValue;

    /* if ATR available check if card supports specified ProtocolType */
    if(dwAtrLength != 0 && pAtr != NULL)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_SetInternalAtr(pAtr, dwAtrLength));
        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_GetGlobal_CheckIfProtocolSupported(bProtocolType, &bSupported));

        if(!bSupported)
        {
            /* not supported so reset to default values */
            PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_SetDefaultValues());
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
    }

    /* Store new protocol type */
    if (bProtocolType != PHHAL_HW_CONTACT_PROTOCOLTYPE_CURRENT)
    {
        pDataParams->bProtocolType = bProtocolType;
    }

    /* Commented out lines are those settings which are not supported by the reader but should actually be changed */
    /* (future implementaion maybe) */

    /* Apply global Parameters */
    /*PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_GetGlobal_Convention(&bConfig));*/
    /*PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_CONVENTION, (uint32_t)bConfig));*/
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_GetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_FD, &wFDIndexValue));

    /* Apply protocol dependent Parameters */
    switch(bProtocolType)
    {
    case PHHAL_HW_CONTACT_PROTOCOLTYPE_T0:
        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_GetT0_WTclk(&wConfig, wFDIndexValue));
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_WT_CLK, wConfig));

        /*PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_GetT0_GTclk(&wConfig, wFDIndexValue));
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_GT_CLK, wConfig));*/

        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE, PHHAL_HW_CONTACT_MODE_DISABLED));

        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalDataParams, PHBAL_REG_SERIALWIN_CONFIG_STOPBITS, PHBAL_REG_SERIALWIN_VALUE_STOPBITS_TWO));
        break;

    case PHHAL_HW_CONTACT_PROTOCOLTYPE_T1:
        /*PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_GetT1_CGTclk(&wConfig, wFDIndexValue));
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_CGT_CLK, wConfig));

        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_GetT1_CWTclk(&wConfig, wFDIndexValue));
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_CWT_CLK, wConfig));*/

        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_GetT1_BWTclk(&wConfig, wFDIndexValue));
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_BWT_CLK, wConfig));

        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_GetT1_RedundancyMode(&bConfig));
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_REDUNDANCY_MODE, (uint32_t)bConfig));

        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalDataParams, PHBAL_REG_SERIALWIN_CONFIG_STOPBITS, PHBAL_REG_SERIALWIN_VALUE_STOPBITS_TWO));

        pDataParams->dwNumExpectedBytes = PHHAL_HW_CONTACT_MAX_T1_BLOCK_SIZE;
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    }

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_PROTOCOLTYPE, bProtocolType));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Transparent_Activate(
                        phhalHwContact_Transparent_DataParams_t * pDataParams,
                        uint8_t * pATRBuffer,
                        uint16_t * pwATRLength
                        )
{
    phStatus_t statusTmp;
    phStatus_t status;
    uint16_t timeOut = 2000;
    uint16_t wAtrTimeout;
    uint8_t bColdReset;

    if(pDataParams->dwATRLength == 0)
    {
        bColdReset = 1;
    }
    else
    {
        bColdReset = 0;
    }

    /* Reset ATR before doing anything else */
    memset(pDataParams->pATR, 0x00, PHHAL_HW_CONTACT_MAX_ATR_SIZE);
    pDataParams->dwATRLength = 0;

    /* Before activation bring reader to default state */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_FD, PHHAL_HW_CONTACT_FD_DEFAULT));
    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_SetDefaultValues());
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_ApplyProtocolSettings(pDataParams, 0, 0, PHHAL_HW_CONTACT_PROTOCOLTYPE_T0));

    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_GetConfig(pDataParams->pBalDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, &timeOut));

    wAtrTimeout = (uint16_t)(pDataParams->wCfgShadow[PHHAL_HW_CONTACT_CONFIG_MAX_CLK_ATR] * 1000 / PHHAL_HW_CONTACT_TRANSPARENT_DEFAULT_CLK);
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, wAtrTimeout));

    if(bColdReset)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalDataParams, PHBAL_REG_SERIALWIN_CONFIG_DTR, PHBAL_REG_SERIALWIN_VALUE_DTR_HIGH));
        PH_CHECK_SUCCESS_FCT(statusTmp, phTools_Sleep(100));
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalDataParams, PHBAL_REG_SERIALWIN_CONFIG_RTS, PHBAL_REG_SERIALWIN_VALUE_RTS_LOW));
        PH_CHECK_SUCCESS_FCT(statusTmp, phTools_Sleep(100));
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalDataParams, PHBAL_REG_SERIALWIN_CONFIG_RTS, PHBAL_REG_SERIALWIN_VALUE_RTS_HIGH));
    }
    else
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalDataParams, PHBAL_REG_SERIALWIN_CONFIG_RTS, PHBAL_REG_SERIALWIN_VALUE_RTS_HIGH));
        PH_CHECK_SUCCESS_FCT(statusTmp, phTools_Sleep(100));
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalDataParams, PHBAL_REG_SERIALWIN_CONFIG_DTR, PHBAL_REG_SERIALWIN_VALUE_DTR_LOW));
        PH_CHECK_SUCCESS_FCT(statusTmp, phTools_Sleep(100));
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalDataParams, PHBAL_REG_SERIALWIN_CONFIG_DTR, PHBAL_REG_SERIALWIN_VALUE_DTR_HIGH));
    }

    PH_CHECK_SUCCESS_FCT(statusTmp, phTools_Sleep(100));
    status = phbalReg_Exchange(pDataParams->pBalDataParams,PH_EXCHANGE_DEFAULT, NULL, 0, PHHAL_HW_CONTACT_MAX_ATR_SIZE, pDataParams->pATR, (uint16_t*)&pDataParams->dwATRLength);

    if((status & PH_ERR_MASK) == PH_ERR_IO_TIMEOUT)
    {
        return PH_ADD_COMPCODE(PH_ERR_IO_TIMEOUT, PH_COMP_HAL);
    }

    if(pDataParams->dwATRLength > PHHAL_HW_CONTACT_MAX_ATR_SIZE)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    /* Copy ATR and set correct length */
    memcpy(pATRBuffer, pDataParams->pATR, pDataParams->dwATRLength);
    *pwATRLength = (uint16_t)pDataParams->dwATRLength;

    /* Set ATR in phToolsAtrParser to parse certain values later */
    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_SetInternalAtr(pDataParams->pATR, pDataParams->dwATRLength));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_ApplyProtocolSettings(pDataParams, 0, 0, PHHAL_HW_CONTACT_PROTOCOLTYPE_T0));

    PH_CHECK_SUCCESS_FCT(statusTmp, phTools_Sleep(100));
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, timeOut));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Transparent_Deactivate(
                        phhalHwContact_Transparent_DataParams_t * pDataParams
                        )
{
    phStatus_t statusTmp;
    uint16_t timeOut = 2000;

    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_GetConfig(pDataParams->pBalDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, &timeOut));
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, 10));

    /* Deactivate */
    PH_CHECK_SUCCESS_FCT(statusTmp, phTools_Sleep(100));
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalDataParams, PHBAL_REG_SERIALWIN_CONFIG_DTR, PHBAL_REG_SERIALWIN_VALUE_DTR_LOW));
    PH_CHECK_SUCCESS_FCT(statusTmp, phTools_Sleep(100));
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalDataParams, PHBAL_REG_SERIALWIN_CONFIG_RTS, PHBAL_REG_SERIALWIN_VALUE_RTS_LOW));

    pDataParams->dwATRLength = 0;
    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_SetDefaultValues());

    PH_CHECK_SUCCESS_FCT(statusTmp, phTools_Sleep(100));
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, timeOut));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Transparent_ClockStop(
                        phhalHwContact_Transparent_DataParams_t * pDataParams,
                        uint16_t wTgClockCount,
                        uint16_t wThClockCount,
                        uint32_t dwPinState
                        )
{
    if( pDataParams || wTgClockCount || wThClockCount || dwPinState);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Transparent_InitContactReader(
                                                        phhalHwContact_Transparent_DataParams_t * pDataParams
                                                        )
{
    phStatus_t statusTmp;

    /* Set internal default values for communication */
    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_SetDefaultValues());

    /* Set default Datarate */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_FD, PHHAL_HW_CONTACT_FD_DEFAULT));

    /* Set default Max Clk Atr timing */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_MAX_CLK_ATR, PHHAL_HW_CONTACT_MAX_CLK_ATR_DEFAULT));

    /* ApplyProtocolSettings to have default values applies */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_ApplyProtocolSettings(pDataParams, 0, 0, PHHAL_HW_CONTACT_PROTOCOLTYPE_T0));

    /* Deactivate set power and ATR length to zero */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Deactivate(pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

#endif /* NXPBUILD__PHHAL_HW_CONTACT_TRANSPARENT */

#endif /* _WIN32 */
