/**********************************************************************
			TXF
                        TeXt File interface
 ***********************************************************************
 Copyright (c) SMARTWARE, 1999-2006 All Rights Reserved.
 ***********************************************************************
 Name of a memory file : "@nnnnnnnn" : first byte addr = 0xnnnnnnnn
 ***********************************************************************
 01/06/2006 - Marc LEGOUX - V6r22d: Add txf__iter_p
 06/04/2006 - Marc LEGOUX - V6r22b: Add txf_write_line()
 22/09/2004 - Marc LEGOUX - V6r16 : Add txf_iter2()
 08/01/2001 - Marc LEGOUX - V6r06 : Add _MemModeWToAlloc
 29/03/1999 - Marc LEGOUX - V6r03 : Add files in memory
 18/09/1995 - Marc LEGOUX Extract from ctf (own module)
 ***********************************************************************/
#ifndef txf_h
#define txf_h

#include <config.h>
#include <err.h>

#if !config_MLOS32
#include <stdio.h>
#endif

/*-------------------------- ERRORS ------------------------------------*/
#ifndef txf__err
extern
err_Type txf__err;			/* Error code offset */
#endif
#define txf__ok		0		/* no error */
#define txf__eof	(txf__err+2)	/* ok but end of file */
#define txf__toobig	(txf__err+3)	/* too big string*/
#define txf__ropen	(txf__err+4)	/* On open to read */
#define txf__wopen	(txf__err+5)	/* On open to write */
#define txf__read	(txf__err+6)	/* On reading */
#define txf__write	(txf__err+7)	/* On writing */
#define txf__seek	(txf__err+8)	/* On seeking */
#define txf__tell	(txf__err+9)	/* On telling */
#define txf__wrong	(txf__err+10)	// Not valid iter
#define txf__busy	(txf__err+11)	// File is taken
#define txf__iter_p	(txf__err+12)	// Text pointer not valid for iterator

/*-------------------------- TYPES -------------------------------------*/
/* Iterator */
typedef
  struct {
    long offs;				/* Offset in file */
    unsigned linenum;			/* line num */
    long lineoffs;			/* line offset */
  } txf_cur_type;

/* Memory mode : Default is ram */
#define txf_MemModeFlash		1	/* Read in flash */
#define txf_MemModeWAlloc		2	/* Write is allocated */
#define txf_MemModeWToAlloc		4	/* Do Alloc to Write */

typedef
  struct txf_iter_type {
    char * line;			/* Last read line buffer or 0 */
    unsigned linemax;			/* Last read line max length */
    unsigned linenum;			/* Last read line num 1..N */
    long lineoffs;			/* last read line offset or -1*/
    long woffs;				/* current next-to-write offset */
    char * fpath;			/* path of read file */
    char * wfpath;			/* path of write file */
    
    unsigned char mode;			/* txf_iter_mode_... */
    unsigned verbose;
    
    #if config_MLOS32
    long rfoffs;			/* next-to-read offset in file */
    long wfoffs;			/* next-to-write offset in file */
    char * raddr;			/* Read area address */
    char * waddr;			/* Write area address */
    unsigned rmaxl;			/* Read area length */
    unsigned wmaxl;			/* Write area length */
    unsigned MemMode;			/* Memory mode */
    void * rarea;			/* Read mem area (with header) */
    void * warea;			/* Write mem area (with header) */
    
    #else
    FILE *file;				/* file to read */
    FILE *wfile;			/* file to write */
    #endif
    
    struct txf_iter_type * fiter;	// 0 for mother iter
    void * FileToken;			// To protect the file
    
    void * RFU[8];
  } txf_iter_type;

#define txf_iter_mode_LineMask		0x03
#define txf_iter_mode_wholeline		0x00
#define txf_iter_mode_partofline	0x01

#define txf_iter_mode_NoLF		0x04	// LF not allowed inside lines
/*-------------------------- FUNCTIONS ---------------------------------*/
/*-----------------------------------------------------------------------*
                        Start/End
 Usefull only to control verbose and to get version
 *-----------------------------------------------------------------------*/
err_Type txf_Start(
        void * Cfg
        );
err_Type txf_End(
        void * Cfg
        );
/*-------------------------------------------------------------------------*
                                FILE ITERATOR
 *-------------------------------------------------------------------------*/
/*-------------------------------------------------------------------------*
                                Init read/write iterator
 WARNING: The path names should stay accessible until _iter_end if mode != 0
 *-------------------------------------------------------------------------*/
err_Type txf_iter(
  char * fname, 	/* "file to read" pathname */
  char * wfname,	/* "file to write" pathname if not NULL */
  long sp,		/* First offset in file (0 for beginning) */
  txf_iter_type * iter	/* (OUT): Iterator */
); 
/*-------------------------------------------------------------------------*
                                2nd iterator 
 Ended by txf_iter_end
 *-------------------------------------------------------------------------*/
err_Type txf_iter2(
  txf_iter_type * fiter,	// Mother iterator
  unsigned Mode, 
  char * wfname,		// "file to write" pathname if not NULL 
  ulong sp,			// Where to start iterator in file
  txf_iter_type * iter		/* (OUT): Iterator */
); 
/*-------------------------------------------------------------------------*
                                End iterator
 Mode : replace read file with write file
        make a .bak with read file
 *-------------------------------------------------------------------------*/
/* Modes : ORed */
#define txf_file_mode_nop	0	/* nop */
#define txf_file_mode_replace	1	/* replace read file with write file */
#define txf_file_mode_bak	3	/* idem and rename read file to .bak */
#define txf_file_mode_clear	4	/* Clear write file if any */
#define txf_file_mode_cpyeof	8	/* Copy rfile to wfile up to eof */

err_Type txf_iter_end(
  txf_iter_type * iter,
  unsigned mode
); 
/*-------------------------------------------------------------------------*
                                Get iterator cursor
 *-------------------------------------------------------------------------*/
err_Type txf_iter_get(
  txf_iter_type * iter,
  char * p,		/* Current pointer in last read line, 0 for bol */
  txf_cur_type * cur	/* (OUT): Cursor */
);
/*-------------------------------------------------------------------------*
                                Set iterator cursor
 Cursors become invalid if file is modified
 Cursor could not be restored from an other file. 
 The last read line buffer is discarded if pp == NULL
 otherwise *pp is restored. 
 The buffer is reloaded if it is not the same line num.
 WARNING : Some trouble if only partoflines are read. 
 *-------------------------------------------------------------------------*/
err_Type txf_iter_set(
  txf_iter_type * iter,	/* (OUT): iterator */
  char ** pp,		/* (OUT): Get Current pointer, or NULL */
  txf_cur_type * cur	/* (IN): Cursor */
);
/*-------------------------------------------------------------------------*
                                Get current offset in read file
 *-------------------------------------------------------------------------*/
err_Type txf_roffs_get(
  txf_iter_type * iter,		/* Iterator */
  long * poffs,			/* (OUT) */
  int bol			/* lineoffs (else cuurent file offset) */
  ); 
/*-------------------------------------------------------------------------*
                                Goto eof
 *-------------------------------------------------------------------------*/
err_Type txf_goto_eof(
  txf_iter_type * iter		/* Iterator */
  ); 
/*-------------------------------------------------------------------------*
                                Goto bol
 *-------------------------------------------------------------------------*/
err_Type txf_goto_bol(
  txf_iter_type * iter		/* Iterator */
  ); 
/*-------------------------------------------------------------------------*
                                Get real path name of write file
 *-------------------------------------------------------------------------*/
err_Type txf_iter_wfname(
        txf_iter_type * iter,
        char * name
        );
/*---------------------------------------------------------------------*
                        Create a unique temporary file name
                        for writing
 *---------------------------------------------------------------------*/
err_Type txf_NameNew(
        char * FileName		/* (IN): Name to derive or "",
                                   (OUT): New file name */
        );
/*-------------------------------------------------------------------------*
                                Read line
 *-------------------------------------------------------------------------*
 The next line is returned as a string (null terminated, no CRLF).
 If the line is too long, only a part of it is returned. 
 *-------------------------------------------------------------------------*/
extern
err_Type txf_read_line(
  txf_iter_type * iter,		/* Iterator */
  unsigned linemax,		/* Max length */
  char * line 			/* (OUT) string */
); 
/*-------------------------------------------------------------------------*
                                Write line
 *-------------------------------------------------------------------------*
 The string is written as a line. 
 CRLF is appended. 
 *-------------------------------------------------------------------------*/
extern
err_Type txf_write_line(
  txf_iter_type * iter,		/* Iterator */
  char * line 			/* string */
); 
/*-------------------------------------------------------------------------*
                                Write record
 *-------------------------------------------------------------------------*
 The current reading position in the last read line is given by 
 a pointer in this line (or NULL for beginning of next line).
 The "read but not yet written" part of the read file is split into 
 two parts by a cursor value (or NULL for beginning of last read line).
 The write file is appended according to mode with :
  copy	    : the whole "rbnyw" part of the read file.
  copybf    : the "rbnyw" part before the cursor. 
  copyaf    : the "rbnyw" part after the cursor. 
  Skip      : Nothing.
 In addition, a string could be written (before) or (after) the "rbnyw"
 or at the cursor position : (middle).
 The string can signal some eols with LF chars. 
 *-------------------------------------------------------------------------*/
/* Mode flags (ORed) */
/* Read write MODE */
#define txf_write_mode_copymsk	0x03	/* copy mode mask */
#define txf_write_mode_copy	3	/* whole part */
#define txf_write_mode_copybf	1	/* part before cursor */
#define txf_write_mode_copyaf	2	/* part after cursor */
#define txf_write_mode_skip	0	/* nothing */

/* String MODE */
#define txf_write_mode_strmsk	0x0C	/* String mode mask */
#define txf_write_mode_before	4	/* write string before */
#define txf_write_mode_after	8	/* write string after */
#define txf_write_mode_middle	0x0C	/* write string middle */

err_Type txf_write_str(
  txf_iter_type * iter, 
  int mode,
  char * p,			/* pointer in last read line (or NULL=after eol) */
  char * stri, 			/* String null terminated or NULL */
  txf_cur_type * cur		/* Cursor position or NULL */
  ); 
/*-------------------------------------------------------------------------*
                                Put string
 The string is written into wfile, at current position. 
 NO MORE : The string can signal some eols with LF chars
           LF are only allowed
 *-------------------------------------------------------------------------*/
err_Type txf_puts(
  txf_iter_type * iter, 
  char * stri 			/* String null terminated or NULL */
  );
/*-------------------------------------------------------------------------*/
#endif
