/***********************************************************************
                M Bus (I2C)
 ***********************************************************************
 The MBUS space is viewed as a set of 128 slave devices. 
 Each device is identified with a 7bit address. 
 One of these devices is the CPU itself (as a slave).
 The CPU is at the same time a master that can access the slave devices but
 not itself. 
 
 A device user interface is defined by a mbus_DeviceType structure. 
 This structure contains a set of the specific procedures: the device methods. 
 
 CPU is MASTER (default):
 The slave devices can be accessed through the device structure. 
 
 CPU is SLAVE :
 (not implemented)
 A mbus_SlaveType structure defines the behavior of the CPU as a slave device. 
 ***********************************************************************
 10/02/2009 - Marc LEGOUX - V2r23p: OBJNET Class
 01/09/2008 - Marc LEGOUX - V2r23n: Creation
 ***********************************************************************/
#ifndef mbus_h
#define mbus_h

#include <err.h>
#include <syst.h>
#include <drv.h>

#define mbus__Err		91030
#define mbus__NoDriver		(mbus__Err)	// No current driver
#define mbus__LostArbitration	(mbus__Err + 1)
#define mbus__Busy		(mbus__Err + 2)	// Bus is busy
#define mbus__Mode		(mbus__Err + 3)	// Wrong mode
#define mbus__NotStarted	(mbus__Err + 4)	// Start signal not sent
#define mbus__Reset		(mbus__Err + 5)	// Line reset
#define mbus__Device		(mbus__Err + 6)	// Wrong device
#define mbus__DeviceExists	(mbus__Err + 7)	// Device already assigned

// OBN Class mbus BEGIN
/*----------------------------------------------------------------------*
			Driver
 *----------------------------------------------------------------------*/
#define mbus_Type void
/*----------------------------------------------------------------------*
                        Config
 *----------------------------------------------------------------------*/
#define mbus_ConfigModePut		0x80	// Put / else Get

#define mbus_ConfigModeMask		0x7F	// Config item selection
#define mbus_ConfigModeFreq		1	// Bus frequency in KHz

extern
err_Type mbus_Config(
        mbus_Type * This, 
        unsigned Mode, 		// Mode
        ulong	Val, 		// New value
        ulong * pVal		// OBN OUT,Length=1: null or (out) Current value
        );
/*----------------------------------------------------------------------*
			Device
 *----------------------------------------------------------------------*/
#define mbus_DeviceRef ulong	// Device Addr (0..127)
/*----------------------------------------------------------------------*
			Read/Write Modes
 *----------------------------------------------------------------------*/
#define mbus_DeviceModeWrite	1	// Write data else Read
#define mbus_DeviceModeNoStop	4	// No Stop Signal at the end (Write only)
#define mbus_DeviceModeReset	8	// Signal a waiting task

#define mbus_DeviceModeStart		2	// Internal mode
/*----------------------------------------------------------------------*
                        Device communication
 *----------------------------------------------------------------------*
 Sends a write or a read command to a device. 
 The Stop signal releases the MBUS. 
 The mode NoStop enables to link several write commands and a terminating
 read command without releasing the bus. 
 *----------------------------------------------------------------------*/
extern
err_Type mbus_DeviceCom(
        mbus_Type * Drv, 	// MBUS driver. 0 for current
        mbus_DeviceRef Device,	// Device address
        unsigned Mode, 		// Mode
        syst_TType Timeout,	// Timeout if the bus is busy
        unsigned Len, 		// Length to transfert
        unsigned * pLen,	// OBN Length=1: null or (out) Transfered length
        uchar * Buf		// OBN IN,Length=Len,OUT,Length=pLen: In Data to write, Out Read data
        );
/*----------------------------------------------------------------------*
                        New device
 Assign a driver to a device
 *----------------------------------------------------------------------*/
extern
err_Type mbus_DeviceNew(
        mbus_Type * This, 
        mbus_DeviceRef Device,  // Device
        unsigned Mode, 		// Mode
        drv_Type * Drv		// OBN Length=1
        );
/*----------------------------------------------------------------------*
                        Remove device
 *----------------------------------------------------------------------*/
extern
err_Type mbus_DeviceDel(
        mbus_Type * This, 
        mbus_DeviceRef Device,  // Device
        unsigned Mode, 		// Mode
        drv_Ref * pDrv		// OBN OUT, Length=1: null or (out) assigned driver
        );
/*--------------------------------------------------------------------*/
// OBN Class mbus END
/************************************************************************
			MODULE
 ************************************************************************/
/*--------------------------------------------------------------------*
                Start
 *--------------------------------------------------------------------*/
extern 
err_Type mbus_Start(void * cfg);
/*--------------------------------------------------------------------*
                End
 *--------------------------------------------------------------------*/
extern 
err_Type mbus_End(void * cfg);

#endif
