/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Specific SAM (AV4 and future SAM's) HAL Component of Reader Library Framework.
 * Implementation of generic HAL interfaces
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHHALHW_SAM_H
#define PHHALHW_SAM_H

#include <ph_Status.h>
#include <phhalHw.h>

#define PHHAL_HW_SAM_DEFAULT_TIMEOUT                                            150U    /**< Default timeout in microseconds. */
#define PHHAL_HW_SAM_HSM_AES_NO_CHAINING                                        0x00U
#define PHHAL_HW_SAM_HSM_AES_CHAINING_NO_SM                                     0x0FU
#define PHHAL_HW_SAM_HSM_AES_CHAINING                                           0xFFU
#define PHHAL_HW_SAM_HSM_AES_NO_SM                                              0x00U
#define PHHAL_HW_SAM_HSM_AES_MAC                                                0x0FU
#define PHHAL_HW_SAM_HSM_AES_ENC                                                0xF0U

#define PHHAL_HW_SAM_HC_SAM_UID_SIZE                                            0x07U   /* Length of the SAM UID */
#define PHHAL_HW_SAM_CMD_GET_VERSION_RESPONSE_LENGTH                            0x1FU   /* Response length for GetVersion command */
#define PHHAL_HW_SAM_CMD_GET_VERSION_RESPONSE_UID_OFFSET                        0x0EU   /* Offset of first UID byte field in the Version field */

/** Macros used for command buffer and other buffer processing. */
#define PHHAL_HW_SAM_ISO7816_HEADER_LENGTH                                      0x05U   /* Length of ISO7816 header */
#define PHHAL_HW_SAM_ISO7816_HEADER_LE_LENGTH                                   0x06U   /* Length of ISO7816 header including LE byte*/
#define PHHAL_HW_SAM_ISO7816_HEADER_NO_LC_LENGTH                                0x04U   /* Length of ISO7816 header excluding LC byte*/
#define PHHAL_HW_SAM_ISO7816_CLA_POS                                            0x00U   /* Position of CLA byte in ISO7816 header */
#define PHHAL_HW_SAM_ISO7816_INS_POS                                            0x01U   /* Position of INS byte in ISO7816 header */
#define PHHAL_HW_SAM_ISO7816_P1_POS                                             0x02U   /* Position of P1 byte in ISO7816 header */
#define PHHAL_HW_SAM_ISO7816_P2_POS                                             0x03U   /* Position of P2 byte in ISO7816 header */
#define PHHAL_HW_SAM_ISO7816_LC_POS                                             0x04U   /* Position of LC byte in ISO7816 header */
#define PHHAL_HW_SAM_ISO7816_SW1SW2_LENGTH                                      0x02U   /* Length of SW1SW2 in ISO7816 */
#define PHHAL_HW_SAM_ISO7816_MAX_LC_MULTIPLE_AESBLOCK                           0xF0U   /* Maximum data field length, which is still a multiple of AES block length. */
#define PHHAL_HW_CMD_SAM_TRUNCATED_MAC_SIZE                                     0x08U   /* Truncated MAC size. */

#define PHHAL_HW_SAM_ISO7816_CLA_BYTE                                           0x80U   /* CLA Byte of SamAV2 Commands */
#define PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE                                    0x00U   /* Default Ins Byte of Sam Commands */
#define PHHAL_HW_SAM_ISO7816_DEFAULT_P2_BYTE                                    0x00U   /* Default Ins Byte of Sam Commands */
#define PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE                                    0x00U   /* Default LC Byte of Sam Commands */
#define PHHAL_HW_SAM_ISO7816_DEFAULT_LE_BYTE                                    0x00U   /* Default Le Byte of Sam Commands */

#define PHHAL_HW_SAM_USHORT_MASK_MSB(Data)                                      (uint8_t) (((Data) & 0xFF00U) >> 8U)
#define PHHAL_HW_SAM_USHORT_MASK_LSB(Data)                                      (uint8_t) ((Data) & 0x00FFU)

#define PHHAL_HW_SAM_USHORT_BYTE(Buffer, BuffLen, Data)                 \
    (Buffer)[(BuffLen)++] = (uint8_t) (((Data) & 0xFF00U) >> 8U);       \
    (Buffer)[(BuffLen)++] = (uint8_t) (((Data) & 0x00FFU));

#define PHHAL_HW_SAM_UINT_BYTE(Buffer, BuffLen, Data)                   \
    (Buffer)[(BuffLen)++] = (uint8_t) (((Data) & 0xFF000000U) >> 24U);  \
    (Buffer)[(BuffLen)++] = (uint8_t) (((Data) & 0x00FF0000U) >> 16U);  \
    (Buffer)[(BuffLen)++] = (uint8_t) (((Data) & 0x0000FF00U) >> 8U);   \
    (Buffer)[(BuffLen)++] = (uint8_t) (((Data) & 0x000000FFU));

#define PHHAL_HW_SAM_GET_PICC_ERROR(Buffer, BuffLen)                    \
    (uint16_t) (((Buffer)[(BuffLen) - 2U] << 8U) | (Buffer)[(BuffLen) - 1U])

/** \addtogroup phhalHw_Sam_Cmd_ErrorCode
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_ErrorCode_Sam ErrorCode_SAM
 * \brief The actual error codes received from SAM. These codes are for reference purpose only.
 * All the interfaces will return codes mentioned in \ref phhalHw_Sam_Cmd_ErrorCodes_Mapped "Mapped Codes" and not these codes.
 * @{
 */
#define PHHAL_HW_SAM_RET_CODE_HW_EEPROM                                         0x6400U /**< SAM Status - EEPROM busy, access collision, memory range cannot be mapped, execution error.
                                                                                         *   Equivalent mapped error code will be #PHHAL_HW_SAM_ERR_HW_EEPROM.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_HW_RC5XX                                          0x6401U /**< SAM Status - Reader IC interface error.
                                                                                         *   Equivalent mapped error code will be #PHHAL_HW_SAM_ERR_HW_RC5XX.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_KEY_CREATE_FAILED                                 0x6501U /**< SAM Status - Key creation failure.
                                                                                         *   Equivalent mapped error code will be #PHHAL_HW_SAM_ERR_KEY_CREATE_FAILED.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_KEY_REF_NO_INVALID                                0x6502U /**< SAM Status - Key reference number invalid.
                                                                                         *   Equivalent mapped error code will be #PHHAL_HW_SAM_ERR_KEY_REF_NO_INVALID.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_KEY_KUC_NO_INVALID                                0x6503U /**< SAM Status - Key usage counter number invalid.
                                                                                         *   Equivalent mapped error code will be #PHHAL_HW_SAM_ERR_KEY_KUC_NO_INVALID.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_HW_EE_HIGH_VOLTAGE                                0x6581U /**< SAM Status - EEPROM high voltage failure; SAM permanently disabled.
                                                                                         *   Equivalent mapped error code will be #PHHAL_HW_SAM_ERR_HW_EE_HIGH_VOLTAGE.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_ISO7816_WRONG_LENGTH_LC                           0x6700U /**< SAM Status - Wrong length of the APDU or wrong LC byte.
                                                                                         *   Equivalent mapped error code will be #PHHAL_HW_SAM_ERR_ISO7816_WRONG_LENGTH_LC.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_INCOMPLETE_CHAINING                               0x6883U /**< SAM Status - Final chained command expected; running command aborted,
                                                                                         *   new command ignored
                                                                                         *   Equivalent mapped error code will be #PHHAL_HW_SAM_ERR_INCOMPLETE_CHAINING.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_COMMAND_CHAINING_NOT_SUPPORTED                    0x6884U /**< SAM Status - [If MFP command] Command-chaining not supported
                                                                                         *   Equivalent mapped error code will be #PHHAL_HW_SAM_ERR_COMMAND_CHAINING_NOT_SUPPORTED.
                                                                                         */

#define PHHAL_HW_SAM_RET_CODE_UNMATCHING_PCD_PD                                 0x6982U /**< SAM Status - [If AuthenticateFirst] Un-matching PCD/PD capabilities
                                                                                         *   Equivalent mapped error code will be #PHHAL_HW_SAM_ERR_UNMATCHING_PCD_PD.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_KEY_INTEGRITY_ERROR                               0x6984U /**< SAM Status - One of the following
                                                                                         *      - CRC mismatch; wrong key handling buffer size
                                                                                         *      - HostRespData received from programmable logic is not
                                                                                         *        multiple of 16 bytes, while chaining applying HostMode.Full
                                                                                         *        is ongoing
                                                                                         *   Equivalent mapped error code will be #PHHAL_HW_SAM_ERR_KEY_INTEGRITY_ERROR.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_COND_USE_NOT_SATISFIED                            0x6985U /**< SAM Status - Conditions of use not satisfied. Multiple meaning for
                                                                                         *   this status, refer the command for actual information.
                                                                                         *   Equivalent mapped error code will be #PHHAL_HW_SAM_ERR_COND_USE_NOT_SATISFIED.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_ISO7816_COMMAND_NOT_ALLOWED                       0x6986U /**< SAM Status - Command not allowed. Multiple meaning for
                                                                                         *   this status, refer the command for actual information.
                                                                                         *   Equivalent mapped error code will be #PHHAL_HW_SAM_ERR_ISO7816_COMMAND_NOT_ALLOWED.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_ISO7816_WRONG_PARAMS_FOR_INS                      0x6A80U /**< SAM Status - Unexpected RFU bit(s) value(s) in data field parameter.
                                                                                         *   Multiple meaning for this status, refer the command for actual
                                                                                         *   information.
                                                                                         *   Equivalent mapped error code will be #PHHAL_HW_SAM_ERR_ISO7816_WRONG_PARAMS_FOR_INS.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_KEY_VERSION_INVALID                               0x6A82U /**< SAM Status - No valid key version found
                                                                                         *   Equivalent mapped error code will be #PHHAL_HW_SAM_ERR_KEY_VERSION_INVALID.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_UNKNOWN_PUBLIC_KEY_INDEX                          0x6A83U /**< SAM Status - Unknown Public Key Index when trying to update
                                                                                         *   Equivalent mapped error code will be #PHHAL_HW_SAM_ERR_UNKNOWN_PUBLIC_KEY_INDEX.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_HOST_PROTECTION_ERROR                             0x6A84U /**< SAM Status - SAM Host protection error
                                                                                         *   Equivalent mapped error code will be #PHHAL_HW_SAM_ERR_HOST_PROTECTION.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_ISO7816_WRONG_P1P2                                0x6A86U /**< SAM Status - Incorrect value for parameter P1 and/or P2
                                                                                         *   Equivalent mapped error code will be #PHHAL_HW_SAM_ERR_ISO7816_WRONG_P1P2.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_ISO7816_WRONG_LE                                  0x6C00U /**< SAM Status - Incorrect LE value
                                                                                         *   Equivalent mapped error code will be #PHHAL_HW_SAM_ERR_ISO7816_WRONG_LE.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_ISO7816_UNKNOWN_INS                               0x6D00U /**< SAM Status - Instruction code not supported or invalid or not available
                                                                                         *   in the current state of the SAM
                                                                                         *   Equivalent mapped error code will be #PHHAL_HW_SAM_ERR_ISO7816_UNKNOWN_INS.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_ISO7816_WRONG_CLASS                               0x6E00U /**< SAM Status - Class not supported
                                                                                         *   Equivalent mapped error code will be #PHHAL_HW_SAM_ERR_ISO7816_UNKNOWN_CLASS.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_NO_PRECISE_DIAGNOSIS                              0x6F00U /**< SAM Status - No precise diagnosis (physical address invalid, EEPROM error,
                                                                                         *   no correct key loaded for cryptographic operation, improper length for cryptographic
                                                                                         *   operation, internal limit exceeded.)
                                                                                         *   Equivalent mapped error code will be #PHHAL_HW_SAM_ERR_NO_PRECISE_DIAGNOSIS.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_OK                                                0x9000U /**< SAM Status - Correct execution. Multiple meaning for this status,
                                                                                         *   refer the command for actual information.
                                                                                         *   Equivalent mapped error code will be #PH_ERR_SUCCESS.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_OK_1BIT                                           0x9001U /**< SAM Status - Correct execution, 1 bit received during ISO14443-3_TransparentExchange
                                                                                         *   command exchange. Equivalent mapped error code will be #PH_ERR_SUCCESS_INCOMPLETE_BYTE.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_OK_2BIT                                           0x9002U /**< SAM Status - Correct execution, 2 bit received during ISO14443-3_TransparentExchange
                                                                                         *   command exchange.
                                                                                         *   Equivalent mapped error code will be #PH_ERR_SUCCESS_INCOMPLETE_BYTE.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_OK_3BIT                                           0x9003U /**< SAM Status - Correct execution, 3 bit received during ISO14443-3_TransparentExchange
                                                                                         *   command exchange.
                                                                                         *   Equivalent mapped error code will be #PH_ERR_SUCCESS_INCOMPLETE_BYTE.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_OK_4BIT                                           0x9004U /**< SAM Status - Correct execution, 4 bit received during ISO14443-3_TransparentExchange
                                                                                         *   command exchange.
                                                                                         *   Equivalent mapped error code will be #PH_ERR_SUCCESS_INCOMPLETE_BYTE.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_OK_5BIT                                           0x9005U /**< SAM Status - Correct execution, 5 bit received during ISO14443-3_TransparentExchange
                                                                                         *   command exchange.
                                                                                         *   Equivalent mapped error code will be #PH_ERR_SUCCESS_INCOMPLETE_BYTE.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_OK_6BIT                                           0x9006U /**< SAM Status - Correct execution, 6 bit received during ISO14443-3_TransparentExchange
                                                                                         *   command exchange.
                                                                                         *   Equivalent mapped error code will be #PH_ERR_SUCCESS_INCOMPLETE_BYTE.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_OK_7BIT                                           0x9007U /**< SAM Status - Correct execution, 7 bit received during ISO14443-3_TransparentExchange
                                                                                         *   command exchange.
                                                                                         *   Equivalent mapped error code will be #PH_ERR_SUCCESS_INCOMPLETE_BYTE.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_CRYPTO_FAILURE                                    0x901EU /**< SAM Status - MAC verification failed. Multiple meaning for
                                                                                         *   this status, refer the command for actual information.
                                                                                         *   Equivalent mapped error code will be #PHHAL_HW_SAM_ERR_CRYPTO.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_OK_CHAINING_ACTIVE_DUOX                           0x90ADU /**< SAM Status - Correct execution - Correct Execution with Cert.A - more data remaining
                                                                                         *   Equivalent mapped error code will be #PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE_DUOX.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_OK_CHAINING_ACTIVE_EXT                            0x90AEU /**< SAM Status - Correct execution - Correct execution - more data expected
                                                                                         *   (SAM-Host secure messaging to be applied on each command exchange)
                                                                                         *   Equivalent mapped error code will be #PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE_EXT.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_OK_CHAINING_ACTIVE                                0x90AFU /**< SAM Status - Correct execution - more data expected
                                                                                         *   Equivalent mapped error code will be #PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_MIFARE_PLUS_INCORRECT_DATA                        0x90BEU /**< SAM Status - MIFARE PICC sent incorrect amount of data (for example
                                                                                          *  wrong length of MAC)
                                                                                         *   Equivalent mapped error code will be #PHHAL_HW_SAM_ERR_MIFARE_PLUS_INCORRECT_DATA.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_MIFARE_PLUS_ERROR                                 0x90BFU /**< SAM Status - Correct SAM execution but MFP response is an error return code
                                                                                         *   Equivalent mapped error code will be #PHHAL_HW_SAM_ERR_MIFARE_PLUS_GEN.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_ISO_UID_INCOMPLETE                                0x90C0U /**< SAM Status - Correct execution - UID not complete (selection sequence
                                                                                         *   has to be continued)
                                                                                         *   Equivalent mapped error code will be #PHHAL_HW_SAM_ERR_ISO_UID_INCOMPLETE.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_DESFIRE_ERROR                                     0x90DFU /**< SAM Status - Correct execution, DESFire returned error or incorrect
                                                                                         *   length
                                                                                         *   Equivalent mapped error code will be #PHHAL_HW_SAM_ERR_DESFIRE_GEN.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_COMM_IO_TIMEOUT                                   0x90E0U /**< SAM Status - No response from card (no card in field)
                                                                                         *   Equivalent mapped error code will be #PH_ERR_IO_TIMEOUT.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_COMM_ISO14443_PROTOCOL_EEROR                      0x90E1U /**< SAM Status - ISO 14443 protocol error
                                                                                         *   Equivalent mapped error code will be #PH_ERR_PROTOCOL_ERROR.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_COMM_PARITY_ERROR                                 0x90E2U /**< SAM Status - Parity Error
                                                                                         *   Equivalent mapped error code will be #PH_ERR_INTEGRITY_ERROR.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_COMM_BUFFER_OVERFLOW                              0x90E3U /**< SAM Status - Buffer Overflow error
                                                                                         *   Equivalent mapped error code will be #PH_ERR_BUFFER_OVERFLOW.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_COMM_CRC_FAILURE                                  0x90E4U /**< SAM Status - (N)ACK expected, CRC mismatch
                                                                                         *   Equivalent mapped error code will be #PH_ERR_INTEGRITY_ERROR.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_COMM_RF_FAILURE                                   0x90E5U /**< SAM Status - RF Field in-active
                                                                                         *   Equivalent mapped error code will be #PH_ERR_RF_ERROR.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_COMM_TEMP_FAILURE                                 0x90E6U /**< SAM Status - Temperature error
                                                                                         *   Equivalent mapped error code will be #PH_ERR_TEMPERATURE_ERROR.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_COMM_FIFO_WRITE                                   0x90E7U /**< SAM Status - FIFO write error
                                                                                         *   Equivalent mapped error code will be #PH_ERR_READ_WRITE_ERROR.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_COMM_COLLISION                                    0x90E8U /**< SAM Status - Collision error
                                                                                         *   Equivalent mapped error code will be #PH_ERR_COLLISION_ERROR.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_COMM_INTERNAL_BUF_OVERFLOW                        0x90E9U /**< SAM Status - Internal transaction buffer overflow (more than 256 bytes
                                                                                         *   received from the card)
                                                                                         *   Equivalent mapped error code will be #PH_ERR_BUFFER_OVERFLOW.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_UID_BCC_INCORRECT                                 0x90EAU /**< SAM Status - UID check byte (BCC) incorrect
                                                                                         *   Equivalent mapped error code will be #PHHAL_HW_SAM_ERR_ISO_UID_BCC_INCORRECT.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_ISO_WRONG_BNR                                     0x90EBU /**< SAM Status - Incorrect block number
                                                                                         *   Equivalent mapped error code will be #PHHAL_HW_SAM_ERR_ISO_WRONG_BNR.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_ISO_INVALID_FORMAT                                0x90ECU /**< SAM Status - One of the following.
                                                                                         *      - ATS length byte does not match actual length of received ATS
                                                                                         *        or PPS response invalid (incorrect length or PPSS byte)
                                                                                         *      - Block length or PCB invalid
                                                                                         *      - Incorrect (no R-block or I-block) answer received
                                                                                         *   Equivalent mapped error code will be #PHHAL_HW_SAM_ERR_ISO_INVALID_FORMAT.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_ISO_INVALID_PARAMETER                             0x90EDU /**< SAM Status - One of the following.
                                                                                         *      - User buffer length below ISO14443-4 minimum
                                                                                         *      - Invalid UID bit count (too many bits received)
                                                                                         *      - CID, NAD either not expected, missing or incorrect
                                                                                         *   Equivalent mapped error code will be #PHHAL_HW_SAM_ERR_ISO_INVALID_PARAMETER.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_PROT_MIFARE_ERROR                                 0x90EFU /**< SAM Status - Correct execution but PICC response is an error return
                                                                                         *   code.
                                                                                         *   Equivalent mapped error code will be #PHHAL_HW_SAM_ERR_MIFARE_GEN.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_PROT_MIFARE_NAK0                                  0x90F0U /**< SAM Status - NACK received from PICC. Lower nibble of SW2
                                                                                         *   depicts status code from the card. MIFARE NACK 0 received.
                                                                                         *   Equivalent mapped error code will be #PHHAL_HW_SAM_ERR_MIFARE_NAK0.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_PROT_MIFARE_NAK1                                  0x90F1U /**< SAM Status - NACK received from PICC. Lower nibble of SW2
                                                                                         *   depicts status code from the card. MIFARE NACK 1 received.
                                                                                         *   Equivalent mapped error code will be #PHHAL_HW_SAM_ERR_MIFARE_NAK1.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_PROT_MIFARE_NAK4                                  0x90F4U /**< SAM Status - NACK received from PICC. Lower nibble of SW2
                                                                                         *   depicts status code from the card. MIFARE NACK 4 received.
                                                                                         *   Equivalent mapped error code will be #PHHAL_HW_SAM_ERR_MIFARE_NAK4.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_PROT_MIFARE_NAK5                                  0x90F5U /**< SAM Status - NACK received from PICC. Lower nibble of SW2
                                                                                         *   depicts status code from the card. MIFARE NACK 5 received.
                                                                                         *   Equivalent mapped error code will be #PHHAL_HW_SAM_ERR_MIFARE_NAK5.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_PROT_MIFARE_NAK6                                  0x90F6U /**< SAM Status - NACK received from PICC. Lower nibble of SW2
                                                                                         *   depicts status code from the card. MIFARE NACK 6 received.
                                                                                         *   Equivalent mapped error code will be #PHHAL_HW_SAM_ERR_MIFARE_NAK6.
                                                                                         */
#define PHHAL_HW_SAM_RET_CODE_PROT_MIFARE_NAK7                                  0x90F7U /**< SAM Status - NACK received from PICC. Lower nibble of SW2
                                                                                         *   depicts status code from the card. MIFARE NACK 7 received.
                                                                                         *   Equivalent mapped error code will be #PHHAL_HW_SAM_ERR_MIFARE_NAK7.
                                                                                         */
/**
 * end of group phhalHw_Sam_Cmd_ErrorCode_Sam
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_ErrorCodes_Mapped ErrorCodes_Mapped
 * \brief Sam Custom Error Codes for the respective status codes returned by Sam hardware.
 * @{
 */
#define PHHAL_HW_SAM_ERR_HW_EEPROM                          (PH_ERR_CUSTOM_BEGIN + 0U)  /**< SAM Custom error code - EEProm failure.
                                                                                         *   This error represents SAM's #PHHAL_HW_SAM_RET_CODE_HW_EEPROM error.
                                                                                         */
#define PHHAL_HW_SAM_ERR_HW_RC5XX                           (PH_ERR_CUSTOM_BEGIN + 1U)  /**< SAM Custom error code - Reader IC interface error.
                                                                                         *   This error represents SAM's #PHHAL_HW_SAM_RET_CODE_HW_RC5XX error.
                                                                                         */
#define PHHAL_HW_SAM_ERR_KEY_CREATE_FAILED                  (PH_ERR_CUSTOM_BEGIN + 2U)  /**< SAM Custom error code - Key creation failure.
                                                                                         *   This error represents SAM's #PHHAL_HW_SAM_RET_CODE_KEY_CREATE_FAILED error.
                                                                                         */
#define PHHAL_HW_SAM_ERR_KEY_REF_NO_INVALID                 (PH_ERR_CUSTOM_BEGIN + 3U)  /**< SAM Custom error code - Key reference number invalid.
                                                                                         *   This error represents SAM's #PHHAL_HW_SAM_RET_CODE_KEY_REF_NO_INVALID error.
                                                                                         */
#define PHHAL_HW_SAM_ERR_KEY_KUC_NO_INVALID                 (PH_ERR_CUSTOM_BEGIN + 4U)  /**< SAM Custom error code - Key usage counter number invalid.
                                                                                         *   This error represents SAM's #PHHAL_HW_SAM_RET_CODE_KEY_KUC_NO_INVALID error.
                                                                                         */
#define PHHAL_HW_SAM_ERR_HW_EE_HIGH_VOLTAGE                 (PH_ERR_CUSTOM_BEGIN + 5U)  /**< SAM Custom error code - EEPROM high voltage failure; SAM permanently disabled.
                                                                                         *   This error represents SAM's #PHHAL_HW_SAM_RET_CODE_HW_EE_HIGH_VOLTAGE error.
                                                                                         */
#define PHHAL_HW_SAM_ERR_ISO7816_WRONG_LENGTH_LC            (PH_ERR_CUSTOM_BEGIN + 6U)  /**< SAM Custom error code - Wrong length of the APDU or wrong LC byte.
                                                                                         *   This error represents SAM's #PHHAL_HW_SAM_RET_CODE_ISO7816_WRONG_LENGTH_LC error.
                                                                                         */
#define PHHAL_HW_SAM_ERR_INCOMPLETE_CHAINING                (PH_ERR_CUSTOM_BEGIN + 7U)  /**< SAM Custom error code - Final chained command expected; running command aborted,
                                                                                         *   new command ignored
                                                                                         *   This error represents SAM's #PHHAL_HW_SAM_RET_CODE_INCOMPLETE_CHAINING error.
                                                                                         */
#define PHHAL_HW_SAM_ERR_UNMATCHING_PCD_PD                  (PH_ERR_CUSTOM_BEGIN + 8U)  /**< SAM Custom error code - [If AuthenticateFirst] Un-matching PCD/PD capabilities
                                                                                         *   This error represents SAM's #PHHAL_HW_SAM_RET_CODE_UNMATCHING_PCD_PD error.
                                                                                         */
#define PHHAL_HW_SAM_ERR_KEY_INTEGRITY_ERROR                (PH_ERR_CUSTOM_BEGIN + 9U)  /**< SAM Custom error code - One of the following
                                                                                         *      - CRC mismatch; wrong key handling buffer size
                                                                                         *      - HostRespData received from programmable logic is not
                                                                                         *        multiple of 16 bytes, while chaining applying HostMode.Full
                                                                                         *        is ongoing
                                                                                         *   This error represents SAM's #PHHAL_HW_SAM_RET_CODE_KEY_INTEGRITY_ERROR error.
                                                                                         */
#define PHHAL_HW_SAM_ERR_COND_USE_NOT_SATISFIED             (PH_ERR_CUSTOM_BEGIN + 10U) /**< SAM Custom error code - Conditions of use not satisfied. Multiple meaning for
                                                                                         *   this status, refer the command for actual information.
                                                                                         *   This error represents SAM's #PHHAL_HW_SAM_RET_CODE_COND_USE_NOT_SATISFIED error.
                                                                                         */
#define PHHAL_HW_SAM_ERR_ISO7816_COMMAND_NOT_ALLOWED        (PH_ERR_CUSTOM_BEGIN + 11U) /**< SAM Custom error code - Command not allowed. Multiple meaning for
                                                                                         *   this status, refer the command for actual information.
                                                                                         *   This error represents SAM's #PHHAL_HW_SAM_RET_CODE_ISO7816_COMMAND_NOT_ALLOWED error.
                                                                                         */
#define PHHAL_HW_SAM_ERR_ISO7816_WRONG_PARAMS_FOR_INS       (PH_ERR_CUSTOM_BEGIN + 12U) /**< SAM Custom error code - Unexpected RFU bit(s) value(s) in data field parameter.
                                                                                         *   Multiple meaning for this status, refer the command for actual
                                                                                         *   information.
                                                                                         *   This error represents SAM's #PHHAL_HW_SAM_RET_CODE_ISO7816_WRONG_PARAMS_FOR_INS error.
                                                                                         */
#define PHHAL_HW_SAM_ERR_KEY_VERSION_INVALID                (PH_ERR_CUSTOM_BEGIN + 13U) /**< SAM Custom error code - No valid key version found.
                                                                                         *   This error represents SAM's #PHHAL_HW_SAM_RET_CODE_KEY_VERSION_INVALID error.
                                                                                         */
#define PHHAL_HW_SAM_ERR_HOST_PROTECTION                    (PH_ERR_CUSTOM_BEGIN + 14U) /**< SAM Custom error code - SAM Host protection error.
                                                                                         *   This error represents SAM's #PHHAL_HW_SAM_RET_CODE_HOST_PROTECTION_ERROR error.
                                                                                         */
#define PHHAL_HW_SAM_ERR_ISO7816_WRONG_P1P2                 (PH_ERR_CUSTOM_BEGIN + 15U) /**< SAM Custom error code - Incorrect value for parameter P1 and/or P2.
                                                                                         *   This error represents SAM's #PHHAL_HW_SAM_RET_CODE_ISO7816_WRONG_P1P2 error.
                                                                                         */
#define PHHAL_HW_SAM_ERR_ISO7816_WRONG_LE                   (PH_ERR_CUSTOM_BEGIN + 16U) /**< SAM Custom error code - Incorrect LE value.
                                                                                         *   This error represents SAM's #PHHAL_HW_SAM_RET_CODE_ISO7816_WRONG_LE error.
                                                                                         */
#define PHHAL_HW_SAM_ERR_ISO7816_UNKNOWN_INS                (PH_ERR_CUSTOM_BEGIN + 17U) /**< SAM Custom error code - Instruction code not supported or invalid or not available
                                                                                         *   in the current state of the SAM.
                                                                                         *   This error represents SAM's #PHHAL_HW_SAM_RET_CODE_ISO7816_UNKNOWN_INS error.
                                                                                         */
#define PHHAL_HW_SAM_ERR_ISO7816_UNKNOWN_CLASS              (PH_ERR_CUSTOM_BEGIN + 18U) /**< SAM Custom error code - Class not supported.
                                                                                         *   This error represents SAM's #PHHAL_HW_SAM_RET_CODE_ISO7816_WRONG_CLASS error.
                                                                                         */
#define PHHAL_HW_SAM_ERR_CRYPTO                             (PH_ERR_CUSTOM_BEGIN + 19U) /**< SAM Custom error code - MAC verification failed. Multiple meaning for
                                                                                         *   this status, refer the command for actual information.
                                                                                         *   This error represents SAM's #PHHAL_HW_SAM_RET_CODE_CRYPTO_FAILURE error.
                                                                                         */
#define PHHAL_HW_SAM_ERR_MIFARE_PLUS_INCORRECT_DATA         (PH_ERR_CUSTOM_BEGIN + 20U) /**< SAM Custom error code - MIFARE PICC sent incorrect amount of data (for example
                                                                                          *  wrong length of MAC)
                                                                                         *   This error represents SAM's #PHHAL_HW_SAM_RET_CODE_MIFARE_PLUS_INCORRECT_DATA error.
                                                                                         */
#define PHHAL_HW_SAM_ERR_MIFARE_PLUS_GEN                    (PH_ERR_CUSTOM_BEGIN + 21U) /**< SAM Custom error code - Correct SAM execution but MFP response is an error return code
                                                                                         *   This error represents SAM's #PHHAL_HW_SAM_RET_CODE_MIFARE_PLUS_ERROR error.
                                                                                         */
#define PHHAL_HW_SAM_ERR_ISO_UID_INCOMPLETE                 (PH_ERR_CUSTOM_BEGIN + 22U) /**< SAM Custom error code - Correct execution - UID not complete (selection sequence
                                                                                         *   has to be continued)
                                                                                         *   This error represents SAM's #PHHAL_HW_SAM_RET_CODE_ISO_UID_INCOMPLETE error.
                                                                                         */
#define PHHAL_HW_SAM_ERR_DESFIRE_GEN                        (PH_ERR_CUSTOM_BEGIN + 23U) /**< SAM Custom error code - Correct execution, DESFire returned error or incorrect
                                                                                         *   length
                                                                                         *   This error represents SAM's #PHHAL_HW_SAM_RET_CODE_DESFIRE_ERROR error.
                                                                                         */
#define PHHAL_HW_SAM_ERR_ISO_WRONG_BNR                      (PH_ERR_CUSTOM_BEGIN + 24U) /**< SAM Custom error code - Incorrect block number
                                                                                         *   This error represents SAM's #PHHAL_HW_SAM_RET_CODE_ISO_WRONG_BNR error.
                                                                                         */
#define PHHAL_HW_SAM_ERR_ISO_INVALID_FORMAT                 (PH_ERR_CUSTOM_BEGIN + 25U) /**< SAM Custom error code - Status - One of the following.
                                                                                         *      - ATS length byte does not match actual length of received ATS
                                                                                         *        or PPS response invalid (incorrect length or PPSS byte)
                                                                                         *      - Block length or PCB invalid
                                                                                         *      - Incorrect (no R-block or I-block) answer received
                                                                                         *   This error represents SAM's #PHHAL_HW_SAM_RET_CODE_ISO_INVALID_FORMAT error.
                                                                                         */
#define PHHAL_HW_SAM_ERR_ISO_INVALID_PARAMETER              (PH_ERR_CUSTOM_BEGIN + 26U) /**< SAM Custom error code - One of the following.
                                                                                         *      - User buffer length below ISO14443-4 minimum
                                                                                         *      - Invalid UID bit count (too many bits received)
                                                                                         *      - CID, NAD either not expected, missing or incorrect
                                                                                         *   This error represents SAM's #PHHAL_HW_SAM_RET_CODE_ISO_INVALID_PARAMETER error.
                                                                                         */
#define PHHAL_HW_SAM_ERR_MIFARE_GEN                         (PH_ERR_CUSTOM_BEGIN + 27U) /**< SAM Custom error code - Correct execution but PICC response is an error return
                                                                                         *   code.
                                                                                         *   This error represents SAM's #PHHAL_HW_SAM_RET_CODE_PROT_MIFARE_ERROR error.
                                                                                         */
#define PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE                 (PH_ERR_CUSTOM_BEGIN + 28U) /**< SAM Custom error code - Correct execution - more data expected
                                                                                         *   This error represents SAM's #PHHAL_HW_SAM_RET_CODE_OK_CHAINING_ACTIVE error.
                                                                                         */
#define PHHAL_HW_SAM_ERR_COMMAND_CHAINING_NOT_SUPPORTED     (PH_ERR_CUSTOM_BEGIN + 29U) /**< SAM Custom error code - [If MFP command] Command-chaining not supported
                                                                                         *   This error represents SAM's #PHHAL_HW_SAM_RET_CODE_COMMAND_CHAINING_NOT_SUPPORTED error.
                                                                                         */
#define PHHAL_HW_SAM_ERR_UNKNOWN_PUBLIC_KEY_INDEX           (PH_ERR_CUSTOM_BEGIN + 30U) /**< SAM Custom error code - Unknown Public Key Index when trying to update
                                                                                         *   This error represents SAM's #PHHAL_HW_SAM_RET_CODE_UNKNOWN_PUBLIC_KEY_INDEX error.
                                                                                         */
#define PHHAL_HW_SAM_ERR_NO_PRECISE_DIAGNOSIS               (PH_ERR_CUSTOM_BEGIN + 31U) /**< SAM Custom error code - No precise diagnosis (physical address invalid, EEPROM error,
                                                                                         *   no correct key loaded for cryptographic operation, improper length for cryptographic
                                                                                         *   operation, internal limit exceeded...)
                                                                                         *   This error represents SAM's #PHHAL_HW_SAM_RET_CODE_NO_PRECISE_DIAGNOSIS error.
                                                                                         */
#define PHHAL_HW_SAM_ERR_PROGRAMMABLE_LOGIC                 (PH_ERR_CUSTOM_BEGIN + 32U) /**< SAM Custom error code - Programmable Logic error. Error code returned by
                                                                                         *   SAM_PLExec command.
                                                                                         *   This error represents SAM's ISOXXXX error.
                                                                                         */
#define PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE_EXT             (PH_ERR_CUSTOM_BEGIN + 33U) /**< SAM Custom error code - Correct execution - Correct execution - more data expected
                                                                                         *   (SAM-Host secure messaging to be applied on each command exchange)
                                                                                         *   This error represents SAM's #PHHAL_HW_SAM_RET_CODE_OK_CHAINING_ACTIVE_EXT error.
                                                                                         */
#define PHHAL_HW_SAM_ERR_ISO_UID_BCC_INCORRECT              (PH_ERR_CUSTOM_BEGIN + 34U) /**< SAM Custom error code - UID check byte (BCC) incorrect.
                                                                                         *   This error represents SAM's #PHHAL_HW_SAM_RET_CODE_UID_BCC_INCORRECT error.
                                                                                         */
#define PHHAL_HW_SAM_ERR_MIFARE_NAK0                        (PH_ERR_CUSTOM_BEGIN + 35U) /**< SAM Custom error code - NACK received from PICC. MIFARE NACK 0 received.
                                                                                         *   This error represents SAM's #PHHAL_HW_SAM_RET_CODE_PROT_MIFARE_NAK0 error.
                                                                                         */
#define PHHAL_HW_SAM_ERR_MIFARE_NAK1                        (PH_ERR_CUSTOM_BEGIN + 36U) /**< SAM Custom error code - NACK received from PICC. MIFARE NACK 1 received.
                                                                                         *   This error represents SAM's #PHHAL_HW_SAM_RET_CODE_PROT_MIFARE_NAK1 error.
                                                                                         */
#define PHHAL_HW_SAM_ERR_MIFARE_NAK4                        (PH_ERR_CUSTOM_BEGIN + 37U) /**< SAM Custom error code - NACK received from PICC. MIFARE NACK 4 received.
                                                                                         *   This error represents SAM's #PHHAL_HW_SAM_RET_CODE_PROT_MIFARE_NAK4 error.
                                                                                         */
#define PHHAL_HW_SAM_ERR_MIFARE_NAK5                        (PH_ERR_CUSTOM_BEGIN + 38U) /**< SAM Custom error code - NACK received from PICC. MIFARE NACK 5 received.
                                                                                         *   This error represents SAM's #PHHAL_HW_SAM_RET_CODE_PROT_MIFARE_NAK5 error.
                                                                                         */
#define PHHAL_HW_SAM_ERR_MIFARE_NAK6                        (PH_ERR_CUSTOM_BEGIN + 39U) /**< SAM Custom error code - NACK received from PICC. MIFARE NACK 6 received.
                                                                                         *   This error represents SAM's #PHHAL_HW_SAM_RET_CODE_PROT_MIFARE_NAK6 error.
                                                                                         */
#define PHHAL_HW_SAM_ERR_MIFARE_NAK7                        (PH_ERR_CUSTOM_BEGIN + 40U) /**< SAM Custom error code - NACK received from PICC. MIFARE NACK 7 received.
                                                                                         *   This error represents SAM's #PHHAL_HW_SAM_RET_CODE_PROT_MIFARE_NAK7 error.
                                                                                         */
#define PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE_DUOX            (PH_ERR_CUSTOM_BEGIN + 41U) /**< SAM Custom error code - Correct Execution with Cert.A - more data remaining
                                                                                         *   This error represents SAM's #PHHAL_HW_SAM_RET_CODE_OK_CHAINING_ACTIVE_DUOX
                                                                                         *   error.
                                                                                         */
/**
 * end of group phhalHw_Sam_Cmd_ErrorCodes_Mapped
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_ErrorCode
 * @}
 */

phStatus_t phhalHw_Sam_Exchange(phhalHw_Sam_DataParams_t * pDataParams, uint16_t wOption, uint8_t * pData, uint16_t wDataLen,
    uint8_t ** ppResponse, uint16_t * pRespLen);

phStatus_t phhalHw_Sam_GetConfig(phhalHw_Sam_DataParams_t * pDataParams, uint16_t wConfig, uint16_t * pValue);

phStatus_t phhalHw_Sam_SetConfig(phhalHw_Sam_DataParams_t * pDataParams, uint16_t wConfig, uint16_t wValue);

phStatus_t phhalHw_Sam_SetMinFDT(phhalHw_Sam_DataParams_t * pDataParams, uint16_t wValue);

phStatus_t phhalHw_Sam_ApplyProtocolSettings(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bCardType);

phStatus_t phhalHw_Sam_ReadRegister(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bAddress, uint8_t * pValue);

phStatus_t phhalHw_Sam_WriteRegister(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bAddress, uint8_t bValue);

phStatus_t phhalHw_Sam_FieldReset(phhalHw_Sam_DataParams_t * pDataParams);

phStatus_t phhalHw_Sam_FieldOn(phhalHw_Sam_DataParams_t * pDataParams);

phStatus_t phhalHw_Sam_FieldOff(phhalHw_Sam_DataParams_t * pDataParams);

phStatus_t phhalHw_Sam_Wait(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bUnit, uint16_t wTimeout);

#endif /* PHHALHW_SAM_H */
