/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
* SAM (AV4 and future SAM's) Specific support interface implementation of Reader Library Framework.
* $Author: Rajendran Kumar (nxp99556) $
* $Revision: 7467 $
* $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
*/

#include <ph_Status.h>
#include <ph_RefDefs.h>
#include <phKeyStore.h>
#include <phCryptoSym.h>

#ifdef NXPBUILD__PHHAL_HW_SAM

#include "../phhalHw_Sam.h"
#include "phhalHw_Sam_HcUtils.h"

/* Private constants */
static const uint8_t PH_MEMLOC_CONST_ROM phhalHw_Sam_HcUtils_ZeroIv[PH_CRYPTOSYM_AES_BLOCK_SIZE] =
 { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

phStatus_t phhalHw_Sam_Utils_GetCheckLcLe(uint8_t * pCmd, uint16_t wCmdLen, uint8_t * pIsLcPresent, uint8_t * pLcLen,
    uint8_t * pIsLePresent)
{
    /* Return error if Command Length is less */
    if(wCmdLen < PHHAL_HW_SAM_ISO7816_HEADER_NO_LC_LENGTH)
    {
        return PH_ADD_COMPCODE(PH_ERR_FRAMING_ERROR, PH_COMP_HAL);
    }

    /* 1. CLA INS P1 P2 */
    if(wCmdLen == PHHAL_HW_SAM_ISO7816_HEADER_NO_LC_LENGTH)
    {
        *pLcLen = 0;
        *pIsLcPresent = PH_OFF;
        *pIsLePresent = PH_OFF;
    }

    /* 2. CLA INS P1 P2 LE */
    else if(wCmdLen == PHHAL_HW_SAM_ISO7816_HEADER_LENGTH)
    {
        *pLcLen = 0;
        *pIsLcPresent = PH_OFF;
        *pIsLePresent = PH_ON;
    }

    /* 3. CLA INS P1 P2 LC DATA */
    else if(wCmdLen == (PHHAL_HW_SAM_ISO7816_HEADER_LENGTH + pCmd[PHHAL_HW_SAM_ISO7816_LC_POS]))
    {
        *pLcLen = pCmd[PHHAL_HW_SAM_ISO7816_LC_POS];
        *pIsLcPresent = PH_ON;
        *pIsLePresent = PH_OFF;
    }

    /* 4. CLA INS P1 P2 LC DATA LE */
    else if(wCmdLen == (PHHAL_HW_SAM_ISO7816_HEADER_LE_LENGTH + pCmd[PHHAL_HW_SAM_ISO7816_LC_POS]))
    {
        *pLcLen = pCmd[PHHAL_HW_SAM_ISO7816_LC_POS];
        *pIsLcPresent = PH_ON;
        *pIsLePresent = PH_ON;
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_FRAMING_ERROR, PH_COMP_HAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Utils_TruncateMacBuffer(uint8_t * pIoBuffer, uint8_t * pMacLen)
{
    uint8_t PH_MEMLOC_REM bCount = 0;
    uint8_t PH_MEMLOC_REM bTruncateCount = 0;

    for(bCount = 1U; bCount < (*pMacLen); bCount += 2U)
    {
        pIoBuffer[bTruncateCount++] = pIoBuffer[bCount];
    }

    *pMacLen = bTruncateCount;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Utils_GetLc(phhalHw_Sam_DataParams_t * pDataParams, uint8_t * pLC)
{
    *pLC = pDataParams->pTxBuffer[PHHAL_HW_SAM_ISO7816_LC_POS];

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Utils_UpdateLc(phhalHw_Sam_DataParams_t * pDataParams)
{
    uint8_t * PH_MEMLOC_REM pBuffer = NULL;

    /* Check for internal error */
    if(pDataParams->wTxBufLen_Cmd < (PHHAL_HW_SAM_ISO7816_LC_POS + 1U))
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }

    /* TxBuffer equals RxBuffer */
    if(pDataParams->pTxBuffer == pDataParams->pRxBuffer)
    {
        /* Start at TxLength if neccessary */
        if((pDataParams->wTxBufStartPos + pDataParams->wTxBufLen) >
            (pDataParams->wRxBufLen))
        {
            pBuffer = &pDataParams->pTxBuffer[pDataParams->wTxBufStartPos + pDataParams->wTxBufLen];
        }
        /* Start at RxLength if neccessary */
        else
        {
            pBuffer = &pDataParams->pTxBuffer[pDataParams->wRxBufLen];
        }
    }
    /* Buffers are different */
    else
    {
        pBuffer = &pDataParams->pTxBuffer[pDataParams->wTxBufLen];
    }

    /* Perform actual update */
    pBuffer[PHHAL_HW_SAM_ISO7816_LC_POS] = (uint8_t) (pDataParams->wTxBufLen_Cmd - PHHAL_HW_SAM_ISO7816_HEADER_LENGTH);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Utils_UpdateP1(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bP1)
{
    uint8_t * PH_MEMLOC_REM pBuffer = NULL;

    /* Check for internal error */
    if(pDataParams->wTxBufLen_Cmd < (PHHAL_HW_SAM_ISO7816_P1_POS + 1U))
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }

    /* TxBuffer equals RxBuffer */
    if(pDataParams->pTxBuffer == pDataParams->pRxBuffer)
    {
        /* Start at TxLength if neccessary */
        if((pDataParams->wTxBufStartPos + pDataParams->wTxBufLen) >
            (pDataParams->wRxBufLen))
        {
            pBuffer = &pDataParams->pTxBuffer[pDataParams->wTxBufStartPos + pDataParams->wTxBufLen];
        }
        /* Start at RxLength if neccessary */
        else
        {
            pBuffer = &pDataParams->pTxBuffer[pDataParams->wRxBufLen];
        }
    }
    /* Buffers are different */
    else
    {
        pBuffer = &pDataParams->pTxBuffer[pDataParams->wTxBufLen];
    }

    /* Perform actual update */
    pBuffer[PHHAL_HW_SAM_ISO7816_P1_POS] = bP1;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Utils_UpdateP2(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bP2)
{
    uint8_t * PH_MEMLOC_REM pBuffer = NULL;

    /* Check for internal error */
    if(pDataParams->wTxBufLen_Cmd < (PHHAL_HW_SAM_ISO7816_P2_POS + 1U))
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }

    /* TxBuffer equals RxBuffer */
    if(pDataParams->pTxBuffer == pDataParams->pRxBuffer)
    {
        /* Start at TxLength if neccessary */
        if((pDataParams->wTxBufStartPos + pDataParams->wTxBufLen) >
            (pDataParams->wRxBufLen))
        {
            pBuffer = &pDataParams->pTxBuffer[pDataParams->wTxBufStartPos + pDataParams->wTxBufLen];
        }
        /* Start at RxLength if neccessary */
        else
        {
            pBuffer = &pDataParams->pTxBuffer[pDataParams->wRxBufLen];
        }
    }
    /* Buffers are different */
    else
    {
        pBuffer = &pDataParams->pTxBuffer[pDataParams->wTxBufLen];
    }

    /* Perform actual update */
    pBuffer[PHHAL_HW_SAM_ISO7816_P2_POS] = bP2;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Utils_ResolveErrorCode(phhalHw_Sam_DataParams_t * pDataParams, uint8_t * pSw1Sw2)
{
    phStatus_t	PH_MEMLOC_REM wStatus = 0;

    pDataParams->wErrorCode = (phStatus_t) ((pSw1Sw2[0] << 8U) | pSw1Sw2[1U]);
    switch(pDataParams->wErrorCode)
    {
        case PHHAL_HW_SAM_RET_CODE_OK:
            wStatus = PH_ERR_SUCCESS;
            break;

        case PHHAL_HW_SAM_RET_CODE_OK_1BIT:
        case PHHAL_HW_SAM_RET_CODE_OK_2BIT:
        case PHHAL_HW_SAM_RET_CODE_OK_3BIT:
        case PHHAL_HW_SAM_RET_CODE_OK_4BIT:
        case PHHAL_HW_SAM_RET_CODE_OK_5BIT:
        case PHHAL_HW_SAM_RET_CODE_OK_6BIT:
        case PHHAL_HW_SAM_RET_CODE_OK_7BIT:
            wStatus = PH_ERR_SUCCESS_INCOMPLETE_BYTE;
            break;

        case PHHAL_HW_SAM_RET_CODE_COMM_IO_TIMEOUT:
            wStatus = PH_ERR_IO_TIMEOUT;
            break;

        case PHHAL_HW_SAM_RET_CODE_COMM_ISO14443_PROTOCOL_EEROR:
            wStatus = PH_ERR_PROTOCOL_ERROR;
            break;

        case PHHAL_HW_SAM_RET_CODE_COMM_PARITY_ERROR:
        case PHHAL_HW_SAM_RET_CODE_COMM_CRC_FAILURE:
            wStatus = PH_ERR_INTEGRITY_ERROR;
            break;

        case PHHAL_HW_SAM_RET_CODE_COMM_BUFFER_OVERFLOW:
        case PHHAL_HW_SAM_RET_CODE_COMM_INTERNAL_BUF_OVERFLOW:
            wStatus = PH_ERR_BUFFER_OVERFLOW;
            break;

        case PHHAL_HW_SAM_RET_CODE_COMM_RF_FAILURE:
            wStatus = PH_ERR_RF_ERROR;
            break;

        case PHHAL_HW_SAM_RET_CODE_COMM_TEMP_FAILURE:
            wStatus = PH_ERR_TEMPERATURE_ERROR;
            break;

        case PHHAL_HW_SAM_RET_CODE_COMM_FIFO_WRITE:
            wStatus = PH_ERR_READ_WRITE_ERROR;
            break;

        case PHHAL_HW_SAM_RET_CODE_COMM_COLLISION:
            wStatus = PH_ERR_COLLISION_ERROR;
            break;

        case  PHHAL_HW_SAM_RET_CODE_HW_EEPROM:
            wStatus = PHHAL_HW_SAM_ERR_HW_EEPROM;
            break;

        case PHHAL_HW_SAM_RET_CODE_HW_RC5XX:
            wStatus = PHHAL_HW_SAM_ERR_HW_RC5XX;
            break;

        case PHHAL_HW_SAM_RET_CODE_KEY_CREATE_FAILED:
            wStatus = PHHAL_HW_SAM_ERR_KEY_CREATE_FAILED;
            break;

        case PHHAL_HW_SAM_RET_CODE_KEY_REF_NO_INVALID:
            wStatus = PHHAL_HW_SAM_ERR_KEY_REF_NO_INVALID;
            break;

        case PHHAL_HW_SAM_RET_CODE_KEY_KUC_NO_INVALID:
            wStatus = PHHAL_HW_SAM_ERR_KEY_KUC_NO_INVALID;
            break;

        case PHHAL_HW_SAM_RET_CODE_HW_EE_HIGH_VOLTAGE:
            wStatus = PHHAL_HW_SAM_ERR_HW_EE_HIGH_VOLTAGE;
            break;

        case PHHAL_HW_SAM_RET_CODE_ISO7816_WRONG_LENGTH_LC:
            wStatus = PHHAL_HW_SAM_ERR_ISO7816_WRONG_LENGTH_LC;
            break;

        case PHHAL_HW_SAM_RET_CODE_INCOMPLETE_CHAINING:
            wStatus = PHHAL_HW_SAM_ERR_INCOMPLETE_CHAINING;
            break;

        case PHHAL_HW_SAM_RET_CODE_UNMATCHING_PCD_PD:
            wStatus = PHHAL_HW_SAM_ERR_UNMATCHING_PCD_PD;
            break;

        case PHHAL_HW_SAM_RET_CODE_KEY_INTEGRITY_ERROR:
            wStatus = PHHAL_HW_SAM_ERR_KEY_INTEGRITY_ERROR;
            break;

        case PHHAL_HW_SAM_RET_CODE_COND_USE_NOT_SATISFIED:
            wStatus = PHHAL_HW_SAM_ERR_COND_USE_NOT_SATISFIED;
            break;

        case PHHAL_HW_SAM_RET_CODE_ISO7816_COMMAND_NOT_ALLOWED:
            wStatus = PHHAL_HW_SAM_ERR_ISO7816_COMMAND_NOT_ALLOWED;
            break;

        case PHHAL_HW_SAM_RET_CODE_ISO7816_WRONG_PARAMS_FOR_INS:
            wStatus = PHHAL_HW_SAM_ERR_ISO7816_WRONG_PARAMS_FOR_INS;
            break;

        case PHHAL_HW_SAM_RET_CODE_KEY_VERSION_INVALID:
            wStatus = PHHAL_HW_SAM_ERR_KEY_VERSION_INVALID;
            break;

        case PHHAL_HW_SAM_RET_CODE_HOST_PROTECTION_ERROR:
            wStatus = PHHAL_HW_SAM_ERR_HOST_PROTECTION;
            break;

        case PHHAL_HW_SAM_RET_CODE_ISO7816_WRONG_P1P2:
            wStatus = PHHAL_HW_SAM_ERR_ISO7816_WRONG_P1P2;
            break;

        case PHHAL_HW_SAM_RET_CODE_ISO7816_WRONG_LE:
            wStatus = PHHAL_HW_SAM_ERR_ISO7816_WRONG_LE;
            break;

        case PHHAL_HW_SAM_RET_CODE_ISO7816_UNKNOWN_INS:
            wStatus = PHHAL_HW_SAM_ERR_ISO7816_UNKNOWN_INS;
            break;

        case PHHAL_HW_SAM_RET_CODE_ISO7816_WRONG_CLASS:
            wStatus = PHHAL_HW_SAM_ERR_ISO7816_UNKNOWN_CLASS;
            break;

        case PHHAL_HW_SAM_RET_CODE_CRYPTO_FAILURE:
            wStatus = PHHAL_HW_SAM_ERR_CRYPTO;
            break;

        case PHHAL_HW_SAM_RET_CODE_MIFARE_PLUS_INCORRECT_DATA:
            wStatus = PHHAL_HW_SAM_ERR_MIFARE_PLUS_INCORRECT_DATA;
            break;

        case PHHAL_HW_SAM_RET_CODE_MIFARE_PLUS_ERROR:
            wStatus = PHHAL_HW_SAM_ERR_MIFARE_PLUS_GEN;
            break;

        case PHHAL_HW_SAM_RET_CODE_ISO_UID_INCOMPLETE:
            wStatus = PHHAL_HW_SAM_ERR_ISO_UID_INCOMPLETE;
            break;

        case PHHAL_HW_SAM_RET_CODE_DESFIRE_ERROR:
            wStatus = PHHAL_HW_SAM_ERR_DESFIRE_GEN;
            break;

        case PHHAL_HW_SAM_RET_CODE_ISO_WRONG_BNR:
            wStatus = PHHAL_HW_SAM_ERR_ISO_WRONG_BNR;
            break;

        case PHHAL_HW_SAM_RET_CODE_ISO_INVALID_FORMAT:
            wStatus = PHHAL_HW_SAM_ERR_ISO_INVALID_FORMAT;
            break;

        case PHHAL_HW_SAM_RET_CODE_ISO_INVALID_PARAMETER:
            wStatus = PHHAL_HW_SAM_ERR_ISO_INVALID_PARAMETER;
            break;

        case PHHAL_HW_SAM_RET_CODE_PROT_MIFARE_ERROR:
            wStatus = PHHAL_HW_SAM_ERR_MIFARE_GEN;
            break;

        case PHHAL_HW_SAM_RET_CODE_OK_CHAINING_ACTIVE:
            wStatus = PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE;
            break;

        case PHHAL_HW_SAM_RET_CODE_COMMAND_CHAINING_NOT_SUPPORTED:
            wStatus = PHHAL_HW_SAM_ERR_COMMAND_CHAINING_NOT_SUPPORTED;
            break;

        case PHHAL_HW_SAM_RET_CODE_UNKNOWN_PUBLIC_KEY_INDEX:
            wStatus = PHHAL_HW_SAM_ERR_UNKNOWN_PUBLIC_KEY_INDEX;
            break;

        case PHHAL_HW_SAM_RET_CODE_NO_PRECISE_DIAGNOSIS:
            wStatus = PHHAL_HW_SAM_ERR_NO_PRECISE_DIAGNOSIS;
            break;

        case PHHAL_HW_SAM_RET_CODE_OK_CHAINING_ACTIVE_DUOX:
            wStatus = PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE_DUOX;
            break;

        case PHHAL_HW_SAM_RET_CODE_OK_CHAINING_ACTIVE_EXT:
            wStatus = PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE_EXT;
            break;

        case PHHAL_HW_SAM_RET_CODE_UID_BCC_INCORRECT:
            wStatus = PHHAL_HW_SAM_ERR_ISO_UID_BCC_INCORRECT;
            break;

        case PHHAL_HW_SAM_RET_CODE_PROT_MIFARE_NAK0:
            wStatus = PHHAL_HW_SAM_ERR_MIFARE_NAK0;
            break;

        case PHHAL_HW_SAM_RET_CODE_PROT_MIFARE_NAK1:
            wStatus = PHHAL_HW_SAM_ERR_MIFARE_NAK1;
            break;

        case PHHAL_HW_SAM_RET_CODE_PROT_MIFARE_NAK4:
            wStatus = PHHAL_HW_SAM_ERR_MIFARE_NAK4;
            break;

        case PHHAL_HW_SAM_RET_CODE_PROT_MIFARE_NAK5:
            wStatus = PHHAL_HW_SAM_ERR_MIFARE_NAK5;
            break;

        case PHHAL_HW_SAM_RET_CODE_PROT_MIFARE_NAK6:
            wStatus = PHHAL_HW_SAM_ERR_MIFARE_NAK6;
            break;

        case PHHAL_HW_SAM_RET_CODE_PROT_MIFARE_NAK7:
            wStatus = PHHAL_HW_SAM_ERR_MIFARE_NAK7;
            break;

        default:
            wStatus = PHHAL_HW_SAM_ERR_PROGRAMMABLE_LOGIC;
            break;
    }

    return PH_ADD_COMPCODE(wStatus, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Utils_GenerateHostAuthSessionKey(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bKeyType,
    uint8_t * pRnd1, uint8_t * pRnd2, uint8_t * pSessionKey, uint8_t * pKeyLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aSv1a[PH_CRYPTOSYM_AES_BLOCK_SIZE];
    uint8_t     PH_MEMLOC_REM aSv1b[PH_CRYPTOSYM_AES_BLOCK_SIZE];
    uint8_t     PH_MEMLOC_REM aKxea[PH_CRYPTOSYM_AES_BLOCK_SIZE];
    uint8_t     PH_MEMLOC_REM aKxeb[PH_CRYPTOSYM_AES_BLOCK_SIZE];
    uint8_t     PH_MEMLOC_COUNT bCount = 0;

    /* Frame the session vector A. */
    memcpy(&aSv1a[0], &pRnd1[7U], 5U);
    memcpy(&aSv1a[5U], &pRnd2[7U], 5U);
    aSv1a[10U] = pRnd1[0] ^ pRnd2[0];
    aSv1a[11U] = pRnd1[1U] ^ pRnd2[1U];
    aSv1a[12U] = pRnd1[2U] ^ pRnd2[2U];
    aSv1a[13U] = pRnd1[3U] ^ pRnd2[3U];
    aSv1a[14U] = pRnd1[4U] ^ pRnd2[4U];

    /* Frame the session vector B. */
    memcpy(&aSv1b[0], &pRnd1[6U], 5U);
    memcpy(&aSv1b[5U], &pRnd2[6U], 5U);
    aSv1b[10U] = pRnd1[1U] ^ pRnd2[1U];
    aSv1b[11U] = pRnd1[2U] ^ pRnd2[2U];
    aSv1b[12U] = pRnd1[3U] ^ pRnd2[3U];
    aSv1b[13U] = pRnd1[4U] ^ pRnd2[4U];
    aSv1b[14U] = pRnd1[5U] ^ pRnd2[5U];

    /* Load initial IV. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pMACCryptoDataParams,
        phhalHw_Sam_HcUtils_ZeroIv,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    /* Add the constant according to keytype and calculate the session key. */
    switch(bKeyType)
    {
        case PH_CRYPTOSYM_KEY_TYPE_AES128:
            aSv1a[15U] = 0x91U;

            /* Calculate the session key using session vector A. */
            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(
                pDataParams->pMACCryptoDataParams,
                PH_CRYPTOSYM_CIPHER_MODE_CBC,
                aSv1a,
                PH_CRYPTOSYM_AES_BLOCK_SIZE,
                pSessionKey));

            /* Update the Session key length. */
            *pKeyLen = PH_CRYPTOSYM_AES128_KEY_SIZE;
            break;

        case PH_CRYPTOSYM_KEY_TYPE_AES192:
            aSv1a[15U] = 0x93U;
            aSv1b[15U] = 0x94U;

            /* Calculate the session key using session vector A. */
            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(
                pDataParams->pMACCryptoDataParams,
                PH_CRYPTOSYM_CIPHER_MODE_CBC,
                aSv1a,
                PH_CRYPTOSYM_AES_BLOCK_SIZE,
                aKxea));

            /* Calculate the session key using session vector B. */
            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(
                pDataParams->pMACCryptoDataParams,
                PH_CRYPTOSYM_CIPHER_MODE_CBC,
                aSv1b,
                PH_CRYPTOSYM_AES_BLOCK_SIZE,
                aKxeb));

            /* Copy calculated Session Key A to Session key parameter. */
            memcpy(pSessionKey, aKxea, PH_CRYPTOSYM_AES_BLOCK_SIZE);

            /* XOR calculated Session Key A with calculated Session Key B. */
            for(bCount = 0; bCount <= 7U; ++bCount)
                pSessionKey[bCount + 8U] ^= aKxeb[bCount];

            /* Copy remaining Session Key B to Session key parameter. */
            memcpy(&pSessionKey[16U], &aKxeb[8U], 8U);

            /* Update the Session key length. */
            *pKeyLen = PH_CRYPTOSYM_AES192_KEY_SIZE;
            break;

        case PH_CRYPTOSYM_KEY_TYPE_AES256:
            aSv1a[15U] = 0x95U;
            aSv1b[15U] = 0x96U;

            /* Calculate the session key using session vector A. */
            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(
                pDataParams->pMACCryptoDataParams,
                PH_CRYPTOSYM_CIPHER_MODE_CBC,
                aSv1a,
                PH_CRYPTOSYM_AES_BLOCK_SIZE,
                aKxea));

            /* Calculate the session key using session vector B. */
            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(
                pDataParams->pMACCryptoDataParams,
                PH_CRYPTOSYM_CIPHER_MODE_CBC,
                aSv1b,
                PH_CRYPTOSYM_AES_BLOCK_SIZE,
                aKxeb));

            /* Compute the session key. */
            /* Kxe = Kxea || Kxeb */
            memcpy(&pSessionKey[0], aKxea, 16U);
            memcpy(&pSessionKey[16U], aKxeb, 16U);
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Utils_GenerateSessionKey(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bKeyType,
    uint8_t * pRnd1, uint8_t * pRnd2, uint8_t * pSessionEncKey, uint8_t * pSessionMacKey, uint8_t * pKeyLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bCount = 0;
    uint8_t     PH_MEMLOC_REM aSv1a[PH_KEYSTORE_KEY_TYPE_AES128_SIZE];
    uint8_t     PH_MEMLOC_REM aSv1b[PH_KEYSTORE_KEY_TYPE_AES128_SIZE];
    uint8_t     PH_MEMLOC_REM aSv2a[PH_KEYSTORE_KEY_TYPE_AES128_SIZE];
    uint8_t     PH_MEMLOC_REM aSv2b[PH_KEYSTORE_KEY_TYPE_AES128_SIZE];
    uint8_t     PH_MEMLOC_REM aSessionEncKeyA[PH_KEYSTORE_KEY_TYPE_AES128_SIZE];
    uint8_t     PH_MEMLOC_REM aSessionEncKeyB[PH_KEYSTORE_KEY_TYPE_AES128_SIZE];
    uint8_t     PH_MEMLOC_REM aSessionMacKeyA[PH_KEYSTORE_KEY_TYPE_AES128_SIZE];
    uint8_t     PH_MEMLOC_REM aSessionMacKeyB[PH_KEYSTORE_KEY_TYPE_AES128_SIZE];

    /* Frame the session vector 1A. */
    memcpy(&aSv1a[0], &pRnd1[11U], 5U);
    memcpy(&aSv1a[5U], &pRnd2[11U], 5U);
    aSv1a[10U] = pRnd1[4U] ^ pRnd2[4U];
    aSv1a[11U] = pRnd1[5U] ^ pRnd2[5U];
    aSv1a[12U] = pRnd1[6U] ^ pRnd2[6U];
    aSv1a[13U] = pRnd1[7U] ^ pRnd2[7U];
    aSv1a[14U] = pRnd1[8U] ^ pRnd2[8U];

    /* Frame the session vector 2A. */
    memcpy(&aSv2a[0], &pRnd1[7U], 5U);
    memcpy(&aSv2a[5U], &pRnd2[7U], 5U);
    aSv2a[10U] = pRnd1[0] ^ pRnd2[0];
    aSv2a[11U] = pRnd1[1U] ^ pRnd2[1U];
    aSv2a[12U] = pRnd1[2U] ^ pRnd2[2U];
    aSv2a[13U] = pRnd1[3U] ^ pRnd2[3U];
    aSv2a[14U] = pRnd1[4U] ^ pRnd2[4U];

    /* Frame the session vector 1B. */
    memcpy(&aSv1b[0], &pRnd1[10U], 5U);
    memcpy(&aSv1b[5U], &pRnd2[10U], 5U);
    aSv1b[10U] = pRnd1[5U] ^ pRnd2[5U];
    aSv1b[11U] = pRnd1[6U] ^ pRnd2[6U];
    aSv1b[12U] = pRnd1[7U] ^ pRnd2[7U];
    aSv1b[13U] = pRnd1[8U] ^ pRnd2[8U];
    aSv1b[14U] = pRnd1[9U] ^ pRnd2[9U];

    /* Frame the session vector 2B. */
    memcpy(&aSv2b[0], &pRnd1[6U], 5U);
    memcpy(&aSv2b[5U], &pRnd2[6U], 5U);
    aSv2b[10U] = pRnd1[1U] ^ pRnd2[1U];
    aSv2b[11U] = pRnd1[2U] ^ pRnd2[2U];
    aSv2b[12U] = pRnd1[3U] ^ pRnd2[3U];
    aSv2b[13U] = pRnd1[4U] ^ pRnd2[4U];
    aSv2b[14U] = pRnd1[5U] ^ pRnd2[5U];

    /* Load initial IV. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pMACCryptoDataParams,
        phhalHw_Sam_HcUtils_ZeroIv,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    /* Add the constant according to keytype and calculate the session key. */
    switch(bKeyType)
    {
        case PH_KEYSTORE_KEY_TYPE_AES128:
            aSv1a[15U] = 0x81U;
            aSv2a[15U] = 0x82U;

            /* Calculate the session encryption key using session vector 1A. */
            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(
                pDataParams->pMACCryptoDataParams,
                PH_CRYPTOSYM_CIPHER_MODE_CBC,
                aSv1a,
                PH_CRYPTOSYM_AES_BLOCK_SIZE,
                pSessionEncKey));

            /* Calculate the session mac key using session vector 2A. */
            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(
                pDataParams->pMACCryptoDataParams,
                PH_CRYPTOSYM_CIPHER_MODE_CBC,
                aSv2a,
                PH_CRYPTOSYM_AES_BLOCK_SIZE,
                pSessionMacKey));

            /* Update the Session key length. */
            *pKeyLen = PH_CRYPTOSYM_AES128_KEY_SIZE;
            break;

        case PH_KEYSTORE_KEY_TYPE_AES192:
            aSv1a[15U] = 0x83U;
            aSv1b[15U] = 0x84U;
            aSv2a[15U] = 0x85U;
            aSv2b[15U] = 0x86U;

            /* Calculate the session encryption key A using session vector 1A. */
            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(
                pDataParams->pMACCryptoDataParams,
                PH_CRYPTOSYM_CIPHER_MODE_CBC,
                aSv1a,
                PH_CRYPTOSYM_AES_BLOCK_SIZE,
                aSessionEncKeyA));

            /* Calculate the session encryption key B using session vector 1B. */
            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(
                pDataParams->pMACCryptoDataParams,
                PH_CRYPTOSYM_CIPHER_MODE_CBC,
                aSv1b,
                PH_CRYPTOSYM_AES_BLOCK_SIZE,
                aSessionEncKeyB));

            /* Calculate the session mac key A using session vector 2A. */
            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(
                pDataParams->pMACCryptoDataParams,
                PH_CRYPTOSYM_CIPHER_MODE_CBC,
                aSv2a,
                PH_CRYPTOSYM_AES_BLOCK_SIZE,
                aSessionMacKeyA));

            /* Calculate the session mac key B using session vector 2B. */
            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(
                pDataParams->pMACCryptoDataParams,
                PH_CRYPTOSYM_CIPHER_MODE_CBC,
                aSv2b,
                PH_CRYPTOSYM_AES_BLOCK_SIZE,
                aSessionMacKeyB));

            /* Copy calculated Enc Session Key A to Enc Session key parameter. */
            memcpy(pSessionEncKey, aSessionEncKeyA, 8U);

            /* XOR calculated Enc Session Key A with calculated Enc Session Key B. */
            for(bCount = 0; bCount <= 7U; ++bCount)
                pSessionEncKey[bCount + 8U] = aSessionEncKeyA[bCount + 8U] ^ aSessionEncKeyB[bCount];

            /* Copy remaining Enc Session Key B to Session key parameter. */
            memcpy(&pSessionEncKey[16U], &aSessionEncKeyB[8U], 8U);

            /* Copy calculated Mac Session Key A to Mac Session key parameter. */
            memcpy(pSessionMacKey, aSessionMacKeyA, 8U);

            /* XOR calculated Mac Session Key A with calculated Mac Session Key B. */
            for(bCount = 0; bCount <= 7U; ++bCount)
                pSessionMacKey[bCount + 8U] = aSessionMacKeyA[bCount + 8U] ^ aSessionMacKeyB[bCount];

            /* Copy remaining Mac Session Key B to Session key parameter. */
            memcpy(&pSessionMacKey[16U], &aSessionMacKeyB[8U], 8U);

            /* Update the Session key length. */
            *pKeyLen = PH_CRYPTOSYM_AES192_KEY_SIZE;
            break;

        case PH_CRYPTOSYM_KEY_TYPE_AES256:
            aSv1a[15U] = 0x87U;
            aSv1b[15U] = 0x88U;
            aSv2a[15U] = 0x89U;
            aSv2b[15U] = 0x8AU;

            /* Calculate the session encryption key A using session vector 1A. */
            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(
                pDataParams->pMACCryptoDataParams,
                PH_CRYPTOSYM_CIPHER_MODE_CBC,
                aSv1a,
                PH_CRYPTOSYM_AES_BLOCK_SIZE,
                aSessionEncKeyA));

            /* Calculate the session encryption key B using session vector 1B. */
            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(
                pDataParams->pMACCryptoDataParams,
                PH_CRYPTOSYM_CIPHER_MODE_CBC,
                aSv1b,
                PH_CRYPTOSYM_AES_BLOCK_SIZE,
                aSessionEncKeyB));

            /* Calculate the session mac key A using session vector 2A. */
            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(
                pDataParams->pMACCryptoDataParams,
                PH_CRYPTOSYM_CIPHER_MODE_CBC,
                aSv2a,
                PH_CRYPTOSYM_AES_BLOCK_SIZE,
                aSessionMacKeyA));

            /* Calculate the session mac key B using session vector 2B. */
            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(
                pDataParams->pMACCryptoDataParams,
                PH_CRYPTOSYM_CIPHER_MODE_CBC,
                aSv2b,
                PH_CRYPTOSYM_AES_BLOCK_SIZE,
                aSessionMacKeyB));

            /* Compute Session ENC key. */
            /* Ke = Kea || Keb */
            memcpy(&pSessionEncKey[0], aSessionEncKeyA, 16U);
            memcpy(&pSessionEncKey[16U], aSessionEncKeyB, 16U);

            /* Compute Session MAC key. */
            /* Km = Kma || Kmb */
            memcpy(&pSessionMacKey[0], aSessionMacKeyA, 16U);
            memcpy(&pSessionMacKey[16U], aSessionMacKeyB, 16U);
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_Sam_Utils_GetSessionUploadKey(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bMode,
    uint16_t wUploadCtr, uint8_t bKeyNo, uint8_t bKeyVer, uint8_t * pSessionKey, uint8_t * pKeyType)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    uint16_t    PH_MEMLOC_REM wKeyType = 0;
    uint8_t     PH_MEMLOC_REM bKeyLen = 0;
    uint8_t     PH_MEMLOC_REM aDefault_Iv[PH_CRYPTOSYM_AES128_KEY_SIZE];
    uint8_t     PH_MEMLOC_REM aSva[PH_CRYPTOSYM_AES128_KEY_SIZE];
    uint8_t     PH_MEMLOC_REM aSvb[PH_CRYPTOSYM_AES128_KEY_SIZE];
    uint8_t     PH_MEMLOC_REM aKey[PH_CRYPTOSYM_AES256_KEY_SIZE];
    uint8_t     PH_MEMLOC_REM aSessionKeyA[PH_CRYPTOSYM_AES128_KEY_SIZE];
    uint8_t     PH_MEMLOC_REM aSessionKeyB[PH_CRYPTOSYM_AES128_KEY_SIZE];
    uint8_t     PH_MEMLOC_REM aSessionKey[PH_CRYPTOSYM_AES256_KEY_SIZE];

    /* Get the Key and Key Type from keystore. */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_GetKey(pDataParams->pKeyStoreDataParams,
        bKeyNo,
        bKeyVer,
        sizeof(aKey),
        aKey,
        &wKeyType));

    /* Update the key type to the parameter. */
    *pKeyType = (uint8_t) wKeyType;

    /* Load the key to CryptoSym for macing the session vector. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
        pDataParams->pPLUpload_MACCryptoDataParams,
        aKey,
        wKeyType));

    /* Load zero IV is required to Crypto data params. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pPLUpload_MACCryptoDataParams,
        aDefault_Iv,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    /* Update the session vector array with the required values. */
    switch(wKeyType)
    {
        case PH_KEYSTORE_KEY_TYPE_AES128:
            memset(aSva, (uint8_t) ((bMode == PHHAL_HW_CMD_SAM_SESSION_KEY_ENC) ? 0x71 : 0x72), sizeof(aSva));

            /* Update the session vector with upload counter value. */
            aSva[0] = (uint8_t) PHHAL_HW_SAM_USHORT_MASK_MSB(wUploadCtr);
            aSva[1U] = (uint8_t) PHHAL_HW_SAM_USHORT_MASK_LSB(wUploadCtr);

            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
                pDataParams->pPLUpload_MACCryptoDataParams,
                (PH_CRYPTOSYM_MAC_MODE_CBCMAC | PH_EXCHANGE_DEFAULT),
                aSva,
                PH_CRYPTOSYM_AES_BLOCK_SIZE,
                pSessionKey,
                &bKeyLen));
            break;

        case PH_KEYSTORE_KEY_TYPE_AES192:
            memset(aSva, (uint8_t) ((bMode == PHHAL_HW_CMD_SAM_SESSION_KEY_ENC) ? 0x73 : 0x75), sizeof(aSva));
            memset(aSvb, (uint8_t) ((bMode == PHHAL_HW_CMD_SAM_SESSION_KEY_ENC) ? 0x74 : 0x76), sizeof(aSvb));

            /* Update the session vector with upload counter value. */
            aSva[0] = (uint8_t) PHHAL_HW_SAM_USHORT_MASK_MSB(wUploadCtr);
            aSva[1U] = (uint8_t) PHHAL_HW_SAM_USHORT_MASK_LSB(wUploadCtr);
            aSvb[0] = (uint8_t) PHHAL_HW_SAM_USHORT_MASK_MSB(wUploadCtr);
            aSvb[1U] = (uint8_t) PHHAL_HW_SAM_USHORT_MASK_LSB(wUploadCtr);

            /* Load the session vector value to CryptoMAC data params. */
            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
                pDataParams->pPLUpload_MACCryptoDataParams,
                (PH_CRYPTOSYM_MAC_MODE_CBCMAC | PH_EXCHANGE_DEFAULT),
                aSva,
                PH_CRYPTOSYM_AES_BLOCK_SIZE,
                aSessionKeyA,
                &bKeyLen));

            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
                pDataParams->pPLUpload_MACCryptoDataParams,
                (PH_CRYPTOSYM_MAC_MODE_CBCMAC | PH_EXCHANGE_DEFAULT),
                aSvb,
                PH_CRYPTOSYM_AES_BLOCK_SIZE,
                aSessionKeyB,
                &bKeyLen));

            /* Compute the session key. */
            /* SessionKey = SessionKeyA[15 : 8]	|| (SessionA[7 : 0] ^ SessionB[15 : 8]) || SessionKeyB[7:0] */
            aSessionKey[0x00] = aSessionKeyA[0x00];
            aSessionKey[0x01U] = aSessionKeyA[0x01U];
            aSessionKey[0x02U] = aSessionKeyA[0x02U];
            aSessionKey[0x03U] = aSessionKeyA[0x03U];
            aSessionKey[0x04U] = aSessionKeyA[0x04U];
            aSessionKey[0x05U] = aSessionKeyA[0x05U];
            aSessionKey[0x06U] = aSessionKeyA[0x06U];
            aSessionKey[0x07U] = aSessionKeyA[0x07U];

            aSessionKey[0x08U] = (uint8_t) (aSessionKeyA[0x08U] ^ aSessionKeyB[0x00]);
            aSessionKey[0x09U] = (uint8_t) (aSessionKeyA[0x09U] ^ aSessionKeyB[0x01U]);
            aSessionKey[0x0AU] = (uint8_t) (aSessionKeyA[0x0AU] ^ aSessionKeyB[0x02U]);
            aSessionKey[0x0BU] = (uint8_t) (aSessionKeyA[0x0BU] ^ aSessionKeyB[0x03U]);
            aSessionKey[0x0CU] = (uint8_t) (aSessionKeyA[0x0CU] ^ aSessionKeyB[0x04U]);
            aSessionKey[0x0DU] = (uint8_t) (aSessionKeyA[0x0DU] ^ aSessionKeyB[0x05U]);
            aSessionKey[0x0EU] = (uint8_t) (aSessionKeyA[0x0EU] ^ aSessionKeyB[0x06U]);
            aSessionKey[0x0FU] = (uint8_t) (aSessionKeyA[0x0FU] ^ aSessionKeyB[0x07U]);

            aSessionKey[0x10U] = aSessionKeyB[0x08U];
            aSessionKey[0x11U] = aSessionKeyB[0x09U];
            aSessionKey[0x12U] = aSessionKeyB[0x0AU];
            aSessionKey[0x13U] = aSessionKeyB[0x0BU];
            aSessionKey[0x14U] = aSessionKeyB[0x0CU];
            aSessionKey[0x15U] = aSessionKeyB[0x0DU];
            aSessionKey[0x16U] = aSessionKeyB[0x0EU];
            aSessionKey[0x17U] = aSessionKeyB[0x0FU];

            /* Update the pSessionKey parameter with the computed session key data. */
            memcpy(pSessionKey, aSessionKey, PH_CRYPTOSYM_AES192_KEY_SIZE);
            break;

        case PH_KEYSTORE_KEY_TYPE_AES256:
            memset(aSva, (uint8_t) ((bMode == PHHAL_HW_CMD_SAM_SESSION_KEY_ENC) ? 0x77U : 0x79U), sizeof(aSva));
            memset(aSvb, (uint8_t) ((bMode == PHHAL_HW_CMD_SAM_SESSION_KEY_ENC) ? 0x78U : 0x7AU), sizeof(aSvb));

            /* Update the session vector with upload counter value. */
            aSva[0] = (uint8_t) PHHAL_HW_SAM_USHORT_MASK_MSB(wUploadCtr);
            aSva[1U] = (uint8_t) PHHAL_HW_SAM_USHORT_MASK_LSB(wUploadCtr);

            aSvb[0] = (uint8_t) PHHAL_HW_SAM_USHORT_MASK_MSB(wUploadCtr);
            aSvb[1U] = (uint8_t) PHHAL_HW_SAM_USHORT_MASK_LSB(wUploadCtr);

            /* Load the session vector value to CryptoMAC data params. */
            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
                pDataParams->pPLUpload_MACCryptoDataParams,
                (PH_CRYPTOSYM_MAC_MODE_CBCMAC | PH_EXCHANGE_DEFAULT),
                aSva,
                PH_CRYPTOSYM_AES_BLOCK_SIZE,
                aSessionKeyA,
                &bKeyLen));

            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
                pDataParams->pPLUpload_MACCryptoDataParams,
                (PH_CRYPTOSYM_MAC_MODE_CBCMAC | PH_EXCHANGE_DEFAULT),
                aSvb,
                PH_CRYPTOSYM_AES_BLOCK_SIZE,
                aSessionKeyB,
                &bKeyLen));

            /* Compute the session key. */
            /* SessionKey = SessionKeyA || SessionKeyB */
            memcpy(&pSessionKey[0], aSessionKeyA, 16U);
            memcpy(&pSessionKey[16U], aSessionKeyB, 16U);
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_KEY, PH_COMP_HAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}
#endif /* NXPBUILD__PHHAL_HW_SAM */
