/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) Specific implementation for RC523 Reader of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHHALHW_SAM_RC_523_H
#define PHHALHW_SAM_RC_523_H

#include <ph_Status.h>

/** \addtogroup phhalHw_Sam_Utils
 * @{
 */

/** \defgroup phhalHw_Sam_Utils_RC523 RC523 Reader IC
 *  Implementation of helper interfaces for RC663 Reader IC
 * @{
 */

#define PHHAL_HW_SAM_RC523_TIMER_FREQ                                           13.56f  /**< RC internal timer frequency */
#define PHHAL_HW_SAM_RC523_ETU_106                                              9.44f   /**< Duration of one ETU at 106 kBit/s in [us] */

#define PHHAL_HW_SAM_RC523_TIMER_MAX_VALUE_US                                   6596996 /**< (RC523) Maximum timer value in microseconds. */
#define PHHAL_HW_SAM_RC523_TIMER_SHIFT                                          4.8f    /**< (RC523) Shift of the internal RC timer in ETUs. */

/**
 * \brief Set configuration parameter.
 *
 * <b> Rc523 Specific </b>
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_UNSUPPORTED_PARAMETER Configuration is not supported or invalid.
 * \retval #PH_ERR_INVALID_PARAMETER Parameter value is invalid.
 * \retval #PH_ERR_PARAMETER_OVERFLOW Setting the parameter value would lead to an overflow.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 */
phStatus_t phhalHw_Sam_Rc523_SetConfig(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wConfig,                                                               /**< [In] Configuration Identifier. */
        uint16_t wValue                                                                 /**< [In] Configuration Value. */
    );

/**
 * \brief Blocks until the time given by \b wTimeout elapsed.
 *
 * <b> Rc523 Specific </b>
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_INVALID_PARAMETER \b bUnit is invalid.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 */
phStatus_t phhalHw_Sam_Rc523_Wait(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bUnit,                                                                  /**< [In] Unit of given timeout value (either #PHHAL_HW_TIME_MICROSECONDS or #PHHAL_HW_TIME_MILLISECONDS). */
        uint16_t wTimeout                                                               /**< [In] Timeout value. */
    );

/**
 * \brief Sets the Frame Delay Time (Timeout).
 *
 * <b> Rc523 Specific </b>
 * \b Note: Frame Delay Time is defined between the last transmitted bit and the first received bit.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Rc523_SetFdt(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bUnit,                                                                  /**< [In] Unit of given timeout value (either #PHHAL_HW_TIME_MICROSECONDS or #PHHAL_HW_TIME_MILLISECONDS). */
        uint16_t wTimeout                                                               /**< [In] Timeout value. */
    );

/**
 * \brief Retrieves the Frame Delay Time of the last command.
 *
 * <b> Rc523 Specific </b>
 * \b Note: Frame Delay Time is defined between the last transmitted bit and the first received bit.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Rc523_GetFdt(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint32_t * pTime                                                                /**< [Out] Calculated time in microseconds from timer contents. */
    );

/**
 * end of group phhalHw_Sam_Utils_RC523
 * @}
 */

/**
 * end of group phhalHw_Sam_Utils
 * @}
 */

#endif /* PHHALHW_SAM_RC_523_H */
