/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) MIFARE Common command implementation of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */



#ifndef PHHALHW_SAM_CMD_COMMON_H
#define PHHALHW_SAM_CMD_COMMON_H

#include <ph_Status.h>
#include <phhalHw.h>

#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */

#ifdef NXPBUILD__PHHAL_HW_SAM

/** \addtogroup phhalHw_Sam_Cmd
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_Common 11_Common
 * \brief Implementation of SAM MIFARE Common commands.
 *  - Has implementation for both X and S mode communication.
 *  - These will be used by MIFARE Plus, DESFire and DUOX PICC's
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_Common_Defines Defines
 * \brief Definitions common for MIFARE Common S and X mode communication
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_Common_Defines_PICC PICC
 * \brief Definitions to identify the PICC for for the command is
 * exchagned to SAM. To be used with below mentioend interfaces.
 *  - \ref phhalHw_Sam_Cmd_SAM_CommitReaderID_Part1 "SAM_CommitReaderID Part1"
 *  - \ref phhalHw_Sam_Cmd_TMRI_CommitReaderID "TMRI_CommitReaderID"
 * @{
 */
#define PHHAL_HW_CMD_SAM_COMMIT_READER_ID_PICC_STATE_MFP                        0x00U   /**< Option to indicate the PICC state as
                                                                                         *   MIFARE Plus PICC.
                                                                                         */
#define PHHAL_HW_CMD_SAM_COMMIT_READER_ID_PICC_STATE_DESFIRE                    0x01U   /**< Option to indicate the PICC state as
                                                                                         *   MIFARE DESFire PICC.
                                                                                         */
/**
 * end of group phhalHw_Sam_Cmd_Common_Defines_PICC
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_Common_Defines
 * @}
 */





/** \defgroup phhalHw_Sam_Cmd_Common_SMode S_Mode
 * \brief Implementation of SAM commands used for MIFARE Common communication in S-Mode.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_Common_SMode_Defines Defines
 * \brief Definitions for SAM commands used for MIFARE Common communication in S-Mode.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_Common_SMode_Defines_CommandCodes CommandCodes
 * \brief Definitions for Instruction codes
 * @{
 */
#define PHHAL_HW_SAM_CMD_INS_SAM_COMMIT_READER_ID                                   0x36U   /**< SAM Insturction code for SAM_CommitReaderID command. */
/**
 * end of group phhalHw_Sam_Cmd_Common_SMode_Defines_CommandCodes
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_Common_SMode_Defines
 * @}
 */

/**
 * \brief Performs CommitReaderID command execution part 1 in S mode. The first part
 * includes the header, block number if its MifarePlus state or only header if its DESFire state.
 *
 * \note
 *  - Post calling this interface it's must to call \ref phhalHw_Sam_Cmd_SAM_CommitReaderID_Part2
 *    "SAM_CommitReaderID Part2" to place SAM into proper completion state if \ref PH_ERR_SUCCESS_CHAINING
 *    "Chaining Status" is returned from Library. Otherwise error will be returned by SAM for any
 *    subsequent commands.
 *  - If \ref PH_ERR_SUCCESS_CHAINING "Chaining Status" is not returned from Library, no need to call
 *    \ref phhalHw_Sam_Cmd_SAM_CommitReaderID_Part2 "SAM_CommitReaderID Part2" interface.
 *
 * \retval #PH_ERR_SUCCESS_CHAINING     Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pRespLen is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_CommitReaderID_Part1(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bState,                                                                 /**< [In] Options for framing the command. Below options
                                                                                         *        should be used.
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_COMMIT_READER_ID_PICC_STATE_MFP
                                                                                         *                 "MIFARE Plus"
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_COMMIT_READER_ID_PICC_STATE_DESFIRE
                                                                                         *                 "MIFARE DESFire"
                                                                                         */
        uint16_t wBlockNr,                                                              /**< [In] Number of the Transaction MAC Configuration Block. Will
                                                                                         *        be used if \b bState is
                                                                                         *        \ref PHHAL_HW_CMD_SAM_COMMIT_READER_ID_PICC_STATE_MFP
                                                                                         *        "MIFARE Plus".
                                                                                         */
        uint8_t ** ppResponse,                                                          /**< [Out] Transaction MAC Reader ID and MAC to be sent to MIFARE Plus
                                                                                         *         or MIFARE DESFire PICC.
                                                                                         *          - TMRI: Transaction MAC Reader ID (16 byte)
                                                                                         *              - UID: SAM UID (7 byte)
                                                                                         *              - Padding: 0x80 || zero padding
                                                                                         *          - MAC: 4 or 8 bytes MAC according to active Secure Messaging
                                                                                         */
        uint16_t * pRespLen                                                             /**< [Out] Length of bytes available in \b ppResponse buffer. */
    );

/**
 * \brief Performs CommitReaderID command execution part 2 in S mode. The last part includes
 * the header and the response received from Mfifare Plus or DESFire PICC.
 *
 * \note This interface should be called only if \ref phhalHw_Sam_Cmd_SAM_CommitReaderID_Part1
 * "SAM_CommitReaderID Part1" returns \ref PH_ERR_SUCCESS_CHAINING "Chaining Status".
 *
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pData and \b pPiccReturnCode are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_CommitReaderID_Part2(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bPiccErrCode,                                                           /**< [In] Error Code sent by MIFARE Plus PICC. For success
                                                                                         *        it should will
                                                                                         *          - 0x90: MIFARE Plus
                                                                                         *          - 0x00: MIFARE DESFire
                                                                                         */
        uint8_t * pData,                                                                /**< [In] The complete data (EncTMRI and MAC) received from
                                                                                         *        MIFARE Plus or MIFARE DESFire PICC.
                                                                                         *          - EncTMRI (16 bytes): Encrypted Transaction MAC ReaderID of
                                                                                         *            the latest successful transaction retuned by the PICC.
                                                                                         *            Present if \b bPiccErrCode = 0x00 or 0x90
                                                                                         *
                                                                                         *          - MAC (8byte): Response MAC protection. Present if
                                                                                         *            \b bPiccErrCode = 0x00 or 0x90 and not applying
                                                                                         *            D40 Secure Messaging
                                                                                         *
                                                                                         *          - Null otherwise
                                                                                         */
        uint8_t bDataLen,                                                               /**< [In] Length of bytes available in \b pData buffer. */
        uint8_t * pPiccReturnCode                                                       /**< [Out] Error code returned by MIFARE PICC. This will of 1 byte
                                                                                         *         in length.
                                                                                         */
    );

/**
 * end of group phhalHw_Sam_Cmd_Common_SMode
 * @}
 */





 /** \defgroup phhalHw_Sam_Cmd_Common_XMode X_Mode
 * \brief SAM commands used for MIFARE common PICC communication in X-Mode.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_Common_XMode_Defines Defines
 * \brief Definitions for SAM commands used for MIFARE Plus communication in X-Mode.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_Common_XMode_Defines_CommandCodes CommandCodes
 * \brief Definitions for Instruction codes
 * @{
 */
#define PHHAL_HW_SAM_CMD_INS_TMRI_COMMIT_READER_ID                              0x37U   /**< SAM Insturction code for TMRI_CommitReaderID command. */
/**
 * end of group phhalHw_Sam_Cmd_Common_XMode_Defines_CommandCodes
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_Common_XMode_Defines_ISOMode ISOMode
 * \brief Definitions for ISO Mode selection
 * @{
 */
#define PHHAL_HW_CMD_SAM_TMRI_ISO_MODE_NATIVE                                   0x00U   /**< Sam CommitReaderID ISO mode selection for Native command
                                                                                         *   set.
                                                                                         */
#define PHHAL_HW_CMD_SAM_TMRI_ISO_MODE_ISO7816                                  0x40U   /**< Sam CommitReaderID ISO mode selection for ISO 7816-4
                                                                                         *   command set.
                                                                                         */
/**
 * end of group phhalHw_Sam_Cmd_Common_XMode_Defines_ISOMode
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_Common_XMode_Defines
 * @}
 */

/**
 * \brief Performs CommitReaderID command execution in X mode. If success is returned the PICC return code will
 * have 0x00 as the value else the actual error code.
 *
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pEncTMRILen and \b pStatusCode are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_TMRI_CommitReaderID(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bISOMode,                                                               /**< [In] Options for P2 information byte.
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_TMRI_ISO_MODE_NATIVE
                                                                                         *            "Command will be sent to PICC using native command set"
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_TMRI_ISO_MODE_ISO7816
                                                                                         *            "Command will be sent to PICC using ISO/IEC 7816-4APDU"
                                                                                         */
        uint8_t bState,                                                                 /**< [In] Options for framing the command. Below options
                                                                                         *        should be used.
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_COMMIT_READER_ID_PICC_STATE_MFP
                                                                                         *                 "MIFARE Plus"
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_COMMIT_READER_ID_PICC_STATE_DESFIRE
                                                                                         *                 "MIFARE DESFire"
                                                                                         */
        uint16_t wBlockNr,                                                              /**< [In] Number of the Transaction MAC Configuration Block. Will
                                                                                         *        be used if \b bState is
                                                                                         *        \ref PHHAL_HW_CMD_SAM_COMMIT_READER_ID_PICC_STATE_MFP
                                                                                         *        "MIFARE Plus".
                                                                                         */
        uint8_t ** ppEncTMRI,                                                           /**< [Out] Encrypted Transaction MAC ReaderID of the latest
                                                                                         *         successful transaction retuned by the PICC.
                                                                                         */
        uint16_t * pEncTMRILen,                                                         /**< [Out] Length of bytes available in \b ppEncTMRI buffer. */
        uint8_t * pStatusCode                                                           /**< [Out] Status code returned by PICC. */
    );

/**
 * end of group phhalHw_Sam_Cmd_Common_XMode
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_Common
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd
 * @}
 */

#endif /* NXPBUILD__PHHAL_HW_SAM */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHHALHW_SAM_CMD_COMMON_H */
