/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) MIFARE Ultralight command implementation of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */


#ifndef PHHALHW_SAM_CMD_UL_H
#define PHHALHW_SAM_CMD_UL_H

#include <ph_Status.h>
#include <phhalHw.h>

#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */

#ifdef NXPBUILD__PHHAL_HW_SAM

/** \addtogroup phhalHw_Sam_Cmd
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_UL 10_MIFARE_Ultralight
 * \brief Implementation of SAM MIFARE Ultralight commands.
 * Has implementation for both X and S mode communication.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_UL_SMode S_Mode
 * \brief Implementation of SAM commands used for MIFARE Ultralight communication in S-Mode.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_UL_SMode_Defines Defines
 * \brief Definitions for SAM commands used for MIFARE Plus communication in S-Mode.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_UL_SMode_Defines_CommandCodes CommandCodes
 * \brief Definitions for Instruction codes
 * @{
 */
#define PHHAL_HW_SAM_CMD_INS_SAM_PWD_AUTH_UL                                    0x0BU   /**< SAM Instruction code for PwdAuthUL command. */
/**
 * end of group phhalHw_Sam_Cmd_UL_SMode_Defines_CommandCodes
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_UL_SMode_Defines
 * @}
 */

/**
 * \brief Performs PwdAuthUL command execution part 1 in S mode. The first part includes the header,
 * Key number, Key Version, Diversification input and returns Message
 *
 * \note
 *  - Post calling this interface it's must to call \ref phhalHw_Sam_Cmd_SAM_PwdAuthUL_Part2
 *    "PwdAuthUL Part2" to place SAM into proper completion state if \ref PH_ERR_SUCCESS_CHAINING
 *    "Chaining Status" is returned from Library. Otherwise error will be returned by SAM for any
 *    subsequent commands.
 *  - If \ref PH_ERR_SUCCESS_CHAINING "Chaining Status" is not returned from Library, no need to call
 *    \ref phhalHw_Sam_Cmd_SAM_PwdAuthUL_Part2 "PwdAuthUL Part2" interface.
 *
 * \retval #PH_ERR_SUCCESS_CHAINING     Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pDivInput and \b pPwdLen
 *                                      are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_PwdAuthUL_Part1(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bKeyNo,                                                                 /**< [In] Key number to be used from SAM. One of the following,
                                                                                         *          - NVM Key: 0x00 - 0x7F
                                                                                         *          - RAM Key: 0xE0 - 0xE3
                                                                                         */
        uint8_t bKeyVer,                                                                /**< [In] Key version to be used from SAM. */
        uint8_t * pDivInput,                                                            /**< [In] Diversification input for key diversification.
                                                                                         *        (1 to 31 byte(s) input).
                                                                                         */
        uint8_t bDivInputLen,                                                           /**< [In] Length of bytes available in \b pDivInput buffer. */
        uint8_t ** ppPwd,                                                               /**< [Out] Password received from SAM (4 byte). */
        uint16_t * pPwdLen                                                              /**< [Out] Length of bytes available in \b ppPwd buffer. */
    );

/**
 * \brief Performs PwdAuthUL command execution part 2 in S mode. The Last part includes
 * the header and Password authentication acknowledge
 *
 * \note This interface should be called only if \ref phhalHw_Sam_Cmd_SAM_PwdAuthUL_Part1
 * "PwdAuthUL Part1" returns \ref PH_ERR_SUCCESS_CHAINING "Chaining Status".
 *
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_PwdAuthUL_Part2(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wPack                                                                  /**< [In] Password Authentication Acknowledge. */
    );

/**
 * end of group phhalHw_Sam_Cmd_UL_SMode
 * @}
 */





/** \defgroup phhalHw_Sam_Cmd_UL_XMode X_Mode
 * \brief Implementation of SAM commands used for MIFARE Ultralight communication in X-Mode.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_UL_XMode_Defines Defines
 * \brief Definitions for SAM commands used for MIFARE Ultralight communication in X-Mode.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_UL_XMode_Defines_CommandCodes CommandCodes
 * \brief Definitions for Instruction codes
 * @{
 */
#define PHHAL_HW_SAM_CMD_INS_UL_PWD_AUTH_PICC                                   0x2DU   /**< Sam X mode Instruction code for PwdAuthUL command. */
#define PHHAL_HW_SAM_CMD_INS_UL_AUTHENTICATE_PICC                               0x2CU   /**< Sam X mode Instruction code for PwdAuthUL command. */
#define PHHAL_HW_SAM_CMD_INS_MF_READ                                            0x30U   /**< Sam X mode Instruction code for MF_Read command. */
#define PHHAL_HW_SAM_CMD_INS_MF_WRITE                                           0xA0U   /**< Sam X mode Instruction code for MF_Write command. */
#define PHHAL_HW_SAM_CMD_INS_UL_SM_EXCHANGE                                     0x7BU   /**< Sam X mode Instruction code for SMExchange command. */
/**
 * end of group phhalHw_Sam_Cmd_UL_XMode_Defines_CommandCodes
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_UL_XMode_Defines_Div Diversification
 * \brief Definitions for Key Diversification.
 * To be used with  \ref phhalHw_Sam_Cmd_UL_AuthenticatePICC "ULC_AuthenticatePICC" interface.
 * @{
 */
#define PHHAL_HW_SAM_CMD_ULC_AUTHENTICATE_DIV_OFF                               0x00U   /**< Option mask to disable the exchange of diversification input. */
#define PHHAL_HW_SAM_CMD_ULC_AUTHENTICATE_DIV_ON                                0x01U   /**< Option mask to enable the exchange of diversification input. */
/**
 * end of group phhalHw_Sam_Cmd_UL_XMode_Defines_Div
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_UL_XMode_Defines_LE ExchagneLE
 * \brief Definitions for LE Exchange.
 * To be used with  \ref phhalHw_Sam_Cmd_UL_AuthenticatePICC "ULC_AuthenticatePICC" interface.
 * @{
 */
#define PHHAL_HW_SAM_CMD_ULC_AUTHENTICATE_EXCLUDE_LE                            0x00U   /**< Option mask to exclude LE in exchange buffer. */
#define PHHAL_HW_SAM_CMD_ULC_AUTHENTICATE_INCLUDE_LE                            0x02U   /**< Option mask to include LE in exchange buffer. */
/**
 * end of group phhalHw_Sam_Cmd_UL_XMode_Defines_LE
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_UL_XMode_Defines_MFWrite MFWrite
 * \brief Definitions for Data to be written.
 * To be used with  \ref phhalHw_Sam_Cmd_MF_Write "MF_Write" interface.
 * @{
 */
#define PHHAL_HW_SAM_CMD_MF_WRITE_PLAIN                                 0x00    /**< Option mask for writing data to MIFARE Ultralight in
                                                                                 *   Plain compatible write (16 byte data blocks).
                                                                                 */
#define PHHAL_HW_SAM_CMD_MF_WRITE_ULTRALIGHT                            0x01    /**< Option mask for writing data to MIFARE Ultralight product
                                                                                 *   (4 byte data block).
                                                                                 */
/**
 * end of group phhalHw_Sam_Cmd_UL_XMode_Defines_MFWrite
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_UL_XMode_Defines
 * @}
 */

/**
 * \brief The UL_PwdAuthPICC supports the X-mode procedure to perform the Password Authentication on the
 * MIFARE Ultralight EV1 PICC. The command generates the password and password acknowledgment bytes and
 * performs the password authentication with the MIFARE Ultralight EV1 PICC. Finally, the command
 * compares the generated password acknowledgment bytes with the PACK bytes returned from the MIFARE
 * Ultralight EV1 PICC.
 *
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pDivInput and \b pStatusCode are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_UL_PwdAuthPICC(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bKeyNo,                                                                 /**< [In] Key number to be used from SAM. One of the following,
                                                                                         *          - NVM Key: 0x00 - 0x7F
                                                                                         *          - RAM Key: 0xE0 - 0xE3
                                                                                         */
        uint8_t bKeyVer,                                                                /**< [In] Key version to be used from SAM. */
        uint8_t * pDivInput,                                                            /**< [In] Diversification input for key diversification.
                                                                                         *        (1 to 31 byte(s) input).
                                                                                         */
        uint8_t bDivInputLen,                                                           /**< [In] Length of bytes available in \b pDivInput buffer. */
        uint8_t * pStatusCode                                                           /**< [Out] Status code returned by PICC.*/
    );

/**
 * \brief Performs AuthenticatePICC command execution in X mode. Includes the header, Key number,
 * Key Version,Diversification input based on \b bOption parameter.
 *
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pDivInput and \b pStatusCode are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_UL_AuthenticatePICC(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                                /**< [In] Options for Key diversification selection and
                                                                                         *        Exchanging LE to SAM. The options can be
                                                                                         *        combined by bitwise OR.
                                                                                         *          - Use Key Diversification. Option for P1
                                                                                         *            information byte.
                                                                                         *              - \ref PHHAL_HW_SAM_CMD_ULC_AUTHENTICATE_DIV_OFF
                                                                                         *                     "No diversification"
                                                                                         *              - \ref PHHAL_HW_SAM_CMD_ULC_AUTHENTICATE_DIV_ON
                                                                                         *                     "Diversify the used key with the given DivInput"
                                                                                         *
                                                                                         *          - Option for exchanging LE to SAM. Used internally by
                                                                                         *            Library.
                                                                                         *              - \ref PHHAL_HW_SAM_CMD_ULC_AUTHENTICATE_EXCLUDE_LE
                                                                                         *                     "LE is not exchanged to SAM"
                                                                                         *              - \ref PHHAL_HW_SAM_CMD_ULC_AUTHENTICATE_INCLUDE_LE
                                                                                         *                     "LE is exchanged to SAM"
                                                                                         */
        uint8_t bKeyNo,                                                                 /**< [In] Key number to be used from SAM. One of the following,
                                                                                         *          - NVM Key: 0x00 - 0x7F
                                                                                         *          - RAM Key: 0xE0 - 0xE3
                                                                                         */
        uint8_t bKeyVer,                                                                /**< [In] Key version to be used from SAM. */
        uint8_t * pDivInput,                                                            /**< [In] Diversification input for key diversification.
                                                                                         *        (1 to 15 byte(s) input).
                                                                                         */
        uint8_t bDivInputLen,                                                           /**< [In] Length of bytes available in \b pDivInput buffer. */
        uint8_t * pStatusCode                                                           /**< [Out] Status code returned by PICC.*/
    );

/**
 * \brief Perform a MIFARE Ultralight Read command in X mode.
 *  - In X-mode, MF_Read reads blocks of a MIFARE Ultralight or MIFARE Ultralight C card in
 *    plain and return the data. The MIFARE SAM AV4 constructs the READ command (Cmd 0x30)
 *    and sends it to the card.
 *  - If more than one block address is issued, the MIFARE SAM AV4 accesses the blocks in the
 *    same order as addresses listed in the command data field. The order of the returned data
 *    is the same as the order of addresses in the data field.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pBlocks and \b pDataLen are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_MF_Read(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t * pBlocks,                                                              /**< [In] The block numbers from where the data should be read.
                                                                                         *         Should be holding an array of block numbers like
                                                                                         *         BNR_1, BNR_2, BNR_3, ..., BNR_N
                                                                                         */
        uint8_t bBlocksLen,                                                             /**< [In] Length pf bytes available in \b pBlocks parameter. */
        uint8_t ** ppData,                                                              /**< [Out] Data returned by Sam. */
        uint16_t * pDataLen                                                             /**< [Out] Length pf bytes available in \b ppData parameter. */
    );

/**
 * \brief Perform a MIFARE Ultralight Write command in X mode.
 *  - In X-mode, MF_Write writes one or several blocks a MIFARE Ultralight or MIFARE
 *    Ultralight C card . The command supports writing 16 bytes or 4 bytes plain. The
 *    length can be selected by bit 0 of parameter byte P2. If 4 byte data blocks is
 *    chosen, the MIFARE SAM AV4 constructs the WRITE command (Cmd 0xA2). If 16 byte
 *    data blocks, the COMPATIBILITY_WRITE(Cmd 0xA0) is used.
 *  - If more than one block is written, the MIFARE SAM AV4 accesses the blocks in the
 *    same order as addresses listed in the command data field.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pData is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_MF_Write(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                                /**< [In] One of the below options. For Updating the P1
                                                                                         *        information byte.
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_MF_WRITE_PLAIN
                                                                                         *                 "Plain compatible write (16 byte data blocks)"
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_MF_WRITE_ULTRALIGHT
                                                                                         *                 "MIFARE Ultralight(4 byte data blocks)"
                                                                                         */
        uint8_t * pData,                                                                /**< [In] Buffer containing the Block and Data pair to be written .
                                                                                         *        to card by Sam hardware. Should be holding an array of
                                                                                         *        block number and data like. \n
                                                                                         *        BNR_1 + Data, BNR_2 + Data, BNR_3 + Data, ..., BNR_N + Data \n
                                                                                         *        Data could be 4 bytes or 16 bytes depending on the option specified.
                                                                                         */
        uint8_t bDataLen                                                                /**< [In] Length pf bytes available in \b pData parameter. */
    );

/**
 * \brief The UL_SMExchange command is used to apply the MIFARE Ultralight AES CMAC-based
 * secure messaging in X-mode on PICC commands.
 *  - The user is responsible for providing the plain command data including the PICC command
 *    code, and if applicable additional parameters and data.
 *  - Supports the following command or MIFARE Ultralight AES product.
 *      - GET_VERSION
 *      - READ
 *      - FAST_READ
 *      - WRITE
 *      - READ_CNT
 *      - INCR_CNT
 *      - READ_SIG
 *      - WRITE_SIG
 *      - LOCK_SIG
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pData and \b pRespLen are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_UL_SMExchange(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t * pData,                                                                /**< [In] Plain command data of PICC. SAM will apply SM
                                                                                         *        and exchange to PICC.
                                                                                         */
        uint8_t bDataLen,                                                               /**< [In] Length pf bytes available in \b pData parameter. */
        uint8_t ** ppResponse,                                                          /**< [Out] Plain Data returned by Sam. */
        uint16_t * pRespLen                                                             /**< [Out] Length pf bytes available in \b ppResponse parameter. */
    );

/**
 * end of group phhalHw_Sam_Cmd_UL_XMode
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_UL
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd
 * @}
 */

#endif /* NXPBUILD__PHHAL_HW_SAM */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHHALHW_SAM_CMD_UL_H */
