/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) MIFARE Plus command implementation of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */


#ifndef PHHALHW_SAM_CMD_PLUS_H
#define PHHALHW_SAM_CMD_PLUS_H

#include <ph_Status.h>
#include <phhalHw.h>

#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */

#ifdef NXPBUILD__PHHAL_HW_SAM

#define PHHAL_HW_SAM_PLUS_COMBINED_READ_MASK                                    0x03U   /* Option mask for Combined Read */
#define PHHAL_HW_SAM_PLUS_COMBINED_WRITE_MASK                                   0x01U   /* Option mask for Combined Write */
#define PHHAL_HW_SAM_PLUS_COMBINED_WRITE_PLIAN_RESPONSE_MASK                    0x02U   /* Option mask for Combined Write Plain */
#define PHHAL_HW_SAM_PLUS_PICC_RESPONSE_CODE                                    0x90U   /* MIFARE Plus PICC Response code */
#define PHHAL_HW_SAM_PLUS_PICC_RESPONSE_CODE_LEN                                0x01U   /* MIFARE Plus PICC Response code length */
#define PHHAL_HW_SAM_PLUS_PICC_CAPABALITIES_LEN                                 0x06U   /* MIFARE Plus PICC Capabalities length */

/** \addtogroup phhalHw_Sam_Cmd
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_Plus 09_MIFARE_Plus
 * \brief Implementation of SAM MIFARE Plus commands.
 * Has implementation for both X and S mode communication.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_Plus_Defines Defines
 * \brief Definitions common for MIFARE Plus S and X mode communication
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_Plus_Defines_Authentication Authenticate
 * \brief Definitions common for MIFARE Plus S and X mode communication for
 * Authentication interfaces.
 * To be used with below mentioned interfaces.
 *  - \ref phhalHw_Sam_Cmd_SAM_AuthenticateMFP_Part1 "SAM_AuthenticateMFP Part1"
 *  - \ref phhalHw_Sam_Cmd_MFP_Authenticate "MFP_Authenticate"
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_Plus_Defines_Authentication_Div Diversification
 * \brief Definitions for Key Diversification Mode.
 * To be used with below mentioned interfaces.
 *  - \ref phhalHw_Sam_Cmd_SAM_AuthenticateMFP_Part1 "SAM_AuthenticateMFP Part1"
 *  - \ref phhalHw_Sam_Cmd_MFP_Authenticate "MFP_Authenticate"
 * @{
 */
#define PHHAL_HW_SAM_MFP_AUTHENTICATE_DIVERSIFICATION_OFF                       0x00U   /**< Option to disable the key diversification. */
#define PHHAL_HW_SAM_MFP_AUTHENTICATE_DIVERSIFICATION_ON                        0x01U   /**< Option to enable the key diversification. */
/**
 * end of group phhalHw_Sam_Cmd_Plus_Defines_Authentication_Div
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_Plus_Defines_Authentication_AuthMode AuthMode
 * \brief Definitions for Authentication Mode.
 * To be used with below mentioned interfaces.
 *  - \ref phhalHw_Sam_Cmd_SAM_AuthenticateMFP_Part1 "SAM_AuthenticateMFP Part1"
 *  - \ref phhalHw_Sam_Cmd_MFP_Authenticate "MFP_Authenticate"
 * @{
 */
#define PHHAL_HW_SAM_MFP_AUTHENTICATE_FIRST                                     0x00U   /**< Option to perform First authentication. */
#define PHHAL_HW_SAM_MFP_AUTHENTICATE_NON_FIRST                                 0x02U   /**< Option to perform NonFirst (following)
                                                                                         *   authentication.
                                                                                         */
/**
 * end of group phhalHw_Sam_Cmd_Plus_Defines_Authentication_AuthMode
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_Plus_Defines_Authentication_SL_KDF SL_KDF
 * \brief Definitions for Security Level and Key derivation information.
 * To be used with below mentioned interfaces.
 *  - \ref phhalHw_Sam_Cmd_SAM_AuthenticateMFP_Part1 "SAM_AuthenticateMFP Part1"
 *  - \ref phhalHw_Sam_Cmd_MFP_Authenticate "MFP_Authenticate"
 * @{
 */
#define PHHAL_HW_SAM_MFP_AUTHENTICATE_SL1_NO_KDF                                0x00U   /**< Option to set the key derivation info for SL0
                                                                                         *   or SL1 layer.
                                                                                         */
#define PHHAL_HW_SAM_MFP_AUTHENTICATE_SL3_KDF                                   0x0CU   /**< Option to set the key derivation info for SL3
                                                                                         *   layer.
                                                                                         */
/**
 * end of group phhalHw_Sam_Cmd_Plus_Defines_Authentication_SL_KDF
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_Plus_Defines_Authentication
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_Plus_Defines_SSAuthentication AuthenticateSectorSwitch
 * \brief Definitions common for MIFARE Plus S and X mode communication for
 * SectorSwitch Authentication interfaces.
 * To be used with below mentioned interfaces.
 *  - \ref phhalHw_Sam_Cmd_SAM_AuthSectorSwitchMFP_Part1 "SAM_AuthSectorSwitchMFP Part1"
 *  - \ref phhalHw_Sam_Cmd_MFP_AuthSectorSwitch "MFP_AuthSectorSwitch"
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_Plus_Defines_SSAuthentication_DivSS Diversify_SectorSwitchKey
 * \brief Definitions for Diversification for Sector Switch Key.
 * To be used with below mentioned interfaces.
 *  - \ref phhalHw_Sam_Cmd_SAM_AuthSectorSwitchMFP_Part1 "SAM_AuthSectorSwitchMFP Part1"
 *  - \ref phhalHw_Sam_Cmd_MFP_AuthSectorSwitch "MFP_AuthSectorSwitch"
 * @{
 */
#define PHHAL_HW_SAM_MFP_SSAUTH_SECTOR_SWITCH_DIV_OFF                           0x00U   /**< Option to disable the Sector Switch key diversification. */
#define PHHAL_HW_SAM_MFP_SSAUTH_SECTOR_SWITCH_DIV_ON                            0x01U   /**< Option to enable the Sector Switch key diversification. */
/**
 * end of group phhalHw_Sam_Cmd_Plus_Defines_SSAuthentication_DivSS
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_Plus_Defines_SSAuthentication_DivS Diversify_SectorKey
 * \brief Definitions for Diversification for Sector Key.
 * To be used with below mentioned interfaces.
 *  - \ref phhalHw_Sam_Cmd_SAM_AuthSectorSwitchMFP_Part1 "SAM_AuthSectorSwitchMFP Part1"
 *  - \ref phhalHw_Sam_Cmd_MFP_AuthSectorSwitch "MFP_AuthSectorSwitch"
 * @{
 */
#define PHHAL_HW_SAM_MFP_SSAUTH_SECTOR_DIV_OFF                                  0x00U   /**< Option to disable the Sector key diversification. */
#define PHHAL_HW_SAM_MFP_SSAUTH_SECTOR_DIV_ON                                   0x02U   /**< Option to enable the Sector key diversification. */
/**
 * end of group phhalHw_Sam_Cmd_Plus_Defines_SSAuthentication_DivS
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_Plus_Defines_SSAuthentication_DivM Diversify_MasterKey
 * \brief Definitions for Diversification for Master Key.
 * To be used with below mentioned interfaces.
 *  - \ref phhalHw_Sam_Cmd_SAM_AuthSectorSwitchMFP_Part1 "SAM_AuthSectorSwitchMFP Part1"
 *  - \ref phhalHw_Sam_Cmd_MFP_AuthSectorSwitch "MFP_AuthSectorSwitch"
 * @{
 */
#define PHHAL_HW_SAM_MFP_SSAUTH_MASTER_SECTOR_DIV_OFF                           0x00U   /**< Option to disable the Master Sector key diversification. */
#define PHHAL_HW_SAM_MFP_SSAUTH_MASTER_SECTOR_DIV_ON                            0x04U   /**< Option to enable the Master Sector key diversification with
                                                                                         *   given sector number.
                                                                                         */
/**
 * end of group phhalHw_Sam_Cmd_Plus_Defines_SSAuthentication_DivM
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_Plus_Defines_SSAuthentication
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_Plus_Defines_PDCAuthentication AuthenticatePDC
 * \brief Definitions common for MIFARE Plus S and X mode communication for
 * Post Delivery Configuration Authentication interfaces.
 * To be used with below mentioned interfaces.
 *  - \ref phhalHw_Sam_Cmd_SAM_AuthenticatePDC_Part1 "SAM_AuthenticatePDC Part1"
 *  - \ref phhalHw_Sam_Cmd_PDC_Authenticate "PDC_Authenticate"
 * @{
 */
#define PHHAL_HW_SAM_PDC_AUTH_DERIVATION_OFF                                    0x00U    /**< Option to disable the key diversification. */
#define PHHAL_HW_SAM_PDC_AUTH_DERIVE_UPGRADE_KEY                                0x02U    /**< Option to indicate the UpgradeKey derivation from
                                                                                          *   ICUpgradeKey given UpgradeInfo.
                                                                                          */
#define PHHAL_HW_SAM_PDC_AUTH_DIVERSIFY_YEAR_UPGRADE_KEY                        0x03U    /**< Option to indicate the diversification of YearUpgradeKey with
                                                                                          *   the given DivInput and then derive the actual UpgradeKey with
                                                                                          *   UpgradeInfo.
                                                                                          */
/**
 * end of group phhalHw_Sam_Cmd_Plus_Defines_PDCAuthentication
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_Plus_Defines_ChangeKey ChangeKey
 * \brief Definitions common for MIFARE Plus S and X mode communication for
 * ChagneKey interfaces for diversification.
 * To be used with below mentioned interfaces.
 *  - \ref phhalHw_Sam_Cmd_SAM_ChangeKeyMFP "SAM_ChangeKeyMFP"
 *  - \ref phhalHw_Sam_Cmd_MFP_ChangeKey "MFP_ChangeKey"
 * @{
 */
#define PHHAL_HW_SAM_MFP_CHANGE_KEY_DIVERSIFICATION_OFF                         0x00U   /**<  Option to disable the key diversification. */
#define PHHAL_HW_SAM_MFP_CHANGE_KEY_DIVERSIFICATION_ON                          0x02U   /**<  Option to enable the key diversification. */
/**
 * end of group phhalHw_Sam_Cmd_Plus_Defines_ChangeKey
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_Plus_Defines
 * @}
 */





/** \defgroup phhalHw_Sam_Cmd_Plus_SMode S_Mode
 * \brief Implementation of SAM commands used for MIFARE Plus communication in S-Mode.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_Plus_SMode_Defines Defines
 * \brief Definitions for SAM commands used for MIFARE Plus communication in S-Mode.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_Plus_SMode_Defines_CommandCodes CommandCodes
 * \brief Definitions for Instruction codes
 * @{
 */
#define PHHAL_HW_SAM_CMD_INS_SAM_AUTHENTICATE_MFP                                   0xA3U   /**< SAM Insturction code for SAM_AuthenticateMFP command */
#define PHHAL_HW_SAM_CMD_INS_SAM_AUTH_SECTOR_SWITCH_MFP                             0x38U   /**< SAM Instruction code for SAM_AuthSectorSwitchMFP command */
#define PHHAL_HW_SAM_CMD_INS_SAM_AUTHENTICATE_PDC                                   0xACU   /**< SAM Instruction code for SAM_AuthenticatePDC command */
#define PHHAL_HW_SAM_CMD_INS_SAM_COMBINED_READ_MFP                                  0x33U   /**< SAM Instruction code for SAM_CombinedReadMFP command */
#define PHHAL_HW_SAM_CMD_INS_SAM_COMBINED_WRITE_MFP                                 0x34U   /**< SAM Instruction code for SAM_CombinedWriteMFP command */
#define PHHAL_HW_SAM_CMD_INS_SAM_CHANGE_KEY_MFP                                     0x35U   /**< SAM Instruction code for SAM_ChangeKeyMFP command */
/**
 * end of group phhalHw_Sam_Cmd_Plus_SMode_Defines_CommandCodes
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_Plus_SMode_Defines_PayloadType PayloadType
 * \brief Definitions for payload type. To be used with below interfaces
 *      - \ref phhalHw_Sam_Cmd_SAM_CombinedReadMFP "SAM_CombinedReadMFP"
 *      - \ref phhalHw_Sam_Cmd_SAM_CombinedWriteMFP "SAM_CombinedWriteMFP"
 *      - \ref phhalHw_Sam_Cmd_SAM_ChangeKeyMFP "SAM_ChangeKeyMFP"
 * @{
 */
#define PHHAL_HW_SAM_OPTION_MFP_PAYLOAD_TYPE_COMMAND                                0x00U   /**< Option value for Payload Type - Command. */
#define PHHAL_HW_SAM_OPTION_MFP_PAYLOAD_TYPE_RESPONSE                               0x01U   /**< Option value for Payload Type - Response. */
#define PHHAL_HW_SAM_OPTION_MFP_PAYLOAD_TYPE_BOTH                                   0x02U   /**< Option value for Payload Type - Both Command and
                                                                                             *   Response.
                                                                                             */
/**
 * end of group phhalHw_Sam_Cmd_Plus_SMode_Defines_PayloadType
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_Plus_SMode_Defines_WriteMode WriteMode
 * \brief Definitions for Combined Write Plain data in response.
 * To be used with \ref phhalHw_Sam_Cmd_SAM_CombinedWriteMFP "SAM_CombinedWriteMFP" interface
 * @{
 */
#define PHHAL_HW_SAM_OPTION_COMBINED_WRITE_RETURN_PLAIN                         0x00U   /**< Option macro to return the plain data in response. */
#define PHHAL_HW_SAM_OPTION_COMBINED_WRITE_SKIP_PLAIN                           0x02U   /**< Option macro to skip the plain data in response. */
/**
 * end of group phhalHw_Sam_Cmd_Plus_SMode_Defines_WriteMode
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_Plus_SMode_Defines
 * @}
 */

/**
 * \brief Perform a MFP Authenticate command part1. This command will generate a
 * 16 byte random number with the one received from PICC and return an 32 byte
 * encrypted data which will be sent to PICC.
 *
 * \note
 *  - Post calling this interface it's must to call \ref phhalHw_Sam_Cmd_SAM_AuthenticateMFP_Part2
 *    "AuthenticateMFP Part2" to place SAM into proper completion state if \ref PH_ERR_SUCCESS_CHAINING
 *    "Chaining Status" is returned from Library. Otherwise error will be returned by SAM for any
 *    subsequent commands.
 *  - If \ref PH_ERR_SUCCESS_CHAINING "Chaining Status" is not returned from Library, no need to call
 *    \ref phhalHw_Sam_Cmd_SAM_AuthenticateMFP_Part2 "AuthenticateMFP Part2" interface.
 *
 * \retval #PH_ERR_SUCCESS_CHAINING     Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pPDChal, \b pDivInput and \b pPCDChalRespLen
 *                                      are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_AuthenticateMFP_Part1(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                                /**< [In] Key diversification & authentication mode selection.
                                                                                         *        Option to update the P1 information. The options can
                                                                                         *        be combined by bitwise oring.
                                                                                         *          - SL and Key derivation info
                                                                                         *              - \ref PHHAL_HW_SAM_MFP_AUTHENTICATE_SL1_NO_KDF
                                                                                         *                "No Key" Derivation information (SL1,
                                                                                         *                originality keys...)
                                                                                         *              - \ref PHHAL_HW_SAM_MFP_AUTHENTICATE_SL3_KDF
                                                                                         *                "Security Level 3" Key Derivatin information
                                                                                         *
                                                                                         *          - Select Authentication Mode
                                                                                         *              - \ref PHHAL_HW_SAM_MFP_AUTHENTICATE_FIRST
                                                                                         *                "Authenticate First"
                                                                                         *              - \ref PHHAL_HW_SAM_MFP_AUTHENTICATE_NON_FIRST
                                                                                         *                "Authenticate NonFirst"
                                                                                         *
                                                                                         *          - Use Key Diversification
                                                                                         *              - \ref PHHAL_HW_SAM_MFP_AUTHENTICATE_DIVERSIFICATION_OFF
                                                                                         *                "No diversification"
                                                                                         *              - \ref PHHAL_HW_SAM_MFP_AUTHENTICATE_DIVERSIFICATION_ON
                                                                                         *                "Diversify the used key with the given DivInput"
                                                                                         */
        uint8_t bKeyNo,                                                                 /**< [In] Key number to be used from SAM. One of the following,
                                                                                         *          - NVM Key: 0x00 - 0x7F
                                                                                         *          - RAM Key: 0xE0 - 0xE3
                                                                                         */
        uint8_t bKeyVer,                                                                /**< [In] Key version to be used from SAM. */
        uint8_t * pPDChal,                                                              /**< [In] Buffer containing the challange generated by the PICC.
                                                                                         *        E(Kx,RndB) (16 bytes).
                                                                                         */
        uint8_t bPDChalLen,                                                             /**< [In] Length of bytes available in \b pPDChal buffer. */
        uint8_t * pDivInput,                                                            /**< [In] Diversification Input used to diversify the key.
                                                                                         *          - If any of diversification option is set in
                                                                                         *            \b bOption parameter, then
                                                                                         *              - 8 (if AV1 key diversification with DES)
                                                                                         *              - 16 (if AV1 key diversification with AES) bytes
                                                                                         *              - 1 to 31 bytes (if AV2 key diversification)
                                                                                         *                diversification input
                                                                                         *          - NULL otherwise
                                                                                         */
        uint8_t bDivInputLen,                                                           /**< [In] Length of bytes available in \b pDivInput buffer. */
        uint8_t ** ppPCDChalResp,                                                       /**< [Out] Buffer containing the PCD challenge response to be
                                                                                         *         sent to PICC. Challenge response as E(Kx, RndA || RndB')
                                                                                         */
        uint16_t * pPCDChalRespLen                                                      /**< [Out] Length of bytes available in \b ppPCDChalResp buffer. */
    );

/**
 * \brief Perform a MFP Authenticate command part2. This command will decrypt the response
 * received from PICC and will return the PC capabilities and PCD capabilites.
 *
 * \note
 *      - This interface should be called only if \ref phhalHw_Sam_Cmd_SAM_AuthenticateMFP_Part1
 *        "AuthenticateMFP Part1" returns \ref PH_ERR_SUCCESS_CHAINING "Chaining Status".
 *      - This interface also support Cmd.SAM_AuthenticateMFPError internally. For this
 *        \b bPiccErrCode should utialized and \b pPiccReturnCode will have the PICC error
 *        code that is exchanged.
 *
 * \retval #PH_ERR_SUCCESS                      Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS          \b pDataParams is NULL.
 * \retval #PHHAL_HW_SAM_ERR_MIFARE_PLUS_GEN    In case of PICC Error exchanged.
 * \retval #PH_ERR_INVALID_PARAMETER            \b pPDResp and \b pPiccReturnCode are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_AuthenticateMFP_Part2(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bPiccErrCode,                                                           /**< [In] Error Code sent by the MIFARE Plus PICC. For success it
                                                                                         *        should be 0x90.
                                                                                         */
        uint8_t * pPDResp,                                                              /**< [In] Buffer containing the input received from PICC.
                                                                                         *          - If no PICC error: PICC authentication response.
                                                                                         *              - If \ref PHHAL_HW_SAM_MFP_AUTHENTICATE_FIRST
                                                                                         *                "Authenticate First": E(Kx,TI || RndA' || adjPDcap2
                                                                                         *                || adjPCDcap2)
                                                                                         *              - If \ref PHHAL_HW_SAM_MFP_AUTHENTICATE_NON_FIRST
                                                                                         *                "Authenticate NonFirst": E(Kx,RndA') (16 bytes)
                                                                                         */
        uint8_t bPDRespLen,                                                             /**< [In] Length of bytes available in \b pPDResp buffer. */
        uint8_t ** ppPDCap2,                                                            /**< [Out] Buffer containing the Output PCD capabilities.
                                                                                         *         This will be of 6 bytes.
                                                                                         */
        uint8_t ** ppPCDCap2,                                                           /**< [Out] Buffer containing the Output PD capabilities.
                                                                                         *         This will be of 6 bytes.
                                                                                         */
        uint8_t * pPiccReturnCode                                                       /**< [Out] Error code returned by MIFARE Plus PICC. This will of 1
                                                                                         *         byte in length.
                                                                                         */
    );

/**
 * \brief Perform a MFP Sector Switch Authenticate command part1. This command will generate a 16 byte random
 * number with the one received from PICC and return an 32 byte encrypted data which will be sent to PICC.
 *
 * \note
 *  - Post calling this interface it's must to call \ref phhalHw_Sam_Cmd_SAM_AuthSectorSwitchMFP_Part2
 *    "AuthSectorSwitchMFP Part2" to place SAM into proper completion state if \ref PH_ERR_SUCCESS_CHAINING
 *    "Chaining Status" is returned from Library. Otherwise error will be returned by SAM for any
 *    subsequent commands.
 *  - If \ref PH_ERR_SUCCESS_CHAINING "Chaining Status" is not returned from Library, no need to call
 *    \ref phhalHw_Sam_Cmd_SAM_AuthSectorSwitchMFP_Part2 "AuthSectorSwitchMFP Part2" interface.
 *
 * \retval #PH_ERR_SUCCESS_CHAINING     Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pPDChal, \b pDivInput and \b pPCDChalRespLen
 *                                      are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_AuthSectorSwitchMFP_Part1(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                                /**< [In] Key diversification selection. Option to update the
                                                                                         *        P1 information. The options can be combined by bitwise
                                                                                         *        oring.
                                                                                         *          - Apply sector number key diversification
                                                                                         *              - \ref PHHAL_HW_SAM_MFP_SSAUTH_MASTER_SECTOR_DIV_OFF
                                                                                         *                "No diversification"
                                                                                         *              - \ref PHHAL_HW_SAM_MFP_SSAUTH_MASTER_SECTOR_DIV_ON
                                                                                         *                "Diversify the master Sector Key with the given sector number"
                                                                                         *
                                                                                         *          - Use Key Diversification for Sector Keys
                                                                                         *              - \ref PHHAL_HW_SAM_MFP_SSAUTH_SECTOR_DIV_OFF
                                                                                         *                "No diversification"
                                                                                         *              - \ref PHHAL_HW_SAM_MFP_SSAUTH_SECTOR_DIV_ON
                                                                                         *                "Diversify the used keys with the given DivInput"
                                                                                         *
                                                                                         *          - Use Key Diversification for Sector Switch Key
                                                                                         *              - \ref PHHAL_HW_SAM_MFP_SSAUTH_SECTOR_SWITCH_DIV_OFF
                                                                                         *                "No diversification"
                                                                                         *              - \ref PHHAL_HW_SAM_MFP_SSAUTH_SECTOR_SWITCH_DIV_ON
                                                                                         *                "Diversify the used key with the given DivInput"
                                                                                         */
        uint8_t * pPDChal,                                                              /**< [In] Buffer containing the challange generated by the PICC.
                                                                                         *        PICC challenge as E ( Kn , ..., E ( K_1 , E ( Kss , RndB))...)
                                                                                         */
        uint8_t bPDChalLen,                                                             /**< [In] Length of bytes available in \b pPDChal buffer. */
        uint16_t wSSKeyBNr,                                                             /**< [In] PICC Sector Switch key block number to be used for
                                                                                         *        authentication.
                                                                                         */
        uint8_t bSSKeyNo,                                                               /**< [In] Sector Switch Key number to be used from SAM (that is: Kss selection).
                                                                                         *        One of the following,
                                                                                         *          - NVM Key: 0x00 - 0x7F
                                                                                         *          - RAM Key: 0xE0 - 0xE3
                                                                                         */
        uint8_t bSSKeyVer,                                                              /**< [In] Sector Switch Key version to be used from SAM (that
                                                                                         *        is: Kss selection)
                                                                                         */
        uint8_t bMSKeyNo,                                                               /**< [In] Master Key number to be used from SAM One of the following,
                                                                                         *          - NVM Key: 0x00 - 0x7F
                                                                                         *          - RAM Key: 0xE0 - 0xE3
                                                                                         */
        uint8_t bMSKeyVer,                                                              /**< [In] Master Key version to be used from SAM. */
        uint8_t bSectorCount,                                                           /**< [In] Number of sectors to be switched inside the PICC.
                                                                                         *          - 0x01 - 0x2A: If \ref PHHAL_HW_SAM_MFP_SSAUTH_MASTER_SECTOR_DIV_OFF
                                                                                         *            "No sector number key diversification"
                                                                                         *          - 0x01 - 0x37: If \ref PHHAL_HW_SAM_MFP_SSAUTH_MASTER_SECTOR_DIV_ON
                                                                                         *            "Sector number key diversification"
                                                                                         */
        uint8_t * pKeyBlocks,                                                           /**< [In] Buffer containing the PICC KeyB block number and reference key number
                                                                                         *        and version to be used form SAM. Should contain the following,
                                                                                         *          - KeyBxBNr: 2 byte Key Block sector number on the PICC (encoded MSB first)
                                                                                         *          - KeyNo: If \ref PHHAL_HW_SAM_MFP_SSAUTH_MASTER_SECTOR_DIV_OFF
                                                                                         *            "No sector number key diversification", SAM Key Number of the block
                                                                                         *            sector key (that is: Kn selection).
                                                                                         *              - NVM Key: 0x00 - 0x7F
                                                                                         *              - RAM Key: 0xE0 - 0xE3
                                                                                         *          - KeyVer: If \ref PHHAL_HW_SAM_MFP_SSAUTH_MASTER_SECTOR_DIV_OFF
                                                                                         *            "No sector number key diversification", SAM Key Version of the block
                                                                                         *            sector key (that is: Kn selection).
                                                                                         *
                                                                                         *          - The format of the buffer should be,
                                                                                         *            KeyBxBNr_1 || [KeyNo_1] || [KeyVer_1] || KeyBxBNr_2 || [KeyNo_2] || [KeyVer_2]
                                                                                         *            || ... || KeyBxBNr_N || [KeyNo_N] || [KeyVer_N], Where N equal to the
                                                                                         *            \b bSectorCount information.
                                                                                         */
        uint8_t bKeyBlocksLen,                                                          /**< [In] Length of bytes available in \b pKeyBlocks buffer. It should
                                                                                         *        be equal to (KeyBxBNr + [KeyNo] + [KeyVer]) * \b bSectorCount
                                                                                         */
        uint8_t * pDivInput,                                                            /**< [In] Buffer containing the diversification inputs to be used for diversifying the key.
                                                                                         *          - SSKeyDivLen: Optional, present if \ref PHHAL_HW_SAM_MFP_SSAUTH_SECTOR_SWITCH_DIV_ON
                                                                                         *            "Sector Switch Diversification is enabled", Length of the Sector Switch Key DivInput
                                                                                         *            (and of Sector Keys DivInput if using same)
                                                                                         *          - SSKeyDivInput: Optional, present if \ref PHHAL_HW_SAM_MFP_SSAUTH_SECTOR_SWITCH_DIV_ON
                                                                                         *            "Sector Switch Diversification is enabled", 1 to 31 bytes diversification input for
                                                                                         *            Sector Switch Key (and for Sector Keys if using same)
                                                                                         *          - KeyBxDivLen: Optional, present if \ref PHHAL_HW_SAM_MFP_SSAUTH_SECTOR_DIV_ON
                                                                                         *            "Sector Diversification is enabled". One of the following,
                                                                                         *              - Using same DivInput for Sector Switch Key and Sector Keys: 0x00
                                                                                         *              - Using different DivInput for Sector keys: 0x01 - 0x1F
                                                                                         *          - KeyBxDivInput: Optional, present if \ref PHHAL_HW_SAM_MFP_SSAUTH_SECTOR_DIV_ON
                                                                                         *            "Sector Diversification is enabled" and KeyBxDivLen != 0x00.
                                                                                         *            1 to 31 bytes diversification input for Sector Keys
                                                                                         *
                                                                                         *          - The format of the buffer should be,
                                                                                         *            [[SSKeyDivLen] || [SSKeyDivInput]] || [[KeyBxDivLen] || [KeyBxDivInput]]
                                                                                         */
        uint8_t bDivInputLen,                                                           /**< [In] Length of bytes available in \b pDivInput buffer. */
        uint8_t ** ppPCDChalResp,                                                       /**< [Out] Buffer containing the PCD challenge response to be sent to PICC.
                                                                                         *         Challenge response as E ( K_n , ..., E ( K_1 , E ( Kss, RndA || RndB'))...)
                                                                                         */
        uint16_t * pPCDChalRespLen                                                      /**< [Out] Length of bytes available in \b ppPCDChalResp buffer. */
    );

/**
 * \brief Perform a MFP Sector Switch Authenticate Authenticate command part2. This command
 * will decrypt the response received from PICC and will return success status if the
 * challanges matches.
 *
 * \note This interface should be called only if \ref phhalHw_Sam_Cmd_SAM_AuthSectorSwitchMFP_Part1
 * "AuthSectorSwitchMFP Part1" returns \ref PH_ERR_SUCCESS_CHAINING "Chaining Status".
 *
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pPDResp and \b pPiccReturnCode are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_AuthSectorSwitchMFP_Part2(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bPiccErrCode,                                                           /**< [In] Error Code sent by the MFP PICC. For success it
                                                                                         *        should be 0x90.
                                                                                         */
        uint8_t * pPDResp,                                                              /**< [In] Buffer containing the input received from PICC.
                                                                                         *        PICC challenge response as E(Kn, ..., E(K1,
                                                                                         *        E(Kss, RndA'))...)
                                                                                         */
        uint8_t bPDRespLen,                                                             /**< [In] Length of bytes available in \b pPDResp buffer. */
        uint8_t * pPiccReturnCode                                                       /**< [Out] Error code returned by MIFARE Plus PICC. This will
                                                                                         *         of 1 byte in length.
                                                                                         */
    );

/**
 * \brief Perform a Post delivery configuration Authenticate command part1. This command will generate a 16
 * byte random number with the one received from PICC and return an 32 byte encrypted data which will be sent
 * to PICC.
 *
 * \note
 *  - Post calling this interface it's must to call \ref phhalHw_Sam_Cmd_SAM_AuthenticatePDC_Part2
 *    "AuthenticatePDC Part2" to place SAM into proper completion state if \ref PH_ERR_SUCCESS_CHAINING
 *    "Chaining Status" is returned from Library. Otherwise error will be returned by SAM for any
 *    subsequent commands.
 *  - If \ref PH_ERR_SUCCESS_CHAINING "Chaining Status" is not returned from Library, no need to call
 *    \ref phhalHw_Sam_Cmd_SAM_AuthenticatePDC_Part2 "AuthenticatePDC Part2" interface.
 *
 * \retval #PH_ERR_SUCCESS_CHAINING     Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pPDChal, \b pUpgradeInfo, \b pDivInput and
 *                                      \b pPCDChalRespLen are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_AuthenticatePDC_Part1(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                                /**< [In] Key diversification selection. Option to set the P1
                                                                                         *        information byte.
                                                                                         *          - \ref PHHAL_HW_SAM_PDC_AUTH_DERIVATION_OFF
                                                                                         *            "No derivation"
                                                                                         *          - \ref PHHAL_HW_SAM_PDC_AUTH_DERIVE_UPGRADE_KEY
                                                                                         *            "Derive UpgradeKey" from targeted ICUpgradeKey given
                                                                                         *            UpgradeInfo
                                                                                         *          - \ref PHHAL_HW_SAM_PDC_AUTH_DIVERSIFY_YEAR_UPGRADE_KEY
                                                                                         *            "Diversify" the targeted YearUpgradeKey with the given
                                                                                         *             DivInput and then derive the actual UpgradeKey with
                                                                                         *             UpgradeInfo
                                                                                         */
        uint8_t bKeyNo,                                                                 /**< [In] Key number to be used from SAM. One of the following,
                                                                                         *          - NVM Key: 0x00 - 0x7F
                                                                                         *          - RAM Key: 0xE0 - 0xE3
                                                                                         */
        uint8_t bKeyVer,                                                                /**< [In] Key version to be used from SAM. */
        uint8_t * pPDChal,                                                              /**< [In] Buffer containing the challange generated by the PICC.
                                                                                         *        Authentication message from the PICC as E ( DivKey (Kx), RndB)
                                                                                         */
        uint8_t bPDChalLen,                                                             /**< [In] Length of bytes available in \b pPDChal buffer. */
        uint8_t * pUpgradeInfo,                                                         /**< [In] Upgrade information of the target product state.  */
        uint8_t bLen,                                                                   /**< [In] Length of bytes available in \b pUpgradeInfo buffer.
                                                                                         *          - 0x00: No key derivation
                                                                                         *          - 0x01 - 0x08: Key derivation involving UpgradeInfo
                                                                                         */
        uint8_t * pDivInput,                                                            /**< [In] Diversification input for generating KeyID.ICUpgrade
                                                                                         *        Key , ie.: VCUID.
                                                                                         */
        uint8_t bDivInputLen,                                                           /**< [In] Length of bytes available in \b pDivInput buffer. */
        uint8_t ** ppPCDChalResp,                                                       /**< [Out] Buffer containing the PCD challenge response to be sent to
                                                                                         *         PICC. Challenge response as E ( DivKey (Kx), RndA || RndB')
                                                                                         */
        uint16_t * pPCDChalRespLen                                                      /**< [Out] Length of bytes available in \b ppPCDChalResp buffer. */
    );

/**
 * \brief Perform a Post delivery configuration Authenticate command part2. This command
 * will decrypt the response received from PICC and will return success status if the
 * challanges matches.
 *
 * \note This interface should be called only if \ref phhalHw_Sam_Cmd_SAM_AuthenticatePDC_Part1
 * "SAM_AuthenticatePDC Part1" returns \ref PH_ERR_SUCCESS_CHAINING "Chaining Status".
 *
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pPDResp and \b pPiccReturnCode are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_AuthenticatePDC_Part2(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bPiccErrCode,                                                           /**< [In] Error Code sent by the MIFARE Plus PICC.
                                                                                         *        For success it should be 0x90.
                                                                                         */
        uint8_t * pPDResp,                                                              /**< [In] Buffer containing the input received from PICC.
                                                                                         *        PICC authentication response as E ( DivKey (Kx), RndA')
                                                                                         */
        uint8_t bPDRespLen,                                                             /**< [In] Length of bytes available in \b pPDResp buffer. */
        uint8_t * pPiccReturnCode                                                       /**< [Out] Error code returned by MIFARE Plus PICC. This will of
                                                                                         *         1 byte in length.
                                                                                         */
    );

/**
 * \brief Perform a MIFARE Plus Combined Read command. This command is used to perform GetVersion,
 * ReadSignature and all Read related operations.
 *
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pData, \b pOutputLen and \b pPiccReturnCode are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_CombinedReadMFP(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bLFI,                                                                   /**< [In] Last Frame Indication (LFI). Option for updating the
                                                                                         *        P2 information of Sam frame.
                                                                                         *          - \ref PHHAL_HW_SAM_ISO7816_LAST_FRAME "Last Frame"
                                                                                         *          - \ref PHHAL_HW_SAM_ISO7816_CHAINED_FRAME "Chainned Frame"
                                                                                         */
        uint16_t wOption,                                                               /**< [In] Buffering options and Paload Type, for exchanging information
                                                                                         *        to SAM. The options can be combined by bitwise oring.
                                                                                         *          - Buffering Options
                                                                                         *              - \ref PH_EXCHANGE_DEFAULT
                                                                                         *                     "Exchange information to SAM and Receive the response"
                                                                                         *
                                                                                         *              - \ref PH_EXCHANGE_BUFFER_FIRST
                                                                                         *                     "Buffer first" set of information. No exchange is
                                                                                         *                     performed
                                                                                         *
                                                                                         *              - \ref PH_EXCHANGE_BUFFER_CONT
                                                                                         *                     "Buffer intermediate" set of information. No exchange
                                                                                         *                     is performed
                                                                                         *
                                                                                         *              - \ref PH_EXCHANGE_BUFFER_LAST
                                                                                         *                     "Buffer last" set of information. Exchange information
                                                                                         *                     to SAM and Receive the response
                                                                                         *
                                                                                         *          - The below flags should be used to switch between command and resposne.
                                                                                         *              - \ref PHHAL_HW_SAM_OPTION_MFP_PAYLOAD_TYPE_COMMAND
                                                                                         *                     "MFP Command"
                                                                                         *              - \ref PHHAL_HW_SAM_OPTION_MFP_PAYLOAD_TYPE_RESPONSE
                                                                                         *                     "MFP Response"
                                                                                         *              - \ref PHHAL_HW_SAM_OPTION_MFP_PAYLOAD_TYPE_BOTH
                                                                                         *                     "MFP Command + Response" (only for unMAC'ed commands)
                                                                                         */
        uint8_t * pData,                                                                /**< [In] The data to be sent to SAM. Should be one of the following.
                                                                                         *          - For \ref PHHAL_HW_SAM_OPTION_MFP_PAYLOAD_TYPE_COMMAND "MFP Command":
                                                                                         *              - For READxyz  : Read Cmd (1byte) || BNR (2byte) || No.Of Blocks (1byte)
                                                                                         *              - For GetV     : GetVersion command (1byte)
                                                                                         *              - For Read_Sign: Read_Sign Cmd (1byte) || Address (1btye)
                                                                                         *
                                                                                         *          - For \ref PHHAL_HW_SAM_OPTION_MFP_PAYLOAD_TYPE_RESPONSE "MFP Response":
                                                                                         *              - 1 - 256 bytes received response from PICC (Maced / Encrypted data)
                                                                                         *              - Error Code
                                                                                         *
                                                                                         *          - For \ref PHHAL_HW_SAM_OPTION_MFP_PAYLOAD_TYPE_BOTH "MFP Command + Response":
                                                                                         *              - READxyU: Read Cmd (1byte) || BNR (2byte) || No.Of Blocks (1byte) ||
                                                                                         *                         RC || Data (N bytes) [|| MAC (8 bytes)]
                                                                                         *              - RC: Response Code
                                                                                         */
        uint8_t bDataLen,                                                               /**< [In] Length of bytes available in \b pData buffer. */
        uint8_t ** ppOutput,                                                            /**< [Out] The complete information received from SAM. This will be information that
                                                                                         *         will be exchagned with PICC. One of the following.
                                                                                         *          - If \ref PHHAL_HW_SAM_OPTION_MFP_PAYLOAD_TYPE_COMMAND "MFP Command":
                                                                                         *            MAC of command
                                                                                         *          - If not \ref PHHAL_HW_SAM_OPTION_MFP_PAYLOAD_TYPE_COMMAND "MFP Command":
                                                                                         *            [Optional, present if READEyz] Decrypted data
                                                                                         */
        uint16_t * pOutputLen,                                                          /**< [Out] Length bytes available in \b ppOutput buffer. */
        uint8_t * pPiccReturnCode                                                       /**< [Out] Error code returned by MIFARE Plus PICC. This will of 1 byte in length. */
    );

/**
 * \brief Perform a MFP CombinedWrite command. This command is common for Write, WriteValue,
 * Increment, Decrement, IncrementTransfer, DecrementTransfer, Transfer and Restore.
 *
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pData, \b pOutputLen and \b pPiccReturnCode are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_CombinedWriteMFP(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wOption,                                                               /**< [In] Buffering options and Paload Type, for exchanging information
                                                                                         *        to SAM. The options can be combined by bitwise oring.
                                                                                         *          - Buffering Options
                                                                                         *              - \ref PH_EXCHANGE_DEFAULT
                                                                                         *                     "Exchange information to SAM and Receive the response"
                                                                                         *
                                                                                         *              - \ref PH_EXCHANGE_BUFFER_FIRST
                                                                                         *                     "Buffer first" set of information. No exchange is
                                                                                         *                     performed
                                                                                         *
                                                                                         *              - \ref PH_EXCHANGE_BUFFER_CONT
                                                                                         *                     "Buffer intermediate" set of information. No exchange
                                                                                         *                     is performed
                                                                                         *
                                                                                         *              - \ref PH_EXCHANGE_BUFFER_LAST
                                                                                         *                     "Buffer last" set of information. Exchange information
                                                                                         *                     to SAM and Receive the response
                                                                                         *
                                                                                         *          - The below flags should be used to switch between command and resposne.
                                                                                         *              - \ref PHHAL_HW_SAM_OPTION_MFP_PAYLOAD_TYPE_COMMAND "MFP Command"
                                                                                         *              - \ref PHHAL_HW_SAM_OPTION_MFP_PAYLOAD_TYPE_RESPONSE "MFP Response"
                                                                                         *
                                                                                         *          - The below flags should be used to update the Plain data in response.
                                                                                         *              - \ref PHHAL_HW_SAM_OPTION_COMBINED_WRITE_RETURN_PLAIN
                                                                                         *                     "Plain data in response"
                                                                                         *                  - If \ref PHHAL_HW_SAM_OPTION_MFP_PAYLOAD_TYPE_RESPONSE
                                                                                         *                    "MFP Response" or \ref PHHAL_HW_SAM_OPTION_MFP_PAYLOAD_TYPE_COMMAND
                                                                                         *                    "MFP Command" with CmdCode different from Cmd.WRITEPy: RFU
                                                                                         *                  - If \ref PHHAL_HW_SAM_OPTION_MFP_PAYLOAD_TYPE_COMMAND "MFP Command"
                                                                                         *                    with CmdCode set to Cmd.WRITEPy: Return plain data in response
                                                                                         *              - \ref PHHAL_HW_SAM_OPTION_COMBINED_WRITE_SKIP_PLAIN
                                                                                         *                     "Skip plain data in response":
                                                                                         *                     If \ref PHHAL_HW_SAM_OPTION_MFP_PAYLOAD_TYPE_COMMAND "MFP Command"
                                                                                         *                     with CmdCode set to Cmd.WRITEPy
                                                                                         */
        uint8_t * pData,                                                                /**< [In] The data to be sent to SAM. Should be one of the following,
                                                                                         *          - For \ref PHHAL_HW_SAM_OPTION_MFP_PAYLOAD_TYPE_COMMAND "MFP Command":
                                                                                         *              - For WRITExy : Write Cmd (1byte) || BNR (2byte) || PlainData (N * 16 bytes)
                                                                                         *              - For INCx    : Inc Cmd (1byte) || BNR (2byte) || PlainData (4bytes)
                                                                                         *              - For DECx    : Dec Cmd (1byte) || BNR (2byte) || PlainData (4bytes)
                                                                                         *              - For INCTx   : Inc Transfer Cmd (1byte) || BNR (4byte) || PlainData (4bytes)
                                                                                         *              - For DECTx   : Dec Transfer Cmd (1byte) || BNR (4byte) || PlainData (4bytes)
                                                                                         *              - For TRFx    : Transfer Cmd (1byte) || BNR (2byte)
                                                                                         *              - For RESx    : Restore Cmd (1byte) || BNR (2byte)
                                                                                         *
                                                                                         *          - For \ref PHHAL_HW_SAM_OPTION_MFP_PAYLOAD_TYPE_RESPONSE "MFP Response":
                                                                                         *              - RC (1byte)
                                                                                         *              - RC (1byte) || MAC (8byte)
                                                                                         *              - RC (1byte) || TMC (4byte) || TMV (8byte)
                                                                                         *              - RC (1byte) || TMC (4byte) || TMV (8byte) || MAC (8byte)
                                                                                         */
        uint8_t bDataLen,                                                               /**< [In] Length of bytes available in \b pData buffer. */
        uint8_t ** ppOutput,                                                            /**< [Out] The complete information received from SAM. This will be information that
                                                                                         *         will be exchagned with PICC. One of the following.
                                                                                         *          - \ref PHHAL_HW_SAM_OPTION_MFP_PAYLOAD_TYPE_COMMAND "MFP Command"
                                                                                         *              - MAC: 8 bytes
                                                                                         *              - Data:
                                                                                         *                  - EncValue (16 byte) || MAC (8 byte)
                                                                                         *                  - EncData (N * 16 + 8 byte (with N = 1..15)) || MAC (8 byte
                                                                                         *          - NULL for \ref PHHAL_HW_SAM_OPTION_MFP_PAYLOAD_TYPE_RESPONSE
                                                                                         *            "MFP Response"
                                                                                         */
        uint16_t * pOutputLen,                                                          /**< [Out] Length bytes available in \b ppOutput buffer. */
        uint8_t * pPiccReturnCode                                                       /**< [Out] Error code returned by MIFARE Plus PICC. This will
                                                                                         *         of 1 byte in length.
                                                                                         */
    );

/**
 * \brief Prepare the MFP Change Key command. This command will return the protected data
 * to be written to PICC.
 *
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pData, \b pProtectedDataLen and
 *                                      \b pPiccReturnCode are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_ChangeKeyMFP(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                                /**< [In] The below flags should be for updating the P1
                                                                                         *        information byte.
                                                                                         *          - Use Key Diversification
                                                                                         *              - \ref PHHAL_HW_SAM_MFP_CHANGE_KEY_DIVERSIFICATION_OFF
                                                                                         *                     "No diversification"
                                                                                         *              - \ref PHHAL_HW_SAM_MFP_CHANGE_KEY_DIVERSIFICATION_ON
                                                                                         *                     "Diversify the used key with the given DivInput"
                                                                                         *
                                                                                         *          - Payload type
                                                                                         *              - \ref PHHAL_HW_SAM_OPTION_MFP_PAYLOAD_TYPE_COMMAND
                                                                                         *                     "MFP Command"
                                                                                         *              - \ref PHHAL_HW_SAM_OPTION_MFP_PAYLOAD_TYPE_RESPONSE
                                                                                         *                     "MFP Response"
                                                                                         */
        uint8_t * pData,                                                                /**< [In] The information to be exchanged to Sam.
                                                                                         *          - For \ref PHHAL_HW_SAM_OPTION_MFP_PAYLOAD_TYPE_COMMAND "MFP Command":
                                                                                         *            MIFARE Plus command data. The buffer should contain,
                                                                                         *              - CmdCode (1 byte): MIFARE Plus command code. Cmd.WRITEEy (0xA0, 0xA1)
                                                                                         *              - KeyBNo (2 byte): MIFARE Plus Key Block number
                                                                                         *                  - 0x4000 ... 0x4FFF, 0x8000 ... 0xAFFF, 0xCXX0, 0xCXX1
                                                                                         *                  - With X any value in 0x0..0xF. Therefore the latter two
                                                                                         *                    values cover: 0xC000, 0xC001, 0xC010, 0xC011, 0xC020, 0xC021,
                                                                                         *                    ... till 0xCFF0, 0xCFF1
                                                                                         *              - KeyNo (1 byte): Key number to be used from SAM. One of the following,
                                                                                         *                  - NVM Key: 0x00 - 0x7F
                                                                                         *                  - RAM Key: 0xE0 - 0xE3
                                                                                         *              - KeyVer (1 byte): Key version to be used from SAM.
                                                                                         *              - DivInput (1 - 31 bytes): Diversification Input. Present if \b bOption
                                                                                         *                has \ref PHHAL_HW_SAM_MFP_CHANGE_KEY_DIVERSIFICATION_ON
                                                                                         *                "Diversification Enabled".
                                                                                         *
                                                                                         *          - For \ref PHHAL_HW_SAM_OPTION_MFP_PAYLOAD_TYPE_RESPONSE "MFP Response":
                                                                                         *            MIFARE Plus response data. The buffer should contain,
                                                                                         *              - PICCReturnCode (1 byte): PICC response code
                                                                                         *              - MAC (8 byte): PICC MAC response
                                                                                         */
        uint8_t bDataLen,                                                               /**< [In] Length of bytes available in \b pData buffer. */
        uint8_t ** ppProtectedData,                                                     /**< [Out] The complete information received from SAM. This will be information that
                                                                                         *         will be exchagned with PICC. One of the following.
                                                                                         *          - \ref PHHAL_HW_SAM_OPTION_MFP_PAYLOAD_TYPE_COMMAND "MFP Command"
                                                                                         *              - Encrypted and MACed data
                                                                                         *                  - EncValue (16 byte) || MAC (8 byte)
                                                                                         *                  - EncData (N * 16 + 8 byte (with N = 1..15)) || MAC (8 byte)
                                                                                         *          - NULL for \ref PHHAL_HW_SAM_OPTION_MFP_PAYLOAD_TYPE_RESPONSE "MFP Response"
                                                                                         */
        uint16_t * pProtectedDataLen,                                                   /**< [Out] Length of bytes available in \b ppProtectedData buffer. */
        uint8_t * pPiccReturnCode                                                       /**< [Out] Error code returned by MIFARE Plus PICC. This will of
                                                                                         *         1 byte in length.
                                                                                         */
    );

/**
 * end of group phhalHw_Sam_Cmd_Plus_SMode
 * @}
 */




/** \defgroup phhalHw_Sam_Cmd_Plus_XMode X_Mode
 * \brief SAM commands used for MIFARE Plus PICC communication in X-Mode.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_Plus_XMode_Defines Defines
 * \brief Definitions for SAM commands used for MIFARE Plus communication in X-Mode.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_Plus_XMode_Defines_CommandCodes CommandCodes
 * \brief Definitions for Instruction codes
 * @{
 */
#define PHHAL_HW_SAM_CMD_INS_MFP_AUTHENTICATE                                   0x70U   /**< SAM Insturction code for MFP_Authenticate command. */
#define PHHAL_HW_SAM_CMD_INS_MFP_AUTH_SECTOR_SWITCH                             0x72U   /**< SAM Insturction code for MFP_AuthentSectorSwitch command. */
#define PHHAL_HW_SAM_CMD_INS_PDC_AUTHENTICATE                                   0x73U   /**< SAM Insturction code for PDC_Authenticate command */
#define PHHAL_HW_SAM_CMD_INS_MFP_COMBINED_READ                                  0x31U   /**< SAM Insturction code for MFP_CombinedRead command. */
#define PHHAL_HW_SAM_CMD_INS_MFP_COMBINED_WRITE                                 0x32U   /**< SAM Insturction code for MFP_CombinedWrite command. */
#define PHHAL_HW_SAM_CMD_INS_MFP_CHANGE_KEY                                     0xA5U   /**< SAM Insturction code for MFPEV1_ChangeKey command */
#define PHHAL_HW_SAM_CMD_INS_MFP_WRITE_PERSO                                    0xA8U   /**< SAM Insturction code for MFP_WritePerso command. */
/**
 * end of group phhalHw_Sam_Cmd_Plus_XMode_Defines_CommandCodes
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_Plus_XMode_Defines_SSAuthentication AuthenticateSectorSwitch
 * \brief Definitions common for MIFARE Plus X mode communication for
 * SectorSwitch Authentication interface.
 * To be used with  \ref phhalHw_Sam_Cmd_MFP_AuthSectorSwitch "MFP_AuthSectorSwitch" interface
 * @{
 */
#define PHHAL_HW_SAM_MFP_SSAUTH_BUFFER_KEY_BLOCKS                               0x10U   /**< Option to buffer the KeyBlocks information. */
#define PHHAL_HW_SAM_MFP_SSAUTH_BUFFER_DIV_INPUT                                0x20U   /**< Option to buffer the Diversification input information. */
/**
 * end of group phhalHw_Sam_Cmd_Plus_XMode_Defines_SSAuthentication
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_Plus_XMode_Defines
 * @}
 */

/**
 * \brief The MFP_Authenticate supports the PICC authentication in X-mode. It performs an
 * authentication with the PICC and upon success generates the session keys.
 *
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pPcdCapsIn, \b pDivInput, \b pPcdCapsOut,
 *                                      \b pPdCaps and \b pPiccReturnCode are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_MFP_Authenticate(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                                /**< [In] Key diversification & authentication mode selection.
                                                                                         *        Option to update the P1 information. The options can
                                                                                         *        be combined by bitwise oring.
                                                                                         *          - SL and Key derivation info
                                                                                         *              - \ref PHHAL_HW_SAM_MFP_AUTHENTICATE_SL1_NO_KDF
                                                                                         *                "No Key" Derivation information (SL1,
                                                                                         *                originality keys...)"
                                                                                         *              - \ref PHHAL_HW_SAM_MFP_AUTHENTICATE_SL3_KDF
                                                                                         *                "Security Level 3" Key Derivatin information
                                                                                         *
                                                                                         *          - Select Authentication Mode
                                                                                         *              - \ref PHHAL_HW_SAM_MFP_AUTHENTICATE_FIRST
                                                                                         *                "Authenticate First"
                                                                                         *              - \ref PHHAL_HW_SAM_MFP_AUTHENTICATE_NON_FIRST
                                                                                         *                "Authenticate NonFirst"
                                                                                         *
                                                                                         *          - Use Key Diversification
                                                                                         *              - \ref PHHAL_HW_SAM_MFP_AUTHENTICATE_DIVERSIFICATION_OFF
                                                                                         *                "No diversification"
                                                                                         *              - \ref PHHAL_HW_SAM_MFP_AUTHENTICATE_DIVERSIFICATION_ON
                                                                                         *                "Diversify the used key with the given DivInput"
                                                                                         */
        uint8_t bKeyNo,                                                                 /**< [In] Key number to be used from SAM. One of the following,
                                                                                         *          - NVM Key: 0x00 - 0x7F
                                                                                         *          - RAM Key: 0xE0 - 0xE3
                                                                                         */
        uint8_t bKeyVer,                                                                /**< [In] Key version to be used from SAM. */
        uint16_t wBlockNo,                                                              /**< [In] MIFARE Plus block number of key to be used for
                                                                                         *        authentication.
                                                                                         */
        uint8_t * pPcdCapsIn,                                                           /**< [In] Buffer containing the Input PCD capabilites. */
        uint8_t bPcdCapsInLen,                                                          /**< [In] Input PCD capabilites Length to be exchanged.
                                                                                         *        One of the following,
                                                                                         *          - If \ref PHHAL_HW_SAM_MFP_AUTHENTICATE_FIRST
                                                                                         *               "AuthenticateFirst": 0x00..0x06
                                                                                         *          - If \ref PHHAL_HW_SAM_MFP_AUTHENTICATE_NON_FIRST
                                                                                         *               "AuthenticateNonFirst": 0x00
                                                                                         */
        uint8_t * pDivInput,                                                            /**< [In] Diversification Input used to diversify the key.
                                                                                         *          - If any of diversification option is set in
                                                                                         *            \b bOption parameter, then
                                                                                         *              - 8 (if AV1 key diversification with DES)
                                                                                         *              - 16 (if AV1 key diversification with AES) bytes
                                                                                         *              - 1 to 31 bytes (if AV2 key diversification)
                                                                                         *                diversification input
                                                                                         *          - NULL otherwise
                                                                                         */
        uint8_t bDivInputLen,                                                           /**< [In] Length of bytes available in \b pDivInput buffer. */
        uint8_t * pPcdCapsOut,                                                          /**< [Out] Buffer containing the Output PCD capabilities.
                                                                                         *         This will be of 6 bytes.
                                                                                         */
        uint8_t * pPdCaps,                                                              /**< [Out] Buffer containing the Output PD capabilities.
                                                                                         *         This will be of 6 bytes.
                                                                                         */
        uint8_t * pPiccReturnCode                                                       /**< [Out] Error code returned by MIFARE Plus PICC. This will of
                                                                                         *         1 byte in length.
                                                                                         */
    );

/**
 * \brief The MFP_AuthSectorSwitch supports the X-mode procedure to switch the security level
 * of MIFARE Plus sectors on the PICC.
 *
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pKeyBlocks, \b pDivInput and
 *                                      \b pPiccReturnCode are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_MFP_AuthSectorSwitch(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wOption,                                                               /**< [In] Buffering options and Key diversification selection, for
                                                                                         *        exchanging information to SAM. The options can be combined
                                                                                         *        by bitwise oring.
                                                                                         *          - Buffering Options:
                                                                                         *              - \ref PH_EXCHANGE_DEFAULT "Default (Exchange to SAM)":
                                                                                         *                Framing the single command which has all the information,
                                                                                         *                to SAM and receive information to SAM.
                                                                                         *              - \ref PH_EXCHANGE_BUFFER_FIRST "Buffer Initial information":
                                                                                         *                For framing ISO7816  command header, SSKeyBNr, SSKeyNo,
                                                                                         *                SSKeyVer, MsKeyNo, MsKeyVer, SectorCount, KeyBlocks.
                                                                                         *                Information is not exchanged to SAM.
                                                                                         *              - \ref PH_EXCHANGE_BUFFER_CONT "Buffer Intermediate Information":
                                                                                         *                For buffering Intermediate / Final Key Blocks or First /
                                                                                         *                Intermediate DivInput. Information is not exchanged to SAM.
                                                                                         *              - \ref PH_EXCHANGE_BUFFER_LAST "Buffer Last Information and
                                                                                         *                Exchange": For buffering Final DivInput and exchanging the
                                                                                         *                information is to SAM. Also receives information from SAM.
                                                                                         *
                                                                                         *          - Key diversification selection. Option to set the P1 information
                                                                                         *            bytes.
                                                                                         *              - Apply sector number key diversification
                                                                                         *                  - \ref PHHAL_HW_SAM_MFP_SSAUTH_MASTER_SECTOR_DIV_OFF
                                                                                         *                    "No diversification"
                                                                                         *                  - \ref PHHAL_HW_SAM_MFP_SSAUTH_MASTER_SECTOR_DIV_ON
                                                                                         *                    "Diversify the master Sector Key with the given sector number"
                                                                                         *
                                                                                         *              - Use Key Diversification for Sector Keys
                                                                                         *                  - \ref PHHAL_HW_SAM_MFP_SSAUTH_SECTOR_DIV_OFF
                                                                                         *                    "No diversification"
                                                                                         *                  - \ref PHHAL_HW_SAM_MFP_SSAUTH_SECTOR_DIV_ON
                                                                                         *                    "Diversify the used keys with the given DivInput"
                                                                                         *
                                                                                         *              - Use Key Diversification for Sector Switch Key
                                                                                         *                  - \ref PHHAL_HW_SAM_MFP_SSAUTH_SECTOR_SWITCH_DIV_OFF
                                                                                         *                    "No diversification"
                                                                                         *                  - \ref PHHAL_HW_SAM_MFP_SSAUTH_SECTOR_SWITCH_DIV_ON
                                                                                         *                    "Diversify the used key with the given DivInput"
                                                                                         *
                                                                                         *          - Additional option. Will not be exchagned to SAM. It's for
                                                                                         *            internal purpose.
                                                                                         *              - \ref PHHAL_HW_SAM_MFP_SSAUTH_BUFFER_KEY_BLOCKS
                                                                                         *                "Buffer the KeyBlocks information"
                                                                                         *              - \ref PHHAL_HW_SAM_MFP_SSAUTH_BUFFER_DIV_INPUT
                                                                                         *                "Buffer the Diversification input information"
                                                                                         */
        uint16_t wSSKeyBNr,                                                             /**< [In] PICC Sector Switch key block number to be used for
                                                                                         *        authentication.
                                                                                         */
        uint8_t bSSKeyNo,                                                               /**< [In] Sector Switch Key number to be used from SAM (that is: Kss selection).
                                                                                         *        One of the following,
                                                                                         *          - NVM Key: 0x00 - 0x7F
                                                                                         *          - RAM Key: 0xE0 - 0xE3
                                                                                         */
        uint8_t bSSKeyVer,                                                              /**< [In] Sector Switch Key version to be used from SAM (that
                                                                                         *        is: Kss selection)
                                                                                         */
        uint8_t bMSKeyNo,                                                               /**< [In] Master Key number to be used from SAM One of the following,
                                                                                         *          - NVM Key: 0x00 - 0x7F
                                                                                         *          - RAM Key: 0xE0 - 0xE3
                                                                                         */
        uint8_t bMSKeyVer,                                                              /**< [In] Master Key version to be used from SAM. */
        uint8_t bSectorCount,                                                           /**< [In] Number of sectors to be switched inside the PICC.
                                                                                         *          - 0x01 - 0x2A: If \ref PHHAL_HW_SAM_MFP_SSAUTH_MASTER_SECTOR_DIV_OFF
                                                                                         *            "No sector number key diversification"
                                                                                         *          - 0x01 - 0x37: If \ref PHHAL_HW_SAM_MFP_SSAUTH_MASTER_SECTOR_DIV_ON
                                                                                         *            "Sector number key diversification"
                                                                                         */
        uint8_t * pKeyBlocks,                                                           /**< [In] Buffer containing the PICC KeyB block number and reference key number
                                                                                         *        and version to be used form SAM. Should contain the following,
                                                                                         *          - KeyBxBNr: 2 byte Key Block sector number on the PICC (encoded MSB
                                                                                         *                      first)
                                                                                         *          - KeyNo: If \ref PHHAL_HW_SAM_MFP_SSAUTH_MASTER_SECTOR_DIV_OFF
                                                                                         *            "no sector number key diversification", SAM Key Number of the block
                                                                                         *            sector key (that is: Kn selection).
                                                                                         *              - NVM Key: 0x00 - 0x7F
                                                                                         *              - RAM Key: 0xE0 - 0xE3
                                                                                         *          - KeyVer: If \ref PHHAL_HW_SAM_MFP_SSAUTH_MASTER_SECTOR_DIV_OFF
                                                                                         *            "no sector number key diversification", SAM Key Version of the block
                                                                                         *            sector key (that is: Kn selection).
                                                                                         *
                                                                                         *          - The format of the buffer should be,
                                                                                         *            KeyBxBNr_1 || [KeyNo_1] || [KeyVer_1] || KeyBxBNr_2 || [KeyNo_2] || [KeyVer_2]
                                                                                         *            || ... || KeyBxBNr_N || [KeyNo_N] || [KeyVer_N], Where N equal to the
                                                                                         *            \b bSectorCount information.
                                                                                         */
        uint8_t bKeyBlocksLen,                                                          /**< [In] Length of bytes available in \b pKeyBlocks buffer. It should
                                                                                         *        be equal to (KeyBxBNr + [KeyNo] + [KeyVer]) * \b bSectorCount
                                                                                         */
        uint8_t * pDivInput,                                                            /**< [In] Buffer containing the diversification inputs to be used for diversifying the key.
                                                                                         *          - SSKeyDivLen: Optional, present if \ref PHHAL_HW_SAM_MFP_SSAUTH_SECTOR_SWITCH_DIV_ON
                                                                                         *            "Sector Switch Diversification is enabled", Length of the Sector Switch Key DivInput
                                                                                         *            (and of Sector Keys DivInput if using same)
                                                                                         *          - SSKeyDivInput: Optional, present if \ref PHHAL_HW_SAM_MFP_SSAUTH_SECTOR_SWITCH_DIV_ON
                                                                                         *            "Sector Switch Diversification is enabled", 1 to 31 bytes diversification input for
                                                                                         *            Sector Switch Key (and for Sector Keys if using same)
                                                                                         *          - KeyBxDivLen: Optional, present if \ref PHHAL_HW_SAM_MFP_SSAUTH_SECTOR_DIV_ON
                                                                                         *            "Sector Diversification is enabled". One of the following,
                                                                                         *              - Using same DivInput for Sector Switch Key and Sector Keys: 0x00
                                                                                         *              - Using different DivInput for Sector keys: 0x01 - 0x1F
                                                                                         *          - KeyBxDivInput: Optional, present if \ref PHHAL_HW_SAM_MFP_SSAUTH_SECTOR_DIV_ON
                                                                                         *            "Sector Diversification is enabled" and KeyBxDivLen != 0x00.
                                                                                         *            1 to 31 bytes diversification input for Sector Keys
                                                                                         *
                                                                                         *          - The format of the buffer should be,
                                                                                         *            [[SSKeyDivLen] || [SSKeyDivInput]] || [[KeyBxDivLen] || [KeyBxDivInput]]
                                                                                         */
        uint8_t bDivInputLen,                                                           /**< [In] Length of bytes available in \b pDivInput buffer. */
        uint8_t * pPiccReturnCode                                                       /**< [Out] Error code returned by MIFARE Plus PICC. This will of 1 byte in length. */
    );

/**
 * \brief The PDC_Authenticate supports the X-mode procedure to perform the
 * Post-Delivery configuration on the PICC
 *
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pUpgradeInfo, \b pDivInput and
 *                                      \b pPiccReturnCode are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_PDC_Authenticate(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                                /**< [In] Key diversification selection. Option to set the P1
                                                                                         *        information byte.
                                                                                         *          - \ref PHHAL_HW_SAM_PDC_AUTH_DERIVATION_OFF
                                                                                         *            "No derivation"
                                                                                         *          - \ref PHHAL_HW_SAM_PDC_AUTH_DERIVE_UPGRADE_KEY
                                                                                         *            "Derive UpgradeKey" from targeted ICUpgradeKey given
                                                                                         *            UpgradeInfo
                                                                                         *          - \ref PHHAL_HW_SAM_PDC_AUTH_DIVERSIFY_YEAR_UPGRADE_KEY
                                                                                         *            "Diversify" the targeted YearUpgradeKey with the given
                                                                                         *             DivInput and then derive the actual UpgradeKey with
                                                                                         *             UpgradeInfo
                                                                                         */
        uint8_t bKeyNo,                                                                 /**< [In] Key number to be used from SAM. One of the following,
                                                                                         *          - NVM Key: 0x00 - 0x7F
                                                                                         *          - RAM Key: 0xE0 - 0xE3
                                                                                         */
        uint8_t bKeyVer,                                                                /**< [In] Key version to be used from SAM. */
        uint16_t wUpgradeKey,                                                           /**< [In] PICC UpgradeKey to be used for authentication. */
        uint8_t * pUpgradeInfo,                                                         /**< [In] Upgrade information of the target product state.  */
        uint8_t bLen,                                                                   /**< [In] Length of bytes available in \b pUpgradeInfo buffer.
                                                                                         *          - 0x00: No key derivation
                                                                                         *          - 0x01 - 0x08: Key derivation involving UpgradeInfo
                                                                                         */
        uint8_t * pDivInput,                                                            /**< [In] Diversification input for generating KeyID.ICUpgrade
                                                                                         *        Key , ie.: VCUID.
                                                                                         */
        uint8_t bDivInputLen,                                                           /**< [In] Length of bytes available in \b pDivInput buffer. */
        uint8_t * pPiccReturnCode                                                       /**< [Out] Error code returned by MIFARE Plus PICC. This will of 1 byte
                                                                                         *         in length.
                                                                                         */
    );

/**
 * \brief Perform a MIFARE Plus Combined Read command in X mode. This command is used to
 * perform GetVersion, ReadSignature and all Read related operations.
 *
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_SUCCESS_CHAINING     Successfull Chaining operation.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pReadCmd, \b pDataLen and
 *                                      \b pPiccReturnCode are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_MFP_CombinedRead(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wOption,                                                               /**< [In] Option for receiving the next data chunk if
                                                                                         *        previous status from SAM was chaining.
                                                                                         *          - \ref PH_EXCHANGE_DEFAULT "Exchange Information" to
                                                                                         *                 MIFARE Plus PICC and receive response.
                                                                                         *          - \ref PH_EXCHANGE_RXCHAINING "Receive remaining"
                                                                                         *                 information from MIFARE Plus PICC.
                                                                                         */
        uint8_t * pReadCmd,                                                             /**< [In] The different types of command to be sent to
                                                                                         *        MIFARE Plus PICC and received the response.
                                                                                         *          - GetVersion: GetVersion cmd (1byte)
                                                                                         *          - ReadSig   : Read Signature cmd (1byte) + Address (1byte)
                                                                                         *          - Read      : Read cmd (1byte) + BlockNr (2byte) + NoBlocks (1byte)
                                                                                         */
        uint8_t bReadCmdLen,                                                            /**< [In] Length of bytes available in \b pReadCmd buffer. */
        uint8_t ** ppData,                                                              /**< [Out] The information returned by SAM for the
                                                                                         *         mentioned command in \b pReadCmd buffer.
                                                                                         */
        uint16_t * pDataLen,                                                            /**< [Out] Length of bytes available in \b ppData buffer. */
        uint8_t * pPiccReturnCode                                                       /**< [Out] Error code returned by MIFARE Plus PICC. This will
                                                                                         *         of 1 byte in length.
                                                                                         */
    );

/**
 * \brief Perform a MIFARE Plus Combined Write command in X mode. This command performs
 * Write, Increment, Decrement, Transfer, Restore, IncrementTransfer and DecrementTransfer
 * commands of the PICC.
 *
 * \return Status code
 *
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pReadCmd, \b pDataLen and
 *                                      \b pPiccReturnCode are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_MFP_CombinedWrite(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wOption,                                                               /**< [In] Option flag to buffer the payload information.
                                                                                         *          - \ref PH_EXCHANGE_DEFAULT
                                                                                         *                 "Exchange information to SAM and Receive the response"
                                                                                         *
                                                                                         *          - \ref PH_EXCHANGE_BUFFER_FIRST
                                                                                         *                 "Buffer first" set of information. No exchange is
                                                                                         *                 performed
                                                                                         *
                                                                                         *          - \ref PH_EXCHANGE_BUFFER_CONT
                                                                                         *                 "Buffer intermediate" set of information. No exchange
                                                                                         *                 is performed
                                                                                         *
                                                                                         *          - \ref PH_EXCHANGE_BUFFER_LAST
                                                                                         *                 "Buffer last" set of information. Exchange information
                                                                                         *                  to SAM and Receive the response
                                                                                         */
        uint8_t * pData,                                                                /**< [In] The different types of command to be sent to
                                                                                         *        MIFARE Plus PICC and received the response,
                                                                                         *          - For WRITExy : Write Cmd (1byte) || BNR (2byte) ||
                                                                                         *                          PlainData (N * 16 bytes)
                                                                                         *          - For INCx    : Inc Cmd (1byte) || BNR (2byte) ||
                                                                                         *                          PlainData (4bytes)
                                                                                         *          - For DECx    : Dec Cmd (1byte) || BNR (2byte) ||
                                                                                         *                          PlainData (4bytes)
                                                                                         *          - For INCTx   : Inc Transfer Cmd (1byte) || BNR (4byte)
                                                                                         *                          || PlainData (4bytes)
                                                                                         *          - For DECTx   : Dec Transfer Cmd (1byte) || BNR (4byte)
                                                                                         *                          || PlainData (4bytes)
                                                                                         *          - For TRFx    : Transfer Cmd (1byte) || BNR (2byte)
                                                                                         *          - For RESx    : Restore Cmd (1byte) || BNR (2byte)
                                                                                         */
        uint8_t bDataLen,                                                               /**< [In] Length of bytes available in \b pData buffer. */
        uint8_t * pTMC,                                                                 /**< [Out] Only available is the block is a TMProtected block.
                                                                                         *         The buffer will have 4 bytes of Transaction MAC counter information.
                                                                                         */
        uint8_t * pTMV,                                                                 /**< [Out] Only available is the block is a TMProtected block.
                                                                                         *         The buffer will have 8 bytes of Transaction MAC value.
                                                                                         */
        uint8_t * pPiccReturnCode                                                       /**< [Out] Error code returned by MIFARE Plus PICC. This will
                                                                                         *         of 1 byte in length.
                                                                                         */
    );

/**
 * \brief Perform a MIFARE Plus Change Key command in X mode.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pDivInput and \b pPiccReturnCode are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_MFP_ChangeKey(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                                /**< [In] The below flags should be for updating the P1
                                                                                         *        information byte.
                                                                                         *          - \ref PHHAL_HW_SAM_MFP_CHANGE_KEY_DIVERSIFICATION_OFF
                                                                                         *                 "No diversification"
                                                                                         *          - \ref PHHAL_HW_SAM_MFP_CHANGE_KEY_DIVERSIFICATION_ON
                                                                                         *                 "Diversify the used key with the given DivInput"
                                                                                         */
        uint8_t bCmdCode,                                                               /**< [In] The MIFARE Plus Write (Cmd.WRITEEy) command code
                                                                                         *        to be used for writting the key. (0xA0 or 0xA1)
                                                                                         */
        uint16_t wBlockNo,                                                              /**< [In] MIFARE Plus Key Block number
                                                                                         *          - 0x4000 ... 0x4FFF, 0x8000 ... 0xAFFF, 0xCXX0, 0xCXX1
                                                                                         *          - With X any value in 0x0..0xF. Therefore the latter two
                                                                                         *            values cover: 0xC000, 0xC001, 0xC010, 0xC011, 0xC020, 0xC021,
                                                                                         *            ... till 0xCFF0, 0xCFF1
                                                                                         */
        uint8_t bKeyNo,                                                                 /**< [In] Key number to be used from SAM. One of the following,
                                                                                         *          - NVM Key: 0x00 - 0x7F
                                                                                         *          - RAM Key: 0xE0 - 0xE3.
                                                                                         */
        uint8_t bKeyVer,                                                                /**< [In] Key version to be used from SAM. */
        uint8_t * pDivInput,                                                            /**< [In] Diversification Input used to diversify the key. */
        uint8_t bDivInputLen,                                                           /**< [In] Length of bytes available in \b pDivInput buffer. */
        uint8_t * pPiccReturnCode                                                       /**< [Out] Error code returned by MIFARE Plus PICC. This will
                                                                                         *         of 1 byte in length.
                                                                                         */
    );

/**
 * \brief MFP_WritePerso is a multi-block write command. It efficiently performs in X-mode up
 * to 13 MIFARE Plus PICC WritePerso commands targeting one block each. If more than 13 blocks
 * are to be updated, several MFP_WritePerso are to be issued. The blocks are transmitted to
 * the SAM in one MFP_WritePerso , then the SAM shall execute the many required PICC WritePerso
 * commands.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pBlocks and \b pPiccReturnCode are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_MFP_WritePerso(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wOption,                                                               /**< [In] Option flag to buffer the payload information.
                                                                                         *          - \ref PH_EXCHANGE_DEFAULT
                                                                                         *                 "Exchange information to SAM and Receive the response"
                                                                                         *
                                                                                         *          - \ref PH_EXCHANGE_BUFFER_FIRST
                                                                                         *                 "Buffer first" set of information. No exchange is
                                                                                         *                 performed
                                                                                         *
                                                                                         *          - \ref PH_EXCHANGE_BUFFER_CONT
                                                                                         *                 "Buffer intermediate" set of information. No exchange
                                                                                         *                 is performed
                                                                                         *
                                                                                         *          - \ref PH_EXCHANGE_BUFFER_LAST
                                                                                         *                 "Buffer last" set of information. Exchange information
                                                                                         *                  to SAM and Receive the response
                                                                                         */
        uint8_t * pBlocks,                                                              /**< [In] Buffer containing the Block and Data pair to be written
                                                                                         *        to MIFAER Plus PICC by SAM. Should be holding an
                                                                                         *        array of block number and data like,
                                                                                         *          - Block Addr: MIFARE Plus Block address (2 bytes)
                                                                                         *          - BlockData: MIFARE Plus Block data to be written at
                                                                                         *            the address (16 bytes)
                                                                                         *          - Ex: BNR_1 || Data || BNR_2 || Data || BNR_3 || Data ||
                                                                                         *            ... || BNR_N || Data BNR_x should be 2 bytes and Data
                                                                                         *            should 16 bytes.
                                                                                         */
        uint8_t bBlocksLen,                                                             /**< [In] Length of bytes available in \b pBlocks buffer. */
        uint8_t * pPiccReturnCode                                                       /**< [Out] Error code returned by MIFARE Plus PICC. This will
                                                                                         *         of 1 byte in length.
                                                                                         */
    );

/**
 * end of group phhalHw_Sam_Cmd_Plus_XMode
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_Plus
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd
 * @}
 */

#endif /* NXPBUILD__PHHAL_HW_SAM */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHHALHW_SAM_CMD_PLUS_H */
