/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) MIFARE DUOX command implementation of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7360 $
 * $Date: 2025-06-26 16:00:10 +0530 (Thu, 26 Jun 2025) $
 */


#ifndef PHHALHW_SAM_CMD_DUOX_H
#define PHHALHW_SAM_CMD_DUOX_H

#include <ph_Status.h>
#include <phhalHw.h>

#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */

#ifdef NXPBUILD__PHHAL_HW_SAM

#define PHHAL_HW_CMD_SAM_AUTH_METHOD_NO_CERT_A_MASK                             0x20U   /* Mask for DUOX Authentication to not include
                                                                                         * Certificate.A
                                                                                         */
#define PHHAL_HW_CMD_SAM_AUTH_METHOD_CERT_B_OPT_MASK                            0x80U   /**< Mask for DUOX Authentication to include
                                                                                         *   Certificate.B Options.
                                                                                         */
#define PHHAL_HW_SAM_DUOX_PICC_NATIVE_STATUS_LEN                                0x01U   /* MIFARE DUOX PICC native status Length */
#define PHHAL_HW_SAM_DUOX_PICC_WRAPPED_STATUS_LEN                               0x02U   /* MIFARE DUOX PICC wrapped status Length */

/** \addtogroup phhalHw_Sam_Cmd
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_DUOX 08_MIFARE_DUOX
 * \brief Implementation of SAM MIFARE DUOX commands.
 * Has implementation for both X and S mode communication.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_DUOX_Defines Defines
 * \brief Definitions common for MIFARE DESFire S and X mode communication
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_DUOX_Defines_CertBOption CertB Option
 * \brief Definitions for MIFARE DUOX CertificateB Processing Option.
 * To be used with below mentioned interfaces,
 *  - \ref phhalHw_Sam_Cmd_SAM_MutualAuthEcc_Part1 "MutualAuthEcc_Part1"
 *  - \ref phhalHw_Sam_Cmd_DUOX_MutualAuthEcc "DUOX_MutualAuthEcc"
 * @{
 */
#define PHHAL_HW_CMD_SAM_CERTB_OPTION_MSB_B_PLAIN                               0x00U   /**< SAM DUOX Certificate B processing option
                                                                                         *   as No Msg.B processing by SAM - Return
                                                                                         *   full Msg.B.Plain
                                                                                         */
#define PHHAL_HW_CMD_SAM_CERTB_OPTION_NO_DATA                                   0x01U   /**< SAM DUOX Certificate B processing option
                                                                                         *   as Sig.B and (if present) Cert.B certificate
                                                                                         *   (chain) validated by SAM - No data returned
                                                                                         */
#define PHHAL_HW_CMD_SAM_CERTB_OPTION_TBS_CERTB                                 0x02U   /**< SAM DUOX Certificate B processing option
                                                                                         *   as Sig.B and (if present) Cert.B certificate
                                                                                         *   (chain) validated by SAM - Return TBSCert.B
                                                                                         */
/**
 * end of group phhalHw_Sam_Cmd_DUOX_Defines_CertBOption
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_DUOX_Defines_ProtOption Protocol_Option
 * \brief Definitions for MIFARE DUOX Protocol Options. To be used with
 * \ref phhalHw_Sam_Cmd_SAM_UnilatAuthEcc_Part1 "SAM_UnilatAuthEcc Part1"
 * @{
 */
#define PHHAL_HW_CMD_SAM_PROTOCOL_OPTION_ISO_INTERNAL_AUTH                      0x00U   /**< SAM DUOX protocol option as no ISOInternal
                                                                                         *   Authentication.
                                                                                         */
#define PHHAL_HW_CMD_SAM_PROTOCOL_OPTION_VDE_ECDSA_SIGN                         0x01U   /**< SAM DUOX protocol option as VDE ECDSA Signing */
/**
 * end of group phhalHw_Sam_Cmd_DUOX_Defines_ProtOption
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_DUOX_Defines
 * @}
 */





/** \defgroup phhalHw_Sam_Cmd_DUOX_SMode S_Mode
 * \brief Implementation of SAM commands used for MIFARE DUOX communication in S-Mode.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_DUOX_SMode_Defines Defines
 * \brief Definitions for SAM commands used for MIFARE DUOX communication in S-Mode.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_DUOX_SMode_Defines_CommandCodes CommandCodes
 * \brief Definitions for Instruction codes
 * @{
 */
#define PHHAL_HW_SAM_CMD_INS_SAM_MUTUAL_AUTH_ECC                                0xD9U   /**< SAM Instruction code for SAM_MutualAuthECC command. */
#define PHHAL_HW_SAM_CMD_INS_SAM_UNILATERAL_AUTH_ECC                            0xD0U   /**< SAM Instruction code for SAM_UnilatAuthEcc command. */
#define PHHAL_HW_SAM_CMD_INS_SAM_BIND_CERTIFICATE                               0xB3U   /**< SAM Instruction code for SAM_BindCertificate command. */
#define PHHAL_HW_SAM_CMD_INS_SAM_IMPORT_ECC_KEY_DUOX                            0xB5U   /**< SAM Instruction code for SAM_ImportEccKeyDUOX command. */
/**
 * end of group phhalHw_Sam_Cmd_DUOX_SMode_Defines_CommandCodes
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_DUOX_SMode_Defines_AuthMethod AuthMethod
 * \brief Definitions for MIFARE DUOX Authentication methods. To be used with
 * \ref phhalHw_Sam_Cmd_SAM_MutualAuthEcc_Part1 "SAM_MutualAuthEcc Part1"
 * @{
 */
#define PHHAL_HW_CMD_SAM_AUTH_METHOD_MUTUAL_AUTH_CERT                           0x80U   /**< SAM DUOX Authentication method as ASymmetric
                                                                                         *   Mutual Authentication with Certificate.A included
                                                                                         */
#define PHHAL_HW_CMD_SAM_AUTH_METHOD_MUTUAL_AUTH_NO_CERT                        0xA0U   /**< SAM DUOX Authentication method as ASymmetric
                                                                                         *   Mutual Authentication with Certificate.A not
                                                                                         *   included
                                                                                         */
#define PHHAL_HW_CMD_SAM_AUTH_METHOD_READER_UNILATERAL_AUTH_CERT                0x40U   /**< SAM DUOX Authentication method as ASymmetric
                                                                                         *   Reader-Unilateral Authentication with Certificate.A
                                                                                         *   included
                                                                                         */
#define PHHAL_HW_CMD_SAM_AUTH_METHOD_READER_UNILATERAL_AUTH_NO_CERT             0x60U   /**< SAM DUOX Authentication method as ASymmetric
                                                                                         *   Reader-Unilateral Authentication with Certificate.A
                                                                                         *   not included
                                                                                         */
/**
 * end of group phhalHw_Sam_Cmd_DUOX_SMode_Defines_AuthMethod
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_DUOX_SMode_Defines
 * @}
 */

/**
 * \brief The SAM_MutualAuthEcc supports in S-mode the ECC-based Mutual and Reader-Unilateral
 * Authentication as implemented by MIFARE DUOX with the ISOGeneralAuthenticate command.
 * Generates its ephemeral key pair for the ECDH-based key agreement and returns the public key
 *
 * \note
 *  - Post calling this interface it's must to call \ref phhalHw_Sam_Cmd_SAM_MutualAuthEcc_Part2
 *    "MutualAuthEcc Part2" and \ref phhalHw_Sam_Cmd_SAM_MutualAuthEcc_Part3 "MutualAuthEcc Part3"
 *     to place SAM into proper completion state if \ref PH_ERR_SUCCESS_CHAINING "Chaining Status"
 *     is returned from Library. Otherwise error will be returned by SAM for any subsequent commands.
 *  - If \ref PH_ERR_SUCCESS_CHAINING "Chaining Status" is not returned from Library, no need to call
 *      - \ref phhalHw_Sam_Cmd_SAM_MutualAuthEcc_Part2 "MutualAuthEcc Part2" interface.
 *      - \ref phhalHw_Sam_Cmd_SAM_MutualAuthEcc_Part3 "MutualAuthEcc Part3" interface.
 *
 * \retval #PH_ERR_SUCCESS_CHAINING     Operation successful with Chaining response.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pSamRespLen is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_MutualAuthEcc_Part1(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                                /**< [In] Option to update the P1 information.
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_AUTH_METHOD_MUTUAL_AUTH_CERT
                                                                                         *                 "ASymmetric Mutual Authentication" with Certificate.A
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_AUTH_METHOD_MUTUAL_AUTH_NO_CERT
                                                                                         *                 "ASymmetric Mutual Authentication" without Certificate.A
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_AUTH_METHOD_READER_UNILATERAL_AUTH_CERT
                                                                                         *                 "ASymmetric Reader-Unilateral Authentication" with
                                                                                         *                 Certificate.A
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_AUTH_METHOD_READER_UNILATERAL_AUTH_NO_CERT
                                                                                         *                 "ASymmetric Reader-Unilateral Authentication" without
                                                                                         *                 Certificate.A
                                                                                         */
        uint8_t bECCKeyNo_Priv,                                                         /**< [In] ECC key entry holding the private key to be used for signing.
                                                                                         *        Will be 0x00 - 0x0F
                                                                                         */
        uint8_t bCertA_FileNo,                                                          /**< [In] The file number of the file holding the Cert.A certificate
                                                                                         *        (chain). Present in command frame only if \b bOption has one
                                                                                         *        of the following,
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_AUTH_METHOD_MUTUAL_AUTH_CERT
                                                                                         *                 "ASymmetric Mutual Authentication" with Certificate.A
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_AUTH_METHOD_READER_UNILATERAL_AUTH_CERT
                                                                                         *                 "ASymmetric Reader-Unilateral Authentication" with
                                                                                         *                 Certificate.A
                                                                                         */
        uint8_t bCertB_Options,                                                         /**< [In] Cert.B processing options. One of the following
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_CERTB_OPTION_MSB_B_PLAIN
                                                                                         *            "No Msg.B processing by SAM". Return full Msg.B.Plain
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_CERTB_OPTION_NO_DATA
                                                                                         *            "Sig.B and (if present) Cert.B certificate (chain)" validated by SAM
                                                                                         *            but there is no response from SAM.
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_CERTB_OPTION_TBS_CERTB
                                                                                         *            "Sig.B and (if present) Cert.B certificate (chain)" validated by SAM
                                                                                         *            and there no response from SAM.
                                                                                         */
        uint8_t bECCKeyNo_CA,                                                           /**< [In] ECC key entry holding the public key of the targeted CA Root Key
                                                                                         *        for Cert.B validation. Will be 0x00 - 0x0F. Present in command
                                                                                         *        frame only if \b bOption has one of the following,
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_AUTH_METHOD_MUTUAL_AUTH_CERT
                                                                                         *                 "ASymmetric Mutual Authentication" with Certificate.A
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_AUTH_METHOD_READER_UNILATERAL_AUTH_CERT
                                                                                         *                 "ASymmetric Reader-Unilateral Authentication" with
                                                                                         *                 Certificate.A
                                                                                         */
        uint8_t ** ppSamResponse,                                                       /**< [Out] Ephemeral public key in uncompressed point representation
                                                                                         *         ( 0x04 || E.Pub.A.x || E.Pub.A.y )
                                                                                         */
        uint16_t * pSamRespLen                                                          /**< [Out] Length of bytes available in \b ppSamResponse buffer. */
    );

/**
 * \brief Performs second part, SAM receives the ephemeral public key of the MIFARE DUOX,
 * executes the ephemeral key agreement and returns an encrypted message containing its
 * certificate chain and signature. Also the status code of PICC will be returned to the
 * caller in case of error.
 *
 * \note This interface should be called only if \ref phhalHw_Sam_Cmd_SAM_MutualAuthEcc_Part1
 * "MutualAuthEcc Part1" returns \ref PH_ERR_SUCCESS_CHAINING "Chaining Status".
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS_CHAINING                     Operation successful with Chaining response.
 * \retval #PHHAL_HW_SAM_ERR_OK_CHAINING_ACTIVE_DUOX    Correct Execution with Cert.A - final frame.
 * \retval #PH_ERR_AUTH_ERROR                           Authentication failed: public key validation
 *                                                      for E.Pub.B failed.
 * \retval #PH_ERR_INVALID_DATA_PARAMS                  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER                    \b pCardResponse and \b pSamRespLen are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_MutualAuthEcc_Part2(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                                /**< [In] Buffering options.
                                                                                         *          - \ref PH_EXCHANGE_DEFAULT "Default": To buffer
                                                                                         *            the initial command information. Use this flag
                                                                                         *            exchange Ephemeral public key (E.Pub.B) or PICC
                                                                                         *            Status code and start reception of encrypted
                                                                                         *            message (Msg.A.enc).
                                                                                         *
                                                                                         *          - To receive remaining \ref PH_EXCHANGE_RXCHAINING
                                                                                         *            "Encrypted Message (Msg.A.enc)"
                                                                                         */
        uint8_t * pPiccErrorCode,                                                       /**< [In] Status code returned by the PICC in case of failure.
                                                                                         *        Should be 2 byte in length with LSB first.
                                                                                         */
        uint8_t * pCardResponse,                                                        /**< [In] Ephemeral public key in uncompressed point representation
                                                                                         *        ( 0x04 || E.Pub.B.x || E.Pub. B.y )
                                                                                         */
        uint8_t bCardRespLen,                                                           /**< [In] Length of bytes available in \b pCardResponse buffer. */
        uint8_t ** ppSamResponse,                                                       /**< [Out] Enciphered Signature and Certificate (Chain, Options)
                                                                                         *          - If AuthMethod = 0x80 or 0x40, Correct execution with
                                                                                         *            Certificate.A.\n
                                                                                         *            Msg.A.enc = Msg.A.enc.0 || Msg.A.enc.1 || .. ||
                                                                                         *            Msg.A.enc.n = E(KSesAuthENC, 0xE0E0 || Cert.A ||
                                                                                         *            Sig.A)
                                                                                         *          - If AuthMethod = 0xA0 or 0x60, Correct execution without
                                                                                         *            Certificate.A.\n
                                                                                         *            Msg.A.enc = E(KSesAuthENC, 0xE0E0 || Sig.A)
                                                                                         */
        uint16_t * pSamRespLen,                                                         /**< [Out] Length of bytes available in \b ppSamResponse buffer. */
        uint8_t * pPiccRetCode                                                          /**< [Out] Status code from MIFARE DUOX PICC if available else zero. */
    );

/**
 * \brief Performs second part, the SAM receives the encrypted message from the MIFARE DUOX
 * and, depending on the protocol choice, returns the decrypted OptsB (Reader-Unilateral
 * Authentication) or additionally validates and/or returns the decrypted certificate chain
 * and signature of the MIFARE DUOX. Also the status code of PICC will be returned to the
 * caller in case of error.
 *
 * \note This interface should be called only if \ref phhalHw_Sam_Cmd_SAM_MutualAuthEcc_Part1
 * "MutualAuthEcc Part1" and \ref phhalHw_Sam_Cmd_SAM_MutualAuthEcc_Part2 "MutualAuthEcc Part2"
 * returns \ref PH_ERR_SUCCESS_CHAINING "Chaining Status".
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_SUCCESS_CHAINING     Operation successful with Chaining response.
 * \retval #PH_ERR_AUTH_ERROR           Authentication failed
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pPiccErrorCode, \b pCardResponse \b pSamRespLen and
 *                                      \b pPiccRetCode are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_MutualAuthEcc_Part3(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                                /**< [In] Buffering options.
                                                                                         *          - \ref PH_EXCHANGE_DEFAULT "Default": To buffer
                                                                                         *            the initial command information. Use this flag
                                                                                         *            exchange Enciphered response from PICC (Msg.B.enc)
                                                                                         *            and start reception of response.
                                                                                         *
                                                                                         *          - To receive remaining \ref PH_EXCHANGE_RXCHAINING
                                                                                         *            "Mutual Authentication response"
                                                                                         */
        uint8_t * pPiccErrorCode,                                                       /**< [In] Status code returned by the PICC in case of failure.
                                                                                         *        Should be 2 byte in length with LSB first.
                                                                                         */
        uint8_t * pCardResponse,                                                        /**< [In] Response from PICC. One of the following
                                                                                         *          - If no PICC error and AuthMethod = 0x40 or 0x40,
                                                                                         *            Reader-unilateral authentication Enciphered response
                                                                                         *            from PICC (length X = 16):\n
                                                                                         *            Msg.B.enc = E(K_{SesAuthEN }, 0xE1E1 || OptsB)
                                                                                         *
                                                                                         *          - If no PICC error and AuthMethod = 0x80 or 0xA0 and
                                                                                         *            CertificateB Options == 0x00, Mutual Authentication
                                                                                         *            with no SAM validation. Enciphered response from PICC
                                                                                         *            (no limitation on accumulated length Sum(X))\n
                                                                                         *            Msg.B.enc = E(K_{SesAuthEN }, 0xE1E1 || OptsB ||
                                                                                         *            Cert.B||Sig.B)
                                                                                         *
                                                                                         *          - If no PICC error and AuthMethod = 0x80 or 0xA0 and
                                                                                         *            CertificateB Options != 0x00, Mutual Authentication
                                                                                         *            with SAM validation. Enciphered response from PICC
                                                                                         *            ( Sum(X) le 960 or 1024 or ... )\n
                                                                                         *            Msg.B.enc = E(KSesAuth ENC, 0xE1E1 || OptsB || Cert.B
                                                                                         *            || Sig.B)
                                                                                         */
        uint16_t wCardRespLen,                                                          /**< [In] Length of bytes available in \b pCardResponse buffer. */
        uint8_t ** ppSamResponse,                                                       /**< [Out] One of the following,
                                                                                         *          - If AuthMethod = 0x80 or 0xA0 and CertificateB Options
                                                                                         *            == 0x00, Mutual Authentication with no SAM validation. \n
                                                                                         *            Opts.B || Cert.B || Sig.B
                                                                                         *          - If AuthMethod = 0x80 or 0xA0 and CertificateB Options
                                                                                         *            == 0x02, Mutual authentication - SAM validation - more
                                                                                         *            data available. \n
                                                                                         *            Opts.B || TBSCert.B.parent || TBSCert.B.leaf
                                                                                         *          - If AuthMethod = 0x80 or 0xA0 and CertificateB Options
                                                                                         *            == 0x01, Mutual authentication - SAM validation - no data
                                                                                         *            returned.\n
                                                                                         *            Opts.B
                                                                                         *          - If AuthMethod = 0x40 or 0x60, Reader-unilateral
                                                                                         *            authentication.\n
                                                                                         *            Opts.B
                                                                                         */
        uint16_t * pSamRespLen,                                                         /**< [Out] Length of bytes available in \b ppSamResponse buffer. */
        uint8_t * pPiccRetCode                                                          /**< [Out] Status code from MIFARE DUOX PICC if available else zero. */
    );

/**
 * \brief The SAM_UnilatAuthEcc supports in S-mode the ECC-based Card-Unilateral
 * Authentication as implemented by MIFARE DUOX with the ISOInternalAuthenticate
 * or VDE_ECDSASign command. In the first part the SAM generates and returns a
 * random challenge. The actual protocol choice is provided to the SAM via the
 * P1 (\b bOpton) parameter
 *
 * \note
 *  - Post calling this interface it's must to call \ref phhalHw_Sam_Cmd_SAM_UnilatAuthEcc_Part2
 *    "UnilatAuthEcc Part2"  to place SAM into proper completion state if \ref PH_ERR_SUCCESS_CHAINING
 *    "Chaining Status" is returned from Library. Otherwise error will be returned by SAM for any
 *    subsequent commands.
 *  - If \ref PH_ERR_SUCCESS_CHAINING "Chaining Status" is not returned from Library, no need to call
 *    \ref phhalHw_Sam_Cmd_SAM_UnilatAuthEcc_Part2 "UnilatAuthEcc Part2" interface.
 *
 * \retval #PH_ERR_SUCCESS_CHAINING     Operation successful with Chaining response.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pSamRespLen is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_UnilatAuthEcc_Part1(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                                /**< [In] Protocol Option to update the P1 information.
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_PROTOCOL_OPTION_ISO_INTERNAL_AUTH
                                                                                         *                 "ISO Internal Authentication"
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_PROTOCOL_OPTION_VDE_ECDSA_SIGN
                                                                                         *                 "VDE_ECDSA Signature generation"
                                                                                         */
        uint8_t bECCKeyNo_Priv,                                                         /**< [In] The key reference number of the ECC key entry to be
                                                                                         *        used for signature verification.
                                                                                         *          - NVM Key: 0x00 - 0x7F
                                                                                         *          - RAM Key: 0xE0 - 0xE3
                                                                                         */
        uint8_t bECC_CurveNo,                                                           /**< [In] The curve reference number of the ECC curve entry to
                                                                                         *        be used for signature verification. Will be 0x00 - 0x03
                                                                                         */
        uint8_t ** ppSamResponse,                                                       /**< [Out] Random challenge from SAM */
        uint16_t * pSamRespLen                                                          /**< [Out] Length of bytes available in \b ppSamResponse buffer. */
    );

/**
 * \brief Performs second part, the SAM receives the signature of the MIFARE DUOX,
 * and executes the signature validation. Also the status code of PICC will be
 * returned to the caller in case of error.
 *
 * \note This interface should be called only if \ref phhalHw_Sam_Cmd_SAM_UnilatAuthEcc_Part1
 * "UnilatAuthEcc Part1" returns \ref PH_ERR_SUCCESS_CHAINING "Chaining Status".
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_AUTH_ERROR           Authentication failed: Sig.B verification failed
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pPiccErrorCode, \b pCardResponse \b pSamRespLen and
 *                                      \b pPiccRetCode are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_UnilatAuthEcc_Part2(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t * pPiccErrorCode,                                                       /**< [In] Status code returned by the PICC in case of failure.
                                                                                         *        Should be 2 byte in length with LSB first.
                                                                                         */
        uint8_t * pCardResponse,                                                        /**< [In] Response from PICC. One of the following
                                                                                         *          - If no PICC error and ProtocolOption = 0x00,
                                                                                         *            16 byte Random challenge from PICC followed
                                                                                         *            by 64 byte Signature. \n
                                                                                         *            RndA || Sig.B = ECDSA_Sign(Priv.B, 0xF0F0
                                                                                         *            [|| OptsA] || RndB || RndA)
                                                                                         *
                                                                                         *          - If no PICC error and ProtocolOption = 0x01,
                                                                                         *            16 byte Random challenge from PICC followed
                                                                                         *            by 64 byte Signature. \n
                                                                                         *            RndA || Sig.B = ECDSA_Sign(Priv.B, RA)
                                                                                         */
        uint8_t bCardRespLen,                                                           /**< [In] Length of bytes available in \b pCardResponse buffer. */
        uint8_t * pPiccRetCode                                                          /**< [Out] Status code from MIFARE DUOX PICC if available else zero. */
    );

/**
 * \brief The SAM_BindCertificate supports in S-mode a key pair generation on MIFARE DUOX with ManageKeyPair
 * command, followed by a certificate signing by the SAM. SAM enables the key pair generation by computing
 * the ManageKeyPair secure messaging. This is equivalent to a SAM_Apply_SM execution. MIFARE DUOX may
 * return an error code during the execution of the protocol, the SAM will accept this also in Part2
 * and Part3. In that case, the certificate binding is interrupted and the PICC error will be echoed
 * with Resp.ISO90DF
 *
 * \note
 *  - Post calling this interface it's must to call \ref phhalHw_Sam_Cmd_SAM_BindCertificate_Part2
 *    "BindCertificate Part2"  to place SAM into proper completion state if \ref PH_ERR_SUCCESS_CHAINING
 *    "Chaining Status" is returned from Library. Otherwise error will be returned by SAM for any
 *    subsequent commands.
 *  - If \ref PH_ERR_SUCCESS_CHAINING "Chaining Status" is not returned from Library, no need to call
 *    \ref phhalHw_Sam_Cmd_SAM_BindCertificate_Part2 "BindCertificate Part2" interface.
 *
 * \retval #PH_ERR_SUCCESS_CHAINING     Operation successful with Chaining response.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pMKPParams and \b pSamRespLen are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_BindCertificate_Part1(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                                /**< [In] Protection mode for ManageKeyPair command towards
                                                                                         *        MIFARE DUOX.
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_COMM_MODE_PLAIN "CommMode.Plain"
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_COMM_MODE_MAC "CommMode.MAC"
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_COMM_MODE_FULL "CommMode.Full"
                                                                                         */
        uint8_t * pMKPParams,                                                           /**< [In] MIFARE DUOX ManageKeyPair Parameters. One of the following
                                                                                         *          - KeyNo: Key number of the key to be managed on MIFARE
                                                                                         *                   DUOX
                                                                                         *
                                                                                         *          - Option: Targeted action: Generate Key Pair
                                                                                         *          - CurveID: Targeted curve on MIFARE DUOX
                                                                                         *          - KeyPolicy: Defines allowed Crypto operations on MIFARE
                                                                                         *                       DUOX
                                                                                         *
                                                                                         *          - WriteAccess: Access right and CommMode for further
                                                                                         *                         updates on MIFARE DUOX
                                                                                         *
                                                                                         *          - KUCLimit: Defines the key usage limit of the targeted
                                                                                         *                      key on MIFARE DUOX
                                                                                         */
        uint8_t bMKPParamsLen,                                                          /**< [In] Length of bytes available in \b pMKPParams buffer. */
        uint8_t bECCKeyNo_Priv,                                                         /**< [In] ECC key entry holding the private key to be used for
                                                                                         *        certificate signing in Part 2. Will be 0x00 - 0x0F
                                                                                         */
        uint8_t ** ppSamResponse,                                                       /**< [Out] MKPMAC: 8 bytes as MACt(KsesAuthMAC, Cmd || MKPParams) */
        uint16_t * pSamRespLen                                                          /**< [Out] Length of bytes available in \b ppSamResponse buffer. */
    );

/**
 * \brief The SAM processed the ManageKeyPair response coming from MIFARE DUOX, extracts
 * the public key, injects in the to-be-signed certificate. Then it signs the certificate
 * and returns the signed certificate.
 *
 * \note This interface should be called only if \ref phhalHw_Sam_Cmd_SAM_BindCertificate_Part1
 * "BindCertificate Part1" returns \ref PH_ERR_SUCCESS_CHAINING "Chaining Status".
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_SUCCESS_CHAINING     Operation successful, chaining ongoing.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pMessage and \b pHashLen are NULL
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_BindCertificate_Part2(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                               /**< [In] Buffering Options:
                                                                                         *          - \ref PH_EXCHANGE_DEFAULT "Default": To buffer
                                                                                         *            the initial command information. Use this flag
                                                                                         *            to exchange MIFARE DUOX ManageKeyPair Parameters,
                                                                                         *            Certificate to be signed and start reception of
                                                                                         *            response.
                                                                                         *
                                                                                         *          - \ref PH_EXCHANGE_RXCHAINING "The LFI is set to AFh"
                                                                                         *            for reception of remaining data from SAM
                                                                                         */
        uint8_t * pData,                                                                /**< [In] Information to be exchanged to SAM.
                                                                                         *          - In case of no PICC error, MIFARE DUOX ManageKeyPair Parameters
                                                                                         *              - Comm.MAC : RC || PublicKey || MACt (KSesAuthMAC, RC || CmdCtr
                                                                                         *                || TI || PublicKey)
                                                                                         *              - Comm.FULL: RC || E (KSesAuthENC, PublicKey) || MACt (RC ||
                                                                                         *                CmdCtr || TI || E (KSesAuthENC, PublicKey))
                                                                                         *              - Certificate to be signed
                                                                                         *          - In case of PICC error: 2 byte PICC error
                                                                                         */
        uint16_t wDataLen,                                                              /**< [In] Length of bytes available in \b pData buffer. */
        uint8_t ** ppResponse,                                                          /**< [Out] Buffer containing the information received from SAM.
                                                                                         *          - Certificate signed by SAM
                                                                                         *          - Status code of PICC
                                                                                         */
        uint16_t * pRespLen                                                             /**< [Out] Length of bytes available in \b pResponse buffer. */
    );

/**
 * \brief The SAM_ImportEccKeyDUOX command supports the procedure to import a private ECC key
 * stored on the SAM into the PICC in S-mode. The command generates and returns the cryptogram
 * to be sent to the PICC. For the ManageKeyPair response received from the PICC, regular
 * secure messaging processing with SAM_Remove_SM can be applied
 *
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pMKPParams and \b pSamRespLen are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_ImportEccKeyDUOX(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bECCKeyNo_Priv,                                                         /**< [In] ECC key entry holding the private key to be imported in
                                                                                         *        MIFARE DUOX. Will be 0x00 - 0x0F
                                                                                         */
        uint8_t * pMKPParams,                                                           /**< [In] MIFARE DUOX ManageKeyPair Parameters. One of the following
                                                                                         *          - KeyNo: Key number of the key to be managed on MIFARE
                                                                                         *                   DUOX
                                                                                         *
                                                                                         *          - Option: Targeted action: Import Key Pair
                                                                                         *          - CurveID: Targeted curve on MIFARE DUOX
                                                                                         *          - KeyPolicy: Defines allowed crypto operations on MIFARE
                                                                                         *                       DUOX
                                                                                         *
                                                                                         *          - WriteAccess: Access right and CommMode for further
                                                                                         *                         updates on MIFARE DUOX
                                                                                         *
                                                                                         *          - KUCLimit: Defines the key usage limit of the targeted
                                                                                         *                      key on MIFARE DUOX
                                                                                         */
        uint8_t bMKPParamsLen,                                                          /**< [In] Length of bytes available in \b pMKPParams buffer. */
        uint8_t ** ppMKPCrypto,                                                         /**< [Out] Cryptogram holding key data: E (KSesAuthENC, Private
                                                                                         *         Key) || MACt (Cmd || CmdCtr || TI || MKPParams ||
                                                                                         *         E (KSesAuthENC, PrivateKey))
                                                                                         */
        uint16_t * pMKPCryptoLen                                                        /**< [Out] Length of bytes available in \b ppMKPCrypto buffer. */
    );

/**
 * end of group phhalHw_Sam_Cmd_DUOX_SMode
 * @}
 */





/** \defgroup phhalHw_Sam_Cmd_DUOX_XMode X_Mode
 * \brief Implementation of SAM commands used for MIFARE DUOX communication in X-Mode.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_DUOX_XMode_Defines Defines
 * \brief Definitions for SAM commands used for MIFARE DUOX communication in X-Mode.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_DUOX_SMode_Defines_CommandCodes CommandCodes
 * \brief Definitions for Instruction codes
 * @{
 */
#define PHHAL_HW_SAM_CMD_INS_DUOX_MUTUAL_AUTH_ECC                               0xDBU   /**< SAM Instruction code for DUOX_MutualAuthECC command. */
#define PHHAL_HW_SAM_CMD_INS_DUOX_UNILATERAL_AUTH_ECC                           0xDFU   /**< SAM Instruction code for DUOX_UnilatAuthEcc command. */
#define PHHAL_HW_SAM_CMD_INS_DUOX_BIND_CERTIFICATE                              0xB4U   /**< SAM Instruction code for DUOX_BindCertificate command. */
#define PHHAL_HW_SAM_CMD_INS_DUOX_IMPORT_ECC_KEY                                0xB6U   /**< SAM Instruction code for DUOX_ImportEccKey command. */
/**
 * end of group phhalHw_Sam_Cmd_DUOX_SMode_Defines_CommandCodes
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_DUOX_XMode_Defines_FlowOption Flow Option
 * \brief Definitions for MIFARE DUOX Flow Option. To be used with
 * \ref phhalHw_Sam_Cmd_DUOX_MutualAuthEcc "DUOX_MutualAuthEcc"
 * @{
 */
#define PHHAL_HW_CMD_SAM_FLOW_OPTION_ISO_GENERAL_AUTH                           0x00U   /**< SAM DUOX flow option as no ISOGeneral
                                                                                         *   Authentication Part 1 and Part2.
                                                                                         */
#define PHHAL_HW_CMD_SAM_FLOW_OPTION_ISO_GENERAL_AUTH_FINAL                     0x01U   /**< SAM DUOX flow option as ISOSelectFile
                                                                                         *   and ISOGeneralAuthenticate Final.
                                                                                         */
/**
 * end of group phhalHw_Sam_Cmd_DUOX_SMode_Defines_ProtOption
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_DUOX_XMode_Defines
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_DUOX_XMode
 * @}
 */

/**
 * \brief The DUOX_MutualAuthEcc supports the ECC-based Mutual and Reader-Unilateral
 * Authentication in X-Mode. The command supports both the flows triggered by
 * ISOGeneralAuthentication and ISOSelectFile as implemented by MIFARE DUOX.
 *
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_SUCCESS_CHAINING     Operation successful, chaining ongoing.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b bOptsA, \b pSamRespLen and
 *                                      \b pPiccReturnCode are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_DUOX_MutualAuthEcc(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                                /**< [In] Flow Option to update the P1 information and receive
                                                                                         *        remaining data from SAM.
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_FLOW_OPTION_ISO_GENERAL_AUTH
                                                                                         *                 "ISOGeneralAuthenticate + ISOGeneralAuthenticate Part2"
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_FLOW_OPTION_ISO_GENERAL_AUTH_FINAL
                                                                                         *                 "ISOSelectFile + ISOGeneralAuthenticateFinal"
                                                                                         *
                                                                                         *          - Use \ref PH_EXCHANGE_RXCHAINING "Receive Remaining Data"
                                                                                         *            from SAM. Should not be merged with above option
                                                                                         */
        uint8_t * pOptsA,                                                               /**< [In] PCD Option (TLV). One of the following
                                                                                         *          - Tag            : 0x80
                                                                                         *          - Length         : Length of Value field. Length is either 0x02 or 0x04
                                                                                         *          - Value          : One of the following,
                                                                                         *              - Auth Method
                                                                                         *                  - 0x80: Asymmetric mutual authentication with Cert.A
                                                                                         *                  - 0xA0: Asymmetric mutual authentication without Cert.A
                                                                                         *                  - 0x40: Asymmetric reader-unilateral authentication
                                                                                         *                          with Cert.A
                                                                                         *                  - 0x60: Asymmetric reader-unilateral authentication
                                                                                         *                          without Cert.A
                                                                                         *
                                                                                         *          - ProtocolVersion: One of the following
                                                                                         *              - 0x00 : Protocol version as supported by SAM
                                                                                         *              - Other: Other protocol version (sent to PICC but ignored by SAM)
                                                                                         *
                                                                                         *          - CertFileNo: Certificate File Number. Present only if
                                                                                         *                        AuthMethod has 0x80 or 0xA0 as values.
                                                                                         *
                                                                                         *          - PrivKeyNo: Private Key Number. Present only if
                                                                                         *                        AuthMethod has 0x80 or 0xA0 as values.
                                                                                         */
        uint8_t bOptsALen,                                                              /**< [In] Length of bytes available in \b pOptsA buffer. */
        uint8_t bDUOX_P2,                                                               /**< [In] Mapped to ISOGeneralAuthenticate or ISOGeneral AuthenticateFinal P2.
                                                                                         *        Refer DataSheet for more information */
        uint8_t bECCKeyNo_Priv,                                                         /**< [In] ECC key entry holding the private key to be used for signing.
                                                                                         *        Will be 0x00 - 0x0F
                                                                                         */
        uint8_t bCertA_FileNo,                                                          /**< [In] The file number of the file holding the Cert.A certificate
                                                                                         *        (chain). Present in command frame only if \b pOptsA has AuthOptions
                                                                                         *        as 0x80 or 0x40
                                                                                         */
        uint8_t bCertB_Options,                                                         /**< [In] Cert.B processing options. One of the following
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_CERTB_OPTION_MSB_B_PLAIN
                                                                                         *            "No Msg.B processing by SAM". Return full Msg.B.Plain
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_CERTB_OPTION_NO_DATA
                                                                                         *            "Sig.B and (if present) Cert.B certificate (chain)"
                                                                                         *            validated by SAM but there is no response from SAM.
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_CERTB_OPTION_TBS_CERTB
                                                                                         *            "Sig.B and (if present) Cert.B certificate (chain)"
                                                                                         *            validated by SAM and there no response from SAM.
                                                                                         */
        uint8_t bECCKeyNo_CA,                                                           /**< [In] ECC key entry holding the public key of the targeted CA Root Key
                                                                                         *        for Cert.B validation. Will be 0x00 - 0x0F. Present in command
                                                                                         *        frame only if \b bOption has one of the following,
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_AUTH_METHOD_MUTUAL_AUTH_CERT
                                                                                         *                 "ASymmetric Mutual Authentication" with Certificate.A
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_AUTH_METHOD_READER_UNILATERAL_AUTH_CERT
                                                                                         *                 "ASymmetric Reader-Unilateral Authentication" with
                                                                                         *                 Certificate.A
                                                                                         */
        uint8_t ** ppSamResponse,                                                       /**< [Out] Buffer containing the information received from SAM.
                                                                                         *          - If AuthMethod = 0x80, 0xA0
                                                                                         *              - CertB_Options = \ref PHHAL_HW_CMD_SAM_CERTB_OPTION_MSB_B_PLAIN
                                                                                         *                                "No Msg.B processing by SAM". Return full
                                                                                         *                                Msg.B.Plain, Mutual authentication. \n
                                                                                         *                                OptB || Cert.B || Sig.B
                                                                                         *              - CertB_Options = \ref PHHAL_HW_CMD_SAM_CERTB_OPTION_TBS_CERTB
                                                                                         *                                "Sig.B and (if present) Cert.B certificate (chain)"
                                                                                         *                                validated by SAM and there no response from SAM,
                                                                                         *                                Mutual authentication - SAM validation. \n
                                                                                         *                                OptB || TBSCert.B.parent || TBSCert.B.leaf
                                                                                         *              - CertB_Options = \ref PHHAL_HW_CMD_SAM_CERTB_OPTION_NO_DATA
                                                                                         *                                "Sig.B and (if present) Cert.B certificate (chain)"
                                                                                         *                                validated by SAM but there is no response from SAM,
                                                                                         *                                Mutual authentication - SAM validation - no data
                                                                                         *                                returned. \n OptB
                                                                                         *
                                                                                         *          - If AuthMethod = 0x80, 0xA0, Reader-unilateral authentication. \n
                                                                                         *            OptB
                                                                                         */
        uint16_t * pSamRespLen,                                                         /**< [Out] Length of bytes available in \b ppSamResponse buffer. */
        uint8_t * pPiccReturnCode                                                       /**< [Out] Status code from PICC, One of the following
                                                                                         *              - ISO/IEC 7816-4 status bytes SW1-SW2 (2 byte)
                                                                                         *              - No response data, indicating unexpected length
                                                                                         *                returned by PICC
                                                                                         */
    );

/**
 * \brief The DUOX_MutualAuthEcc supports in X-mode the ECC-based Card-Unilateral Authentication
 * as implemented by MIFARE DUOX with the ISOInternalAuthenticate or VDE_ECDSASign command. The
 * actual protocol choice is provided to the SAM via the P1 parameter.
 *
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pPiccReturnCode is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_DUOX_UnilatAuthEcc(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                                /**< [In] Protocol Option to update the P1 information.
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_PROTOCOL_OPTION_ISO_INTERNAL_AUTH
                                                                                         *                 "ISO Internal Authentication"
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_PROTOCOL_OPTION_VDE_ECDSA_SIGN
                                                                                         *                 "VDE_ECDSA Signature generation"
                                                                                         */
        uint8_t bECC_KeyNo,                                                             /**< [In] The key reference number of the ECC key entry to be
                                                                                         *        used for signature verification.
                                                                                         *          - NVM Key: 0x00 - 0x7F
                                                                                         *          - RAM Key: 0xE0 - 0xE3
                                                                                         */
        uint8_t bECC_CurveNo,                                                           /**< [In] The curve reference number of the ECC curve entry to
                                                                                         *        be used for signature verification. Will be 0x00 - 0x03
                                                                                         */
        uint8_t * pPiccReturnCode                                                       /**< [Out] Status code from PICC, One of the following
                                                                                         *              - ISO/IEC 7816-4 status bytes SW1-SW2 (2 byte)
                                                                                         *              - No response data, indicating unexpected length
                                                                                         *                returned by PICC
                                                                                         */
    );

/**
 * \brief The DUOX_BindCertificate supports in X-mode a key pair generation on MIFARE DUOX with
 * ManageKeyPair command, followed by a certificate signing by the SAM
 *
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_SUCCESS_CHAINING     Operation successful with Chaining response.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pMKPParams and \b pSamRespLen are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_DUOX_BindCertificate(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                                /**< [In] Configuration for ManageKeyPair command towards
                                                                                         *        MIFARE DUOX.
                                                                                         *          - ISO mode selection
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_ISO_MODE_NATIVE
                                                                                         *                "Command will be sent to PICC using native command set"
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_ISO_MODE_ISO7816
                                                                                         *                "Command will be sent to PICC using ISO/IEC 7816-4APDU"
                                                                                         *
                                                                                         *          - Communication mode. To be ORed with above options.
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_COMM_MODE_MAC "CommMode.MAC"
                                                                                         *              - \ref PHHAL_HW_CMD_SAM_COMM_MODE_FULL "CommMode.Full"
                                                                                         *
                                                                                         *          - Use \ref PH_EXCHANGE_RXCHAINING "Receive Remaining Data"
                                                                                         *            from SAM. Should not be merged with above option
                                                                                         */
        uint8_t * pMKPParams,                                                           /**< [In] MIFARE DUOX ManageKeyPair Parameters. One of the following
                                                                                         *          - KeyNo: Key number of the key to be managed on MIFARE
                                                                                         *                   DUOX
                                                                                         *
                                                                                         *          - Option: Targeted action: Generate Key Pair
                                                                                         *          - CurveID: Targeted curve on MIFARE DUOX
                                                                                         *          - KeyPolicy: Defines allowed crypto operations on MIFARE
                                                                                         *                       DUOX
                                                                                         *
                                                                                         *          - WriteAccess: Access right and CommMode for further
                                                                                         *                         updates on MIFARE DUOX
                                                                                         *
                                                                                         *          - KUCLimit: Defines the key usage limit of the targeted
                                                                                         *                      key on MIFARE DUOX
                                                                                         */
        uint8_t bMKPParamsLen,                                                          /**< [In] Length of bytes available in \b pMKPParams buffer. */
        uint8_t bECCKeyNo_Priv,                                                         /**< [In] ECC key entry holding the private key to be used for
                                                                                         *        certificate signing in Part 2. Will be 0x00 - 0x0F
                                                                                         */
        uint8_t * pTBSCertificate,                                                      /**< [In] Certificate to be signed */
        uint16_t wTBSCertLen,                                                           /**< [In] Length of bytes available in \b pTBSCertificate buffer. */
        uint8_t ** ppSamResponse,                                                       /**< [Out] Certificate signed by SAM. */
        uint16_t * pSamRespLen,                                                         /**< [Out] Length of bytes available in \b ppSamResponse buffer. */
        uint8_t * pPiccReturnCode                                                       /**< [Out] Status code from PICC, One of the following
                                                                                         *              - Native return code (1 byte) if \b bOption has
                                                                                         *                \ref PHHAL_HW_CMD_SAM_ISO_MODE_NATIVE "Native command"
                                                                                         *              - ISO/IEC 7816-4 status bytes SW1-SW2 (2 byte) if
                                                                                         *                \b bOption has \ref PHHAL_HW_CMD_SAM_ISO_MODE_ISO7816
                                                                                         *                "ISO/IEC 7816-4 command"
                                                                                         */
    );

/**
 * \brief The SAM_ImportEccKeyDUOX command supports the procedure to import a private ECC key
 * stored on the SAM into the PICC in S-mode. The command generates and returns the cryptogram
 * to be sent to the PICC. For the ManageKeyPair response received from the PICC, regular
 * secure messaging processing with SAM_Remove_SM can be applied
 *
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pMKPParams and \b pSamRespLen are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_DUOX_ImportEccKey(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                                /**< [In] Configuration for ManageKeyPair command towards
                                                                                         *        MIFARE DUOX. ISO mode selection,
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_ISO_MODE_NATIVE
                                                                                         *            "Command will be sent to PICC using native command set"
                                                                                         *          - \ref PHHAL_HW_CMD_SAM_ISO_MODE_ISO7816
                                                                                         *            "Command will be sent to PICC using ISO/IEC 7816-4APDU"
                                                                                         */
        uint8_t * pMKPParams,                                                           /**< [In] MIFARE DUOX ManageKeyPair Parameters. One of the following
                                                                                         *          - KeyNo: Key number of the key to be managed on MIFARE
                                                                                         *                   DUOX
                                                                                         *
                                                                                         *          - Option: Targeted action: Import Key Pair
                                                                                         *          - CurveID: Targeted curve on MIFARE DUOX
                                                                                         *          - KeyPolicy: Defines allowed crypto operations on MIFARE
                                                                                         *                       DUOX
                                                                                         *
                                                                                         *          - WriteAccess: Access right and CommMode for further
                                                                                         *                         updates on MIFARE DUOX
                                                                                         *
                                                                                         *          - KUCLimit: Defines the key usage limit of the targeted
                                                                                         *                      key on MIFARE DUOX
                                                                                         */
        uint8_t bMKPParamsLen,                                                          /**< [In] Length of bytes available in \b pMKPParams buffer. */
        uint8_t bECCKeyNo_Priv,                                                         /**< [In] ECC key entry holding the private key to be imported in
                                                                                         *        MIFARE DUOX. Will be of 0x00 - 0x0F
                                                                                         */
        uint8_t * pPiccReturnCode                                                       /**< [Out] Status code from PICC, One of the following
                                                                                         *              - Native return code (1 byte) if \b bOption has
                                                                                         *                \ref PHHAL_HW_CMD_SAM_ISO_MODE_NATIVE "Native command"
                                                                                         *              - ISO/IEC 7816-4 status bytes SW1-SW2 (2 byte) if
                                                                                         *                \b bOption has \ref PHHAL_HW_CMD_SAM_ISO_MODE_ISO7816
                                                                                         *                "ISO/IEC 7816-4 command"
                                                                                         *              - No response data, indicating unexpected length
                                                                                         *                returned by PICC
                                                                                         */
    );

/**
 * end of group phhalHw_Sam_Cmd_DUOX
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd
 * @}
 */

#endif /* NXPBUILD__PHHAL_HW_SAM */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHHALHW_SAM_CMD_DUOX_H */
