/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) Virtual Card command implementation of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */


#ifndef PHHALHW_SAM_CMD_VC_H
#define PHHALHW_SAM_CMD_VC_H

#include <ph_Status.h>
#include <phhalHw.h>

#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */

#ifdef NXPBUILD__PHHAL_HW_SAM

/** \addtogroup phhalHw_Sam_Cmd
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_VCA 06_VirtualCard
 * \brief Implementation of SAM Virtual Card and ProximityCheck commands.
 * Has implementation for both X and S mode communication.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_VCA_Defines Defines
 * \brief Definitions common for Virtual Card S anad X mode communication
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_VCA_Defines_SelectVC SelectVC
 * \brief Definitions for Key diversification selection. To be used with below mentioned interfaces.
 *  - \ref phhalHw_Sam_Cmd_SAM_SelectVC "SAM_SelectVC"
 *  - \ref phhalHw_Sam_Cmd_VCA_Select "VCA_Select"
 * @{
 */
#define PHHAL_HW_SAM_CMD_SELECT_DIV_DEFAULT                                     0x00U   /**< Default option mask to disable the diversification of
                                                                                         *   VcSelect MAC and ENC key.
                                                                                         */
#define PHHAL_HW_SAM_CMD_SELECT_ENC_KEY_DIV_INPUT                               0x01U   /**< Option mask to perform diversification of VC SelectENC
                                                                                         *   key using the diversification input provided.
                                                                                         */
#define PHHAL_HW_SAM_CMD_SELECT_MAC_KEY_DIV_INPUT                               0x02U   /**< Option mask to perform diversification of VC SelectMAC
                                                                                         *   key using the diversification input provided.
                                                                                         */
#define PHHAL_HW_SAM_CMD_SELECT_MAC_KEY_DIV_VCUID                               0x04U   /**< Option mask to perform diversification of VC SelectMAC
                                                                                         *   key using Virtual Card Identifier.
                                                                                         */
#define PHHAL_HW_SAM_CMD_SELECT_MAC_KEY_DIV_INPUT_VCUID                         0x06U   /**< Option mask to perform diversification of VC SelectMAC
                                                                                         *   key using Virtual Card Identifier and Diversification
                                                                                         *   input provided.
                                                                                         */
/**
 * end of group phhalHw_Sam_Cmd_VCA_Defines_SelectVC
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_VCA_Defines_PC ProximityCheck
 * \brief Definitions for Key diversification selection. To be used with below mentioned interfaces.
 *  - \ref phhalHw_Sam_Cmd_SAM_ProximityCheck_Part1 "SAM_ProximityCheck Part1"
 *  - \ref phhalHw_Sam_Cmd_VCA_ProximityCheck "VCA_ProximityCheck"
 * @{
 */
#define PHHAL_HW_SAM_CMD_PROXIMITY_CHECK_DIV_ON                                 0x01U   /**< Option mask to perform diversification of ProximityCheck
                                                                                         *   key using the diversification input provided.
                                                                                         */
/**
 * end of group phhalHw_Sam_Cmd_VCA_Defines_PC
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_VCA_Defines
 * @}
 */





/** \defgroup phhalHw_Sam_Cmd_VCA_SMode S_Mode
 * \brief Implementation of SAM commands used for Virtual Card and ProximityCheck communication in S Mode.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_VCA_SMode_Defines Defines
 * \brief Definitions for SAM commands used for Virtual Card and ProximityCheck communication in S Mode.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_VCA_SMode_Defines_CommandCodes CommandCodes
 * \brief Definitions for Instruction codes
 * @{
 */
#define PHHAL_HW_SAM_CMD_INS_SAM_SELECT_VC                                      0x44U   /**< SAM Insturction code for SAM_SelectVC command. */
#define PHHAL_HW_SAM_CMD_INS_SAM_PROXIMITY_CHECK                                0xFAU   /**< SAM Insturction code for SAM_ProximityCheck command. */
/**
 * end of group phhalHw_Sam_Cmd_VCA_SMode_Defines_CommandCodes
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_VCA_SMode_Defines
 * @}
 */

/**
 * \brief Performs Virtual card selection based on the response of ISOSelect command.
 *  - The SAM_SelectVC command is designed to support in S-mode the VC Selection with PCD authentication,
 *    that is if the VC is configured for AuthVCMandatory.
 *  - The AuthVCMandatory configuration is deducted from the presence of a valid FCI (that is: expected
 *    4-bytes TLV headers and payload size) in the response of the PICC ISOSelect command.
 *  - If the PICC ISOSelect command does not return a FCI or returns an invalid FCI, there is no need to
 *    execute SAM_SelectVC
 *  - The command supports key diversification.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pData, \b pDivInput and \b pRespLen is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_SelectVC(
        phhalHw_Sam_DataParams_t *pDataParams,                                          /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                                /**< [In] Option to update the P1 information. The options
                                                                                         *        can be combined by bitwise oring.
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SELECT_ENC_KEY_DIV_INPUT
                                                                                         *                 "Diversification of VCSelectENCKey" uisng
                                                                                         *                  Diversification Input
                                                                                         *
                                                                                         *          - Option for diversification of VCSelectMACKey.
                                                                                         *              - \ref PHHAL_HW_SAM_CMD_SELECT_MAC_KEY_DIV_INPUT
                                                                                         *                "Diversify the used key with the given DivInput"
                                                                                         *              - \ref PHHAL_HW_SAM_CMD_SELECT_MAC_KEY_DIV_VCUID
                                                                                         *                "Diversify with the Virtual Card Unique Identifier"
                                                                                         *                (VCUID)
                                                                                         *              - \ref PHHAL_HW_SAM_CMD_SELECT_MAC_KEY_DIV_INPUT_VCUID
                                                                                         *                "Diversify with the VCUID and the given DivInput"
                                                                                         *
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SELECT_DIV_DEFAULT "Option to disable"
                                                                                         *                 the diversification for VcSelect MAC and ENC keys.
                                                                                         *
                                                                                         */
        uint8_t bEncKeyNo,                                                              /**< [In] KeyID.VCSelectENCKey SAM Key Number.
                                                                                         *          - 0x00 - 0x7F: Standard (NVRam) Key
                                                                                         *          - 0xE0 - 0xE3: RAM Key
                                                                                         */
        uint8_t bEncKeyVer,                                                             /**< [In] KeyID.VCSelectENCKey SAM Key Version. */
        uint8_t bMacKeyNo,                                                              /**< [In] KeyID.VCSelectMACKey SAM Key Number.
                                                                                         *          - 0x00 - 0x7F: Standard (NVRam) Key
                                                                                         *          - 0xE0 - 0xE3: RAM Key
                                                                                         */
        uint8_t bMacKeyVer,                                                             /**< [In] KeyID.VCSelectMACKey SAM Key Version. */
        uint8_t * pData,                                                                /**< [In] ISOSelect response payload (32 Bytes) without TLV headers,
                                                                                         *        including the VC related data.
                                                                                         */
        uint8_t bDataLen,                                                               /**< [In] Length of bytes available in \b pData buffer. */
        uint8_t * pDivInput,                                                            /**< [In] Diversification Input used to diversify the key. */
        uint8_t bDivInputLen,                                                           /**< [In] Length of bytes available in \b pDivInput buffer. */
        uint8_t ** ppResponse,                                                          /**< [Out] Response received from SAM (MAC || VCData)
                                                                                         *          - MAC: Response to Challenge as MACt(DivKey(KVCSelMAC)),
                                                                                         *            RndChal || VCData)
                                                                                         *          - VCData: Decrypted Virtual Card (VC) Data
                                                                                         */
        uint16_t * pRespLen                                                             /**< [Out] Length of bytes available in \b ppResponse buffer. */
    );

/**
 * \brief Performs Proximity Check Part 1 command execution.
 * \note
 *  - Post calling this interface it's must to call \ref phhalHw_Sam_Cmd_SAM_ProximityCheck_Part2
 *    "ProximityCheck Part2" to place SAM into proper completion state if \ref PH_ERR_SUCCESS_CHAINING
 *    "Chaining Status" is returned from Library. Otherwise error will be returned by SAM for any
 *    subsequent commands.
 *  - If \ref PH_ERR_SUCCESS_CHAINING "Chaining Status" is not returned from Library, no need to call
 *    \ref phhalHw_Sam_Cmd_SAM_ProximityCheck_Part2 "ProximityCheck Part2" interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS_CHAINING     Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pData, \b pDivInput and \b pRespLen is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_ProximityCheck_Part1(
        phhalHw_Sam_DataParams_t *pDataParams,                                          /**< [In] Pointer to this layer's parameter structure. */
        uint8_t  bOption,                                                               /**< [In] \ref PHHAL_HW_SAM_CMD_PROXIMITY_CHECK_DIV_ON
                                                                                         *        "Use Key Diversification". Instructs SAM to
                                                                                         *        Diversify the Key.
                                                                                         */
        uint8_t bKeyNo,                                                                 /**< [In] Key number to be used in SAM.
                                                                                         *          - 0x00 - 0x7F: Standard (NVRam) Key
                                                                                         *          - 0xE0 - 0xE3: RAM Key
                                                                                         */
        uint8_t bKeyVer,                                                                /**< [In] Key version to be used in SAM. */
        uint8_t * pPPCData,                                                             /**< [In] PICC_PreparePC response data: Option || Pub
                                                                                         *        RespTime [|| PPS1] [|| ActBitRate]
                                                                                         */
        uint8_t bPPCDataLen,                                                            /**< [In] Length of bytes available in \b pPPCData buffer. */
        uint8_t * pPCData,                                                              /**< [In] Response and challenge bytes exchanged during the
                                                                                         *        proximity check protocol as (pRndR1 || pRndC1) ||
                                                                                         *        ... || (pRndR8 || pRndC8)
                                                                                         */
        uint8_t  bPCDataLen,                                                            /**< [In] Length of bytes available in \b pPCData buffer. */
        uint8_t * pDivInput,                                                            /**< [In] Diversification Input used to diversify the key. */
        uint8_t bDivInputLen,                                                           /**< [In] Length of bytes available in \b pDivInput buffer. */
        uint8_t ** ppMac,                                                               /**< [Out] The MAC to be exchanged to the PICC.
                                                                                         *         MACt (KPC, code || Option || PubRespTime [|| PPS1] || (pRndR1
                                                                                         *         || pRndC1) ||...|| (pRndR8 || pRndC8))
                                                                                         */
        uint16_t * pMacLen                                                              /**< [Out] Length of bytes available in \b ppMac buffer. */
    );

/**
 * \brief Performs Proximity Check Part 2 command execution.
 * \note This interface should be called only if \ref phhalHw_Sam_Cmd_SAM_ProximityCheck_Part1
 * "ProximityCheck Part1" returns \ref PH_ERR_SUCCESS_CHAINING "Chaining Status".
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS               Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pData, \b pDivInput and \b pPiccRetCode is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_ProximityCheck_Part2(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t * pData,                                                                /**< [In] The data to be exchaned to SAM hardware.
                                                                                         *          - For PICC Error  : The PICC error code should be
                                                                                         *            passed and \b bDataLen should be 1.
                                                                                         *          - For PICC Success: The MAC received from PICC
                                                                                         *            should be passed and \b bDataLen should be 8.
                                                                                         */
        uint8_t bDataLen,                                                               /**< [In] Length of bytes available in \b pData buffer. */
        uint8_t * pPiccRetCode                                                          /**< [Out] The PICC response code echoed back by the SAM. */
    );

/**
 * end of group phhalHw_Sam_Cmd_VCA_SMode
 * @}
 */





/** \defgroup phhalHw_Sam_Cmd_VCA_XMode X_Mode
 * \brief Implementation of SAM commands used for Virtual Card and ProximityCheck communication in X Mode.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_VCA_XMode_Defines Defines
 * \brief Definitions for SAM commands used for Virtual Card and ProximityCheck communication in X Mode.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_VCA_XMode_Defines_CommandCodes CommandCodes
 * \brief Definitions for Instruction codes
 * @{
 */
#define PHHAL_HW_SAM_CMD_INS_VCA_SELECT                                         0x45U   /**< Sam Instruction code for VCA_SelectVC command. */
#define PHHAL_HW_SAM_CMD_INS_VCA_PROXIMITY_CHECK                                0xFBU   /**< Sam Instruction code for VCA_ProximityCheck command. */
/**
 * end of group phhalHw_Sam_Cmd_VCA_XMode_Defines_CommandCodes
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_VCA_XMode_Defines_VCASelect Select
 * \brief Definitions for VCASelect command variants. To be used with \ref phhalHw_Sam_Cmd_VCA_Select
 * "VCA_Select" interface.
 * @{
 */
#define PHHAL_HW_SAM_CMD_VCA_SELECT_VARIANT_PART1                               0x00U   /**< Option mask for complete VC selection in 1 part. */
#define PHHAL_HW_SAM_CMD_VCA_SELECT_VARIANT_PART2                               0x08U   /**< Option mask for complete VC selection in 2 part. */
/**
 * end of group phhalHw_Sam_Cmd_VCA_XMode_Defines_VCASelect
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_VCA_XMode_Defines_Format Format
 * \brief Definitions for command format (ISO7816-4 mode) to be used for Proximity Check command.
 * To be used with \ref phhalHw_Sam_Cmd_VCA_ProximityCheck "VCA_ProximityCheck" interface.
 * @{
 */
#define PHHAL_HW_SAM_CMD_PC_NATIVE_FORMAT                                       0x00U   /**< Option mask to perform Proximity Check in native format. */
#define PHHAL_HW_SAM_CMD_PC_WRAPPED_FORMAT                                      0x04U   /**< Option mask to perform Proximity Check in Iso7816-4 wrapped format. */
/**
 * end of group phhalHw_Sam_Cmd_VCA_XMode_Defines_Format
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_VCA_XMode_Defines_Processing Processing
 * \brief Definitions for VerifyPC processing to be used for Proximity Check command.
 * To be used with \ref phhalHw_Sam_Cmd_VCA_ProximityCheck "VCA_ProximityCheck" interface.
 * @{
 */
#define PHHAL_HW_SAM_CMD_PC_NORMAL_PROCESSING                                   0x00U   /**< Option mask to perform Normal VerifyPC processing. */
#define PHHAL_HW_SAM_CMD_PC_RANDOM_PROCESSING                                   0x02U   /**< Option mask to perform Random VerifyPC processing. */
/**
 * end of group phhalHw_Sam_Cmd_VCA_XMode_Defines_Processing
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_VCA_XMode_Defines
 * @}
 */

/**
 * \brief The VCA_Select command handles in X-mode a complete VC Selection with or without PCD
 * authentication depending on the AuthVCMandatory configuration of the VC. The SAM shall deduct
 * the AuthVCMandatory configuration from the presence of a valid FCI (that is: expected 4-bytes
 * TLV headers and payload size) in the response of the PICC ISOSelect.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS               Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pIID, \b pDivInput \b pRespLen and
 *                                      \b pPiccRetCode is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_VCA_Select(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                                /**< [In] Option to update the P1 information. The options
                                                                                         *        can be combined by bitwise oring.
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SELECT_ENC_KEY_DIV_INPUT
                                                                                         *                 "Diversification of VCSelectENCKey" uisng
                                                                                         *                  Diversification Input
                                                                                         *
                                                                                         *          - Option for diversification of VCSelectMACKey.
                                                                                         *              - \ref PHHAL_HW_SAM_CMD_SELECT_MAC_KEY_DIV_INPUT
                                                                                         *                "Diversify the used key with the given DivInput"
                                                                                         *              - \ref PHHAL_HW_SAM_CMD_SELECT_MAC_KEY_DIV_VCUID
                                                                                         *                "Diversify with the Virtual Card Unique Identifier"
                                                                                         *                (VCUID)
                                                                                         *              - \ref PHHAL_HW_SAM_CMD_SELECT_MAC_KEY_DIV_INPUT_VCUID
                                                                                         *                "Diversify with the VCUID and the given DivInput"
                                                                                         *
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SELECT_DIV_DEFAULT "Option to disable"
                                                                                         *                 the diversification for VcSelect MAC and ENC keys.
                                                                                         *
                                                                                         *          - VC selection can be done in 1-part of 2-parts.
                                                                                         *              - \ref PHHAL_HW_SAM_CMD_VCA_SELECT_VARIANT_PART1
                                                                                         *                "The complete VC Selection is performed in 1-part"
                                                                                         *              - \ref PHHAL_HW_SAM_CMD_VCA_SELECT_VARIANT_PART2
                                                                                         *                "The complete VC Selection is performed in 2-parts",
                                                                                         *                Postpone Cmd.PICC_ISO_ExternalAuthenticate to second part
                                                                                         */
        uint8_t bEncKeyNo,                                                              /**< [In] KeyID.VCSelectENCKey SAM Key Number.
                                                                                         *          - 0x00 - 0x7F: Standard (NVRam) Key
                                                                                         *          - 0xE0 - 0xE3: RAM Key
                                                                                         */
        uint8_t bEncKeyVer,                                                             /**< [In] KeyID.VCSelectENCKey SAM Key Version. */
        uint8_t bMacKeyNo,                                                              /**< [In] KeyID.VCSelectMACKey SAM Key Number.
                                                                                         *          - 0x00 - 0x7F: Standard (NVRam) Key
                                                                                         *          - 0xE0 - 0xE3: RAM Key
                                                                                         */
        uint8_t bMacKeyVer,                                                             /**< [In] KeyID.VCSelectMACKey SAM Key Version. */
        uint8_t * pIID,                                                                 /**< [In] The Installation Identifier (IID) to be selected. */
        uint8_t bIIDLen,                                                                /**< [In] Length of bytes available in \b pIID buffer. */
        uint8_t * pDivInput,                                                            /**< [In] Diversification Input used to diversify the key. */
        uint8_t bDivInputLen,                                                           /**< [In] Length of bytes available in \b pDivInput buffer. */
        uint8_t ** ppResponse,                                                          /**< [Out] Response received from SAM based on the variants and
                                                                                         *         AuthVCMandatory settings
                                                                                         *          - AuthVC Not Mandatory & Variant 1 : Conf0 (1byte), FCI (N bytes)
                                                                                         *          - AuthVC Mandatory & Variant 1     : Conf1 (1byte), Decrypted VC
                                                                                         *                                               Data (16 bytes)
                                                                                         *          - AuthVC Mandatory & Variant 2     : Decrypted VC Data (16 bytes)
                                                                                         */
        uint16_t * pRespLen,                                                            /**< [Out] Length of bytes available in \b ppResponse buffer. */
        uint16_t * pPiccRetCode                                                         /**< [Out] The status code returned from the PICC. This will be applicable
                                                                                         *         for both the variants.
                                                                                         */
    );



/**
 * \brief The VCA_ProximityCheck command performs the complete PICC proximity check between the SAM
 * and the PICC. It performs the PICC PreparePC command, the one or more PICC ProxmityChecks and the
 * PICC VerifyPC commands.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS               Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pDivInput and \b pRespLen is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_VCA_ProximityCheck(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t  bOption,                                                               /**< [In] Option to indicate diversification, processing and
                                                                                        *         format options. Options can be ORed.
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_PROXIMITY_CHECK_DIV_ON
                                                                                         *            "Use Key Diversification". Instructs SAM to
                                                                                         *            Diversify the Key.
                                                                                         *
                                                                                         *          - Option to indicate VerifyPC processing.
                                                                                         *              - \ref PHHAL_HW_SAM_CMD_PC_NORMAL_PROCESSING
                                                                                         *                "Normal Cmd.PICC_VerifyPC processing"
                                                                                         *              - \ref PHHAL_HW_SAM_CMD_PC_RANDOM_PROCESSING
                                                                                         *                "Random Cmd.PICC_VerifyPC processing"
                                                                                         *
                                                                                         *          - The command format (ISO7816-4 mode) to be used to
                                                                                         *            communicated to PICC.
                                                                                         *              - \ref PHHAL_HW_SAM_CMD_PC_NATIVE_FORMAT
                                                                                         *                "Use native commands"
                                                                                         *              - \ref PHHAL_HW_SAM_CMD_PC_WRAPPED_FORMAT
                                                                                         *                "Wrap commands in ISO7816-4 compliant format"
                                                                                         */
        uint8_t bKeyNo,                                                                 /**< [In] Key number to be used in SAM.
                                                                                         *          - 0x00 - 0x7F: Standard (NVRam) Key
                                                                                         *          - 0xE0 - 0xE3: RAM Key
                                                                                         */
        uint8_t bKeyVer,                                                                /**< [In] Key version to be used in SAM. */
        uint8_t  bNumOfRand,                                                            /**< [In] Maximum number of random bytes sent in one Cmd.
                                                                                         *        PICC_ProximityCheck
                                                                                         *          - M == 8: Only one iteration is made, SAM send all 8
                                                                                         *            random bytes in one Cmd.PICC_ProximityCheck
                                                                                         *          - M == 7: The SAM sends the first 7 random bytes in one
                                                                                         *            Cmd.PICC_ProximityCheck and the remaining random byte in
                                                                                         *            another one
                                                                                         *          - And so on as
                                                                                         *              - M = 6 + 2
                                                                                         *              - M = 5 + 3
                                                                                         *              - M = 4 + 4
                                                                                         *              - M = 3 + 3 + 2
                                                                                         *              - M = 2 + 2 + 2 + 2
                                                                                         *              - Finally M == 1
                                                                                         *          - The SAM sends 8 Cmd.PICC_ProximityCheck with one random
                                                                                         *            byte.
                                                                                         */
        uint8_t * pDivInput,                                                            /**< [In] Diversification Input used to diversify the key. */
        uint8_t bDivInputLen,                                                           /**< [In] Length bytes available in \b pDivInput buffer. */
        uint8_t ** ppResponse,                                                          /**< [Out] Response received from SAM.
                                                                                         *          - For PICC Error  : The PICC error code will be returned.
                                                                                         *          - For PICC Success: The PPCDataLen and PPCData will be returned.
                                                                                         *              - PPCDataLen: Length of Cmd.PICC_PreparePC response data
                                                                                         *              - PPCData: Cmd.PICC_PreparePC response data: Option || PubRespTime
                                                                                         *                [|| PPS1] [|| ActBitRate]
                                                                                         */
        uint16_t * pRespLen                                                             /**< [Out] Length bytes available in \b ppResponse buffer. */
    );

/**
 * end of group phhalHw_Sam_Cmd_VCA_XMode
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_VCA
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd
 * @}
 */

#endif /* NXPBUILD__PHHAL_HW_SAM */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHHALHW_SAM_CMD_VC_H */
