/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) Security and Configuration command implementation of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */


#ifndef PHHALHW_SAM_CMD_SC_H
#define PHHALHW_SAM_CMD_SC_H

#include <ph_Status.h>
#include <phhalHw.h>

#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */

#ifdef NXPBUILD__PHHAL_HW_SAM

/** \addtogroup phhalHw_Sam_Cmd
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_SC 02_SecurityConfiguration
 * \brief Implementation of SAM Security and Configuration commands
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_SC_Defines Defines
 * \brief Definitions for SAM Security and Configuration commands.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_SC_Defines_CommandCodes CommandCodes
 * \brief Definitions for Instruction codes
 * @{
 */
#define PHHAL_HW_SAM_CMD_INS_SAM_GET_VERSION                                    0x60U   /**< Sam Instruction code for SAM_GetVersion command. */
#define PHHAL_HW_SAM_CMD_INS_SAM_DISABLE_CRYPTO                                 0xDCU   /**< Sam Instruction code for SAM_DisableCrypto command. */
#define PHHAL_HW_SAM_CMD_INS_SAM_ACTIVATE_OFFLINE_KEY                           0x01U   /**< Sam Instruction code for SAM_ActivateOfflineKey command. */
#define PHHAL_HW_SAM_CMD_INS_SAM_LOAD_INIT_VECTOR                               0x71U   /**< Sam Instruction code for SAM_LoadInitVector command. */
#define PHHAL_HW_SAM_CMD_INS_SAM_KILL_AUTHENTICATION                            0xCAU   /**< Sam Instruction code for SAM_KillAuthenticate command. */
#define PHHAL_HW_SAM_CMD_INS_SAM_SELECT_APPLICATION                             0x5AU   /**< Sam Instruction code for SAM_SelectApplication command. */
#define PHHAL_HW_SAM_CMD_INS_SAM_GET_RANDOM                                     0X84U   /**< Sam Instruction code for SAM_GetChallenge command. */
#define PHHAL_HW_SAM_CMD_INS_SAM_SLEEP                                          0x51U   /**< Sam Instruction code for SAM_Sleep command. */
#define PHHAL_HW_SAM_CMD_INS_SAM_SET_CONFIGURATION                              0x3CU   /**< Sam Instruction code for SAM_SetConfiguration command. */
/**
 * end of group phhalHw_Sam_Cmd_SC_Defines_CommandCodes
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_SC_Defines_DisableCrypto DisableCrypto
 * \brief Definitions for disabling Crypto functionalities. To be used with \ref phhalHw_Sam_Cmd_SAM_DisableCrypto
 * "SAM_DisableCrypto" interface.
 * @{
 */
#define PHHAL_HW_SAM_CMD_SAM_DISABLE_CRYPTO_NO_CHANGE                       0x0000U    /**< Option mask for Disable Crypto with no change for programming mask bit. */
#define PHHAL_HW_SAM_CMD_SAM_DISABLE_CRYPTO_DES_PICC_CHANGE_KEY             0x0800U    /**< Option mask for Disable Crypto to disable DESFire Key change. */
#define PHHAL_HW_SAM_CMD_SAM_DISABLE_CRYPTO_DECRYPTION                      0x1000U    /**< Option mask for Disable Crypto to disable the decryption of data. */
#define PHHAL_HW_SAM_CMD_SAM_DISABLE_CRYPTO_ENCRYPTION                      0x2000U    /**< Option mask for Disable Crypto to disable encryption of data. */
#define PHHAL_HW_SAM_CMD_SAM_DISABLE_CRYPTO_MAC_VERIFICATION                0x4000U    /**< Option mask for Disable Crypto to disable verification of MAC. */
#define PHHAL_HW_SAM_CMD_SAM_DISABLE_CRYPTO_MAC_GENERATION                  0x8000U    /**< Option mask for Disable Crypto to disable generation of MAC. */
/**
 * end of group phhalHw_Sam_Cmd_SC_Defines_DisableCrypto
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_SC_Defines_LRP LRPUpdateKeys
 * \brief Definitions for LRP update keys. To be used with \ref phhalHw_Sam_Cmd_SAM_ActivateOfflineKey
 * "SAM_ActivateOfflineKey" interface.
 * @{
 */
#define PHHAL_HW_SAM_CMD_SAM_AO_NO_LRP_UPDATE_KEY                               0x00U   /**< Option mask for ActivateOffline with P1 information Bit 1 set to zero in case of
                                                                                         *   non AES_128 LRP keytype.
                                                                                         */
#define PHHAL_HW_SAM_CMD_SAM_AO_LRP_UPDATE_KEY_ONE                              0x00U   /**< Option mask for ActivateOffline with LRP Update keys to generate One
                                                                                         *   updated key (KeyID.LRPUpdate).
                                                                                         */
#define PHHAL_HW_SAM_CMD_SAM_AO_LRP_UPDATE_KEY_TWO                              0x02U   /**< Option mask for ActivateOffline with LRP Update keys to generate Two
                                                                                         *   updated key (KeyID.LRPMACUpdate and KeyID.LRPENCUpdate).
                                                                                         */
/**
 * end of group phhalHw_Sam_Cmd_SC_Defines_LRP
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_SC_Defines_LoadIV LoadIV
 * \brief Definitions for Load Initialization Vector. To be used with \ref phhalHw_Sam_Cmd_SAM_LoadInitVector
 * "SAM_LoadInitVector" interface.
 * @{
 */
#define PHHAL_HW_SAM_CMD_SAM_LOAD_IV_MODE_SET_IV                                0x00U  /**< Option mask for Load Init Vector to set the IV. */
#define PHHAL_HW_SAM_CMD_SAM_LOAD_IV_MODE_SET_LRP_ENC_CTR                       0x01U  /**< Option mask for Load Init Vector to set the LRP_EncCtr. */
/**
 * end of group phhalHw_Sam_Cmd_SC_Defines_LoadIV
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_SC_Defines_KillAuth KillAuthentication
 * \brief Definitions to kill authentication. To be used with \ref phhalHw_Sam_Cmd_SAM_KillAuthentication
 * "SAM_KillAuthentication" interface.
 * @{
 */
#define PHHAL_HW_SAM_CMD_SAM_KILL_AUTHENTICATION_FULL                           0x00U  /**< Option mask for killing any authentication on corresponding LC. */
#define PHHAL_HW_SAM_CMD_SAM_KILL_AUTHENTICATION_PARTIAL                        0x01U  /**< Option mask for killing PICC or offline key activation but preserving
                                                                                        *   any Host Authentication.
                                                                                        */
/**
 * end of group phhalHw_Sam_Cmd_SC_Defines_KillAuth
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_SC_Defines_SetConfig SetConfiguration
 * \brief Definitions to update SAM configuration settings. To be used with \ref phhalHw_Sam_Cmd_SAM_SetConfiguration
 * "SAM_SetConfiguration" interface.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_SC_Defines_SetConfig_ConfigID Configuration
 * \brief To be used with \ref phhalHw_Sam_Cmd_SAM_SetConfiguration "SAM_SetConfiguration" interface,
 * \b bOption parameter.
 * @{
 */
#define PHHAL_HW_SAM_CMD_SAM_SET_CONFIGURATION_HISTORICAL_BYTES                 0x00U  /**< Option mask for exchanging the historical bytes. */
#define PHHAL_HW_SAM_CMD_SAM_SET_CONFIGURATION_READER_IC_CONFIG                 0x01U  /**< Option mask for exchanging the reader IC configuration. */
#define PHHAL_HW_SAM_CMD_SAM_SET_CONFIGURATION_I2C_CLOCK_SPEED                  0x02U  /**< Option mask for exchanging the I2C processing clock speed configuration. */
#define PHHAL_HW_SAM_CMD_SAM_SET_CONFIGURATION_FULL_ATR                         0x04U  /**< Option mask for exchanging the Full ATR configuration. */
/**
 * end of group phhalHw_Sam_Cmd_SC_Defines_SetConfig_ConfigID
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_SC_Defines_SetConfig_RIC ReaderIC
 * \brief To be used with \ref phhalHw_Sam_Cmd_SAM_SetConfiguration "SAM_SetConfiguration" interface,
 * \b pData parameter for \b bOption = \ref PHHAL_HW_SAM_CMD_SAM_SET_CONFIGURATION_READER_IC_CONFIG
 * "Reader IC configuration".
 * @{
 */
#define PHHAL_HW_SAM_CMD_SAM_READER_IC_CONFIG_RC512                             0x01U  /**< Option mask for exchanging the reader IC configuration as RC512. */
#define PHHAL_HW_SAM_CMD_SAM_READER_IC_CONFIG_RC523                             0x02U  /**< Option mask for exchanging the reader IC configuration as RC523 (Default). */
#define PHHAL_HW_SAM_CMD_SAM_READER_IC_CONFIG_RC663                             0x03U  /**< Option mask for exchanging the reader IC configuration as RC663. */
/**
 * end of group phhalHw_Sam_Cmd_SC_Defines_SetConfig_RIC
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_SC_Defines_SetConfig_I2C I2CSpeed
 * \brief To be used with \ref phhalHw_Sam_Cmd_SAM_SetConfiguration "SAM_SetConfiguration" interface,
 * \b pData parameter for \b bOption = \ref PHHAL_HW_SAM_CMD_SAM_SET_CONFIGURATION_I2C_CLOCK_SPEED
 * "I2C processing clock speed".
 * @{
 */
#define PHHAL_HW_SAM_CMD_SAM_I2C_CLOCK_SPEED_0_5_MHZ                            0x04U  /**< Option mask I2C processing clock speed as 0.5 MHz. */
#define PHHAL_HW_SAM_CMD_SAM_I2C_CLOCK_SPEED_1_MHZ                              0x05U  /**< Option mask I2C processing clock speed as 1 MHz. */
#define PHHAL_HW_SAM_CMD_SAM_I2C_CLOCK_SPEED_2_MHZ                              0x06U  /**< Option mask I2C processing clock speed as 2 MHz. */
#define PHHAL_HW_SAM_CMD_SAM_I2C_CLOCK_SPEED_3_MHZ                              0x07U  /**< Option mask I2C processing clock speed as 3 MHz. */
#define PHHAL_HW_SAM_CMD_SAM_I2C_CLOCK_SPEED_4_MHZ                              0x08U  /**< Option mask I2C processing clock speed as 4 MHz. */
#define PHHAL_HW_SAM_CMD_SAM_I2C_CLOCK_SPEED_6_MHZ                              0x09U  /**< Option mask I2C processing clock speed as 6 MHz. */
#define PHHAL_HW_SAM_CMD_SAM_I2C_CLOCK_SPEED_8_MHZ                              0x0AU  /**< Option mask I2C processing clock speed as 8 MHz. */
#define PHHAL_HW_SAM_CMD_SAM_I2C_CLOCK_SPEED_12_MHZ                             0x0BU  /**< Option mask I2C processing clock speed as 12 MHz (Default). */
#define PHHAL_HW_SAM_CMD_SAM_I2C_CLOCK_SPEED_16_MHZ                             0x0CU  /**< Option mask I2C processing clock speed as 16 MHz. */
#define PHHAL_HW_SAM_CMD_SAM_I2C_CLOCK_SPEED_FREE_RUN                           0x0FU  /**< Option mask I2C processing clock speed as free running mode. */
/**
 * end of group phhalHw_Sam_Cmd_SC_Defines_SetConfig_I2C
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_SC_Defines_SetConfig
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_SC_Defines
 * @}
 */

/**
 * \brief Get version information from the SAM. Returns
 *  - <b>Bytes[0 - 6]: Hardware version information</b>
 *      - Byte[0]      : Vendor ID (set to 0x04 for NXP)
 *      - Byte[1]      : Type
 *      - Byte[2]      : Subtype
 *      - Byte[3]      : Major version number
 *      - Byte[4]      : Minor version number
 *      - Byte[5]      : Storage size
 *      - Byte[6]      : Communication protocol type (set to 0x01, meaning T=1)
 *
 *  - <b>Bytes[7 - 13]: Software version information</b>
 *      - Byte[7]      : Vendor ID (set to 0x04 for NXP)
 *      - Byte[8]      : Type
 *      - Byte[9]      : Subtype
 *      - Byte[10]     : Major version number
 *      - Byte[11]     : Minor version number
 *      - Byte[12]     : Storage size
 *      - Byte[13]     : Communication protocol type (set to 0x01, meaning T=1)
 *
 *  - <b>Bytes[14 - 30]: Manufacturer data</b>
 *      - Byte[14 - 20]: Code the unique serial number
 *      - Byte[11 - 25]: Code the production batch number
 *      - Byte[26]     : Code the day of production [0x00..0x1F]
 *      - Byte[27]     : Code the month of production [0x00..0x0C]
 *      - Byte[28]     : Code the year of production (year is 2000 + code)
 *      - Byte[29]     : Global crypto settings; most significant byte of CryptoSecRow
 *      - Byte[30]     : Code the SAM version and activated state (set to 0x0X for the unactivated MIFARE SAM AVx,
 *                       set to 0xAX after SAM Activation). For Ex: 0x04 will be for SAM AV4 Unactivated and 0xA4 will
 *                       be for SAM AV4 after SAM Activation. For future SAM version the number will increment from
 *                       lower nibble.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pVersion and \b pVerLen is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_GetVersion(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t * pVersion,                                                             /**< [Out] Buffer containing the read version. */
        uint8_t * pVerLen                                                               /**< [Out] Amount of valid bytes in the \b pVersion buffer. */
    );

/**
 * \brief Disable Crypto-related features of the SAM permanently and is irreversible.
 *  - The SAM shall maintain the status of the cryptographic functionality in the Global Crypto Security Row
 *    (CryptoSecRow).
 *  - The command shall apply the ProMas bits to the CryptoSecRow and shall not be able to unset the bit (cannot
 *    re-enable the functionality).
 *  - The initial value of the CryptoSecRow are set to 0x0000, therefore full cryptographic functionality is available.
 *  - The configuration settings (SET) of all KST key entries shall reflect the changes after the execution of
 *    SAM_DisableCrypto.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_DisableCrypto(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wProMas                                                                /**< [In] Two byte mask to specify the desired settings for
                                                                                         *        cryptography-related features.
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_DISABLE_CRYPTO_NO_CHANGE
                                                                                         *                 "No change for programming mask bit"
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_DISABLE_CRYPTO_DES_PICC_CHANGE_KEY
                                                                                         *                 "Disable DESFire (PICC) key change"
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_DISABLE_CRYPTO_DECRYPTION
                                                                                         *                 "Disable decryption"
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_DISABLE_CRYPTO_ENCRYPTION
                                                                                         *                 "Disable encryption"
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_DISABLE_CRYPTO_MAC_VERIFICATION
                                                                                         *                 "Disable MAC verification"
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_DISABLE_CRYPTO_MAC_GENERATION
                                                                                         *                 "Disable MAC generation"
                                                                                         */
    );



/**
 * \brief Activation of an OfflineCrypto or an OfflineChange Key.
 * SAM_ActivateOfflineKey shall activate Offline keys for further offline data processing.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pDivInput is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_ActivateOfflineKey(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                                /**< [In] Option to update the P1 information if Keytype is AES_128LRP.
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_AO_LRP_UPDATE_KEY_ONE
                                                                                         *                 "Generate one updated key"
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_AO_LRP_UPDATE_KEY_TWO
                                                                                         *                 "Generate two updated keys"
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_AO_NO_LRP_UPDATE_KEY
                                                                                         *                 "NO LRP Update", if keytype is
                                                                                         *                 other than AES_128LRP.
                                                                                         */
        uint8_t bKeyNo,                                                                 /**< [In] Key reference number in SAM KeyStore. */
        uint8_t bKeyVer,                                                                /**< [In] Key version to be used in SAM KeyStore. */
        uint8_t * pDivInput,                                                            /**< [In] Diversification Input used to diversify the key. */
        uint8_t bDivInputLen                                                            /**< [In] Length bytes available in \b pDivInput buffer. */
    );

/**
 * \brief Load an Init Vector for the next cryptographic operation into the SAM.
 *  - SAM_LoadInitVector are used to load an init vector (IV) or encryption counter (LRP_EncCtr) for next
 *    cryptographic operations.
 *  - The loaded IV will be applied in the next cryptographic operation independent from the <b>Keep IV</b>
 *    setting of the key entry, except for the authentication commands where the init vector is reset
 *    to zero.
 *  - If a KeyType.AES_128LRP key is loaded, the command can also be used to set the LRP_EncCtr used during
 *    encryption and decryption.
 *  - This is indicated by P1. If the LRP_EncCtr is to be set, a value of up to 16 bytes is given.
 *  - Once set, the counter is further maintained and incremented by the SAM.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pData is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_LoadInitVector(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                                /**< [In] One of the below option.
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_LOAD_IV_MODE_SET_IV
                                                                                         *                 "Set Initialization Vector"
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_LOAD_IV_MODE_SET_LRP_ENC_CTR
                                                                                         *                 "Set LRP Encryption Counter"
                                                                                         */
        uint8_t * pData,                                                                /**< [In] Data based on the option selected.
                                                                                         *          - If \ref PHHAL_HW_SAM_CMD_SAM_LOAD_IV_MODE_SET_IV
                                                                                         *            "Set Initialization Vector", 8 bytes of initialization
                                                                                         *            vector for DES or
                                                                                         *            16 bytes of initialization vector for AES.
                                                                                         *          - If \ref PHHAL_HW_SAM_CMD_SAM_LOAD_IV_MODE_SET_LRP_ENC_CTR
                                                                                         *            "Set LRP Encryption Counter", 1 - 16 bytes Encryption counter.
                                                                                         */
        uint8_t bDataLen                                                                /**< [In] The length of bytes available in \b pData buffer. */
    );

/**
 * \brief Kill all active authentications in this logical channel.
 * SAM_KillAuthentication invalidates either any authentication or all but the Host Authentication
 * on the LC on which the command is applied
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_KillAuthentication(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption                                                                 /**< [In] The type of authentication to be killed.
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_KILL_AUTHENTICATION_FULL
                                                                                         *                 "Kill any authentication on corresponding LC"
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_SAM_KILL_AUTHENTICATION_PARTIAL
                                                                                         *                 "Kill PICC authentication or Offline Key activation but
                                                                                         *                  preserves any Host Authentication
                                                                                         */
    );

/**
 * \brief Select an application by the DF_AID
 *  - SAM_SelectApplication is the equivalent of the SelectApplication command of DESFire.
 *  - The SAM generates a list of available keys linked to the specified Application ID as defined
 *    in the key entry property DF_AID.
 *  - For every key number, up to 6 key versions can be stored in the list (so it can read the keys
 *    from maximum two key entries per DESFire AID and DESFire key number).
 *  - This list is filled starting with key entry zero. If the KST contains more than 6 key versions
 *    per DESFire AID and DESFire key number, only the first 6 versions will be listed.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pDF_Aid is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_SelectApplication(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t * pDF_Aid                                                               /**< [In] DESFire application identifier (3 bytes). */
    );

/**
 * \brief Gets a random number from SAM.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pRnd is NULL.
 - \retval #PH_ERR_LENGTH_ERROR         If Number of returned bytes are not equal expected number.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_GetRandom(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bExpLen,                                                                /**< [In] The length of random bytes expected from SAM. */
        uint8_t * pRnd                                                                  /**< [Out] The random number returned by Sam. */
    );

/**
 * \brief Set the SAM into power down mode.
 *  - SAM_Sleep will force the SAM to put a connected reader chip into sleep mode and itself into idle mode to
 *    reduce power consumption
 *  - The SAM will answer the command and afterwards switch to idle mode.
 *  - The SAM will automatically return to normal operation after receiving the first character of the next command.
 *  - The reader chip will stay in sleep mode until a command is issued which utilizes the reader IC.
 *  - Then the SAM automatically carries out the wake-up sequence before starting the requested operation.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_Sleep(
        phhalHw_Sam_DataParams_t * pDataParams                                          /**< [In] Pointer to this layer's parameter structure. */
    );

/**
 * \brief Used to update SAM configuration settings.
 * A successful Host Authentication with one of the three SAM master keys (KeyNo set to 0x00) is
 * required to perform the command
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pData is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_SAM_SetConfiguration(
        phhalHw_Sam_DataParams_t * pDataParams,                                 /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                        /**< [In] Configuration setting ID. Define length and content of the Data parameter.
                                                                                 *          - \ref PHHAL_HW_SAM_CMD_SAM_SET_CONFIGURATION_HISTORICAL_BYTES
                                                                                 *                 "ATR historical bytes"
                                                                                 *          - \ref PHHAL_HW_SAM_CMD_SAM_SET_CONFIGURATION_READER_IC_CONFIG
                                                                                 *                 "Reader IC configuration"
                                                                                 *          - \ref PHHAL_HW_SAM_CMD_SAM_SET_CONFIGURATION_I2C_CLOCK_SPEED
                                                                                 *                 "I2C processing clock speed"
                                                                                 *          - \ref PHHAL_HW_SAM_CMD_SAM_SET_CONFIGURATION_FULL_ATR
                                                                                 *                 "Full ATR configuration"
                                                                                 */
        uint8_t * pData,                                                        /**< [In] Configuration setting data.
                                                                                 *          - If \ref PHHAL_HW_SAM_CMD_SAM_SET_CONFIGURATION_HISTORICAL_BYTES
                                                                                 *            "ATR historical bytes", the historical bytes should be exchanged.
                                                                                 *
                                                                                 *          - If \ref PHHAL_HW_SAM_CMD_SAM_SET_CONFIGURATION_READER_IC_CONFIG
                                                                                 *            "Reader IC configuration", one of the below values,
                                                                                 *              - \ref PHHAL_HW_SAM_CMD_SAM_READER_IC_CONFIG_RC512 "RC512"
                                                                                 *              - \ref PHHAL_HW_SAM_CMD_SAM_READER_IC_CONFIG_RC523 "RC523 (Default)"
                                                                                 *              - \ref PHHAL_HW_SAM_CMD_SAM_READER_IC_CONFIG_RC663 "RC663"
                                                                                 *
                                                                                 *          - If \ref PHHAL_HW_SAM_CMD_SAM_SET_CONFIGURATION_I2C_CLOCK_SPEED
                                                                                 *            "I2C processing clock speed", one of the below values,
                                                                                 *              - \ref PHHAL_HW_SAM_CMD_SAM_I2C_CLOCK_SPEED_0_5_MHZ "0.5 MHz"
                                                                                 *              - \ref PHHAL_HW_SAM_CMD_SAM_I2C_CLOCK_SPEED_1_MHZ "1 MHz"
                                                                                 *              - \ref PHHAL_HW_SAM_CMD_SAM_I2C_CLOCK_SPEED_2_MHZ "2 MHz"
                                                                                 *              - \ref PHHAL_HW_SAM_CMD_SAM_I2C_CLOCK_SPEED_3_MHZ "3 MHz"
                                                                                 *              - \ref PHHAL_HW_SAM_CMD_SAM_I2C_CLOCK_SPEED_4_MHZ "4 MHz"
                                                                                 *              - \ref PHHAL_HW_SAM_CMD_SAM_I2C_CLOCK_SPEED_6_MHZ "6 MHz"
                                                                                 *              - \ref PHHAL_HW_SAM_CMD_SAM_I2C_CLOCK_SPEED_8_MHZ "8 MHz"
                                                                                 *              - \ref PHHAL_HW_SAM_CMD_SAM_I2C_CLOCK_SPEED_12_MHZ "12 MHz (Default)"
                                                                                 *              - \ref PHHAL_HW_SAM_CMD_SAM_I2C_CLOCK_SPEED_16_MHZ "16 MHz"
                                                                                 *              - \ref PHHAL_HW_SAM_CMD_SAM_I2C_CLOCK_SPEED_FREE_RUN "Free running mode"
                                                                                 *
                                                                                 *          - If \ref PHHAL_HW_SAM_CMD_SAM_SET_CONFIGURATION_FULL_ATR "Full ATR update"
                                                                                 */
        uint8_t bDataLen                                                        /**< [In] Length of Configuration setting data. */
    );

/**
 * end of group phhalHw_Sam_Cmd_SC
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd
 * @}
 */

#endif /* NXPBUILD__PHHAL_HW_SAM */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHHALHW_SAM_CMD_SC_H */
