/*
 * Copyright 2013, 2017, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

#ifndef PH_DEFINES_H
#define PH_DEFINES_H

#define USE_FOR_NEW_READER_LIB

/* Disable 'cast truncates constant value' warnings */
#pragma warning(disable:4310)

#define PH_RDL_PARAM_MASK 0xFFFF

/**
 * PARAMETER GROUPS
 */

/**
* The Definition of the IDs for the deferent Layers of the HAL Stack
*/
enum phHalObjects
{
    PH_RDL_HAL_BEGIN                    = 0x0000,           /*!< \brief 0x0000 Indicates the first value of the HAL Stack Ids. */
    PH_RDL_HAL_READER                   = PH_RDL_HAL_BEGIN, /*!< \brief 0x0000 Indicates that id for the Reader Object in the HAL stack. */
    PH_RDL_HAL_TIMEMEASURE,                                 /*!< \brief 0x0001 Indicates that id for the time measurement layer in the HAL stack. */
#ifndef USE_FOR_NEW_READER_LIB
    PH_RDL_HAL_MIFARE,                                      /*!< \brief 0x0002 Indicates that id for the mifare layer in the HAL stack. */
#endif
    PH_RDL_HAL_PARITYCRC,                                   /*!< \brief 0x0003 Indicates that id for the parity crc layer in the HAL stack. */
#ifndef USE_FOR_NEW_READER_LIB
    PH_RDL_HAL_ISO14443_3,                                  /*!< \brief 0x0004 Indicates that id for the iso14443-3 layer in the HAL stack. */
    PH_RDL_HAL_ISO14443_4,                                  /*!< \brief 0x0005 Indicates that id for the iso14443-4 layer in the HAL stack. */
#endif
    PH_RDL_HAL_END,

    PH_RDL_SUPPORT_HAL_BEGIN = PH_RDL_HAL_END,
    PH_RDL_SUPPORT_HAL_FPGA = PH_RDL_SUPPORT_HAL_BEGIN,     /*!< \brief 0x0006 Indicates that id for the FPGA in the HAL stack. */
    PH_RDL_SUPPORT_HAL_END,

#ifndef USE_FOR_NEW_READER_LIB
    PH_RDL_APPL_BEGIN = PH_RDL_SUPPORT_HAL_END,             /*!< \brief 0x0008 Indicates that it is not an HAL Stack object. */
    PH_RDL_APPL_MIFARE=PH_RDL_APPL_BEGIN,
    PH_RDL_APPL_MIFARE_PLUS,
    PH_RDL_APPL_END,
    PH_RDL_HAL_OBJECTS_LENGTH = PH_RDL_HAL_END-PH_RDL_HAL_BEGIN + PH_RDL_SUPPORT_HAL_END-PH_RDL_SUPPORT_HAL_BEGIN  + PH_RDL_APPL_END-PH_RDL_APPL_BEGIN,
#else
    PH_RDL_HAL_OBJECTS_LENGTH = PH_RDL_HAL_END-PH_RDL_HAL_BEGIN + PH_RDL_SUPPORT_HAL_END,
#endif
    PH_RDL_HAL_NOHALOBJECT                                  /*!< \brief 0x0007 Indicates that it is not an HAL Stack object. */
};

/**
 * PARAMETERS
 */

/* READER PARAMETERS */
enum phReaderParams
{
    PH_RDL_READER_PARAMETERS_BEGIN              = PH_RDL_HAL_READER << 16,          /*!< \brief 0x00000000 Indicates the Begin of the Reader Parameters */
    PH_RDL_READER_RESET_OFF_TIME                = PH_RDL_READER_PARAMETERS_BEGIN,   /*!< \brief 0x00000000 Indicates that id for the time of a HF Pause in [ms] */
    PH_RDL_READER_RESET_ON_TIME,                /*!< \brief 0x00000001 Indicates that id for the time the reader does not modulate after a HF Pause. */
    PH_RDL_READER_COMM_UPLINK_SPEED_MAX,        /*!< \brief 0x00000002 Indicates that id for the maximum communication speed from PCD to PCCD (see #phCommSpeed for params). */
    PH_RDL_READER_COMM_DOWNLINK_SPEED_MAX,      /*!< \brief 0x00000003 Indicates that id for the maximum communication speed from PCCD to PCD (see #phCommSpeed for params). */
    PH_RDL_READER_COMM_UPLINK_SPEED,            /*!< \brief 0x00000004 Indicates that id for the current communication speed from PCD to PCCD (see #phCommSpeed for params). */
    PH_RDL_READER_COMM_DOWNLINK_SPEED,          /*!< \brief 0x00000005 Indicates that id for the current communication speed from PCCD to PCD (see #phCommSpeed for params). */
    PH_RDL_READER_COMM_ALLOW_ASYNC_SPEED,       /*!< \brief 0x00000006 Indicates that id for the permission of asynchronous communication speeds (see #phCommSpeed for params)*/
    PH_RDL_READER_TIMEOUT_ETU,                  /*!< \brief 0x00000007 Indicates that id for the time (in ETUs; 1 ETU = 128/fc) the reader waits for a response until it detects a timeout. */
    PH_RDL_READER_TIMEOUT_BEHAVIOR,             /*!< \brief 0x00000008 Indicates that id for the timeout behavior (see #phCommTimeOut for params). */
    PH_RDL_READER_FSDI_MAX,                     /*!< \brief 0x00000009 Indicates that id for the maximum FSDI (note: the FSDI codes the FSD which codes the maximum Number of Bytes, the reader is able to receive) the reader is able to receive. */
    PH_RDL_READER_ACCESS_REGISTER,              /*!< \brief 0x0000000A Indicates that id for the access of a joiner register. Note: When writing to the register the param has to be coded as ((0xFF00 & adress) | (0x00FF & value)); When reading the param is the desired adress. */
    PH_RDL_READER_CRC_MODE,                     /*!< \brief 0x0000000B Indicates that id for the setting of the CRC modes (see #phHALConfig for params). */
    PH_RDL_READER_PARITY_ENABLED,               /*!< \brief 0x0000000C Indicates that id for the use of the patity bit (see #phHALConfig for params). */
    PH_RDL_READER_PARITY_MODE,                  /*!< \brief 0x0000000D Indicates that id for the mode of parity bit (odd or even) (see #phHALConfig for params). */
    PH_RDL_READER_RXALIGN,                      /*!< \brief 0x0000000E Indicates that id for the return buffer alignment. Or in other words: indicates where the first bit is stored in the return buffer. */
    PH_RDL_READER_EMD_SUPPRESSION,              /*!< \brief 0x0000000F Indicates that id for the activation or deactivation of the Electro-Magnetic Distortion (EMD) Suppression.*/
    PH_RDL_READER_RF_FREQUENCY,                 /*!< \brief 0x00000010 Indicates that id for the RF frequency. */
    PH_RDL_READER_RF_FIELD_STRENGTH,            /*!< \brief 0x00000011 Indicates that id for the RF field strength. */
    PH_RDL_READER_LOGGING_STATUS,               /*!< \brief 0x00000012 Indicates that id for the logging status (see #phHALConfig for params).*/
    PH_RDL_READER_CARD_TYPE,                    /*!< \brief 0x00000013 Indicates that id for the card type (see #phCardTypes for params).  @warning Its is necessary to manually load the corresponding pattern for the selected card type.*/
    PH_RDL_READER_VALUES_AFTER_COLL,            /*!< \brief 0x00000014 Indicates that id for the reader behaviour ofter collisions (note: #phHALConfig::PH_RDL_DEACTIVATED joiner stops listening after a collision; #phHALConfig::PH_RDL_ACTIVATED joiner continues listening after a collision). */
    PH_RDL_READER_PARAMETERS_END,               /*!< \brief 0x00000015 Indicates that the id is not a id of #phReaderParams */
    PH_RDL_READER_PARAMETERS_LENGTH = (PH_RDL_READER_PARAMETERS_END - PH_RDL_READER_PARAMETERS_BEGIN)
};

/* TIMING MEASUREMENT PARAMETERS */
enum phTimeMeasureParams
{
    PH_RDL_TIMEMEASURE_PARAMETERS_BEGIN         = PH_RDL_HAL_TIMEMEASURE << 16,         /*!< \brief 0x00010000 Indicates the Begin of the Time Measurement Parameters */
    PH_RDL_TIMEMEASURE_ACTIVATION_STATUS        = PH_RDL_TIMEMEASURE_PARAMETERS_BEGIN,  /*!< \brief 0x00010000 Indicates that id for the parameter to activate or deactivate the Time Measurement HAL (see #phHALConfig::PH_RDL_DEACTIVATED and #phHALConfig::PH_RDL_ACTIVATED for params). */
    PH_RDL_TIMEMEASURE_START_POS,               /*!< \brief 0x00010001 Indicates that id for the start position for the time measurement (see #phHALConfig for params). */
    PH_RDL_TIMEMEASURE_END_POS,                 /*!< \brief 0x00010002 Indicates that id for the end position for the time measurement (see #phHALConfig for params). */
    PH_RDL_TIMEMEASURE_LOGGING_STATUS,          /*!< \brief 0x00010003 Indicates that id for the logging status (see #phHALConfig for params). */
    PH_RDL_TIMEMEASURE_PARAMETERS_END,          /*!< \brief 0x00010004 Indicates that the id is not a id of #phTimeMeasureParams */
    PH_RDL_TIMEMEASURE_PARAMETERS_LENGTH = (PH_RDL_TIMEMEASURE_PARAMETERS_END - PH_RDL_TIMEMEASURE_PARAMETERS_BEGIN)
};

#ifndef USE_FOR_NEW_READER_LIB
/* MIFARE CRYPTO PARAMETERS */
enum phMifareParams
{
    PH_RDL_MIFARE_PARAMETERS_BEGIN              = PH_RDL_HAL_MIFARE << 16,
    PH_RDL_MIFARE_ACTIVATION_STATUS             = PH_RDL_MIFARE_PARAMETERS_BEGIN,
    PH_RDL_MIFARE_RESET_CRYPTO1,
    PH_RDL_MIFARE_LOGGING_STATUS,
    PH_RDL_MIFARE_PARAMETERS_END,
    PH_RDL_MIFARE_PARAMETERS_LENGTH = (PH_RDL_MIFARE_PARAMETERS_END - PH_RDL_MIFARE_PARAMETERS_BEGIN)
};
#endif

/* PARITY CRC PARAMETERS */
enum phParityCRCParams
{
    PH_RDL_PARITYCRC_PARAMETERS_BEGIN           = PH_RDL_HAL_PARITYCRC << 16,           /*!< \brief 0x00030000 Indicates the Begin of the Parity and CRC Parameters */
    PH_RDL_PARITYCRC_ACTIVATION_STATUS          = PH_RDL_PARITYCRC_PARAMETERS_BEGIN,    /*!< \brief 0x00030000 Indicates that id for the parameter to activate or deactivate the Parity and CRC HAL (see #phHALConfig::PH_RDL_DEACTIVATED and #phHALConfig::PH_RDL_ACTIVATED for params). */
    PH_RDL_PARITYCRC_CRC_MODE,              /*!< \brief 0x00030001 Indicates that id for the crc mode (see #phHALConfig for params). */
    PH_RDL_PARITYCRC_PARITY_ENABLED,        /*!< \brief 0x00030002 Indicates that id for the use of the patity bit (see #phHALConfig for params). */
    PH_RDL_PARITYCRC_PARITY_MODE,           /*!< \brief 0x00030003 Indicates that id for the mode of parity bit (odd or even) (see #phHALConfig for params). */
    PH_RDL_PARITYCRC_RXALIGN,               /*!< \brief 0x00030004 Indicates that id for the return buffer alignment.  Or in other words: indicates where the first bit is stored in the return buffer. */
    PH_RDL_PARITYCRC_LOGGING_STATUS,        /*!< \brief 0x00030005 Indicates that id for the logging status (see #phHALConfig for params). */
    PH_RDL_PARITYCRC_PARAMETERS_END,        /*!< \brief 0x00030006 Indicates that the id is not a id of #phParityCRCParams */
    PH_RDL_PARITYCRC_PARAMETERS_LENGTH = (PH_RDL_PARITYCRC_PARAMETERS_END - PH_RDL_PARITYCRC_PARAMETERS_BEGIN)
};

#ifndef USE_FOR_NEW_READER_LIB
/* ISO 14443_3 PARAMETERS */
enum phISO144433Params
{
    PH_RDL_ISO14443_3_PARAMETERS_BEGIN          = PH_RDL_HAL_ISO14443_3 << 16,          /*!< \brief 0x00040000 Indicates the Begin of the ISO14443 Layer 3 Parameters */
    PH_RDL_ISO14443_3_ACTIVATION_STATUS         = PH_RDL_ISO14443_3_PARAMETERS_BEGIN,   /*!< \brief 0x00040000 Indicates that id for the parameter to activate or deactivate the ISO14443 Layer 3 HAL (see #phHALConfig::PH_RDL_DEACTIVATED and #phHALConfig::PH_RDL_ACTIVATED for params). */
    PH_RDL_ISO14443_3_LOGGING_STATUS,                                                   /*!< \brief 0x00040001 Indicates that id for the logging status (see #phHALConfig for params). */
    PH_RDL_ISO14443_3_EXT_ATQB,                                                         /*!< \brief 0x00040002 Indicates that id for the use of extended AtqB. (see #phHALConfig for params)*/
    PH_RDL_ISO14443_3_PARAMETERS_END,                                                   /*!< \brief 0x00040003 Indicates that the id is not a id of #phISO144433Params */
    PH_RDL_ISO14443_3_PARAMETERS_LENGTH = (PH_RDL_ISO14443_3_PARAMETERS_END - PH_RDL_ISO14443_3_PARAMETERS_BEGIN)
};

/* ISO 14443_4 PARAMETERS */
enum phISO144434Params
{
    PH_RDL_ISO14443_4_PARAMETERS_BEGIN          = PH_RDL_HAL_ISO14443_4 << 16,          /*!< \brief 0x00050000 Indicates the Begin of the ISO14443 Layer 4 Parameters */
    PH_RDL_ISO14443_4_ACTIVATION_STATUS         = PH_RDL_ISO14443_4_PARAMETERS_BEGIN,   /*!< \brief 0x00050000 Indicates that id for the parameter to activate or deactivate the ISO14443 Layer 4 HAL (see #phHALConfig::PH_RDL_DEACTIVATED and #phHALConfig::PH_RDL_ACTIVATED for params). */
    PH_RDL_ISO14443_4_BNR,                          /*!< \brief 0x00050001 Indicates that id for the setting of a block number (note: params can either be 0 or 1). */
    PH_RDL_ISO14443_4_CID_EN,                       /*!< \brief 0x00050002 Indicates that id for the activation or deactivation of CID (note: params can either be 0 or 1). */
    PH_RDL_ISO14443_4_NAD_EN,                       /*!< \brief 0x00050003 Indicates that id for the activation or deactivation of NAD (note: params can either be 0 or 1). */
    PH_RDL_ISO14443_4_NAD,                          /*!< \brief 0x00050004 Indicates that id for the NAD. */
    PH_RDL_ISO14443_4_NUM_RETRANSMIT,               /*!< \brief 0x00050005 Indicates that id for the number of retransmitts in case of a transmission error. */
    PH_RDL_ISO14443_4_FWI,                          /*!< \brief 0x00050006 Indicates that id for the FWI (frame waiting time integer). */
    PH_RDL_ISO14443_4_ACTIVE_CID,                   /*!< \brief 0x00050007 Indicates that id for the active CID. */
    PH_RDL_ISO14443_4_NEXT_FREE_CID,                /*!< \brief 0x00050008 Indicates that id for the next free CID. */
    PH_RDL_ISO14443_4_FREE_CID,                     /*!< \brief 0x00050009 Indicates that id for freeing a CID. (note: param is CID to free). */
    PH_RDL_ISO14443_4_CARD_TYPE,                    /*!< \brief 0x0005000A Indicates that id for card type (see #phCardTypes for params). */
    PH_RDL_ISO14443_4_FORCE_PPS_SPEED,              /*!< \brief 0x0005000B Indicates that id for activating or deactivating the forcing of an other PPS speed. Note: if activated the reader forces the PPS speed desired and returns an error message in case that speed is higher than supported by the PICC; if deactivated the highest speed supported by the PICC and the PCD is chosen. */
    PH_RDL_ISO14443_4_FSCI,                         /*!< \brief 0x0005000C Indicates that id for the FSCI. */
    PH_RDL_ISO14443_4_CID_SUPPORT,                  /*!< \brief 0x0005000D Indicates that id for the the CID support of the selected PICC. */
    PH_RDL_ISO14443_4_PROXIMITY_STATE,              /*!< \brief 0x0005000E */
    PH_RDL_ISO14443_4_SEND_DESELECT_TO_LOST_CARD,   /*!< \brief 0x0005000F Defines if after retransmits are done, a DESELECT is sent to the card */
    PH_RDL_ISO14443_4_ALWAYS_TOGGLE_BLKNR,          /*!< \brief 0x00050010 Defines if the Block Number is always changed also if the card does not respond. */
    PH_RDL_ISO14443_4_LOGGING_STATUS,               /*!< \brief 0x00050011 Indicates that id for the logging status (see #phHALConfig for params). */
    PH_RDL_ISO14443_4_FSDI,                         /*!< \brief 0x00050012 Indicates that id for the FSDI. */
    PH_RDL_ISO14443_4_SOF,                          /*!< \brief 0x00050013 Indicates that id for the SOF for TypeB. (0 indicates SOF required, 1 inicates SOF not required). */
    PH_RDL_ISO14443_4_EOF,                          /*!< \brief 0x00050014 Indicates that id for the EOF for TypeB. (0 indicates EOF required, 1 inicates EOF not required). */
    PH_RDL_ISO14443_4_TR0,                          /*!< \brief 0x00050015 Indicates that id for the TR0 for TypeB. */
    PH_RDL_ISO14443_4_TR1,                          /*!< \brief 0x00050016 Indicates that id for the TR1 for TypeB. */
    PH_RDL_ISO14443_4_SINGLE_ACTIVATION,            /*!< \brief 0x00050017 Indicates that id for the single card activation. i.e. activating cards with CID 0. */
    PH_RDL_ISO14443_4_PARAMETERS_END,               /*!< \brief 0x00050018 Indicates that the id is not a id of #phISO144434Params. */
    PH_RDL_ISO14443_4_PARAMETERS_LENGTH = (PH_RDL_ISO14443_4_PARAMETERS_END - PH_RDL_ISO14443_4_PARAMETERS_BEGIN)
};
#endif

#ifndef USE_FOR_NEW_READER_LIB
/* APP_MIFARE_PLUS PARAMETERS */
enum phApplMifarePlusParams
{
    PH_RDL_APPL_MIFARE_PLUS_PARAMETERS_BEGIN          = PH_RDL_APPL_MIFARE_PLUS << 16,
    PH_RDL_APPL_MIFARE_PLUS_LOGGING_STATUS            = PH_RDL_APPL_MIFARE_PLUS_PARAMETERS_BEGIN,
    PH_RDL_APPL_MIFARE_PLUS_PARAMETERS_END,
    PH_RDL_APPL_MIFARE_PLUS_PARAMETERS_LENGTH = (PH_RDL_APPL_MIFARE_PLUS_PARAMETERS_END - PH_RDL_APPL_MIFARE_PLUS_PARAMETERS_BEGIN)
};

/* APP_MIFARE PARAMETERS */
enum phApplMifareParams
{
    PH_RDL_APPL_MIFARE_PARAMETERS_BEGIN       = PH_RDL_APPL_MIFARE << 16,
    PH_RDL_APPL_MIFARE_LOGGING_STATUS         = PH_RDL_APPL_MIFARE_PARAMETERS_BEGIN,
    PH_RDL_APPL_MIFARE_PARAMETERS_END,
    PH_RDL_APPL_MIFARE_PARAMETERS_LENGTH      = (PH_RDL_APPL_MIFARE_PARAMETERS_END - PH_RDL_APPL_MIFARE_PARAMETERS_BEGIN)
};
#endif

/* FPGA PARAMS */
enum phFPGAParams
{
    PH_RDL_FPGA_PARAMETERS_BEGIN              = PH_RDL_SUPPORT_HAL_FPGA << 16,  /*!< \brief 0x00060000 Indicates the Begin of the FPGA Parameters */
    PH_RDL_FPGA_ACTIVATION_STATUS             = PH_RDL_FPGA_PARAMETERS_BEGIN,   /*!< \brief 0x00060000 Indicates that id for the parameter to activate or deactivate the FPGA (see #phFPGAValues). */
    PH_RDL_FPGA_SEQUENCE_ON_TRIGGER,         /*!< \brief 0x00060001 Indicates that id for the selection of the data sequence to play on trigger (see #phFPGAValues). */
    PH_RDL_FPGA_SELECTED_SEQUENCE,           /*!< \brief 0x00060002 Indicates that id for the selection of the data sequence to to set some parameter (see #phFPGAValues). */
    PH_RDL_FPGA_RESET_CYCLES_OFF,            /*!< \brief 0x00060003 Indicates that id for the setting of the number of OFF cycles of the rf pause. */
    PH_RDL_FPGA_RESET_CYCLES_ON,             /*!< \brief 0x00060004 Indicates that id for the setting of the number of ON cycles after the rf pause. */
    PH_RDL_FPGA_DEFAULT_SEQUENCE_RELOAD_MODE,/*!< \brief 0x00060005 Indicates that id for the setting of the reload mode of the patterns (see #phFPGAValues). */
    PH_RDL_FPGA_TIMEOUT_BEHAVIOUR,           /*!< \brief 0x00060006 Indicates that id for the setting of the timeout behavior of the FPGA (see #phFPGAValues) */
    PH_RDL_FPGA_SEQUENCE_FDT_OFFSET_PCD,     /*!< \brief 0x00060007 Indicates that id for the setting of the FDT Offset PCD (the pattern must be selected previous with #PH_RDL_FPGA_SELECTED_SEQUENCE) */
    PH_RDL_FPGA_AUTO_ADJUST_SEQUENCE,        /*!< \brief 0x00060008 Indicates that id for the setting of the Auto Ajust Sequence mode (see #phFPGAValues). */
    PH_RDL_FPGA_FILTER_MODE,                 /*!< \brief 0x00060009 Indicates that id for the setting of the Filter Mode (see #phFPGAValues). */
    PH_RDL_FPGA_DEFAULT_ENVELOPED_VALUE,     /*!< \brief 0x0006000A Indicates that id for the setting of the default enveloped value (0..4095). */
    PH_RDL_FPGA_DEBUG_OUTPUT,                /*!< \brief 0x0006000B Indicates that id for the setting of the Debug Output Mode (see #phFPGAValues). */
    PH_RDL_FPGA_FAST_EXCHANGE_MODE,          /*!< \brief 0x0006000C Indicates that id for the setting of the Fast Exchange Mode (valid values are 0 (is deactivated) and 1 (is activated) ). */
    PH_RDL_FPGA_ANTENNA_TYPE,                /*!< \brief 0x0006000D Indicates that id for the setting of the antenna type (see #phFPGAValues) */

    //vhbr psk
    PH_RDL_FPGA_SOC_MODE,                    /*!< \brief 0x0006000E Indicates that id for the setting of the antenna type (see #phFPGAValues) */
    PH_RDL_FPGA_EOC_MODE,                    /*!< \brief 0x0006000F Indicates that id for the setting of the antenna type (see #phFPGAValues) */
    PH_RDL_FPGA_CHANNEL_MODE,                /*!< \brief 0x00060010 Indicates that id for the setting of the antenna type (see #phFPGAValues) */
    PH_RDL_FPGA_BITS_PER_SYMBOL,             /*!< \brief 0x00060011 Indicates that id for the setting of the antenna type (see #phFPGAValues) */
    PH_RDL_FPGA_ETU,                         /*!< \brief 0x00060012 Indicates that id for the setting of the antenna type (see #phFPGAValues) */
    PH_RDL_FPGA_SOC_SEQ,                     /*!< \brief 0x00060013 Indicates that id for the setting of the antenna type (see #phFPGAValues) */

    //vhbr ask
    PH_RDL_FPGA_BIT_RATES,                   /*!< \brief 0x00060014 Indicates that id for the setting of the antenna type (see #phFPGAValues) */
    PH_RDL_FPGA_START_BIT_MODE,              /*!< \brief 0x00060015 Indicates that id for the setting of the antenna type (see #phFPGAValues) */
    PH_RDL_FPGA_STOP_BIT_MODE,               /*!< \brief 0x00060016 Indicates that id for the setting of the antenna type (see #phFPGAValues) */
    PH_RDL_FPGA_EGT,                         /*!< \brief 0x00060017 Indicates that id for the setting of the antenna type (see #phFPGAValues) */
    PH_RDL_FPGA_SOF_LOW_TIME,                /*!< \brief 0x00060018 Indicates that id for the setting of the antenna type (see #phFPGAValues) */
    PH_RDL_FPGA_SOF_HIGH_TIME,               /*!< \brief 0x00060019 Indicates that id for the setting of the antenna type (see #phFPGAValues) */
    PH_RDL_FPGA_EOF_LOW_TIME,                /*!< \brief 0x0006001A Indicates that id for the setting of the antenna type (see #phFPGAValues) */
    PH_RDL_FPGA_EOF_HIGH_TIME,               /*!< \brief 0x0006001B Indicates that id for the setting of the antenna type (see #phFPGAValues) */
    PH_RDL_FPGA_BIT_BOUNDARIES_HIGH,         /*!< \brief 0x0006001C Indicates that id for the setting of the antenna type (see #phFPGAValues) */
    PH_RDL_FPGA_BIT_BOUNDARIES_LOW,          /*!< \brief 0x0006001D Indicates that id for the setting of the antenna type (see #phFPGAValues) */
    PH_RDL_FPGA_VHBR_ASK_MODE,               /*!< \brief 0x0006001E Indicates that id for the setting of the antenna type (see #phFPGAValues) */
    PH_RDL_FPGA_VHBR_PSK_MODE,               /*!< \brief 0x0006001F Indicates that id for the setting of the antenna type (see #phFPGAValues) */

    PH_RDL_FPGA_PARAMETERS_END,              /*!< \brief 0x00060020 Indicates that the id is not a id of #phFPGAParams */
    PH_RDL_FPGA_PARAMETERS_LENGTH             = (PH_RDL_FPGA_PARAMETERS_END - PH_RDL_FPGA_PARAMETERS_BEGIN)
};

/* FPGA String Params */
enum phFPGAStringParams
{
    PH_RDL_FPGA_STRING_PARAMETERS_BEGIN      = PH_RDL_FPGA_PARAMETERS_END,
    PH_RDL_FPGA_STRING_SEQUENCE_FILE         = PH_RDL_FPGA_STRING_PARAMETERS_BEGIN,
    PH_RDL_FPGA_STRING_LOG_FILE_NAME,        /*!< \brief 0x0006000D Indicates that id for the setting of the Debug Log File Name (Note: max file name len is 256)*/
    PH_RDL_FPGA_STRING_PARAMETERS_END,
    PH_RDL_FPGA_STRING_PARAMETERS_LENGTH     = (PH_RDL_FPGA_STRING_PARAMETERS_END - PH_RDL_FPGA_STRING_PARAMETERS_BEGIN)
};



/**
 * CONSTANTS
 */

/**
* The Definition of the PARAMETER SETTING MODEs
*/
enum phParamSetMode
{
    PH_RDL_PARAM_SETMODE_NORMAL         = 0x0001, /*!< \brief 0x0001 Set the value directly */
    PH_RDL_PARAM_SETMODE_DEFAULT        = 0x0002, /*!< \brief 0x0002 Set the value as default value */
    PH_RDL_PARAM_SETMODE_ALL            = 0x0003  /*!< \brief 0x0003 Set the value directly and as default value */
};

/**
* The Definition of the READER TYPES
*/
enum phReaderTypes
{
    PH_RDL_READER_NONE                  = 0x0000,   /*!< \brief 0x0000 Indicates that no reader is selected. */
    PH_RDL_READER_RD70X,                            /*!< \brief 0x0001 Indicates that the Pegoda reader is selected. */
    PH_RDL_READER_RC523_PN512,                      /*!< \brief 0x0002 Indicates that the Joiner reader is selected. */
    PH_RDL_READER_ISO_LAB,                          /*!< \brief 0x0003 Indicates that the ISOLab reader is selected. */
    PH_RDL_READER_PCSC,                             /*!< \brief 0x0004 Indicates that a PCSC reader is selected. */
    PH_RDL_READER_ULTRASMART_TCL,                   /*!< \brief 0x0005 Indicates that the UltraSmart reader is selected. */
    PH_RDL_READER_DF8_KEIL_SIM,                     /*!< \brief 0x0006 Indicates that the Keil Desfire Simulator is selected. */
    PH_RDL_READER_MICROPROSS,                       /*!< \brief 0x0007 Indicates that the Micropross reader is selected. */
    PH_RDL_READER_SMARTMX_KEIL_SIM,                 /*!< \brief 0x0008 Indicates that the SmartMX Keil simulator is selected. */
    PH_RDL_READER_NCSIM_SIM,                        /*!< \brief 0x0009 Indicates that the NCSIM simulator is selected. */
    PH_RDL_READER_LAST                              /*!< \brief Indicates that this is not an Reader. */
};

/**
* The Definition of the Time Units for the Paramter
*/
enum phTimeUnit
{
    PH_RDL_TIME_UNIT_MILLI_SECONDS      = 0x0000,   /*!< 0x0000 Timings are in milli seconds */
    PH_RDL_TIME_UNIT_MICRO_SECONDS,                 /*!< 0x0001 Timings are in micro seconds */
    PH_RDL_TIME_UNIT_ETU,                           /*!< 0x0002 Timings are in etu: 1 etu is 128/fc = 128/13560000 = 9,44e-6 */
    PH_RDL_TIME_UNIT_FC                             /*!< 0x0003 Timings are in fc: 1 1/fc = 1/13560000 = 73,7e-9 */
};

/**
* The Definition of the READER COMMUNICATION SPEEDS
*/
enum phCommSpeed
{
    PH_RDL_SPEED_106K_ISO14443          = 0x0000,   /*!< 0x0000 Communication Speed 106 */
    PH_RDL_SPEED_212K_ISO14443,                     /*!< 0x0001 Communication Speed 212 */
    PH_RDL_SPEED_424K_ISO14443,                     /*!< 0x0002 Communication Speed 424 */
    PH_RDL_SPEED_848K_ISO14443,                     /*!< 0x0003 Communication Speed 848 */
    PH_RDL_SPEED_ASYNC_NOT_ALLOWED,                 /*!< 0x0004 Asyncron Communication Speed not allowed */
    PH_RDL_SPEED_ASYNC_ALLOWED                      /*!< 0x0005 Asyncron Communication Speed allowed */
};

/**
* The Definition of the READER_TIMEOUT_BEHAVIOR
*/
enum phCommTimeOut
{
    PH_RDL_TIMEOUT_BEHAVIOR_GEN_ERROR   = 0x0001, /*!< 0x0001 On Timeout Generate an Error */
    PH_RDL_TIMEOUT_BEHAVIOR_RESET       = PH_RDL_TIMEOUT_BEHAVIOR_GEN_ERROR+1  /*!< 0x0002 On Timeout Perform an HF Reset */
};

/**
* The Definition of the Values for the Paramter of the HAL Config
*/
enum phHALConfig
{
    /* FEATURE ACTIVATION */
    PH_RDL_DEACTIVATED                  = 0x0000,   /*!< 0x0000 Deactivate Feature */
    PH_RDL_ACTIVATED                    = 0x0001,   /*!< 0x0001 Activate Feature */

    /* CRC MODES */
    PH_RDL_CRC_OFF                      = 0x0000,   /*!< 0x0000 CRC Mode Off */
    PH_RDL_CRC_UPLINK                   = 0x0001,   /*!< 0x0001 CRC Mode Uplink */
    PH_RDL_CRC_DOWNLINK                 = 0x0002,   /*!< 0x0002 CRC Mode Downlink */
    PH_RDL_CRC_BOTH                     = PH_RDL_CRC_UPLINK | PH_RDL_CRC_DOWNLINK, /*!< 0x0003 = 0x0001|0x0002 CRC Mode Both */
    PH_RDL_CRC_MIFARE                   = 0x0004 | PH_RDL_CRC_UPLINK, /*!< 0x0005 = 0x0004|0x0001 CRC Mode Mifare (CRC on Uplink and CRC on downlink if not ACK or NAK) */

    /* PARITY */
    PH_RDL_PARITY_OFF                   = 0x0000,   /*!< 0x0000 Deactivate Parity Mode */
    PH_RDL_PARITY_ON                    = 0x0001,   /*!< 0x0001 Activate Parity Mode */
    PH_RDL_PARITY_EVEN                  = 0x0000,   /*!< 0x0000 Parity Mode Even */
    PH_RDL_PARITY_ODD                   = 0x0001,   /*!< 0x0001 Parity Mode Odd */

    /* TIMEMEASUREMENT */
    PH_RDL_TX_SOC_START                 = 0x0001,    /*!< 0x0001 Time Measurment Point@: TX: Start of SOC (Start of Communication) */
    PH_RDL_TX_SOC_END                   = 0x0002,    /*!< 0x0002 Time Measurment Point@: TX: End of SOC */
    PH_RDL_TX_EOC_START                 = 0x0003,    /*!< 0x0003 Time Measurment Point@: TX: Start of EOC (End of Communication) */
    PH_RDL_TX_EOC_END                   = 0x0004,    /*!< 0x0004 Time Measurment Point@: TX: End of EOC */
    PH_RDL_RX_SOC_START                 = 0x0005,    /*!< 0x0005 Time Measurment Point@: RX: Start of SOC */
    PH_RDL_RX_EOC_END                   = 0x0006,    /*!< 0x0006 Time Measurment Point@: RX: End of End EOC */

    /* LOGGING PARAMETERS */
    PH_RDL_LOG_DEBUG                    = 0x0000,   /*!< 0x0000 ID for Debuging Messages */
    PH_RDL_LOG_INFO                     = 0x0001,   /*!< 0x0001 ID for Info Messages */
    PH_RDL_LOG_WARN                     = 0x0002,   /*!< 0x0002 ID for Warn Messages */
    PH_RDL_LOG_ERR                      = 0x0003,   /*!< 0x0003 ID for Error Messages */
    PH_RDL_LOG_FATAL                    = 0x0004,   /*!< 0x0004 ID for Fatal Error Messages */
};

/**
* The Definition of the READER_CARD_TYPES
*/
enum phCardTypes
{
    PH_RDL_READER_CARD_TYPE_ISO14443A       = 0x0000, /*!< \brief 0x0000 ID for ISO14443A */
    PH_RDL_READER_CARD_TYPE_ISO14443B,      /*!< \brief 0x0001 ID for ISO14443B */
    PH_RDL_READER_CARD_TYPE_ISO15693,       /*!< \brief 0x0002 ID for ISO15693 */
    PH_RDL_READER_CARD_TYPE_REINIT,         /*!< \brief 0x0003 ID for REINIT */
    PH_RDL_READER_CARD_TYPE_ISO7816_T0,     /*!< \brief 0x0004 ID for ISO7816 T=0 */
    PH_RDL_READER_CARD_TYPE_ISO7816_T1,     /*!< \brief 0x0005 ID for ISO7816 T=1 */
    PH_RDL_READER_CARD_TYPE_ISO7816_RAW,    /*!< \brief 0x0006 ID for ISO7816 RAW */
    PH_RDL_READER_CARD_TYPE_FELICA,         /*!< \brief 0x0007 ID for Felica */
    PH_RDL_READER_CARD_TYPE_LAST            /*!< \brief 0x0008 Not an Card Type */
};

/**
* The Definition of the Mifare KeyTypes
*/
enum phMfKeyTypes
{
    MF_KEY_A = 0x60,    /*!< 0x60 ID for Mifare KeyA */
    MF_KEY_B = 0x61     /*!< 0x61 ID for Mifare KeyB */
};

enum phInitOptions
{
    PH_RDL_INIT_KEEP = 0x00
};

/**
* The Definition of the Error code
*/
enum phErrors
{
    /* LAYER ERRORS */
    PH_RDL_ERR_OK                       = ( 0x0000),  /*!< \brief 0x0000 Indicates a successful termination. */
    PH_RDL_ERR_IO_TIMEOUT               = (-0x0001),  /*!< \brief -0x0001 No reply received, e.g. PICC removal. */
    PH_RDL_ERR_CRC_ERROR                = (-0x0002),  /*!< \brief -0x0002 Wrong CRC detected by RC or library. */
    PH_RDL_ERR_PARITY_ERROR             = (-0x0003),  /*!< \brief -0x0003 Parity error detected by RC or library. */
    PH_RDL_ERR_BITCOUNT_ERROR           = (-0x0004),  /*!< \brief -0x0004 Typically, the RC reports such an error. */
    PH_RDL_ERR_FRAMING_ERROR            = (-0x0005),  /*!< \brief -0x0005 Invalid frame format. */
    PH_RDL_ERR_COLLISION_ERROR          = (-0x0006),  /*!< \brief -0x0006 Typically, the RC repors such an error. */
    PH_RDL_ERR_NOT_MATCH_ERROR          = (-0x0007),  /*!< \brief -0x0007 There was an dismatch between the different channels of the Joiners. */
    PH_RDL_ERR_PROTOCOL_ERROR           = (-0x0008),  /*!< \brief -0x0008 Protocol error detected. */

    /* HAL ERRORS */
    PH_RDL_ERR_BUF_TOO_SMALL            = (-0x0101),  /*!< \brief -0x0101 Communication buffer size insufficient. */
    PH_RDL_ERR_INVALID_PARAMETER        = (-0x0102),  /*!< \brief -0x0102 Indicates that an invalid parameter was supplied. */
    PH_RDL_ERR_READER_OPENING_FAILED    = (-0x0103),  /*!< \brief -0x0103 The library failed to open the reader connection. */
    PH_RDL_ERR_PARAM_HAL_SETTING        = (-0x0104),  /*!< \brief -0x0104 An error in a HAL setting (likely an incompatibility) was detected. */
    PH_RDL_ERR_PARAM_NOT_SUPPORTED      = (-0x0105),  /*!< \brief -0x0105 An error indicating that a parameter was is not supported in a GetParam or SetParam function. */

    /* OTHER ERRORS */
    PH_RDL_ERR_NOT_IMPLEMENTED          = (-0x01FE),  /*!< \brief -0x01FE The called function does not implement any functionalty */
    PH_RDL_ERR_INTERNAL_ERROR           = (-0x01FF),  /*!< \brief -0x01FF An internal error occured (this is bad, contact the architect) */
};



/**
* The Definition of the FPGA values
*/
enum phFPGAValues
{
    /* ACTIVATION */
    PH_FPGA_DEACTIVATE                        = ( 0x0000),    /*!< \brief 0x0000 Indicates that the FPGA is deactivated. */
    PH_FPGA_ACTIVATE                          = ( 0x0001),    /*!< \brief 0x0001 Indicates that the FPGA is activated. */

    /* SEQUENCES */
    PH_FPGA_SINUS                             = ( 0x0000),    /*!< \brief 0x0000 Index for a simple Sinus. */
    PH_FPGA_SEQUENCE_1                        = ( 0x0001),    /*!< \brief 0x0001 Index for the data sequence 1. */
    PH_FPGA_SEQUENCE_2                        = ( 0x0002),    /*!< \brief 0x0002 Index for the data sequence 2 */
    PH_FPGA_SEQUENCE_3                        = ( 0x0003),    /*!< \brief 0x0003 Index for the data sequence 3. */
    PH_FPGA_SEQUENCE_4                        = ( 0x0004),    /*!< \brief 0x0004 Index for the data sequence 4. */
    PH_FPGA_SEQUENCE_5                        = ( 0x0005),    /*!< \brief 0x0005 Index for the data sequence 5. */
    PH_FPGA_SEQUENCE_6                        = ( 0x0006),    /*!< \brief 0x0006 Index for the data sequence 6. */
    PH_FPGA_SEQUENCE_7                        = ( 0x0007),    /*!< \brief 0x0007 Index for the data sequence 7. */
    PH_FPGA_SEQUENCE_8                        = ( 0x0008),    /*!< \brief 0x0008 Index for the data sequence 8. */
    PH_FPGA_TYPE_B                            = ( 0x0009),

    /* DEFAULT SEQUENCE RELOAD MODI */
    PH_FPGA_DEFAULT_RELOAD_DEACTIVATED        = ( 0x0000),    /*!< \brief 0x0000 No Reload of the default patterns is done. */
    PH_FPGA_DEFAULT_RELOAD_ON_INIT            = ( 0x0001),    /*!< \brief 0x0001 On init/reset a reload of the default patterns is done. */
    PH_FPGA_DEFAULT_RELOAD_ON_SELECTION       = ( 0x0002),    /*!< \brief 0x0002 If a pattern is selected and not the default pattern is loaded a reload is done. */
    PH_FPGA_DEFAULT_RELOAD_IMMEDIATLY         = ( 0x0003),    /*!< \brief 0x0003 A reload is done immedialy. */

    /* TIMEOUT BEHAVIOUR */
    PH_FPGA_HF_PAUSE                          = ( 0x0000),
    PH_FPGA_LABEL_RESET                       = ( 0x0001),
    /* AUTO ADJUST */
    PH_FPGA_AUTO_ADJUST_DISABLED              = ( 0x0000),    /*!< \brief 0x0000 The Sequence is selected according the datarate automatically (DS1=106 -- DS4=848). */
    PH_FPGA_AUTO_ADJUST_ENABLED               = ( 0x0001),    /*!< \brief 0x0001 The Sequence is NOT selected according the datarate */

    PH_FPGA_FILTER_MODE_DEFAULT               = ( 0x0000),    /*!< \brief 0 Default Filter Mode use the filter. */
    PH_FPGA_FILTER_MODE_CARRIER               = ( 0x0001),    /*!< \brief 1 Play only the carrier and do not use the filter. */
    PH_FPGA_FILTER_MODE_ENVELOPED             = ( 0x0002),    /*!< \brief 2 Play only the enveloped and do not use the filter. */
    PH_FPGA_FILTER_MODE_MODULATED             = ( 0x0003),    /*!< \brief 3 Play the modulated carrier (carrier*enveloped) and do not use the filter. */

    PH_FPGA_OUTPUT_CONSOLE_DEBUG              = ( 0x0001),    /*!< \brief 0x0001 Enable Output of Debug Messages to the console or callback. (See #PH_FPGA_OUTPUT_CALLBACK for settings) */
    PH_FPGA_OUTPUT_CONSOLE_INFO               = ( 0x0002),    /*!< \brief 0x0002 Enable Output of Info Messages to the console or callback. (See #PH_FPGA_OUTPUT_CALLBACK for settings)*/
    PH_FPGA_OUTPUT_CONSOLE_WARNING            = ( 0x0004),    /*!< \brief 0x0004 Enable Output of Warning Messages to the console or callback. (See #PH_FPGA_OUTPUT_CALLBACK for settings)*/
    PH_FPGA_OUTPUT_CONSOLE_ERROR              = ( 0x0008),    /*!< \brief 0x0008 Enable Output of Error Messages to the console or callback. (See #PH_FPGA_OUTPUT_CALLBACK for settings)*/
    PH_FPGA_OUTPUT_CONSOLE_FATAL_ERROR        = ( 0x0010),    /*!< \brief 0x0010 Enable Output of Fatal Error Messages to the console or callback. (See #PH_FPGA_OUTPUT_CALLBACK for settings)*/
    PH_FPGA_OUTPUT_CONSOLE_FUNCTION_CALL      = ( 0x0020),    /*!< \brief 0x0020 Enable Output of Function Calls to the console or callback. (See #PH_FPGA_OUTPUT_CALLBACK for settings)*/
    PH_FPGA_OUTPUT_CONSOLE_TIMESTAMP          = ( 0x0040),    /*!< \brief 0x0040 Enable Adding of Timestamp to every Message printed to the console or callback. (See #PH_FPGA_OUTPUT_CALLBACK for settings)*/
    PH_FPGA_OUTPUT_FILE_DEBUG                 = ( 0x0100),    /*!< \brief 0x0100 Enable Output of Debug Messages to the file. */
    PH_FPGA_OUTPUT_FILE_INFO                  = ( 0x0200),    /*!< \brief 0x0200 Enable Output of Info Messages to the file. */
    PH_FPGA_OUTPUT_FILE_WARNING               = ( 0x0400),    /*!< \brief 0x0400 Enable Output of Warning Messages to the file. */
    PH_FPGA_OUTPUT_FILE_ERROR                 = ( 0x0800),    /*!< \brief 0x0800 Enable Output of Error Messages to the file. */
    PH_FPGA_OUTPUT_FILE_FATAL_ERROR           = ( 0x1000),    /*!< \brief 0x1000 Enable Output of Fatal Error Messages to the file. */
    PH_FPGA_OUTPUT_FILE_FUNCTION_CALL         = ( 0x2000),    /*!< \brief 0x2000 Enable Output of Function Calls to the file. */
    PH_FPGA_OUTPUT_FILE_TIMESTAMP             = ( 0x4000),    /*!< \brief 0x4000 Enable Adding of Timestamp to every Message printed to the file */
    PH_FPGA_OUTPUT_CALLBACK                   = ( 0x8000)     /*!< \brief 0x8000 Enable Callback to Library. Note: if enabled console output gets disabled. Desired Console Flags still need to be set and messages get redirected to the callback function. */
};



/**
 * MIFARE APPLICATION
 */

/**
* The Definition of the Mifare Error Codes
* Do not change; defined here to avoid dependency on reader lib header files
*/
enum phAppMifareErrors
{
    MF_ERR_BASE_REPONSE_CODE            = 0xF000,
    MF_ERR_NACK0                        = MF_ERR_BASE_REPONSE_CODE + 0x00,  /*!< \brief 0x00 NACK 0 received */
    MF_ERR_NACK1,                                                           /*!< \brief 0x00 NACK 1 received */
    MF_ERR_NACK4                        = MF_ERR_BASE_REPONSE_CODE + 0x04,  /*!< \brief 0x00 NACK 4 received */
    MF_ERR_NACK5                        = MF_ERR_BASE_REPONSE_CODE + 0x05,  /*!< \brief 0x00 NACK 5 received */
    MF_ERR_ACK                          = 0x00,                             /*!< \brief 0x00 Operation OK */
    MF_ERR_NACKF                        = 0x0F                              /*!< \brief 0x00 NACK F received */
};

/**
* The Definition of the Mifare Application Options
*/
enum phAppMifareOptions
{
    MF_OPTION_SELECT_REQUEST            = 0x00,     /*!< \brief 0x00 in Mifare Application use a Request in Activation */
    MF_OPTION_SELECT_WAKEUP             = 0x01      /*!< \brief 0x00 in Mifare Application use a Wakeup in Activation */
};

/*
 * MIFARE PLUS APPLICATION
 */

enum phAppMifarePlusModes
{
    MFP_MODE_USE_CMDLIB                 = 0x00,
    MFP_MODE_USE_RD70X                  = 0x01,
    MFP_MODE_USE_BFL_OLD                = 0x02
};

enum phAppMifarePlusErrors
{
    MFP_ERR_BASE_INTERNAL               = 0x0D00,   /*!< Base address for internal errors, fits with errors defined in phcsBflStatus.h) */
    MFP_ERR_BASE_REPONSE_CODE           = 0xF000,   /*!< response from PICC has unexpected value */
    MFP_ERR_BASE_LENGTH                 = 0xF100,   /*!< response from PICC has unexpected length */
    MFP_ERR_OK                          = 0x00,     /*!< Do not change; defined here to avoid dependency on reader lib header files */
    MFP_ERR_CRYPTOLIB_FAILURE           = 0x01,     /*!< CryptoLib failure on opening or processing */
    MFP_ERR_AUTH_FAILURE                = 0x02,     /*!< Authentication with PICC failed */
    MFP_ERR_PARAMETER_NOT_SUPPORTED     = 0x03,     /*!< Parameter value is not supported */
    MFP_ERR_COMMAND_NOT_SUPPORTED       = 0x04,     /*!< Command is not supported */
    MFP_ERR_OTHER                       = 0x05,     /*!< unspecified internal failure */
    MFP_PICC_ERR_AUTH                   = 0x06,     /*!< Card Error: Authentication */
    MFP_PICC_ERR_CMD_OVERFLOW           = 0x07,     /*!< Card Error: Overflow */
    MFP_PICC_ERR_MAC_PCD                = 0x08,     /*!< Card Error: MAC */
    MFP_PICC_ERR_BNR                    = 0x09,     /*!< Card Error: Block Number */
    MFP_PICC_ERR_EXT                    = 0x0A,     /*!< Card Error: EXT */
    MFP_PICC_ERR_CMD_INVALID            = 0x0B,     /*!< Card Error: Invalid Cmd */
    MFP_PICC_ERR_FORMAT                 = 0x0C,     /*!< Card Error: Format */
    MFP_PICC_ERR_GEN_FAILURE            = 0x0F      /*!< Card Error: Generic Failure */
};

enum phAppMifarePlusOptions
{
    MFP_OPTION_SELECT_ISO14443_3        = 0x01,
    MFP_OPTION_SELECT_ISO14443_4        = 0x02,
    MFP_OPTION_SELECT_WAKEUP            = 0x10
};

/**
 * DEFINITIONS
 */

/* Trigger Source */
#define TRIGGER_INTERN                  0x0001      /*!< 1 Intern FPGA trigger. */
#define TRIGGER_EXTERN                  0x0002      /*!< 1 Extern FPGA trigger. */

/* Clock Source */
#define CLOCK_DEFAULT                   0x0000      /*!< 0x00 default clock. */
#define CLOCK_EXTERN_2                  0x0002      /*!< 0x01 extern clock*2. */
#define CLOCK_EXTERN                    0x0001      /*!< 0x02 extern clock. */

/* Resonance Frequency Repeat Rate */
#define REPEAT_RATE_0_5                 0x0001    	/*!< 0x01 0.5Hz / 2000ms */
#define REPEAT_RATE_1                   0x0002	    /*!< 0x02 1Hz / 1000ms */
#define REPEAT_RATE_2                   0x0003	    /*!< 0x03 2Hz / 500ms */
#define REPEAT_RATE_4                   0x0004	    /*!< 0x04 4Hz / 250ms */
#define REPEAT_RATE_8                   0x0005	    /*!< 0x05 8Hz / 125ms */
#define REPEAT_RATE_16_13               0x0006	    /*!< 0x06 16.13Hz / 62ms */
#define REPEAT_RATE_32_25               0x0007	    /*!< 0x07 32.25Hz / 31ms */
#define REPEAT_RATE_62_5                0x0008	    /*!< 0x08 62.5Hz / 16ms */
#define REPEAT_RATE_125                 0x0009	    /*!< 0x09 125Hz / 8ms */
#define REPEAT_RATE_250                 0x000A	    /*!< 0x0A 250Hz / 4ms */

/**
 * The Definition of the selected Joiner
 */

#define TARGET_JOINER_NONE              0x0000      /*!< 0x0000 do not send a command */
#define TARGET_JOINER_JOINER_I          0x8000      /*!< 0x8000 send command to Joiner I */
#define TARGET_JOINER_JOINER_Q          0x8001      /*!< 0x8001 send command to Joiner Q*/
#define TARGET_JOINER_BOTH              0x8002      /*!< 0x8002 send command to both Joiner */


#endif /* PH_DEFINES_H */
