/*
 * Copyright 2013, 2015, 2017, 2019, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/*
 * RdCardSim specific HAL-Component of Reader Library Framework.
 * $RCSfile $
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phbalReg.h>
#include <phhalHw.h>
#include <ph_RefDefs.h>
#include <ctype.h>

#ifdef NXPBUILD__PHHAL_HW_RDCARDSIM

#include "phhalHw_RdCardSim.h"
#include "phhalHw_RdCardSim_Int.h"
#include <phhalHw_RdCardSim_Cmd.h>

static const uint8_t * PH_MEMLOC_CONST_ROM gpkphhalHw_FrameHeaderStr[END_HEADER_LIST] =
{
    (uint8_t*)"IVD",      /**< Invalid header */
    (uint8_t*)"ENV",      /**< NCSIM testbench environment command header string of the PIPE frame */
    (uint8_t*)"PIC",      /**< Reader model and PICC exchange header string of the PIPE frame */
    (uint8_t*)"TUS"       /**< Time unit measurement header string of the PIPE frame */,
    (uint8_t*)"PMW",      /**< Write operation to the testbench registers header string of the PIPE frame */
    (uint8_t*)"PMR",      /**< Read operation to the testbench registers header string of the PIPE frame */
    (uint8_t*)"MEM",      /**< Memory operations to the simulator header string of the PIPE frame */
	(uint8_t*)"WAF",      /**< Wafer Test Communication Mode */
  	(uint8_t*)"PCC"      /**< UART Communication Mode */
};

static const uint8_t * PH_MEMLOC_CONST_ROM gpkphhalHw_FrameStatusCodesStr[END_STATUS_LIST] =
{
    (uint8_t*)"IVD",      /**< invalid cknowledge code string of the PIPE frame */
    (uint8_t*)"ACK",      /**< Acknowledge code string of the PIPE frame */
    (uint8_t*)"NAK",      /**< Command not successful */
    (uint8_t*)"ERP",      /**< PIPE Protocol error code string of the PIPE frame */
    (uint8_t*)"ERS",      /**< Error in the transmited string in the frame (unrecognised header) */
    (uint8_t*)"ETO",      /**< Error, caused by Timeout */
    (uint8_t*)"ECR",      /**< Error, caused by Invalid CRC */
    (uint8_t*)"EPR",      /**< Error, caused by Invalid Parity */
    (uint8_t*)"EFR",      /**< Error, caused by Miller Error (Framing Error) */
};

static const uint8_t * PH_MEMLOC_CONST_ROM gpkphhalHw_TBENVBaudRateStr[] =
{
    (uint8_t*)"106",    /**< String to the testbench to indicate 106 baud rate */
    (uint8_t*)"212",    /**< String to the testbench to indicate 212 baud rate */
    (uint8_t*)"424",    /**< String to the testbench to indicate 424 baud rate */
    (uint8_t*)"848",    /**< String to the testbench to indicate 848 baud rate */
    (uint8_t*)"1696",   /**< String to the testbench to indicate 1696 baud rate */
    (uint8_t*)"3392",   /**< String to the testbench to indicate 3392 baud rate */
};

static const uint8_t * PH_MEMLOC_CONST_ROM gpkphhalHw_TBENVSchemeStr[] =
{
    (uint8_t*)"MANCHESTER",    /**< String to the testbench to indicate Manchester Encoding */
    (uint8_t*)"BPSK",          /**< String to the testbench to indicate BPSK Encoding */
 };

static const uint8_t * PH_MEMLOC_CONST_ROM gpkphhalHw_TBENVParity[] =
{
    (uint8_t*)"CL_OFF",
    (uint8_t*)"CL_ODD",
    (uint8_t*)"CL_EVEN",
};

const uint8_t * phhalHw_RdCardSim_BaudRateStr(
    uint8_t bBaud
    )
{

    /* Security check */
    if (bBaud >= (sizeof(gpkphhalHw_TBENVBaudRateStr) / sizeof(gpkphhalHw_TBENVBaudRateStr[0])))
    {
        bBaud = sizeof(gpkphhalHw_TBENVBaudRateStr) / sizeof(gpkphhalHw_TBENVBaudRateStr[0]) - 1;
    }
    return gpkphhalHw_TBENVBaudRateStr[ bBaud ];
}

const uint8_t * phhalHw_RdCardSim_SchemeStr(
    uint8_t bBaud
    )
{
    if (bBaud == 0 )
    {
	    return gpkphhalHw_TBENVSchemeStr[ 0 ];
    }
	else
	{
	    /* All others use the same encoding. */
		return gpkphhalHw_TBENVSchemeStr[ 1 ];
	}
}

const uint8_t * phhalHw_RdCardSim_ParityStr(
    uint8_t bParity
    )
{
    /* Security check */
    if (bParity > 2)
    {
        bParity = 2;
    }
    return gpkphhalHw_TBENVParity[ bParity ];
}

uint16_t phhalHw_RdCardSim_StrFind (
                                    char  pStr[],
                                    const char  pKey[]
)
{
    uint16_t    PH_MEMLOC_REM wFound;
    uint16_t    PH_MEMLOC_REM wIdx;
    uint16_t    PH_MEMLOC_REM wStrLength;
    uint16_t    PH_MEMLOC_REM wKeyLength;
    uint8_t     PH_MEMLOC_REM *pTmpStr;

    wStrLength = (uint16_t)strlen(pStr);
    wKeyLength = (uint16_t)strlen(pKey);

    wFound = 1;
    for (wIdx = 0; ((wIdx < wStrLength) && (wFound != 0)) ; wIdx++)
    {
        pTmpStr  = (uint8_t*)&pStr[wIdx];
        wFound = (uint16_t)strncmp((char*)pTmpStr, pKey, wKeyLength);
    }

    return wFound;
}

uint8_t phhalHw_RdCardSim_NumBufferAlloc (
    phhalHw_RdCardSim_DataParams_t * pDataParams
    )
{
    uint8_t alloccnt;
    uint8_t bufcnt;

    alloccnt = 0;

    for (bufcnt = 0 ; bufcnt < PHHAL_HW_RDCARDSIM_INTBUFFER ; bufcnt++)
    {
        if (pDataParams->wIntBufferAlloc & (uint16_t)((uint8_t)1U << bufcnt))
        {
            alloccnt++;
        }
    }
    return alloccnt;
}

phStatus_t  phhalHw_RdCardSim_Init_Buffer (
    phhalHw_RdCardSim_DataParams_t * pDataParams,
    uint8_t *  pBuffer
    )
{
    phStatus_t status;
    uint8_t bufcnt, bufcntLog;
    uint8_t found;
    uint16_t wTmpBufLength;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phhalHw_RdCardSim_Init_Buffer");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bufcnt);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    found = 0;
    wTmpBufLength = pDataParams->wIntBufferLen / PHHAL_HW_RDCARDSIM_INTBUFFER;
    for (bufcnt = 0 ; (bufcnt < PHHAL_HW_RDCARDSIM_INTBUFFER && found == 0); bufcnt++)
    {
        if (pBuffer == (pDataParams->pIntBuffer + (bufcnt * wTmpBufLength)))
        {
            found = 1;
            pDataParams->wIntBufferAlloc &= (uint16_t)~(uint16_t)((uint8_t)1U << bufcnt); /* PRQA S 290 */
            memset(pDataParams->pIntBuffer + (bufcnt * wTmpBufLength), 0x00, wTmpBufLength); /* PRQA S 3200 */
            PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
            bufcntLog = bufcnt;
            PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_INFO, bufcnt_log, &bufcntLog);
            PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_GEN);
        }
    }
    if (found == 1)
    {
        status = PH_ERR_SUCCESS;
        bufcnt--;
    }
    else
    {
        status = PH_ERR_PARAMETER_OVERFLOW;
        bufcnt = 0xEE;
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_INFO, bufcnt_log, &bufcnt);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return PH_ADD_COMPCODE(status, PH_COMP_HAL);

}

static phStatus_t  phhalHw_RdCardSim_BufferAllocCheckSize(
		phhalHw_RdCardSim_DataParams_t * pDataParams,
		uint16_t wNeededSize)
{
    phStatus_t status;
	uint16_t wAvailableSize = pDataParams->wIntBufferLen / PHHAL_HW_RDCARDSIM_INTBUFFER;

	if (wNeededSize > (pDataParams->wIntBufferLen / PHHAL_HW_RDCARDSIM_INTBUFFER))
	{
	    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phhalHw_RdCardSim_BufferAllocCheckSize");
		PH_LOG_HELPER_ALLOCATE_PARAMNAME(wNeededSize);
		PH_LOG_HELPER_ALLOCATE_PARAMNAME(wAvailableSize);
		PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, bFunctionName);
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_ERROR, wNeededSize_log, &(wNeededSize));
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_ERROR, wAvailableSize_log, &(wAvailableSize));
		PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_GEN);
        status = PH_ERR_PARAMETER_OVERFLOW;
	}
	else
	{
        status = PH_ERR_SUCCESS;
	}
    return PH_ADD_COMPCODE(status, PH_COMP_HAL);
}

phStatus_t  phhalHw_RdCardSim_BufferAlloc (
    phhalHw_RdCardSim_DataParams_t * pDataParams,
    uint8_t   ** ppBuffer
    )
{

    phStatus_t status;
    uint8_t bufcnt, bufcntLog;
    uint8_t found;
    uint8_t alloccnt;
    uint16_t wTmpBufLength;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phhalHw_RdCardSim_BufferAlloc");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bufcnt);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wIntBufferLen);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wIntBufferAlloc);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wTmpBufLength);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(alloccnt);

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, wIntBufferLen_log, &(pDataParams->wIntBufferLen));
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, wIntBufferAlloc_log, &(pDataParams->wIntBufferAlloc));
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);


    /* sequentially go through the buffer list and look for available buffers */
    found = 0;
    wTmpBufLength = pDataParams->wIntBufferLen / PHHAL_HW_RDCARDSIM_INTBUFFER;
    for (bufcnt = 0 ; (bufcnt < PHHAL_HW_RDCARDSIM_INTBUFFER && found == 0); bufcnt++)
    {
        if (!(pDataParams->wIntBufferAlloc & (uint16_t)((uint8_t)1U << bufcnt)))
        {
            found = 1;
            pDataParams->wIntBufferAlloc |= (uint16_t)((uint8_t)1U << bufcnt);
            *ppBuffer = pDataParams->pIntBuffer + (bufcnt * wTmpBufLength);
            memset(pDataParams->pIntBuffer + (bufcnt * wTmpBufLength), 0x00, wTmpBufLength); /* PRQA S 3200 */
            PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
            bufcntLog = bufcnt;
            PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_INFO, bufcnt_log, &bufcntLog);
            PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, wIntBufferAlloc_log, &(pDataParams->wIntBufferAlloc));
            PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, wTmpBufLength_log, &wTmpBufLength);
            PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_GEN);
        }
    }
    if (found == 1)
    {
        status = PH_ERR_SUCCESS;
        bufcnt--;
    }
    else
    {
        status = PH_ERR_PARAMETER_OVERFLOW;
        bufcnt = 0xEE;
    }
    alloccnt = phhalHw_RdCardSim_NumBufferAlloc (pDataParams);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_INFO, alloccnt_log, &alloccnt);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_INFO, bufcnt_log, &bufcnt);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, wIntBufferAlloc_log, &(pDataParams->wIntBufferAlloc));
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return PH_ADD_COMPCODE(status, PH_COMP_HAL);
}


phStatus_t  phhalHw_RdCardSim_BufferDeAlloc (
    phhalHw_RdCardSim_DataParams_t * pDataParams,
    uint8_t   * pBuffer
    )
{

    phStatus_t status;
    uint8_t bufcnt, bufcntLog;
    uint8_t found;
    uint8_t alloccnt;
    uint16_t wTmpBufLength;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phhalHw_RdCardSim_BufferDeAlloc");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bufcnt);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wIntBufferLen);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wIntBufferAlloc);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(alloccnt);

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, wIntBufferLen_log, &(pDataParams->wIntBufferLen));
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, wIntBufferAlloc_log, &(pDataParams->wIntBufferAlloc));
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);


    /* sequentially go through the buffer list and look for available buffers */
    found = 0;
    wTmpBufLength = pDataParams->wIntBufferLen / PHHAL_HW_RDCARDSIM_INTBUFFER;
    for (bufcnt = 0 ; (bufcnt < PHHAL_HW_RDCARDSIM_INTBUFFER && found == 0); bufcnt++)
    {
        if (pBuffer == (pDataParams->pIntBuffer + (bufcnt * wTmpBufLength)))
        {
            found = 1;
            pDataParams->wIntBufferAlloc &= (uint16_t)~(uint16_t)((uint8_t)1U << bufcnt); /* PRQA S 290 */
            memset(pDataParams->pIntBuffer + (bufcnt * wTmpBufLength), 0x00, wTmpBufLength); /* PRQA S 3200 */
            PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
            bufcntLog = bufcnt;
            PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_INFO, bufcnt_log, &bufcntLog);
            PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, wIntBufferAlloc_log, &(pDataParams->wIntBufferAlloc));
            PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_GEN);
        }
    }
    if (found == 1)
    {
        status = PH_ERR_SUCCESS;
        bufcnt--;
    }
    else
    {
        status = PH_ERR_PARAMETER_OVERFLOW;
        bufcnt = 0xEE;
    }

    alloccnt = phhalHw_RdCardSim_NumBufferAlloc (pDataParams);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_INFO, alloccnt_log, &alloccnt);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_INFO, bufcnt_log, &bufcnt);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, wIntBufferAlloc_log, &(pDataParams->wIntBufferAlloc));
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return PH_ADD_COMPCODE(status, PH_COMP_HAL);


}

phStatus_t  phhalHw_RdCardSim_TxFrame (
                                       phhalHw_RdCardSim_DataParams_t * pDataParams,
                                       uint8_t     bHeader,
                                       uint16_t    wNBytes,
                                       uint16_t    wNBits,
                                       uint8_t   * pPayld,
                                       uint8_t   * pTxFrame
                                       )
{
    uint8_t     PH_MEMLOC_REM Xch[3];
    uint16_t    PH_MEMLOC_REM wTxBytes;
    uint16_t    PH_MEMLOC_REM wIdx;
    phStatus_t  PH_MEMLOC_REM status;
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint8_t *   PH_MEMLOC_REM pPayldCh;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phhalHw_RdCardSim_TxFrame");
    /*PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDataParams);*/
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bHeader);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wNBytes);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wNBits);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pPayld);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);

    PH_LOG_HELPER_ALLOCATE_TEXT(bTX_Payld, ", TX_Payld = ");
    PH_LOG_HELPER_ALLOCATE_TEXT(bTX_Frame, "TX_Frame = ");

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_INFO, bHeader_log, &bHeader);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, wNBytes_log, &wNBytes);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, wNBits_log, &wNBits);
    if ((bHeader == CMD_PIC) ||(bHeader == CMD_WAF || (bHeader==CMD_PCC)))
    {
        PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_INFO, pPayld_log, pPayld, (uint16_t)strlen((char*)pPayld));
    }
    else
    {
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bTX_Payld);
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, (char*)pPayld);
    }
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);


    switch (bHeader)
    {
    case CMD_ENV:
        if (pDataParams->bC40ModeEnabled == PH_OFF)
        {
            sprintf((char*)pTxFrame, "%s%s", (const char*)gpkphhalHw_FrameHeaderStr[CMD_ENV], (char*)pPayld); /* PRQA S 3200 */
        }
        else
        {
            if (wNBytes == 0)
            {
                /* here the caller must ensure the string is 0 termimated (as is done internally RdCardSim_Cmd) */
            sprintf((char*)pTxFrame, "%s%s", (const char*)gpkphhalHw_FrameHeaderStr[CMD_ENV], (char*)pPayld); /* PRQA S 3200 */
            }
            else
            {
                /* we got a length of the input data, just copy that portion only, and don't rely on a 0 terminator */
                sprintf((char*)pTxFrame, "%s", (const char*)gpkphhalHw_FrameHeaderStr[CMD_ENV]);
                /* use strncat to append just the input bytes.  it will also add a 0 terminator to pTxFrame */
                strncat((char*)pTxFrame, (char*)pPayld, wNBytes);
            }
        }
        break;
    case CMD_PIC:
	case CMD_WAF:
    case CMD_PCC:
        if (pDataParams->bC40ModeEnabled == PH_ON)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferAllocCheckSize(pDataParams, 2*wNBytes+9));
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferAlloc(pDataParams, &pPayldCh));
        /* convert the uint pPayld to chars and adjust wNBytes*/
		wTxBytes = wNBytes;
        if (wNBits != 0)
        {
            wTxBytes--;
        }
        for (wIdx = 0; wIdx < wNBytes; wIdx++)
        {
            memset(Xch, 0x00, 3);  /* PRQA S 3200 */
            sprintf((char*)Xch, "%02X\0", pPayld[wIdx]); /* PRQA S 3200 */
            /*              strcpy((char*)&PayldCh[bIdx*2], (char*)Xch); */ /* PRQA S 3200 */
            strcpy((char*)(pPayldCh+(wIdx*2)), (char*)Xch); /* PRQA S 3200 */
        }

        sprintf((char*)pTxFrame, "%s%04X%02X%s", gpkphhalHw_FrameHeaderStr[bHeader], wTxBytes, wNBits, (char*)pPayldCh); /* PRQA S 3200 */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferDeAlloc(pDataParams, pPayldCh));
        break;
    case CMD_TUS:
        sprintf((char*)pTxFrame, "%s%s", gpkphhalHw_FrameHeaderStr[CMD_TUS], gpkphhalHw_FrameHeaderStr[CMD_TUS]); /* PRQA S 3200 */
        break;
    case CMD_PMW:
        sprintf((char*)pTxFrame, "%s%s", gpkphhalHw_FrameHeaderStr[CMD_PMW], (char*)pPayld); /* PRQA S 3200 */
        break;
    case CMD_PMR:
        sprintf((char*)pTxFrame, "%s%s", gpkphhalHw_FrameHeaderStr[CMD_PMR], (char*)pPayld); /* PRQA S 3200 */
        break;
    case CMD_MEM:
        sprintf((char*)pTxFrame, "%s%s", gpkphhalHw_FrameHeaderStr[CMD_MEM], (char*)pPayld); /* PRQA S 3200 */
        break;
    default:
        printf( "phhalHw_RdCardSim_TxFrame: ERROR - Unknown frame header value of (0x%X)\n", bHeader); /* PRQA S 3200 */
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    if (strlen((char*)pTxFrame) > 0)
    {
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bTX_Frame);
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, (char*)pTxFrame);
        pTxFrame[strlen((char*)pTxFrame)]=0x00;
        status = PH_ERR_SUCCESS;
    }
    else
    {
        status = PH_ERR_INTERFACE_ERROR;
    }

    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return PH_ADD_COMPCODE(status, PH_COMP_HAL);
}


/***********************************************************************************************
*        RxFrame
************************************************************************************************/
phStatus_t phhalHw_RdCardSim_RxFrame (
                                      uint8_t * pRxFrame,
                                      uint8_t bHeader,
                                      uint16_t * pNBytes,
                                      uint8_t * pNBits,
                                      uint8_t * pPayld,
                                      uint8_t bC40ModeEnabled
                                      )
{
    uint8_t     PH_MEMLOC_REM aTmpStr[FRAME_FIELD_MAX_NCHAR];
    uint8_t *   PH_MEMLOC_REM pStrBuffer;
    uint16_t    PH_MEMLOC_REM wHeaderStart;
    uint16_t    PH_MEMLOC_REM wStrBufferIdx;
    uint16_t    PH_MEMLOC_REM wRxtatus;
    uint16_t    PH_MEMLOC_COUNT wIdx;
    uint16_t    PH_MEMLOC_REM wCompCode = PH_COMP_HAL | PHHAL_HW_RDCARDSIM_ID;
    void *      PH_MEMLOC_REM pDataParams = &wCompCode;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName,        "phhalHw_RdCardSim_RxFrame");
    PH_LOG_HELPER_ALLOCATE_TEXT(bErrorNoHeader,       "ERROR - did not find header from ");
    PH_LOG_HELPER_ALLOCATE_TEXT(bErrorInvalidHeader,  "ERROR - invalid header.");
    PH_LOG_HELPER_ALLOCATE_TEXT(bErrorNoStatus,       "ERROR - did not find status in ");
    PH_LOG_HELPER_ALLOCATE_TEXT(bErrorByte,           "ERROR - invalid byte frame section in ");
    PH_LOG_HELPER_ALLOCATE_TEXT(bErrorBit,            "ERROR - invalid bit frame section in ");
    PH_LOG_HELPER_ALLOCATE_TEXT(bErrorPayLoad,        "ERROR - invalid pay load section in ");
    PH_LOG_HELPER_ALLOCATE_TEXT(bErrorTime,           "ERROR - invalid time section in ");
    PH_LOG_HELPER_ALLOCATE_TEXT(bErrorRead,           "ERROR - invalid read section in ");

    PH_LOG_HELPER_ALLOCATE_TEXT(bInfoHeader,          "Found header = ");
    PH_LOG_HELPER_ALLOCATE_TEXT(bInfoStatus,          "Found status = ");
    PH_LOG_HELPER_ALLOCATE_TEXT(bInfoTimeValue,       "Time = ");
    PH_LOG_HELPER_ALLOCATE_TEXT(bInfoReadValue,       "ReadData = ");

    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pRxFrame);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bHeader);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pNBytes);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pNBits);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pPayld);
    /*PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);  MK: you do not use this variable at this time???*/

    /* Parameter check */
    if (bHeader >= END_HEADER_LIST)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }

    *pNBytes      = 0;
    *pNBits       = 0;
    wRxtatus      = RXSTAT_ERROR;
    pStrBuffer    = pRxFrame;
    wStrBufferIdx = 0;

    memset(aTmpStr, 0x00, FRAME_FIELD_MAX_NCHAR);  /* PRQA S 3200 */

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_INFO, bHeader_log, &bHeader);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, " = ");
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, (const char*)gpkphhalHw_FrameHeaderStr[bHeader]);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, ", pRxFrame = ");
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, (char*)pRxFrame);
    PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_INFO, pRxFrame_log, pRxFrame, (uint16_t)strlen((char*)pRxFrame));
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* Decode the header */
    wHeaderStart = phhalHw_RdCardSim_StrFind((char*)pRxFrame, (const char*)gpkphhalHw_FrameHeaderStr[bHeader]);
    if (wHeaderStart <= strlen((char*)pRxFrame))
    {
        /*  If there are any strip the extra characters before the header. */
        if (wHeaderStart > 0)
        {
            wStrBufferIdx = wHeaderStart;
            pStrBuffer = &pRxFrame[wStrBufferIdx];
        }
        /* Check for a valid header and remove it */
        if (strncmp((char*)pStrBuffer, (const char*)gpkphhalHw_FrameHeaderStr[bHeader], FRAME_HEADER_NCHAR) == 0)
        {
            wStrBufferIdx = wStrBufferIdx + FRAME_HEADER_NCHAR;
            /* Security check */
            if (wStrBufferIdx >= strlen((char*)pRxFrame))
            {
                return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
            }
            pStrBuffer = &pRxFrame[wStrBufferIdx];
            PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
            PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bInfoHeader);
            PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, (const char*)gpkphhalHw_FrameHeaderStr[bHeader]);
            PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_GEN);

        }
        else
        {
            wRxtatus = RXSTAT_INVALID_HDR;
            PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, bFunctionName);
            PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, bErrorInvalidHeader);
            PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_GEN);
        }
    }
    else
    {
        wRxtatus = RXSTAT_NO_HDR;
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, bFunctionName);
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, bErrorNoHeader);
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, (char*)pRxFrame);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_GEN);
    }

    /* When there is no header errors decode the status codes */
    if (wRxtatus == RXSTAT_ERROR)
    {
        /* Security check */
        if (strlen((char*)pStrBuffer) < FRAME_STATUS_NCHAR)
        {
            return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
        }
        /* Check if there is a match with the valid status codes */
        for (wIdx = 0 ; wIdx < END_STATUS_LIST ; ++wIdx)
        {
            if (strncmp((char*)pStrBuffer, (const char*)gpkphhalHw_FrameStatusCodesStr[wIdx], FRAME_STATUS_NCHAR) == 0)
            {
                wRxtatus = wIdx;
                break;
            }
        }
        /* Strip off the status code  */
        if (wRxtatus < END_STATUS_LIST)
        {
            wStrBufferIdx = wStrBufferIdx + FRAME_STATUS_NCHAR;
            pStrBuffer = &pRxFrame[wStrBufferIdx];
            /* Security check */
            if (wStrBufferIdx > strlen((char*)pRxFrame))
            {
                return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
            }
            PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
            PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bInfoStatus);
            PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, (const char*)gpkphhalHw_FrameStatusCodesStr[wRxtatus]);
            PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_GEN);
        }
        else
        {
            wRxtatus = RXSTAT_INVALID_STATUS;
            PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, bFunctionName);
            PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, bErrorNoStatus);
            PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, (char*)pRxFrame);
            PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_GEN);
        }
    }

	if ((bHeader==CMD_ENV) && wRxtatus == STAT_ACK)
	{
		/*
		Check for ATR frame on POR (walkaround)
		When final solution is ready leave only else statment content - delete averything in and including if statement
		*/
		/*(if(strncmp("RST_POR", (const char*)pTxBuffer, strlen("RST_POR")) == 0){
		* Security check */
		uint16_t tmp_bufLen = (uint16_t)(strlen((char*)pStrBuffer));
        if (bC40ModeEnabled == PH_OFF && (tmp_bufLen < (FRAME_NBYTES_NCHAR + FRAME_NBITS_NCHAR + 3)) && tmp_bufLen > 1 ){
			return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
		}
        /* if we have extra data, it should be in the normal format: 4 char byte length, 2 char bit length.
         * all the rest is optional and should match the byte/bit length */
        else if (bC40ModeEnabled == PH_ON && (tmp_bufLen < (FRAME_NBYTES_NCHAR + FRAME_NBITS_NCHAR)) && tmp_bufLen > 1 )
		{
			return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
		}
		else if(tmp_bufLen > 1)
		{
			/* Decode the received bytes when the status code is ACK, i.e. when wRxtatus code contains valid status */
			strncpy((char*)aTmpStr, (char*)pStrBuffer, FRAME_NBYTES_NCHAR);  /* PRQA S 3200 */
			aTmpStr[FRAME_NBYTES_NCHAR] = 0x00;
			if (isxdigit((int)aTmpStr[0]) && isxdigit((int)aTmpStr[1]) && isxdigit((int)aTmpStr[2]) && isxdigit((int)aTmpStr[3]))
			{
				/* Extract number of bytes */
				*pNBytes = (uint16_t)strtol((char*)aTmpStr, NULL, 16);
				wStrBufferIdx = wStrBufferIdx + FRAME_NBYTES_NCHAR;
				pStrBuffer = &pRxFrame[wStrBufferIdx];

				/* Extract number of bits */
				strncpy((char*)aTmpStr, (char*)pStrBuffer, FRAME_NBITS_NCHAR);  /* PRQA S 3200 */
				aTmpStr[FRAME_NBITS_NCHAR] = 0x00;
				if (isxdigit((int)aTmpStr[0]) && isxdigit((int)aTmpStr[1]))
				{
					*pNBits = (uint8_t)strtol((char*)aTmpStr, NULL, 16);

					if (*pNBits < 8)
					{
						wStrBufferIdx = wStrBufferIdx + FRAME_NBITS_NCHAR;
						pStrBuffer = &pRxFrame[wStrBufferIdx];
					}
					else
					{
						*pNBits = 0;
						wRxtatus = RXSTAT_INVALID_NBITS;
						PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, bFunctionName);
						PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, bErrorBit);
						PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, (char*)pRxFrame);
						PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_GEN);
					}
				}
				else
				{
					wRxtatus = RXSTAT_INVALID_NBITS;
					PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, bFunctionName);
					PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, bErrorBit);
					PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, (char*)pRxFrame);
					PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_GEN);
				}
			}
			else
			{
				wRxtatus = RXSTAT_INVALID_NBYTES;
				PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, bFunctionName);
				PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, bErrorByte);
				PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, (char*)pRxFrame);
				PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_GEN);
			}

			PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
			PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, pNBytes_log, pNBytes);
			PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_INFO, pNBits_log, pNBits);
			PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_GEN);

            if (bC40ModeEnabled == PH_OFF)
            {
                if ((uint16_t)strlen((char*)pStrBuffer) != (2 * (*pNBytes))) {
				    *pNBytes = 0;
				    wRxtatus = RXSTAT_INVALID_NPAYLD;
				    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, bFunctionName);
				    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, bErrorPayLoad);
				    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, (char*)pRxFrame);
				    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_GEN);

			    }

			    /*Quick check for default ATR header*/
			    if (*pNBytes > 2) {
				    for (wIdx = 0; wIdx < *pNBytes; ++wIdx)
				    {
					    strncpy((char*)aTmpStr, (char*)&pStrBuffer[2*wIdx], 2);  /* PRQA S 3200 */
					    aTmpStr[2] = 0x00;
					    pStrBuffer[wIdx] = (uint8_t)strtol((char*)aTmpStr, NULL, 16);
					    if (pPayld) {
						    /* if payload needs to be passed back to caller, copy it */
						    pPayld[wIdx] = pStrBuffer[wIdx];
					    }
				    }

				    if(pStrBuffer[0] != 0x3b && pStrBuffer[0] != 0x3f) {
					    wRxtatus = RXSTAT_INVALID_NPAYLD;
				    }
			    }
			    else {
				    wRxtatus = RXSTAT_INVALID_NPAYLD;
			    }
            }
            else
            {
				if ((uint16_t)strlen((char*)pStrBuffer) == (2 * (*pNBytes)))
				{
					for (wIdx = 0; wIdx < *pNBytes; ++wIdx)
					{
						strncpy((char*)aTmpStr, (char*)&pStrBuffer[2*wIdx], 2);  /* PRQA S 3200 */
						aTmpStr[2] = 0x00;
						pPayld[wIdx] = (uint8_t)strtol((char*)aTmpStr, NULL, 16);
						PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
						PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_INFO, pPayld_log, pPayld, wIdx+1);
						PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_GEN);
					}
				}
				else
				{
					*pNBytes = 0;
					wRxtatus = RXSTAT_INVALID_NPAYLD;
					PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, bFunctionName);
					PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, bErrorPayLoad);
					PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, (char*)pRxFrame);
					PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_GEN);
				}
            }
		}
	}
    /* Decode the PIC frame payload  */
    if (((bHeader == CMD_PIC) || (bHeader==CMD_WAF) || (bHeader==CMD_PCC)) && wRxtatus == STAT_ACK)
    {
        /* Security check */
        if (strlen((char*)pStrBuffer) < (FRAME_NBYTES_NCHAR + FRAME_NBITS_NCHAR))
        {
            return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
        }

        /* Decode the received bytes when the status code is ACK, i.e. when wRxtatus code contains valid status */
        strncpy((char*)aTmpStr, (char*)pStrBuffer, FRAME_NBYTES_NCHAR);  /* PRQA S 3200 */
        aTmpStr[FRAME_NBYTES_NCHAR] = 0x00;
        if (isxdigit((int)aTmpStr[0]) && isxdigit((int)aTmpStr[1]) && isxdigit((int)aTmpStr[2]) && isxdigit((int)aTmpStr[3]))
        {
            /* Extract number of bytes */
            *pNBytes = (uint16_t)strtol((char*)aTmpStr, NULL, 16);
            wStrBufferIdx = wStrBufferIdx + FRAME_NBYTES_NCHAR;
            pStrBuffer = &pRxFrame[wStrBufferIdx];

            /* Extract number of bits */
            strncpy((char*)aTmpStr, (char*)pStrBuffer, FRAME_NBITS_NCHAR);  /* PRQA S 3200 */
            aTmpStr[FRAME_NBITS_NCHAR] = 0x00;
            if (isxdigit((int)aTmpStr[0]) && isxdigit((int)aTmpStr[1]))
            {
                *pNBits = (uint8_t)strtol((char*)aTmpStr, NULL, 16);

                if (*pNBits < 8)
                {
                    wStrBufferIdx = wStrBufferIdx + FRAME_NBITS_NCHAR;
                    pStrBuffer = &pRxFrame[wStrBufferIdx];
                }
                else
                {
                    *pNBits = 0;
                    wRxtatus = RXSTAT_INVALID_NBITS;
                    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, bFunctionName);
                    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, bErrorBit);
                    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, (char*)pRxFrame);
                    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_GEN);
                }
            }
            else
            {
                wRxtatus = RXSTAT_INVALID_NBITS;
                PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, bFunctionName);
                PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, bErrorBit);
                PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, (char*)pRxFrame);
                PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_GEN);
            }
        }
        else
        {
            wRxtatus = RXSTAT_INVALID_NBYTES;
            PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, bFunctionName);
            PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, bErrorByte);
            PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, (char*)pRxFrame);
            PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_GEN);
        }

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, pNBytes_log, pNBytes);
        PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_INFO, pNBits_log, pNBits);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_GEN);

        if ((uint16_t)strlen((char*)pStrBuffer) == (2 * (*pNBytes)))
        {
            for (wIdx = 0; wIdx < *pNBytes; ++wIdx)
            {
                strncpy((char*)aTmpStr, (char*)&pStrBuffer[2*wIdx], 2);  /* PRQA S 3200 */
                aTmpStr[2] = 0x00;
                pPayld[wIdx] = (uint8_t)strtol((char*)aTmpStr, NULL, 16);
                PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
                PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_INFO, pPayld_log, pPayld, wIdx+1);
                PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_GEN);
            }
        }
        else
        {
            *pNBytes = 0;
            wRxtatus = RXSTAT_INVALID_NPAYLD;
            PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, bFunctionName);
            PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, bErrorPayLoad);
            PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, (char*)pRxFrame);
            PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_GEN);
        }
    }

    /* Decode the TUS frame payload */
    if (bHeader == CMD_TUS && wRxtatus == STAT_ACK)
    {
        /* Security check */
        if (strlen((char*)pStrBuffer) < FRAME_TIME_NCHAR)
        {
            return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
        }

        /* Decode the received bytes when the status code is ACK, i.e. when wRxtatus code contains valid status */
        strncpy((char*)aTmpStr, (char*)pStrBuffer, FRAME_TIME_NCHAR);  /* PRQA S 3200 */
        aTmpStr[FRAME_TIME_NCHAR] = 0x00;
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bInfoTimeValue);
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, (char*)aTmpStr);

        if (isxdigit((int)aTmpStr[0]) && isxdigit((int)aTmpStr[1]) && isxdigit((int)aTmpStr[2]) && isxdigit((int)aTmpStr[3]) &&
            isxdigit((int)aTmpStr[4]) && isxdigit((int)aTmpStr[5]) && isxdigit((int)aTmpStr[6]) && isxdigit((int)aTmpStr[7]))
        {
            memcpy(pPayld, (char*)pStrBuffer, strlen((char*)pStrBuffer)); /* PRQA S 3200 */
            pPayld[strlen((char*)pStrBuffer)] = 0x00;
        }
        else
        {
            wRxtatus = RXSTAT_INVALID_NBYTES;
            PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, bFunctionName);
            PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, bErrorTime);
            PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, (char*)pRxFrame);
            PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_GEN);
        }
    }

    /*   Decode the PMR frame payload  */
    if (bHeader == CMD_PMR && wRxtatus == STAT_ACK)
    {
        /* Security check */
        if (strlen((char*)pStrBuffer) < FRAME_TIME_NCHAR)
        {
            return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
        }

        /* Decode the received bytes when the status code is ACK, i.e. when wRxtatus code contains valid status */
        strncpy((char*)aTmpStr, (char*)pStrBuffer, FRAME_TIME_NCHAR);  /* PRQA S 3200 */
        aTmpStr[FRAME_TIME_NCHAR] = 0x00;

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bInfoReadValue);
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, (char*)aTmpStr);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_GEN);

        if (isxdigit((int)aTmpStr[0]) && isxdigit((int)aTmpStr[1]) && isxdigit((int)aTmpStr[2]) && isxdigit((int)aTmpStr[3]) &&
            isxdigit((int)aTmpStr[4]) && isxdigit((int)aTmpStr[5]) && isxdigit((int)aTmpStr[6]) && isxdigit((int)aTmpStr[7]))
        {
            memcpy(pPayld, pStrBuffer, FRAME_PRMDATA_NCHAR); /* PRQA S 3200 */
            pPayld[FRAME_PRMDATA_NCHAR] = 0x00;
        }
        else
        {
            wRxtatus = RXSTAT_INVALID_NBYTES;
            PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, bFunctionName);
            PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, bErrorRead);
            PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_ERROR, (char*)pRxFrame);
            PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_GEN);
        }
    }

    /*   Decode the MEM frame payload  */
    if (bHeader == CMD_MEM && wRxtatus == STAT_ACK)
    {
        /* Security check 1 ok if no chars returned i.e for MEM WRITE*/
         if (strlen((char*)pStrBuffer) > 0)
         {
            /* Security check 2 */
            if (strlen((char*)pStrBuffer) < FRAME_MEM_NBYTES)
            {
                return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
            }
            if (pPayld == NULL)
            {
                return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
            }
        }
    }

    switch (wRxtatus)
    {
    case RXSTAT_ACK :
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    case RXSTAT_NAK :
        return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_HAL);
    case RXSTAT_ETO :
        return PH_ADD_COMPCODE(PH_ERR_IO_TIMEOUT, PH_COMP_HAL);
    case RXSTAT_ECR :
        return PH_ADD_COMPCODE(PH_ERR_INTEGRITY_ERROR, PH_COMP_HAL);
    case RXSTAT_EPR :
        return PH_ADD_COMPCODE(PH_ERR_INTEGRITY_ERROR, PH_COMP_HAL);
    case RXSTAT_EFR :
        return PH_ADD_COMPCODE(PH_ERR_FRAMING_ERROR, PH_COMP_HAL);
    default :
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_HAL);
    }
}

phStatus_t phhalHw_RdCardSim_GetSimTime (
    phhalHw_RdCardSim_DataParams_t * pDataParams,
    uint8_t   * pSimTime
    )
{
    phStatus_t  status, statusTmp;
    uint16_t    wTmp;
    uint8_t     bTmp;
    uint8_t *   pTxFrame;
    uint8_t *   pRxFrame;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phhalHw_RdCardSim_GetSimTime");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pSimTime);

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    status = PH_ERR_SUCCESS;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferAlloc(pDataParams, &pTxFrame));
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_TxFrame(pDataParams, CMD_TUS, 0, 0, (uint8_t *)NULL, pTxFrame));

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferAlloc(pDataParams, &pRxFrame));
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalDataParams,
        PH_EXCHANGE_DEFAULT,
        pTxFrame,
        (uint16_t)strlen((char*)pTxFrame),
        pDataParams->wRxBufSize,
        pRxFrame,
        &wTmp));

    /* Length check */
    if (strlen((char*)pRxFrame) != (uint16_t)(wTmp - 1))
    {
        status = PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_HAL);
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_INFO, pSimTime_log, pSimTime, (uint16_t)strlen((char*)pSimTime));
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
        return status;
    }

    /* Remove the newline character from the input stream */
    pRxFrame[wTmp - 1] = 0x00;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferDeAlloc(pDataParams, pTxFrame));

    status = phhalHw_RdCardSim_RxFrame(pRxFrame, CMD_TUS, &wTmp, &bTmp, pSimTime, pDataParams->bC40ModeEnabled);
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdCardSim_BufferDeAlloc(pDataParams, pRxFrame));

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_INFO, pSimTime_log, pSimTime, (uint16_t)strlen((char*)pSimTime));
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return PH_ADD_COMPCODE( status, PH_COMP_HAL);
}

#endif /* NXPBUILD__PHHAL_HW_RDCARDSIM */
