/*
 * Copyright 2017, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal functions for PCSC HAL.
 *
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phhalHw.h>
#include <ph_RefDefs.h>
#include <ph_Status.h>

#ifdef NXPBUILD__PHHAL_HW_PCSC

#include "phhalHw_Pcsc.h"
#include "phhalHw_Pcsc_Int.h"

phStatus_t phhalHw_Pcsc_Int_GetTxBuffer(
                                        phhalHw_Pcsc_DataParams_t * pDataParams,
                                        uint8_t ** pTxBuffer,
                                        uint16_t * pTxBufferLen,
                                        uint16_t * pTxBufferSize
                                        )
{
    /* We need to watch that we do not overwrite content below the RxStartPos though */
    if (pDataParams->pTxBuffer == pDataParams->pRxBuffer)
    {
        *pTxBuffer = &pDataParams->pTxBuffer[pDataParams->wRxBufStartPos];
        *pTxBufferSize = pDataParams->wTxBufSize - pDataParams->wRxBufStartPos;
    }
    /* Else just return the actual Buffer */
    else
    {
        *pTxBuffer = pDataParams->pTxBuffer;
        *pTxBufferSize = pDataParams->wTxBufSize;
    }

    /* Return stored length */
    *pTxBufferLen = pDataParams->wTxBufLen;

    return PH_ERR_SUCCESS;
}

phStatus_t phhalHw_Pcsc_Int_GetRxBuffer(
                                        phhalHw_Pcsc_DataParams_t * pDataParams,
                                        uint8_t ** pRxBuffer,
                                        uint16_t * pRxBufferLen,
                                        uint16_t * pRxBufferSize
                                        )
{
    /* Get RX buffer details */
    *pRxBuffer = &pDataParams->pRxBuffer[pDataParams->wRxBufStartPos];
    *pRxBufferSize = pDataParams->wRxBufSize - pDataParams->wRxBufStartPos;
    *pRxBufferLen = pDataParams->wRxBufLen;

    return PH_ERR_SUCCESS;
}

phStatus_t phhalHw_Pcsc_Int_CheckErrorCode(
                                           phhalHw_Pcsc_DataParams_t * pDataParams,
                                           uint8_t * pErrorCode
                                           )
{
    /* Update error information */
    pDataParams->wErrorCode = ((((uint16_t)pErrorCode[0] << 8U) & 0xFF00U) | (pErrorCode[1]));

    if (pDataParams->wErrorCode == 0x9000U)
    {
        /* Command success */
        return PH_ERR_SUCCESS;
    }
    else if (pDataParams->wErrorCode == 0x6401U)
    {
        /* Timeout */
        return PH_ADD_COMPCODE(PH_ERR_IO_TIMEOUT, PH_COMP_HAL);
    }
    else
    {
        /* Refer bErrorCode data parameter for more information */
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }
}

phStatus_t phhalHw_Pcsc_Int_CheckResponse(
                                          phhalHw_Pcsc_DataParams_t * pDataParams,
                                          uint8_t * pInData,
                                          uint16_t wInDataLength,
                                          uint8_t ** ppOutData,
                                          uint16_t * pOutDataLength
                                          )
{
    phStatus_t PH_MEMLOC_REM statusTmp;
    uint16_t   PH_MEMLOC_REM wIndex = 0U;
    uint8_t *  PH_MEMLOC_REM pOutData = NULL;
    uint16_t   PH_MEMLOC_REM wOutDataLength = 0U;

    pDataParams->bRxLastBits = 0x00U;

    if (ppOutData == NULL)
    {
        ppOutData = &pOutData;
    }

    if (pOutDataLength == NULL)
    {
        pOutDataLength = &wOutDataLength;
    }

    /* Note: This check will not work if TLVs are combined in C-APDU */
    while (wIndex < wInDataLength)
    {
        /* Error status */
        if (pInData[wIndex] == 0xC0U)
        {
            if (pInData[++wIndex] != 0x03U)
            {
                return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
            }

            /* Number of bad data object - not checked */
            wIndex++;

            /* Check for error */
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_Int_CheckErrorCode(
                pDataParams,
                &pInData[wIndex + 1U]));

            /* Increment data index */
            wIndex += 2U;
        }
        /* Version */
        else if (pInData[wIndex] == 0x80U)
        {
            if (pInData[++wIndex] != 0x03U)
            {
                return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
            }

            *pOutDataLength = 3U;
            *ppOutData = &pInData[wIndex + 1U];

            /* Increment data index */
            wIndex += *pOutDataLength;
        }
        /* RX last bits */
        else if (pInData[wIndex] == 0x92U)
        {
            if (pInData[++wIndex] != 0x01U)
            {
                return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
            }

            if (pInData[++wIndex] <= 0x07U)
            {
                pDataParams->bRxLastBits = pInData[wIndex];
            }
            else
            {
                return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
            }
        }
        /* Response status */
        else if (pInData[wIndex] == 0x96U)
        {
            if (pInData[++wIndex] != 0x02U)
            {
                return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
            }

            /* Increment to error info */
            wIndex++;

            /* Check for error types */
            if (pInData[wIndex] & 0x01U)
            {
                return PH_ADD_COMPCODE(PH_ERR_INTEGRITY_ERROR, PH_COMP_HAL);
            }
            else if (pInData[wIndex] & 0x02U)
            {
                /* Update collision position */
                pDataParams->bCollPos = pInData[wIndex + 1U];

                return PH_ADD_COMPCODE(PH_ERR_COLLISION_ERROR, PH_COMP_HAL);
            }
            else if (pInData[wIndex] & 0x04U)
            {
                return PH_ADD_COMPCODE(PH_ERR_INTEGRITY_ERROR, PH_COMP_HAL);
            }
            else if (pInData[wIndex] & 0x08U)
            {
                return PH_ADD_COMPCODE(PH_ERR_FRAMING_ERROR, PH_COMP_HAL);
            }
            else
            {
                /* Do nothing */
            }

            /* Increment data index */
            wIndex++;
        }
        /* Card response */
        else if (pInData[wIndex] == 0x97U)
        {
            /* Response data */
            *pOutDataLength = pInData[++wIndex];
            *ppOutData = &pInData[wIndex + 1U];

            wIndex += *pOutDataLength;
        }
        /* Get parameter */
        else if ((pInData[wIndex] == 0xFFU) && (pInData[wIndex + 1U] == 0x6DU))
        {
            wIndex++;

            /* Length */
            if ((pInData[++wIndex] != 0x06U) && (pInData[wIndex] != 0x03U))
            {
                return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
            }

            /* Get parameter tag */
            if (pInData[++wIndex] > 0x0BU)
            {
                return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_HAL);
            }

            /* Get parameter length */
            if ((pInData[++wIndex] != 0x04U) && (pInData[wIndex] != 0x01U))
            {
                return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
            }

            /* Update length */
            *pOutDataLength = pInData[wIndex];

            /* Get parameter value */
            *ppOutData = &pInData[wIndex + 1U];

            wIndex += *pOutDataLength;
        }
        /* Switch protocol response */
        else if (pInData[wIndex] == 0x8FU)
        {
            wIndex++;

            /* Type A Layer 3 SAK */
            if (pInData[wIndex] == 0x01U)
            {
                /* Update SAK */
                pDataParams->bSak = pInData[++wIndex];
            }
            /* Type B Layer 3 Protocol Info */
            else if (pInData[wIndex] == 0x03U)
            {
                /* Note: Type B support not implemented */
                wIndex += 3U;
            }
            else
            {
                return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
            }
        }
        else
        {
            /* Invalid tag type */
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_HAL);
        }

        /* Move to next TLV */
        wIndex++;
    }

    return PH_ERR_SUCCESS;
}

phStatus_t phhalHw_Pcsc_Int_IsTransparentSession(
                                                 phhalHw_Pcsc_DataParams_t * pDataParams
                                                 )
{
    if (pDataParams->bTransparentSession == PH_ON)
    {
        /* Command success */
        return PH_ERR_SUCCESS;
    }
    else
    {
        /* Command needs transparent session */
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_HAL);
    }
}

void phhalHw_Pcsc_Int_GetLcValue(
                                 uint16_t * pLc,
                                 uint16_t wTxLength,
                                 uint8_t bTxBitsLen
                                 )
{
    *pLc = wTxLength + bTxBitsLen;

    /* Transceive TLV Length */
    if (wTxLength <= 0x7FU)
    {
        /* BER - TLV one byte length format */
        *pLc += 2U;
    }
    else if ((wTxLength > 0x7FU) && (wTxLength <= 0xFFU))
    {
        /* BER - TLV two byte length format */
        *pLc += 3U;
    }
    else
    {
        /* BER - TLV three byte length format */
        *pLc += 4U;
    }
}

phStatus_t phhalHw_Pcsc_Int_SetMinFDT(
                                      phhalHw_Pcsc_DataParams_t * pDataParams,
                                      uint16_t wValue
                                      )
{
    phStatus_t PH_MEMLOC_REM statusTmp = 0;
    uint16_t   PH_MEMLOC_REM wTxRate = 0;
    uint16_t   PH_MEMLOC_REM wTimer = 0;

    if (wValue == PH_ON)
    {
        /* Backup current timeout value */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_GetConfig(
            pDataParams,
            PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS,
            &wTimer));

        pDataParams->dwFdtPc = wTimer;

        /* Get the data rate */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_GetConfig(
            pDataParams,
            PHHAL_HW_CONFIG_TXDATARATE,
            &wTxRate));

        switch(wTxRate)
        {
            case PHHAL_HW_RF_DATARATE_106:
                wTimer = PHHAL_HW_MINFDT_106_US;
                break;

            case PHHAL_HW_RF_DATARATE_212:
                wTimer = PHHAL_HW_MINFDT_212_US;
                break;

            case PHHAL_HW_RF_DATARATE_424:
                wTimer = PHHAL_HW_MINFDT_424_US;
                break;

            case PHHAL_HW_RF_DATARATE_848:
                wTimer = PHHAL_HW_MINFDT_848_US;
                break;

            default:
                break;
        }

        /* Set minimum timeout value */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_SetConfig(
            pDataParams,
            PHHAL_HW_CONFIG_TIMEOUT_VALUE_US,
            wTimer));
    }
    else if (wValue == PH_OFF)
    {
        /* Restore back timeout value */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_Pcsc_SetConfig(
            pDataParams,
            PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS,
            pDataParams->dwFdtPc));
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    return PH_ERR_SUCCESS;
}

#endif  /* NXPBUILD__PHHAL_HW_PCSC */
