/*
 * Copyright 2013, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * RD710 specific HAL-Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phhalHw.h>
#include <phbalReg.h>
#include <ph_RefDefs.h>
#include <phTools.h>

#ifdef NXPBUILD__PHHAL_HW_MFCCRYPTO

#include "phhalHw_MfcCrypto.h"
#include "phhalHw_MfcCrypto_Cmd.h"
#include "phhalHw_MfcCrypto_Int.h"

/*
* Perform Crypto1 on data
*/
phStatus_t phhalHw_MfcCrypto_Cmd_CodeData(
    phhalHw_MfcCrypto_DataParams_t * pDataParams,
    uint8_t * pDataIn,
    uint32_t dwDataBitLen,
    uint8_t * pDataOut
    )
{
    uint32_t    PH_MEMLOC_REM dwIndex;
    uint8_t     PH_MEMLOC_REM bTmp;
    uint8_t     PH_MEMLOC_REM bTmp2;

    /* if we have less then 8 bits, then we have not received a hole byte
    and this means we have no parity. All we have to do then is to "decrypt"
    the few bits that we have */
    if (dwDataBitLen < 8)
    {
        /* only simple feedback mode is possible here (normal encryption) */
        for (dwIndex = 0; dwIndex < dwDataBitLen; ++dwIndex)
        {
            /* Retrieve bit */
            bTmp = 0x00;
            phhalHw_MfcCrypto_GetBits(pDataIn, dwDataBitLen, dwIndex, &bTmp, 1, 1);

            bTmp = (uint8_t)bTmp ^ (uint8_t)phhalHw_MfcCrypto_SimFeedCryptoReg(pDataParams->cryptoReg);

            /* Replace changed bit */
            phhalHw_MfcCrypto_InsertBits(pDataOut, &dwDataBitLen, dwIndex, &bTmp, 1, 1, 1);
        }
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    }

    for (dwIndex = 0; dwIndex < dwDataBitLen; ++dwIndex)
    {
        /* write byte */
        if ((dwIndex % 9) != 8)
        {
            /* Retrieve bit */
            bTmp = 0x00;
            phhalHw_MfcCrypto_GetBits(pDataIn, dwDataBitLen, dwIndex, &bTmp, 1, 1);

            switch (pDataParams->bCryptoState)
            {
                /* encrypt if necessary */
            case PHHAL_HW_MFCCRYPTO_STATE_SIM_FEED:
                /* simple feedback mode: normal encryption */
                bTmp = (uint8_t)bTmp ^ (uint8_t)phhalHw_MfcCrypto_SimFeedCryptoReg(pDataParams->cryptoReg);
                break;
            case PHHAL_HW_MFCCRYPTO_STATE_SP_FEED:
                /* special feedback 1 mode: for encrypting and feeding RA  */
                bTmp = (uint8_t)bTmp ^ (uint8_t)phhalHw_MfcCrypto_Feed1CryptoReg(pDataParams->cryptoReg, (uint8_t)(bTmp ^ (pDataParams->uIdForSpFeed1 & 1)));
                pDataParams->uIdForSpFeed1 = pDataParams->uIdForSpFeed1 >> 1;
                break;
            case PHHAL_HW_MFCCRYPTO_STATE_ENCRYPT_OFF:
                /* no encryption */
                /* pBitArray[index] = pBitArray[index]; */
                break;

            case PHHAL_HW_MFCCRYPTO_STATE_SP_FEED1_INIT:
                /* special feedback 1 mode: for feeding RB during initial authentication */
                bTmp2 = phhalHw_MfcCrypto_Feed1CryptoReg(pDataParams->cryptoReg, (uint8_t)(bTmp ^ (uint8_t)((uint8_t)pDataParams->uIdForSpFeed1 & 1)));
                pDataParams->uIdForSpFeed1 = pDataParams->uIdForSpFeed1 >> 1;
                break;
            case PHHAL_HW_MFCCRYPTO_STATE_SP_FEED2:
                /* special feedback 2 mode: for decrypting and feeding RB */

                bTmp = (uint8_t)bTmp ^ (uint8_t)phhalHw_MfcCrypto_Feed2CryptoReg(pDataParams->cryptoReg, (uint8_t)(bTmp ^ (pDataParams->uIdForSpFeed1 & 1)));
                pDataParams->uIdForSpFeed1 = pDataParams->uIdForSpFeed1 >> 1;
                break;
            default:
                break;
            }

            /* Replace changed bit */
            phhalHw_MfcCrypto_InsertBits(pDataOut, &dwDataBitLen, dwIndex, &bTmp, 1, 1, 1);
        }
        else
        {
            if ((pDataParams->bCryptoState == PHHAL_HW_MFCCRYPTO_STATE_SIM_FEED) || (pDataParams->bCryptoState == PHHAL_HW_MFCCRYPTO_STATE_SP_FEED) || (pDataParams->bCryptoState == PHHAL_HW_MFCCRYPTO_STATE_SP_FEED2))
            {
                /* Retrieve bit */
                bTmp = 0x00;
                phhalHw_MfcCrypto_GetBits(pDataIn, dwDataBitLen, dwIndex, &bTmp, 1, 1);

                /* parity bit: do not clock register ! */
                bTmp = bTmp ^ phhalHw_MfcCrypto_ComputeF(pDataParams->cryptoReg);

                /* Replace changed bit */
                phhalHw_MfcCrypto_InsertBits(pDataOut, &dwDataBitLen, dwIndex, &bTmp, 1, 1, 1);
            }
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

/*!
* 	Decodes AUTHENTICATION (1st part) response and updates crypto statemachine.
*/
phStatus_t phhalHw_MfcCrypto_Cmd_DecodeAuth1(
    phhalHw_MfcCrypto_DataParams_t * pDataParams,
    uint8_t* pAuth1Reply,
    uint16_t wAuth1ReplyLength,
    uint8_t bAuth1ReplyBits,
    uint8_t* pUid,
    uint8_t* pKey,
    uint8_t* pRndA,
    uint8_t* pRndB
    )
{
    phStatus_t  PH_MEMLOC_REM status;
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint8_t     PH_MEMLOC_REM aTmpKey[6];
    uint8_t     PH_MEMLOC_REM bOutBits;

    if ((wAuth1ReplyLength != 5) || (bAuth1ReplyBits != 4))
    {
        return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_HAL);
    }

    /* adapt pDataParams->bCryptoState after sending */
    if (pDataParams->bCryptoState == PHHAL_HW_MFCCRYPTO_STATE_ENCRYPT_OFF)
    {
        pDataParams->bCryptoState = PHHAL_HW_MFCCRYPTO_STATE_SP_FEED1_INIT; /* expecting unencrypted RB */
    }
    else
    {
        pDataParams->bCryptoState = PHHAL_HW_MFCCRYPTO_STATE_SP_FEED2; /* expecting encrypted E(RB) */
    }

    /* Reverse rA Key */
    memcpy(aTmpKey, pKey, 6); /* PRQA S 3200 */
    phhalHw_MfcCrypto_SwapByteArray (aTmpKey, 6);

    /* load key into crypto reg */
    phhalHw_MfcCrypto_LoadCryptoRegWithBuffer(pDataParams->cryptoReg, aTmpKey, MF_KEYSIZE_BYTES);

    /* process received RB */
    /* Remember UID */
    pDataParams->uIdForSpFeed1  = pUid[0];
    pDataParams->uIdForSpFeed1 |= (uint32_t)((uint32_t)pUid[1] << 8);
    pDataParams->uIdForSpFeed1 |= (uint32_t)((uint32_t)pUid[2] << 16);
    pDataParams->uIdForSpFeed1 |= (uint32_t)((uint32_t)pUid[3] << 24);
    memcpy(pDataParams->rA_buf, pRndA, 4);     /* PRQA S 3200 */

    /* WARNING: here the UID & RA must already be set .. */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_MfcCrypto_Cmd_CodeData(
        pDataParams,
        pAuth1Reply,
        (uint32_t)(((uint32_t)(wAuth1ReplyLength - 1) << 0x03U) + bAuth1ReplyBits),
        pAuth1Reply));

    /* Verify parity */
    status = phTools_DecodeParity(
        PH_TOOLS_PARITY_OPTION_ODD,
        pAuth1Reply,
        wAuth1ReplyLength,
        bAuth1ReplyBits,
        5,
        pAuth1Reply,
        &wAuth1ReplyLength,
        &bOutBits);

    /* Error at parity verification means AUTH error */
    if ((status & PH_ERR_MASK) != PH_ERR_SUCCESS)
    {
        return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_HAL);
    }

    /* Reverse rB Key */
    memcpy(pRndB, pAuth1Reply, wAuth1ReplyLength); /* PRQA S 3200 */
    phhalHw_MfcCrypto_SwapByteArray(pRndB, wAuth1ReplyLength);
    phhalHw_MfcCrypto_SeedRandRegWithBuffer(pDataParams->randReg, pRndB, MF_RANDSIZE_BYTES);

    /* adapt pDataParams->bCryptoState after processing received RB */
    pDataParams->bCryptoState = PHHAL_HW_MFCCRYPTO_STATE_SP_FEED; /* for feeding and encrypting RA */

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

/*!
* 	Decodes AUTHENTICATION (2nd part) response and updates crypto statemachine.
*/
phStatus_t phhalHw_MfcCrypto_Cmd_DecodeAuth2(
    phhalHw_MfcCrypto_DataParams_t * pDataParams,
    uint8_t * pAuth2Reply,
    uint16_t wAuth2ReplyLength,
    uint8_t bAuth2ReplyBits,
    uint8_t * pRndB2
    )
{
    phStatus_t  PH_MEMLOC_REM status;
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint8_t     PH_MEMLOC_REM aRndTmp[4];
    uint8_t     PH_MEMLOC_REM bOutBits;

    if ((wAuth2ReplyLength != 5) || (bAuth2ReplyBits != 4))
    {
        return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_HAL);
    }

    /* read and process RB'' */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_MfcCrypto_Cmd_CodeData(
        pDataParams,
        pAuth2Reply,
        (uint32_t)(((uint32_t)(wAuth2ReplyLength - 1) << 0x03U) + bAuth2ReplyBits),
        pAuth2Reply));

    status = phTools_DecodeParity(
        PH_TOOLS_PARITY_OPTION_ODD,
        pAuth2Reply,
        wAuth2ReplyLength,
        bAuth2ReplyBits,
        5,
        pAuth2Reply,
        &wAuth2ReplyLength,
        &bOutBits);

    /* Error at parity verification means AUTH error */
    if ((status & PH_ERR_MASK) != PH_ERR_SUCCESS)
    {
        return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_HAL);
    }

    memcpy(pRndB2, pAuth2Reply, wAuth2ReplyLength); /* PRQA S 3200 */

    /* generate and compare rB'' */
    phhalHw_MfcCrypto_ClockRandRegWithBuffer(pDataParams->randReg, pDataParams->rBp_buf+2, MF_RANDSIZE_BYTES >> 1); /* lower part of rB'', PRNG now contains upper part of rB'' */
    phhalHw_MfcCrypto_ClockRandRegWithBuffer(pDataParams->randReg, pDataParams->rBp_buf, MF_RANDSIZE_BYTES >> 1); /* upper part of rB'' */

    memcpy(aRndTmp, pDataParams->rBp_buf, wAuth2ReplyLength); /* PRQA S 3200 */
    phhalHw_MfcCrypto_SwapByteArray( aRndTmp, wAuth2ReplyLength );
    if (memcmp(aRndTmp, pRndB2, MF_RANDSIZE_BYTES) != 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_HAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

/*!
* 	Generates AUTHENTICATION (1st part) command.
*
* 	\param[in] 	authType 	The command code
* 	\param[in] 	blockAddr 	The data block address
* 	\param[out] auth1Msg    Encrypted Message
*   \return Length of encrypted message
*/
phStatus_t phhalHw_MfcCrypto_Cmd_EncodeAuth1(
    phhalHw_MfcCrypto_DataParams_t * pDataParams,
    uint8_t bAuthType,
    uint8_t bBlockAddr,
    uint8_t* pAuth1Msg
    )
{
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint16_t    PH_MEMLOC_REM wByteLength;
    uint8_t     PH_MEMLOC_REM aCmd[4];
    uint16_t    PH_MEMLOC_REM wCrc;
    uint16_t    PH_MEMLOC_REM wBitLength;
    uint8_t     PH_MEMLOC_REM bBits;

    /* Build message */
    aCmd[0] = bAuthType;
    aCmd[1] = bBlockAddr;

    PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc16(
        PH_TOOLS_CRC_OPTION_DEFAULT,
        PH_TOOLS_CRC16_PRESET_ISO14443A,
        PH_TOOLS_CRC16_POLY_ISO14443,
        aCmd,
        2,
        &wCrc));

    aCmd[2] = (uint8_t)(wCrc);
    aCmd[3] = (uint8_t)(wCrc >> 8);
    PH_CHECK_SUCCESS_FCT(statusTmp, phTools_EncodeParity(
        PH_TOOLS_PARITY_OPTION_ODD,
        aCmd,
        4,
        0,
        5,
        pAuth1Msg,
        &wByteLength,
        &bBits));

    wBitLength = (uint16_t)(bBits) ? (((uint16_t)(wByteLength - 1) << 0x03U) + bBits) : ((uint16_t)wByteLength << 0x03U);

    /* Perform Crypto1 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_MfcCrypto_Cmd_CodeData(
        pDataParams,
        pAuth1Msg,
        wBitLength,
        pAuth1Msg));

    /* Return command length */
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

/*!
* 	Generates AUTHENTICATION (2nd part) command.
*
* 	\param[out] auth2Msg    Encrypted Message
*   \return Length of encrypted message
*/
phStatus_t phhalHw_MfcCrypto_Cmd_EncodeAuth2(
    phhalHw_MfcCrypto_DataParams_t * pDataParams,
    uint8_t* pAuth2Msg
    )
{
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint8_t     PH_MEMLOC_REM aTmpRaIn[4];
    uint8_t     PH_MEMLOC_REM aTmpRa[5];
    uint8_t     PH_MEMLOC_REM aTmpRbIn[4];
    uint8_t     PH_MEMLOC_REM aTmpRb[5];
    uint16_t    PH_MEMLOC_REM wByteLength;
    uint16_t    PH_MEMLOC_REM wBitLength;
    uint8_t     PH_MEMLOC_REM bBits;
    uint32_t    PH_MEMLOC_REM dwBitCount;

    memcpy(aTmpRaIn, pDataParams->rA_buf, 4); /* PRQA S 3200 */

    /* Encrypt rA */
    PH_CHECK_SUCCESS_FCT(statusTmp, phTools_EncodeParity(
        PH_TOOLS_PARITY_OPTION_ODD,
        aTmpRaIn,
        4,
        0,
        sizeof(aTmpRa),
        aTmpRa,
        &wByteLength,
        &bBits));

    wBitLength = (uint16_t)(bBits) ? (((uint16_t)(wByteLength - 1) << 0x03U) + bBits) : ((uint16_t)wByteLength << 0x03U);
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_MfcCrypto_Cmd_CodeData(
        pDataParams,
        aTmpRa,
        wBitLength,
        aTmpRa));

    /* adapt pDataParams->bCryptoState after processing RA */
    pDataParams->bCryptoState = PHHAL_HW_MFCCRYPTO_STATE_SIM_FEED; /* from now on simple feedback (= normal encryption) */

    /* generate  rB' */
    phhalHw_MfcCrypto_ClockRandRegWithBuffer(pDataParams->randReg, pDataParams->rBp_buf/*dummy*/, MF_RANDSIZE_BYTES/2); /* not used: outputting upper part of rB */
    phhalHw_MfcCrypto_ClockRandRegWithBuffer(pDataParams->randReg, pDataParams->rBp_buf/*dummy*/, MF_RANDSIZE_BYTES); /* not used: clocks during rA, PRNG now contains lower part of rB */
    phhalHw_MfcCrypto_ClockRandRegWithBuffer(pDataParams->randReg, pDataParams->rBp_buf+2, MF_RANDSIZE_BYTES/2); /* lower part of rB', PRNG now contains upper part of rB' */
    phhalHw_MfcCrypto_ClockRandRegWithBuffer(pDataParams->randReg, pDataParams->rBp_buf, MF_RANDSIZE_BYTES/2); /* upper part of rB' */

    /* Encrypt rB */
    memcpy(aTmpRbIn, pDataParams->rBp_buf, 4); /* PRQA S 3200 */
    phhalHw_MfcCrypto_SwapByteArray(aTmpRbIn, 4);

    /* Encrypt rB */
    PH_CHECK_SUCCESS_FCT(statusTmp, phTools_EncodeParity(
        PH_TOOLS_PARITY_OPTION_ODD,
        aTmpRbIn,
        4,
        0,
        sizeof(aTmpRb),
        aTmpRb,
        &wByteLength,
        &bBits));

    wBitLength = (uint16_t)(bBits) ? (((uint16_t)(wByteLength - 1) << 0x03U) + bBits) : ((uint16_t)wByteLength << 0x03U);
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_MfcCrypto_Cmd_CodeData(
        pDataParams,
        aTmpRb,
        wBitLength,
        aTmpRb));

    /* prepare output buffer */
    dwBitCount = 0;
    phhalHw_MfcCrypto_InsertBits(pAuth2Msg, &dwBitCount, 0, aTmpRa, wBitLength, 1, 0);
    phhalHw_MfcCrypto_InsertBits(pAuth2Msg, &dwBitCount, dwBitCount, aTmpRb, wBitLength, 1, 0);

    /* Return command length */
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

#endif /* NXPBUILD__PHHAL_HW_MFCCRYPTO */
