/*
 * Copyright 2013, 2016 - 2017, 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/*
 * Licensee wrapper specific HAL-Component of Reader Library Framework.
 * $RCSfile $
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */


#include <ph_Status.h>
#include <phbalReg.h>
#include <phhalHw.h>
#ifdef NXPBUILD__PH_KEYSTORE
#include <phKeyStore.h>
#endif /* NXPBUILD__PH_KEYSTORE */
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHHAL_HW_LICENSEEWRAPPER

/* MS Windows PC/SC Subsystem */
#ifdef _WIN32
	#pragma warning(push)                   /* PRQA S 3116 */
	#pragma warning(disable:4200)           /* PRQA S 3116 */
	#pragma warning(disable:4201)           /* PRQA S 3116 */
	#include <windows.h>
	#pragma warning(pop)                    /* PRQA S 3116 */
#endif

#include "phhalHw_LicenseeWrapper.h"

/**
 * Macro to log an internal error
 */
#define RET_ON_ERR(err) if ((err)!=0) goto RET

/*==========================================================================*/
/*                                                                          */
/*    FUNCTION NAME: phhalHw_LicenseeWrapper_Init                           */
/*                                                                          */
/*==========================================================================*/
phStatus_t phhalHw_LicenseeWrapper_Init(phhalHw_LicenseeWrapper_DataParams_t  * UNALIGNED  pDataParams,
										uint16_t wSizeOfDataParams,
										uint8_t * pTxBuffer,
										uint16_t  wTxBufSize,
										uint8_t * pRxBuffer,
										uint16_t  wRxBufSize,
										int8_t  * pDllFile)
{
	phStatus_t  status = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phhalHw_LicenseeWrapper_Init");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wSizeOfDataParams);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wTxBufSize);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wRxBufSize);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wSizeOfDataParams_log, &wSizeOfDataParams);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wTxBufSize_log, &wTxBufSize);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wRxBufSize_log, &wRxBufSize);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pTxBuffer);
    PH_ASSERT_NULL (pRxBuffer);

	pDataParams->wId = PH_COMP_HAL | PHHAL_HW_LICENSEEWRAPPER_ID;
	pDataParams->pTxBuffer = pTxBuffer;
	pDataParams->wTxBufSize = wTxBufSize;
	pDataParams->pRxBuffer = pRxBuffer;
	pDataParams->wRxBufSize = wRxBufSize;

    /* Verify exchange buffers */
    if ((wRxBufSize == 0) || (wTxBufSize == 0))
    {
        RET_ON_ERR(status =  PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL));
    }



	/* Load the Licensee HAL DLL */
	pDataParams->LicenseeHalDll = (HINSTANCE)LoadLibrary(pDllFile);
	if (pDataParams->LicenseeHalDll == NULL) RET_ON_ERR(status = PH_ADD_COMPCODE(PH_ERR_LICENSEE_HAL, PH_COMP_HAL));

	/* Map the functions of the Licensee HAL DLL */
	pDataParams->pCbInit = (phhalHw_LicenseeWrapper_Init_t) GetProcAddress((HMODULE) pDataParams->LicenseeHalDll, "Init");
	pDataParams->pCbTerm = (phhalHw_LicenseeWrapper_Term_t) GetProcAddress((HMODULE) pDataParams->LicenseeHalDll, "Term");
	pDataParams->pCbSetConfig = (phhalHw_LicenseeWrapper_SetConfig_t) GetProcAddress((HMODULE) pDataParams->LicenseeHalDll, "SetConfig");
	pDataParams->pCbGetConfig = (phhalHw_LicenseeWrapper_GetConfig_t) GetProcAddress((HMODULE) pDataParams->LicenseeHalDll, "GetConfig");
	pDataParams->pCbFieldReset = (phhalHw_LicenseeWrapper_FieldReset_t) GetProcAddress((HMODULE) pDataParams->LicenseeHalDll, "FieldReset");
	pDataParams->pCbWait = (phhalHw_LicenseeWrapper_Wait_t) GetProcAddress((HMODULE) pDataParams->LicenseeHalDll, "Wait");
	pDataParams->pCbExchange = (phhalHw_LicenseeWrapper_Exchange_t) GetProcAddress((HMODULE) pDataParams->LicenseeHalDll, "Exchange");
	pDataParams->pCbApplyProtocolSettings = (phhalHw_LicenseeWrapper_ApplyProtocolSettings_t) GetProcAddress((HMODULE) pDataParams->LicenseeHalDll, "ApplyProtocolSettings");

	/* Check all the mandatory DLL APIs */
	if ((pDataParams->pCbSetConfig == NULL) ||
		(pDataParams->pCbGetConfig  == NULL) ||
		(pDataParams->pCbFieldReset == NULL) ||
		(pDataParams->pCbWait == NULL) ||
		(pDataParams->pCbExchange == NULL) ||
		(pDataParams->pCbApplyProtocolSettings == NULL))
	{
		RET_ON_ERR(status =  PH_ADD_COMPCODE(PH_ERR_LICENSEE_HAL, PH_COMP_HAL));
	}


	/* Call the init DLL API if present */
	if (pDataParams->pCbInit != NULL)
	{
		if (pDataParams->pCbInit((void ** UNALIGNED) &pDataParams->pContext) != 0) RET_ON_ERR(status = PH_ADD_COMPCODE(PH_ERR_LICENSEE_HAL, PH_COMP_HAL));
	}



RET:
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return status;
}


/*==========================================================================*/
/*                                                                          */
/*    FUNCTION NAME: phhalHw_LicenseeWrapper_Term                           */
/*                                                                          */
/*==========================================================================*/
phStatus_t phhalHw_LicenseeWrapper_Term(phhalHw_LicenseeWrapper_DataParams_t * pDataParams)
{
	phStatus_t  status = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);


	/* Call the Term DLL API if present */
	if (pDataParams->pCbTerm != NULL)
	{
		if (pDataParams->pCbTerm(pDataParams->pContext) != 0) status = PH_ADD_COMPCODE(PH_ERR_LICENSEE_HAL, PH_COMP_HAL);
	}

	/* Set all function pointers to NULL */
	pDataParams->pCbInit = (phhalHw_LicenseeWrapper_Init_t)NULL;
	pDataParams->pCbTerm = (phhalHw_LicenseeWrapper_Term_t)NULL;
	pDataParams->pCbSetConfig = (phhalHw_LicenseeWrapper_SetConfig_t)NULL;
	pDataParams->pCbGetConfig = (phhalHw_LicenseeWrapper_GetConfig_t)NULL;
	pDataParams->pCbFieldReset = (phhalHw_LicenseeWrapper_FieldReset_t)NULL;
	pDataParams->pCbWait = (phhalHw_LicenseeWrapper_Wait_t)NULL;
	pDataParams->pCbExchange = (phhalHw_LicenseeWrapper_Exchange_t)NULL;
	pDataParams->pCbApplyProtocolSettings = (phhalHw_LicenseeWrapper_ApplyProtocolSettings_t)NULL;

	/* ALways unload the Licensee HAL DLL */
	if (FreeLibrary(pDataParams->LicenseeHalDll) == FALSE)
	{
		status = PH_ADD_COMPCODE(PH_ERR_LICENSEE_HAL, PH_COMP_HAL);
	}


	/* Set the DLL handle to NULL */
	pDataParams->LicenseeHalDll = NULL;


	return status;
}

/*==========================================================================*/
/*                                                                          */
/*    FUNCTION NAME: phhalHw_LicenseeWrapper_SetConfig                      */
/*                                                                          */
/*==========================================================================*/
phStatus_t phhalHw_LicenseeWrapper_SetConfig(phhalHw_LicenseeWrapper_DataParams_t * pDataParams,
											 uint16_t  wConfig,
                                             uint16_t  wValue)
{



	phStatus_t  status =  PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);


    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phhalHw_LicenseeWrapper_SetConfig");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wConfig);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wValue);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, wConfig_log, &wConfig);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, wValue_log, &wValue);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);


	/* Call the SetConfig DLL API if present */
	if (pDataParams->pCbSetConfig != NULL)
	{
		if (pDataParams->pCbSetConfig(wConfig, wValue, pDataParams->pContext) != 0) status = PH_ADD_COMPCODE(PH_ERR_LICENSEE_HAL, PH_COMP_HAL);
	}




	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return status;
}


/*==========================================================================*/
/*                                                                          */
/*    FUNCTION NAME: phhalHw_LicenseeWrapper_GetConfig                      */
/*                                                                          */
/*==========================================================================*/
phStatus_t phhalHw_LicenseeWrapper_GetConfig(phhalHw_LicenseeWrapper_DataParams_t * pDataParams,
											 uint16_t   wConfig,
											 uint16_t * pValue)
{
	phStatus_t status = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);


    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phhalHw_LicenseeWrapper_GetConfig");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wConfig);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pValue);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, wConfig_log, &wConfig);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);


	/* Call the GetConfig DLL API if present */
	if (pDataParams->pCbGetConfig != NULL)
	{
		if (pDataParams->pCbGetConfig(wConfig, pValue, pDataParams->pContext) != 0) status = PH_ADD_COMPCODE(PH_ERR_LICENSEE_HAL, PH_COMP_HAL);
	}


	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, wConfig_log, &wConfig);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, pValue_log, pValue);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return status;

}


/*==========================================================================*/
/*                                                                          */
/*    FUNCTION NAME: phhalHw_LicenseeWrapper_FieldReset                     */
/*                                                                          */
/*==========================================================================*/
phStatus_t phhalHw_LicenseeWrapper_FieldReset(phhalHw_LicenseeWrapper_DataParams_t * pDataParams)
{
	phStatus_t status = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phhalHw_LicenseeWrapper_FieldReset");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Call the FieldReset DLL API if present */
	if (pDataParams->pCbFieldReset != NULL)
	{
		if (pDataParams->pCbFieldReset(pDataParams->pContext) != 0) status = PH_ADD_COMPCODE(PH_ERR_LICENSEE_HAL, PH_COMP_HAL);
	}


	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}

/*==========================================================================*/
/*                                                                          */
/*    FUNCTION NAME: phhalHw_LicenseeWrapper_Wait                           */
/*                                                                          */
/*==========================================================================*/
phStatus_t phhalHw_LicenseeWrapper_Wait(phhalHw_LicenseeWrapper_DataParams_t * pDataParams,
										uint8_t   bUnit,
										uint16_t  wTimeout)
{
    phStatus_t status = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phhalHw_LicenseeWrapper_Wait");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bUnit);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wTimeout);

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bUnit_log, &bUnit);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wTimeout_log, &wTimeout);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

	/* Call the Wait DLL API if present */
	if (pDataParams->pCbWait != NULL)
	{
		if (pDataParams->pCbWait(bUnit, wTimeout, pDataParams->pContext) != 0) status = PH_ADD_COMPCODE(PH_ERR_LICENSEE_HAL, PH_COMP_HAL);
	}



	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);


	return status;
}


/*==========================================================================*/
/*                                                                          */
/*    FUNCTION NAME: phhalHw_LicenseeWrapper_Exchange                       */
/*                                                                          */
/*==========================================================================*/
phStatus_t phhalHw_LicenseeWrapper_Exchange(phhalHw_LicenseeWrapper_DataParams_t * pDataParams,
											uint16_t    wOption,
											uint8_t   * pTxBuffer,
											uint16_t    wTxLength,
											uint8_t  ** ppRxBuffer,
											uint16_t  * pRxLength)
{

	phStatus_t status = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
	uint16_t rxLen, i;
	uint8_t * pRx;
	int32_t retCode = PH_ERR_SUCCESS;


	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phhalHw_LicenseeWrapper_Exchange");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wOption);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pTxBuffer);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wTxLength);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(ppRxBuffer);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pRxLength);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, wOption_log, &wOption);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, wTxLength_log, &wTxLength);
    PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_INFO, pTxBuffer_log, pTxBuffer, wTxLength);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);


	/* Check the transmit buffer */
	if (pTxBuffer == NULL)
	{
		pTxBuffer = pDataParams->pTxBuffer;
		wTxLength = pDataParams->wTxBufLen;
	}
	/* Reset the received number of bytes */
	pDataParams->wRxBufLen = 0;



	/* Call the Exchange DLL API if present */
	if (pDataParams->pCbExchange != NULL)
	{
		retCode = pDataParams->pCbExchange(wOption, pTxBuffer, wTxLength, &pRx, &rxLen, pDataParams->pContext);

		/* Check if RX buffer is big enough */
		if (rxLen > pDataParams->wRxBufSize) RET_ON_ERR(retCode = PH_ERR_PARAMETER_OVERFLOW);

		/* Copy the received bytes to the RX buffer */
		for (i = 0; i < rxLen; i++) pDataParams->pRxBuffer[i] = pRx[i];
		pDataParams->wRxBufLen = rxLen;

	}



	/* Update the output parameters */
	if (ppRxBuffer != NULL)	*ppRxBuffer = pDataParams->pRxBuffer;
	if (pRxLength != NULL) *pRxLength = pDataParams->wRxBufLen;
	/* Reset the transmit buffer length */
	pDataParams->wTxBufLen = 0;



RET:
	status = (phStatus_t) PH_ADD_COMPCODE(retCode, PH_COMP_HAL);
	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);

	if ((ppRxBuffer != NULL) && (pRxLength != NULL))
	{
		PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, pRxLength_log, pRxLength);
		PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_INFO, ppRxBuffer_log, *ppRxBuffer, (*pRxLength));
	}
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);


    return status;
}


/*==========================================================================*/
/*                                                                          */
/*    FUNCTION NAME: phhalHw_LicenseeWrapper_ApplyProtocolSettings          */
/*                                                                          */
/*==========================================================================*/
phStatus_t phhalHw_LicenseeWrapper_ApplyProtocolSettings(phhalHw_LicenseeWrapper_DataParams_t * pDataParams,
														 uint8_t bCardType)
{
	phStatus_t status = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);

	PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phhalHw_LicenseeWrapper_ApplyProtocolSettings");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bCardType);
	PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bCardType_log, &bCardType);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);


	/* Call the ApplyProtocolSettings DLL API if present */
	if (pDataParams->pCbApplyProtocolSettings != NULL)
	{
		if (pDataParams->pCbApplyProtocolSettings(bCardType, pDataParams->pContext) != 0) status = PH_ADD_COMPCODE(PH_ERR_LICENSEE_HAL, PH_COMP_HAL);
	}

	PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
	PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

	return status;
}


#endif /* NXPBUILD__PHHAL_HW_LICENSEEWRAPPER */

