/*
 * Copyright 2018 - 2020, 2023 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * DUT specific HAL-Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7463 $
 * $Date: 2025-08-29 15:56:52 +0530 (Fri, 29 Aug 2025) $
 *
 * History:
 *      Created On February 14, 2018
 *
 */

#include <ph_Status.h>

#ifdef NXPBUILD__PHHAL_HW_DUT
#include <ph_RefDefs.h>
#include <phhalHw_DUT_Cmd.h>

#include "phhalHw_DUT.h"

phStatus_t phhalHw_DUT_Init(phhalHw_DUT_DataParams_t * pDataParams, uint16_t wSizeOfDataParams, void * pBalDataParams, uint8_t bSlotNumber,
    uint8_t * pTxBuffer, uint16_t wTxBufSize, uint8_t * pRxBuffer, uint16_t wRxBufSize)
{
    /* Dataparam check */
    if(sizeof(phhalHw_DUT_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_HAL);
    }
    if((wTxBufSize < PHHAL_HW_DUT_RESERVED_BUFFER_LEN) || (wRxBufSize < PHHAL_HW_DUT_RESERVED_BUFFER_LEN))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }
    /* Pointer check*/
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_DATA_PARAM(pBalDataParams, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pTxBuffer, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pRxBuffer, PH_COMP_HAL);

    /* Modify Tx-/RxBuffer pointers/lengths */
    pTxBuffer += PHHAL_HW_DUT_RESERVED_BUFFER_LEN;
    wTxBufSize = wTxBufSize - PHHAL_HW_DUT_RESERVED_BUFFER_LEN;
    pRxBuffer += PHHAL_HW_DUT_RESERVED_BUFFER_LEN;
    wRxBufSize = wRxBufSize - PHHAL_HW_DUT_RESERVED_BUFFER_LEN;

    /* Init. private data */
    pDataParams->wId = PH_COMP_HAL | PHHAL_HW_DUT_ID;
    pDataParams->pBalDataParams = pBalDataParams;
    pDataParams->bSlotNumber = bSlotNumber;
    pDataParams->pTxBuffer = pTxBuffer;
    pDataParams->wTxBufSize = wTxBufSize;
    pDataParams->wTxBufLen = 0;
    pDataParams->wTxBufStart = 0;
    pDataParams->pRxBuffer = pRxBuffer;
    pDataParams->wRxBufSize = wRxBufSize;
    pDataParams->wRxBufLen = 0;
    pDataParams->wRxBufStartPos = 0;
    pDataParams->wTxBufStartPos = 0;
    pDataParams->bCardType = PHHAL_HW_CARDTYPE_ISO14443A;

    return PH_ERR_SUCCESS;
}

phStatus_t phhalHw_DUT_InitReader(phhalHw_DUT_DataParams_t * pDataParams, uint16_t wCmd)
{
    return PHHALHW_DUT_INT(pDataParams, wCmd, PH_EXCHANGE_DEFAULT, NULL, 0, NULL, NULL);
}






phStatus_t phhalHw_DUT_Int(phhalHw_DUT_DataParams_t * pDataParams, uint16_t wCmd, uint16_t wSlaveAddr, uint16_t wOption, uint8_t * pTxBuff,
    uint16_t wTxLen, uint16_t wBytesToRead, uint8_t ** ppRxBuff, uint16_t * pRxLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
    uint8_t *   PH_MEMLOC_REM pTxBuff_Tmp = NULL;
    uint16_t    PH_MEMLOC_REM wTxBuffSize_Tmp = 0;
    uint16_t    PH_MEMLOC_REM wTxBuffOffset_Tmp = 0;
    uint8_t *   PH_MEMLOC_REM pRxBuff_Tmp = NULL;
    uint16_t    PH_MEMLOC_REM wRxBuffSize_Tmp = 0;
    uint16_t    PH_MEMLOC_REM wRxBuffLen_Tmp = 0;
    uint8_t     PH_MEMLOC_REM bUseBuffering = 0;
    uint8_t     PH_MEMLOC_REM aTxBuff_Bkp[PHHAL_HW_DUT_RESERVED_BUFFER_LEN];
    uint8_t     PH_MEMLOC_REM aRxBuff_Bkp[PHHAL_HW_DUT_FRAME_HEADER_LEN];
    uint16_t    PH_MEMLOC_REM wTransmitStart = 0;
    uint16_t    PH_MEMLOC_REM wReceiveStart = 0;

    uint8_t		PH_MEMLOC_REM bExchangeTx = PH_ON;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;
    uint16_t    PH_MEMLOC_REM wTotRespLen = 0;
    uint16_t    PH_MEMLOC_REM wRemRespLen = 0;
    int16_t		PH_MEMLOC_REM wReadLen = 0;

    uint8_t     PH_MEMLOC_REM bRetries = 0U;

#ifndef NXPBUILD__PHHAL_HW_DUT_CMD_I2C
    PH_UNUSED_VARIABLE(wSlaveAddr);
    PH_UNUSED_VARIABLE(wBytesToRead);
#endif /* NXPBUILD__PHHAL_HW_DUT_CMD_I2C */

    /* Check if buffering needs to be used */
    switch(wCmd)
    {
        case PHHAL_HW_DUT_CMD_HAL_EXCHANGE:
        case PHHAL_HW_DUT_CMD_HAL_TRANSMIT:
#ifdef NXPBUILD__PHHAL_HW_DUT_CMD_I2C
        case PHHAL_HW_DUT_CMD_I2C_EXCHANGE:
        case PHHAL_HW_DUT_CMD_I2C_TRANSMIT:
        case PHHAL_HW_DUT_CMD_I2C_RECEIVE:
#endif /* NXPBUILD__PHHAL_HW_DUT_CMD_I2C */
            bUseBuffering = PH_OFF;
            break;

        default:
            bUseBuffering = PH_ON;
            break;
    }

    wTransmitStart = PHHAL_HW_DUT_RESERVED_BUFFER_LEN + pDataParams->wTxBufStartPos;
    wReceiveStart = PHHAL_HW_DUT_FRAME_HEADER_LEN + pDataParams->wRxBufStartPos;

    /* Init. temporary wStatus (satisfy compiler) */
    wStatus1 = PH_ERR_SUCCESS;

    if(pRxLen != NULL)
    {
        *pRxLen = 0;
    }

    /* */
    /* Evaluate temporary buffer */
    /* */
    if(bUseBuffering != PH_OFF)
    {
        /* TxBuffer equals RxBuffer */
        if(pDataParams->pTxBuffer == pDataParams->pRxBuffer)
        {
            /* Start at TxLenght if necessary */
            if(pDataParams->wTxBufLen >= pDataParams->wRxBufLen)
            {
                pTxBuff_Tmp = &pDataParams->pTxBuffer[pDataParams->wTxBufLen + wTransmitStart];
                wTxBuffSize_Tmp = pDataParams->wTxBufSize - pDataParams->wTxBufLen - wTransmitStart;
            }
            /* Else start at RxLenght */
            else
            {
                pTxBuff_Tmp = &pDataParams->pRxBuffer[pDataParams->wRxBufLen + wTransmitStart];
                wTxBuffSize_Tmp = pDataParams->wRxBufSize - pDataParams->wRxBufLen - wTransmitStart;
            }

            /* RxBuffer uses same buffer position as TxBuffer */
            pRxBuff_Tmp = pTxBuff_Tmp;
            wRxBuffSize_Tmp = wTxBuffSize_Tmp;
        }
        /* Buffers are different */
        else
        {
            pTxBuff_Tmp = &pDataParams->pTxBuffer[pDataParams->wTxBufLen + wTransmitStart];
            wTxBuffSize_Tmp = pDataParams->wTxBufSize - pDataParams->wTxBufLen - wTransmitStart;
            pRxBuff_Tmp = &pDataParams->pRxBuffer[pDataParams->wRxBufLen + wReceiveStart];
            wRxBuffSize_Tmp = pDataParams->wRxBufSize - pDataParams->wRxBufLen - wReceiveStart;
        }

        /* If a buffered command begins, reset TxBufferStart position */
        if((0U == ((wOption & PH_EXCHANGE_LEAVE_BUFFER_BIT))))
        {
            pDataParams->wTxBufStart = pDataParams->wTxBufLen;
        }

        /* If a buffered command is being sent, save Tx-/RxBuffer Contents */
        if((0U == ((wOption & PH_EXCHANGE_BUFFERED_BIT))))
        {
            (void) memcpy(aTxBuff_Bkp, pTxBuff_Tmp - PHHAL_HW_DUT_RESERVED_BUFFER_LEN, PHHAL_HW_DUT_RESERVED_BUFFER_LEN);
            (void) memcpy(aRxBuff_Bkp, pRxBuff_Tmp - PHHAL_HW_DUT_FRAME_HEADER_LEN, PHHAL_HW_DUT_FRAME_HEADER_LEN);
        }
    }
    else
    {
        pTxBuff_Tmp = &pDataParams->pTxBuffer[wTransmitStart];
        wTxBuffSize_Tmp = pDataParams->wTxBufSize;
        pRxBuff_Tmp = &pDataParams->pRxBuffer[wReceiveStart];
        wRxBuffSize_Tmp = pDataParams->wRxBufSize - wReceiveStart;
        pDataParams->wTxBufLen = 0;
        pDataParams->wRxBufLen = 0;
    }

    /* Buffer overflow check */
    if(wTxLen > wTxBuffSize_Tmp)
    {
        pDataParams->wTxBufLen = pDataParams->wTxBufStart;
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    /* Copy transmit buffer */
    if(pTxBuff != NULL)
    {
        (void) memcpy(pTxBuff_Tmp, pTxBuff, wTxLen);
    }

    /* Set Buffer pointer to start */
    if(bUseBuffering != PH_OFF)
    {
        /* Increment TxLenght */
        pDataParams->wTxBufLen = pDataParams->wTxBufLen + wTxLen;

        /* Shall we already perform the Exchange */
        if(0U != (wOption & PH_EXCHANGE_BUFFERED_BIT))
        {
            return PH_ERR_SUCCESS;
        }

        /* Save buffer length then Reset our transmit buffer contents */
        wTxLen = pDataParams->wTxBufLen - pDataParams->wTxBufStart;
        pDataParams->wTxBufLen = pDataParams->wTxBufStart;
    }

    /* Move RxBuffer pointer to start of receive header */
    pRxBuff_Tmp -= PHHAL_HW_DUT_FRAME_HEADER_LEN;
    wRxBuffSize_Tmp = wRxBuffSize_Tmp + PHHAL_HW_DUT_FRAME_HEADER_LEN;

    switch(wCmd)
    {
        /* VERSION COMMANDS ------------------------------------------------------------------------------ */
        case PHHAL_HW_DUT_CMD_GETVERSION:
        case PHHAL_HW_DUT_CMD_GETVERSION_NFC:
        case PHHAL_HW_DUT_CMD_GETVERSION_DATE_TIME:
        case PHHAL_HW_DUT_CMD_GETVERSION_READER_IC_TYPE:

        /* READER OPERATION ------------------------------------------------------------------------------ */
        case PHHAL_HW_DUT_CMD_RD_OPS_GET_READER_STATUS:
        case PHHAL_HW_DUT_CMD_RD_OPS_READ_REGISTER:
        case PHHAL_HW_DUT_CMD_RD_OPS_WRITE_REGISTER:
        case PHHAL_HW_DUT_CMD_RD_OPS_FIELD_ON:
        case PHHAL_HW_DUT_CMD_RD_OPS_FIELD_OFF:
        case PHHAL_HW_DUT_CMD_RD_OPS_FIELD_RESET:
        case PHHAL_HW_DUT_CMD_RD_OPS_READ_EEPROM:
        case PHHAL_HW_DUT_CMD_RD_OPS_WRITE_EEPROM:

#ifdef NXPBUILD__PHHAL_HW_DUT_CMD_CC
        /* CONTACT CARD COMMUNICATION -------------------------------------------------------------------- */
        case PHHAL_HW_DUT_INIT_CONTACT_CARD:
        case PHHAL_HW_DUT_CMD_CC_ACTIVATE:
        case PHHAL_HW_DUT_CMD_CC_COLD_RESET:
        case PHHAL_HW_DUT_CMD_CC_WARM_RESET:
        case PHHAL_HW_DUT_CMD_CC_STOP_CLOCK:
        case PHHAL_HW_DUT_CMD_CC_START_CLOCK:
        case PHHAL_HW_DUT_CMD_CC_DEACTIVATE:
        case PHHAL_HW_DUT_CMD_CC_PRESENCE_CHECK:
        case PHHAL_HW_DUT_CMD_CC_TRANSMIT_DATA:
        case PHHAL_HW_DUT_CMD_CC_SEND_PPS:
#endif /* NXPBUILD__PHHAL_HW_DUT_CMD_CC */
            wTxBuffOffset_Tmp = 4;
            pTxBuff_Tmp -= wTxBuffOffset_Tmp;
            pTxBuff_Tmp[0] = (uint8_t) ((uint16_t) wCmd >> 8U);
            pTxBuff_Tmp[1] = (uint8_t) wCmd;
            pTxBuff_Tmp[2] = (uint8_t) (wTxLen);
            pTxBuff_Tmp[3] = (uint8_t) (wTxLen >> 8U);
            wTxLen = wTxLen + wTxBuffOffset_Tmp;
            break;

        /* HAL ------------------------------------------------------------------------------------------- */
        case PHHAL_HW_DUT_INIT_CONTACTLESS:
        case PHHAL_HW_DUT_CMD_HAL_SET_CONFIG:
        case PHHAL_HW_DUT_CMD_HAL_GET_CONFIG:
        case PHHAL_HW_DUT_CMD_HAL_APPLY_PROT_SETTINGS:
        case PHHAL_HW_DUT_CMD_HAL_WAIT:
        case PHHAL_HW_DUT_CMD_HAL_MFC_AUTH:
        case PHHAL_HW_DUT_CMD_HAL_EXECUTE:
        case PHHAL_HW_DUT_CMD_HAL_MFC_AUTH_KEY_NO:
        case PHHAL_HW_DUT_CMD_HAL_I18000P3M3_INVENTORY:
        case PHHAL_HW_DUT_CMD_HAL_I18000P3M3_RESUME_INVENTORY:
            wTxBuffOffset_Tmp = 5;
            pTxBuff_Tmp -= wTxBuffOffset_Tmp;
            pTxBuff_Tmp[0] = (uint8_t) ((uint16_t) wCmd >> 8U);
            pTxBuff_Tmp[1] = (uint8_t) wCmd;
            pTxBuff_Tmp[2] = (uint8_t) (wTxLen + 1U);
            pTxBuff_Tmp[3] = (uint8_t) ((wTxLen + 1U) >> 8U);
            pTxBuff_Tmp[4] = pDataParams->bSlotNumber;
            wTxLen = wTxLen + wTxBuffOffset_Tmp;
            break;

        /* CONTACTLESS EXCHANGE COMMANDS ----------------------------------------------------------------- */
        case PHHAL_HW_DUT_CMD_HAL_EXCHANGE:
        case PHHAL_HW_DUT_CMD_HAL_TRANSMIT:
        case PHHAL_HW_DUT_CMD_HAL_RECEIVE:
            wTxBuffOffset_Tmp = 7;
            pTxBuff_Tmp -= wTxBuffOffset_Tmp;
            pTxBuff_Tmp[0] = (uint8_t) ((uint16_t) wCmd >> 8U);
            pTxBuff_Tmp[1] = (uint8_t) wCmd;
            pTxBuff_Tmp[2] = (uint8_t) (wTxLen + 3U);
            pTxBuff_Tmp[3] = (uint8_t) ((wTxLen + 3U) >> 8U);
            pTxBuff_Tmp[4] = pDataParams->bSlotNumber;
            pTxBuff_Tmp[5] = (uint8_t) (wOption);
            pTxBuff_Tmp[6] = (uint8_t) (wOption >> 8U);
            wTxLen = wTxLen + wTxBuffOffset_Tmp;
            break;

#ifdef NXPBUILD__PHHAL_HW_DUT_CMD_I2C
        /* I2C EXCHANGE COMMANDS ------------------------------------------------------------------------- */
        case PHHAL_HW_DUT_INIT_I2C:
        case PHHAL_HW_DUT_CMD_I2C_EXCHANGE:
        case PHHAL_HW_DUT_CMD_I2C_TRANSMIT:
        case PHHAL_HW_DUT_CMD_I2C_RECEIVE:
            wTxBuffOffset_Tmp = 10;
            pTxBuff_Tmp -= wTxBuffOffset_Tmp;
            pTxBuff_Tmp[0] = (uint8_t) ((uint16_t) wCmd >> 8U);
            pTxBuff_Tmp[1] = (uint8_t) wCmd;
            pTxBuff_Tmp[2] = (uint8_t) (wTxLen + 2U);
            pTxBuff_Tmp[3] = (uint8_t) ((wTxLen + 2U) >> 8U);
            pTxBuff_Tmp[4] = (uint8_t) (wSlaveAddr);
            pTxBuff_Tmp[5] = (uint8_t) (wSlaveAddr >> 8U);
            pTxBuff_Tmp[4] = (uint8_t) (wOption);
            pTxBuff_Tmp[5] = (uint8_t) (wOption >> 8U);
            pTxBuff_Tmp[4] = (uint8_t) (wBytesToRead);
            pTxBuff_Tmp[5] = (uint8_t) (wBytesToRead >> 8U);
            wTxLen = wTxLen + wTxBuffOffset_Tmp;
            break;

        case PHHAL_HW_DUT_CMD_I2C_START:
        case PHHAL_HW_DUT_CMD_I2C_STOP:
        case PHHAL_HW_DUT_CMD_I2C_REPEAT_START:
            wTxBuffOffset_Tmp = 4;
            pTxBuff_Tmp -= wTxBuffOffset_Tmp;
            pTxBuff_Tmp[0] = (uint8_t) ((uint16_t) wCmd >> 8U);
            pTxBuff_Tmp[1] = (uint8_t) wCmd;
            pTxBuff_Tmp[2] = (uint8_t) (wTxLen);
            pTxBuff_Tmp[3] = (uint8_t) (wTxLen >> 8U);
            wTxLen = wTxLen + wTxBuffOffset_Tmp;
            break;
#endif /* NXPBUILD__PHHAL_HW_DUT_CMD_I2C */

#ifdef NXPBUILD__PHHAL_HW_DUT_CMD_CRYPTOSYM
            /* SYMMETRIC CRYPTO OPERATION --------------------------------------------------------------------- */
        case PHHAL_HW_DUT_CMD_CRYPTOSYM_INIT:
        case PHHAL_HW_DUT_CMD_CRYPTOSYM_DEINIT:
        case PHHAL_HW_DUT_CMD_CRYPTOSYM_GET_STATUS:
        case PHHAL_HW_DUT_CMD_CRYPTOSYM_INVALIDATE_KEY:
        case PHHAL_HW_DUT_CMD_CRYPTOSYM_SET_CONFIG:
        case PHHAL_HW_DUT_CMD_CRYPTOSYM_GET_CONFIG:
        case PHHAL_HW_DUT_CMD_CRYPTOSYM_ENCRYPT:
        case PHHAL_HW_DUT_CMD_CRYPTOSYM_DECRYPT:
        case PHHAL_HW_DUT_CMD_CRYPTOSYM_CALCULATE_MAC:
        case PHHAL_HW_DUT_CMD_CRYPTOSYM_LOAD_IV:
        case PHHAL_HW_DUT_CMD_CRYPTOSYM_LOAD_KEY:
        case PHHAL_HW_DUT_CMD_CRYPTOSYM_LOAD_KEY_DIRECT:
        case PHHAL_HW_DUT_CMD_CRYPTOSYM_DIVERSIFY_KEY:
        case PHHAL_HW_DUT_CMD_CRYPTOSYM_DIVERSIFY_DIRECT_KEY:
            wTxBuffOffset_Tmp = 4;
            pTxBuff_Tmp -= wTxBuffOffset_Tmp;
            pTxBuff_Tmp[0] = (uint8_t) ((uint16_t) wCmd >> 8U);
            pTxBuff_Tmp[1] = (uint8_t) wCmd;
            pTxBuff_Tmp[2] = (uint8_t) (wTxLen);
            pTxBuff_Tmp[3] = (uint8_t) (wTxLen >> 8U);
            wTxLen = wTxLen + wTxBuffOffset_Tmp;
            break;
#endif /* NXPBUILD__PHHAL_HW_DUT_CMD_CRYPTOSYM */


#ifdef NXPBUILD__PHHAL_HW_DUT_CMD_CRYPTORNG
            /* RANDOM NUMBER GENERATION OPERATION ------------------------------------------------------------ */
        case PHHAL_HW_DUT_CMD_CRYPTORNG_INIT:
        case PHHAL_HW_DUT_CMD_CRYPTORNG_DEINIT:
        case PHHAL_HW_DUT_CMD_CRYPTORNG_GET_STATUS:
        case PHHAL_HW_DUT_CMD_CRYPTORNG_SEED:
        case PHHAL_HW_DUT_CMD_CRYPTORNG_RND:
            wTxBuffOffset_Tmp = 4;
            pTxBuff_Tmp -= wTxBuffOffset_Tmp;
            pTxBuff_Tmp[0] = (uint8_t) ((uint16_t) wCmd >> 8U);
            pTxBuff_Tmp[1] = (uint8_t) wCmd;
            pTxBuff_Tmp[2] = (uint8_t) (wTxLen);
            pTxBuff_Tmp[3] = (uint8_t) (wTxLen >> 8U);
            wTxLen = wTxLen + wTxBuffOffset_Tmp;
            break;
#endif /* NXPBUILD__PHHAL_HW_DUT_CMD_CRYPTORNG */

#ifdef NXPBUILD__PHHAL_HW_DUT_CMD_KEYSTORE
            /* KEYSTORE OPERATION ----------------------------------------------------------------------------- */
        case PHHAL_HW_DUT_CMD_KEYSTORE_INIT:
        case PHHAL_HW_DUT_CMD_KEYSTORE_DEINIT:
        case PHHAL_HW_DUT_CMD_KEYSTORE_FORMAT_KEY_ENTRY:
        case PHHAL_HW_DUT_CMD_KEYSTORE_SET_KUC:
        case PHHAL_HW_DUT_CMD_KEYSTORE_GET_KUC:
        case PHHAL_HW_DUT_CMD_KEYSTORE_CHANGE_KUC:
        case PHHAL_HW_DUT_CMD_KEYSTORE_SET_CONFIG:
        case PHHAL_HW_DUT_CMD_KEYSTORE_SET_CONFIG_BUFFER:
        case PHHAL_HW_DUT_CMD_KEYSTORE_GET_CONFIG:
        case PHHAL_HW_DUT_CMD_KEYSTORE_GET_CONFIG_BUFFER:
        case PHHAL_HW_DUT_CMD_KEYSTORE_SET_KEY:
        case PHHAL_HW_DUT_CMD_KEYSTORE_SET_KEY_AT_POS:
        case PHHAL_HW_DUT_CMD_KEYSTORE_SET_FULL_KEY_ENTRY:
        case PHHAL_HW_DUT_CMD_KEYSTORE_GET_KEY_ENTRY:
        case PHHAL_HW_DUT_CMD_KEYSTORE_GET_KEY:
        case PHHAL_HW_DUT_CMD_KEYSTORE_SET_KEY_ASYM:
        case PHHAL_HW_DUT_CMD_KEYSTORE_GET_KEY_ASYM:
#ifdef NXPBUILD__PHHAL_HW_DUT_CMD_KEYSTORE_NDA
            /* KEYSTORE PROVISIONING OPERATION ---------------------------------------------------------------- */
        case PHHAL_HW_DUT_CMD_KEYSTORE_PROVISION_INIT:
        case PHHAL_HW_DUT_CMD_KEYSTORE_PROVISION_DEINIT:
        case PHHAL_HW_DUT_CMD_KEYSTORE_PROVISION_SET_CONFIG:
        case PHHAL_HW_DUT_CMD_KEYSTORE_PROVISION_GET_CONFIG:
        case PHHAL_HW_DUT_CMD_KEYSTORE_PROVISION_APP_FIXED_KEYS:
#endif /* NXPBUILD__PHHAL_HW_DUT_CMD_KEYSTORE_NDA */
            wTxBuffOffset_Tmp = 4;
            pTxBuff_Tmp -= wTxBuffOffset_Tmp;
            pTxBuff_Tmp[0] = (uint8_t) ((uint16_t) wCmd >> 8U);
            pTxBuff_Tmp[1] = (uint8_t) wCmd;
            pTxBuff_Tmp[2] = (uint8_t) (wTxLen);
            pTxBuff_Tmp[3] = (uint8_t) (wTxLen >> 8U);
            wTxLen = wTxLen + wTxBuffOffset_Tmp;
            break;
#endif /* NXPBUILD__PHHAL_HW_DUT_CMD_KEYSTORE */

        default:
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_HAL);
    }

    wRxBuffLen_Tmp = 0;

    /* Read the Header Information. */
    wRemRespLen = 6;
    do
    {
        wStatus = phbalReg_Exchange(
            pDataParams->pBalDataParams,
            (uint16_t) (PH_EXCHANGE_DEFAULT | PHBAL_REG_SERIALWIN_SUPRESS_CHECKS),
            bExchangeTx ? pTxBuff_Tmp : NULL,
            (uint16_t) (bExchangeTx ? wTxLen : 0U),
            wRemRespLen,
            &pRxBuff_Tmp[wTotRespLen],
            &wRespLen);

        wTotRespLen += wRespLen;
        wRemRespLen -= wRespLen;
        bExchangeTx = PH_OFF;

        bRetries++;
        if(bRetries == 10U)
            break;

    } while(wTotRespLen != 6U);

    /* Update the remaining bytes to read. */
    wRemRespLen = (uint16_t) (pRxBuff_Tmp[4] | (pRxBuff_Tmp[5] << 8U));
    wRemRespLen = (uint16_t) ((wTotRespLen != 0) ? wRemRespLen : 0);

    wReadLen = wRemRespLen;

    /* Read the payload is available. */
    if(wRemRespLen != 0U)
    {
        do
        {
            wStatus = phbalReg_Exchange(
                pDataParams->pBalDataParams,
                (uint16_t) (PH_EXCHANGE_DEFAULT | PHBAL_REG_SERIALWIN_SUPRESS_CHECKS),
                NULL,
                0U,
                wReadLen,
                &pRxBuff_Tmp[wTotRespLen],
                &wRespLen);

            wTotRespLen += wRespLen;
            wReadLen -= wRespLen;
        } while(((wTotRespLen - 6U) != wRemRespLen) && (wReadLen > 0U));
    }

    wRxBuffLen_Tmp = wTotRespLen;

    /* Frame check */
    if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        /* Frame has to be at least 6 bytes */
        if(wRxBuffLen_Tmp < PHHAL_HW_DUT_FRAME_HEADER_LEN)
        {
            wStatus = PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_HAL);
        }
        else
        {
            /* Retrieve length */
            wRxBuffSize_Tmp = pRxBuff_Tmp[PHHAL_HW_DUT_FRAME_LEN_POS];
            wRxBuffSize_Tmp |= ((uint16_t) pRxBuff_Tmp[PHHAL_HW_DUT_FRAME_LEN_POS + 1U] << 8U);

            /* Length should match */
            if((wRxBuffLen_Tmp - PHHAL_HW_DUT_FRAME_HEADER_LEN) != wRxBuffSize_Tmp)
            {
                wStatus = PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_HAL);
            }
            else
            {
                /* Command Code should match */
                if((pRxBuff_Tmp[PHHAL_HW_DUT_FRAME_CMD_POS] != (uint8_t) ((uint16_t) wCmd >> 8U)) ||
                    (pRxBuff_Tmp[PHHAL_HW_DUT_FRAME_CMD_POS + 1U] != ((uint8_t) wCmd | 0x80U)))
                {
                    wStatus = PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_HAL);
                }
                else
                {
                    /* Retrieve wStatus */
                    wStatus1 = PH_ADD_COMPCODE(
                        (uint16_t) pRxBuff_Tmp[PHHAL_HW_DUT_FRAME_STATUS_POS],
                        ((uint16_t) pRxBuff_Tmp[PHHAL_HW_DUT_FRAME_STATUS_POS + 1U] << 8U));
                }
            }
        }
    }

    /* If a buffered command has been sent, restore Tx-/RxBuffer Contents */
    if(bUseBuffering != PH_OFF)
    {
        (void) memcpy(pTxBuff_Tmp, aTxBuff_Bkp, wTxBuffOffset_Tmp);
        (void) memcpy(pRxBuff_Tmp, aRxBuff_Bkp, PHHAL_HW_DUT_FRAME_HEADER_LEN);
    }

    /* Bail out on error */
    PH_CHECK_SUCCESS(wStatus);

    /* Return RxBuffer length */
    if(pRxLen != NULL)
    {
        *pRxLen = wRxBuffLen_Tmp - PHHAL_HW_DUT_FRAME_HEADER_LEN;
    }

    /* Return RxBuffer pointer */
    if((ppRxBuff != NULL) && (pRxLen != NULL))
    {
        (void) memcpy(&pRxBuff_Tmp[PHHAL_HW_DUT_FRAME_PAYLOAD_POS - PHHAL_HW_DUT_FRAME_HEADER_LEN], &pRxBuff_Tmp[PHHAL_HW_DUT_FRAME_PAYLOAD_POS], *pRxLen);
        *ppRxBuff = pRxBuff_Tmp;
    }

    /* Unbuffered command: store RxLength */
    if(bUseBuffering == PH_OFF)
    {
        pDataParams->wRxBufLen = wRxBuffLen_Tmp - PHHAL_HW_DUT_FRAME_HEADER_LEN;
    }

    return wStatus1;
}

phStatus_t phhalHw_DUT_SetMinFDT(phhalHw_DUT_DataParams_t * pDataParams, uint16_t wValue)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wTxRate = 0;
    uint16_t    PH_MEMLOC_REM wTimer = 0;

    if(wValue == PH_ON)
    {
        /*Backup the old Timer values and set min FDT*/
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_DUT_GetConfig(pDataParams,
            PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS, &wTimer));
        pDataParams->dwFdtPc = wTimer;

        /* Get the data rate */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_DUT_GetConfig(pDataParams,
            PHHAL_HW_CONFIG_TXDATARATE, &wTxRate));
        switch(wTxRate)
        {
            case PHHAL_HW_RF_DATARATE_106:
                wTimer = PHHAL_HW_MINFDT_106_US;
                break;

            case PHHAL_HW_RF_DATARATE_212:
                wTimer = PHHAL_HW_MINFDT_212_US;
                break;

            case PHHAL_HW_RF_DATARATE_424:
                wTimer = PHHAL_HW_MINFDT_424_US;
                break;

            case PHHAL_HW_RF_DATARATE_848:
                wTimer = PHHAL_HW_MINFDT_848_US;
                break;

            default:
                break;
        }
        /* Calculate values for Microsecond values */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_DUT_SetConfig(pDataParams,
            PHHAL_HW_CONFIG_TIMEOUT_VALUE_US, wTimer));
    }
    else if(wValue == PH_OFF)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_DUT_SetConfig(pDataParams,
            PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS, pDataParams->dwFdtPc));
    }
    else
    {
        /* Do nothing*/
    }
    return wStatus;
}

#endif /* NXPBUILD__PHHAL_HW_DUT */
