/*
 * Copyright 2022 - 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Hal DUT specific definition of Reader Operation commands.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7463 $
 * $Date: 2025-08-29 15:56:52 +0530 (Fri, 29 Aug 2025) $
 *
 * History:
 *   Created On 11 October, 2022
 */

#include <ph_Status.h>

#ifdef NXPBUILD__PHHAL_HW_DUT_CMD_RDOPS
#include <ph_RefDefs.h>
#include <phhalHw_DUT_Cmd.h>

#include "../phhalHw_DUT.h"

phStatus_t phhalHw_DUT_Cmd_RdOps_GetReaderStatus(phhalHw_DUT_DataParams_t * pDataParams, uint8_t ** ppReaderStatus,
    uint16_t * pReaderStatusLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Execute via Hal Command */
    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_RD_OPS_GET_READER_STATUS,
        PH_EXCHANGE_DEFAULT,
        NULL,
        0,
        ppReaderStatus,
        pReaderStatusLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_DUT_Cmd_RdOps_Read_Register(phhalHw_DUT_DataParams_t * pDataParams, uint32_t dwRegAddr,
    uint32_t * pRegData)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM * pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRspLen = 0;

    /* Execute via Hal Command */
    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_RD_OPS_READ_REGISTER,
        PH_EXCHANGE_DEFAULT,
        (uint8_t *) &dwRegAddr,
        4U,
        &pResponse,
        &wRspLen));

    /* Response Length Check. */
    if(wRspLen != 4U)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_HAL);
    }

    /* Copy the Response to return parameter. */
    *pRegData = (uint32_t) (pResponse[0U] | (pResponse[1U] << 8U) | (pResponse[2U] << 16U) | (pResponse[3U] << 24U));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_DUT_Cmd_RdOps_Write_Register(phhalHw_DUT_DataParams_t * pDataParams, uint32_t dwRegAddr,
    uint32_t dwRegData)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Execute via Hal Command */
    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_RD_OPS_WRITE_REGISTER,
        PH_EXCHANGE_BUFFER_FIRST,
        (uint8_t *) &dwRegAddr,
        4U,
        NULL,
        NULL));

    /* Execute via Hal Command */
    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_RD_OPS_WRITE_REGISTER,
        PH_EXCHANGE_BUFFER_LAST,
        (uint8_t *) &dwRegData,
        4U,
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_DUT_Cmd_RdOps_FieldOn(phhalHw_DUT_DataParams_t * pDataParams)
{
    /* Execute via Hal Command */
    return PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_RD_OPS_FIELD_ON,
        PH_EXCHANGE_DEFAULT,
        NULL,
        0,
        NULL,
        NULL);
}

phStatus_t phhalHw_DUT_Cmd_RdOps_FieldOff(phhalHw_DUT_DataParams_t * pDataParams)
{
    /* Execute via Hal Command */
    return PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_RD_OPS_FIELD_OFF,
        PH_EXCHANGE_DEFAULT,
        NULL,
        0,
        NULL,
        NULL);
}

phStatus_t phhalHw_DUT_Cmd_RdOps_FieldReset(phhalHw_DUT_DataParams_t * pDataParams)
{
    /* Execute via Hal Command */
    return PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_RD_OPS_FIELD_RESET,
        PH_EXCHANGE_DEFAULT,
        NULL,
        0,
        NULL,
        NULL);
}

phStatus_t phhalHw_DUT_Cmd_RdOps_Read_EEPROM(phhalHw_DUT_DataParams_t * pDataParams, uint16_t wOption,
    uint32_t dwAddr_ProtIndex, uint32_t dwNoOfBytesToRead, uint8_t ** ppResponse, uint16_t * pRespLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wRespDataStart = 0;

    uint8_t     PH_MEMLOC_REM aCmdData[9];
    uint8_t     PH_MEMLOC_REM bCmdDataLen = 0;

    /* Clear the buffer. */
    (void) memset(aCmdData, 0x00, sizeof(aCmdData));

    /* Read EEPROM command framing for Pegoda - 3 reader. */
    if((wOption == PHHALHW_TAG_EEPROM_USER_AREA) || (wOption == PHHALHW_TAG_EEPROM_SECURE_LIB_AREA) ||
        (wOption == PHHALHW_TAG_EEPROM_IC_CONFIG_AREA) || (wOption == PHHALHW_TAG_EEPROM_PROTOCOL_AREA))
    {
        /* Add 2 byte RFU */
        aCmdData[bCmdDataLen++] = 0;
        aCmdData[bCmdDataLen++] = 0;

        /* Add Tag of TLV packet */
        aCmdData[bCmdDataLen++] = (uint8_t) wOption;

        /* Add Protocol Index */
        if(wOption == PHHALHW_TAG_EEPROM_PROTOCOL_AREA)
        {
            /* Add Len of TLV packet */
            aCmdData[bCmdDataLen++] = 0x01U;
            aCmdData[bCmdDataLen++] = 0x00;

            (void) memcpy(&aCmdData[bCmdDataLen], (uint8_t *) &dwAddr_ProtIndex, 1U);
            bCmdDataLen += 1U;

            /* Position where the response data starts */
            wRespDataStart = 4U;
        }

        /* Add EEPROM Address and No Of bytes to Read */
        else
        {
            /* Add Len of TLV packet */
            aCmdData[bCmdDataLen++] = 0x04U;
            aCmdData[bCmdDataLen++] = 0x00;

            /* Add EEPROM Address. */
            (void) memcpy(&aCmdData[bCmdDataLen], (uint8_t *) &dwAddr_ProtIndex, 2U);
            bCmdDataLen += 2U;

            /* Add Number of bytes to read. */
            (void) memcpy(&aCmdData[bCmdDataLen], (uint8_t *) &dwNoOfBytesToRead, 2U);
            bCmdDataLen += 2U;

            /* Position where the response data starts */
            wRespDataStart = 5U;
        }
    }
    else
    {
        /* Add EEPROM Address. */
        (void) memcpy(&aCmdData[bCmdDataLen], (uint8_t *) &dwAddr_ProtIndex, 4U);
        bCmdDataLen += 4U;

        /* Add Number of bytes to read. */
        (void) memcpy(&aCmdData[bCmdDataLen], (uint8_t *) &dwNoOfBytesToRead, 4U);
        bCmdDataLen += 4U;

        if(wOption != 0)
        {
            aCmdData[bCmdDataLen++] = (uint8_t) wOption;
        }
    }

    /* Execute via Hal Command */
    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_RD_OPS_READ_EEPROM,
        PH_EXCHANGE_DEFAULT,
        aCmdData,
        bCmdDataLen,
        ppResponse,
        pRespLen));

    /* Remove Initial Information */
    ppResponse[0] += wRespDataStart;
    *pRespLen -= wRespDataStart;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_DUT_Cmd_RdOps_Write_EEPROM(phhalHw_DUT_DataParams_t * pDataParams, uint16_t wOption,
    uint32_t dwStartAddr, uint8_t * pData, uint16_t wDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wTotLen = 0;
    uint16_t    PH_MEMLOC_REM wCmdDataLen = 0;

    /* Calculate start position */
    wCmdDataLen = PHHAL_HW_DUT_RESERVED_BUFFER_LEN;

    /* Write EEPROM command framing for Pegoda - 3 reader. */
    if((wOption == PHHALHW_TAG_EEPROM_USER_AREA) || (wOption == PHHALHW_TAG_EEPROM_SECURE_LIB_AREA) ||
        (wOption == PHHALHW_TAG_EEPROM_IC_CONFIG_AREA) || (wOption == PHHALHW_TAG_EEPROM_PROTOCOL_AREA))
    {
        /* Add 2 byte RFU */
        pDataParams->pTxBuffer[wCmdDataLen++] = 0;
        pDataParams->pTxBuffer[wCmdDataLen++] = 0;

        /* Add Tag  */
        pDataParams->pTxBuffer[wCmdDataLen++] = (uint8_t) wOption;

        /* Add Value */
        if(wOption == PHHALHW_TAG_EEPROM_PROTOCOL_AREA)
        {
            /* Buffer overflow check */
            if((5U + wDataLen) > pDataParams->wTxBufSize)
            {
                pDataParams->wTxBufLen = 0;
                return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
            }

            /* Sum the total Length to be used for TLV frame */
            wTotLen = (uint16_t) (1U + wDataLen);

            /* Add Len of TLV packet */
            (void) memcpy(&pDataParams->pTxBuffer[wCmdDataLen], (uint8_t *) &wTotLen, 2U);
            wCmdDataLen += 2U;

            /* Add EEPROM Address. */
            (void) memcpy(&pDataParams->pTxBuffer[wCmdDataLen], (uint8_t *) &dwStartAddr, 1U);
            wCmdDataLen += 1U;
        }
        else
        {
            /* Buffer overflow check */
            if((7U + wDataLen) > pDataParams->wTxBufSize)
            {
                pDataParams->wTxBufLen = 0;
                return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
            }

            /* Sum the total Length to be used for TLV frame */
            wTotLen = (uint16_t) (2U + wDataLen);

            /* Add Len of TLV packet */
            (void) memcpy(&pDataParams->pTxBuffer[wCmdDataLen], (uint8_t *) &wTotLen, 2U);
            wCmdDataLen += 2U;

            /* Add EEPROM Address. */
            (void) memcpy(&pDataParams->pTxBuffer[wCmdDataLen], (uint8_t *) &dwStartAddr, 2U);
            wCmdDataLen += 2U;
        }
    }
    else
    {
        /* Buffer overflow check */
        if((5U + wDataLen) > pDataParams->wTxBufSize)
        {
            pDataParams->wTxBufLen = 0;
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }

        /* Add EEPROM Address. */
        (void) memcpy(&pDataParams->pTxBuffer[wCmdDataLen], (uint8_t *) &dwStartAddr, 4U);
        wCmdDataLen += 4U;

        if(wOption != 0)
        {
            pDataParams->pTxBuffer[wCmdDataLen++] = (uint8_t) wOption;
        }
    }

    /* Copy data to Transmit buffer. */
    memcpy(&pDataParams->pTxBuffer[wCmdDataLen], pData, wDataLen);
    wCmdDataLen += wDataLen;

    /* Execute via Hal Command */
    PH_CHECK_SUCCESS_FCT(wStatus, PHHALHW_DUT_INT(
        pDataParams,
        PHHAL_HW_DUT_CMD_RD_OPS_WRITE_EEPROM,
        PH_EXCHANGE_DEFAULT,
        NULL,
        (uint16_t) (wCmdDataLen - PHHAL_HW_DUT_RESERVED_BUFFER_LEN),
        NULL,
        NULL));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

#endif /* NXPBUILD__PHHAL_HW_DUT_CMD_RDOPS */
