/*
 * Copyright 2013, 2017, 2019 - 2020, 2024, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Hardare Thermostream TP04030 Component Application of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phbalReg.h>
#include <phdlThstrm.h>
#include <ph_RefDefs.h>
#include <phTools.h>

#ifdef NXPBUILD__PHDL_THSTRM_TP04030

#include "phdlThstrm_TP04030.h"
#include "phdlThstrm_TP04030_Int.h"
#include "../phdlThstrm_Int.h"

phStatus_t phdlThstrm_TP04030_Init(
                                   phdlThstrm_TP04030_DataParams_t * pDataParams,
                                   uint16_t wSizeOfDataParams,
                                   void * pBalRegDataParams
                                   )
{
    if (sizeof(phdlThstrm_TP04030_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_THSTRM);
    }
    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pBalRegDataParams);

    /* init private data */
    pDataParams->wId                    = PH_COMP_DL_THSTRM | PHDL_THSTRM_TP04030_ID;
    pDataParams->pBalRegDataParams      = pBalRegDataParams;
    pDataParams->wWaitTempTimeoutSec    = 3600;
    pDataParams->bCancellationToken     = PH_OFF;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_THSTRM);
}

phStatus_t phdlThstrm_TP04030_SetConfig(
                                        phdlThstrm_TP04030_DataParams_t * pDataParams,
                                        uint16_t wIdentifier,
                                        uint16_t wValue
                                        )
{
    switch (wIdentifier)
    {
    case PHDL_THSTRM_CONFIG_SELECT_SETP:
        return phdlThstrm_TP04030_Int_SetSelectedSetPoint(pDataParams, (uint8_t)wValue);

    case PHDL_THSTRM_CONFIG_SETP_TEMP:
        /* set setpoint temperature convert the unsigned uint16 value, which has to hold values according
        to the two's complement representation into a int16 value */
        return phdlThstrm_TP04030_Int_SetSetPointTemp(pDataParams, (int16_t)wValue);

    case PHDL_THSTRM_CONFIG_SETP_SOAKTIME:
        return phdlThstrm_TP04030_Int_SetSetPointSoakTime(pDataParams, wValue);

    case PHDL_THSTRM_CONFIG_SETP_WINDOW:
        /* here the cast doesn't matter as no negative temperatures for the window size are allowed */
        return phdlThstrm_TP04030_Int_SetSetPointWindow(pDataParams, (uint8_t)wValue);

    case PHDL_THSTRM_CONFIG_HEAD:
        return phdlThstrm_TP04030_Int_SetHead(pDataParams, (uint8_t)wValue);

    case PHDL_THSTRM_CONFIG_FLOW:
        return phdlThstrm_TP04030_Int_SetFlow(pDataParams, (uint8_t)wValue);

    case PHDL_THSTRM_CONFIG_FLOWRATE:
        return phdlThstrm_TP04030_Int_SetFlowRate(pDataParams, (uint8_t)wValue);

    case PHDL_THSTRM_CONFIG_SETP_TEMP_REACHED:
        /* read only */
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_THSTRM);

    case PHDL_THSTRM_CONFIG_CURR_TEMP:
        /* read only */
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_THSTRM);

    case PHDL_THSTRM_CONFIG_COMPRESSOR:
        return phdlThstrm_TP04030_Int_SetComprState(pDataParams, (uint8_t)wValue);

    case PHDL_THSTRM_CONFIG_HEADLOCKED:
        return phdlThstrm_TP04030_Int_SetHeadLockedState(pDataParams, (uint8_t)wValue);

    case PHDL_THSTRM_CONFIG_WAITTEMP_TIMEOUT_SEC:
        pDataParams->wWaitTempTimeoutSec = wValue;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_THSTRM);

    case PHDL_THSTRM_CONFIG_CANCELLATION_TOKEN:
        /* check value */
        if (wValue != PH_ON && wValue != PH_OFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
        }
        pDataParams->bCancellationToken = (uint8_t)wValue;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_THSTRM);

    case PHDL_THSTRM_CONFIG_EDIT_MODE:
        return phdlThstrm_TP04030_Int_SetEditMode(pDataParams, (uint8_t)wValue);

    case PHDL_THSTRM_CONFIG_TEMPERATURE_PERFORMANCE_MODE:
        return phdlThstrm_TP04030_Int_SetTemperaturePerformanceMode(pDataParams, (uint8_t)wValue);

    default:
        /* return error code */
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_THSTRM);
    }
}

phStatus_t phdlThstrm_TP04030_GetConfig(
                                        phdlThstrm_TP04030_DataParams_t * pDataParams,
                                        uint16_t wIdentifier,
                                        uint16_t * wValue
                                        )
{
    phStatus_t statusTmp;
    uint8_t bTmp;
    uint16_t wTmp;
    int16_t wTmp2;

    switch (wIdentifier)
    {
    case PHDL_THSTRM_CONFIG_SELECT_SETP:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_TP04030_Int_GetSelectedSetPoint(pDataParams, &bTmp));
        *wValue = (uint16_t) bTmp;
        break;

    case PHDL_THSTRM_CONFIG_SETP_TEMP:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_TP04030_Int_GetSetPointTemp(pDataParams, &wTmp2));
        /* cast signed temperature to unsigned dwValue, resulting in a two's complement representation */
        *wValue = (uint16_t) wTmp2;
        break;

    case PHDL_THSTRM_CONFIG_SETP_SOAKTIME:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_TP04030_Int_GetSetPointSoakTime(pDataParams, &wTmp));
        *wValue = (uint16_t) wTmp;
        break;

    case PHDL_THSTRM_CONFIG_SETP_WINDOW:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_TP04030_Int_GetSetPointWindow(pDataParams, &bTmp));
        *wValue = (uint16_t) bTmp;
        break;

    case PHDL_THSTRM_CONFIG_HEAD:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_TP04030_Int_GetHead(pDataParams, &bTmp));
        *wValue = (uint16_t) bTmp;
        break;

    case PHDL_THSTRM_CONFIG_FLOW:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_TP04030_Int_GetFlow(pDataParams, &bTmp));
        *wValue = (uint16_t) bTmp;
        break;

    case PHDL_THSTRM_CONFIG_FLOWRATE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_TP04030_Int_GetFlowRate(pDataParams, &bTmp));
        *wValue = (uint16_t) bTmp;
        break;

    case PHDL_THSTRM_CONFIG_SETP_TEMP_REACHED:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_TP04030_Int_GetSetPointTempReached(pDataParams, &bTmp));
        *wValue = (uint16_t) bTmp;
        break;

    case PHDL_THSTRM_CONFIG_CURR_TEMP:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_TP04030_Int_GetCurrentTemp(pDataParams, &wTmp2));
        /* cast signed temperature to unsigned dwValue, resulting in a two's complement representation */
        *wValue = (uint16_t) wTmp2;
        break;

    case PHDL_THSTRM_CONFIG_COMPRESSOR:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_TP04030_Int_GetComprState(pDataParams, &bTmp));
        if (bTmp == 0)
        {
            *wValue = PH_OFF;
        }
        else
        {
            *wValue = PH_ON;
        }
        break;

    case PHDL_THSTRM_CONFIG_HEADLOCKED:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_TP04030_Int_GetHeadLockedState(pDataParams, &bTmp));
        *wValue = (uint16_t) bTmp;
        break;

    case PHDL_THSTRM_CONFIG_WAITTEMP_TIMEOUT_SEC:
        *wValue = pDataParams->wWaitTempTimeoutSec;
        break;

    case PHDL_THSTRM_CONFIG_CANCELLATION_TOKEN:
        *wValue = pDataParams->bCancellationToken;
        break;

    case PHDL_THSTRM_CONFIG_EDIT_MODE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_TP04030_Int_GetEditMode(pDataParams, &bTmp));
        if (bTmp == 0)
        {
            *wValue = PH_OFF;
        }
        else
        {
            *wValue = PH_ON;
        }
        break;

    case PHDL_THSTRM_CONFIG_TEMPERATURE_PERFORMANCE_MODE:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_TP04030_Int_GetTemperaturePerformanceMode(pDataParams, &bTmp));
        if (bTmp == 0)
        {
            *wValue = PH_OFF;
        }
        else
        {
            *wValue = PH_ON;
        }
        break;

    default:
        /* return error code */
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_THSTRM);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_THSTRM);
}

phStatus_t phdlThstrm_TP04030_PowerOff(
                                       phdlThstrm_TP04030_DataParams_t * pDataParams
                                       )
{
    phStatus_t statusTmp;

    /* RETURN TO LOCAL MODE */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_TP04030_Int_Send(pDataParams, (uint8_t *)"%GL\n", 4));

    /* SET SHUTDOWN TIMER */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_TP04030_Int_Send(pDataParams, (uint8_t *)"STIM 1\n", 7));

    /* check if cmd is processed correct */
    return phdlThstrm_TP04030_Int_CheckCmdExec((phdlThstrm_TP04030_DataParams_t *)pDataParams);
}

phStatus_t phdlThstrm_TP04030_SoftReset(
                                        phdlThstrm_TP04030_DataParams_t * pDataParams
                                        )
{
    phStatus_t statusTmp;

    /* Check for cancellation */
    if (pDataParams->bCancellationToken == PH_ON)
    {
        pDataParams->bCancellationToken = PH_OFF;
        return PH_ADD_COMPCODE(PH_ERR_CANCELED, PH_COMP_DL_MSTAMPOSC);
    }

    /* clear the ESR register */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_TP04030_Int_Send(pDataParams, (uint8_t *)"*CLS\n", 5));

    /* clear the EROR register and wait 4 seconds after CLER command until device is ready to receive new commands */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_TP04030_Int_SendWait(pDataParams, (uint8_t *)"CLER\n", 5, 4000));

    /* check if cmd is processed correct */
    statusTmp = phdlThstrm_TP04030_Int_CheckCmdExec((phdlThstrm_TP04030_DataParams_t *)pDataParams);

    /* handle expected querry error (caused by reading and timeout) */
    if ((statusTmp & PH_ERR_MASK) == PHDL_THSTRM_ERR_QUERY_ERROR)
    {
        statusTmp = PH_ERR_SUCCESS;
    }

    return PH_ADD_COMPCODE(statusTmp, PH_COMP_DL_THSTRM);
}

phStatus_t phdlThstrm_TP04030_InitThstrm(
    phdlThstrm_TP04030_DataParams_t * pDataParams
    )
{
    phStatus_t statusTmp;
    uint8_t bRx[10];
    uint16_t wRxlen, wTemp;

    /* Check for cancellation */
    if (pDataParams->bCancellationToken == PH_ON)
    {
        pDataParams->bCancellationToken = PH_OFF;
        return PH_ADD_COMPCODE(PH_ERR_CANCELED, PH_COMP_DL_MSTAMPOSC);
    }

    /* read the error codes once just to clear them before starting remote connection */
    statusTmp = phdlThstrm_TP04030_Int_CheckCmdExec((phdlThstrm_TP04030_DataParams_t *)pDataParams);

    /* put device into remote mode */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_TP04030_Int_Send(pDataParams, (uint8_t *)"%RM\n", 4));

    /* read the status byte */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
        pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        (uint8_t*)"*STB?\n",
        6,
        10,
        (uint8_t*)bRx,
        &wRxlen));

    /* convert the response */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_Int_Atoui16((int8_t*)bRx, &wTemp));


    /* evaluate the status byte */
    if (((uint8_t) wTemp & 0x80) != 0x80)
    {
        return PH_ADD_COMPCODE(PHDL_THSTRM_ERR_NOT_READY, PH_COMP_DL_THSTRM);
    }
    return phdlThstrm_TP04030_Int_ConfigureThermostream((phdlThstrm_TP04030_DataParams_t *)pDataParams);
}

phStatus_t phdlThstrm_TP04030_WaitTemp(
                                       phdlThstrm_TP04030_DataParams_t * pDataParams
                                       )
{
    phStatus_t statusTmp;
    uint8_t bRx[10];
    uint16_t wRxlen, wTemp;
    uint8_t bTempReached;
    uint16_t wElapsedTime = 0;

    do
    {
        /* check for the at temperature (soak time has elapsed)-bit in Temperature event condition register */
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
            pDataParams->pBalRegDataParams,
            PH_EXCHANGE_DEFAULT,
            (uint8_t*)"TECR?\n",
            6,
            10,
            (uint8_t*)bRx,
            &wRxlen));

        /* convert response */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlThstrm_Int_Atoui16((int8_t*)bRx, &wTemp));
        bTempReached = (uint8_t)wTemp;

        /* if temp not reached sleep 1s and retry */
        if(!(bTempReached & 0x01))
        {
            if (wElapsedTime >= pDataParams->wWaitTempTimeoutSec)
            {
                return PH_ADD_COMPCODE(PHDL_THSTRM_ERR_EXECUTION_ERROR, PH_COMP_DL_THSTRM);
            }

            /* Check for cancellation */
            if (pDataParams->bCancellationToken == PH_ON)
            {
                pDataParams->bCancellationToken = PH_OFF;
                return PH_ADD_COMPCODE(PH_ERR_CANCELED, PH_COMP_DL_MSTAMPOSC);
            }

            phTools_Sleep(1000);
            wElapsedTime++;
        }

    } while (!(bTempReached & 0x01));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_THSTRM);
}

#endif /* NXPBUILD__PHDL_THSTRM_TP04030 */
