/*
 * Copyright 2013, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Hardware Steppper V1 Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phbalReg.h>
#include <ph_RefDefs.h>
#include <math.h>

#ifdef NXPBUILD__PHDL_STEPPER_V1

#include <phdlStepper.h>
#include "phdlStepper_V1.h"
#include "phdlStepper_V1_Int.h"
#include "../phdlStepper_Int.h"

phStatus_t phdlStepper_V1_Init(
                               phdlStepper_V1_DataParams_t * pDataParams,
                               uint16_t wSizeOfDataParams,
                               void * pBalRegDataParams
                               )
{
    if (sizeof(phdlStepper_V1_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_STEPPER);
    }
    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pBalRegDataParams);

    /* init private data */
    pDataParams->wId                    = PH_COMP_DL_STEPPER | PH_COMP_DL_STEPPER_V1_ID;
    pDataParams->pBalRegDataParams		= pBalRegDataParams;
    pDataParams->dwMax_Position			= 0xFFFFFFFF;
    pDataParams->bInit					= PH_OFF;
    pDataParams->bState					= PHDL_STEPPER_V1_INT_STATE_STDBY;
    pDataParams->wRamp_delay			= 2;
    pDataParams->bRamp_on_of			= PH_ON;
    pDataParams->bDirection				= PHDL_STEPPER_DIR_DOWN;
    pDataParams->wStart_Speed			= 0xF424;
    pDataParams->wMax_Speed				= 0xFC0B;
    pDataParams->wSteps_Way_Ratio		= 400;
    pDataParams->wDistance				= 10;
    pDataParams->bMaxCurrent			= 100;
    pDataParams->bStdby_Current			= 35;
    pDataParams->bStdby_Current_Flag	= 0;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t  phdlStepper_V1_Initialize(
                                      phdlStepper_V1_DataParams_t * pDataParams
                                      )
{
    phStatus_t PH_MEMLOC_REM status;

    /* Performe action to get the stepper into a defined starting state. */
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_V1_Int_Initparam(pDataParams));

    /* set init to two to indicate that we are in init and so that we can use the move functions */
    pDataParams->bInit = 2;

    /* get limits */
    status = phdlStepper_V1_Int_GetLimits(pDataParams);
    /* check the success, if fail -> reset bInit */
    if ((status & PH_ERR_MASK) != PH_ERR_SUCCESS)
    {
        pDataParams->bInit = PH_OFF;
    }
    else
    {
        pDataParams->bInit = PH_ON;
    }

    return PH_ADD_COMPCODE(status, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_V1_MoveDistance(
                                       phdlStepper_V1_DataParams_t * pDataParams,
                                       uint16_t wSpeed,
                                       uint8_t bDirection,
                                       uint32_t dwDistance,
                                       uint8_t bBlocking
                                       )
{
    uint32_t PH_MEMLOC_REM dwSteps;

    /* Test if Stepper is Initialised. */
    if (!pDataParams->bInit)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
    }

    /* Convert Distance in mm to Steps */
    dwSteps = (uint32_t)((float32_t)dwDistance / 1000.0f * (float32_t)pDataParams->wSteps_Way_Ratio);

    /* Use Move_Steps to perform task. */
    return phdlStepper_V1_MoveSteps(pDataParams,wSpeed , bDirection, dwSteps, bBlocking);
}

phStatus_t phdlStepper_V1_MoveSteps(
                                    phdlStepper_V1_DataParams_t * pDataParams,
                                    uint16_t wSpeed,
                                    uint8_t bDirection,
                                    uint32_t dwSteps,
                                    uint8_t bBlocking
                                    )
{
    phStatus_t  PH_MEMLOC_REM status;
    uint8_t     PH_MEMLOC_REM aTx[11];
    uint16_t    PH_MEMLOC_REM wTxLength;
    uint16_t    PH_MEMLOC_REM wRealSpeed;
    uint32_t    PH_MEMLOC_REM dwPosCount;
    uint32_t    PH_MEMLOC_REM dwCalcEndPos;
    uint32_t    PH_MEMLOC_REM dwCounter = 0;

    /* Test if Stepper is Initialised. */
    if (!pDataParams->bInit)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
    }

    /* Calculate real speed value with reference to the Stepper documentation. */
    wRealSpeed= (uint16_t)(0xFFFF - (uint16_t)(1000000000.0f / ((float32_t)pDataParams->wSteps_Way_Ratio * (float32_t)wSpeed)));

    PH_CHECK_SUCCESS_FCT(status, phdlStepper_V1_GetConfig32(pDataParams, PHDL_STEPPER_CONFIG32_POSITION_COUNTER, &dwPosCount));

    /* Check if the requested position is within the limit indicators. */
    if ((bDirection == PHDL_STEPPER_DIR_UP && (dwPosCount - dwSteps) > dwPosCount && pDataParams->bInit == PH_ON) ||
        (bDirection == PHDL_STEPPER_DIR_DOWN && dwSteps > (pDataParams->dwMax_Position - dwPosCount) && pDataParams->bInit == PH_ON))
    {
        return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS, PH_COMP_DL_STEPPER);
    }

    /* calculate the end position */
    dwCalcEndPos = dwPosCount + (uint32_t)(pow(-1.0f, (float32_t)bDirection) * (float32_t)dwSteps);

    /* Release from Limits if needed. */
    if (pDataParams->bState == PHDL_STEPPER_V1_INT_STATE_ERROR)
    {
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_V1_Int_ReleaseFromLimit(pDataParams));

        /* Recalculate steps if released from Limit. */
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_V1_GetConfig32(pDataParams, PHDL_STEPPER_CONFIG32_POSITION_COUNTER, &dwPosCount));

        if (dwCalcEndPos < dwPosCount)
        {
            dwSteps = dwPosCount - dwCalcEndPos;
            bDirection = PHDL_STEPPER_DIR_UP;
        }
        else
        {
            dwSteps = dwCalcEndPos - dwPosCount;
            bDirection = PHDL_STEPPER_DIR_DOWN;
        }
    }

    /* Set Stepper parameter accordingly */
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_SetConfig(pDataParams, PHDL_STEPPER_CONFIG_DIR, (uint16_t) bDirection));
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_V1_Int_GetFrame(PHDL_STEPPER_V1_INT_SND_STEPS, dwSteps, &aTx[0], &wTxLength));
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_Int_Send_Wait(pDataParams, &aTx[0], wTxLength, 100));
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_SetConfig(pDataParams, PHDL_STEPPER_CONFIG_MAX_SPEED, (uint16_t) wRealSpeed));

    PH_CHECK_SUCCESS_FCT(status, phdlStepper_Int_Wait(100));

    /* Start movement,  build command */
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_V1_Int_GetFrame(PHDL_STEPPER_V1_INT_STATE_DRIVE, 0, &aTx[0], &wTxLength));

    /* Send comand. */
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_Int_Send_Wait(pDataParams, &aTx[0], wTxLength, 20));

    /* set current state */
    pDataParams->bState = PHDL_STEPPER_V1_INT_STATE_DRIVE;

    /* request position counter untill the stepper answers again (when stepper is the standby state). */
    do
    {
        /* request postion counter */
        status = phdlStepper_V1_GetConfig32(pDataParams, PHDL_STEPPER_CONFIG32_POSITION_COUNTER, &dwPosCount);

        /* avoid to get stuck */
        dwCounter++;
        if (dwCounter >= 0xFFFFFFFE)
        {
            return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_DRIVE_FAILURE, PH_COMP_DL_STEPPER);
        }
    } while ((status & PH_ERR_MASK) == PH_ERR_IO_TIMEOUT);

    /* check status for any error else than timeout */
    PH_CHECK_SUCCESS(status);

    /* Check which error occured. */
    switch (dwPosCount >> 24)
    {
    case PHDL_STEPPER_V1_LIMIT_UPPER:
        pDataParams->bState = PHDL_STEPPER_V1_INT_STATE_ERROR;
        return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_UPPER_LIMIT_INDICATOR, PH_COMP_DL_STEPPER);

    case PHDL_STEPPER_V1_LIMIT_LOWER:
        pDataParams->bState = PHDL_STEPPER_V1_INT_STATE_ERROR;
        return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_LOWER_LIMIT_INDICATOR, PH_COMP_DL_STEPPER);

    default: /* Work around if Stepper can't catch the error massage */
        if ((abs((int16_t)(dwCalcEndPos - dwPosCount)) > 2)  && (bBlocking == PH_ON))
        {
            pDataParams->bState = PHDL_STEPPER_V1_INT_STATE_ERROR;

            if (bDirection == PHDL_STEPPER_DIR_DOWN)
            {
                return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_LOWER_LIMIT_INDICATOR, PH_COMP_DL_STEPPER);
            }
            else
            {
                return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_UPPER_LIMIT_INDICATOR, PH_COMP_DL_STEPPER);
            }
        }
        else
        {
            pDataParams->bState = PHDL_STEPPER_V1_INT_STATE_STDBY;
            return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
        }
    }
}

phStatus_t phdlStepper_V1_GoToPosition(
                                       phdlStepper_V1_DataParams_t * pDataParams,
                                       uint16_t wSpeed,
                                       uint32_t dwPosition,
                                       uint8_t bBlocking
                                       )
{
    phStatus_t  PH_MEMLOC_REM status;
    uint8_t     PH_MEMLOC_REM bDirection;
    uint32_t    PH_MEMLOC_REM dwSteps;
    uint32_t    PH_MEMLOC_REM dwPosCount;

    /* Test if Stepper is Initialised. */
    if (!pDataParams->bInit)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
    }

    /* Check if the requested position is within the limit indicators. */
    if (dwPosition > (uint32_t)(1000.0f * (float32_t)pDataParams->dwMax_Position / (float32_t)pDataParams->wSteps_Way_Ratio))
    {
        return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS, PH_COMP_DL_STEPPER);
    }

    /* Release from limits if necessary. */
    if (pDataParams->bState == PHDL_STEPPER_V1_INT_STATE_ERROR)
    {
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_V1_Int_ReleaseFromLimit(pDataParams));
    }

    /* Get current position */
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_V1_GetConfig32(pDataParams, PHDL_STEPPER_CONFIG32_POSITION_COUNTER, &dwPosCount));

    /* Calculate steps to position and direction. */
    if ((uint32_t)(dwPosition * pDataParams->wSteps_Way_Ratio / 1000) < (dwPosCount))
    {
        bDirection	= PHDL_STEPPER_DIR_UP;
        dwSteps = dwPosCount - (uint32_t)((float32_t)dwPosition / 1000.0f * (float32_t)pDataParams->wSteps_Way_Ratio);
    }
    else
    {
        bDirection	= PHDL_STEPPER_DIR_DOWN;
        dwSteps = (uint32_t)((float32_t)dwPosition / 1000.0f * (float32_t)pDataParams->wSteps_Way_Ratio - (float32_t)dwPosCount);
    }

    /* Start movement. */
    return phdlStepper_V1_MoveSteps(pDataParams, wSpeed , bDirection, dwSteps, bBlocking);
}

phStatus_t phdlStepper_V1_SetConfig(
                                    phdlStepper_V1_DataParams_t * pDataParams,
                                    uint16_t wConfig,
                                    uint16_t wValue
                                    )
{
    phStatus_t  PH_MEMLOC_REM status;
    uint8_t     PH_MEMLOC_REM aFrame[11];
    uint16_t    PH_MEMLOC_REM wFrameLength = 0;

    /* Write Intputvalue to the Stepper Parameters */
    switch (wConfig)
    {
    case PHDL_STEPPER_CONFIG_RAMP_ON_OFF:
        /* check value */
        if (wValue != PH_ON && wValue != PH_OFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        /* build command */
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_V1_Int_GetFrame(PHDL_STEPPER_V1_INT_SND_RAMP_ON_OFF, (uint32_t)wValue, &aFrame[0], &wFrameLength));
        pDataParams->bRamp_on_of = (uint8_t)wValue;
        break;

    case PHDL_STEPPER_CONFIG_DIR:
        /* check value */
        if (wValue != PHDL_STEPPER_DIR_DOWN && wValue != PHDL_STEPPER_DIR_UP)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        /* build command */
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_V1_Int_GetFrame(PHDL_STEPPER_V1_INT_SND_DIRECTION, (uint32_t)wValue, &aFrame[0], &wFrameLength));
        pDataParams->bDirection = (uint8_t)wValue;
        break;

    case PHDL_STEPPER_CONFIG_MAX_CURRENT:
        /* check value */
        if (wValue > 0xFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        /* build command */
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_V1_Int_GetFrame(PHDL_STEPPER_V1_INT_SND_MAX_CURRENT, (uint32_t)wValue, &aFrame[0], &wFrameLength));
        pDataParams->bMaxCurrent = (uint8_t)wValue;
        break;

    case PHDL_STEPPER_CONFIG_STDBY_CURRENT:
        /* check value */
        if (wValue > 0xFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        /* build command */
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_V1_Int_GetFrame(PHDL_STEPPER_V1_INT_SND_SBY_CURRENT, (uint32_t)wValue, &aFrame[0], &wFrameLength));
        pDataParams->bStdby_Current = (uint8_t)wValue;
        break;

    case PHDL_STEPPER_CONFIG_STDBY_CURRENT_FLAG:
        /* check value */
        if (wValue != PH_ON && wValue != PH_OFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        /* build command */
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_V1_Int_GetFrame(PHDL_STEPPER_V1_INT_SND_SBY_FLAG_CURRENT, (uint32_t)wValue, &aFrame[0], &wFrameLength));
        pDataParams->bStdby_Current_Flag = (uint8_t)wValue;
        break;

    case PHDL_STEPPER_CONFIG_RAMP_DELAY:
        /* build command */
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_V1_Int_GetFrame(PHDL_STEPPER_V1_INT_SND_RAMP_DELAY, (uint32_t)wValue, &aFrame[0], &wFrameLength));
        pDataParams->wRamp_delay = wValue;
        break;

    case PHDL_STEPPER_CONFIG_START_SPEED:
        /* check value */
        if (wValue > PHDL_STEPPER_V1_MAX_START_SPEED)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        /* build command */
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_V1_Int_GetFrame(PHDL_STEPPER_V1_INT_SND_START_SPEED, (uint32_t)wValue, &aFrame[0], &wFrameLength));
        pDataParams->wStart_Speed = wValue;
        break;

    case PHDL_STEPPER_CONFIG_MAX_SPEED:
        /* check value */
        if (wValue > PHDL_STEPPER_V1_MAX_MAX_SPEED)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        /* build command */
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_V1_Int_GetFrame(PHDL_STEPPER_V1_INT_SND_MAX_SPEED, (uint32_t)wValue, &aFrame[0], &wFrameLength));
        pDataParams->wMax_Speed = wValue;
        break;

    case PHDL_STEPPER_CONFIG_STEPS_WAY_RATIO:
        /* build command */
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_V1_Int_GetFrame(PHDL_STEPPER_V1_INT_SND_STEPS_WAY_RATIO, (uint32_t)wValue, &aFrame[0], &wFrameLength));
        pDataParams->wSteps_Way_Ratio = wValue;
        break;

    case PHDL_STEPPER_CONFIG_DISTANCE:
        /* build command */
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_V1_Int_GetFrame(PHDL_STEPPER_V1_INT_SND_DISTANCE, (uint32_t)wValue, &aFrame[0], &wFrameLength));
        pDataParams->wDistance = wValue;
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_DL_STEPPER);
    }

    /* Send frame and wait 100ms for stepper to process command */
    return phdlStepper_Int_Send_Wait(pDataParams, &aFrame[0], wFrameLength, 100);
}

phStatus_t phdlStepper_V1_SetConfig32(
                                      phdlStepper_V1_DataParams_t * pDataParams,
                                      uint16_t wConfig,
                                      uint32_t dwValue
                                      )
{
    phStatus_t  PH_MEMLOC_REM status;
    uint8_t     PH_MEMLOC_REM aFrame[11];
    uint16_t    PH_MEMLOC_REM wFrameLength = 0;

    /* Write Intputvalue to the Stepper Parameters */
    switch (wConfig)
    {
    case PHDL_STEPPER_CONFIG32_POSITION_COUNTER:
        /* build command */
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_V1_Int_GetFrame(PHDL_STEPPER_V1_INT_SND_POSITION_COUNTER, dwValue, &aFrame[0], &wFrameLength));
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_DL_STEPPER);
    }

    /* Send frame and wait 100ms for stepper to process command */
    return phdlStepper_Int_Send_Wait(pDataParams, &aFrame[0], wFrameLength, 100);
}

phStatus_t phdlStepper_V1_GetConfig(
                                    phdlStepper_V1_DataParams_t * pDataParams,
                                    uint16_t wConfig,
                                    uint16_t * pValue
                                    )
{
    switch(wConfig)
    {
        /* Values from the parameter set */
    case PHDL_STEPPER_CONFIG_RAMP_DELAY:
        *pValue = pDataParams->wRamp_delay;
        break;

    case PHDL_STEPPER_CONFIG_RAMP_ON_OFF:
        *pValue = (uint16_t) pDataParams->bRamp_on_of;
        break;

    case PHDL_STEPPER_CONFIG_DIR:
        *pValue = (uint16_t) pDataParams->bDirection;
        break;

    case PHDL_STEPPER_CONFIG_START_SPEED:
        *pValue = pDataParams->wStart_Speed;
        break;
    case PHDL_STEPPER_CONFIG_MAX_SPEED:
        *pValue = pDataParams->wMax_Speed;
        break;

    case PHDL_STEPPER_CONFIG_STEPS_WAY_RATIO:
        *pValue = pDataParams->wSteps_Way_Ratio;
        break;

    case PHDL_STEPPER_CONFIG_DISTANCE:
        *pValue = pDataParams->wDistance;
        break;

    case PHDL_STEPPER_CONFIG_MAX_CURRENT:
        *pValue = (uint16_t) pDataParams->bMaxCurrent;
        break;

    case PHDL_STEPPER_CONFIG_STDBY_CURRENT:
        *pValue = (uint16_t) pDataParams->bStdby_Current;
        break;

    case PHDL_STEPPER_CONFIG_STDBY_CURRENT_FLAG:
        *pValue = (uint16_t) pDataParams->bStdby_Current_Flag;
        break;

        /* Unknown Identifier */
    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_DL_STEPPER);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_V1_GetConfig32(
                                      phdlStepper_V1_DataParams_t * pDataParams,
                                      uint16_t wConfig,
                                      uint32_t * pValue
                                      )
{
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint8_t     PH_MEMLOC_REM aTx[11];
    uint8_t	    PH_MEMLOC_REM aRx[4];
    uint16_t    PH_MEMLOC_REM wRxLength;
    uint16_t    PH_MEMLOC_REM wFrameLength = 0;

    switch(wConfig)
    {
    case PHDL_STEPPER_CONFIG32_POSITION_COUNTER:

        /* Construct frame to send */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_V1_Int_GetFrame(PHDL_STEPPER_V1_INT_REQ_POSITION_COUNTER, 0, &aTx[0], &wFrameLength));

        /* send data */
        statusTmp = phbalReg_Exchange(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT,&aTx[0] ,wFrameLength, 4, &aRx[0] , &wRxLength);

        /* convert response */
        if (statusTmp == PH_ERR_SUCCESS)
        {
            *pValue = ((uint32_t) aRx[0] <<24) | ((uint32_t) aRx[1] <<16) | ((uint32_t) aRx[2] <<8) | aRx[3];
        }
        else
        {
            *pValue = 0;
        }
        return statusTmp;

    case PHDL_STEPPER_CONFIG32_MAX_POSITION:
        *pValue = (uint32_t) pDataParams->dwMax_Position;
        break;

        /* Unknown Identifier */
    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_DL_STEPPER);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

#endif /* NXPBUILD__PHDL_STEPPER_V1 */
