/*
 * Copyright 2017, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal Stepper Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phbalReg.h>
#include <phdlStepper.h>
#include <ph_RefDefs.h>
#include <phTools.h>
#include <ph_Status.h>

#ifdef NXPBUILD__PHDL_STEPPER_ROBOT

#include "phdlStepper_Robot.h"
#include "phdlStepper_Robot_Int.h"
#include "../phdlStepper_Int.h"

#ifndef _WIN32
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <stdio.h>              /* PRQA S 5124 */
#else
#pragma warning(push)           /* PRQA S 3116 */
#pragma warning(disable:4001)   /* PRQA S 3116 */
#include <ctype.h>
#include <stdlib.h>
#include <stdio.h>              /* PRQA S 5124 */
#pragma warning(pop)            /* PRQA S 3116 */
#endif

phStatus_t phdlStepper_Robot_Int_ConvertToStep(
                                        phdlStepper_Robot_DataParams_t * pDataParams,
                                        int32_t dwPos,
                                        int32_t * dwStepPos
                                        )
{
    /* pos is in 1/1000 mm -> 1 rotation (65536 units) is 4 mm
        -> val = 65536 / 4000 * pos */
    double unit = dwPos * ((double)pDataParams->wStepsWayRatio / 1000.0);

    if( unit < 0.0 )
    {
        *dwStepPos = -(int32_t)((-unit)+0.5);
    }
    else
    {
        *dwStepPos = (int32_t)((unit)+0.5);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Robot_Int_ConvertFromStep(
                                        phdlStepper_Robot_DataParams_t * pDataParams,
                                        int32_t dwStepPos,
                                        int32_t * dwPos
                                        )
{
    double unit = dwStepPos * (1000.0 / (double)pDataParams->wStepsWayRatio);

    if( unit < 0.0 )
    {
        *dwPos = -(int32_t)((-unit)+0.5);
    }
    else
    {
        *dwPos = (int32_t)((unit)+0.5);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Robot_Int_Reset(
                                        phdlStepper_Robot_DataParams_t * pDataParams
                                        )
{
    /*lots of garbage will come, but finally there's a "ready!" + 0x0A + 0x0D */
    uint8_t expected[] = { 'r', 'e', 'a', 'd', 'y', '!', 0x0a, 0x0d };
    uint8_t buf[1024];
    uint16_t wRxLen = 0, wTotalLen = 0, wCompareIndex = 0;
    uint16_t wValue=0;
    phStatus_t statusTmp;
    uint8_t repeat_cnt = 0;

    /* read current IO timeout  */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_GetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, &wValue));
    /* set IO timeout to long for RESET */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, 5000));

    /* send data */
    for( repeat_cnt = 0; repeat_cnt < pDataParams->wMaxGetPositionRetrys; repeat_cnt++ )
    {
        statusTmp = phbalReg_Exchange(pDataParams->pBalRegDataParams,
            PH_EXCHANGE_DEFAULT,
            PHDL_STEPPER_ROBOT_INT_SND_RESET,
            (uint16_t)strlen((char *)PHDL_STEPPER_ROBOT_INT_SND_RESET),
            sizeof(buf),
            buf,
            &wRxLen);

        if ((statusTmp & PH_ERR_MASK) == PH_ERR_SUCCESS && wRxLen > 10)
            break;
    }

    PH_CHECK_SUCCESS(statusTmp)
    wTotalLen = wRxLen;

    wCompareIndex = 0;
    while( wTotalLen < 8 || memcmp( buf+wCompareIndex, expected, 8 ) != 0 )
    {
        /* All available date analysed load new data */
        if( wCompareIndex+8 >= wTotalLen )
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
                pDataParams->pBalRegDataParams,
                PH_EXCHANGE_DEFAULT,
                NULL,
                0,
                sizeof(buf)-wTotalLen,
                (uint8_t *)(buf + wTotalLen),
                &wRxLen));
                wTotalLen = wTotalLen + wRxLen;
        }
        else
        {
            wCompareIndex++;
        }
    }

    /* restore IO timeout */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, wValue));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Robot_Approval(
                                        phdlStepper_Robot_DataParams_t * pDataParams
                                        )
{
    uint8_t buf[50];
    uint16_t wRxLen = 0;
    phStatus_t statusTmp;

    /* send data */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        PHDL_STEPPER_ROBOT_INT_SND_APPROVAL,
        (uint16_t)strlen((char *)PHDL_STEPPER_ROBOT_INT_SND_APPROVAL),
        sizeof(buf),
        buf,
        &wRxLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Robot_Int_GetStep(
                                        phdlStepper_Robot_DataParams_t * pDataParams,
                                        int32_t * dwStep
                                        )
{
    uint8_t buf[50];
    uint16_t wRxLen = 0;
    uint8_t repeat_cnt = 0;
    uint8_t clear_input_cnt = 0;
    uint16_t wValue=0;
    phStatus_t statusTmp, statusRestore;

    for( repeat_cnt = 0; repeat_cnt < pDataParams->wMaxGetPositionRetrys; repeat_cnt++ )
    {
        /* send data */
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams,
            PH_EXCHANGE_DEFAULT,
            PHDL_STEPPER_ROBOT_INT_SND_GET_POSITION,
            (uint16_t)strlen((char *)PHDL_STEPPER_ROBOT_INT_SND_GET_POSITION),
            sizeof(buf),
            buf,
            &wRxLen));

        if( memcmp((const char *) buf+5,"CPXIST:",7) == 0 )
        {
            if( sscanf( (const char *) buf+12, "%08x", dwStep ) == 1 )
            {
                return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
            }
        }

        /* Some Error at getting Position so clear input buffer
           Reading until get timeout (but max wMaxGetPositionRetrys times */

        /* read current IO timeout  */
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_GetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, &wValue));
        /* set IO timeout to a short value */
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, 10));

        for( clear_input_cnt = 0; clear_input_cnt < pDataParams->wMaxGetPositionRetrys; clear_input_cnt++ )
        {
            statusTmp = phbalReg_Exchange(
                pDataParams->pBalRegDataParams,
                PH_EXCHANGE_DEFAULT,
                NULL,
                0,
                sizeof(buf),
                buf,
                &wRxLen);
            /* A timeout is expected because the the buffer is cleared -> break */
            if ((statusTmp & PH_ERR_MASK) == PH_ERR_IO_TIMEOUT)
            {
                break;
            }
            /* If a overflow occured still more data available -> next round */
            if ((statusTmp & PH_ERR_MASK) == PH_ERR_BUFFER_OVERFLOW)
            {
                continue;
            }
            /* only next round with success -> any other case throw error */
            if (statusTmp != PH_ERR_SUCCESS)
            {
                /* restore IO timeout */
                PH_CHECK_SUCCESS_FCT(statusRestore, phbalReg_SetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, wValue));
                return statusTmp;
            }
        }
        /* restore IO timeout */
        PH_CHECK_SUCCESS_FCT(statusRestore, phbalReg_SetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, wValue));
    }
    *dwStep = 0;
    return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Robot_Int_WaitMoveFinished(
                                        phdlStepper_Robot_DataParams_t * pDataParams,
                                        int32_t dwTargetStep,
                                        uint16_t bDolerance
                                        )
{
    int32_t act_step = 0, old_step = 0;
    uint16_t wRetryCnt = 0;
    phStatus_t status;

    PH_CHECK_SUCCESS_FCT(status, phdlStepper_Robot_Int_GetStep(pDataParams, &act_step));
	old_step = act_step;

    while( abs( act_step-dwTargetStep ) > bDolerance || old_step != act_step )
	{
        if( old_step == act_step )
        {
            wRetryCnt++;
            if( wRetryCnt >= pDataParams->wWaitPositionTimeout )
            {
                return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS, PH_COMP_DL_STEPPER);
            }
        }
        else
        {
            wRetryCnt = 0;
        }
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_Int_Wait(10));
		old_step = act_step;
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_Robot_Int_GetStep(pDataParams, &act_step));
	}
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Robot_Int_SetTarget(
                                        phdlStepper_Robot_DataParams_t * pDataParams,
                                        int32_t dwTarget)
{
    uint8_t cmd[128];
    uint8_t buf[50];
    uint16_t wRxLen = 0;
    phStatus_t statusTmp;

    phdlStepper_Robot_Int_SetMode(pDataParams,3);

    sprintf( (char *)cmd, (char *)PHDL_STEPPER_ROBOT_INT_SND_SET_TARGET, dwTarget );

    /* send data */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        cmd,
        (uint16_t)strlen((char *)cmd),
        sizeof(buf),
        buf,
        &wRxLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Robot_Int_Start(
                                        phdlStepper_Robot_DataParams_t * pDataParams
                                        )
{
    uint8_t buf[50];
    uint16_t wRxLen = 0;
    phStatus_t statusTmp;

    /* send data */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        PHDL_STEPPER_ROBOT_INT_SND_START,
        (uint16_t)strlen((char *)PHDL_STEPPER_ROBOT_INT_SND_START),
        sizeof(buf),
        buf,
        &wRxLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Robot_Int_SetMode(
                                        phdlStepper_Robot_DataParams_t * pDataParams,
                                        uint8_t mode
                                        )
{
    uint8_t cmd[128];
    uint8_t buf[50];
    uint16_t wRxLen = 0;
    phStatus_t statusTmp;

    sprintf( (char *)cmd, (char *)PHDL_STEPPER_ROBOT_INT_SND_SET_POS_MODE, mode );

    /* send data */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        cmd,
        (uint16_t)strlen((char *)cmd),
        sizeof(buf),
        buf,
        &wRxLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Robot_Int_SetMaxUpm(
                                        phdlStepper_Robot_DataParams_t * pDataParams,
                                        uint16_t wUpm
                                        )
{
    uint8_t cmd[128];
    uint8_t buf[50];
    uint16_t wRxLen = 0;
    phStatus_t statusTmp;

    wUpm = wUpm / 64;
    if( wUpm == 0 )
    {
        wUpm = 1;
    }

    sprintf( (char *)cmd, (char *)PHDL_STEPPER_ROBOT_INT_SND_SET_MAX_UPM, wUpm );

    /* send data */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        cmd,
        (uint16_t)strlen((char *)cmd),
        sizeof(buf),
        buf,
        &wRxLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Robot_Int_GoToStep(
                                       phdlStepper_Robot_DataParams_t * pDataParams,
                                       uint16_t wSpeed,
                                       int32_t dwStep,
                                       uint8_t bBlocking
                                       )
{
    phStatus_t status;
    uint16_t limit = 400;

    if( wSpeed > pDataParams->wMaxSpeed ||
        (bBlocking != PH_ON && bBlocking != PH_OFF))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
    }

    if( dwStep < 0 || (uint32_t)dwStep > pDataParams->dwMaxPosition )
    {
        return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS, PH_COMP_DL_STEPPER);
    }

    PH_CHECK_SUCCESS_FCT(status, phdlStepper_Robot_Int_SetMaxUpm(pDataParams, wSpeed));
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_Robot_Int_SetTarget(pDataParams, dwStep));
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_Robot_Int_Start(pDataParams));

    if( bBlocking == PH_ON )
    {
        if( wSpeed > 1000 )
        {
            limit = 800;
        }
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_Robot_Int_WaitMoveFinished(pDataParams,dwStep,limit));
    }
    return PH_ADD_COMPCODE(status, PH_COMP_DL_STEPPER);
}

/*int Robot::Reference()
{
  string cmd = "REFGO!";

  unsigned char buf[16];
  unsigned int len = 12;
  int err;

  err = Serial::Write( cmd );
  if( err < 0 )
    return err;

  err = Read( buf, len );
  return err;
}

int Robot::SetPos( long pos )
{
  pos = -ConvertToRobot( pos );
  unsigned char cmd[32];
  sprintf( (char *) cmd, "SPSETX:%08X", pos );
  int len = 15;
  int err;

  err = Write( cmd, len );
  if( err < 0 )
    return err;

  unsigned char buf[32];
  err = Read( buf, 21 );
  return err;
}

int Robot::StepDown( int rel )
{
  rel = ConvertToRobot( rel );
  int err = SetPosMode( posModeRel | posStartBreak );
  if( err < 0 )
    return err;

  char cmd[32];
  sprintf( cmd, "SPX:01:%08X", rel );
  int len = 15;

  err = Write( (unsigned char *) cmd, len );
  if( err < 0 )
    return err;

  unsigned char buf[32];
  err = Read( buf, 21 );
  if( err < 0 )
    return err;

  return Start();
}

int Robot::StepUp( int rel )
{
  return StepDown( -rel );
}




*/
#endif /* NXPBUILD__PHDL_STEPPER_ROBOT */
