/*
 * Copyright 2017, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Hardware Steppper Robot Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phbalReg.h>
#include <ph_RefDefs.h>
#include <math.h>

#ifdef NXPBUILD__PHDL_STEPPER_ROBOT

#include <phdlStepper.h>
#include "phdlStepper_Robot.h"
#include "phdlStepper_Robot_Int.h"
#include "../phdlStepper_Int.h"

phStatus_t phdlStepper_Robot_Init(
                               phdlStepper_Robot_DataParams_t * pDataParams,
                               uint16_t wSizeOfDataParams,
                               void * pBalRegDataParams
                               )
{
    if (sizeof(phdlStepper_Robot_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_STEPPER);
    }
    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pBalRegDataParams);

    /* init private data */
    pDataParams->wId                    = PH_COMP_DL_STEPPER | PH_COMP_DL_STEPPER_ROBOT_ID;
    pDataParams->pBalRegDataParams      = pBalRegDataParams;
    pDataParams->dwMaxPosition          = PHDL_STEPPER_ROBOT_LIMIT_LOWER;
    pDataParams->bInit                  = PH_OFF;
    pDataParams->wMaxSpeed              = 50000;   /* 50mm/sec */
    pDataParams->wStepsWayRatio         = 16384;   /* 1 rotation (65536 units) is 4 mm */
    pDataParams->wMaxGetPositionRetrys  = 3;
    pDataParams->wWaitPositionTimeout   = 100;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t  phdlStepper_Robot_Initialize(
                                      phdlStepper_Robot_DataParams_t * pDataParams
                                      )
{
    phStatus_t PH_MEMLOC_REM status;

    PH_ASSERT_NULL (pDataParams);

    /* Reset Robot and wait until top position is reached */
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_Robot_Int_Reset(pDataParams));
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_Robot_Approval(pDataParams));
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_Int_Wait(1000));
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_Robot_Int_WaitMoveFinished(pDataParams,0,100));

    pDataParams->bInit = PH_ON;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Robot_MoveDistance(
                                       phdlStepper_Robot_DataParams_t * pDataParams,
                                       uint16_t wSpeed,
                                       uint8_t bDirection,
                                       uint32_t dwDistance,
                                       uint8_t bBlocking
                                       )
{
    phStatus_t PH_MEMLOC_REM status;
    int32_t    PH_MEMLOC_REM dwSteps;

    PH_ASSERT_NULL (pDataParams);

    /* Test if Stepper is Initialised. */
    if (!pDataParams->bInit)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
    }

    /* Convert Distance in mm to Steps */
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_Robot_Int_ConvertToStep(pDataParams, (int32_t)dwDistance, &dwSteps));

    /* Use Move_Steps to perform task. */
    return phdlStepper_Robot_MoveSteps(pDataParams, wSpeed, bDirection, (uint32_t)dwSteps, bBlocking);
}

phStatus_t phdlStepper_Robot_MoveSteps(
                                    phdlStepper_Robot_DataParams_t * pDataParams,
                                    uint16_t wSpeed,
                                    uint8_t bDirection,
                                    uint32_t dwSteps,
                                    uint8_t bBlocking
                                    )
{
    phStatus_t PH_MEMLOC_REM status;
    int32_t    PH_MEMLOC_REM dwActStep;
    int32_t    PH_MEMLOC_REM dwCalcEndPos;

    PH_ASSERT_NULL (pDataParams);
    if( bDirection != PHDL_STEPPER_DIR_DOWN && bDirection != PHDL_STEPPER_DIR_UP )
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
    }

    /* Test if Stepper is Initialised. */
    if (!pDataParams->bInit)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
    }

    PH_CHECK_SUCCESS_FCT(status, phdlStepper_Robot_Int_GetStep(pDataParams, &dwActStep));

    /* calculate the end position */
    dwCalcEndPos = dwActStep + (int32_t)(pow(-1.0f, (float32_t)bDirection) * (float32_t)dwSteps);

    return phdlStepper_Robot_Int_GoToStep(pDataParams, wSpeed, dwCalcEndPos, bBlocking);
}

phStatus_t phdlStepper_Robot_GoToPosition(
                                       phdlStepper_Robot_DataParams_t * pDataParams,
                                       uint16_t wSpeed,
                                       uint32_t dwPosition,
                                       uint8_t bBlocking
                                       )
{
    phStatus_t  PH_MEMLOC_REM status;
    int32_t     PH_MEMLOC_REM dwSteps;

    PH_ASSERT_NULL (pDataParams);

    /* Test if Stepper is Initialised. */
    if (!pDataParams->bInit)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
    }

    /* Convert Distance in mm to Steps */
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_Robot_Int_ConvertToStep(pDataParams,(int32_t)dwPosition,&dwSteps));

    return phdlStepper_Robot_Int_GoToStep(pDataParams, wSpeed, dwSteps, bBlocking);
}

phStatus_t phdlStepper_Robot_SetConfig(
                                    phdlStepper_Robot_DataParams_t * pDataParams,
                                    uint16_t wConfig,
                                    uint16_t wValue
                                    )
{
    PH_ASSERT_NULL (pDataParams);

    switch (wConfig)
    {
    case PHDL_STEPPER_CONFIG_MAX_SPEED:
        /* check value */
        if (wValue > PHDL_STEPPER_ROBOT_MAX_MAX_SPEED)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        pDataParams->wMaxSpeed = wValue;
        break;

    case PHDL_STEPPER_CONFIG_STEPS_WAY_RATIO:
        pDataParams->wStepsWayRatio = wValue;
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_DL_STEPPER);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Robot_SetConfig32(
                                      phdlStepper_Robot_DataParams_t * pDataParams,
                                      uint16_t wConfig,
                                      uint32_t dwValue
                                      )
{
    phStatus_t  PH_MEMLOC_REM statusTmp;

    PH_ASSERT_NULL (pDataParams);

    switch (wConfig)
    {
    case PHDL_STEPPER_CONFIG32_POSITION_COUNTER:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Robot_Int_GoToStep(pDataParams, pDataParams->wMaxSpeed/2, dwValue, PH_ON));
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_DL_STEPPER);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Robot_GetConfig(
                                    phdlStepper_Robot_DataParams_t * pDataParams,
                                    uint16_t wConfig,
                                    uint16_t * pValue
                                    )
{
    PH_ASSERT_NULL (pDataParams);

    switch(wConfig)
    {
    case PHDL_STEPPER_CONFIG_MAX_SPEED:
        *pValue = pDataParams->wMaxSpeed;
        break;

    case PHDL_STEPPER_CONFIG_STEPS_WAY_RATIO:
        *pValue = pDataParams->wStepsWayRatio;
        break;

        /* Unknown Identifier */
    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_DL_STEPPER);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_Robot_GetConfig32(
                                      phdlStepper_Robot_DataParams_t * pDataParams,
                                      uint16_t wConfig,
                                      uint32_t * pValue
                                      )
{
    phStatus_t  PH_MEMLOC_REM statusTmp;
    int32_t     PH_MEMLOC_REM dwTmp = 0;

    PH_ASSERT_NULL (pDataParams);

    switch(wConfig)
    {
    case PHDL_STEPPER_CONFIG32_POSITION_COUNTER:

        /* Construct frame to send */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_Robot_Int_GetStep(pDataParams, &dwTmp));
        if( dwTmp < 0 )
        {
            dwTmp = 0;
        }
        *pValue = (uint32_t)dwTmp;
        break;

    case PHDL_STEPPER_CONFIG32_MAX_POSITION:
        *pValue = (uint32_t) pDataParams->dwMaxPosition;
        break;

        /* Unknown Identifier */
    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_DL_STEPPER);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

#endif /* NXPBUILD__PHDL_STEPPER_ROBOT */
