/*
 * Copyright 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal Stepper Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHDLSTEPPER_HIGHZ_INT_H
#define PHDLSTEPPER_HIGHZ_INT_H

#include <ph_Status.h>

#if _MSC_VER <= 1600
#define DOUBLE
#else
#define DOUBLE	(double)
#endif

/**
* \brief Union for HighZ Status.
*/
typedef union
{
    struct
    {
        uint8_t drive_x  : 1; /**< These bits signal whether the robot is currently driving in X axis. */
        uint8_t drive_y  : 1; /**< These bits signal whether the robot is currently driving in Y axis. */
        uint8_t drive_z  : 1; /**< These bits signal whether the robot is currently driving in Z axis. */
        uint8_t drive_c  : 1; /**< These bits signal whether the robot is currently driving in C axis. */
        uint8_t end_z    : 1; /**< These bits signal whether a end switch of Z axis is set or not. */
        uint8_t end_y    : 1; /**< These bits signal whether a end switch of Y axis is set or not. */
        uint8_t end_x    : 1; /**< These bits signal whether a end switch of X axis is set or not. */
        uint8_t em_stop  : 1; /**< Signals whether the emergency stop is set or not. This bit is also 1 if the power supply is disabled using the POWER_OFF command. */
    } items;
    uint8_t data;
} phdlStepper_HighZ_HighZStatus_t;

#define PHDL_STEPPER_HIGHZ_INT_PREFIX_1             (uint8_t)'P'
#define PHDL_STEPPER_HIGHZ_INT_PREFIX_2             (uint8_t)'r'
#define PHDL_STEPPER_HIGHZ_INT_PREFIX_3             (uint8_t)'E'

#define PHDL_STEPPER_HIGHZ_INT_STANDBY              0x00
#define PHDL_STEPPER_HIGHZ_INT_DRIVE                0x01
#define PHDL_STEPPER_HIGHZ_INT_RECEIVE_SETTINGS     0x02
#define PHDL_STEPPER_HIGHZ_INT_GO_REFERENCE         0x03
#define PHDL_STEPPER_HIGHZ_INT_READ_POSITION        0x04
#define PHDL_STEPPER_HIGHZ_INT_WRITE_POSITION       0x05
#define PHDL_STEPPER_HIGHZ_INT_READ_STATUS          0x06
#define PHDL_STEPPER_HIGHZ_INT_DRIVE_CONTINUOUS     0x07
#define PHDL_STEPPER_HIGHZ_INT_POWER_ON             0x08
#define PHDL_STEPPER_HIGHZ_INT_POWER_OFF            0x09
#define PHDL_STEPPER_HIGHZ_INT_RELEASE              0x0A

#define PHDL_STEPPER_HIGHZ_INT_COORDINATE_X         0
#define PHDL_STEPPER_HIGHZ_INT_COORDINATE_Y         1
#define PHDL_STEPPER_HIGHZ_INT_COORDINATE_Z         2

extern uint8_t PHDL_STEPPER_HIGHZ_INT_COORDINATE_SEQUENCE[];

#define PHDL_STEPPER_HIGHZ_INT_FORWARD_X            true
#define PHDL_STEPPER_HIGHZ_INT_FORWARD_Y            false
#define PHDL_STEPPER_HIGHZ_INT_FORWARD_Z            true

#define PHDL_STEPPER_HIGHZ_INT_POWER_WAIT           500  /* ms seconds, a value lower than 400 will result in lost movement
                                                            when driving immediately after power_on */

#define PHDL_STEPPER_HIGHZ_INT_DRIVE_STEP_MAX       0xFFFF  /* drive message has two bytes per coordinate */
#define PHDL_STEPPER_HIGHZ_INT_DRIVE_TIMEOUT        60      /* seconds */

#define PHDL_STEPPER_HIGHZ_INT_GO_REFERENCE_TIMEOUT 120     /* seconds, large number since go reference may drive slowly for a long distance
                                                               as a method to handle errors during startup */

#define PHDL_STEPPER_HIGHZ_INT_RELEASE_RETRY_MAX    5
#define PHDL_STEPPER_HIGHZ_INT_RELEASE_WAIT_TIME_MAX    5000  /* milliseconds, process takes 1.5 sec */
#define PHDL_STEPPER_HIGHZ_INT_RELEASE_DISTANCE     1000  /* steps */

#define PHDL_STEPPER_HIGHZ_INT_CRC16_POLY           0x1021U


phStatus_t phdlStepper_HighZ_Int_ReadStatus(
                                        phdlStepper_HighZ_DataParams_t * pDataParams,
                                        phdlStepper_HighZ_HighZStatus_t *pbStatus);

phStatus_t phdlStepper_HighZ_Int_Exchange(
                                        phdlStepper_HighZ_DataParams_t * pDataParams,
                                        uint8_t * pCommand,
                                        uint16_t wTxLength,
                                        uint8_t *pResponse,
                                        uint16_t wExpectedLength,
                                        uint16_t wTimeoutMs);

phStatus_t phdlStepper_HighZ_Int_CreateMessage(
                                        uint8_t bCommand,
                                        uint8_t *pData,
                                        uint16_t wDataLength,
                                        uint8_t *pMessage,
                                        uint16_t wMessageSize,
                                        uint16_t *pwMessageLength);

int32_t phdlStepper_HighZ_Int_ConvertMicrometerToSteps(
                                        phdlStepper_HighZ_DataParams_t * pDataParams,
                                        int32_t dwMicrometer,
                                        uint8_t bAxis
                                        );

int32_t phdlStepper_HighZ_Int_ConvertStepsToMicrometer(
                                        phdlStepper_HighZ_DataParams_t * pDataParams,
                                        int32_t dwSteps,
                                        uint8_t bAxis
                                        );

phStatus_t phdlStepper_HighZ_Int_PowerOn(phdlStepper_HighZ_DataParams_t * pDataParams);

phStatus_t phdlStepper_HighZ_Int_PowerOff(phdlStepper_HighZ_DataParams_t * pDataParams);

phStatus_t phdlStepper_HighZ_Int_ReadPosition(
                                        phdlStepper_HighZ_DataParams_t * pDataParams,
                                        int32_t *pdwXPosAbs,
                                        int32_t *pdwYPosAbs,
                                        int32_t *pdwZPosAbs,
                                        int32_t *pdwXPosRel,
                                        int32_t *pdwYPosRel,
                                        int32_t *pdwZPosRel
                                        );

phStatus_t phdlStepper_HighZ_Int_ReadPositionAbs(
                                        phdlStepper_HighZ_DataParams_t * pDataParams,
                                        int32_t *pdwXPosAbs,
                                        int32_t *pdwYPosAbs,
                                        int32_t *pdwZPosAbs
                                        );

phStatus_t phdlStepper_HighZ_Int_ReadPositionRel(
                                        phdlStepper_HighZ_DataParams_t * pDataParams,
                                        int32_t *pdwXPosRel,
                                        int32_t *pdwYPosRel,
                                        int32_t *pdwZPosRel
                                        );

phStatus_t phdlStepper_HighZ_Int_WritePosition(
                                        phdlStepper_HighZ_DataParams_t * pDataParams,
                                        int32_t dwXPos,
                                        int32_t dwYPos,
                                        int32_t dwZPos
                                        );

phStatus_t phdlStepper_HighZ_Int_ReleaseFromEndpoints(
                                        phdlStepper_HighZ_DataParams_t * pDataParams
                                        );

phStatus_t phdlStepper_HighZ_Int_DriveToAbsolutePosition(
                                        phdlStepper_HighZ_DataParams_t * pDataParams,
                                        int32_t dwXPosAbs,
                                        int32_t dwYPosAbs,
                                        int32_t dwZPosAbs
                                        );

phStatus_t phdlStepper_HighZ_Int_DriveToRelativePosition(
                                        phdlStepper_HighZ_DataParams_t * pDataParams,
                                        int32_t dwX,
                                        int32_t dwY,
                                        int32_t dwZ
                                        );

phStatus_t phdlStepper_HighZ_Int_ExecuteDrive(
                                        phdlStepper_HighZ_DataParams_t * pDataParams,
                                        int32_t dwXDist,
                                        int32_t dwYDist,
                                        int32_t dwZDist
                                        );

phStatus_t phdlStepper_HighZ_Int_SendDriveSettings(
                                        phdlStepper_HighZ_DataParams_t * pDataParams,
                                        uint16_t wXStep,
                                        uint16_t wYStep,
                                        uint16_t wZStep,
                                        uint8_t bXDir,
                                        uint8_t bYDir,
                                        uint8_t bZDir
                                        );

phStatus_t phdlStepper_HighZ_Int_Drive(phdlStepper_HighZ_DataParams_t * pDataParams);

phStatus_t phdlStepper_HighZ_Int_WaitDriveEnd(phdlStepper_HighZ_DataParams_t * pDataParams);

phStatus_t phdlStepper_HighZ_Int_WaitEndpoint(phdlStepper_HighZ_DataParams_t * pDataParams,
                                        uint8_t bWaitX,
                                        uint8_t bWaitY,
                                        uint8_t bWaitZ,
                                        phdlStepper_HighZ_HighZStatus_t *pbStatus
                                        );

phStatus_t phdlStepper_HighZ_Int_WaitEndpointOnAllCoordinates(phdlStepper_HighZ_DataParams_t * pDataParams);

phStatus_t phdlStepper_HighZ_Int_GoReference(phdlStepper_HighZ_DataParams_t * pDataParams);

phStatus_t phdlStepper_HighZ_Int_Calibrate(phdlStepper_HighZ_DataParams_t * pDataParams);

phStatus_t phdlStepper_HighZ_Int_DriveToZeroPositionFast(phdlStepper_HighZ_DataParams_t * pDataParams);

phStatus_t phdlStepper_HighZ_Int_DriveToZeroPositionSafe(phdlStepper_HighZ_DataParams_t * pDataParams);

phStatus_t phdlStepper_HighZ_Int_DriveToZeroPositionAbsolute(phdlStepper_HighZ_DataParams_t * pDataParams);

phStatus_t phdlStepper_HighZ_Int_DriveToZeroPositionRelative(phdlStepper_HighZ_DataParams_t * pDataParams);

phStatus_t phdlStepper_HighZ_Int_CheckOutOfBounds(phdlStepper_HighZ_DataParams_t * pDataParams, uint8_t *bOutOfBounds);

phStatus_t phdlStepper_HighZ_Int_Read(phdlStepper_HighZ_DataParams_t * pDataParams,
                                        uint8_t *pResponse,
                                        uint16_t wExpectedLength,
                                        uint16_t wTimeoutMs);

phStatus_t phdlStepper_HighZ_Int_WaitRelease(phdlStepper_HighZ_DataParams_t * pDataParams);

phStatus_t phdlStepper_HighZ_Int_GetSpeed(phdlStepper_HighZ_DataParams_t * pDataParams,
                                        uint16_t *pwSpeed);

phStatus_t phdlStepper_HighZ_Int_SetSpeed(phdlStepper_HighZ_DataParams_t * pDataParams,
                                        uint16_t wSpeed);

#endif /* PHDLSTEPPER_HIGHZ_INT_H */
