/*
 * Copyright 2017, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Hardware Steppper DensoVS60 Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phbalReg.h>
#include <ph_RefDefs.h>
#include <math.h>

#ifdef NXPBUILD__PHDL_STEPPER_DENSO_VS60

#include <phdlStepper.h>
#include "phdlStepper_DensoVS60.h"
#include "phdlStepper_DensoVS60_Int.h"
#include "phdlStepper_DensoVS60_Cmd.h"
#include "../phdlStepper_Int.h"

phStatus_t phdlStepper_DensoVS60_Init(
                               phdlStepper_DensoVS60_DataParams_t * pDataParams,
                               uint16_t wSizeOfDataParams
                               )
{

    phStatus_t PH_MEMLOC_REM status;
    phdlStepper_DensoVS60_DataParams_XY_coord_t    structTargetXY;
    phdlStepper_DensoVS60_DataParams_Joint_coord_t structTargetJoint;

    if (sizeof(phdlStepper_DensoVS60_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_STEPPER);
    }
    PH_ASSERT_NULL (pDataParams);

    /* Initialize the structures in order to prevent warning notifications */
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_External_InitializeCoordsStructures(&structTargetXY, &structTargetJoint));

    /* init private data */
    pDataParams->wId                    = PH_COMP_DL_STEPPER | PH_COMP_DL_STEPPER_DENSO_VS60_ID;
    pDataParams->bInit                  = PH_OFF;
    pDataParams->wStepsWayRatio         = 400; /* Use the same as in Stepper V1 */
    pDataParams->bPositionMode          = PHDL_STEPPER_DENSO_VS60_POS_MODE_LOCAL;
    pDataParams->bPositionModeCorrect   = 0;
    pDataParams->bActiveAxis            = PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_Z;
    pDataParams->bActiveAxisType        = 0;
    pDataParams->bArmTaken              = 0;
    pDataParams->bActiveRotation        = 1;
    pDataParams->wRobotSpeed            = 100;
    pDataParams->bRobotAcceeration      = 100;
    pDataParams->structTargetXY         = structTargetXY;
    pDataParams->structTargetJoint      = structTargetJoint;
    strcpy((char *)pDataParams->pPortName, "192.168.106.208");

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t  phdlStepper_DensoVS60_Initialize(
                                      phdlStepper_DensoVS60_DataParams_t * pDataParams
                                      )
{
    phStatus_t PH_MEMLOC_REM status;

    PH_ASSERT_NULL (pDataParams);

    /* If the stepper is already initialized */
    if( pDataParams->bInit == PH_ON)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
    }

    /* Initialize robot arm */
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_External_CompleteInternalInitialization(pDataParams));

    /* Start the robot's motor */
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_External_StartMotor(pDataParams));

    /* Set default internal speed to 10 */
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_External_SetRobotSpeedInternal(pDataParams, pDataParams->wRobotSpeed));

    /* Set default external speed to 100% */
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_External_SetRobotSpeedExternal(pDataParams,100));

    /* Set default internal acceleration of the robot to the maximum(100) */
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_External_SetRobotAcceleration(pDataParams,(float)(pDataParams->bRobotAcceeration)));

    /* Set mode to remote ->robot contol goes through software */
    pDataParams->bPositionMode = PHDL_STEPPER_DENSO_VS60_POS_MODE_REMOTE;

    /* Compete initialization */
    pDataParams->bInit = PH_ON;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_MoveDistance(
                                       phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                       uint16_t wSpeed,
                                       uint8_t bDirection,
                                       uint32_t dwDistance,
                                       uint8_t bBlocking
                                       )
{

    phStatus_t PH_MEMLOC_REM status;
    int32_t    PH_MEMLOC_REM dwSignedDistance;
    int32_t    PH_MEMLOC_REM dwActPosLimitLow;
    int32_t    PH_MEMLOC_REM dwActPosLimitHigh;


    /* Initialize stuctures for both axis types */
    phdlStepper_DensoVS60_DataParams_XY_coord_t    pStructCurrentXY;
    phdlStepper_DensoVS60_DataParams_Joint_coord_t pStructCurrentJoint;

    /* Set the current coodinates for both axis types */
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_Int_GetPositionXY(pDataParams,  &pStructCurrentXY));
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_Int_GetPositionJoint(pDataParams,  &pStructCurrentJoint));

    PH_ASSERT_NULL (pDataParams);
    if( bDirection != PHDL_STEPPER_DIR_DOWN && bDirection != PHDL_STEPPER_DIR_UP )
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
    }

    /* Test if Stepper is Initialised. */
    if (!pDataParams->bInit)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
    }

    /* If the position is set local on the robot this is not allowed */
    if (pDataParams->bPositionMode == PHDL_STEPPER_DENSO_VS60_POS_MODE_LOCAL)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
    }

    /* limit the distance to the max available number / 2 */
    if (dwDistance > 0x7FFFFFFF)
    {
        return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS, PH_COMP_DL_STEPPER);
    }

    if (bDirection == PHDL_STEPPER_DIR_DOWN)
    {
        dwSignedDistance = (int32_t)(0-dwDistance);
        dwActPosLimitLow = -0x7FFFFFFF - dwSignedDistance;
        dwActPosLimitHigh = 0x7FFFFFFF;
    }

    else
    {
        dwSignedDistance = dwDistance;
        dwActPosLimitLow = -0x7FFFFFFF;
        dwActPosLimitHigh = 0x7FFFFFFF - dwSignedDistance;
    }

    /* Set internal speed of the robot arm */
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_External_SetRobotSpeedInternal(pDataParams, wSpeed));

    switch (pDataParams->bActiveAxis)
    {
    case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_X:
        /* calculate the end position */
        if (pStructCurrentXY.dwPositionX < dwActPosLimitLow || pStructCurrentXY.dwPositionX > dwActPosLimitHigh)
        {
            return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS, PH_COMP_DL_STEPPER);
        }
        pDataParams->structTargetXY.dwPositionX += dwSignedDistance;
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_Int_GoToParameterXY(pDataParams,bBlocking));
        break;
    case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_Y:
        /* calculate the end position */
        if (pStructCurrentXY.dwPositionY < dwActPosLimitLow || pStructCurrentXY.dwPositionY > dwActPosLimitHigh)
        {
            return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS, PH_COMP_DL_STEPPER);
        }
        pDataParams->structTargetXY.dwPositionY += dwSignedDistance;
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_Int_GoToParameterXY(pDataParams, bBlocking));
        break;
    case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_Z:
        /* calculate the end position */
        if (pStructCurrentXY.dwPositionZ < dwActPosLimitLow || pStructCurrentXY.dwPositionZ > dwActPosLimitHigh)
        {
            return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS, PH_COMP_DL_STEPPER);
        }
        pDataParams->structTargetXY.dwPositionZ += dwSignedDistance;
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_Int_GoToParameterXY(pDataParams,  bBlocking));
        break;
    case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_RX:
        /* calculate the end position */
        if (pStructCurrentXY.dwPositionRx < dwActPosLimitLow || pStructCurrentXY.dwPositionRx > dwActPosLimitHigh)
        {
            return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS, PH_COMP_DL_STEPPER);
        }
        pDataParams->structTargetXY.dwPositionRx += dwSignedDistance;
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_Int_GoToParameterXY(pDataParams,  bBlocking));
        break;
    case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_RY:
        /* calculate the end position */
        if (pStructCurrentXY.dwPositionRy < dwActPosLimitLow || pStructCurrentXY.dwPositionRy > dwActPosLimitHigh)
        {
            return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS, PH_COMP_DL_STEPPER);
        }
        pDataParams->structTargetXY.dwPositionRy += dwSignedDistance;
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_Int_GoToParameterXY(pDataParams,  bBlocking));
        break;
    case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_RZ:
        /* calculate the end position */
        if (pStructCurrentXY.dwPositionRz < dwActPosLimitLow || pStructCurrentXY.dwPositionRz > dwActPosLimitHigh)
        {
            return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS, PH_COMP_DL_STEPPER);
        }
        pDataParams->structTargetXY.dwPositionRz += dwSignedDistance;
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_Int_GoToParameterXY(pDataParams,  bBlocking));
        break;
    case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_J1:
        /* calculate the end position */
        if (pStructCurrentJoint.dwPositionJ1 < dwActPosLimitLow || pStructCurrentJoint.dwPositionJ1 > dwActPosLimitHigh)
        {
            return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS, PH_COMP_DL_STEPPER);
        }
        pDataParams->structTargetJoint.dwPositionJ1 += dwSignedDistance;
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_Int_GoToParameterJoint(pDataParams, bBlocking));
        break;
    case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_J2:
        /* calculate the end position */
        if (pStructCurrentJoint.dwPositionJ2 < dwActPosLimitLow || pStructCurrentJoint.dwPositionJ2 > dwActPosLimitHigh)
        {
            return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS, PH_COMP_DL_STEPPER);
        }
        pDataParams->structTargetJoint.dwPositionJ2 += dwSignedDistance;
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_Int_GoToParameterJoint(pDataParams, bBlocking));
        break;
    case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_J3:
        /* calculate the end position */
        if (pStructCurrentJoint.dwPositionJ3 < dwActPosLimitLow || pStructCurrentJoint.dwPositionJ3 > dwActPosLimitHigh)
        {
            return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS, PH_COMP_DL_STEPPER);
        }
        pDataParams->structTargetJoint.dwPositionJ3 += dwSignedDistance;
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_Int_GoToParameterJoint(pDataParams, bBlocking));
        break;
    case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_J4:
        /* calculate the end position */
        if (pStructCurrentJoint.dwPositionJ4 < dwActPosLimitLow || pStructCurrentJoint.dwPositionJ4 > dwActPosLimitHigh)
        {
            return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS, PH_COMP_DL_STEPPER);
        }
        pDataParams->structTargetJoint.dwPositionJ4 += dwSignedDistance;
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_Int_GoToParameterJoint(pDataParams, bBlocking));
        break;
    case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_J5:
        /* calculate the end position */
        if (pStructCurrentJoint.dwPositionJ5 < dwActPosLimitLow || pStructCurrentJoint.dwPositionJ5 > dwActPosLimitHigh)
        {
            return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS, PH_COMP_DL_STEPPER);
        }
        pDataParams->structTargetJoint.dwPositionJ5 += dwSignedDistance;
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_Int_GoToParameterJoint(pDataParams, bBlocking));
        break;
    case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_J6:
        /* calculate the end position */
        if (pStructCurrentJoint.dwPositionJ6 < dwActPosLimitLow || pStructCurrentJoint.dwPositionJ6 > dwActPosLimitHigh)
        {
            return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS, PH_COMP_DL_STEPPER);
        }
        pDataParams->structTargetJoint.dwPositionJ6 += dwSignedDistance;
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_Int_GoToParameterJoint(pDataParams, bBlocking));
        break;
    case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_J7:
        /* calculate the end position */
        if (pStructCurrentJoint.dwPositionJ7 < dwActPosLimitLow || pStructCurrentJoint.dwPositionJ7 > dwActPosLimitHigh)
        {
            return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS, PH_COMP_DL_STEPPER);
        }
        pDataParams->structTargetJoint.dwPositionJ7 += dwSignedDistance;
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_Int_GoToParameterJoint(pDataParams, bBlocking));
        break;
    case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_J8:
        /* calculate the end position */
        if (pStructCurrentJoint.dwPositionJ8 < dwActPosLimitLow || pStructCurrentJoint.dwPositionJ8 > dwActPosLimitHigh)
        {
            return PH_ADD_COMPCODE(PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS, PH_COMP_DL_STEPPER);
        }
        pDataParams->structTargetJoint.dwPositionJ8 += dwSignedDistance;
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_Int_GoToParameterJoint(pDataParams, bBlocking));
        break;
    default:
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_STEPPER);
    }

    return PH_ADD_COMPCODE(status, PH_COMP_DL_STEPPER);
}


phStatus_t phdlStepper_DensoVS60_MoveSteps(
                                           phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                           uint16_t wSpeed,
                                           uint8_t bDirection,
                                           uint32_t dwSteps,
                                           uint8_t bBlocking
                                          )
{

    phStatus_t PH_MEMLOC_REM status;
    int32_t    PH_MEMLOC_REM dwDistance;

    PH_ASSERT_NULL (pDataParams);

    /* Test if Stepper is Initialised. */
    if (!pDataParams->bInit)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
    }

    PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_Int_ConvertFromStep(pDataParams, dwSteps, &dwDistance));

    /* Use MoveDistance to perform task. */
    return phdlStepper_DensoVS60_MoveDistance(pDataParams, wSpeed, bDirection, dwDistance, bBlocking);
}


phStatus_t phdlStepper_DensoVS60_GoToPosition(
                                       phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                       uint16_t wSpeed,
                                       uint32_t dwPosition,
                                       uint8_t bBlocking
                                       )
{
    /* Function variables */
    phStatus_t PH_MEMLOC_REM status;
    int32_t dwSignedPosition;

    PH_ASSERT_NULL (pDataParams);

    /* Test if Stepper is Initialised. */
    if (!pDataParams->bInit)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
    }

    /* If the position is set local on the robot this is not allowed */
    if (pDataParams->bPositionMode == PHDL_STEPPER_DENSO_VS60_POS_MODE_LOCAL)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
    }

    /* Set internal speed */
    PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_External_SetRobotSpeedInternal(pDataParams, wSpeed));

    /* Cast the value of the position */
    dwSignedPosition = (int32_t)dwPosition;

    switch (pDataParams->bActiveAxis)
    {
    case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_X:
        pDataParams->structTargetXY.dwPositionX = dwSignedPosition;
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_Int_GoToParameterXY(pDataParams, bBlocking));
        break;
    case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_Y:
        pDataParams->structTargetXY.dwPositionY = dwSignedPosition;
         PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_Int_GoToParameterXY(pDataParams, bBlocking));
        break;
    case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_Z:
        pDataParams->structTargetXY.dwPositionZ = dwSignedPosition;
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_Int_GoToParameterXY(pDataParams, bBlocking));
        break;
    case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_RX:
        pDataParams->structTargetXY.dwPositionRx = dwSignedPosition;
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_Int_GoToParameterXY(pDataParams, bBlocking));
        break;
    case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_RY:
        pDataParams->structTargetXY.dwPositionRy = dwSignedPosition;
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_Int_GoToParameterXY(pDataParams, bBlocking));
        break;
    case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_RZ:
        pDataParams->structTargetXY.dwPositionRz = dwSignedPosition;
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_Int_GoToParameterXY(pDataParams, bBlocking));
        break;
    case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_FIG:
        pDataParams->structTargetXY.wFigure = (int16_t)dwSignedPosition;
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_Int_GoToParameterXY(pDataParams, bBlocking));
        break;
    case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_J1:
        pDataParams->structTargetJoint.dwPositionJ1 = dwSignedPosition;
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_Int_GoToParameterJoint(pDataParams, bBlocking));
        break;
    case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_J2:
        pDataParams->structTargetJoint.dwPositionJ2 = dwSignedPosition;
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_Int_GoToParameterJoint(pDataParams, bBlocking));
        break;
    case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_J3:
        pDataParams->structTargetJoint.dwPositionJ3 = dwSignedPosition;
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_Int_GoToParameterJoint(pDataParams, bBlocking));
        break;
    case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_J4:
        pDataParams->structTargetJoint.dwPositionJ4 = dwSignedPosition;
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_Int_GoToParameterJoint(pDataParams, bBlocking));
        break;
    case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_J5:
        pDataParams->structTargetJoint.dwPositionJ5 = dwSignedPosition;
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_Int_GoToParameterJoint(pDataParams, bBlocking));
        break;
    case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_J6:
        pDataParams->structTargetJoint.dwPositionJ6 = dwSignedPosition;
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_Int_GoToParameterJoint(pDataParams, bBlocking));
        break;
    case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_J7:
        pDataParams->structTargetJoint.dwPositionJ7 = dwSignedPosition;
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_Int_GoToParameterJoint(pDataParams, bBlocking));
        break;
    case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_J8:
        pDataParams->structTargetJoint.dwPositionJ8 = dwSignedPosition;
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_Int_GoToParameterJoint(pDataParams, bBlocking));
        break;
    default:
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_STEPPER);
    }

    return PH_ADD_COMPCODE(status, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_SetConfig(
                                    phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                    uint16_t wConfig,
                                    uint16_t wValue
                                    )
{
    phStatus_t  PH_MEMLOC_REM status;

    /* When starting for the first time*/
    if(pDataParams->bArmTaken == 0)
    {
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_Initialize(pDataParams));
        pDataParams->bArmTaken = 1;
    }

    PH_ASSERT_NULL (pDataParams);

    switch (wConfig)
    {
    case PHDL_STEPPER_CONFIG_STEPS_WAY_RATIO:
        pDataParams->wStepsWayRatio = wValue;
        break;

    case PHDL_STEPPER_DENSO_VS60_CONFIG_POSITION_MODE:
        switch (wValue)
        {
        /* First case clear offset for both axis types */
        case PHDL_STEPPER_DENSO_VS60_POS_MODE_SET_CLEAR_ZERO_OFFSET:
            PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_External_ResetOffset(pDataParams,0));
            PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_External_SetCurrentXYPositionWithOffset(pDataParams, &pDataParams->structTargetXY));
            PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_External_ResetOffset(pDataParams,1));
            PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_External_SetCurrentJointPositionWithOffset(pDataParams,&pDataParams->structTargetJoint));
            break;
        case PHDL_STEPPER_DENSO_VS60_POS_MODE_SET_CURRENT_AS_ZERO:
            PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_Int_SetAsZero(pDataParams));
            break;
        case PHDL_STEPPER_DENSO_VS60_POS_MODE_REMOTE:
            /* Request Robot, Start Motor */
            if (pDataParams->bPositionMode != PHDL_STEPPER_DENSO_VS60_POS_MODE_REMOTE)
            {
                PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_Initialize(pDataParams));
            }
            pDataParams->bPositionMode = PHDL_STEPPER_DENSO_VS60_POS_MODE_REMOTE;
            break;
        case PHDL_STEPPER_DENSO_VS60_POS_MODE_LOCAL:
            /* Stop Motor, Release Robot */
            if (pDataParams->bPositionMode != PHDL_STEPPER_DENSO_VS60_POS_MODE_LOCAL)
            {
                PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_Int_ManualMode(pDataParams));
            }
            /* Set the control of the robot back to the controller */
            pDataParams->bPositionMode = PHDL_STEPPER_DENSO_VS60_POS_MODE_LOCAL;
            break;
        case PHDL_STEPPER_DENSO_VS60_POS_MODE_COLLECT:
            /* If we start this mode we get the current position because there may be a non blocking move before */
            if (pDataParams->bPositionMode != PHDL_STEPPER_DENSO_VS60_POS_MODE_COLLECT)
            {
                pDataParams->bPositionMode = PHDL_STEPPER_DENSO_VS60_POS_MODE_COLLECT;
            }
            break;
        case PHDL_STEPPER_DENSO_VS60_POS_MODE_START:
            if(pDataParams->bPositionModeCorrect == 1)
            {
                if(pDataParams->bActiveAxisType == 0)
                {
                    PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_External_SetCurrentCoordsAsOffset(pDataParams,0));
                    PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_External_ResetCurrentCooridnates( pDataParams,0));
                }
                else
                {
                    PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_External_SetCurrentCoordsAsOffset(pDataParams,1));
                    PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_External_ResetCurrentCooridnates( pDataParams,1));
                }

              pDataParams->bPositionModeCorrect = 0;
              return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
            }
            status = phdlStepper_DensoVS60_Int_GoToXYAxis(pDataParams, PH_ON);
            if (((status & PH_ERR_MASK) == PH_ERR_SUCCESS) || ((status & PH_ERR_MASK) == PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS))
            {
                pDataParams->bPositionMode = PHDL_STEPPER_DENSO_VS60_POS_MODE_REMOTE;
            }
            PH_CHECK_SUCCESS(status);
            break;
        case PHDL_STEPPER_DENSO_VS60_POS_MODE_START_NONBLOCKING:
            if(pDataParams->bPositionModeCorrect == 1)
            {
                if(pDataParams->bActiveAxisType == 0)
                {
                    PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_External_SetCurrentCoordsAsOffset(pDataParams,0));
                    PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_External_ResetCurrentCooridnates( pDataParams,0));
                }
                else
                {
                    PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_External_SetCurrentCoordsAsOffset(pDataParams,1));
                    PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_External_ResetCurrentCooridnates( pDataParams,1));
                }

              pDataParams->bPositionModeCorrect = 0;
              return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
            }
            status = phdlStepper_DensoVS60_Int_GoToXYAxis(pDataParams, PH_OFF);
            if (((status & PH_ERR_MASK) == PH_ERR_SUCCESS) || ((status & PH_ERR_MASK) == PHDL_STEPPER_ERR_POS_OUT_OF_LIMITS))
            {
                pDataParams->bPositionMode = PHDL_STEPPER_DENSO_VS60_POS_MODE_REMOTE;
            }
            PH_CHECK_SUCCESS(status);
            break;
        case PHDL_STEPPER_DENSO_VS60_POS_MODE_HALT_MOVE:
            /* If we start this mode we get the current position because there may be a non blocking move before */
            if (pDataParams->bPositionMode != PHDL_STEPPER_DENSO_VS60_POS_MODE_COLLECT)
            {
                PH_CHECK_SUCCESS_FCT(status,phdlStepper_DensoVS60_External_ExecuteHaltMove(pDataParams));
            }
            break;
        default:
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        break;

    case PHDL_STEPPER_DENSO_VS60_CONFIG_ACTIVE_AXIS:
        if( wValue != PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_X &&
            wValue != PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_Y &&
            wValue != PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_Z &&
            wValue != PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_RX &&
            wValue != PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_RY &&
            wValue != PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_RZ &&
            wValue != PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_J1 &&
            wValue != PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_J2 &&
            wValue != PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_J3 &&
            wValue != PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_J4 &&
            wValue != PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_J5 &&
            wValue != PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_J6 &&
            wValue != PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_J7 &&
            wValue != PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_J8)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        pDataParams->bActiveAxis = (uint8_t)wValue;
        break;

    case PHDL_STEPPER_DENSO_VS60_CONFIG_ACTIVATE_LASER:
        if( wValue != PH_ON &&
            wValue != PH_OFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_External_SetIODeviceState(pDataParams,26,(wValue == PH_ON) ? 1 : 0));
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG_ACTIVATE_PUMP:
        if( wValue != PH_ON &&
            wValue != PH_OFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_External_SetIODeviceState(pDataParams,25,(wValue == PH_ON) ? 1 : 0));
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG_CHECK_HEAD_SENSOR:
        if( wValue != PH_ON &&
            wValue != PH_OFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        pDataParams->bActiveRotation = (uint8_t)wValue;
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG_ACTIVE_ROTATION:
        if( wValue != PH_ON &&
            wValue != PH_OFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_External_SetIODeviceState(pDataParams,10,(wValue == PH_ON) ? 1 : 0));
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG_ROBOT_SPEED:
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_External_SetRobotSpeedInternal(pDataParams, wValue));
        break;
    case  PHDL_STEPPER_DENSO_VS60_CONFIG_ROBOT_ACCELERATION:
        PH_CHECK_SUCCESS_FCT(status, phdlStepper_DensoVS60_External_SetRobotAcceleration(pDataParams, wValue));
        break;

    case PHDL_STEPPER_DENSO_VS60_CONFIG_CORRECT_COLLECT:
        if( wValue != PH_ON &&
            wValue != PH_OFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_STEPPER);
        }
        pDataParams->bPositionModeCorrect = (uint8_t)wValue;
        break;
    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_DL_STEPPER);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_SetConfig32(
                                             phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                             uint16_t wConfig,
                                             uint32_t dwValue
                                            )
{
    /* Function variables */
    phStatus_t  PH_MEMLOC_REM statusTmp;
    int32_t dwSignedValue;

    PH_ASSERT_NULL (pDataParams);

    /* Convert the value to signed integer */
    dwSignedValue = (int32_t)dwValue;

    switch (wConfig)
    {
    case PHDL_STEPPER_CONFIG32_POSITION_COUNTER: /* POSITION_COUNTER uses steps so convert */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_Int_ConvertFromStep(pDataParams, dwSignedValue, &dwSignedValue));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_GoToPosition(pDataParams, 10, (uint32_t)dwSignedValue, PH_ON));
        break;

    case PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_X:
        /* If the position is set local on the robot this is not allowed */
        if (pDataParams->bPositionMode == PHDL_STEPPER_DENSO_VS60_POS_MODE_LOCAL || pDataParams->bActiveAxisType != 0)
        {
            pDataParams->bPositionModeCorrect = 1;
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
        }
        pDataParams->bActiveAxisType = 0;
        pDataParams->structTargetXY.dwPositionX = dwSignedValue;
        if (pDataParams->bPositionMode != PHDL_STEPPER_DENSO_VS60_POS_MODE_COLLECT)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_Int_GoToXYAxis(pDataParams, PH_ON));
        }
        break;

    case PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_Y:
        /* If the position is set local on the robot this is not allowed */
        if (pDataParams->bPositionMode == PHDL_STEPPER_DENSO_VS60_POS_MODE_LOCAL || pDataParams->bActiveAxisType != 0)
        {
            pDataParams->bPositionModeCorrect = 1;
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
        }
        pDataParams->bActiveAxisType = 0;
        pDataParams->structTargetXY.dwPositionY = dwSignedValue;
        if (pDataParams->bPositionMode != PHDL_STEPPER_DENSO_VS60_POS_MODE_COLLECT)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_Int_GoToXYAxis(pDataParams, PH_ON));
        }
        break;

    case PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_Z:
        /* If the position is set local on the robot this is not allowed */
        if (pDataParams->bPositionMode == PHDL_STEPPER_DENSO_VS60_POS_MODE_LOCAL || pDataParams->bActiveAxisType != 0)
        {
            pDataParams->bPositionModeCorrect = 1;
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
        }
        pDataParams->bActiveAxisType = 0;
        pDataParams->structTargetXY.dwPositionZ = dwSignedValue;
        if (pDataParams->bPositionMode != PHDL_STEPPER_DENSO_VS60_POS_MODE_COLLECT)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_Int_GoToXYAxis(pDataParams, PH_ON));
        }
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_RX:
        /* If the position is set local on the robot this is not allowed */
        if (pDataParams->bPositionMode == PHDL_STEPPER_DENSO_VS60_POS_MODE_LOCAL || pDataParams->bActiveAxisType != 0)
        {
            pDataParams->bPositionModeCorrect = 1;
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
        }
        pDataParams->bActiveAxisType = 0;
        pDataParams->structTargetXY.dwPositionRx = dwSignedValue;
        if (pDataParams->bPositionMode != PHDL_STEPPER_DENSO_VS60_POS_MODE_COLLECT)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_Int_GoToXYAxis(pDataParams, PH_ON));
        }
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_RY:
        /* If the position is set local on the robot this is not allowed */
        if (pDataParams->bPositionMode == PHDL_STEPPER_DENSO_VS60_POS_MODE_LOCAL || pDataParams->bActiveAxisType != 0)
        {
            pDataParams->bPositionModeCorrect = 1;
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
        }
        pDataParams->bActiveAxisType = 0;
        pDataParams->structTargetXY.dwPositionRy = dwSignedValue;
        if (pDataParams->bPositionMode != PHDL_STEPPER_DENSO_VS60_POS_MODE_COLLECT)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_Int_GoToXYAxis(pDataParams, PH_ON));
        }
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_RZ:
        /* If the position is set local on the robot this is not allowed */
        if (pDataParams->bPositionMode == PHDL_STEPPER_DENSO_VS60_POS_MODE_LOCAL || pDataParams->bActiveAxisType != 0)
        {
            pDataParams->bPositionModeCorrect = 1;
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
        }
        pDataParams->bActiveAxisType = 0;
        pDataParams->structTargetXY.dwPositionRz = dwSignedValue;
        if (pDataParams->bPositionMode != PHDL_STEPPER_DENSO_VS60_POS_MODE_COLLECT)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_Int_GoToXYAxis(pDataParams, PH_ON));
        }
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_FIG:
        /* If the position is set local on the robot this is not allowed */
        if (pDataParams->bPositionMode == PHDL_STEPPER_DENSO_VS60_POS_MODE_LOCAL || pDataParams->bActiveAxisType != 0)
        {
            pDataParams->bPositionModeCorrect = 1;
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
        }
        pDataParams->bActiveAxisType = 0;
        pDataParams->structTargetXY.wFigure = (int16_t)dwSignedValue;
        if (pDataParams->bPositionMode != PHDL_STEPPER_DENSO_VS60_POS_MODE_COLLECT)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_Int_GoToXYAxis(pDataParams, PH_ON));
        }
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_J1:
        /* If the position is set local on the robot this is not allowed */
        if (pDataParams->bPositionMode == PHDL_STEPPER_DENSO_VS60_POS_MODE_LOCAL || pDataParams->bActiveAxisType != 1)
        {
            pDataParams->bPositionModeCorrect = 1;
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
        }
        pDataParams->bActiveAxisType = 1;
        pDataParams->structTargetJoint.dwPositionJ1 = dwSignedValue;
        if (pDataParams->bPositionMode != PHDL_STEPPER_DENSO_VS60_POS_MODE_COLLECT)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_Int_GoToJointAxis(pDataParams, PH_ON));
        }
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_J2:
        /* If the position is set local on the robot this is not allowed */
        if (pDataParams->bPositionMode == PHDL_STEPPER_DENSO_VS60_POS_MODE_LOCAL ||  pDataParams->bActiveAxisType != 1)
        {
            pDataParams->bPositionModeCorrect = 1;
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
        }
        pDataParams->bActiveAxisType = 1;
        pDataParams->structTargetJoint.dwPositionJ2 = dwSignedValue;
        if (pDataParams->bPositionMode != PHDL_STEPPER_DENSO_VS60_POS_MODE_COLLECT)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_Int_GoToJointAxis(pDataParams, PH_ON));
        }
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_J3:
        /* If the position is set local on the robot this is not allowed */
        if (pDataParams->bPositionMode == PHDL_STEPPER_DENSO_VS60_POS_MODE_LOCAL ||  pDataParams->bActiveAxisType != 1)
        {
            pDataParams->bPositionModeCorrect = 1;
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
        }
        pDataParams->bActiveAxisType = 1;
        pDataParams->structTargetJoint.dwPositionJ3 = dwSignedValue;
        if (pDataParams->bPositionMode != PHDL_STEPPER_DENSO_VS60_POS_MODE_COLLECT)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_Int_GoToJointAxis(pDataParams, PH_ON));
        }
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_J4:
        /* If the position is set local on the robot this is not allowed */
        if (pDataParams->bPositionMode == PHDL_STEPPER_DENSO_VS60_POS_MODE_LOCAL ||  pDataParams->bActiveAxisType != 1)
        {
            pDataParams->bPositionModeCorrect = 1;
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
        }
        pDataParams->bActiveAxisType = 1;
        pDataParams->structTargetJoint.dwPositionJ4 = dwSignedValue;
        if (pDataParams->bPositionMode != PHDL_STEPPER_DENSO_VS60_POS_MODE_COLLECT)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_Int_GoToJointAxis(pDataParams, PH_ON));
        }
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_J5:
        /* If the position is set local on the robot this is not allowed */
        if (pDataParams->bPositionMode == PHDL_STEPPER_DENSO_VS60_POS_MODE_LOCAL ||  pDataParams->bActiveAxisType != 1)
        {
            pDataParams->bPositionModeCorrect = 1;
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
        }
        pDataParams->bActiveAxisType = 1;
        pDataParams->structTargetJoint.dwPositionJ5 = dwSignedValue;
        if (pDataParams->bPositionMode != PHDL_STEPPER_DENSO_VS60_POS_MODE_COLLECT)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_Int_GoToJointAxis(pDataParams, PH_ON));
        }
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_J6:
        /* If the position is set local on the robot this is not allowed */
        if (pDataParams->bPositionMode == PHDL_STEPPER_DENSO_VS60_POS_MODE_LOCAL ||  pDataParams->bActiveAxisType != 1)
        {
            pDataParams->bPositionModeCorrect = 1;
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
        }
        pDataParams->bActiveAxisType = 1;
        pDataParams->structTargetJoint.dwPositionJ6 = dwSignedValue;
        if (pDataParams->bPositionMode != PHDL_STEPPER_DENSO_VS60_POS_MODE_COLLECT)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_Int_GoToJointAxis(pDataParams, PH_ON));
        }
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_J7:
        /* If the position is set local on the robot this is not allowed */
        if (pDataParams->bPositionMode == PHDL_STEPPER_DENSO_VS60_POS_MODE_LOCAL ||  pDataParams->bActiveAxisType != 1)
        {
            pDataParams->bPositionModeCorrect = 1;
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
        }
        pDataParams->bActiveAxisType = 1;
        pDataParams->structTargetJoint.dwPositionJ7 = dwSignedValue;
        if (pDataParams->bPositionMode != PHDL_STEPPER_DENSO_VS60_POS_MODE_COLLECT)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_Int_GoToJointAxis(pDataParams, PH_ON));
        }
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_J8:
        /* If the position is set local on the robot this is not allowed */
        if (pDataParams->bPositionMode == PHDL_STEPPER_DENSO_VS60_POS_MODE_LOCAL ||  pDataParams->bActiveAxisType != 1)
        {
            pDataParams->bPositionModeCorrect = 1;
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
        }
        pDataParams->bActiveAxisType = 1;
        pDataParams->structTargetJoint.dwPositionJ8 = dwSignedValue;
        if (pDataParams->bPositionMode != PHDL_STEPPER_DENSO_VS60_POS_MODE_COLLECT)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_Int_GoToJointAxis(pDataParams, PH_ON));
        }
        break;

    case PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_X:
        /* If the position is set local on the robot this is not allowed */
        if (pDataParams->bPositionMode == PHDL_STEPPER_DENSO_VS60_POS_MODE_LOCAL)
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
        }
        pDataParams->structTargetXY.offsetCoords.dwPositionX = ((double)dwSignedValue)/1000;
        if (pDataParams->bPositionMode != PHDL_STEPPER_DENSO_VS60_POS_MODE_COLLECT)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_Int_GoToXYAxis(pDataParams, PH_ON));
        }
        break;

    case PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_Y:
        /* If the position is set local on the robot this is not allowed */
        if (pDataParams->bPositionMode == PHDL_STEPPER_DENSO_VS60_POS_MODE_LOCAL)
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
        }
        pDataParams->structTargetXY.offsetCoords.dwPositionY = ((double)dwSignedValue)/1000;
        if (pDataParams->bPositionMode != PHDL_STEPPER_DENSO_VS60_POS_MODE_COLLECT)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_Int_GoToXYAxis(pDataParams, PH_ON));
        }
        break;

    case PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_Z:
        /* If the position is set local on the robot this is not allowed */
        if (pDataParams->bPositionMode == PHDL_STEPPER_DENSO_VS60_POS_MODE_LOCAL)
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
        }
        pDataParams->structTargetXY.offsetCoords.dwPositionZ = ((double)dwSignedValue)/1000;
        if (pDataParams->bPositionMode != PHDL_STEPPER_DENSO_VS60_POS_MODE_COLLECT)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_Int_GoToXYAxis(pDataParams, PH_ON));
        }
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_RX:
        /* If the position is set local on the robot this is not allowed */
        if (pDataParams->bPositionMode == PHDL_STEPPER_DENSO_VS60_POS_MODE_LOCAL)
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
        }
        pDataParams->structTargetXY.offsetCoords.dwPositionRx = ((double)dwSignedValue)/10;
        if (pDataParams->bPositionMode != PHDL_STEPPER_DENSO_VS60_POS_MODE_COLLECT)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_Int_GoToXYAxis(pDataParams, PH_ON));
        }
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_RY:
        /* If the position is set local on the robot this is not allowed */
        if (pDataParams->bPositionMode == PHDL_STEPPER_DENSO_VS60_POS_MODE_LOCAL)
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
        }
        pDataParams->structTargetXY.offsetCoords.dwPositionRy = ((double)dwSignedValue)/10;
        if (pDataParams->bPositionMode != PHDL_STEPPER_DENSO_VS60_POS_MODE_COLLECT)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_Int_GoToXYAxis(pDataParams, PH_ON));
        }
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_RZ:
        /* If the position is set local on the robot this is not allowed */
        if (pDataParams->bPositionMode == PHDL_STEPPER_DENSO_VS60_POS_MODE_LOCAL)
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
        }
        pDataParams->structTargetXY.offsetCoords.dwPositionRz = ((double)dwSignedValue)/10;
        if (pDataParams->bPositionMode != PHDL_STEPPER_DENSO_VS60_POS_MODE_COLLECT)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_Int_GoToXYAxis(pDataParams, PH_ON));
        }
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_FIG:
        /* If the position is set local on the robot this is not allowed */
        if (pDataParams->bPositionMode == PHDL_STEPPER_DENSO_VS60_POS_MODE_LOCAL)
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
        }
        pDataParams->structTargetXY.offsetCoords.wFigure = ((double)dwSignedValue)/10;
        if (pDataParams->bPositionMode != PHDL_STEPPER_DENSO_VS60_POS_MODE_COLLECT)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_Int_GoToXYAxis(pDataParams, PH_ON));
        }
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_J1:
        /* If the position is set local on the robot this is not allowed */
        if (pDataParams->bPositionMode == PHDL_STEPPER_DENSO_VS60_POS_MODE_LOCAL)
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
        }
        pDataParams->structTargetJoint.offsetCoords.dwPositionJ1 = ((double)dwSignedValue)/1000;
        if (pDataParams->bPositionMode != PHDL_STEPPER_DENSO_VS60_POS_MODE_COLLECT)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_Int_GoToJointAxis(pDataParams, PH_ON));
        }
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_J2:
        /* If the position is set local on the robot this is not allowed */
        if (pDataParams->bPositionMode == PHDL_STEPPER_DENSO_VS60_POS_MODE_LOCAL)
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
        }
        pDataParams->structTargetJoint.offsetCoords.dwPositionJ2 = ((double)dwSignedValue)/1000;
        if (pDataParams->bPositionMode != PHDL_STEPPER_DENSO_VS60_POS_MODE_COLLECT)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_Int_GoToJointAxis(pDataParams, PH_ON));
        }
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_J3:
        /* If the position is set local on the robot this is not allowed */
        if (pDataParams->bPositionMode == PHDL_STEPPER_DENSO_VS60_POS_MODE_LOCAL)
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
        }
        pDataParams->structTargetJoint.offsetCoords.dwPositionJ3 = ((double)dwSignedValue)/1000;
        if (pDataParams->bPositionMode != PHDL_STEPPER_DENSO_VS60_POS_MODE_COLLECT)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_Int_GoToJointAxis(pDataParams, PH_ON));
        }
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_J4:
        /* If the position is set local on the robot this is not allowed */
        if (pDataParams->bPositionMode == PHDL_STEPPER_DENSO_VS60_POS_MODE_LOCAL)
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
        }
        pDataParams->structTargetJoint.offsetCoords.dwPositionJ4 = ((double)dwSignedValue)/1000;
        if (pDataParams->bPositionMode != PHDL_STEPPER_DENSO_VS60_POS_MODE_COLLECT)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_Int_GoToJointAxis(pDataParams, PH_ON));
        }
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_J5:
        /* If the position is set local on the robot this is not allowed */
        if (pDataParams->bPositionMode == PHDL_STEPPER_DENSO_VS60_POS_MODE_LOCAL)
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
        }
        pDataParams->structTargetJoint.offsetCoords.dwPositionJ5 = ((double)dwSignedValue)/1000;
        if (pDataParams->bPositionMode != PHDL_STEPPER_DENSO_VS60_POS_MODE_COLLECT)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_Int_GoToJointAxis(pDataParams, PH_ON));
        }
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_J6:
        /* If the position is set local on the robot this is not allowed */
        if (pDataParams->bPositionMode == PHDL_STEPPER_DENSO_VS60_POS_MODE_LOCAL)
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
        }
        pDataParams->structTargetJoint.offsetCoords.dwPositionJ6 = ((double)dwSignedValue)/1000;
        if (pDataParams->bPositionMode != PHDL_STEPPER_DENSO_VS60_POS_MODE_COLLECT)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_Int_GoToJointAxis(pDataParams, PH_ON));
        }
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_J7:
        /* If the position is set local on the robot this is not allowed */
        if (pDataParams->bPositionMode == PHDL_STEPPER_DENSO_VS60_POS_MODE_LOCAL)
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
        }
        pDataParams->structTargetJoint.offsetCoords.dwPositionJ7 = ((double)dwSignedValue)/1000;
        if (pDataParams->bPositionMode != PHDL_STEPPER_DENSO_VS60_POS_MODE_COLLECT)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_Int_GoToJointAxis(pDataParams, PH_ON));
        }
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_J8:
        /* If the position is set local on the robot this is not allowed */
        if (pDataParams->bPositionMode == PHDL_STEPPER_DENSO_VS60_POS_MODE_LOCAL)
        {
            return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_STEPPER);
        }
        pDataParams->structTargetJoint.offsetCoords.dwPositionJ8 = ((double)dwSignedValue)/1000;
        if (pDataParams->bPositionMode != PHDL_STEPPER_DENSO_VS60_POS_MODE_COLLECT)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_Int_GoToJointAxis(pDataParams, PH_ON));
        }
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_DL_STEPPER);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_GetConfig(
                                    phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                    uint16_t wConfig,
                                    uint16_t * pValue
                                    )
{
    phStatus_t status;
    PH_ASSERT_NULL (pDataParams);

    switch(wConfig)
    {
    case PHDL_STEPPER_CONFIG_STEPS_WAY_RATIO:
        *pValue = pDataParams->wStepsWayRatio;
        break;

    case PHDL_STEPPER_DENSO_VS60_CONFIG_POSITION_MODE:
        *pValue = pDataParams->bPositionMode;
        break;

    case PHDL_STEPPER_DENSO_VS60_CONFIG_ACTIVE_AXIS:
        *pValue = pDataParams->bActiveAxis;
        break;

    case PHDL_STEPPER_DENSO_VS60_CONFIG_ACTIVATE_LASER:
         status = phdlStepper_DensoVS60_External_GetIODeviceState(pDataParams,26);
         if(status != 0)
         {
             status = 1;
         }
        *pValue = status;
        break;

    case PHDL_STEPPER_DENSO_VS60_CONFIG_ACTIVATE_PUMP:
         status = phdlStepper_DensoVS60_External_GetIODeviceState(pDataParams,25);
         if(status != 0)
         {
             status = 1;
         }
        *pValue = status;
         break;

    case PHDL_STEPPER_DENSO_VS60_CONFIG_ACTIVE_ROTATION:
        *pValue = pDataParams->bActiveRotation;
         break;

    case PHDL_STEPPER_DENSO_VS60_CONFIG_CHECK_HEAD_SENSOR:
         status = phdlStepper_DensoVS60_External_GetIODeviceState(pDataParams,10);
         if(status != 0)
         {
             status = 1;
         }
        *pValue = status;
         break;

    case PHDL_STEPPER_DENSO_VS60_CONFIG_ROBOT_SPEED:
        *pValue = pDataParams->wRobotSpeed;
        break;

    case  PHDL_STEPPER_DENSO_VS60_CONFIG_ROBOT_ACCELERATION:
        *pValue = pDataParams->bRobotAcceeration;
        break;

    case PHDL_STEPPER_DENSO_VS60_CONFIG_CORRECT_COLLECT:
        *pValue = pDataParams->bPositionModeCorrect;
        break;
        /* Unknown Identifier */
    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_DL_STEPPER);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

phStatus_t phdlStepper_DensoVS60_GetConfig32(
                                      phdlStepper_DensoVS60_DataParams_t * pDataParams,
                                      uint16_t wConfig,
                                      uint32_t * pValue
                                      )
{

    phStatus_t  PH_MEMLOC_REM statusTmp;
    int32_t    PH_MEMLOC_REM dwAct;
    double dwAbsolutValue;
    dwAbsolutValue = 0;
    PH_ASSERT_NULL (pDataParams);


    switch(wConfig)
    {
    case PHDL_STEPPER_CONFIG32_POSITION_COUNTER: /* POSITION_COUNTER uses steps so convert */
        switch (pDataParams->bActiveAxis)
        {
        /* Substract offset of the active coordiante from its current value in order to get the distance traveled*/
        case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_X:
            dwAct = pDataParams->structTargetXY.dwPositionX;
            break;
        case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_Y:
            dwAct = pDataParams->structTargetXY.dwPositionY;
            break;
        case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_Z:
            dwAct = pDataParams->structTargetXY.dwPositionZ;
            break;
        case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_RX:
            dwAct = pDataParams->structTargetXY.dwPositionRx;
            break;
        case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_RY:
            dwAct = pDataParams->structTargetXY.dwPositionRy;
            break;
        case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_RZ:
            dwAct = pDataParams->structTargetXY.dwPositionRz;
            break;
        case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_FIG:
            dwAct = pDataParams->structTargetXY.wFigure;
            break;
        case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_J1:
            dwAct = pDataParams->structTargetJoint.dwPositionJ1;
            break;
        case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_J2:
            dwAct = pDataParams->structTargetJoint.dwPositionJ2;
            break;
        case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_J3:
            dwAct = pDataParams->structTargetJoint.dwPositionJ3;
            break;
        case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_J4:
            dwAct = pDataParams->structTargetJoint.dwPositionJ4;
            break;
        case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_J5:
            dwAct = pDataParams->structTargetJoint.dwPositionJ5;
            break;
        case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_J6:
            dwAct = pDataParams->structTargetJoint.dwPositionJ6;
            break;
        case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_J7:
            dwAct = pDataParams->structTargetJoint.dwPositionJ7;
            break;
        case PHDL_STEPPER_DENSO_VS60_ACTIVE_AXIS_J8:
            dwAct = pDataParams->structTargetJoint.dwPositionJ8;
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_STEPPER);
        }
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlStepper_DensoVS60_Int_ConvertToStep(pDataParams, dwAct, &dwAct));
        *pValue = (uint32_t)dwAct;
        break;

    case PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_X:
        phdlStepper_DensoVS60_External_GetParticularAbsoluteCurrentPosition( pDataParams,0,1,&dwAbsolutValue);
        *pValue = (uint32_t)((dwAbsolutValue - pDataParams->structTargetXY.offsetCoords.dwPositionX)*1000);
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_Y:
        phdlStepper_DensoVS60_External_GetParticularAbsoluteCurrentPosition( pDataParams,0,2,&dwAbsolutValue);
        *pValue = (uint32_t)((dwAbsolutValue - pDataParams->structTargetXY.offsetCoords.dwPositionY)*1000);
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_Z:
        phdlStepper_DensoVS60_External_GetParticularAbsoluteCurrentPosition( pDataParams,0,3,&dwAbsolutValue);
        *pValue = (uint32_t)((dwAbsolutValue - pDataParams->structTargetXY.offsetCoords.dwPositionZ)*1000);
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_RX:
        phdlStepper_DensoVS60_External_GetParticularAbsoluteCurrentPosition( pDataParams,0,4,&dwAbsolutValue);
        *pValue = (uint32_t)((dwAbsolutValue - pDataParams->structTargetXY.offsetCoords.dwPositionRx)*10);
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_RY:
        phdlStepper_DensoVS60_External_GetParticularAbsoluteCurrentPosition( pDataParams,0,5,&dwAbsolutValue);
        *pValue = (uint32_t)((dwAbsolutValue - pDataParams->structTargetXY.offsetCoords.dwPositionRy)*10);
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_RZ:
        phdlStepper_DensoVS60_External_GetParticularAbsoluteCurrentPosition( pDataParams,0,6,&dwAbsolutValue);
        *pValue = (uint32_t)((dwAbsolutValue - pDataParams->structTargetXY.offsetCoords.dwPositionRz)*10);
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_FIG:
        phdlStepper_DensoVS60_External_GetParticularAbsoluteCurrentPosition( pDataParams,0,7,&dwAbsolutValue);
        *pValue = (uint32_t)((dwAbsolutValue - pDataParams->structTargetXY.offsetCoords.wFigure)*10);
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_J1:
        phdlStepper_DensoVS60_External_GetParticularAbsoluteCurrentPosition( pDataParams,1,1,&dwAbsolutValue);
        *pValue = (uint32_t)((dwAbsolutValue - pDataParams->structTargetJoint.offsetCoords.dwPositionJ1)*1000);
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_J2:
        phdlStepper_DensoVS60_External_GetParticularAbsoluteCurrentPosition( pDataParams,1,2,&dwAbsolutValue);
        *pValue = (uint32_t)((dwAbsolutValue - pDataParams->structTargetJoint.offsetCoords.dwPositionJ2)*1000);
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_J3:
         phdlStepper_DensoVS60_External_GetParticularAbsoluteCurrentPosition( pDataParams,1,3,&dwAbsolutValue);
        *pValue = (uint32_t)((dwAbsolutValue - pDataParams->structTargetJoint.offsetCoords.dwPositionJ3)*1000);
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_J4:
        phdlStepper_DensoVS60_External_GetParticularAbsoluteCurrentPosition( pDataParams,1,4,&dwAbsolutValue);
        *pValue = (uint32_t)((dwAbsolutValue - pDataParams->structTargetJoint.offsetCoords.dwPositionJ4)*1000);
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_J5:
        phdlStepper_DensoVS60_External_GetParticularAbsoluteCurrentPosition( pDataParams,1,5,&dwAbsolutValue);
        *pValue = (uint32_t)((dwAbsolutValue - pDataParams->structTargetJoint.offsetCoords.dwPositionJ5)*1000);
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_J6:
         phdlStepper_DensoVS60_External_GetParticularAbsoluteCurrentPosition( pDataParams,1,6,&dwAbsolutValue);
        *pValue = (uint32_t)((dwAbsolutValue - pDataParams->structTargetJoint.offsetCoords.dwPositionJ6)*1000);
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_J7:
        phdlStepper_DensoVS60_External_GetParticularAbsoluteCurrentPosition( pDataParams,1,7,&dwAbsolutValue);
        *pValue = (uint32_t)((dwAbsolutValue - pDataParams->structTargetJoint.offsetCoords.dwPositionJ7)*1000);
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_POSITION_J8:
        phdlStepper_DensoVS60_External_GetParticularAbsoluteCurrentPosition( pDataParams,1,8,&dwAbsolutValue);
        *pValue = (uint32_t)((dwAbsolutValue - pDataParams->structTargetJoint.offsetCoords.dwPositionJ8)*1000); ;
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_X:
        *pValue = (uint32_t)(1000*(pDataParams->structTargetXY.offsetCoords.dwPositionX));
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_Y:
        *pValue = (uint32_t)(1000*(pDataParams->structTargetXY.offsetCoords.dwPositionY));
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_Z:
        *pValue = (uint32_t)(1000*(pDataParams->structTargetXY.offsetCoords.dwPositionZ));
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_RX:
        *pValue = (uint32_t)(10*(pDataParams->structTargetXY.offsetCoords.dwPositionRx));
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_RY:
        *pValue = (uint32_t)(10*(pDataParams->structTargetXY.offsetCoords.dwPositionRy));
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_RZ:
        *pValue =  (uint32_t)(10*(pDataParams->structTargetXY.offsetCoords.dwPositionRz));
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_FIG:
        *pValue = (uint32_t)(10*(pDataParams->structTargetXY.offsetCoords.wFigure));
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_J1:
        *pValue = (uint32_t)(1000*(pDataParams->structTargetJoint.offsetCoords.dwPositionJ1));
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_J2:
        *pValue = (uint32_t)(1000*(pDataParams->structTargetJoint.offsetCoords.dwPositionJ2));
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_J3:
        *pValue = (uint32_t)(1000*(pDataParams->structTargetJoint.offsetCoords.dwPositionJ3));
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_J4:
        *pValue = (uint32_t)(1000*(pDataParams->structTargetJoint.offsetCoords.dwPositionJ4));
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_J5:
        *pValue = (uint32_t)(1000*(pDataParams->structTargetJoint.offsetCoords.dwPositionJ5));
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_J6:
        *pValue = (uint32_t)(1000*(pDataParams->structTargetJoint.offsetCoords.dwPositionJ6));
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_J7:
        *pValue = (uint32_t)(1000*(pDataParams->structTargetJoint.offsetCoords.dwPositionJ7));
        break;
    case PHDL_STEPPER_DENSO_VS60_CONFIG32_OFFSET_J8:
        *pValue = (uint32_t)(1000*(pDataParams->structTargetJoint.offsetCoords.dwPositionJ8));
        break;

        /* Unknown Identifier */
    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_DL_STEPPER);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_STEPPER);
}

#endif /* NXPBUILD__PHDL_STEPPER_DENSO_VS60 */
