/*
 * Copyright 2017 - 2018, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal Generic Powersupply Device Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>

#ifdef NXPBUILD__PHDL_POWERSUPPLY_E3631A
#include "E3631A/phdlPowerSupply_E3631A.h"
#endif /* NXPBUILD__PHDL_POWERSUPPLY_E3631A */

#ifdef NXPBUILD__PHDL_POWERSUPPLY
#include "phdlPowerSupply_Int.h"

phStatus_t phdlPowerSupply_Int_Send(
                             phdlPowerSupply_E3631A_DataParams_t * pDataParams,
                             uint8_t * pTxBuffer,
                             uint16_t wTxLength
                             )
{
    phStatus_t statusTmp, status;
    uint16_t wValue=0;
    uint8_t bRx;
    uint16_t wRxLen;

    /* read current IO timeout */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_GetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, &wValue));
    /* set IO timeout to zero to speed up timeout error */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, 100));

    /* send data */
    status = phbalReg_Exchange(
        pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        pTxBuffer,
        wTxLength,
        1,
        &bRx,
        &wRxLen);

    /* restore IO timeout use statusTmp2 to not overwrite response from send */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, wValue));

    /* handle expected IO-timeouterror occured and ignore error as no response is excpected */
    if ((status & PH_ERR_MASK) == PH_ERR_IO_TIMEOUT)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
    }

    return status;
}

phStatus_t phdlPowerSupply_Int_SendWait(
                                 phdlPowerSupply_E3631A_DataParams_t * pDataParams,
                                 uint8_t * pTxBuffer,
                                 uint16_t wTxLength,
                                 uint16_t wWaittime
                                 )
{
    phStatus_t statusTmp, status;
    uint16_t wValue=0;
    uint8_t bRx;
    uint16_t wRxLen;

    /* read current IO timeout */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_GetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, &wValue));
    /* set IO timeout to zero to speed up timeout error */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, wWaittime));

    /* send data */
    status = phbalReg_Exchange(
        pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        pTxBuffer,
        wTxLength,
        1,
        &bRx,
        &wRxLen);

    /* restore IO timeout use statusTmp2 to not overwrite response from send */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, wValue));

    /* handle expected IO-timeouterror occured and ignore error as no response is excpected */
    if ((status & PH_ERR_MASK) == PH_ERR_IO_TIMEOUT)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_POWERSUPPLY);
    }

    return status;
}

/* helperfunction asci to integer */
phStatus_t phdlPowerSupply_Int_Atoui16(
                                int8_t * pAsci,
                                uint16_t * wURetVal
                                )
{
    uint16_t wTemp=0;

    if (*pAsci >= '0' && *pAsci <= '9')
    {
        do
        {
            if (*pAsci != '.' )
            {
                wTemp *= 10;
                wTemp += (*pAsci-'0');
            }
            pAsci++;
        } while ((*pAsci >= '0' && *pAsci <= '9') || *pAsci == '.');
    }

    *wURetVal = wTemp;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_POWERSUPPLY);
}

/* helperfunction asci to integer */
phStatus_t phdlPowerSupply_Int_Atoi16(
                               int8_t * pAsci,
                               int16_t * wIRetVal
                               )
{
    int16_t wTemp=0;

    if ((*pAsci >= '0' && *pAsci <= '9') || *pAsci == '-')
    {
        if (*pAsci == '-')
        {
            /* increse index, so that minus sign is ignored */
            pAsci++;
            do
            {
                if (*pAsci != '.')
                {
                    wTemp *= 10;
                    wTemp -= (*pAsci-'0');
                }
                pAsci++;
            } while ((*pAsci >= '0' && *pAsci <= '9') || *pAsci == '.');
        }
        else
        {
            do
            {
                if (*pAsci != '.')
                {
                    wTemp *= 10;
                    wTemp += (*pAsci-'0');
                }
                pAsci++;
            }while ((*pAsci >= '0' && *pAsci <= '9') || *pAsci == '.');
        }
    }
    *wIRetVal = wTemp;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_POWERSUPPLY);
}

phStatus_t phdlPowerSupply_Int_FindChar(
                                 int8_t * pAsci,
                                 int8_t bRef,
                                 uint8_t * bRetVal
                                 )
{
    uint8_t bCount=0;

    do
    {
        if (*pAsci == bRef)
        {
            break;
        }
        bCount++;
        pAsci++;
    } while (!(*pAsci == 0 || *pAsci == 10|| *pAsci == 13));

    *bRetVal= bCount;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_POWERSUPPLY);
}

phStatus_t phdlPowerSupply_Int_FindStr(int8_t * pAsci,
                                int8_t * pRef,
                                uint8_t bRefSize,
                                uint8_t * pbStrFound
                                )
{
    uint8_t bI, bRefCount;

    *pbStrFound = 0;
    do
    {
        bRefCount = 0;
        /* check for string */
        for (bI = 0; bI<bRefSize ; bI++)
        {
            if (*(pAsci+bI) == *(pRef+bI))
            {
                bRefCount++;
            }
        }
        if (bRefCount == bRefSize)
        {
            *pbStrFound = 1;
            break;
        }
        /* incease index */
        pAsci++;
    } while (!(*(pAsci + bRefSize -1) == 0 || *(pAsci + bRefSize -1) == 10|| *(pAsci + bRefSize -1) == 13));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_POWERSUPPLY);
}

phStatus_t  phdlPowerSupply_Int_Atof(
                              int8_t * pAsci,
                              float32_t * fRetVal
                              )
{
    phStatus_t statusTmp;
    float32_t fTemp = 0;
    uint8_t bDiv =0, bNeg = 0;
    int16_t iNegExp = 0;
    int16_t iExp = 0;
    int16_t iTemp;

    /* check if negative number */
    if (*pAsci == '-')
    {
        bNeg = 1;
        pAsci++;
    }
    /* check if positive number */
    if (*pAsci == '+')
    {
		bNeg = 0;
        pAsci++;
    }
    /* check for space */
    if (*pAsci == ' ')
    {
        pAsci++;
    }

    /* get significand */
    if (*pAsci >= '0' && *pAsci <= '9')
    {
        do
        {
            if (*pAsci != '.')
            {
                fTemp *= 10;
                fTemp += (float) (*pAsci-'0');
                /* if decimal increase division */
                if (bDiv)
                {
                    iNegExp++;
                }
            }
            else
            {
                bDiv = 1;
            }
            /* increase pointer */
            pAsci++;
        } while ((*pAsci >= '0' && *pAsci <= '9') || *pAsci == '.');
    }

    /* check for exponent */
    if (*pAsci == 'E')
    {
        pAsci++;
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlPowerSupply_Int_Atoi16((int8_t*)pAsci, &iTemp));
        iExp = (int16_t) iTemp;
    }

    /* calculate exponent */
    iExp = iExp  - iNegExp;

    if (iExp>0)
    {
        while(iExp!=0)
        {
            fTemp *= 10;
            iExp--;
        }
    }
    else if (iExp < 0)
    {
        while( iExp != 0)
        {
            fTemp /= 10;
            iExp++;
        }
    }
    else
    {
        /* do nothing */
    }

    /* check for sign */
    if (bNeg)
    {
        fTemp = (float32_t) -fTemp;
    }
    *fRetVal = fTemp;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_POWERSUPPLY);
}

#endif /* NXPBUILD__PHDL_POWERSUPPLY */
