/*
 * Copyright 2017, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Hardware Oscilloscope Waverunner L64Xi Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phbalReg.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHDL_POWERSUPPLY_E3631A

#include <phdlPowerSupply.h>
#include "phdlPowerSupply_E3631A.h"
#include "phdlPowerSupply_E3631A_Int.h"
#include "../phdlPowerSupply_Int.h"

phStatus_t phdlPowerSupply_E3631A_Init(
                                phdlPowerSupply_E3631A_DataParams_t * pDataParams,
                                uint16_t wSizeOfDataParams,
                                phbalReg_Visa_DataParams_t * pBalRegDataParams
                                )
{
    phStatus_t statusTmp;

    if (sizeof(phdlPowerSupply_E3631A_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_DL_POWERSUPPLY);
    }
    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pBalRegDataParams);



    /* init private data */
    pDataParams->wId                 = PH_COMP_DL_POWERSUPPLY | PHDL_POWERSUPPLY_E3631A_ID;
    pDataParams->pBalRegDataParams   = pBalRegDataParams;

    pDataParams->wMillivolts[0] = 0;
    pDataParams->wMillivolts[1] = 0;
    pDataParams->wMillivolts[2] = 0;

    pDataParams->bStateOfChannels[0] = PH_OFF;
    pDataParams->bStateOfChannels[1] = PH_OFF;
    pDataParams->bStateOfChannels[2] = PH_OFF;

    pDataParams->bChannel = PHDL_POWERSUPPLY_CHANNEL_ONE;

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlPowerSupply_E3631A_Int_ResetDevice(pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_POWERSUPPLY);
}

phStatus_t phdlPowerSupply_E3631A_InitPowerSupply(
                                    phdlPowerSupply_E3631A_DataParams_t * pDataParams
                                    )
{
    phStatus_t statusTmp;

    /* clear all status bits */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlPowerSupply_Int_Send(pDataParams, (uint8_t *)"*CLS\n", 5));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_POWERSUPPLY);
}

phStatus_t phdlPowerSupply_E3631A_SetConfig(
                                     phdlPowerSupply_E3631A_DataParams_t * pDataParams,
                                     uint16_t wIdentifier,
                                     uint16_t wValue
                                     )
{
    phStatus_t statusTmp;

    switch (wIdentifier)
    {
    case PHDL_POWERSUPPLY_CONFIG_VOLT:
        wValue = wValue * 1000;
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlPowerSupply_E3631A_SetConfig(pDataParams, PHDL_POWERSUPPLY_CONFIG_MILLIVOLT, wValue));
        break;

    case PHDL_POWERSUPPLY_CONFIG_MILLIVOLT:
        if (pDataParams->bChannel == PHDL_POWERSUPPLY_CHANNEL_ONE)
        {
            if (wValue > 6000)
            {
                return(PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_POWERSUPPLY));
            }
        }
        else if ( (pDataParams->bChannel == PHDL_POWERSUPPLY_CHANNEL_TWO)
                  || (pDataParams->bChannel == PHDL_POWERSUPPLY_CHANNEL_THREE))
        {
            if (wValue > 25000)
            {
                return(PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_POWERSUPPLY));
            }
        }
        else
        {
            return(PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_POWERSUPPLY));
        }
        pDataParams->wMillivolts[pDataParams->bChannel] = wValue;
        break;

    case PHDL_POWERSUPPLY_CONFIG_CHANNEL:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlPowerSupply_E3631A_Int_SetChannel(pDataParams, (uint8_t)wValue));
        pDataParams->bChannel = (uint8_t)wValue;
        break;

    default:
        /* return error code */
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_POWERSUPPLY);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_POWERSUPPLY);
}

phStatus_t phdlPowerSupply_E3631A_GetConfig(
                                     phdlPowerSupply_E3631A_DataParams_t * pDataParams,
                                     uint16_t wIdentifier,
                                     uint16_t * pwValue
                                     )
{
    switch (wIdentifier)
    {
    case PHDL_POWERSUPPLY_CONFIG_VOLT:
        *pwValue = pDataParams->wMillivolts[pDataParams->bChannel];
        break;

    case PHDL_POWERSUPPLY_CONFIG_MILLIVOLT:
        *pwValue = pDataParams->wMillivolts[pDataParams->bChannel];
        break;

    case PHDL_POWERSUPPLY_CONFIG_CHANNEL:
        *pwValue = pDataParams->bChannel;
        break;

    default:
        /* return error code */
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_POWERSUPPLY);
}

phStatus_t phdlPowerSupply_E3631A_PowerOn(
                                   phdlPowerSupply_E3631A_DataParams_t * pDataParams,
                                   uint8_t  bChannel
                                   )
{

    return phdlPowerSupply_E3631A_Int_PowerOn(pDataParams, bChannel);
}

phStatus_t phdlPowerSupply_E3631A_PowerOff(
                                   phdlPowerSupply_E3631A_DataParams_t * pDataParams,
                                   uint8_t  bChannel
                                   )
{
    return phdlPowerSupply_E3631A_Int_PowerOff(pDataParams, bChannel);
}

#endif /* NXPBUILD__PHDL_POWERSUPPLY_E3631A */
