/*
 * Copyright 2013 - 2014, 2017 - 2020, 2022 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal Hardware Oscilloscope Waverunner 64Xi Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phbalReg.h>
#include <ph_RefDefs.h>
#include <phTools.h>
#include <math.h>
#include <stdio.h>              /* PRQA S 5124 */

#ifdef NXPBUILD__PHDL_OSCI_LW64XI

#include "phdlOsci_LW64Xi.h"
#include "phdlOsci_LW64Xi_Int.h"
#include "../phdlOsci_Int.h"

/* for visa functionality displaced function exchange large */
#include "../../../phbalReg/src/Visa/external/visa.h"
#include "../../../phbalReg/src/Visa/phbalReg_Visa_Cmd.h"

phStatus_t phdlOsci_LW64Xi_Int_ChkLastCmd(
    phdlOsci_LW64Xi_DataParams_t * pDataParams
    )
{
    phStatus_t statusTmp;
    uint8_t bRx[10];
    uint16_t wRxlen, wTemp;
    uint8_t bCmdCode;

    /* read command errror register */
    statusTmp = phbalReg_Exchange(
        pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        (uint8_t *)"CMR?\n",
        5,
        10,
        (uint8_t*)bRx,
        &wRxlen);
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_CheckClearError(pDataParams, statusTmp));

    /* convert response */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atoui16((int8_t*)bRx, &wTemp));
    bCmdCode = (uint8_t)wTemp;

    /* analyse response */
    switch (bCmdCode)
    {
    case 0:
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);

    case 1:
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_CMD_UNKNOWN_COMMAND, PH_COMP_DL_OSCI);

    case 2:
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_CMD_ILLEGAL_HEADER_PATH, PH_COMP_DL_OSCI);

    case 3:
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_CMD_ILLEGAL_NUMBER, PH_COMP_DL_OSCI);

    case 4:
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_CMD_ILLEGAL_NUMBER_SUFFIX, PH_COMP_DL_OSCI);

    case 5:
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_CMD_UNKNOWN_KEYWORD, PH_COMP_DL_OSCI);

    case 6:
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_CMD_STRING_ERROR, PH_COMP_DL_OSCI);

    case 7:
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_CMD_GET_EMBEDDED_IN_ANOTHER_MESSAGE, PH_COMP_DL_OSCI);

    case 10:
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_CMD_ARBITRARY_DATA_BLOCK_EXPECTED, PH_COMP_DL_OSCI);

    case 11:
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_CMD_NONDIGIT_CHAR_IN_BYTE_COUNT_FIELD, PH_COMP_DL_OSCI);

    case 12:
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_CMD_EOI_DETECTED, PH_COMP_DL_OSCI);

    case 13:
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_CMD_EXTRA_BYTES_DETECTED, PH_COMP_DL_OSCI);

    default:
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
    }
}

phStatus_t phdlOsci_LW64Xi_Int_ChkLastExe(
    phdlOsci_LW64Xi_DataParams_t * pDataParams
    )
{
    phStatus_t statusTmp;
    uint8_t bRx[10];
    uint16_t wRxlen,wTemp;
    uint8_t bCmdCode;

    /* read command execute register */
    statusTmp = phbalReg_Exchange(
        pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        (uint8_t*)"EXR?\n",
        5,
        10,
        (uint8_t*)bRx,
        &wRxlen);
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_CheckClearError(pDataParams, statusTmp));

    /* convert response */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atoui16((int8_t*)bRx, &wTemp));
    bCmdCode = (uint8_t)wTemp;

    /* analyse response */
    switch (bCmdCode)
    {
    case 0:
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);

    case 21:
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_EXE_PERMISSION_ERROR, PH_COMP_DL_OSCI);

    case 22:
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_EXE_ENVIRONMENT_ERROR, PH_COMP_DL_OSCI);

    case 23:
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_EXE_OPTION_ERROR, PH_COMP_DL_OSCI);

    case 24:
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_EXE_UNRESOLVED_PARSING, PH_COMP_DL_OSCI);

    case 25:
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_EXE_PARAMETER_ERROR, PH_COMP_DL_OSCI);

    case 26:
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_EXE_CMD_NOT_IMPLEMENTED, PH_COMP_DL_OSCI);

    case 27:
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_EXE_PARAM_MISSING, PH_COMP_DL_OSCI);

    case 30:
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_EXE_HEX_DATA_ERROR, PH_COMP_DL_OSCI);

    case 31:
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_EXE_WAVEFORM_ERROR, PH_COMP_DL_OSCI);

    case 32:
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_EXE_WAVEFORM_DESCRIPTION_ERROR, PH_COMP_DL_OSCI);

    case 33:
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_EXE_WAVEFORM_TEXT_ERROR, PH_COMP_DL_OSCI);

    case 34:
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_EXE_WAVEFORM_TIME_ERROR, PH_COMP_DL_OSCI);

    case 35:
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_EXE_WAVEFORM_DATA_ERROR, PH_COMP_DL_OSCI);

    case 36:
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_EXE_PANEL_SETUP_ERROR, PH_COMP_DL_OSCI);

    case 50:
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_EXE_NO_MASS_STORAGE_PRESENT, PH_COMP_DL_OSCI);

    case 51:
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_EXE_MASS_STORAGE_NOT_FORMATTED, PH_COMP_DL_OSCI);

    case 53:
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_EXE_MASS_STORAGE_WAS_WRITE_PROTECTED, PH_COMP_DL_OSCI);

    case 54:
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_EXE_BAD_MASS_STORAGE_DETECTED, PH_COMP_DL_OSCI);

    case 55:
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_EXE_MASS_STOR_ROOT_DIR_FULL, PH_COMP_DL_OSCI);

    case 56:
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_EXE_MASS_STOR_FULL, PH_COMP_DL_OSCI);

    case 57:
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_EXE_MASS_STOR_FILE_SEQ_NUM_EXHAUSTED, PH_COMP_DL_OSCI);

    case 58:
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_EXE_MASS_STOR_FILE_NOT_FOUND, PH_COMP_DL_OSCI);

    case 59:
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_EXE_REQ_DIR_NOT_FOUND, PH_COMP_DL_OSCI);

    case 61:
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_EXE_FILENAME_ILLEGAL, PH_COMP_DL_OSCI);

    case 62:
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_EXE_MASS_STORAGE_FILENAME_ALREADY_EXISTS, PH_COMP_DL_OSCI);

    default:
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
    }
}

phStatus_t phdlOsci_LW64Xi_Int_WaitExe(
                                       phdlOsci_LW64Xi_DataParams_t * pDataParams
                                       )
{
    phStatus_t statusTmp = PH_ERR_SUCCESS;
    uint8_t bRx[10];
    uint16_t wRxlen;
    uint8_t bRetryCounter;

    /* Add a retry for this command because interaction with LeCroy HDO6034A shows sometimes a BAL,IO_TIMEOUT on OPC
       altought scope log shows that a response was send */
    for (bRetryCounter = 0; bRetryCounter < 2; bRetryCounter++)
    {
        /* read operation complete, osci only responds if the last operation is completed */
        statusTmp = phbalReg_Exchange(
            pDataParams->pBalRegDataParams,
            PH_EXCHANGE_DEFAULT,
            (uint8_t*)"*OPC?\n",
            6,
            10,
            (uint8_t*)bRx,
            &wRxlen);

        /* Only retry in case of timeout */
        if ((statusTmp & PH_ERR_MASK) != PH_ERR_IO_TIMEOUT)
            break;
    }
    return phdlOsci_Int_CheckClearError(pDataParams, statusTmp);
}

phStatus_t phdlOsci_LW64Xi_Int_ChkCmd(
                                      phdlOsci_LW64Xi_DataParams_t * pDataParams
                                      )
{
    phStatus_t statusTmp;

    /* wait for command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_WaitExe(pDataParams));

    /* check if last command was interpreted correct */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_ChkLastCmd(pDataParams));

    /* check if last command was executed correct */
    return phdlOsci_LW64Xi_Int_ChkLastExe(pDataParams);
}

phStatus_t phdlOsci_LW64Xi_Int_GetAmpl(
                                       phdlOsci_LW64Xi_DataParams_t * pDataParams,
                                       uint8_t  pbSource[2],
                                       float32_t * pfAmplitude
                                       )
{
    phStatus_t statusTmp;
    float32_t fTemp, fTemp2;
    uint8_t bRx[20], bPos;
    uint16_t wI;
    uint16_t wRxlen;
    uint8_t abCmd[20] = "C1:PAVA? AMPL\n";

    /* check if source is F than change, otherwise discard all and use default */
    if (pbSource[0] == 'F')
    {
        abCmd[0] = 'F';
    }

    /* check channel number */
    if (pbSource[1] == '2' || pbSource[1] == '3' || pbSource[1] == '4')
    {
        abCmd[1] = pbSource[1];
    }

    /* reset value for amplitude */
    *pfAmplitude = 0.0;
    fTemp = 0.0;

    /* average over wAverageFact measurements */
    for (wI = 0; wI < pDataParams->wAverageFact; wI++)
    {
        /* Set Single trigger and force trigger to get new data */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_SetTriggerMode(pDataParams, PHDL_OSCI_TRIGGER_SINGLE));
        if ((pDataParams->bMeasurementOptions&PHDL_OSCI_MEASUREMENT_SUPPRESS_FORCE_TRIGGER) != PHDL_OSCI_MEASUREMENT_SUPPRESS_FORCE_TRIGGER)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_SendWait(pDataParams, (uint8_t *)"FRTR\n", 5, 10));
        }

        /* check for operation complete before reading any data */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_WaitExe(pDataParams));

        if ((pDataParams->bMeasurementOptions&PHDL_OSCI_MEASUREMENT_WAIT_STOP) == PHDL_OSCI_MEASUREMENT_WAIT_STOP)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_WaitForStateStop(pDataParams));
        }

        /* read the measurement data */
        statusTmp = phbalReg_Exchange(
            pDataParams->pBalRegDataParams,
            PH_EXCHANGE_DEFAULT,
            (uint8_t*)abCmd,
            14,
            20,
            (uint8_t*)bRx,
            &wRxlen);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_CheckClearError(pDataParams, statusTmp));

        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_FindChar((int8_t *)bRx, ',', &bPos));

        /* convert string, considering offset through ',' */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atof(((int8_t *)bRx) + bPos + 1, &fTemp2));
        fTemp += fTemp2;
    }

    /* check if amplitude is ok (too small or negative */
    if (fTemp < 1e-6)
    {
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_AMPLITUDE_VAL, PH_COMP_DL_OSCI);
    }

    /* calculate average */
    *pfAmplitude = fTemp / (float32_t) pDataParams->wAverageFact;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_LW64Xi_Int_GetRMS(
                                       phdlOsci_LW64Xi_DataParams_t * pDataParams,
                                       uint8_t  pbSource[2],
                                       float32_t * pfRMS
                                       )
{
    phStatus_t statusTmp;
    float32_t fTemp, fTemp2;
    uint8_t bRx[20], bPos;
    uint16_t wI;
    uint16_t wRxlen;
    uint8_t abCmd[20] = "C1:PAVA? RMS\n";

    /* check if source is F than change, otherwise discard all and use default */
    if (pbSource[0] == 'F')
    {
        abCmd[0] = 'F';
    }

    /* check channel number */
    if (pbSource[1] == '2' || pbSource[1] == '3' || pbSource[1] == '4')
    {
        abCmd[1] = pbSource[1];
    }

    /* reset value for amplitude */
    *pfRMS = 0.0;
    fTemp = 0.0;

    /* average over wAverageFact measurements */
    for (wI = 0; wI < pDataParams->wAverageFact; wI++)
    {
        /* Set Single trigger and force trigger to get new data */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_SetTriggerMode(pDataParams, PHDL_OSCI_TRIGGER_SINGLE));
        if ((pDataParams->bMeasurementOptions&PHDL_OSCI_MEASUREMENT_SUPPRESS_FORCE_TRIGGER) != PHDL_OSCI_MEASUREMENT_SUPPRESS_FORCE_TRIGGER)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_SendWait(pDataParams, (uint8_t *)"FRTR\n", 5, 10));
        }

        /* check for operation complete before reading any data */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_WaitExe(pDataParams));

        if ((pDataParams->bMeasurementOptions&PHDL_OSCI_MEASUREMENT_WAIT_STOP) == PHDL_OSCI_MEASUREMENT_WAIT_STOP)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_WaitForStateStop(pDataParams));
        }

        /* read the measurement data */
        statusTmp = phbalReg_Exchange(
            pDataParams->pBalRegDataParams,
            PH_EXCHANGE_DEFAULT,
            (uint8_t*)abCmd,
            13,
            20,
            (uint8_t*)bRx,
            &wRxlen);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_CheckClearError(pDataParams, statusTmp));

        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_FindChar((int8_t *)bRx, ',', &bPos));

        /* convert string, considering offset through ',' */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atof(((int8_t *)bRx) + bPos + 1, &fTemp2));
        fTemp += fTemp2 * fTemp2;
    }

    /* check if amplitude is ok (too small or negative */
    if (fTemp < 1e-12)
    {
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_AMPLITUDE_VAL, PH_COMP_DL_OSCI);
    }

    /* calculate average */
    *pfRMS = (float32_t)sqrt(fTemp / (float32_t) pDataParams->wAverageFact);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_LW64Xi_Int_GetPeakPeak(
    phdlOsci_LW64Xi_DataParams_t * pDataParams,
    uint8_t  bSource[2],
    float32_t * pfAmplitude
    )
{
    phStatus_t statusTmp;
    float32_t fTemp, fTemp2;
    uint8_t bRx[40], bPos;
    uint16_t wI;
    uint16_t wRxlen;
    uint8_t abCmd[20] = "C1:PAVA? PKPK\n";

    /* check if source is F than change, otherwise discard all and use default */
    if (bSource[0] == 'F')
    {
        abCmd[0] = 'F';
    }

    /* check channel number */
    if (bSource[1] == '2' || bSource[1] == '3' || bSource[1] == '4')
    {
        abCmd[1] = bSource[1];
    }

    /* reset value for amplitude */
    *pfAmplitude = 0.0;
    fTemp = 0.0;

    /* average over wAverageFact measurements */
    for (wI = 0; wI < pDataParams->wAverageFact; wI++)
    {
        /* Set Single trigger and force trigger to get new data */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_SetTriggerMode(pDataParams, PHDL_OSCI_TRIGGER_SINGLE));
        if ((pDataParams->bMeasurementOptions&PHDL_OSCI_MEASUREMENT_SUPPRESS_FORCE_TRIGGER) != PHDL_OSCI_MEASUREMENT_SUPPRESS_FORCE_TRIGGER)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_SendWait(pDataParams, (uint8_t *)"FRTR\n", 5, 10));
        }

        /* check for operation complete before reading any data */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_WaitExe(pDataParams));

        if ((pDataParams->bMeasurementOptions&PHDL_OSCI_MEASUREMENT_WAIT_STOP) == PHDL_OSCI_MEASUREMENT_WAIT_STOP)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_WaitForStateStop(pDataParams));
        }

        /* read the measurement data */
        statusTmp = phbalReg_Exchange(
            pDataParams->pBalRegDataParams,
            PH_EXCHANGE_DEFAULT,
            (uint8_t*)abCmd,
            14,
            40,
            (uint8_t*)bRx,
            &wRxlen);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_CheckClearError(pDataParams, statusTmp));

        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_FindChar((int8_t *)bRx, ',', &bPos));
        /* convert string, considering offset through ',' */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atof(((int8_t *)bRx) + bPos + 1, &fTemp2));
        fTemp += fTemp2;
    }

    /* check if amplitude is ok (too small or negative) */
    if (fTemp < 1e-6)
    {
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_AMPLITUDE_VAL, PH_COMP_DL_OSCI);
    }

    /* calculate average */
    *pfAmplitude = fTemp / (float32_t) pDataParams->wAverageFact;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_LW64Xi_Int_OverUndershoot(
    phdlOsci_LW64Xi_DataParams_t * pDataParams,
    uint8_t  bSource[2],
    uint8_t * pBIsOverUnderShoot
    )
{
    phStatus_t statusTmp;
    uint8_t bRx[40], bTemp;
    uint8_t bRxPkpk[40];
    uint16_t wRxlen;
    uint8_t abCmd[20] = "C1:PAVA? OVSN\n";
    uint8_t abCmdPkpk[20] = "C1:PAVA? PKPK\n";

    /* check if source is F than change, otherwise discard all and use default */
    if (bSource[0] == 'F')
    {
        abCmd[0] = 'F';
        abCmdPkpk[0] = 'F';
    }

    /* check channel number */
    if (bSource[1] == '2' || bSource[1] == '3' || bSource[1] == '4')
    {
        abCmd[1] = bSource[1];
        abCmdPkpk[1] = bSource[1];
    }

    /* Set Single trigger and force trigger to get new data */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_SetTriggerMode(pDataParams, PHDL_OSCI_TRIGGER_SINGLE));
    if ((pDataParams->bMeasurementOptions&PHDL_OSCI_MEASUREMENT_SUPPRESS_FORCE_TRIGGER) != PHDL_OSCI_MEASUREMENT_SUPPRESS_FORCE_TRIGGER)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_SendWait(pDataParams, (uint8_t *)"FRTR\n", 5, 10));
    }

    /* check for operation complete before reading any data */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_WaitExe(pDataParams));

    if ((pDataParams->bMeasurementOptions&PHDL_OSCI_MEASUREMENT_WAIT_STOP) == PHDL_OSCI_MEASUREMENT_WAIT_STOP)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_WaitForStateStop(pDataParams));
    }

    /* read the measurement data */
    statusTmp = phbalReg_Exchange(
        pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        (uint8_t*)abCmd,
        14,
        40,
        (uint8_t*)bRx,
        &wRxlen);
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_CheckClearError(pDataParams, statusTmp));

    /* evaluate response */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_FindStr((int8_t *)bRx, (int8_t *)"OK", 2, &bTemp));
    if (bTemp)
    {
        *pBIsOverUnderShoot = 0;
    }
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_FindStr((int8_t *)bRx, (int8_t *)"OU", 2, &bTemp));
    if (bTemp)
    {
        *pBIsOverUnderShoot = 1;
    }
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_FindStr((int8_t *)bRx, (int8_t *)"UF", 2, &bTemp));
    if (bTemp)
    {
        *pBIsOverUnderShoot = 1;
    }
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_FindStr((int8_t *)bRx, (int8_t *)"IV", 2, &bTemp));
    if (bTemp)
    {
        /* It could be a to low signal for OVSN (unterflow, 2) or a overflow of the range (1) */
        /* read the PKPK value to deferent the cases */
        statusTmp = phbalReg_Exchange(
            pDataParams->pBalRegDataParams,
            PH_EXCHANGE_DEFAULT,
            (uint8_t*)abCmdPkpk,
            14,
            40,
            (uint8_t*)bRxPkpk,
            &wRxlen);
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_CheckClearError(pDataParams, statusTmp));

        /* If PKPK is ok then it must be and to low value */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_FindStr((int8_t *)bRxPkpk, (int8_t *)"OK", 2, &bTemp));
        if (bTemp)
        {
            *pBIsOverUnderShoot = 2;
        }
        else /* In any other case it is an overflow */
        {
            *pBIsOverUnderShoot = 1;
        }
    }
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_FindStr((int8_t *)bRx, (int8_t *)"OF", 2, &bTemp));
    if (bTemp)
    {
        *pBIsOverUnderShoot = 1;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_LW64Xi_Int_CorrDiv(
                                       uint32_t  dwOldDiv,
                                       uint32_t * dwNewDiv
                                       )
{
    if (dwOldDiv < 2)
    {
        /* set to smallest DIV */
        *dwNewDiv =2;
    }
    else if (dwOldDiv < 200)
    {
        /* 0.002-0.2V -> 0.002V steps */
        *dwNewDiv = dwOldDiv + (2-(dwOldDiv%2))%2;
    }
    else if (dwOldDiv < 500)
    {
        /* 0.2-0.5V -> 0.005V steps */
        *dwNewDiv = dwOldDiv + (5-(dwOldDiv%5))%5;
    }
    else if (dwOldDiv < 1000)
    {
        /* 0.5-1.0V -> 0.010V steps */
        *dwNewDiv = dwOldDiv + (10-(dwOldDiv%10))%10;
    }
    else if (dwOldDiv < 2000)
    {
        /* 1.0-2.0V -> 0.020V steps */
        *dwNewDiv = dwOldDiv + (20-(dwOldDiv%20))%20;
    }
    else if (dwOldDiv < 5000)
    {
        /* 2.0-5.0V -> 0.050V steps */
        *dwNewDiv = dwOldDiv + (50-(dwOldDiv%50))%50;
    }
    else if (dwOldDiv < 10000)
    {
        /* 5.0-10.0V -> 0.100V steps */
        *dwNewDiv = dwOldDiv + (100-(dwOldDiv%100))%100;
    }
    else if (dwOldDiv < 20000)
    {
        /* 10.0-20.0V -> 0.200V steps */
        *dwNewDiv = dwOldDiv + (200-(dwOldDiv%200))%200;
    }
    else if (dwOldDiv < 50000)
    {
        /* 20.0-50.0V -> 0.500V steps */
        *dwNewDiv = dwOldDiv + (500-(dwOldDiv%500))%500;
    }
    else if (dwOldDiv < 100000)
    {
        /* 50.0-100.0V -> 1.000V steps */
        *dwNewDiv = dwOldDiv + (1000-(dwOldDiv%1000))%1000;
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_LW64Xi_Int_ConvertDivInCmd(
    uint32_t  dwDiv,
    uint8_t  bSource[2],
    uint8_t * pCmdLength,
    uint8_t * pCmd
    )
{
    phStatus_t statusTmp;

    /* correct Divisor */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_CorrDiv(dwDiv, &dwDiv));

    /* prepare common section of command */
    pCmd[0] = bSource[0];
    pCmd[1] = bSource[1];
    pCmd[2] = ':';
    pCmd[3] = 'V';
    pCmd[4] = 'D';
    pCmd[5] = 'I';
    pCmd[6] = 'V';
    pCmd[7] = ' ';

    if (dwDiv < 10) /* Div >= 0.001V and <0.010V */
    {
        /* build command */
        pCmd[8] = (uint8_t)'0' + (uint8_t) (dwDiv%10);
        /* add unit*/
        pCmd[9] = 'M';
        pCmd[10] = 'V';
        pCmd[11] = '\n';
        *pCmdLength = 12;
    }
    else if (dwDiv < 100) /* Div >= 0.01V and <0.10V */
    {
        /* build command */
        pCmd[9] = (uint8_t)'0' + (uint8_t) (dwDiv%10);
        dwDiv /=10;
        pCmd[8] = (uint8_t)'0' + (uint8_t) (dwDiv%10);
        /* add unit*/
        pCmd[10] = 'M';
        pCmd[11] = 'V';
        pCmd[12] = '\n';
        *pCmdLength = 13;
    }
    else if (dwDiv < 1000) /* Div >= 0.1V and <1.0V */
    {
        /* build command */
        pCmd[11] = (uint8_t)'0' + (uint8_t) (dwDiv%10);
        dwDiv /=10;
        pCmd[10] = (uint8_t)'0' + (uint8_t) (dwDiv%10);
        dwDiv /=10;
        pCmd[9] = '.';
        pCmd[8] = (uint8_t)'0' + (uint8_t) (dwDiv%10);
        /* add unit */
        pCmd[12] = 'E';
        pCmd[13] = '-';
        pCmd[14] = '1';
        pCmd[15] = ' ';
        pCmd[16] = 'V';
        pCmd[17] = '\n';
        *pCmdLength = 18;
    }
    else if (dwDiv < 10000) /* Div >= 1.0V and <10.0V */
    {
        /* build command */
        dwDiv /=10;
        pCmd[11] = (uint8_t)'0' + (uint8_t)(dwDiv%10);
        dwDiv /=10;
        pCmd[10] = (uint8_t)'0' + (uint8_t)(dwDiv%10);
        dwDiv /=10;
        pCmd[9] = '.';
        pCmd[8] = (uint8_t)'0' + (uint8_t)(dwDiv%10);
        /* add unit */
        pCmd[12] = ' ';
        pCmd[13] = 'V';
        pCmd[14] = '\n';
        *pCmdLength = 15;
    }
    else if (dwDiv < 100000) /* Div >= 10.0V and <100.0V */
    {
        /* build command */
        dwDiv /=10;
        pCmd[12] = (uint8_t)'0' + (uint8_t)(dwDiv%10);
        dwDiv /=10;
        pCmd[11] = (uint8_t)'0' + (uint8_t)(dwDiv%10);
        dwDiv /=10;
        pCmd[10] = '.';
        pCmd[9] = (uint8_t)'0' + (uint8_t)(dwDiv%10);
        dwDiv /=10;
        pCmd[8] = (uint8_t)'0' + (uint8_t)(dwDiv%10);
        /* add unit */
        pCmd[13] = ' ';
        pCmd[14] = 'V';
        pCmd[15] = '\n';
        *pCmdLength = 16;
    }
    else /* divisor bigger than or exact 100 is not useable */
    {
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_COULD_NOT_CONVERT_DIVISOR_INTO_CMD, PH_COMP_DL_OSCI);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_LW64Xi_Int_GetDiv(
                                      phdlOsci_LW64Xi_DataParams_t * pDataParams,
                                      uint8_t  bSource[2],
                                      uint32_t * dwDiv
                                      )
{
    phStatus_t statusTmp;
    float32_t fTemp;
    uint8_t bRx[40];
    uint16_t wRxlen;
    uint8_t abCmd[11] = "C1:VDIV?\n";

    /* check if source is F than change, otherwise discard all and use default */
    if (bSource[0] == 'F')
    {
        abCmd[0] = 'F';
    }

    /* check channel number */
    if (bSource[1] == '2' || bSource[1] == '3' || bSource[1] == '4')
    {
        abCmd[1] = bSource[1];
    }

    /* read out divisions */
    statusTmp = phbalReg_Exchange(
        pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        (uint8_t*)abCmd,
        9,
        40,
        (uint8_t*)bRx,
        &wRxlen);
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_CheckClearError(pDataParams, statusTmp));

    /* convert response */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Atof((int8_t *)bRx, &fTemp));

    /* return divisions */
    fTemp *=1000;
    *dwDiv = (uint32_t) fTemp;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_LW64Xi_Int_GetBestDiv(
    phdlOsci_LW64Xi_DataParams_t * pDataParams,
    uint32_t dwOldDiv,
    float32_t fPeakPeak,
    uint32_t * dwNewDiv
    )
{
    phStatus_t statusTmp;
    float32_t lower_limit, upper_limit;

    /* correct Divisor (to ensure non zero) */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_CorrDiv(dwOldDiv, &dwOldDiv));

    /* calc limits */
    lower_limit = (float32_t)pDataParams->wRangeMin / (float32_t)PHDL_OSCI_RANGE_DIVISOR * (float32_t)dwOldDiv;
    upper_limit = (float32_t)pDataParams->wRangeMax / (float32_t)PHDL_OSCI_RANGE_DIVISOR * (float32_t)dwOldDiv;

    /* check if in amplitude in range */
    if (fPeakPeak >= lower_limit && fPeakPeak <= upper_limit)
    {
        /* amplitude is ok, return old DIV */
        *dwNewDiv = dwOldDiv;
    }
    else
    {
        /* calculate optimum DIV */
        dwOldDiv = (uint32_t)(fPeakPeak * 2 * (float32_t)PHDL_OSCI_RANGE_DIVISOR / (float32_t)(pDataParams->wRangeMax + pDataParams->wRangeMin));

        /* correct Divisor and copy it to new one */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_CorrDiv(dwOldDiv, dwNewDiv));

        /* Recalulate limit and round up if still not in limit */
        upper_limit = (float32_t)pDataParams->wRangeMax / (float32_t)PHDL_OSCI_RANGE_DIVISOR * (float32_t)*dwNewDiv;
        if (fPeakPeak > upper_limit)
        {
            dwOldDiv++;
            /* correct Divisor and copy it to new one */
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_CorrDiv(dwOldDiv, dwNewDiv));
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_LW64Xi_Int_SetCorrRange(
    phdlOsci_LW64Xi_DataParams_t * pDataParams,
    uint8_t  bSource[2]
)
{
    phStatus_t statusTmp;
    uint8_t bOvershoot, bCounter = 0;
    uint16_t wAverageFact;
    uint32_t dwTempDiv;
    uint32_t dwNewDiv;
    float32_t fPeakPeak;
    uint8_t  bCmdLength;
    uint8_t	abCmd[20];

    /* correct osci range */

    /* check status of channel */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_OverUndershoot(pDataParams, bSource, &bOvershoot));

    /* handle undershoot */
    if (bOvershoot == 2)
    {
        /* set division to 500 mV */
        dwTempDiv = 500;

        /* generate command */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_ConvertDivInCmd(dwTempDiv, bSource, &bCmdLength, (uint8_t *)abCmd));

        /* send command and wait PHDL_OSCI_LW64XI_SET_VDIV_DELAY ms */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_SendWait(pDataParams, (uint8_t *)abCmd, bCmdLength, 10));

        /* check command */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_ChkCmd(pDataParams));

        /* check for overshoot */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_OverUndershoot(pDataParams, bSource, &bOvershoot));
    }

    /* handle overshoot */
    if (bOvershoot == 1)
    {
        /* if overshoot get current DIV */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_GetDiv(pDataParams, bSource, &dwTempDiv));

        do{
            /* double DIV to be fast */
            dwTempDiv *=4;

            /* generate command */
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_ConvertDivInCmd(dwTempDiv, bSource, &bCmdLength, (uint8_t *)abCmd));

            /* send command and wait PHDL_OSCI_LW64XI_SET_VDIV_DELAY ms */
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_SendWait(pDataParams, (uint8_t *)abCmd, bCmdLength, 10));

            /* check command */
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_ChkCmd(pDataParams));

            /* check for overshoot */
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_OverUndershoot(pDataParams, bSource, &bOvershoot));

            /* in case  of getting stuck exit with error */
            if (bCounter >10)
            {
                return PH_ADD_COMPCODE(PHDL_OSCI_ERR_CORR_RANGE_FAIL, PH_COMP_DL_OSCI);
            }
            bCounter++;
        } while (bOvershoot == 1);
    }

    /* now signal is in measurable range */

    /* get current amplitude */
    /* Just use a smaller average factor */
    wAverageFact = pDataParams->wAverageFact;
    pDataParams->wAverageFact = 2;
    statusTmp = phdlOsci_LW64Xi_Int_GetPeakPeak(pDataParams, bSource, &fPeakPeak);
    fPeakPeak = fPeakPeak * (float32_t)PHDL_OSCI_FIELD_MULTIPLIER_DIVISOR;
    pDataParams->wAverageFact = wAverageFact;
    PH_CHECK_SUCCESS(statusTmp);

    /* get current DIV */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_GetDiv(pDataParams, bSource, &dwTempDiv));

    /* get the best divisor */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_GetBestDiv(pDataParams, dwTempDiv, fPeakPeak, &dwNewDiv));

    /* if the new divisor is different we have to act */
    if (dwNewDiv != dwTempDiv)
    {
        /* generate command */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_ConvertDivInCmd(dwNewDiv, bSource, &bCmdLength, (uint8_t *)abCmd));

        /* send command and wait PHDL_OSCI_LW64XI_SET_VDIV_DELAY ms */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_SendWait(pDataParams, (uint8_t *)abCmd, bCmdLength, 10));

        /* check command */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_ChkCmd(pDataParams));

        /* check for overshoot */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_OverUndershoot(pDataParams, bSource, &bOvershoot));

        /* remove eventually still existent overshoot */
        bCounter = 0;

        if (bOvershoot == 1)
        {
            do
            {
                /* check for overshoot at the begining to ensure that the divisions */
                /* are increased by 20% after the overshoot is removed as well */
                PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_OverUndershoot(pDataParams, bSource, &bOvershoot));

                /* get current DIV */
                PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_GetDiv(pDataParams, bSource, &dwTempDiv));

                /* increase Div by 20% or at least by one */
                if (dwTempDiv < 5)
                {
                    dwTempDiv++;
                }
                else
                {
                    dwTempDiv = (uint32_t)((float32_t)dwTempDiv * 1.2);
                }

                /* generate command */
                PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_ConvertDivInCmd(dwTempDiv, bSource, &bCmdLength, (uint8_t *)abCmd));

                /* send command */
                PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_SendWait(pDataParams, (uint8_t *) abCmd, bCmdLength, PHDL_OSCI_LW64XI_SET_VDIV_DELAY));

                /* check command */
                PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_ChkCmd(pDataParams));


                /* in case  of getting stuck exit with error */
                if (bCounter >50)
                {
                    return PH_ADD_COMPCODE(PHDL_OSCI_ERR_CORR_RANGE_FAIL, PH_COMP_DL_OSCI);
                }
                bCounter++;
            } while (bOvershoot == 1);
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_LW64Xi_Int_GetWaveForm(
    phdlOsci_LW64Xi_DataParams_t * pDataParams,
    uint8_t  bSource[2],
    uint32_t dwWaveFormBufferSize,
    uint32_t * pdwWaveFormLength,
    uint8_t * pWaveFormBuffer,
    uint8_t * pbHeaderOffset
    )
{
    phStatus_t statusTmp;
    uint8_t abCmd[12] = "C1:WF? ALL\n";

    /* check if source is F than change, otherwise discard all and use default */
    if (bSource[0] == 'F')
    {
        abCmd[0] = 'F';
    }

    /* check channel number */
    if (bSource[1] == '2' || bSource[1] == '3' || bSource[1] == '4')
    {
        abCmd[1] = bSource[1];
    }

    /* turn comm header off */
    /* send command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"CHDR OFF\n", 9));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_ChkCmd(pDataParams));

    /* configure waveform setup */
    /* set word precision */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"CFMT DEF9,WORD,BIN\n", 19));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_ChkCmd(pDataParams));

    /*set little endian*/
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"CORD LO\n", 8)); /* word precision */

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_ChkCmd(pDataParams));

    /* initiate file transfer */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_BalReg_VisaExchangeLarge(pDataParams, (uint8_t *)abCmd, 11, dwWaveFormBufferSize, pWaveFormBuffer, pdwWaveFormLength));

    /* check if Buffer too small (return BufferSize and Byte Count are the same) */
    if (dwWaveFormBufferSize == *pdwWaveFormLength)
    {
        /* clear errors on oscilloscope */
        statusTmp = phdlOsci_LW64Xi_Int_ChkCmd(pDataParams);
        /* exit with error */
        return PH_ADD_COMPCODE(PHDL_OSCI_ERR_READ_WAVEFORM_BUFFER_TOO_SMALL, PH_COMP_DL_OSCI);
    }

    /* set osci header offset (4 Byte) (ALL;) */
    *pbHeaderOffset = 4;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_LW64Xi_Int_ConfigFDT_PCD(
    phdlOsci_LW64Xi_DataParams_t * pDataParams
    )
{
    /* FDT PCD config */
    phStatus_t statusTmp;

    /* FDTPCD_TRG_LEVEL */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C4:TRLV 1V\n", 11));
    /* FDTPCD_TDIV */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TDIV 2US\n", 9));
    /* Filter channel 1 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"VBS 'app.Acquisition.C1.EnhanceResType = 4'\n", 44));
    /* set the number of samples to 1E5 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"VBS 'app.Acquisition.Horizontal.MaxSamples = 1E5'\n", 50));
    /* FDTPCD_TRG_DELAY */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRDL 10.2US\n", 12));
    /* FDTPCD_TRG_MODE */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRMD NORM\n", 10));
    /* FDTPCD_TRG_SELECT */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRSE EDGE,SR,C4,HT,OFF\n", 23));
    /* FDTPCD_TRG_SLOPE */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C4:TRSL POS\n", 12));
    /* FDTPCD_CHANNEL4_VDIV */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_SendWait(pDataParams, (uint8_t *)"C4:VDIV 1V\n", 11, 200));

    /* check command */
    return phdlOsci_LW64Xi_Int_ChkCmd(pDataParams);
}

phStatus_t phdlOsci_LW64Xi_Int_ConfigFDT_PICC_106_L3(
    phdlOsci_LW64Xi_DataParams_t * pDataParams
    )
{
    /* FDT PICC config */
    phStatus_t statusTmp;

    /* FDTPICC_TRG_LEVEL */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C4:TRLV 1V\n", 11));
    /* FDTPICC_TDIV */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TDIV 20US\n", 10));
    /* Filter channel 1 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"VBS 'app.Acquisition.C1.EnhanceResType = 0'\n", 44));
    /* set the number of samples to 1E5 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"VBS 'app.Acquisition.Horizontal.MaxSamples = 1E5'\n", 50));
    /* FDTPICC_TRG_DELAY */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRDL 40US\n", 10));
    /* FDTPICC_TRG_MODE */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRMD NORM\n", 10));
    /* FDTPICC_TRG_SELECT */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRSE EDGE,SR,C4,HT,OFF\n", 23));
    /* FDTPICC_TRG_SLOPE */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C4:TRSL NEG\n", 12));
    /* FDTPICC_CHANNEL4_VDIV */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C4:VDIV 1V\n", 11));

    /* check command */
    return phdlOsci_LW64Xi_Int_ChkCmd(pDataParams);
}

phStatus_t phdlOsci_LW64Xi_Int_ConfigFDT_PICC_106_L4(
    phdlOsci_LW64Xi_DataParams_t * pDataParams
    )
{
    /* FDT PICC config */
    phStatus_t statusTmp;

    /* FDTPICC_TRG_LEVEL */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C4:TRLV 1V\n", 11));
    /* FDTPICC_TDIV */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TDIV 20US\n", 10));
    /* Filter channel 1 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"VBS 'app.Acquisition.C1.EnhanceResType = 0'\n", 44));
    /* set the number of samples to 1E5 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"VBS 'app.Acquisition.Horizontal.MaxSamples = 1E5'\n", 50));
    /* FDTPICC_TRG_DELAY */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRDL 110US\n", 11));
    /* FDTPICC_TRG_MODE */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRMD NORM\n", 10));
    /* FDTPICC_TRG_SELECT */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRSE EDGE,SR,C4,HT,OFF\n", 23));
    /* FDTPICC_TRG_SLOPE */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C4:TRSL NEG\n", 12));
    /* FDTPICC_CHANNEL4_VDIV */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C4:VDIV 1V\n", 11));

    /* check command */
    return phdlOsci_LW64Xi_Int_ChkCmd(pDataParams);
}

phStatus_t phdlOsci_LW64Xi_Int_ConfigFDT_PICC_212_L4(
    phdlOsci_LW64Xi_DataParams_t * pDataParams
    )
{
    /* FDT PICC config */
    phStatus_t statusTmp;

    /* FDTPICC_TRG_LEVEL */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C4:TRLV 1V\n", 11));
    /* FDTPICC_TDIV */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TDIV 20US\n", 10));
    /* Filter channel 1 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"VBS 'app.Acquisition.C1.EnhanceResType = 0'\n", 44));
    /* set the number of samples to 1E5 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"VBS 'app.Acquisition.Horizontal.MaxSamples = 1E5'\n", 50));
    /* FDTPICC_TRG_DELAY */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRDL 60US\n", 10));
    /* FDTPICC_TRG_MODE */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRMD NORM\n", 10));
    /* FDTPICC_TRG_SELECT */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRSE EDGE,SR,C4,HT,OFF\n", 23));
    /* FDTPICC_TRG_SLOPE */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C4:TRSL NEG\n", 12));
    /* FDTPICC_CHANNEL4_VDIV */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C4:VDIV 1V\n", 11));

    /* check command */
    return phdlOsci_LW64Xi_Int_ChkCmd(pDataParams);
}

phStatus_t phdlOsci_LW64Xi_Int_ConfigFDT_PICC_424_L4(
    phdlOsci_LW64Xi_DataParams_t * pDataParams
    )
{
    /* FDT PICC config */
    phStatus_t statusTmp;

    /* FDTPICC_TRG_LEVEL */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C4:TRLV 1V\n", 11));
    /* FDTPICC_TDIV */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TDIV 20US\n", 10));
    /* Filter channel 1 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"VBS 'app.Acquisition.C1.EnhanceResType = 0'\n", 44));
    /* set the number of samples to 1E5 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"VBS 'app.Acquisition.Horizontal.MaxSamples = 1E5'\n", 50));
    /* FDTPICC_TRG_DELAY */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRDL 10US\n", 10));
    /* FDTPICC_TRG_MODE */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRMD NORM\n", 10));
    /* FDTPICC_TRG_SELECT */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRSE EDGE,SR,C4,HT,OFF\n", 23));
    /* FDTPICC_TRG_SLOPE */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C4:TRSL NEG\n", 12));
    /* FDTPICC_CHANNEL4_VDIV */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C4:VDIV 1V\n", 11));

    /* check command */
    return phdlOsci_LW64Xi_Int_ChkCmd(pDataParams);
}

phStatus_t phdlOsci_LW64Xi_Int_ConfigFDT_PICC_848_L4(
    phdlOsci_LW64Xi_DataParams_t * pDataParams
    )
{
    /* FDT PICC config */
    phStatus_t statusTmp;

    /* FDTPICC_TRG_LEVEL */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C4:TRLV 1V\n", 11));
    /* FDTPICC_TDIV */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TDIV 10US\n", 10));
    /* Filter channel 1 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"VBS 'app.Acquisition.C1.EnhanceResType = 0'\n", 44));
    /* set the number of samples to 1E5 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"VBS 'app.Acquisition.Horizontal.MaxSamples = 1E5'\n", 50));
    /* FDTPICC_TRG_DELAY */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRDL 30US\n", 10));
    /* FDTPICC_TRG_MODE */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRMD NORM\n", 10));
    /* FDTPICC_TRG_SELECT */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRSE EDGE,SR,C4,HT,OFF\n", 23));
    /* FDTPICC_TRG_SLOPE */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C4:TRSL NEG\n", 12));
    /* FDTPICC_CHANNEL4_VDIV */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C4:VDIV 1V\n", 11));

    /* check command */
    return phdlOsci_LW64Xi_Int_ChkCmd(pDataParams);
}

phStatus_t phdlOsci_LW64Xi_Int_ConfigLive(
    phdlOsci_LW64Xi_DataParams_t * pDataParams
    )
{
    /* Live config */
    phStatus_t statusTmp;

    /* DISABLE_MATH_F1 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"F1:TRA OFF\n", 11));
    /* LIVE_TIME_DIV */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TDIV 2US\n", 9));
    /* Filter channel 1 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"VBS 'app.Acquisition.C1.EnhanceResType = 4'\n", 44));
    /* set the number of samples to 1E5 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"VBS 'app.Acquisition.Horizontal.MaxSamples = 1E5'\n", 50));
    /* LIVE_TRG_SELECT */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRSE EDGE,SR,EX10,HT,OFF\n", 25));
    /* LIVE_TRG_COUPLING */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"EX:TRCP AC\n", 11));
    /* LIVE_TRG_SLOPE */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"EX:TRSL NEG\n", 12));
    /* LIVE_TRG_LEVEL */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"EX:TRLV 10mV\n", 13));
    /* LIVE_TRG_DELAY */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRDL 0S\n", 8));
    /* OSCI_TRIGGER_AUTO */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRMD AUTO\n", 10));

    /* check command */
    return phdlOsci_LW64Xi_Int_ChkCmd(pDataParams);
}
phStatus_t phdlOsci_LW64Xi_Int_ConfigVHD(
    phdlOsci_LW64Xi_DataParams_t * pDataParams
    )
{
    /* VHD config */
    phStatus_t statusTmp;

    /* DISABLE_MATH_F1 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"F1:TRA OFF\n", 11));
    /* LIVE_TIME_DIV */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TDIV 500US\n", 11));
    /* Filter channel 1 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"VBS 'app.Acquisition.C1.EnhanceResType = 0'\n", 44));
    /* set the number of samples to 1E6 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"VBS 'app.Acquisition.Horizontal.MaxSamples = 2.5E6'\n", 52));
    /* LIVE_TRG_SELECT */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRSE EDGE,SR,EX10,HT,OFF\n", 25));
    /* LIVE_TRG_COUPLING */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"EX:TRCP DC\n", 11));
    /* LIVE_TRG_SLOPE */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"EX:TRSL NEG\n", 12));
    /* LIVE_TRG_LEVEL */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"EX:TRLV 750mV\n", 14));
    /* LIVE_TRG_DELAY */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRDL -1000US\n", 13));
    /* OSCI_TRIGGER_AUTO */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRMD NORM\n", 10));

    /* check command */
    return phdlOsci_LW64Xi_Int_ChkCmd(pDataParams);
}

phStatus_t phdlOsci_LW64Xi_Int_ConfigEMD(
    phdlOsci_LW64Xi_DataParams_t * pDataParams
    )
{
    /* EMD config */
    phStatus_t statusTmp;

    /* DISABLE_MATH_F1 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"F1:TRA OFF\n", 11));
    /* TIME_DIV */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TDIV 50US\n", 10));
	/*PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TDIV 500US\n", 11));*/
    /* Filter channel 1 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"VBS 'app.Acquisition.C1.EnhanceResType = 0'\n", 44));
    /* set the number of samples to 500E5 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"VBS 'app.Acquisition.Horizontal.MaxSamples = 5E5'\n", 50));
    /*PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"VBS 'app.Acquisition.Horizontal.MaxSamples = 5E6'\n", 50));*/
    /* TRG_DELAY */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRDL 200US\n", 11));
    /*PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRDL 2000US\n", 12));*/
    /* TRG_MODE */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRMD NORM\n", 10));
    /* TRG_SELECT */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRSE EDGE,SR,C4,HT,OFF\n", 23));
    /* TRG_LEVEL */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C4:TRLV 1V\n", 11));
    /* RG_SLOPE */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C4:TRSL NEG\n", 12));
    /* CHANNEL4_VDIV */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C4:VDIV 1V\n", 11));
    /* TRACE_C4 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C4:TRA ON\n", 10));

    /* check command */
    return phdlOsci_LW64Xi_Int_ChkCmd(pDataParams);
}

phStatus_t phdlOsci_LW64Xi_Int_ConfigSet(
    phdlOsci_LW64Xi_DataParams_t * pDataParams
    )
{
    /* Set config */
    phStatus_t statusTmp;

    /* SET_TIME_DIV */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TDIV 500NS\n", 11));
    /* Filter channel 1 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"VBS 'app.Acquisition.C1.EnhanceResType = 4'\n", 44));
    /* SET_TRG_SELECT */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRSE EDGE,SR,C1,HT,OFF\n", 23));

    /* SET_TRG_COUPLING */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C1:TRCP AC\n", 11));
    /* SET_TRG_SLOPE */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C1:TRSL NEG\n", 12));
    /* SET_TRG_LEVEL */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C1:TRLV 10mV\n", 13));
    /* SET_TRG_DELAY */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRDL 0S\n", 8));
    /* OSZI_TRIGGER_AUTO */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRMD AUTO\n", 10));

    /* check command */
    return phdlOsci_LW64Xi_Int_ChkCmd(pDataParams);
}

phStatus_t phdlOsci_LW64Xi_Int_ConfigGlobal(
    phdlOsci_LW64Xi_DataParams_t * pDataParams
    )
{
    /* global config */
    phStatus_t statusTmp;

    /* RESET */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"*RST\n", 5));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_ChkCmd(pDataParams));

    /* GENERAL CONFIG */
    /* set up communication parameters */
    /* CLEAR_STATUS_BITS */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"*CLS\n", 5));
    /* COMMHEADER */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"CHDR OFF\n", 9));
    /* AUTOCALIBRATE */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"ACAL OFF\n", 9));
    /* GRID */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"GRID SINGLE\n", 12));
    /* CHANNELS CONFIG */
    /* TRACE_C1 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C1:TRA ON\n", 10));
    /* TRACE_C2 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C2:TRA ON\n", 10));

    /* DEFINE CONFIG INIT */

    if(pDataParams->bAutoAdjustChannels == PH_ON)
    {
        /* COUPLING_C1*/
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C1:CPL A1M\n", 11));
        /* COUPLING_C2 */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C2:CPL DC50\n", 12));
        /* COUPLING_C3 */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C3:CPL DC50\n", 12));
        /* COUPLING_C4 */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C4:CPL DC50\n", 12));
    }

    /* BANDWITH_C1 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"BWL C1,OFF\n", 11));
    /* BANDWITH_C2 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"BWL C2,OFF\n", 11));
    /* VDIV_C1 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C1:VDIV 100MV\n", 14));
    /* VDIV_C2 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C2:VDIV 100MV\n", 14));
    /* Filter channel 1 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"VBS 'app.Acquisition.C1.EnhanceResType = 4'\n", 44));

    /* TRIGGER CONFIG */
    /* TRG_DELAY */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRDL 0s\n", 8));
    /* TRG_MODE */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRMD AUTO\n", 10));

    /* put Oszi to LIVE mode */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_ConfigLive(pDataParams));

    /* check command */
    return phdlOsci_LW64Xi_Int_ChkCmd(pDataParams);
}

phStatus_t phdlOsci_LW64Xi_Int_ConfigHPause_XSmall(
    phdlOsci_LW64Xi_DataParams_t * pDataParams
    )
{
    /* FDT PCD config */
    phStatus_t statusTmp;

    /* FDTPCD_TRG_LEVEL */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C1:TRLV 0.5V\n", 13));
    /* FDTPCD_TDIV */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TDIV 2US\n", 9));
    /* Filter channel 1 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"VBS 'app.Acquisition.C1.EnhanceResType = 0'\n", 44));
    /* FDTPCD_TRG_DELAY */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRDL 5US\n", 9));
    /* FDTPCD_TRG_MODE */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRMD NORM\n", 10));
    /* FDTPCD_TRG_SELECT */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRSE WIDTH,SR,C1,HT,PL,HV,5E-7S\n", 32));
    /* FDTPCD_TRG_SLOPE */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C1:TRSL NEG\n", 12));
    /* FDTPCD_CHANNEL4_VDIV */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_SendWait(pDataParams, (uint8_t *)"C1:VDIV 0.5V\n", 13, 200));

	PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"VBS 'app.Acquisition.Horizontal.MaxSamples = 25000000'\n", 55));

    /* check command */
    return phdlOsci_LW64Xi_Int_ChkCmd(pDataParams);
}

phStatus_t phdlOsci_LW64Xi_Int_ConfigHPause_Small(
    phdlOsci_LW64Xi_DataParams_t * pDataParams
    )
{
    /* FDT PCD config */
    phStatus_t statusTmp;

    /* FDTPCD_TRG_LEVEL */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C1:TRLV 0.5V\n", 13));
    /* FDTPCD_TDIV */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TDIV 2MS\n", 9));
    /* Filter channel 1 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"VBS 'app.Acquisition.C1.EnhanceResType = 0'\n", 44));
    /* FDTPCD_TRG_DELAY */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRDL 5MS\n", 9));
    /* FDTPCD_TRG_MODE */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRMD NORM\n", 10));
    /* FDTPCD_TRG_SELECT */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRSE WIDTH,SR,C1,HT,PL,HV,5E-4S\n", 32));
    /* FDTPCD_TRG_SLOPE */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C1:TRSL NEG\n", 12));
    /* FDTPCD_CHANNEL4_VDIV */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_SendWait(pDataParams, (uint8_t *)"C1:VDIV 0.5V\n", 13, 200));

	PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"VBS 'app.Acquisition.Horizontal.MaxSamples = 1000000'\n", 54));

    /* check command */
    return phdlOsci_LW64Xi_Int_ChkCmd(pDataParams);
}

phStatus_t phdlOsci_LW64Xi_Int_ConfigHPause_Medium(
    phdlOsci_LW64Xi_DataParams_t * pDataParams
    )
{
    /* FDT PCD config */
    phStatus_t statusTmp;

    /* FDTPCD_TRG_LEVEL */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C1:TRLV 0.5V\n", 13));
    /* FDTPCD_TDIV */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TDIV 20MS\n", 10));
    /* Filter channel 1 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"VBS 'app.Acquisition.C1.EnhanceResType = 0'\n", 44));
    /* FDTPCD_TRG_DELAY */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRDL 50MS\n", 10));
    /* FDTPCD_TRG_MODE */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRMD NORM\n", 10));
    /* FDTPCD_TRG_SELECT */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRSE WIDTH,SR,C1,HT,PL,HV,5E-4S\n", 32));
    /* FDTPCD_TRG_SLOPE */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C1:TRSL NEG\n", 12));
    /* FDTPCD_CHANNEL4_VDIV */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_SendWait(pDataParams, (uint8_t *)"C1:VDIV 0.5V\n", 13, 200));

	PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"VBS 'app.Acquisition.Horizontal.MaxSamples = 1000000'\n", 54));

    /* check command */
    return phdlOsci_LW64Xi_Int_ChkCmd(pDataParams);
}

phStatus_t phdlOsci_LW64Xi_Int_ConfigHPause_Large(
    phdlOsci_LW64Xi_DataParams_t * pDataParams
    )
{
    /* FDT PCD config */
    phStatus_t statusTmp;

    /* FDTPCD_TRG_LEVEL */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C1:TRLV 0.5V\n", 13));
    /* FDTPCD_TDIV */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TDIV 50MS\n", 10));
    /* Filter channel 1 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"VBS 'app.Acquisition.C1.EnhanceResType = 0'\n", 44));
    /* FDTPCD_TRG_DELAY */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRDL 200MS\n", 11));
    /* FDTPCD_TRG_MODE */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRMD NORM\n", 10));
    /* FDTPCD_TRG_SELECT */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRSE WIDTH,SR,C1,HT,PL,HV,5E-4S\n", 32));
    /* FDTPCD_TRG_SLOPE */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C1:TRSL NEG\n", 12));
    /* FDTPCD_CHANNEL4_VDIV */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_SendWait(pDataParams, (uint8_t *)"C1:VDIV 0.5V\n", 13, 200));

	PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"VBS 'app.Acquisition.Horizontal.MaxSamples = 1000000'\n", 54));

    /* check command */
    return phdlOsci_LW64Xi_Int_ChkCmd(pDataParams);
}

phStatus_t phdlOsci_LW64Xi_Int_ConfigAntiTearing(
    phdlOsci_LW64Xi_DataParams_t * pDataParams
    )
{
    /* FDT PICC config */
    phStatus_t statusTmp;

    /* FDTPICC_TRG_LEVEL */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C4:TRLV 1V\n", 11));
    /* FDTPICC_TDIV */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TDIV 50US\n", 10));
    /* Filter channel 1 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"VBS 'app.Acquisition.C1.EnhanceResType = 0'\n", 44));
    /* FDTPICC_TRG_DELAY */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRDL -220US\n", 12));
    /* FDTPICC_TRG_MODE */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRMD NORM\n", 10));
    /* FDTPICC_TRG_SELECT */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRSE EDGE,SR,C4,HT,OFF\n", 23));
    /* FDTPICC_TRG_SLOPE */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C4:TRSL POS\n", 12));

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_SendWait(pDataParams, (uint8_t *)"C4:VDIV 1V\n", 11, 200));

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"VBS 'app.Acquisition.C4.View = true'\n", 37));

	PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"VBS 'app.Acquisition.Horizontal.MaxSamples = 1000000'\n", 54));

    /* check command */
    return phdlOsci_LW64Xi_Int_ChkCmd(pDataParams);
}

phStatus_t phdlOsci_LW64Xi_Int_ConfigAntiTearing_Deselect(
    phdlOsci_LW64Xi_DataParams_t * pDataParams
    )
{
    /* FDT PICC config */
    phStatus_t statusTmp;

    /* FDTPICC_TRG_LEVEL */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C4:TRLV 1V\n", 11));
    /* FDTPICC_TDIV */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TDIV 200US\n", 11));
    /* Filter channel 1 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"VBS 'app.Acquisition.C1.EnhanceResType = 0'\n", 44));
    /* FDTPICC_TRG_DELAY */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRDL -920US\n", 12));
    /* FDTPICC_TRG_MODE */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRMD NORM\n", 10));
    /* FDTPICC_TRG_SELECT */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRSE EDGE,SR,C4,HT,OFF\n", 23));
    /* FDTPICC_TRG_SLOPE */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"C4:TRSL POS\n", 12));

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_SendWait(pDataParams, (uint8_t *)"C4:VDIV 1V\n", 11, 200));

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"VBS 'app.Acquisition.C4.View = true'\n", 37));

	PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"VBS 'app.Acquisition.Horizontal.MaxSamples = 1000000'\n", 54));

    /* check command */
    return phdlOsci_LW64Xi_Int_ChkCmd(pDataParams);
}

phStatus_t phdlOsci_LW64Xi_Int_ConfigLabelResetWithCommand(
    phdlOsci_LW64Xi_DataParams_t * pDataParams
    )
{
    /* FDT PICC config */
    phStatus_t statusTmp;

    /* FDTPICC_TRG_LEVEL */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"EX:TRLV 750mV\n", 14));
    /* FDTPICC_TDIV */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TDIV 1MS\n", 9));
    /* Filter channel 1 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"VBS 'app.Acquisition.C1.EnhanceResType = 0'\n", 44));
    /* FDTPICC_TRG_DELAY */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRDL -4.5MS\n", 12));
    /* FDTPICC_TRG_MODE */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRMD NORM\n", 10));
    /* FDTPICC_TRG_SELECT */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRSE EDGE,SR,EX,HT,OFF\n", 23));
    /* FDTPICC_TRG_SLOPE */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"EX:TRSL NEG\n", 12));

	PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"VBS 'app.Acquisition.Horizontal.MaxSamples = 1000000'\n", 54));

    /* check command */
    return phdlOsci_LW64Xi_Int_ChkCmd(pDataParams);
}


phStatus_t phdlOsci_LW64Xi_Int_SetTriggerMode(
    phdlOsci_LW64Xi_DataParams_t * pDataParams,
    uint16_t wTriggerMode
    )
{
    phStatus_t statusTmp;
    int64_t qwCurrentTimebasePosition;
    int64_t qwCurrentTimebaseRange;

    switch (wTriggerMode)
    {
    case PHDL_OSCI_TRIGGER_AUTO:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRMD AUTO\n", 10));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_ChkCmd(pDataParams));
        break;

    case PHDL_OSCI_TRIGGER_NORMAL:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRMD NORM\n", 10));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_ChkCmd(pDataParams));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TIMEBASE_POSITION_NS, &qwCurrentTimebasePosition));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TIMEBASE_RANGE_NS, &qwCurrentTimebaseRange));
        if ((qwCurrentTimebaseRange / 2 - qwCurrentTimebasePosition + 999999) / 1000000 > 0)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phTools_Sleep((uint32_t)((qwCurrentTimebaseRange / 2 - qwCurrentTimebasePosition + 999999) / 1000000)));
        }
        break;

    case PHDL_OSCI_TRIGGER_SINGLE:
        /* set to stopped mode before going to single mode (osci bug?) */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRMD STOP\n", 10));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_ChkCmd(pDataParams));

        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRMD SINGLE\n", 12));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_ChkCmd(pDataParams));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_WaitForStateArmedOrStop(pDataParams));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TIMEBASE_POSITION_NS, &qwCurrentTimebasePosition));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TIMEBASE_RANGE_NS, &qwCurrentTimebaseRange));
        if ((qwCurrentTimebaseRange / 2 - qwCurrentTimebasePosition + 999999) / 1000000 > 0)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phTools_Sleep((uint32_t)((qwCurrentTimebaseRange / 2 - qwCurrentTimebasePosition + 999999) / 1000000)));
        }
        break;

    case PHDL_OSCI_TRIGGER_STOP:
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_Send(pDataParams, (uint8_t *)"TRMD STOP\n", 10));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_ChkCmd(pDataParams));
        break;

    default:
        /* return error code */
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
    }

    /* check command */
    return phdlOsci_LW64Xi_Int_ChkCmd(pDataParams);
}

phStatus_t phdlOsci_LW64Xi_Int_GetTriggerMode(
    phdlOsci_LW64Xi_DataParams_t * pDataParams,
    uint16_t * pwTriggerMode
    )
{
    phStatus_t statusTmp;

    uint8_t bRx[20];
    uint16_t wRxlen;
    uint8_t bTemp;

    /* read command execute register */
    statusTmp = phbalReg_Exchange(
        pDataParams->pBalRegDataParams,
        PH_EXCHANGE_DEFAULT,
        (uint8_t*)"TRMD?\n",
        6,
        20,
        (uint8_t*)bRx,
        &wRxlen);
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_CheckClearError(pDataParams, statusTmp));

    /* evaluate response */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_FindStr((int8_t *)bRx, (int8_t *)"AUTO", 4, &bTemp));
    if (bTemp)
    {
        *pwTriggerMode = PHDL_OSCI_TRIGGER_AUTO;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
    }

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_FindStr((int8_t *)bRx, (int8_t *)"NORM", 4, &bTemp));
    if (bTemp)
    {
        *pwTriggerMode = PHDL_OSCI_TRIGGER_NORMAL;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
    }
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_FindStr((int8_t *)bRx, (int8_t *)"SINGLE", 6, &bTemp));
    if (bTemp)
    {
        *pwTriggerMode = PHDL_OSCI_TRIGGER_SINGLE;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
    }

    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_FindStr((int8_t *)bRx, (int8_t *)"STOP", 4, &bTemp));
    if (bTemp)
    {
        *pwTriggerMode = PHDL_OSCI_TRIGGER_STOP;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
    }

    return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_LW64Xi_Int_ConfigOsci(
    phdlOsci_LW64Xi_DataParams_t * pDataParams,
    uint8_t bConfig
    )
{
    switch (bConfig)
    {
    case PHDL_OSCI_MODE_FDT_PCD:
        return phdlOsci_LW64Xi_Int_ConfigFDT_PCD(pDataParams);

    case PHDL_OSCI_MODE_FDT_PICC_106_L3:
        return phdlOsci_LW64Xi_Int_ConfigFDT_PICC_106_L3(pDataParams);

    case PHDL_OSCI_MODE_FDT_PICC_106_L4:
        return phdlOsci_LW64Xi_Int_ConfigFDT_PICC_106_L4(pDataParams);

    case PHDL_OSCI_MODE_FDT_PICC_212_L4:
        return phdlOsci_LW64Xi_Int_ConfigFDT_PICC_212_L4(pDataParams);

    case PHDL_OSCI_MODE_FDT_PICC_424_L4:
        return phdlOsci_LW64Xi_Int_ConfigFDT_PICC_424_L4(pDataParams);

    case PHDL_OSCI_MODE_FDT_PICC_848_L4:
        return phdlOsci_LW64Xi_Int_ConfigFDT_PICC_848_L4(pDataParams);

    case PHDL_OSCI_MODE_LIVE:
        return phdlOsci_LW64Xi_Int_ConfigLive(pDataParams);

    case PHDL_OSCI_MODE_SET:
        return phdlOsci_LW64Xi_Int_ConfigSet(pDataParams);

    case PHDL_OSCI_MODE_GLOBAL:
        return phdlOsci_LW64Xi_Int_ConfigGlobal(pDataParams);

    case PHDL_OSCI_MODE_VHD:
        return phdlOsci_LW64Xi_Int_ConfigVHD(pDataParams);

    case PHDL_OSCI_MODE_EMD:
        return phdlOsci_LW64Xi_Int_ConfigEMD(pDataParams);

    case PHDL_OSCI_MODE_HFPAUSE_XS:
        return phdlOsci_LW64Xi_Int_ConfigHPause_XSmall(pDataParams);
    case PHDL_OSCI_MODE_HFPAUSE_S:
        return phdlOsci_LW64Xi_Int_ConfigHPause_Small(pDataParams);
    case PHDL_OSCI_MODE_HFPAUSE_M:
        return phdlOsci_LW64Xi_Int_ConfigHPause_Medium(pDataParams);
    case PHDL_OSCI_MODE_HFPAUSE_L:
        return phdlOsci_LW64Xi_Int_ConfigHPause_Large(pDataParams);
    case PHDL_OSCI_MODE_ANTITEARING:
        return phdlOsci_LW64Xi_Int_ConfigAntiTearing(pDataParams);
    case PHDL_OSCI_MODE_ANTITEARINGDESELECT:
        return phdlOsci_LW64Xi_Int_ConfigAntiTearing_Deselect(pDataParams);
    case PHDL_OSCI_MODE_LABELRESETWITHCOMMAND:
        return phdlOsci_LW64Xi_Int_ConfigLabelResetWithCommand(pDataParams);

    default:
        /* return error code */
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_OSCI);
    }
}

phStatus_t phdlOsci_LW64Xi_Int_BalReg_VisaExchangeLarge(
    phdlOsci_LW64Xi_DataParams_t * pDataParams,
    uint8_t * pTxBuffer,
    uint16_t wTxLength,
    uint32_t dwRxBufSize,
    uint8_t * pRxBuffer,
    uint32_t * pRxLength
    )
{
#ifdef _WIN32
    ViStatus statusVi;
    uint32_t dwBytesWritten, dwBytesRead;

    phStatus_t statusTmp;
    uint16_t wValue=0;

    /* write the data */
    statusVi = (pDataParams->pBalRegDataParams->pviWrite)(pDataParams->pBalRegDataParams->instr, (ViBuf)pTxBuffer, (ViUInt32)wTxLength, &dwBytesWritten);
    /* error handling */
    if (statusVi < VI_SUCCESS)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
    }
    if (dwBytesWritten != wTxLength)
    {
        /* write error not all data send */
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
    }

    /* read current IO timeout */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_GetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, &wValue));

    /* set IO timeout to high because transfering data can take a long time */
    statusVi = (pDataParams->pBalRegDataParams->pviSetAttribute) (pDataParams->pBalRegDataParams->instr, VI_ATTR_TMO_VALUE, 150000);
    if (statusVi < VI_SUCCESS)
    {
        /* Cannot open a session to the device */
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }

    /* read the data */
    statusVi = (pDataParams->pBalRegDataParams->pviRead)(pDataParams->pBalRegDataParams->instr, (ViPBuf)pRxBuffer, (ViUInt32)dwRxBufSize, &dwBytesRead);

    /* restore IO timeout */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_SetConfig(pDataParams->pBalRegDataParams, PHBAL_REG_CONFIG_READ_TIMEOUT_MS, wValue));

    /* Return received length */
    *pRxLength = dwBytesRead;

    /* Check for timeout error */
    if (dwBytesRead == 0 || statusVi == VI_ERROR_TMO)
    {
        return PH_ADD_COMPCODE(PH_ERR_IO_TIMEOUT, PH_COMP_BAL);
    }
    /* error handling */
    if (statusVi < VI_SUCCESS)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
#else
    /* satisfy compiler */
    pDataParams = NULL;
    pTxBuffer = NULL;
    wTxLength = 0;
    wRxBufSize = 0;
    pRxBuffer = NULL;
    pRxLength = NULL;

    return PH_ADD_COMPCODE(PH_ERR_NOT_AVAILABLE, PH_COMP_BAL);
#endif
}

phStatus_t phdlOsci_LW64Xi_Int_GetChannelConfig(
    phdlOsci_LW64Xi_DataParams_t * pDataParams,
    uint8_t bChannel,
    uint8_t *pParam,
    uint8_t *pValue,
    uint16_t wValueSize
    )
{
    phStatus_t statusTmp;
    int32_t dwPrintLength = 0;
    uint8_t abCmd[50];
    uint16_t wRxlen;
    uint8_t* pChannel;

    switch(bChannel)
    {
    case 1:
        pChannel = (uint8_t*)"C1";
        break;
    case 2:
        pChannel = (uint8_t*)"C2";
        break;
    case 3:
        pChannel = (uint8_t*)"C3";
        break;
    case 4:
        pChannel = (uint8_t*)"C4";
        break;
    case 5:
        pChannel = (uint8_t*)"EX";
        break;
    case 6:
        pChannel = (uint8_t*)"LINE";
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
    }

    dwPrintLength = sprintf((char *)abCmd, "%s:%s?\n", pChannel, pParam);

    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, (uint8_t *)abCmd, (uint16_t)dwPrintLength , wValueSize, (uint8_t *)pValue, &wRxlen));

    /* there is an errorcode on stack because no answer is expected */
    PH_CHECK_SUCCESS_FCT(statusTmp,phdlOsci_LW64Xi_Int_ChkLastCmd(pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_LW64Xi_Int_GetParseChannelValue(
    uint8_t bChannel,
    uint8_t *pBuffer,
    uint8_t *pValue
    )
{
    char * pTmpVal;
    uint8_t searchChannel[5];

    sprintf((char *)searchChannel, "C%d", bChannel);
    pTmpVal = strstr((char *)pBuffer, (char *)searchChannel);

    if(pTmpVal == NULL)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
    }

    if(strncmp(pTmpVal + 3, PH_VALUE_ON, 2) == 0)
    {
        *pValue = PH_ON;
    }
    else if(strncmp(pTmpVal + 3, PH_VALUE_OFF, 3) == 0)
    {
        *pValue = PH_OFF;
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_LW64Xi_Int_GetNoiseFilterValue(
    phdlOsci_LW64Xi_DataParams_t * pDataParams,
    uint8_t *pValue
    )
{
    phStatus_t statusTmp;
    int32_t dwPrintLength = 0;
    int8_t abBuffer[50];
    uint16_t wTmpValue = 0;

    dwPrintLength = sprintf((char *)abBuffer, "VBS? 'Return=app.Acquisition.C%d.EnhanceResType'\n", pDataParams->bSelectedChannel);

    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT,
        (uint8_t *)abBuffer, (uint16_t)dwPrintLength , 50, (uint8_t *)abBuffer, &wTmpValue));

    if(strncmp((char *)abBuffer, "None", 4) == 0)
    {
        *pValue = PHDL_OSCI_LW64XI_NOISE_FILTER_NONE;
    }
    else if(strncmp((char *)abBuffer, "0.5bits", 7) == 0)
    {
        *pValue = PHDL_OSCI_LW64XI_NOISE_FILTER_HALF_BIT;
    }
    else if(strncmp((char *)abBuffer, "1bits", 5) == 0)
    {
        *pValue = PHDL_OSCI_LW64XI_NOISE_FILTER_ONE_BIT;
    }
    else if(strncmp((char *)abBuffer, "1.5bits", 7) == 0)
    {
        *pValue = PHDL_OSCI_LW64XI_NOISE_FILTER_ONEHALF_BIT;
    }
    else if(strncmp((char *)abBuffer, "2bits", 5) == 0)
    {
        *pValue = PHDL_OSCI_LW64XI_NOISE_FILTER_TWO_BIT;
    }
    else if(strncmp((char *)abBuffer, "2.5bits", 7) == 0)
    {
        *pValue = PHDL_OSCI_LW64XI_NOISE_FILTER_TWOHALF_BIT;
    }
    else if(strncmp((char *)abBuffer, "3bits", 5) == 0)
    {
        *pValue = PHDL_OSCI_LW64XI_NOISE_FILTER_THREE_BIT;
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_DL_OSCI);
    }

   return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_LW64Xi_Int_ConvertStringToInt64(
    uint8_t *abBuffer,
    int64_t qwMultiplier,
    int64_t *pqwValue
    )
{
    float32_t fTmpValue;

    if (sscanf((char *)abBuffer, "%f", &fTmpValue) != 1)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
    }
    if (fTmpValue >= 0.0)
    {
        *pqwValue = (int64_t)(0.5 + ((float32_t)qwMultiplier * fTmpValue));
    }
    else
    {
        *pqwValue = (int64_t)(-0.5 + ((float32_t)qwMultiplier * fTmpValue));
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_LW64Xi_Int_GetParseTriggerSource(
    phdlOsci_LW64Xi_DataParams_t * pDataParams,
    uint8_t *pBuffer,
    uint8_t *pValue
    )
{
    char * pTmpVal;

    /* satisfy compiler */
    if (pDataParams);

    /* Go to second occurrence of ',' as this is the param delimeter */
    pTmpVal = strchr((char *)pBuffer, ',');
    pTmpVal++;
    pTmpVal = strchr(pTmpVal, ',');

    if(pTmpVal == NULL)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
    }

    pTmpVal++;

    if(strncmp(pTmpVal, "C1", 2) == 0)
    {
        *pValue = PHDL_OSCI_CHANNEL_1;
    }
    else if(strncmp(pTmpVal, "C2", 2) == 0)
    {
        *pValue = PHDL_OSCI_CHANNEL_2;
    }
    else if(strncmp(pTmpVal, "C3", 2) == 0)
    {
        *pValue = PHDL_OSCI_CHANNEL_3;
    }
    else if(strncmp(pTmpVal, "C4", 2) == 0)
    {
        *pValue = PHDL_OSCI_CHANNEL_4;
    }
    else if(strncmp(pTmpVal, "EX", 2) == 0)
    {
        *pValue = PHDL_OSCI_CHANNEL_EXTERNAL;
    }
    else if(strncmp(pTmpVal, "LINE", 4) == 0)
    {
        *pValue = PHDL_OSCI_CHANNEL_LINE;
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_LW64Xi_Int_SetChannelConfig(
    phdlOsci_LW64Xi_DataParams_t * pDataParams,
    uint8_t bChannel,
    uint8_t *pParam,
    uint8_t *pValue
    )
{
    phStatus_t statusTmp;
    int32_t dwPrintLength = 0;
    uint8_t abCmd[50];
    uint8_t* pChannel;

    switch(bChannel)
    {
    case 1:
        pChannel = (uint8_t*)"C1";
        break;
    case 2:
        pChannel = (uint8_t*)"C2";
        break;
    case 3:
        pChannel = (uint8_t*)"C3";
        break;
    case 4:
        pChannel = (uint8_t*)"C4";
        break;
    case 5:
        pChannel = (uint8_t*)"EX";
        break;
    case 6:
        pChannel = (uint8_t*)"LINE";
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
    }

    dwPrintLength = sprintf((char *)abCmd, "%s:%s %s\n", pChannel, pParam, pValue);

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalReg_Visa_Cmd_Transmit(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, (uint8_t *)abCmd, (uint16_t)dwPrintLength));

    /* check command */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_Int_ChkCmd(pDataParams))

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_LW64Xi_Int_SetTriggerConfig(
    phdlOsci_LW64Xi_DataParams_t * pDataParams,
    uint8_t *pSource
    )
{
    phStatus_t statusTmp;
    int32_t dwPrintLength = 0;
    uint8_t abConfig[50];
    uint8_t abCmd[50];
    char * pEndValues;
    int8_t pStartValues[20];

    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT,
            (uint8_t *)"TRSE?", (uint16_t)5 , 50, (uint8_t *)abConfig, (uint16_t*)&dwPrintLength));

    abConfig[dwPrintLength] = '\0';

    pEndValues = strchr((char *)abConfig, ',');
    pEndValues++;
    pEndValues = strchr((char *)pEndValues, ',');

    memcpy(pStartValues, abConfig, pEndValues - (char *)abConfig + 1);
    pStartValues[pEndValues - (char *)abConfig + 1] = '\0';

    pEndValues++;
    pEndValues = strchr((char *)pEndValues, ',');

    dwPrintLength = sprintf((char *)abCmd, "TRSE %s%s%s\n", pStartValues, pSource, pEndValues);

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalReg_Visa_Cmd_Transmit(pDataParams->pBalRegDataParams, PH_EXCHANGE_DEFAULT, (uint8_t *)abCmd, (uint16_t)dwPrintLength));

    /* there is a errorcode on stack because no answer is expected */
    PH_CHECK_SUCCESS_FCT(statusTmp,phdlOsci_LW64Xi_Int_ChkLastCmd(pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_LW64Xi_Int_SetMaxSamplesRange(
    phdlOsci_LW64Xi_DataParams_t * pDataParams
    )
{
    phStatus_t statusTmp;
    int64_t qwPreviousMaxSamples;
    int64_t qwtmpConfigValue;

    /* get current maxsamples value */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TIMEBASE_MAX_SAMPLES, &qwPreviousMaxSamples));

    /* set to very small value and get value to get minimum value for maxsamples */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_SetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TIMEBASE_MAX_SAMPLES, pDataParams->dwMaxSamplesMinValue));
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TIMEBASE_MAX_SAMPLES, &qwtmpConfigValue));
    pDataParams->dwMaxSamplesMinValue = (uint32_t)qwtmpConfigValue;

    /* set to very large value and get value to get maximum value for maxsamples */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_SetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TIMEBASE_MAX_SAMPLES, pDataParams->dwMaxSamplesMaxValue));
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_GetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TIMEBASE_MAX_SAMPLES, &qwtmpConfigValue));
    pDataParams->dwMaxSamplesMaxValue = (uint32_t)qwtmpConfigValue;

    /* restore maxsamples value */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_LW64Xi_SetConfig64(pDataParams, PHDL_OSCI_CONFIG64_TIMEBASE_MAX_SAMPLES, qwPreviousMaxSamples));
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_LW64Xi_Int_WaitForStateArmedOrStop(
    phdlOsci_LW64Xi_DataParams_t * pDataParams
    )
{
    phStatus_t statusTmp;

    uint8_t bRx[20];
    uint16_t wRxlen;
    uint8_t bTemp;
    uint32_t dwMaxWait = 0;

    while(dwMaxWait < 2000)
    {
        /* read command execute register */
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
            pDataParams->pBalRegDataParams,
            PH_EXCHANGE_DEFAULT,
            (uint8_t*)"vbs? return = app.Acquisition.AcquisitionState\n",
            47,
            20,
            (uint8_t*)bRx,
            &wRxlen));

        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_CheckClearError(pDataParams, statusTmp));

        /* evaluate response */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_FindStr((int8_t *)bRx, (int8_t *)"Armed", 5, &bTemp));
        if (bTemp)
        {
            return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
        }

        /* if not armed check if already triggered and in stoped state */
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
            pDataParams->pBalRegDataParams,
            PH_EXCHANGE_DEFAULT,
            (uint8_t*)"TRMD?\n",
            6,
            20,
            (uint8_t*)bRx,
            &wRxlen));

        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_CheckClearError(pDataParams, statusTmp));

        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_FindStr((int8_t *)bRx, (int8_t *)"STOP", 4, &bTemp));
        if (bTemp)
        {
            return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
        }
        dwMaxWait++;
    }
    return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_OSCI);
}

phStatus_t phdlOsci_LW64Xi_Int_WaitForStateStop(
    phdlOsci_LW64Xi_DataParams_t * pDataParams
)
{
    phStatus_t statusTmp;

    uint8_t bRx[20];
    uint16_t wRxlen;
    uint8_t bTemp;
    uint32_t dwMaxWait = 0;

    while (dwMaxWait < 2000)
    {
        /* check if already triggered and in stoped state */
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Exchange(
            pDataParams->pBalRegDataParams,
            PH_EXCHANGE_DEFAULT,
            (uint8_t*)"TRMD?\n",
            6,
            20,
            (uint8_t*)bRx,
            &wRxlen));

        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_CheckClearError(pDataParams, statusTmp));

        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_Int_FindStr((int8_t *)bRx, (int8_t *)"STOP", 4, &bTemp));
        if (bTemp)
        {
            return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_OSCI);
        }
        /* Delay in loop until aquisition is finished */
        PH_CHECK_SUCCESS_FCT(statusTmp, phTools_Sleep(10));
        dwMaxWait++;
    }
    return PH_ADD_COMPCODE(PHDL_OSCI_ERR_TRIGGER_NOT_FIRED , PH_COMP_DL_OSCI);
}
#endif /* NXPBUILD__PHDL_OSCI_LW64Xi */
