/*
 * Copyright 2017, 2019 - 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal Software Master Amplifier Oscilloscope Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <phbalReg.h>
#include <phdlMstAmpOsc.h>
#include <ph_RefDefs.h>
#include <math.h>
#include <phTools.h>

#ifdef NXPBUILD__PHDL_MSTAMPOSC_MP300

#include "phdlMstAmpOsc_Mp300.h"
#include "phdlMstAmpOsc_Mp300_Int.h"

#include <phdlAmp.h>
#include <phdlOsci.h>
#include <phdlRdFpga.h>

#include <stdio.h>              /* PRQA S 5124 */

/*
From
http://www.mech.uq.edu.au/staff/jacobs/nm_lib/cmathsrc/spline.c
*/
phStatus_t phdlMstAmpOsc_Mp300_Int_Spline (uint16_t n, int end1, int end2,
    double slope1, double slope2,
    double x[], double y[],
    double b[], double c[], double d[],
    uint16_t *iflag)
{
    int16_t nm1, ib, i;
    double  t;
    uint8_t ascend;

    nm1 = n - 1;
    *iflag = 0;

    if (n < 2)
    {   /* no possible interpolation */
        *iflag = 1;
        return PH_ADD_COMPCODE(PHDL_MSTAMPOSC_ERR_COULD_NOT_FIND_GAIN_VALS, PH_COMP_DL_MSTAMPOSC);
    }

    ascend = 1;
    for (i = 1; i < n; ++i)
    {
        if (x[i] <= x[i-1])
        {
            ascend = 0;
        }
    }
    if (ascend == 0)
    {
        *iflag = 2;
        return PH_ADD_COMPCODE(PHDL_MSTAMPOSC_ERR_COULD_NOT_FIND_GAIN_VALS, PH_COMP_DL_MSTAMPOSC);
    }

    if (n >= 3)
    {   /* ---- At least quadratic ---- */

        /* ---- Set up the symmetric tri-diagonal system
            b = diagonal
            d = offdiagonal
            c = right-hand-side  */
        d[0] = x[1] - x[0];
        c[1] = (y[1] - y[0]) / d[0];
        for (i = 1; i < nm1; ++i)
        {
            d[i]   = x[i+1] - x[i];
            b[i]   = 2.0 * (d[i-1] + d[i]);
            c[i+1] = (y[i+1] - y[i]) / d[i];
            c[i]   = c[i+1] - c[i];
        }

        /* ---- Default End conditions
            Third derivatives at x[0] and x[n-1] obtained
            from divided differences  */
        b[0]   = -d[0];
        b[nm1] = -d[n-2];
        c[0]   = 0.0;
        c[nm1] = 0.0;
        if (n != 3)
        {
            c[0]   = c[2] / (x[3] - x[1]) - c[1] / (x[2] - x[0]);
            c[nm1] = c[n-2] / (x[nm1] - x[n-3]) - c[n-3] / (x[n-2] - x[n-4]);
            c[0]   = c[0] * d[0] * d[0] / (x[3] - x[0]);
            c[nm1] = -c[nm1] * d[n-2] * d[n-2] / (x[nm1] - x[n-4]);
        }

        /* Alternative end conditions -- known slopes */
        if (end1 == 1)
        {
            b[0] = 2.0 * (x[1] - x[0]);
            c[0] = (y[1] - y[0]) / (x[1] - x[0]) - slope1;
        }
        if (end2 == 1)
        {
            b[nm1] = 2.0 * (x[nm1] - x[n-2]);
            c[nm1] = slope2 - (y[nm1] - y[n-2]) / (x[nm1] - x[n-2]);
        }

        /* Forward elimination */
        for (i = 1; i < n; ++i)
        {
            t    = d[i-1] / b[i-1];
            b[i] = b[i] - t * d[i-1];
            c[i] = c[i] - t * c[i-1];
        }

        /* Back substitution */
        c[nm1] = c[nm1] / b[nm1];
        for (ib = 0; ib < nm1; ++ib)
        {
            i    = n - ib - 2;
            c[i] = (c[i] - d[i] * c[i+1]) / b[i];
        }

        /* c[i] is now the sigma[i] of the text */

        /* Compute the polynomial coefficients */
        b[nm1] = (y[nm1] - y[n-2]) / d[n-2] + d[n-2] * (c[n-2] + 2.0 * c[nm1]);
        for (i = 0; i < nm1; ++i)
        {
            b[i] = (y[i+1] - y[i]) / d[i] - d[i] * (c[i+1] + 2.0 * c[i]);
            d[i] = (c[i+1] - c[i]) / d[i];
            c[i] = 3.0 * c[i];
        }
        c[nm1] = 3.0 * c[nm1];
        d[nm1] = d[n-2];

    }   /* at least quadratic */
    else  /* if n >= 3 */
    {   /* linear segment only  */
        b[0] = (y[1] - y[0]) / (x[1] - x[0]);
        c[0] = 0.0;
        d[0] = 0.0;
        b[1] = b[0];
        c[1] = 0.0;
        d[1] = 0.0;
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
}

phStatus_t phdlMstAmpOsc_Mp300_Int_seval (uint16_t n, double u,
              double x[], double y[],
              double b[], double c[], double d[],
              uint16_t *last, double * result)
{
    uint16_t    i, j, k;
    double w;

    i = *last;
    if (i >= n-1)
    {
        i = 0;
    }
    if (i < 0)
    {
        i = 0;
    }

    if ((x[i] > u) || (x[i+1] < u))
    {   /* ---- perform a binary search ---- */
        i = 0;
        j = n;
        do
        {
            k = (i + j) / 2;    /* split the domain to search */
            if (u < x[k])
            {
                j = k;          /* move the upper bound */
            }
            if (u >= x[k])
            {
                i = k;          /* move the lower bound */
            }
        }                       /* there are no more segments to search */
        while (j > i+1);
    }
    *last = i;

    /* ---- Evaluate the spline ---- */
    w = u - x[i];
    *result = y[i] + w * (b[i] + w * (c[i] + w * d[i]));
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
}
/*-------------------------------------------------------------------*/

phStatus_t phdlMstAmpOsc_Mp300_Int_InterpolateSpline(
                                uint16_t n,
                                double x[], double y[],
                                double b[], double c[], double d[],
                                double point, double * result)
{
    uint16_t segment_index = 0;

    if ( point > x[n] )
    {
        return PH_ADD_COMPCODE(PHDL_MSTAMPOSC_ERR_FIELD_TOO_HIGH, PH_COMP_DL_MSTAMPOSC);
    }
    return phdlMstAmpOsc_Mp300_Int_seval(n, point, x, y, b, c, d, &segment_index, result);
}


phStatus_t phdlMstAmpOsc_Mp300_Int_CalcGainValues(
                                phdlMstAmpOsc_Mp300_DataParams_t * pDataParams,
                                uint16_t wFieldStrength,
                                uint16_t * pwGain,
                                uint16_t * wMp300GainValueThousand
                                )
{
    phStatus_t statusTmp;
    double value = 0.0;
    uint16_t wLowGain = 0;
    double dLowH = 0.0;
    double dHighH = 0.0;
    double dLowError = 0.0;
    double dHighError = 0.0;
    float32_t dFieldstrenthAmp = 0.0;
    uint16_t wMaxMPGain;

    /* Mp300 gain has to be 100%, except if fieldstrength is too low to be reachable only with varying with amplifier */
    *wMp300GainValueThousand = pDataParams->wDefaultMp300Gain;
    if (wFieldStrength < pDataParams->pdFieldstrength[0])
    {
        *wMp300GainValueThousand = (uint16_t)((float32_t)wFieldStrength / ((float32_t)pDataParams->pdFieldstrength[0] / (float32_t)pDataParams->wDefaultMp300Gain)+0.5);
        *pwGain = (uint16_t)pDataParams->pdGainValues[0];
    }
    else
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_InterpolateSpline(
                        pDataParams->bMaxMeasuredInterpolationPoint,
                        pDataParams->pdFieldstrength, pDataParams->pdGainValues,
                        pDataParams->pdbH2Gain, pDataParams->pdcH2Gain, pDataParams->pddH2Gain,
                        wFieldStrength, &value));

        wLowGain = (uint16_t)value;

        /* Check if gain + 1 would also be within the measured limits */
        if (wLowGain + 1 > pDataParams->pdGainValues[pDataParams->bMaxMeasuredInterpolationPoint])
        {   /* If not just use one gain step below */
            wLowGain--;
        }

        /* Calc the error from lower and higher gain value */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_InterpolateSpline(
                        pDataParams->bMaxMeasuredInterpolationPoint,
                        pDataParams->pdGainValues, pDataParams->pdFieldstrength,
                        pDataParams->pdbGain2H, pDataParams->pdcGain2H, pDataParams->pddGain2H,
                        wLowGain, &dLowH));
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_InterpolateSpline(
                pDataParams->bMaxMeasuredInterpolationPoint,
                pDataParams->pdGainValues, pDataParams->pdFieldstrength,
                pDataParams->pdbGain2H, pDataParams->pdcGain2H, pDataParams->pddGain2H,
                wLowGain+1, &dHighH));

        dLowError = abs((double)wFieldStrength - dLowH);
        dHighError = abs((double)wFieldStrength - dHighH);

        /* Round where the error is lower */
        *pwGain = (dLowError <= dHighError) ? wLowGain : (wLowGain+1);
        dFieldstrenthAmp = (dLowError < dHighError) ? (float32_t)dLowH : (float32_t)dHighH;
        /* Now error is known so calc the corrected reader gain */
        *wMp300GainValueThousand = (uint16_t)((float32_t)wFieldStrength / (dFieldstrenthAmp / (float32_t)pDataParams->wDefaultMp300Gain)+0.5);

        /* If reader is available and the lower value is used, check if the gain is supported by the reader */
        if (pDataParams->pDlRdGainDataParams != NULL)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_GetMaxOutputGain(pDataParams->pDlRdGainDataParams, &wMaxMPGain));
            /* If the calculated gain is to high use the higher value as fallback */
            if (*wMp300GainValueThousand > wMaxMPGain)
            {
                *pwGain = (wLowGain+1);
                dFieldstrenthAmp = (float32_t)dHighH;
                *wMp300GainValueThousand = (uint16_t)((float32_t)wFieldStrength / (dFieldstrenthAmp / (float32_t)pDataParams->wDefaultMp300Gain)+0.5);
            }

            /* If still out of limit we just use the max gain */
            if (*wMp300GainValueThousand > wMaxMPGain)
            {
                *wMp300GainValueThousand = wMaxMPGain;
            }
        }
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
}

phStatus_t phdlMstAmpOsc_Mp300_Int_SetSecureState(
    phdlMstAmpOsc_Mp300_DataParams_t * pDataParams
    )
{
    phStatus_t statusTmp;

    if (pDataParams->pDlAmpDataParams != NULL)
    {
        /* set amplification to minimum and mp300gain to normal(100%) if too low */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_SetGain(pDataParams->pDlAmpDataParams, pDataParams->wMinAmpGain));
        pDataParams->wCurrAmpGain = pDataParams->wMinAmpGain;
        if (pDataParams->pDlRdGainDataParams != NULL)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_SetOutputGain(pDataParams->pDlRdGainDataParams, pDataParams->wDefaultMp300Gain));
        }
    }
    else /* Only Mp300 */
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_SetOutputGain(pDataParams->pDlRdGainDataParams, pDataParams->wMinAmpGain));
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
}

phStatus_t phdlMstAmpOsc_Mp300_Int_GetFieldStrengthRangeChecked(
    phdlMstAmpOsc_Mp300_DataParams_t * pDataParams,
    uint16_t wLimitLow,
    uint16_t wLimitHigh,
    uint16_t *pwFieldStrengthMeas
    )
{
    phStatus_t statusTmp;
    uint8_t bRetries = 0;

    do
    {
        /* read field strength */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_GetFieldStrength(pDataParams->pDlOsciDataParams, pDataParams->bChannel, pwFieldStrengthMeas));
        bRetries++;
    }
    while(bRetries < 4 && (*pwFieldStrengthMeas < wLimitLow || *pwFieldStrengthMeas > wLimitHigh));

    /* If after 4 retries we ar not within the limits it should not be a measurement error so return the measured value */
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
}

phStatus_t phdlMstAmpOsc_Mp300_Int_SetOutputGain(
    void * pDlRdGainDataParams,
    uint16_t wGain
    )
{
    if (pDlRdGainDataParams == NULL)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_MSTAMPOSC);
    }

#ifdef NXPBUILD__PHHAL_HW_MP300
    if (((phhalHw_Mp300_DataParams_t *)pDlRdGainDataParams)->wId == (PH_COMP_HAL | PHHAL_HW_MP300_ID))
    {
        return phhalHw_SetConfig(pDlRdGainDataParams, PHHAL_HW_MP300_CONFIG_FIELDSTRENGTH_THOUSAND, wGain);
    }
#endif /* NXPBUILD__PHHAL_HW_MP300 */
#ifdef NXPBUILD__PHHAL_HW_PROXILAB
    if (((phhalHw_ProxiLAB_DataParams_t *)pDlRdGainDataParams)->wId == (PH_COMP_HAL | PHHAL_HW_PROXILAB_ID))
    {
        return phhalHw_SetConfig(pDlRdGainDataParams, PHHAL_HW_PROXILAB_CONFIG_POWERLEVEL, wGain);
    }
#endif /* NXPBUILD__PHHAL_HW_PROXILAB */
#ifdef NXPBUILD__PHDL_RDFPGA_V6
    if (((phdlRdFpga_V6_DataParams_t *)pDlRdGainDataParams)->wId == (PH_COMP_DL_RDFPGA | PHDL_RDFPGA_ID))
    {
        return phdlRdFpga_SetPotiPos(pDlRdGainDataParams, wGain);
    }
#endif /* NXPBUILD__PHDL_RDFPGA_V6 */

    return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_MSTAMPOSC);
}

phStatus_t phdlMstAmpOsc_Mp300_Int_GetMaxOutputGain(
    void * pDlRdGainDataParams,
    uint16_t * pwMaxGain
    )
{
    if (pDlRdGainDataParams == NULL)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_MSTAMPOSC);
    }

    if (pwMaxGain == NULL)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
    }

#ifdef NXPBUILD__PHHAL_HW_MP300
    if (((phhalHw_Mp300_DataParams_t *)pDlRdGainDataParams)->wId == (PH_COMP_HAL | PHHAL_HW_MP300_ID))
    {
        return phhalHw_GetConfig(pDlRdGainDataParams, PHHAL_HW_MP300_CONFIG_MAX_FIELDSTRENGTH_THOUSAND, pwMaxGain);
    }
#endif /* NXPBUILD__PHHAL_HW_MP300 */
#ifdef NXPBUILD__PHHAL_HW_PROXILAB
    if (((phhalHw_ProxiLAB_DataParams_t *)pDlRdGainDataParams)->wId == (PH_COMP_HAL | PHHAL_HW_PROXILAB_ID))
    {
        *pwMaxGain = 1023;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
    }
#endif /* NXPBUILD__PHHAL_HW_PROXILAB */
#ifdef NXPBUILD__PHDL_RDFPGA_V6
    if (((phdlRdFpga_V6_DataParams_t *)pDlRdGainDataParams)->wId == (PH_COMP_DL_RDFPGA | PHDL_RDFPGA_ID))
    {
        return phdlRdFpga_GetConfig(pDlRdGainDataParams, PHDL_RDFPGA_CONFIG_MAX_POTI, pwMaxGain);
    }
#endif /* NXPBUILD__PHDL_RDFPGA_V6 */
    return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_MSTAMPOSC);
}

phStatus_t phdlMstAmpOsc_Mp300_Int_GetMinOutputGain(
    void * pDlRdGainDataParams,
    uint16_t * pwMinGain
    )
{
    if (pDlRdGainDataParams == NULL)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_MSTAMPOSC);
    }

    if (pwMinGain == NULL)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
    }

#ifdef NXPBUILD__PHHAL_HW_MP300
    if (((phhalHw_Mp300_DataParams_t *)pDlRdGainDataParams)->wId == (PH_COMP_HAL | PHHAL_HW_MP300_ID))
    {
        *pwMinGain = 0;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
    }
#endif /* NXPBUILD__PHHAL_HW_MP300 */
#ifdef NXPBUILD__PHHAL_HW_PROXILAB
    if (((phhalHw_ProxiLAB_DataParams_t *)pDlRdGainDataParams)->wId == (PH_COMP_HAL | PHHAL_HW_PROXILAB_ID))
    {
        *pwMinGain = 0;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
    }
#endif /* NXPBUILD__PHHAL_HW_PROXILAB */
#ifdef NXPBUILD__PHDL_RDFPGA_V6
    if (((phdlRdFpga_V6_DataParams_t *)pDlRdGainDataParams)->wId == (PH_COMP_DL_RDFPGA | PHDL_RDFPGA_ID))
    {
        return phdlRdFpga_GetConfig(pDlRdGainDataParams, PHDL_RDFPGA_CONFIG_MIN_POTI, pwMinGain);
    }
#endif /* NXPBUILD__PHDL_RDFPGA_V6 */
    return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_MSTAMPOSC);
}

phStatus_t phdlMstAmpOsc_Mp300_Int_SetRxGain(
    void * pDlRdGainDataParams,
    uint16_t wGain
    )
{
    if (pDlRdGainDataParams == NULL)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_MSTAMPOSC);
    }

#ifdef NXPBUILD__PHHAL_HW_MP300
    if (((phhalHw_Mp300_DataParams_t *)pDlRdGainDataParams)->wId == (PH_COMP_HAL | PHHAL_HW_MP300_ID))
    {
        if (((phhalHw_Mp300_DataParams_t *)pDlRdGainDataParams)->wCouplerResourceId == PHHAL_HW_MP300_RESSOURCE_TCL3)
        {
            return phhalHw_SetConfig(pDlRdGainDataParams, PHHAL_HW_MP300_CONFIG_RX_GAIN, wGain);
        }
        else
        {
            return phhalHw_SetConfig(pDlRdGainDataParams, PHHAL_HW_MP300_CONFIG_RX_GAIN_TARGET, wGain);
        }
    }
#endif /* NXPBUILD__PHHAL_HW_MP300 */
#ifdef NXPBUILD__PHHAL_HW_PROXILAB
    if (((phhalHw_ProxiLAB_DataParams_t *)pDlRdGainDataParams)->wId == (PH_COMP_HAL | PHHAL_HW_PROXILAB_ID))
    {
        return phhalHw_SetConfig(pDlRdGainDataParams, PHHAL_HW_PROXILAB_CONFIG_DEMODULATOR_THRESHOLD, wGain);
    }
#endif /* NXPBUILD__PHHAL_HW_PROXILAB */
#ifdef NXPBUILD__PHDL_RDFPGA_V6
    if (((phdlRdFpga_V6_DataParams_t *)pDlRdGainDataParams)->wId == (PH_COMP_DL_RDFPGA | PHDL_RDFPGA_ID))
    {
        return phdlRdFpga_SetJoinerRxGain(pDlRdGainDataParams,(uint8_t) wGain);
    }
#endif /* NXPBUILD__PHDL_RDFPGA_V6 */
    return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_MSTAMPOSC);
}

phStatus_t phdlMstAmpOsc_Mp300_Int_GetMaxRxGain(
    void * pDlRdGainDataParams,
    uint16_t * pwMaxGain
    )
{
    float32_t fDriverVersion;

    if (pDlRdGainDataParams == NULL)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_MSTAMPOSC);
    }

    if (pwMaxGain == NULL)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
    }

#ifdef NXPBUILD__PHHAL_HW_MP300
    if (((phhalHw_Mp300_DataParams_t *)pDlRdGainDataParams)->wId == (PH_COMP_HAL | PHHAL_HW_MP300_ID))
    {
        if (((phhalHw_Mp300_DataParams_t *)pDlRdGainDataParams)->wCouplerResourceId == PHHAL_HW_MP300_RESSOURCE_TCL3)
        {
            if (sscanf((const char *)((phhalHw_Mp300_DataParams_t *)pDlRdGainDataParams)->bCouplerDriverVersion, "%f", &fDriverVersion) != 1)
            {
                return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
            }

            if (fDriverVersion < 1.11)
            {
                *pwMaxGain = 100;
            }
            else
            {
                *pwMaxGain = 200;
            }
        }
        else
        {
            *pwMaxGain = 2000;
        }

        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
    }
#endif /* NXPBUILD__PHHAL_HW_MP300 */
#ifdef NXPBUILD__PHHAL_HW_PROXILAB
    if (((phhalHw_ProxiLAB_DataParams_t *)pDlRdGainDataParams)->wId == (PH_COMP_HAL | PHHAL_HW_PROXILAB_ID))
    {
        *pwMaxGain = 255;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
    }
#endif /* NXPBUILD__PHHAL_HW_PROXILAB */
#ifdef NXPBUILD__PHDL_RDFPGA_V6
    if (((phdlRdFpga_V6_DataParams_t *)pDlRdGainDataParams)->wId == (PH_COMP_DL_RDFPGA | PHDL_RDFPGA_ID))
    {
        *pwMaxGain = 9;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
    }
#endif /* NXPBUILD__PHDL_RDFPGA_V6 */
    return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_MSTAMPOSC);
}

phStatus_t phdlMstAmpOsc_Mp300_Int_GetMinRxGain(
    void * pDlRdGainDataParams,
    uint16_t * pwMinGain
    )
{
    if (pDlRdGainDataParams == NULL)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_DL_MSTAMPOSC);
    }

    if (pwMinGain == NULL)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
    }

#ifdef NXPBUILD__PHHAL_HW_MP300
    if (((phhalHw_Mp300_DataParams_t *)pDlRdGainDataParams)->wId == (PH_COMP_HAL | PHHAL_HW_MP300_ID))
    {
        *pwMinGain = 0;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
    }
#endif /* NXPBUILD__PHHAL_HW_MP300 */
#ifdef NXPBUILD__PHHAL_HW_PROXILAB
    if (((phhalHw_ProxiLAB_DataParams_t *)pDlRdGainDataParams)->wId == (PH_COMP_HAL | PHHAL_HW_PROXILAB_ID))
    {
        *pwMinGain = 0;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
    }
#endif /* NXPBUILD__PHHAL_HW_PROXILAB */
#ifdef NXPBUILD__PHDL_RDFPGA_V6
    if (((phdlRdFpga_V6_DataParams_t *)pDlRdGainDataParams)->wId == (PH_COMP_DL_RDFPGA | PHDL_RDFPGA_ID))
    {
        *pwMinGain = 2;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
    }
#endif /* NXPBUILD__PHDL_RDFPGA_V6 */
    return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_MSTAMPOSC);
}

phStatus_t phdlMstAmpOsc_Mp300_Int_AdjustReaderRxGain(
    phdlMstAmpOsc_Mp300_DataParams_t * pDataParams,
    uint16_t wFieldStrength
    )
{
    phStatus_t statusTmp = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_DL_MSTAMPOSC);
    float64_t fGainExpected = 0.0;
    uint16_t wGainExpected = 0;
    uint16_t wGainLimit = 0;

    /* Adjust the RxGain Target acording to the FS */
    if (pDataParams->pDlRdGainDataParams != NULL && pDataParams->bRxGainMode != PHDL_MSTAMPOSC_MP300_RX_GAIN_TARGET_MODE_DISABLED)
    {
        switch (pDataParams->bRxGainMode)
        {
        case PHDL_MSTAMPOSC_MP300_RX_GAIN_TARGET_MODE_CONSTANT:
            wGainExpected = pDataParams->wMinMp300RxGain;
            break;

        case PHDL_MSTAMPOSC_MP300_RX_GAIN_TARGET_MODE_LINEAR:
        case PHDL_MSTAMPOSC_MP300_RX_GAIN_TARGET_MODE_LINEAR_LIMIT:
        case PHDL_MSTAMPOSC_MP300_RX_GAIN_TARGET_MODE_LINEAR_LIMIT_UP:
        case PHDL_MSTAMPOSC_MP300_RX_GAIN_TARGET_MODE_LINEAR_LIMIT_DOWN:
            /* Interpolation does not work if FS for min and max is the same */
            if (pDataParams->wMinMp300RxGainFS == pDataParams->wMaxMp300RxGainFS)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_DL_MSTAMPOSC);
            }

            /* Linear interpolation */
            fGainExpected = (double)pDataParams->wMinMp300RxGain +
                ((double)(wFieldStrength - pDataParams->wMinMp300RxGainFS)*
                ((double)(pDataParams->wMaxMp300RxGain - pDataParams->wMinMp300RxGain)/(double)(pDataParams->wMaxMp300RxGainFS - pDataParams->wMinMp300RxGainFS)));

            /* Limit depending on mode */
            if ((pDataParams->bRxGainMode == PHDL_MSTAMPOSC_MP300_RX_GAIN_TARGET_MODE_LINEAR_LIMIT || pDataParams->bRxGainMode == PHDL_MSTAMPOSC_MP300_RX_GAIN_TARGET_MODE_LINEAR_LIMIT_DOWN) &&
                wFieldStrength < pDataParams->wMinMp300RxGainFS)
            {
                fGainExpected = pDataParams->wMinMp300RxGain;
            }
            if ((pDataParams->bRxGainMode == PHDL_MSTAMPOSC_MP300_RX_GAIN_TARGET_MODE_LINEAR_LIMIT || pDataParams->bRxGainMode == PHDL_MSTAMPOSC_MP300_RX_GAIN_TARGET_MODE_LINEAR_LIMIT_UP) &&
                wFieldStrength > pDataParams->wMaxMp300RxGainFS)
            {
                fGainExpected = pDataParams->wMaxMp300RxGain;
            }

            /* Limit also for the max and min possible value */
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_GetMinRxGain(pDataParams->pDlRdGainDataParams, &wGainLimit));
            if (fGainExpected < wGainLimit)
            {
                fGainExpected = wGainLimit;
            }
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_GetMaxRxGain(pDataParams->pDlRdGainDataParams, &wGainLimit));
            if (fGainExpected > wGainLimit)
            {
                fGainExpected = wGainLimit;
            }

            wGainExpected = (uint16_t)(0.5 + fGainExpected);
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_MSTAMPOSC);
        }

        PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_GetMinRxGain(pDataParams->pDlRdGainDataParams, &wGainLimit));
        if (wGainExpected < wGainLimit)
        {
            wGainExpected = wGainLimit;
        }

        PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_GetMaxRxGain(pDataParams->pDlRdGainDataParams, &wGainLimit));
        if (wGainExpected > wGainLimit)
        {
            wGainExpected = wGainLimit;
        }

        PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_SetRxGain(pDataParams->pDlRdGainDataParams, wGainExpected));
    }

    return statusTmp;
}

phStatus_t phdlMstAmpOsc_Mp300_Int_ReCalibrate(
    phdlMstAmpOsc_Mp300_DataParams_t * pDataParams,
    uint16_t wFieldStrength,
    uint8_t bRecalCounter)
{
    phStatus_t statusTmp;
    uint8_t bRecalErrorCnt = 0;

    do
    {
        /* if we tried to recalibrate 3 times we assume it is not possible */
        if (bRecalCounter >= PHDL_MSTAMPOSC_NUM_OF_RECAL)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_SetSecureState(pDataParams));
            /* allready tried to recalibrate, precision not achieveable */
            return PH_ADD_COMPCODE(PHDL_MSTAMPOSC_ERR_FIELD_PRECISION_NOT_ACHIEVEABLE, PH_COMP_DL_MSTAMPOSC);
        }

        if (pDataParams->bFastRecalibrateEnabled == PH_OFF)
        {
            /* Clear calibration flag */
            pDataParams->bCal = PH_OFF;

            /* try to recalibrate, assume thermal drift */
            /* We must measure the values new so supplied values are invalid */
            pDataParams->bFieldstrengthValuesSupplied = PH_OFF;
            statusTmp = phdlMstAmpOsc_Mp300_Cal(pDataParams);
        }
        else
        {
            /* Fast mode is used */
            statusTmp = phdlMstAmpOsc_Mp300_Int_ReCalibrateFast(pDataParams, wFieldStrength);
        }

        /* If canceled not retry */
        if ((statusTmp & PH_ERR_MASK) == PH_ERR_CANCELED)
        {
            return statusTmp;
        }
        if (statusTmp != PH_ERR_SUCCESS)
        {
            bRecalErrorCnt++;
            if (bRecalErrorCnt >= PHDL_MSTAMPOSC_NUM_OF_RECAL)
            {
                if (pDataParams->bUseNormalCalIfFastFail == PH_OFF)
                {
                    return statusTmp;
                }

                /* Clear calibration flag */
                pDataParams->bCal = PH_OFF;

                /* Perform a full calibration if fast fails to often */
                /* But only one time */
                pDataParams->bFieldstrengthValuesSupplied = PH_OFF;
                return phdlMstAmpOsc_Mp300_Cal(pDataParams);
            }
        }
    } while (statusTmp != PH_ERR_SUCCESS);
    return statusTmp;
}

/* ReCalibrateFast either keeps the current calibration state or it returns on success with the new calibration data
 * At all time the calibration stays valid.
 */
phStatus_t phdlMstAmpOsc_Mp300_Int_ReCalibrateFast(
    phdlMstAmpOsc_Mp300_DataParams_t * pDataParams,
    uint16_t wFieldStrength)
{
    phStatus_t statusTmp;
    phStatus_t statusCalResult;
    uint16_t wGainValue, wReaderGainValueThousand;
    uint8_t bInsertIndex = 0;
    uint16_t wFieldStrengthMeas;
    uint16_t wCounter;
    double dCalError = 0.0;
    uint8_t bNumberErrorsSummed = 0;
    uint8_t bActInterpolationPoint;
    uint8_t bMaxMeasuredInterpolationPointTmp = 0;
    uint8_t bCalibrationRetryCnt = 0;
    uint8_t bPointWasAdded = PH_OFF;

    /* Find position */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_CalcGainValues(pDataParams, wFieldStrength, &wGainValue, &wReaderGainValueThousand));
    bInsertIndex = 0;
    while(wGainValue > pDataParams->pdGainValues[bInsertIndex] && bInsertIndex <= pDataParams->bMaxMeasuredInterpolationPoint)
        bInsertIndex++;

    /* If insertIndex > bMaxMeasuredInterpolationPoint this should not happen return with error */
    if (bInsertIndex > pDataParams->bMaxMeasuredInterpolationPoint)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_SetSecureState(pDataParams));
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_MSTAMPOSC);
    }

    /* if insertIndex == 0 it must match the first calibrated point */
    if (bInsertIndex == 0 && wGainValue != pDataParams->pdGainValues[bInsertIndex])
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_SetSecureState(pDataParams));
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_DL_MSTAMPOSC);
    }

    /* remeasure all points above the current FS and store in tmp and calc the error to the previus measurement */
    dCalError = 0.0;
    bNumberErrorsSummed = 0;
    for( bActInterpolationPoint = bInsertIndex; bActInterpolationPoint < pDataParams->bNumberPoints; )
    {
        statusCalResult = phdlMstAmpOsc_Mp300_Int_MeasureCalibrationPoint(pDataParams, (uint16_t)pDataParams->pdGainValues[bActInterpolationPoint], &wFieldStrengthMeas);
        if ((statusCalResult & PH_ERR_MASK) != PH_ERR_SUCCESS)
        {
            /* If not something bad is going on return error */
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_SetSecureState(pDataParams));
            return statusCalResult;
        }

        /* Check if the FS is monoton rising */
        if ((bActInterpolationPoint != bInsertIndex) && (wFieldStrengthMeas <= pDataParams->pdTempBuffer[bActInterpolationPoint-1]))
        {
            bCalibrationRetryCnt++;
            if (bCalibrationRetryCnt == 3)
            {
                /* If not something bad is going on return error */
                PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_SetSecureState(pDataParams));
                return PH_ADD_COMPCODE(PHDL_MSTAMPOSC_ERR_COULD_NOT_FIND_GAIN_VALS, PH_COMP_DL_MSTAMPOSC);
            }
            /* Restart from begin */
            bActInterpolationPoint = bInsertIndex;
            dCalError = 0.0;
            bNumberErrorsSummed = 0;
            continue;
        }
        /* Store data in tmp buffer to keep original data in case of recalibration error */
        pDataParams->pdTempBuffer[bActInterpolationPoint] = wFieldStrengthMeas;
        bMaxMeasuredInterpolationPointTmp = bActInterpolationPoint;

        /* Calc error only if values was also measured in previous calibration) */
        if (bActInterpolationPoint <= pDataParams->bMaxMeasuredInterpolationPoint)
        {
            dCalError += (pDataParams->pdTempBuffer[bActInterpolationPoint] / pDataParams->pdFieldstrength[bActInterpolationPoint]);
            bNumberErrorsSummed++;
        }
        bActInterpolationPoint++;

        /* damper check */
        if (wFieldStrengthMeas > pDataParams->wMaxFieldstrengthUsed)
        {
            break;
        }
    }
    dCalError /= bNumberErrorsSummed;

    /* After calibration go back to applied value */
    if (pDataParams->pDlAmpDataParams != NULL)
    {
        /* set gain value */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_SetGain(pDataParams->pDlAmpDataParams, wGainValue));
        pDataParams->wCurrAmpGain = wGainValue;
        if (pDataParams->pDlRdGainDataParams != NULL)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_SetOutputGain(pDataParams->pDlRdGainDataParams, wReaderGainValueThousand));
        }
    }
    else /* Only Mp300 */
    {
        wReaderGainValueThousand = wGainValue; /* The gain value is used for the mp300 */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_SetOutputGain(pDataParams->pDlRdGainDataParams, wReaderGainValueThousand));
    }

    /* Also adjust lower points with calculated error */
    for( bActInterpolationPoint = 0; bActInterpolationPoint < bInsertIndex; bActInterpolationPoint++)
    {
        pDataParams->pdTempBuffer[bActInterpolationPoint] = dCalError * pDataParams->pdFieldstrength[bActInterpolationPoint];
    }
    /* Check if last point of calculated values is lower than first measured */
    if ((bInsertIndex != 0) && (pDataParams->pdTempBuffer[bInsertIndex] <= pDataParams->pdTempBuffer[bInsertIndex-1]))
    {
        /* If not something bad is going on return error */
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_SetSecureState(pDataParams));
        return PH_ADD_COMPCODE(PHDL_MSTAMPOSC_ERR_COULD_NOT_FIND_GAIN_VALS, PH_COMP_DL_MSTAMPOSC);
    }

    /* if we did not hit one of the calibration points we may have to insert the point */
    if (wGainValue != pDataParams->pdGainValues[bInsertIndex])
    {
        /* Check if the fieldstrength between this 2 points already measured is high enougth to insert point */
        if (pDataParams->pdTempBuffer[bInsertIndex] / pDataParams->pdTempBuffer[bInsertIndex-1] > 1.05)
        {
            /* If no more points are available return error */
            if (pDataParams->bNumberPoints == pDataParams->bMaxPoints)
            {
                /* Not Points available could not use fast calibration */
                PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_SetSecureState(pDataParams));
                return PH_ADD_COMPCODE(PHDL_MSTAMPOSC_ERR_FIELD_PRECISION_NOT_ACHIEVEABLE, PH_COMP_DL_MSTAMPOSC);
            }

            /* Calc the gain value that should be inbetween the two existing points */
            PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_CalcGainValues(pDataParams,
                (uint16_t)(((pDataParams->pdFieldstrength[bInsertIndex-1] + pDataParams->pdFieldstrength[bInsertIndex])/2.0) + 0.5),
                &wGainValue,
                &wReaderGainValueThousand));

            /* Check if new point is valid */
            if (wGainValue <= pDataParams->pdGainValues[bInsertIndex-1] || wGainValue >= pDataParams->pdGainValues[bInsertIndex])
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_SetSecureState(pDataParams));
                return PH_ADD_COMPCODE(PHDL_MSTAMPOSC_ERR_FIELD_PRECISION_NOT_ACHIEVEABLE, PH_COMP_DL_MSTAMPOSC);
            }

            /* Measure point and insert */
            statusTmp = phdlMstAmpOsc_Mp300_Int_MeasureCalibrationPoint(pDataParams, wGainValue, &wFieldStrengthMeas);
            if ((statusTmp & PH_ERR_MASK) != PH_ERR_SUCCESS)
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_SetSecureState(pDataParams));
                return statusTmp;
            }

            /* Check if the FS is monoton rising */
            if (wFieldStrengthMeas <= pDataParams->pdTempBuffer[bInsertIndex-1] ||
                wFieldStrengthMeas >= pDataParams->pdTempBuffer[bInsertIndex])
            {
                PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_SetSecureState(pDataParams));
                return PH_ADD_COMPCODE(PHDL_MSTAMPOSC_ERR_FIELD_PRECISION_NOT_ACHIEVEABLE, PH_COMP_DL_MSTAMPOSC);
            }

            /* Insert data */
            for (wCounter = pDataParams->bMaxPoints-1; wCounter > bInsertIndex; wCounter--)
            {
                pDataParams->pdTempBuffer[wCounter] = pDataParams->pdTempBuffer[wCounter-1];
                pDataParams->pdGainValues[wCounter] = pDataParams->pdGainValues[wCounter-1];
            }
            bMaxMeasuredInterpolationPointTmp = bMaxMeasuredInterpolationPointTmp + 1;
            pDataParams->bNumberPoints++;
            pDataParams->pdTempBuffer[bInsertIndex] = wFieldStrengthMeas;
            pDataParams->pdGainValues[bInsertIndex] = wGainValue;
            bPointWasAdded = PH_ON;
        }
    }

    /* Recalulate Spline */
    statusTmp = phdlMstAmpOsc_Mp300_Int_CalcSplineValues(
        bMaxMeasuredInterpolationPointTmp, pDataParams->pdTempBuffer, pDataParams->pdGainValues,
        pDataParams->pdbH2Gain, pDataParams->pdcH2Gain, pDataParams->pddH2Gain,
        pDataParams->pdbGain2H, pDataParams->pdcGain2H, pDataParams->pddGain2H);

    if (statusTmp != PH_ERR_SUCCESS)
    {
        /* Restore old cal values */
        if (bPointWasAdded)
        {
            pDataParams->bNumberPoints--;
            /* remove insert point */
            for (wCounter = bInsertIndex; wCounter < pDataParams->bMaxPoints-1; wCounter++)
            {
                pDataParams->pdGainValues[wCounter] = pDataParams->pdGainValues[wCounter+1];
            }
            /* Recalculate old spline values (must work) */
            phdlMstAmpOsc_Mp300_Int_CalcSplineValues(
                pDataParams->bMaxMeasuredInterpolationPoint, pDataParams->pdFieldstrength, pDataParams->pdGainValues,
                pDataParams->pdbH2Gain, pDataParams->pdcH2Gain, pDataParams->pddH2Gain,
                pDataParams->pdbGain2H, pDataParams->pdcGain2H, pDataParams->pddGain2H);
        }
        return statusTmp;
    }

    /* If Everything is pass copy new data */
    /* set the new max point */
    pDataParams->bMaxMeasuredInterpolationPoint = bMaxMeasuredInterpolationPointTmp;

    /* Copy Data to correct buffer */
    for (wCounter = 0; wCounter < pDataParams->bNumberPoints; wCounter++)
    {
        pDataParams->pdFieldstrength[wCounter] = pDataParams->pdTempBuffer[wCounter];
    }

    return statusTmp;
}

phStatus_t phdlMstAmpOsc_Mp300_Int_CalcSplineValues(
    uint8_t bMaxMeasuredInterpolationPoint,
    double pdFieldstrength[], double pdGainValues[],
    double pdbH2Gain[], double pdcH2Gain[], double pddH2Gain[],
    double pdbGain2H[], double pdcGain2H[], double pddGain2H[])
{
    phStatus_t statusTmp;
    uint16_t wSplineResult;
    uint8_t i;
    double x1, x2;
    double y1, y2;
    double b, c, d;

    /* Calc the Spline Values for interpolation from H to Gain */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_Spline(bMaxMeasuredInterpolationPoint+1, 0, 0,
            0, 0,
            pdFieldstrength, pdGainValues,
            pdbH2Gain, pdcH2Gain, pddH2Gain,
            &wSplineResult));

    /* Check if spline calculation could never get negative */
    /* Minimum for each segment must be > start value of segment
     * Spline formula:
     * w = u - x[i]:
     * *result = y[i] + w * b[i] + w*w * c[i] + w*w*w * d[i];
     * gradient if the q spline:
     * 3dx + 2cx + b = 0
     *             -2c  +/- sqrt((2c) - 4*3d*b)
     * -> x1,2 = -------------------------------
     *                         2*3d
     * use this points (only positive one) to calculate y
     * none of this points is allowed to be less that zero */
    for ( i = 0; i < bMaxMeasuredInterpolationPoint; i++)
    {
        b = pdbH2Gain[i];
        c = pdcH2Gain[i];
        d = pddH2Gain[i];
        x1 = (-2.0*c + sqrt(pow(2.0*c,2) - 4.0*3.0*d*b)) / (2.0*3.0*d);
        x2 = (-2.0*c - sqrt(pow(2.0*c,2) - 4.0*3.0*d*b)) / (2.0*3.0*d);
        y1 = pdGainValues[i] + x1 * (b + x1 * (c + x1 * d));
        y2 = pdGainValues[i] + x2 * (b + x2 * (c + x2 * d));

        if (x1 >= 0.0 && x1 <= pdFieldstrength[i+1] - pdFieldstrength[i] && y1 < 0.0)
        {
            wSplineResult = 1;
        }
        if (x2 >= 0.0 && x2 <= pdFieldstrength[i+1] - pdFieldstrength[i] && y2 < 0.0)
        {
            wSplineResult = 1;
        }
    }


    if (wSplineResult != 0)
    {   /* If spline calulation fails, return error */
        return PH_ADD_COMPCODE(PHDL_MSTAMPOSC_ERR_COULD_NOT_FIND_GAIN_VALS, PH_COMP_DL_MSTAMPOSC);
    }

    /* Calc the Spline Values for interpolation from Gain to H */
    PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_Spline(bMaxMeasuredInterpolationPoint+1, 0, 0,
            0, 0,
            pdGainValues, pdFieldstrength,
            pdbGain2H, pdcGain2H, pddGain2H,
            &wSplineResult));

    for ( i = 0; i < bMaxMeasuredInterpolationPoint; i++)
    {
        b = pdbGain2H[i];
        c = pdcGain2H[i];
        d = pddGain2H[i];
        x1 = (-2.0*c + sqrt(pow(2.0*c,2) - 4.0*3.0*d*b)) / (2.0*3.0*d);
        x2 = (-2.0*c - sqrt(pow(2.0*c,2) - 4.0*3.0*d*b)) / (2.0*3.0*d);
        y1 = pdFieldstrength[i] + x1 * (b + x1 * (c + x1 * d));
        y2 = pdFieldstrength[i] + x2 * (b + x2 * (c + x2 * d));

        if (x1 >= 0.0 && x1 <= pdGainValues[i+1] - pdGainValues[i] && y1 < 0.0)
        {
            wSplineResult = 1;
        }
        if (x2 >= 0.0 && x1 <= pdGainValues[i+1] - pdGainValues[i] && y2 < 0.0)
        {
            wSplineResult = 1;
        }
    }

    if (wSplineResult != 0)
    {   /* If spline calulation fails, return error */
        return PH_ADD_COMPCODE(PHDL_MSTAMPOSC_ERR_COULD_NOT_FIND_GAIN_VALS, PH_COMP_DL_MSTAMPOSC);
    }

    return statusTmp;
}

phStatus_t phdlMstAmpOsc_Mp300_Int_MeasureCalibrationPoint(
    phdlMstAmpOsc_Mp300_DataParams_t * pDataParams,
    uint16_t wGain,
    uint16_t * pwFieldStrengthMeas)
{
    phStatus_t statusTmp;
    uint8_t bMeasurmentRetryCnt = 0;
    uint16_t pwFieldStrengthMeasSenseCoil;
    uint16_t wSenseCoilBalanceCheckLimit;
    uint16_t wFieldMultiplier;

    /* Set MP300 Fieldstrength to pDataParams->wDefaultMp300Gain (just if also amplifier is used) */
    if (pDataParams->pDlRdGainDataParams != NULL && pDataParams->pDlAmpDataParams != NULL)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_SetOutputGain(pDataParams->pDlRdGainDataParams, pDataParams->wDefaultMp300Gain));
    }

    if (pDataParams->pDlAmpDataParams != NULL)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlAmp_SetGain(pDataParams->pDlAmpDataParams, wGain));
        pDataParams->wCurrAmpGain = wGain;
    }
    else /* Use Mp300 */
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlMstAmpOsc_Mp300_Int_SetOutputGain(pDataParams->pDlRdGainDataParams, wGain));
    }
    phTools_Sleep(50);

    do
    {
        /* Check for cancellation */
        if (pDataParams->bCancellationToken == PH_ON)
        {
            pDataParams->bCancellationToken = PH_OFF;
            return PH_ADD_COMPCODE(PH_ERR_CANCELED, PH_COMP_DL_MSTAMPOSC);
        }
        statusTmp = phdlOsci_GetFieldStrength(pDataParams->pDlOsciDataParams, pDataParams->bChannel, pwFieldStrengthMeas);
    }
    while (((statusTmp & PH_ERR_MASK) != PH_ERR_SUCCESS) && bMeasurmentRetryCnt++ < 3);
    if (pDataParams->bSenseCoilBalanceChannel != 0)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phdlOsci_GetConfig(pDataParams->pDlOsciDataParams, PHDL_OSCI_CONFIG_FIELD_MULTIPLIER ,&wFieldMultiplier));
        if (wFieldMultiplier < 200)
        {
            wSenseCoilBalanceCheckLimit = pDataParams->bSenseCoilBalanceCheckLimit * 4;
        }
        else
        {
            wSenseCoilBalanceCheckLimit = pDataParams->bSenseCoilBalanceCheckLimit;
        }
        do
        {
            /* Check for cancellation */
            if (pDataParams->bCancellationToken == PH_ON)
            {
                pDataParams->bCancellationToken = PH_OFF;
                return PH_ADD_COMPCODE(PH_ERR_CANCELED, PH_COMP_DL_MSTAMPOSC);
            }
            statusTmp = phdlOsci_GetFieldStrength(pDataParams->pDlOsciDataParams, pDataParams->bSenseCoilBalanceChannel, &pwFieldStrengthMeasSenseCoil);
        }
        while (((statusTmp & PH_ERR_MASK) != PH_ERR_SUCCESS) && bMeasurmentRetryCnt++ < 3);

        /* If get Fieldstength for C2 failes return error */
        if ((statusTmp & PH_ERR_MASK) != PH_ERR_SUCCESS)
        {
            return statusTmp;
        }

        if ((pwFieldStrengthMeasSenseCoil > wSenseCoilBalanceCheckLimit || *pwFieldStrengthMeas > wSenseCoilBalanceCheckLimit) &&
            pwFieldStrengthMeasSenseCoil > *pwFieldStrengthMeas)
        {
            return PH_ADD_COMPCODE(PHDL_MSTAMPOSC_ERR_CHANNEL_VALUE_MISSMATCH, PH_COMP_DL_MSTAMPOSC);
        }

        if (pwFieldStrengthMeasSenseCoil > pDataParams->wMaxFieldstrengthUsed)
        {
            return PH_ADD_COMPCODE(PHDL_MSTAMPOSC_ERR_CHANNEL_VALUE_MISSMATCH, PH_COMP_DL_MSTAMPOSC);
        }
    }
    return statusTmp;
}
#endif /* NXPBUILD__PHDL_MSTAMPOSC_MP300 */
