/*
 * Copyright 2019, 2022, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * ISO3 Setup Software Master Amplifier Oscilloscope Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHDLMSTAMPOSC_ISO3_INT_H
#define PHDLMSTAMPOSC_ISO3_INT_H

#include <ph_Status.h>

/**
* \name Master Amplifier Osci ISO3 Internal Defines
*/
/*@{*/
#define PHDL_MSTAMPOSC_ISO3_GAIN_CTRL_ADDR           0x0000000e         /**< Gain control register */
#define PHDL_MSTAMPOSC_ISO3_DAC_MIN_GAIN             0x00007fff         /**< Measured min gain */
#define PHDL_MSTAMPOSC_ISO3_DAC_MAX_GAIN             0x00300000         /**< Max possible gain */
#define PHDL_MSTAMPOSC_ISO3_STEERING_MAX_VAL         0x0001ffff         /**< Oversteering value */
#define PHDL_MSTAMPOSC_ISO3_CAL_COIL_VOLT_ADDR       0x0000000a         /**< Coil volt register */
#define PHDL_MSTAMPOSC_ISO3_STEERING_VAR_ADDR        0x0000000b         /**< Steering register */
/*@}*/

/**
* \brief Mls scaling value register addr 0x12 of ISO3 Box
*/
typedef struct GainCtrlReg_t
{
    uint32_t amplify_deviation :  4;
    uint32_t agc_reference     : 24;
    uint32_t rfu               :  4;
} phdlMstAmpOsc_ISO3_GainCtrlReg_t;

phStatus_t phdlMstAmpOsc_ISO3_Int_BinSearchMaxFieldStrength(
    phdlMstAmpOsc_ISO3_DataParams_t * pDataParams,
    uint16_t wFieldStrengthLimitMilliAPM,
    uint16_t * pwMaxFieldStrengthMilliAPM
    );

phStatus_t phdlMstAmpOsc_ISO3_Int_FindMaxFieldStrength(
    phdlMstAmpOsc_ISO3_DataParams_t * pDataParams,
    uint16_t wFieldStrengthLimitMilliAPM,
    uint16_t * pwMaxFieldStrengthMilliAPM
    );

phStatus_t phdlMstAmpOsc_ISO3_Int_GetFieldStrength(
    phdlMstAmpOsc_ISO3_DataParams_t * pDataParams,
    uint16_t * pdwFieldStrengthMilliAPM,
    uint8_t bAntennaType
    );

/**
 * \brief Make an analysis of a large number of values from the steering
 * register. If the variance is too high it indicates that the AGC got
 * unstable and the field strength needs to be reduced.
 * \param pDataParams Pointer to the ISO3 MstAmpOsc parameter struct
 * \return PH_ERR_SUCCESS if no communication error occured
 * \return PH_ERR_INTERNAL if there was a communication error
 */
phStatus_t phdlMstAmpOsc_ISO3_Int_AgcStabilityCheck(
    phdlMstAmpOsc_ISO3_DataParams_t * pDataParams,
    uint8_t * pAgcIsStable
    );

/**
 * \brief Reads and avrages AGC gain value
 * \param pDataParams Pointer to the ISO3 MstAmpOsc parameter struct
 * \param pAvgAgcGain Pointer to average gain value
 * \return PH_ERR_SUCCESS if no communication error occured
 */
phStatus_t phdlMstAmpOsc_ISO3_Int_AvgAgcGain(phdlMstAmpOsc_ISO3_DataParams_t * pDataParams, uint32_t * pAvgAgcGain);

/**
 * \brief Adjusts the reader sensitivity
 * \param pDataParams Pointer to the ISO3 MstAmpOsc parameter struct
 * \param wFieldStrength fs to calculate sensitivity
 * \return PH_ERR_SUCCESS if no communication error occured
 */
phStatus_t phdlMstAmpOsc_ISO3_Int_AdjustReaderRxThreshold(
    phdlMstAmpOsc_ISO3_DataParams_t * pDataParams,
    uint16_t wFieldStrength
    );

/**
* \brief Set the reader output target to zero
* \param pDataParams Pointer to the ISO3 MstAmpOsc parameter struct
* \return PH_ERR_SUCCESS if secure state is set correct
*/
phStatus_t phdlMstAmpOsc_Iso3_Int_SetAgcRefZero(phdlMstAmpOsc_ISO3_DataParams_t * pDataParams);

#endif /* PHDLMSTAMPOSC_ISOSETUP_INT_H */
