/*
 * Copyright 2022 - 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Smartware BAL Component of Reader Library Framework.
 * Author: santosh.d.araballi
 * Revision: 1.12
 * Date: Thu Apr 14 15:36:57 2011
 */

#include <ph_Status.h>
#include <phbalReg.h>
#include <ph_RefDefs.h>
#include <string.h>
#include "phbalReg_Smartware_Int.h"
#include "external\phbalReg_Smartware_Ex.h"

#ifdef NXPBUILD__PHBAL_REG_SMARTWARE

phStatus_t phbalReg_Smartware_Init(
                                   phbalReg_Smartware_DataParams_t * pDataParams,
                                   uint16_t wSizeOfDataParams
                                   )
{
    phStatus_t statusTmp;

    if (sizeof(phbalReg_Smartware_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_BAL);
    }
    PH_ASSERT_NULL (pDataParams);

    /* initialize the data parameters to default values */
    pDataParams->wId = PH_COMP_BAL | PHBAL_REG_SMARTWARE_ID;
    pDataParams->dwInstrumentNode = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Smartware_Int_CheckInternalBuffers(pDataParams));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

phStatus_t phbalReg_Smartware_GetPortList(
                                          phbalReg_Smartware_DataParams_t * pDataParams,
                                          uint16_t wPortBufSize,
                                          uint8_t * pPortNames,
                                          uint16_t * pNumOfPorts
                                          )
{
    uint8_t abPortName[] = "Smartware Reader";
    uint16_t wLen;

    *pNumOfPorts = 0;

    /* Try to open port if successful return reader */
    if(phbalReg_OpenPort(pDataParams) == PH_ERR_SUCCESS)
    {
        phbalReg_ClosePort(pDataParams);
        if (wPortBufSize)
        {
            wLen = (uint16_t)strlen( (const char *) abPortName);
            memcpy(pPortNames, abPortName, wLen); /* PRQA S 3200 */
            *pNumOfPorts = 1;
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_BAL);
        }
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

phStatus_t phbalReg_Smartware_SetPort(
                                      phbalReg_Smartware_DataParams_t * pDataParams,
                                      uint8_t * pPortName
                                      )
{
    /* satisfy compiler */
    if(pDataParams || pPortName);

    /* No Port needs to be set for this reader so just return success */
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

phStatus_t phbalReg_Smartware_Exchange(
                                       phbalReg_Smartware_DataParams_t * pDataParams,
                                       uint16_t wOption,
                                       uint8_t * pTxBuffer,
                                       uint16_t wTxLength,
                                       uint16_t wRxBufSize,
                                       uint8_t * pRxBuffer,
                                       uint16_t * pRxLength
                                       )
{
    /* satisfy compiler */
    if(pDataParams || wOption || pTxBuffer || wTxLength || wRxBufSize || pRxBuffer || pRxLength);
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

phStatus_t phbalReg_Smartware_OpenPort(
                                       phbalReg_Smartware_DataParams_t * pDataParams
                                       )
{
    phStatus_t statusTmp;

    /* load the function pointers */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Smartware_Ex_LoadFunctions(pDataParams));
    if(statusTmp != PH_ERR_SUCCESS)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_Smartware_Ex_UnLoadFunctions(pDataParams));
    }

    statusTmp = phbalReg_Smartware_Int_StartReader(pDataParams);
    if(statusTmp != PH_ERR_SUCCESS)
    {
        phbalReg_Smartware_Ex_UnLoadFunctions(pDataParams);
        return PH_ADD_COMPCODE(PH_ERR_IO_TIMEOUT, PH_COMP_BAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

phStatus_t phbalReg_Smartware_ClosePort(
                                        phbalReg_Smartware_DataParams_t * pDataParams
                                        )
{
    phbalReg_Smartware_Int_ConvertError(pDataParams, phbalReg_Smartware_Int_StopReader(pDataParams));
    phbalReg_Smartware_Ex_UnLoadFunctions(pDataParams);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

phStatus_t phbalReg_Smartware_SetConfig(
                                        phbalReg_Smartware_DataParams_t * pDataParams,
                                        uint16_t wConfig,
                                        uint16_t wValue
                                        )
{
    /* satisfy compiler */
    if(pDataParams || wConfig || wValue);
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

phStatus_t phbalReg_Smartware_GetConfig(
                                        phbalReg_Smartware_DataParams_t * pDataParams,
                                        uint16_t wConfig,
                                        uint16_t * pValue
                                        )
{
    /* satisfy compiler */
    if(pDataParams || wConfig || pValue);
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

#endif /* NXPBUILD__PHBAL_REG_SMARTWARE */
