/*
 * Copyright 2019, 2022, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * BAL MultiSocket Internal Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phTools.h>
#include <phbalReg.h>

#ifdef NXPBUILD__PHBAL_REG_MULTISOCKET

#include "phbalReg_MultiSocket_Int.h"
#include "phbalReg_MultiSocket.h"

phStatus_t phbalReg_MultiSocket_Int_Send(
    phbalReg_MultiSocket_DataParams_t * pDataParams,
    uint8_t * pbTxBuffer,
    uint16_t wTxSizeBytes
    )
{
    uint32_t dwPacketSize = 1024;
    uint32_t dwPacketNumb = wTxSizeBytes / dwPacketSize;
    uint32_t dwPacketRest = wTxSizeBytes % dwPacketSize;
    uint32_t dwCounter;
    uint32_t dwSentBytesNow;
    uint32_t dwSentBytesSum = 0;
    uint32_t dwSentBytes;
    uint16_t wTimeWaited = 0;

    /* Send packets */
    for( dwCounter = 0; dwCounter < dwPacketNumb; dwCounter++ )
    {
        dwSentBytes = 0;
        while(dwSentBytes < dwPacketSize &&
            wTimeWaited <= pDataParams->wTxTimeout)
        {
            dwSentBytesNow = send(pDataParams->dwSendFd,
                ((const char *) pbTxBuffer) + (dwCounter * dwPacketSize) + dwSentBytes,
                dwPacketSize - dwSentBytes,
                0);

            if(dwSentBytesNow == -1)
            {
                /* Interface error */
                return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
            }
            else if(dwSentBytesNow == 0)
            {
                /* Nothing transmitted */
                phTools_Sleep(100);
                wTimeWaited += 100;
            }
            else
            {
                /* Successfully transmitted, reset waited time */
                wTimeWaited = 0;
                dwSentBytes += dwSentBytesNow;
            }
        }
        dwSentBytesSum += dwPacketSize;
    }

    /* Send remaining bytes */
    if(dwPacketRest > 0)
    {
        dwSentBytes = 0;
        while(dwSentBytes < dwPacketRest &&
            wTimeWaited <= pDataParams->wTxTimeout)
        {
            dwSentBytesNow = send(pDataParams->dwSendFd,
                ((const char *) pbTxBuffer) + (dwCounter * dwPacketSize) + dwSentBytes,
                dwPacketRest - dwSentBytes,
                0);

            if (dwSentBytesNow == -1)
            {
                /* Interface error */
                return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
            }
            else if(dwSentBytesNow == 0)
            {
                /* Nothing transmitted */
                phTools_Sleep(100);
                wTimeWaited += 100;
            }
            else
            {
                /* Successfully transmitted, reset waited time */
                wTimeWaited = 0;
                dwSentBytes += dwSentBytesNow;
            }
        }
        dwSentBytesSum += dwPacketRest;
    }

    /* Check for tx timeout */
    if(wTimeWaited > pDataParams->wTxTimeout)
    {
        phbalReg_MultiSocket_ClosePort(pDataParams);
        return PH_ADD_COMPCODE(PH_ERR_IO_TIMEOUT, PH_COMP_BAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}


phStatus_t phbalReg_MultiSocket_Int_Recv(
    phbalReg_MultiSocket_DataParams_t * pDataParams,
    uint8_t * pbRxBuffer,
    uint32_t dwBufferSizeBytes
    )
{
    phStatus_t statusTmp;
    uint32_t dwRecvdBytesNow;
    uint32_t dwRecvdBytesSum = 0;
    uint16_t wTimeWaited = 0;

    while (dwRecvdBytesSum < dwBufferSizeBytes &&
        wTimeWaited <= pDataParams->wRxTimeout)
    {
        dwRecvdBytesNow = recv(pDataParams->dwRecvFd,
            (int8_t*)pbRxBuffer + dwRecvdBytesSum,
            dwBufferSizeBytes - dwRecvdBytesSum,
            0);

        if(dwRecvdBytesNow == -1)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_MultiSocket_ClosePort(pDataParams));
            return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
        }
        else if(dwRecvdBytesNow == 0)
        {
            /* Nothing received */
            phTools_Sleep(100);
            wTimeWaited += 100;
        }
        else
        {
            /* Sum up, reset waited time */
            wTimeWaited = 0;
            dwRecvdBytesSum += dwRecvdBytesNow;
        }
    }

    /* Check for rx timeout */
    if(wTimeWaited > pDataParams->wRxTimeout)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_MultiSocket_ClosePort(pDataParams));
        return PH_ADD_COMPCODE(PH_ERR_IO_TIMEOUT, PH_COMP_BAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

phStatus_t phbalReg_Int_ParseWindowsError(
    uint32_t dwWinErrorCode)
{
#ifdef _WIN32
    WSACleanup();
    switch (dwWinErrorCode)
    {
    case WSAETIMEDOUT:
        return PH_ADD_COMPCODE(PH_ERR_IO_TIMEOUT, PH_COMP_BAL);
    case WSAECONNRESET:
    case WSAEHOSTDOWN:
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    default:
        break;
    }
#endif
    return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
}

phStatus_t phbalReg_MultiSocket_Int_OpenSendPort(
    phbalReg_MultiSocket_DataParams_t * pDataParams
    )
{
    /* Declare variables */
    struct sockaddr_in sendAddress;
    BOOL bOptVal = FALSE;
    int bOptLen = sizeof (BOOL);
    uint32_t pdwIpAddress[4];
    uint32_t dwIpAddress;

    /* Check param */
    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL(pDataParams->pHost);
    PH_ASSERT_NULL(pDataParams->pSendPort);

    /* Parse address */
    if (sscanf((const char *)pDataParams->pHost, "%d.%d.%d.%d", &pdwIpAddress[0], &pdwIpAddress[1], &pdwIpAddress[2], &pdwIpAddress[3]) != 4)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }
    dwIpAddress = (pdwIpAddress[3] << 24) + (pdwIpAddress[2] << 16) + (pdwIpAddress[1] << 8) + pdwIpAddress[0];

    /* Init address struct */
    memset(&sendAddress, '\0', sizeof(struct sockaddr_in));

    /* Set sendAddress values */
    sendAddress.sin_family = AF_INET;
    sendAddress.sin_addr.s_addr = dwIpAddress;
    sendAddress.sin_port = htons((uint16_t) strtol((const char *) pDataParams->pSendPort, NULL, 10));

    /* Check socket file descriptors */
    if (pDataParams->dwSendFd == INVALID_SOCKET)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
    }

    bOptVal = TRUE;
    if (setsockopt(pDataParams->dwSendFd, IPPROTO_TCP, TCP_NODELAY, (char *) &bOptVal, bOptLen) == SOCKET_ERROR)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_BAL);
    }

    /* Connect send socket */
    if (connect(pDataParams->dwSendFd, (struct sockaddr *) &sendAddress, sizeof(struct sockaddr)) == SOCKET_ERROR)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

phStatus_t phbalReg_MultiSocket_Int_OpenRecvPort(
    phbalReg_MultiSocket_DataParams_t * pDataParams
    )
{
    /* Declare variables */
    struct sockaddr_in recvAddress;
    BOOL bOptVal = FALSE;
    int bOptLen = sizeof (BOOL);
    uint32_t pdwIpAddress[4];
    uint32_t dwIpAddress;

    /* Check param */
    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL(pDataParams->pHost);
    PH_ASSERT_NULL(pDataParams->pRecvPort);

    /* Parse address */
    if (sscanf((const char *)pDataParams->pHost, "%d.%d.%d.%d", &pdwIpAddress[0], &pdwIpAddress[1], &pdwIpAddress[2], &pdwIpAddress[3]) != 4)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }
    dwIpAddress = (pdwIpAddress[3] << 24) + (pdwIpAddress[2] << 16) + (pdwIpAddress[1] << 8) + pdwIpAddress[0];

    /* Init address struct */
    memset(&recvAddress, '\0', sizeof(struct sockaddr_in));

    /* Set recvAddress values */
    recvAddress.sin_family = AF_INET;
    recvAddress.sin_addr.s_addr = dwIpAddress;
    recvAddress.sin_port = htons((uint16_t) strtol((const char *) pDataParams->pRecvPort, NULL, 10));

    /* Check socket file descriptors */
    if (pDataParams->dwRecvFd == INVALID_SOCKET)
    {
        pDataParams->dwRecvFd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
        if (pDataParams->dwRecvFd == INVALID_SOCKET)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
        }
    }

    bOptVal = TRUE;
    if (setsockopt(pDataParams->dwRecvFd, IPPROTO_TCP, TCP_NODELAY, (char *) &bOptVal, bOptLen) == SOCKET_ERROR)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_BAL);
    }

    /* Connect recv sockets */
    if (connect(pDataParams->dwRecvFd, (struct sockaddr *) &recvAddress, sizeof(struct sockaddr)) == SOCKET_ERROR)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

phStatus_t phbalReg_MultiSocket_Int_CloseSendPort(
    phbalReg_MultiSocket_DataParams_t * pDataParams
    )
{
    /* Check param */
    PH_ASSERT_NULL(pDataParams);

    /* Check socket file descriptor */
    if(pDataParams->dwSendFd == INVALID_SOCKET)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
    }

    /* Close sockets */
#ifdef _WIN32
    shutdown(pDataParams->dwSendFd, SD_BOTH);
    closesocket(pDataParams->dwSendFd);
    if(pDataParams->dwRecvFd == INVALID_SOCKET)
        WSACleanup();
#else
    closesocket(pDataParams->dwSendFd);
#endif

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

phStatus_t phbalReg_MultiSocket_Int_CloseRecvPort(
    phbalReg_MultiSocket_DataParams_t * pDataParams
    )
{
    /* Check param */
    PH_ASSERT_NULL(pDataParams);

    /* Check socket file descriptor */
    if(pDataParams->dwRecvFd == INVALID_SOCKET)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
    }

    /* Close sockets */
#ifdef _WIN32
    shutdown(pDataParams->dwRecvFd, SD_BOTH);
    closesocket(pDataParams->dwRecvFd);
    pDataParams->dwRecvFd = INVALID_SOCKET;
    if(pDataParams->dwSendFd == INVALID_SOCKET)
        WSACleanup();
#else
    closesocket(pDataParams->dwRecvFd);
#endif

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

#endif /* NXPBUILD__PHBAL_REG_MULTISOCKET */
