/*
 * Copyright 2019, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * BAL MultiSocket Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phbalReg.h>

#ifdef NXPBUILD__PHBAL_REG_MULTISOCKET

#include "phbalReg_MultiSocket.h"
#include "phbalReg_MultiSocket_Int.h"

phStatus_t phbalReg_MultiSocket_Init(
    phbalReg_MultiSocket_DataParams_t * pDataParams
    )
{
    /* Declare variables */
    SOCKET_T sendSock = INVALID_SOCKET;
    SOCKET_T recvSock = INVALID_SOCKET;

#ifdef _WIN32
    /* Initialize WSADATA structure for Windows Socket */
    WSADATA wsaData;
    uint16_t wVersionRequested = MAKEWORD(2,2);
    uint32_t dwError;

    dwError = WSAStartup(wVersionRequested, &wsaData);

    if(dwError != 0)
    {
        /* No usable WinSock DLL found! */
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_BAL);
    }

    if (LOBYTE(wsaData.wVersion) != 2 || HIBYTE(wsaData.wVersion) != 2) {
        /* No usable version of Winsock.dll found! */
        WSACleanup();
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_BAL);
    }

#endif

    /* Check param */
    PH_ASSERT_NULL(pDataParams);

    /* Create two TCP sockets
    * AF_INET:     IPv4
    * SOCK_STREAM: 2-Way communication via TCP and IPv4
    * IPPROTO_TCP: TCP protocol
    */
    sendSock = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    recvSock = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);

    if (sendSock == INVALID_SOCKET || recvSock == INVALID_SOCKET) {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_BAL);
    }

    /* Init data params with default values */
    pDataParams->wId              = (uint16_t) PH_COMP_BAL | PHBAL_REG_MULTISOCKET_ID;
    pDataParams->wTxTimeout       = (uint16_t) PHBAL_REG_MULTISOCKET_TX_TIMEOUT_DEFAULT;
    pDataParams->wRxTimeout       = (uint16_t) PHBAL_REG_MULTISOCKET_RX_TIMEOUT_DEFAULT;
    pDataParams->pHost            = (uint8_t *) NULL;
    pDataParams->pSendPort        = (uint8_t *) NULL;
    pDataParams->pRecvPort        = (uint8_t *) NULL;
    pDataParams->dwSendFd         = sendSock;
    pDataParams->dwRecvFd         = recvSock;

    /* Set default timeout for rx und tx */
    phbalReg_MultiSocket_SetConfig(pDataParams, PHBAL_REG_MULTISOCKET_CONFIG_RX_TIMEOUT, pDataParams->wRxTimeout);
    phbalReg_MultiSocket_SetConfig(pDataParams, PHBAL_REG_MULTISOCKET_CONFIG_TX_TIMEOUT, pDataParams->wTxTimeout);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

phStatus_t phbalReg_MultiSocket_GetPortList(
    phbalReg_MultiSocket_DataParams_t * pDataParams,
    uint16_t wPortBufSize,
    uint8_t * pPortNames,
    uint16_t * pNumOfPorts
    )
{
    /* Satisfy compiler */
    if(pDataParams || wPortBufSize || pPortNames || pNumOfPorts);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_BAL);
}

phStatus_t phbalReg_MultiSocket_SetPort(
    phbalReg_MultiSocket_DataParams_t * pDataParams,
    uint8_t * pPortName
    )
{
    /* Check params */
    PH_ASSERT_NULL(pDataParams);
    PH_ASSERT_NULL(pPortName);

    /* pPortName looks something like this: "10.0.2.15" + "\0" + "22221" + "\0" + "22223" + "\0" */

    if (strlen((char *) pPortName) == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
    }

    /* Host IP */
    pDataParams->pHost = pPortName;

    pPortName += strlen((char *) pPortName) + 1;

    if (strlen((char *) pPortName) == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
    }

    /* Send Port */
    pDataParams->pSendPort = pPortName;

    pPortName += strlen((char *) pPortName) + 1;

    if (strlen((char *) pPortName) == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
    }

    /* Recv Port */
    pDataParams->pRecvPort = pPortName;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

phStatus_t phbalReg_MultiSocket_OpenPort(
    phbalReg_MultiSocket_DataParams_t * pDataParams
    )
{
    phStatus_t statusTmp;

    /* Check param */
    PH_ASSERT_NULL(pDataParams);

    /* Open send and recv port */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_MultiSocket_Int_OpenSendPort(pDataParams));

    if (strcmp((char*)pDataParams->pRecvPort, (char*)pDataParams->pSendPort) != 0)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_MultiSocket_Int_OpenRecvPort(pDataParams));
    }
    else
    {
        pDataParams->dwRecvFd = pDataParams->dwSendFd;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

phStatus_t phbalReg_MultiSocket_ClosePort(
    phbalReg_MultiSocket_DataParams_t * pDataParams
    )
{
    phStatus_t statusTmp;
    /* Check param */
    PH_ASSERT_NULL(pDataParams);

    /* Close send and recv socket */
    PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_MultiSocket_Int_CloseSendPort(pDataParams));

    if (strcmp((char*)pDataParams->pRecvPort, (char*)pDataParams->pSendPort) != 0)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_MultiSocket_Int_CloseRecvPort(pDataParams));
    }

    pDataParams->dwSendFd = INVALID_SOCKET;
    pDataParams->dwRecvFd = INVALID_SOCKET;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

phStatus_t phbalReg_MultiSocket_Exchange(
    phbalReg_MultiSocket_DataParams_t * pDataParams,
    uint16_t wOption,
    uint8_t * pTxBuffer,
    uint16_t wTxLength,
    uint16_t wRxBufSize,
    uint8_t * pRxBuffer,
    uint16_t * pRxLength
    )
{
    phStatus_t statusTmp;
    /* Check param */
    PH_ASSERT_NULL(pDataParams);

    if(wOption != PH_EXCHANGE_DEFAULT)
    {
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_BAL);
    }

    if(pDataParams->dwSendFd == INVALID_SOCKET || pDataParams->dwRecvFd == INVALID_SOCKET)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_BAL);
    }

    /* Send data */
    if(pTxBuffer != NULL)
    {
        if(wTxLength == 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_BAL);
        }

        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_MultiSocket_Int_Send(
            pDataParams,
            pTxBuffer,
            wTxLength));
    }

    /* Recv data */
    if(pRxBuffer != NULL && wRxBufSize > 0)
    {
        if(pRxLength == NULL)
        {
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_BAL);
        }

        *pRxLength = 0;
        PH_CHECK_SUCCESS_FCT(statusTmp, phbalReg_MultiSocket_Int_Recv(
            pDataParams,
            pRxBuffer,
            wRxBufSize));

        *pRxLength = wRxBufSize;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

phStatus_t phbalReg_MultiSocket_SetConfig(
    phbalReg_MultiSocket_DataParams_t * pDataParams,
    uint16_t wConfig,
    uint16_t wValue
    )
{
    /* Declare variable */
    struct timeval timeout;

    /* Check params */
    PH_ASSERT_NULL(pDataParams);

    switch(wConfig)
    {
    case PHBAL_REG_MULTISOCKET_CONFIG_TX_TIMEOUT:
    case PHBAL_REG_MULTISOCKET_CONFIG_RX_TIMEOUT:
        /* Check value */
        if (wValue < 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_BAL);
        }

        /* Set value */
        if (wConfig == PHBAL_REG_MULTISOCKET_CONFIG_TX_TIMEOUT)
        {
            /* Set tx value */
            pDataParams->wTxTimeout = wValue;
        }
        else
        {
            /* Set rx value */
            pDataParams->wRxTimeout = wValue;
        }

        /* Init timeout struct */
        timeout.tv_sec = wValue;
        timeout.tv_usec = 0;

        if (wConfig == PHBAL_REG_MULTISOCKET_CONFIG_TX_TIMEOUT)
        {
            /* Set tx timeout */
            if (setsockopt(pDataParams->dwSendFd, SOL_SOCKET, SO_SNDTIMEO, (const char *) &timeout, sizeof(struct timeval)) < 0)
            {
                return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_BAL);
            }
        }
        else
        {
            /* Set rx timeout */
            if (setsockopt(pDataParams->dwRecvFd, SOL_SOCKET, SO_RCVTIMEO, (const char *) &timeout, sizeof(struct timeval)) < 0)
            {
                return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_BAL);
            }
        }

        break;
    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_BAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

phStatus_t phbalReg_MultiSocket_GetConfig(
    phbalReg_MultiSocket_DataParams_t * pDataParams,
    uint16_t wConfig,
    uint16_t * pValue
    )
{
    /* Check params */
    PH_ASSERT_NULL(pDataParams);

    switch(wConfig)
    {
    case PHBAL_REG_MULTISOCKET_CONFIG_TX_TIMEOUT:
        *pValue = pDataParams->wTxTimeout;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
    case PHBAL_REG_MULTISOCKET_CONFIG_RX_TIMEOUT:
        *pValue = pDataParams->wRxTimeout;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_BAL);
    }
}

#endif /* NXPBUILD__PHBAL_REG_MULTISOCKET */
