/*
 * Copyright 2022 - 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal FPGA BOX BAL Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phbalReg.h>
#include <ph_RefDefs.h>

#pragma warning(push)           /* PRQA S 3116 */
#pragma warning(disable:4001)   /* PRQA S 3116 */
#pragma warning( disable : 4996 )
#include <ctype.h>
#include <cwchar>
#include <iostream>
#pragma warning(pop)            /* PRQA S 3116 */

#import "..\..\..\phHalHwContact\src\ContactKeolabs\ex\ContactLAB.exe" no_namespace

#ifdef NXPBUILD__PHBAL_REG_CONTACTLAB

#include "phbalReg_ContactLAB_Int.h"

phStatus_t phbalReg_ContactLAB_Int_OpenPort(
                                            phbalReg_ContactLAB_DataParams_t * pDataParams
                                            )
{
    IContactLABPtr ContactLAB;
    phStatus_t status;

    /* Create ProxiLAB object */
    HRESULT hr = ContactLAB.CreateInstance("KEOLABS.ContactLAB");

    if (FAILED(hr))
    {
        /* Failed to create COM object */
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }

    /* Check if Reader is connected */
    if (!ContactLAB->IsConnected)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }

    PH_CHECK_SUCCESS_FCT(status, phbalReg_ContactLAB_Int_ConvertError(pDataParams, ContactLAB, ContactLAB->Settings->LoadDefaultConfig()));
    PH_CHECK_SUCCESS_FCT(status, phbalReg_ContactLAB_Int_ApplicationNeedsRecompiling(pDataParams, ContactLAB));

    strcpy_s((char *)pDataParams->bBoardNumber, PHBAL_REG_PROXILAB_VERSION_SIZE, (char *)ContactLAB->BoardNumber);
    strcpy_s((char *)pDataParams->bHardwareVersion, PHBAL_REG_PROXILAB_VERSION_SIZE, (char *)ContactLAB->HardwareVersion);
    strcpy_s((char *)pDataParams->bFirmwareVersion, PHBAL_REG_PROXILAB_VERSION_SIZE, (char *)ContactLAB->FirmwareVersion);
    strcpy_s((char *)pDataParams->bSoftwareVersion, PHBAL_REG_PROXILAB_VERSION_SIZE, (char *)ContactLAB->SoftwareVersion);
    strcpy_s((char *)pDataParams->bContactLABSerial, PHBAL_REG_PROXILAB_VERSION_SIZE, (char *)ContactLAB->ContactLABSerial);
    pDataParams->bBoardNumber[PHBAL_REG_PROXILAB_VERSION_SIZE-1] = 0;
    pDataParams->bHardwareVersion[PHBAL_REG_PROXILAB_VERSION_SIZE-1] = 0;
    pDataParams->bFirmwareVersion[PHBAL_REG_PROXILAB_VERSION_SIZE-1] = 0;
    pDataParams->bSoftwareVersion[PHBAL_REG_PROXILAB_VERSION_SIZE-1] = 0;
    pDataParams->bContactLABSerial[PHBAL_REG_PROXILAB_VERSION_SIZE-1] = 0;

    /* Pass IContact to pDataParams Pointer */
    pDataParams->pContactLAB = ContactLAB;
    ContactLAB.AddRef();

    /* Disable Logging (increased performance) */
    ContactLAB->Display->Verbose = 0;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

phStatus_t phbalReg_ContactLAB_Int_ClosePort(
                                             phbalReg_ContactLAB_DataParams_t * pDataParams
                                             )
{
    /* Release the IContactLAB Smart Pointer */
    IContactLABPtr ContactLAB = (IContactLAB*)pDataParams->pContactLAB;

    if(ContactLAB == NULL)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_BAL);
    }

    ContactLAB.Release();

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

phStatus_t phbalReg_ContactLAB_Int_ConvertError(
                                                phbalReg_ContactLAB_DataParams_t * pDataParams,
                                                void* pContactLAB,
                                                uint32_t status
                                                )
{
    phStatus_t convertedStatus;
    IContactLABPtr ContactLAB = (IContactLAB*)pContactLAB;

    switch(status)
    {
    case ERR_SUCCESSFUL:
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);

    default:
        convertedStatus = PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
        break;
    }

    /* Store the error message */
    sprintf_s((char *)pDataParams->bLastErrorMessage, PHBAL_REG_PROXILAB_ERROR_MESSAGE_SIZE, (char *)ContactLAB->GetErrorInfo(status));
    return convertedStatus;
}

//This method should be called at the start of any application importing the type
//library ProxiLAB, to check if recompiling is required.
phStatus_t phbalReg_ContactLAB_Int_ApplicationNeedsRecompiling(
                                                               phbalReg_ContactLAB_DataParams_t * pDataParams,
                                                               void* pContactLAB
                                                               )
{
    uint8_t aBufferIn[16] = {
        0xD0, 0xD1, 0xD2, 0xD3, 0xD4, 0xD5, 0xD6, 0xD7,
        0xD8, 0xD9, 0xDA, 0xDB, 0xDC, 0xDD, 0xDE, 0xDF};
    uint8_t aBufferOut[16];
    unsigned long dwBufferOutSize;
    uint32_t dwRet;

    IContactLABPtr ContactLAB = (IContactLAB*)pContactLAB;

    /*Send the table to the server, get it back and check they are identical*/
    dwRet = ContactLAB->EchoArray(aBufferIn, 16, aBufferOut, 16, &dwBufferOutSize);
    if (dwRet)
    {
        /*An error occurred (usually a bad parameter, this should not happen*/
        return phbalReg_ContactLAB_Int_ConvertError(pDataParams, pContactLAB, dwRet);
    }

    if (dwBufferOutSize != 16)
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);

    /*The EchoArray method returned an array: compare results*/
    for(int i = 0; i < 16; i++)
    {
        if(aBufferIn[i] != aBufferOut[i])
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_BAL);
}

#endif /* NXPBUILD__PHBAL_REG_CONTACTLAB */
