/*
 * Copyright 2013 - 2018, 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Generic Virtual Card Architecture (R) Application Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phalVca.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHAL_VCA_SW
#include "Sw/phalVca_Sw.h"
#endif /* NXPBUILD__PHAL_VCA_SW */

#ifdef NXPBUILD__PHAL_VCA_SAM_X
#include "Sam_X/phalVca_Sam_X.h"
#endif /* NXPBUILD__PHAL_VCA_SAM_X */

#ifdef NXPBUILD__PHAL_VCA_SAM_NONX
#include "Sam_NonX/phalVca_Sam_NonX.h"
#endif /* NXPBUILD__PHAL_VCA_SAM_NONX */

#ifdef NXPBUILD__PHAL_VCA
phStatus_t phalVca_StartCardSelection(void * pDataParams)
{
    phStatus_t PH_MEMLOC_REM status;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalVca_StartCardSelection");
    /*PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDataParams);*/
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* Validate the parameters */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_VCA);

    /* Check data parameters */
    if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_VCA)
    {
        status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return status;
    }

    /* Perform operation on active layer */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_VCA_SW
        case PHAL_VCA_SW_ID:
            status = phalVca_Sw_StartCardSelection((phalVca_Sw_DataParams_t *) pDataParams);
            break;
#endif /* NXPBUILD__PHAL_VCA_SW */

#ifdef NXPBUILD__PHAL_VCA_SAM_NONX
        case PHAL_VCA_SAMAV2_NONX_ID:
            status = phalVca_Sam_NonX_StartCardSelection((phalVca_SamAV2_DataParams_t *) pDataParams);
            break;
#endif /* NXPBUILD__PHAL_VCA_SAM_NONX */

#ifdef NXPBUILD__PHAL_VCA_SAM_X
        case PHAL_VCA_SAMAV2_X_ID:
            status = phalVca_Sam_X_StartCardSelection((phalVca_SamAV2_X_DataParams_t *) pDataParams);
            break;

#ifdef NXPBUILD__PHAL_VCA_SAMAV3_X
        case PHAL_VCA_SAMAV3_X_ID:
#endif /* NXPBUILD__PHAL_VCA_SAMAV3_X */
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
        case PHAL_VCA_SAM_NONX_ID:
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */
            status = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_VCA);
            break;
#endif /* NXPBUILD__PHAL_VCA_SAM_X */

        default:
            status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);
            break;
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return status;
}

phStatus_t phalVca_FinalizeCardSelection(void * pDataParams, uint16_t * pNumValidIids)
{
    phStatus_t PH_MEMLOC_REM status;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalVca_FinalizeCardSelection");
    /*PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDataParams);*/
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pNumValidIids);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* Validate the parameters */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_VCA);
    PH_ASSERT_NULL_PARAM(pNumValidIids, PH_COMP_AL_VCA);

    /* Check data parameters */
    if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_VCA)
    {
        status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return status;
    }

    /* Perform operation on active layer */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_VCA_SW
        case PHAL_VCA_SW_ID:
            status = phalVca_Sw_FinalizeCardSelection((phalVca_Sw_DataParams_t *) pDataParams, pNumValidIids);
            break;
#endif /* NXPBUILD__PHAL_VCA_SW */

#ifdef NXPBUILD__PHAL_VCA_SAM_NONX
        case PHAL_VCA_SAMAV2_NONX_ID:
            status = phalVca_Sam_NonX_FinalizeCardSelection((phalVca_SamAV2_DataParams_t *) pDataParams, pNumValidIids);
            break;
#endif /* NXPBUILD__PHAL_VCA_SAM_NONX */

#ifdef NXPBUILD__PHAL_VCA_SAM_X
        case PHAL_VCA_SAMAV2_X_ID:
            status = phalVca_Sam_X_FinalizeCardSelection((phalVca_SamAV2_X_DataParams_t *) pDataParams, pNumValidIids);
            break;

#ifdef NXPBUILD__PHAL_VCA_SAMAV3_X
        case PHAL_VCA_SAMAV3_X_ID:
#endif /* NXPBUILD__PHAL_VCA_SAMAV3_X */
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
        case PHAL_VCA_SAM_NONX_ID:
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */
            status = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_VCA);
            break;
#endif /* NXPBUILD__PHAL_VCA_SAM_X */

        default:
            status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);
            break;
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, pNumValidIids_log, pNumValidIids);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return status;
}

#ifdef NXPBUILD__PH_CRYPTOSYM
phStatus_t phalVca_SelectVc(void * pDataParams, uint16_t wValidIidIndex, uint16_t wKeyNumber, uint16_t wKeyVersion)
{
    phStatus_t PH_MEMLOC_REM status;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalVca_SelectVc");
    /*PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDataParams);*/
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wValidIidIndex);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wKeyNumber);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wKeyVersion);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wValidIidIndex_log, &wValidIidIndex);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wKeyNumber_log, &wKeyNumber);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wKeyVersion_log, &wKeyVersion);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* Validate the parameters */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_VCA);

    /* Check data parameters */
    if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_VCA)
    {
        status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return status;
    }

    /* Perform operation on active layer */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_VCA_SW
        case PHAL_VCA_SW_ID:
            status = phalVca_Sw_SelectVc((phalVca_Sw_DataParams_t *) pDataParams, wValidIidIndex, wKeyNumber, wKeyVersion);
            break;
#endif /* NXPBUILD__PHAL_VCA_SW */

#ifdef NXPBUILD__PHAL_VCA_SAM_NONX
        case PHAL_VCA_SAMAV2_NONX_ID:
            status = phalVca_Sam_NonX_SelectVc((phalVca_SamAV2_DataParams_t *) pDataParams, wValidIidIndex, wKeyNumber, wKeyVersion);
            break;
#endif /* NXPBUILD__PHAL_VCA_SAM_NONX */

#ifdef NXPBUILD__PHAL_VCA_SAM_X
        case PHAL_VCA_SAMAV2_X_ID:
            status = phalVca_Sam_X_SelectVc((phalVca_SamAV2_X_DataParams_t *) pDataParams, wValidIidIndex, wKeyNumber, wKeyVersion);
            break;

#ifdef NXPBUILD__PHAL_VCA_SAMAV3_X
        case PHAL_VCA_SAMAV3_X_ID:
#endif /* NXPBUILD__PHAL_VCA_SAMAV3_X */
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
        case PHAL_VCA_SAM_NONX_ID:
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */
            status = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_VCA);
            break;
#endif /* NXPBUILD__PHAL_VCA_SAM_X */

        default:
            status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);
            break;
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return status;
}
#endif /* NXPBUILD__PH_CRYPTOSYM */

phStatus_t phalVca_DeselectVc(void * pDataParams)
{
    phStatus_t PH_MEMLOC_REM status;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalVca_DeselectVc");
    /*PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDataParams);*/
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* Validate the parameters */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_VCA);

    /* Check data parameters */
    if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_VCA)
    {
        status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return status;
    }

    /* Perform operation on active layer */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_VCA_SW
        case PHAL_VCA_SW_ID:
            status = phalVca_Sw_DeselectVc((phalVca_Sw_DataParams_t *) pDataParams);
            break;
#endif /* NXPBUILD__PHAL_VCA_SW */

#ifdef NXPBUILD__PHAL_VCA_SAM_NONX
        case PHAL_VCA_SAMAV2_NONX_ID:
            status = phalVca_Sam_NonX_DeselectVc((phalVca_SamAV2_DataParams_t *) pDataParams);
            break;
#endif /* NXPBUILD__PHAL_VCA_SAM_NONX */

#ifdef NXPBUILD__PHAL_VCA_SAM_X
        case PHAL_VCA_SAMAV2_X_ID:
            status = phalVca_Sam_X_DeselectVc((phalVca_SamAV2_X_DataParams_t *) pDataParams);
            break;

#ifdef NXPBUILD__PHAL_VCA_SAMAV3_X
        case PHAL_VCA_SAMAV3_X_ID:
#endif /* NXPBUILD__PHAL_VCA_SAMAV3_X */
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
        case PHAL_VCA_SAM_NONX_ID:
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */
            status = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_VCA);
            break;
#endif /* NXPBUILD__PHAL_VCA_SAM_X */

        default:
            status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);
            break;
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return status;
}

phStatus_t phalVca_VcSupport(void * pDataParams, uint8_t * pIid, uint16_t wKeyEncNumber, uint16_t wKeyEncVersion, uint16_t wKeyMacNumber,
    uint16_t wKeyMacVersion)
{
    phStatus_t PH_MEMLOC_REM status;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalVca_VcSupport");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pIid);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wKeyEncNumber);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wKeyEncVersion);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wKeyMacNumber);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wKeyMacVersion);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pIid_log, pIid, 16);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wKeyEncNumber_log, &wKeyEncNumber);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wKeyEncVersion_log, &wKeyEncVersion);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wKeyMacNumber_log, &wKeyMacNumber);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wKeyMacVersion_log, &wKeyMacVersion);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* Validate the parameters */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_VCA);
    PH_ASSERT_NULL_PARAM(pIid, PH_COMP_AL_VCA);

    /* Check data parameters */
    if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_VCA)
    {
        status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return status;
    }

    /* Perform operation on active layer */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_VCA_SW
        case PHAL_VCA_SW_ID:
            status = phalVca_Sw_VcSupport((phalVca_Sw_DataParams_t *) pDataParams, pIid, wKeyEncNumber, wKeyEncVersion, wKeyMacNumber, wKeyMacVersion);
            break;
#endif /* NXPBUILD__PHAL_VCA_SW */

#ifdef NXPBUILD__PHAL_VCA_SAM_NONX
        case PHAL_VCA_SAMAV2_NONX_ID:
            status = phalVca_Sam_NonX_VcSupport((phalVca_SamAV2_DataParams_t *) pDataParams, pIid, wKeyEncNumber, wKeyEncVersion, wKeyMacNumber, wKeyMacVersion);
            break;
#endif /* NXPBUILD__PHAL_VCA_SAM_NONX */

#ifdef NXPBUILD__PHAL_VCA_SAM_X
        case PHAL_VCA_SAMAV2_X_ID:
            status = phalVca_Sam_X_VcSupport((phalVca_SamAV2_X_DataParams_t *) pDataParams, pIid, wKeyEncNumber, wKeyEncVersion, wKeyMacNumber, wKeyMacVersion);
            break;

#ifdef NXPBUILD__PHAL_VCA_SAMAV3_X
        case PHAL_VCA_SAMAV3_X_ID:
#endif /* NXPBUILD__PHAL_VCA_SAMAV3_X */
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
        case PHAL_VCA_SAM_NONX_ID:
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */
            status = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_VCA);
            break;
#endif /* NXPBUILD__PHAL_VCA_SAM_X */
        default:
            status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);
            break;
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return status;
}

#ifdef NXPBUILD__PH_CRYPTOSYM
phStatus_t phalVca_VcSupportLast(void * pDataParams, uint8_t * pIid, uint8_t bLenCap, uint8_t * pPcdCapabilities, uint16_t wKeyEncNumber,
    uint16_t wKeyEncVersion, uint16_t wKeyMacNumber, uint16_t wKeyMacVersion)
{
    phStatus_t PH_MEMLOC_REM status;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalVca_VcSupportLast");
    /*PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDataParams);*/
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pIid);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bLenCap);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pPcdCapabilities);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wKeyEncNumber);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wKeyEncVersion);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wKeyMacNumber);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wKeyMacVersion);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pIid_log, pIid, 16);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bLenCap_log, &bLenCap);
    PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pPcdCapabilities_log, pPcdCapabilities, bLenCap);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wKeyEncNumber_log, &wKeyEncNumber);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wKeyEncVersion_log, &wKeyEncVersion);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wKeyMacNumber_log, &wKeyMacNumber);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wKeyMacVersion_log, &wKeyMacVersion);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* Validate the parameters */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_VCA);
    PH_ASSERT_NULL_PARAM(pIid, PH_COMP_AL_VCA);
    if(bLenCap) PH_ASSERT_NULL_PARAM(pPcdCapabilities, PH_COMP_AL_VCA);

    /* Check data parameters */
    if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_VCA)
    {
        status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return status;
    }

    /* Perform operation on active layer */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_VCA_SW
        case PHAL_VCA_SW_ID:
            status = phalVca_Sw_VcSupportLast((phalVca_Sw_DataParams_t *) pDataParams, pIid, bLenCap, pPcdCapabilities,
                wKeyEncNumber, wKeyEncVersion, wKeyMacNumber, wKeyMacVersion);
            break;
#endif /* NXPBUILD__PHAL_VCA_SW */

#ifdef NXPBUILD__PHAL_VCA_SAM_NONX
        case PHAL_VCA_SAMAV2_NONX_ID:
            status = phalVca_Sam_NonX_VcSupportLast((phalVca_SamAV2_DataParams_t *) pDataParams, pIid, bLenCap, pPcdCapabilities,
                wKeyEncNumber, wKeyEncVersion, wKeyMacNumber, wKeyMacVersion);
            break;
#endif /* NXPBUILD__PHAL_VCA_SAM_NONX */

#ifdef NXPBUILD__PHAL_VCA_SAM_X
        case PHAL_VCA_SAMAV2_X_ID:
            status = phalVca_Sam_X_VcSupportLast((phalVca_SamAV2_X_DataParams_t *) pDataParams, pIid, bLenCap, pPcdCapabilities,
                wKeyEncNumber, wKeyEncVersion, wKeyMacNumber, wKeyMacVersion);
            break;

#ifdef NXPBUILD__PHAL_VCA_SAMAV3_X
        case PHAL_VCA_SAMAV3_X_ID:
#endif /* NXPBUILD__PHAL_VCA_SAMAV3_X */
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
        case PHAL_VCA_SAM_NONX_ID:
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */
            status = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_VCA);
            break;
#endif /* NXPBUILD__PHAL_VCA_SAM_X */

        default:
            status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);
            break;
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return status;
}
#endif /* NXPBUILD__PH_CRYPTOSYM */

phStatus_t phalVca_GetIidInfo(void * pDataParams, uint16_t wValidIidIndex, uint16_t * pIidIndex, uint8_t * pVcUidSize, uint8_t * pVcUid,
    uint8_t * pInfo, uint8_t * pPdCapabilities)
{
    phStatus_t PH_MEMLOC_REM status;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalVca_GetIidInfo");
    /*PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDataParams);*/
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wValidIidIndex);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pIidIndex);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pVcUidSize);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pVcUid);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pInfo);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pPdCapabilities);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wValidIidIndex_log, &wValidIidIndex);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* Validate the parameters */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_VCA);
    PH_ASSERT_NULL_PARAM(pIidIndex, PH_COMP_AL_VCA);
    PH_ASSERT_NULL_PARAM(pVcUidSize, PH_COMP_AL_VCA);
    PH_ASSERT_NULL_PARAM(pVcUid, PH_COMP_AL_VCA);
    PH_ASSERT_NULL_PARAM(pInfo, PH_COMP_AL_VCA);
    PH_ASSERT_NULL_PARAM(pPdCapabilities, PH_COMP_AL_VCA);

    /* Check data parameters */
    if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_VCA)
    {
        status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return status;
    }

    /* Perform operation on active layer */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_VCA_SW
        case PHAL_VCA_SW_ID:
            status = phalVca_Sw_GetIidInfo((phalVca_Sw_DataParams_t *) pDataParams, wValidIidIndex, pIidIndex, pVcUidSize,
                pVcUid, pInfo, pPdCapabilities);
            break;
#endif /* NXPBUILD__PHAL_VCA_SW */

#ifdef NXPBUILD__PHAL_VCA_SAM_NONX
        case PHAL_VCA_SAMAV2_NONX_ID:
            status = phalVca_Sam_NonX_GetIidInfo((phalVca_SamAV2_DataParams_t *) pDataParams, wValidIidIndex, pIidIndex,
                pVcUidSize, pVcUid, pInfo, pPdCapabilities);
            break;
#endif /* NXPBUILD__PHAL_VCA_SAM_NONX */

#ifdef NXPBUILD__PHAL_VCA_SAM_X
        case PHAL_VCA_SAMAV2_X_ID:
            status = phalVca_Sam_X_GetIidInfo((phalVca_SamAV2_X_DataParams_t *) pDataParams, wValidIidIndex, pIidIndex,
                pVcUidSize, pVcUid, pInfo, pPdCapabilities);
            break;

#ifdef NXPBUILD__PHAL_VCA_SAMAV3_X
        case PHAL_VCA_SAMAV3_X_ID:
#endif /* NXPBUILD__PHAL_VCA_SAMAV3_X */
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
        case PHAL_VCA_SAM_NONX_ID:
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */
            status = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_VCA);
            break;
#endif /* NXPBUILD__PHAL_VCA_SAM_X */

        default:
            status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);
            break;
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
#ifdef NXPBUILD__PH_LOG
    if((status & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, pIidIndex_log, pIidIndex);
        PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, pVcUidSize_log, pVcUidSize);
        PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pVcUid_log, pVcUid, *pVcUidSize);
        PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, pInfo_log, pInfo);
        PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pPdCapabilities_log, pPdCapabilities, 2);
    }
#endif
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return status;
}

#ifdef NXPBUILD__PH_CRYPTOSYM
phStatus_t phalVca_IsoSelect(void * pDataParams, uint8_t bSelectionControl, uint8_t bOption, uint8_t  bDFnameLen, uint8_t * pDFname,
    uint8_t * pDivInput, uint8_t bDivInputLen, uint8_t bEncKeyNo, uint8_t bEncKeyVer, uint8_t bMacKeyNo, uint8_t bMacKeyVer,
    uint8_t * pResponse, uint16_t * pRespLen)
{
    phStatus_t  PH_MEMLOC_REM status;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalVca_IsoSelect");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bSelectionControl);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bOption);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDFname);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bDFnameLen);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDivInput);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bDivInputLen);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bEncKeyNo);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bEncKeyVer);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bMacKeyNo);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bMacKeyVer);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pResponse);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pRespLen);

    /* Validate the parameters */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_VCA);
    PH_ASSERT_NULL_PARAM(pDFname, PH_COMP_AL_VCA);
    PH_ASSERT_NULL_PARAM(pResponse, PH_COMP_AL_VCA);
    PH_ASSERT_NULL_PARAM(pRespLen, PH_COMP_AL_VCA);

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bSelectionControl_log, &bSelectionControl);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bOption_log, &bOption);
    PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pDFname_log, pDFname, bDFnameLen);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bDFnameLen_log, &bDFnameLen);

    if(pDivInput != NULL)
        PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pDivInput_log, pDivInput, (uint16_t) bDivInputLen);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bDivInputLen_log, &bDivInputLen);

    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bEncKeyNo_log, &bEncKeyNo);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bEncKeyVer_log, &bEncKeyVer);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bMacKeyNo_log, &bMacKeyNo);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bMacKeyVer_log, &bMacKeyVer);

    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* Check data parameters */
    if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_VCA)
    {
        status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return status;
    }

    /* Perform operation on active layer */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_VCA_SW
        case PHAL_VCA_SW_ID:
            status = phalVca_Sw_IsoSelect((phalVca_Sw_DataParams_t *) pDataParams, bSelectionControl, bOption, bDFnameLen,
                pDFname, pResponse, pRespLen);
            break;
#endif /* NXPBUILD__PHAL_VCA_SW */

#ifdef NXPBUILD__PHAL_VCA_SAM_NONX
        case PHAL_VCA_SAMAV2_NONX_ID:
            status = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_VCA);
            break;

#ifdef NXPBUILD__PHAL_VCA_SAMAV3_NONX
        case PHAL_VCA_SAMAV3_NONX_ID:
#endif /* NXPBUILD__PHAL_VCA_SAMAV3_NONX */
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
        case PHAL_VCA_SAM_NONX_ID:
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */
            status = phalVca_Sam_NonX_IsoSelect(pDataParams, bSelectionControl, bOption, bDFnameLen, pDFname,
                pDivInput, bDivInputLen, bEncKeyNo, bEncKeyVer, bMacKeyNo, bMacKeyVer, pResponse, pRespLen);
            break;
#endif /* NXPBUILD__PHAL_VCA_SAM_NONX */

#ifdef NXPBUILD__PHAL_VCA_SAM_X
        case PHAL_VCA_SAMAV2_X_ID:
            status = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_VCA);
            break;

#ifdef NXPBUILD__PHAL_VCA_SAMAV3_X
        case PHAL_VCA_SAMAV3_X_ID:
#endif /* NXPBUILD__PHAL_VCA_SAMAV3_X */
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
        case PHAL_VCA_SAM_X_ID:
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */
            status = phalVca_Sam_X_IsoSelect(pDataParams, bOption, bDFnameLen,
                pDFname, pDivInput, bDivInputLen, bEncKeyNo, bEncKeyVer, bMacKeyNo, bMacKeyVer, pResponse, pRespLen);
            break;
#endif /* NXPBUILD__PHAL_VCA_SAM_X */

        default:
            status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);
            break;
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    if(*pRespLen != 0)
        PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pResponse_log, pResponse, *pRespLen);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, pRespLen_log, pRespLen);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return status;
}

phStatus_t phalVca_IsoExternalAuthenticate(void  * pDataParams, uint8_t * pInData, uint16_t wKeyNumber, uint16_t wKeyVersion)
{
    phStatus_t PH_MEMLOC_REM status;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalVca_IsoExternalAuthenticate");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wKeyNumber);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wKeyVersion);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wKeyNumber_log, &wKeyNumber);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wKeyVersion_log, &wKeyVersion);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* Validate the parameters */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_VCA);
    PH_ASSERT_NULL_PARAM(pInData, PH_COMP_AL_VCA);

    /* Check data parameters */
    if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_VCA)
    {
        status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return status;
    }

    /* Perform operation on active layer */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_VCA_SW
        case PHAL_VCA_SW_ID:
            status = phalVca_Sw_IsoExternalAuthenticate((phalVca_Sw_DataParams_t *) pDataParams, pInData, wKeyNumber, wKeyVersion);
            break;
#endif /* NXPBUILD__PHAL_VCA_SW */

#ifdef NXPBUILD__PHAL_VCA_SAM_NONX
        case PHAL_VCA_SAMAV2_NONX_ID:
            status = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_VCA);
            break;

#ifdef NXPBUILD__PHAL_VCA_SAMAV3_NONX
        case PHAL_VCA_SAMAV3_NONX_ID:
#endif /* NXPBUILD__PHAL_VCA_SAMAV3_NONX */
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
        case PHAL_VCA_SAM_NONX_ID:
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */
            status = phalVca_Sam_NonX_IsoExternalAuthenticate(pDataParams, pInData);
            break;
#endif /* NXPBUILD__PHAL_VCA_SAM_NONX */

#ifdef NXPBUILD__PHAL_VCA_SAM_X
        case PHAL_VCA_SAMAV2_X_ID:
#ifdef NXPBUILD__PHAL_VCA_SAMAV3_X
        case PHAL_VCA_SAMAV3_X_ID:
#endif /* NXPBUILD__PHAL_VCA_SAMAV3_X */
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
        case PHAL_VCA_SAM_X_ID:
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */
            status = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_VCA);
            break;
#endif /* NXPBUILD__PHAL_VCA_SAM_X */

        default:
            status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);
            break;
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return status;
}
#endif /* NXPBUILD__PH_CRYPTOSYM */




#ifdef NXPBUILD__PH_CRYPTOSYM
phStatus_t phalVca_ProximityCheck(void * pDataParams, uint8_t bGenerateRndC, uint8_t * pRndC, uint8_t bPps1, uint8_t bNumSteps,
    uint16_t wKeyNumber, uint16_t wKeyVersion, uint8_t * pUsedRndC)
{
    phStatus_t PH_MEMLOC_REM status;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalVca_ProximityCheck");
    /*PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDataParams);*/
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bGenerateRndC);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pRndC);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bPps1);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bNumSteps);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wKeyNumber);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wKeyVersion);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pUsedRndC);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bGenerateRndC_log, &bGenerateRndC);
#ifdef NXPBUILD__PH_LOG
    if(!bGenerateRndC)
    {
        PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pRndC_log, pRndC, 7);
    }
#endif
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bPps1_log, &bPps1);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bNumSteps_log, &bNumSteps);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wKeyNumber_log, &wKeyNumber);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wKeyVersion_log, &wKeyVersion);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* Validate the parameters */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_VCA);
    if(!bGenerateRndC) PH_ASSERT_NULL_PARAM(pRndC, PH_COMP_AL_VCA);
    PH_ASSERT_NULL_PARAM(pUsedRndC, PH_COMP_AL_VCA);

    /* Check data parameters */
    if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_VCA)
    {
        status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return status;
    }

    /* Perform operation on active layer */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_VCA_SW
        case PHAL_VCA_SW_ID:
            status = phalVca_Sw_ProximityCheck((phalVca_Sw_DataParams_t *) pDataParams, bGenerateRndC, pRndC, bPps1, bNumSteps, wKeyNumber,
                wKeyVersion, pUsedRndC);
            break;
#endif /* NXPBUILD__PHAL_VCA_SW */

#ifdef NXPBUILD__PHAL_VCA_SAM_NONX
        case PHAL_VCA_SAMAV2_NONX_ID:
            status = phalVca_Sam_NonX_ProximityCheck((phalVca_SamAV2_DataParams_t *) pDataParams, bGenerateRndC, pRndC, bPps1, bNumSteps, wKeyNumber,
                wKeyVersion, pUsedRndC);
            break;
#endif /* NXPBUILD__PHAL_VCA_SAM_NONX */

#ifdef NXPBUILD__PHAL_VCA_SAM_X
        case PHAL_VCA_SAMAV2_X_ID:
#ifdef NXPBUILD__PHAL_VCA_SAMAV3_X
        case PHAL_VCA_SAMAV3_X_ID:
#endif /* NXPBUILD__PHAL_VCA_SAMAV3_X */
            status = phalVca_Sam_X_ProximityCheck((phalVca_SamAV2_X_DataParams_t *) pDataParams, bGenerateRndC, pRndC, bPps1, bNumSteps, wKeyNumber,
                wKeyVersion, pUsedRndC);
            break;
#endif /* NXPBUILD__PHAL_VCA_SAM_X */

        default:
            status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);
            break;
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
#ifdef NXPBUILD__PH_LOG
    if((status & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pUsedRndC_log, pUsedRndC, 7);
    }
#endif
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return status;
}

phStatus_t phalVca_ProximityCheckNew(void * pDataParams, uint8_t bOption, uint8_t * pPrndC, uint8_t  bNumSteps, uint16_t wKeyNumber, uint16_t wKeyVersion,
    uint8_t * pDivInput, uint8_t bDivInputLen, uint8_t * pOption, uint8_t * pPubRespTime, uint8_t * pResponse, uint16_t * pRespLen, uint8_t * pCumRndRC)
{
    phStatus_t PH_MEMLOC_REM status;

    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalVca_ProximityCheckNew");
    /*PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDataParams);*/
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bOption);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pPrndC);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pOption);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pPubRespTime);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bNumSteps);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wKeyNumber);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wKeyVersion);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pDivInput);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bDivInputLen);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pResponse);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pCumRndRC);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);

    /* Validate the parameters */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_VCA);
    PH_ASSERT_NULL_PARAM(pResponse, PH_COMP_AL_VCA);
    PH_ASSERT_NULL_PARAM(pRespLen, PH_COMP_AL_VCA);
    PH_ASSERT_NULL_PARAM(pCumRndRC, PH_COMP_AL_VCA);

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bOption_log, &bOption);
    if(!bOption) PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pPrndC_log, pPrndC, 8);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bNumSteps_log, &bNumSteps);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wKeyNumber_log, &wKeyNumber);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wKeyVersion_log, &wKeyVersion);
    if(pDivInput != NULL)
        PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pDivInput_log, pDivInput, (uint16_t) bDivInputLen);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bDivInputLen_log, &bDivInputLen);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* Check data parameters */
    if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_VCA)
    {
        status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);

        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

        return status;
    }

    /* Perform operation on active layer */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_VCA_SW
        case PHAL_VCA_SW_ID:
            /* Validate the parameters. */
            if(!bOption) PH_ASSERT_NULL_PARAM(pPrndC, PH_COMP_AL_VCA);

            status = phalVca_Sw_ProximityCheckNew((phalVca_Sw_DataParams_t *) pDataParams, bOption, pPrndC, bNumSteps, wKeyNumber, wKeyVersion,
                pOption, pPubRespTime, pResponse, pRespLen, pCumRndRC);
            break;
#endif /* NXPBUILD__PHAL_VCA_SW */

#ifdef NXPBUILD__PHAL_VCA_SAM_NONX
        case PHAL_VCA_SAMAV2_NONX_ID:
            status = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_VCA);
            break;

#ifdef NXPBUILD__PHAL_VCA_SAMAV3_NONX
        case PHAL_VCA_SAMAV3_NONX_ID:
#endif /* NXPBUILD__PHAL_VCA_SAMAV3_NONX */
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
        case PHAL_VCA_SAM_NONX_ID:
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */
            status = phalVca_Sam_NonX_ProximityCheckNew(pDataParams, bOption, pPrndC, bNumSteps, wKeyNumber,
                wKeyVersion, pDivInput, bDivInputLen, pOption, pPubRespTime, pResponse, pRespLen, pCumRndRC);
            break;
#endif /* NXPBUILD__PHAL_VCA_SAM_NONX */

#ifdef NXPBUILD__PHAL_VCA_SAM_X
        case PHAL_VCA_SAMAV2_X_ID:
            status = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_VCA);
            break;

#ifdef NXPBUILD__PHAL_VCA_SAMAV3_X
        case PHAL_VCA_SAMAV3_X_ID:
#endif /* NXPBUILD__PHAL_VCA_SAMAV3_NONX */
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
        case PHAL_VCA_SAM_X_ID:
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */
            status = phalVca_Sam_X_ProximityCheckNew(pDataParams, bOption, wKeyNumber, wKeyVersion,
                bNumSteps, pDivInput, bDivInputLen, pOption, pPubRespTime, pResponse, pRespLen);
            break;
#endif /* NXPBUILD__PHAL_VCA_SAM_X */

        default:
            status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);
            break;
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
#ifdef NXPBUILD__PH_LOG
    if ((status & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pOption_log, pOption, 1);
        PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pPubRespTime_log, pPubRespTime, 2);
        PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pResponse_log, pResponse, *pRespLen);
        PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pCumRndRC_log, pCumRndRC, 16);
    }
#endif
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);

    return status;
}

phStatus_t phalVca_PrepareProximityCheckNew(void * pDataParams, uint8_t * bOption, uint8_t * pPubRespTime, uint8_t * pResponse, uint16_t * pRespLen)
{
    phStatus_t PH_MEMLOC_REM status;
    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalVca_PrepareProximityCheckNew");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bOption);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pPubRespTime);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pResponse);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* Validate the parameters */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_VCA);

    /* Check data parameters */
    if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_VCA)
    {
        status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
        return status;
    }

    /* Perform operation on active layer */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_VCA_SW
    case PHAL_VCA_SW_ID:
        status = phalVca_Sw_Cmd_PrepareProximityCheckNew((phalVca_Sw_DataParams_t *)pDataParams, bOption, pPubRespTime, pResponse, pRespLen);
        break;
#endif /* NXPBUILD__PHAL_VCA_SW */

#ifdef NXPBUILD__PHAL_VCA_SAM_NONX
        case PHAL_VCA_SAMAV2_NONX_ID:
#ifdef NXPBUILD__PHAL_VCA_SAMAV3_NONX
        case PHAL_VCA_SAMAV3_NONX_ID:
#endif /* NXPBUILD__PHAL_VCA_SAMAV3_NONX */
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
        case PHAL_VCA_SAM_NONX_ID:
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */
            status = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_VCA);
            break;
#endif /* NXPBUILD__PHAL_VCA_SAM_NONX */

#ifdef NXPBUILD__PHAL_VCA_SAM_X
        case PHAL_VCA_SAMAV2_X_ID:
#ifdef NXPBUILD__PHAL_VCA_SAMAV3_X
        case PHAL_VCA_SAMAV3_X_ID:
#endif /* NXPBUILD__PHAL_VCA_SAMAV3_X */
            status = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_VCA);
            break;
#endif /* NXPBUILD__PHAL_VCA_SAM_X */

        default:
            status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);
            break;
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
#ifdef NXPBUILD__PH_LOG
    if ((status & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, bOption_log, bOption, 1);
        PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pPubRespTime_log, pPubRespTime, 2);
        PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pResponse_log, pResponse, *pRespLen);
    }
#endif
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
    return status;
}

phStatus_t phalVca_ExecuteProximityCheckNew(void * pDataParams, uint8_t bGenerateRndC, uint8_t * pPrndC, uint8_t bNumSteps, uint8_t * pPubRespTime,
    uint8_t * pCumRndRC)
{
    phStatus_t PH_MEMLOC_REM status;
    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalVca_ExecuteProximityCheckNew");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bGenerateRndC);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pPrndC);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bNumSteps);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pPubRespTime);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pCumRndRC);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bGenerateRndC_log, &bGenerateRndC);
#ifdef NXPBUILD__PH_LOG
    if(!bGenerateRndC)
    {
        PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pPrndC_log, pPrndC, 8);
    }
#endif
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bNumSteps_log, &bNumSteps);
    PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pPubRespTime_log, pPubRespTime, 2);
    PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pCumRndRC_log, pCumRndRC, 8);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* Validate the parameters */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_VCA);
    if(!bGenerateRndC) PH_ASSERT_NULL_PARAM(pPrndC, PH_COMP_AL_VCA);
    PH_ASSERT_NULL_PARAM(pCumRndRC, PH_COMP_AL_VCA);

    /* Check data parameters */
    if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_VCA)
    {
        status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
        return status;
    }

    /* Perform operation on active layer */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_VCA_SW
        case PHAL_VCA_SW_ID:
            status = phalVca_Sw_Cmd_ProximityCheckNew((phalVca_Sw_DataParams_t *) pDataParams, bGenerateRndC, pPrndC, bNumSteps,
                pPubRespTime, pCumRndRC);
            break;
#endif /* NXPBUILD__PHAL_VCA_SW */

#ifdef NXPBUILD__PHAL_VCA_SAM_NONX
        case PHAL_VCA_SAMAV2_NONX_ID:
#ifdef NXPBUILD__PHAL_VCA_SAMAV3_NONX
        case PHAL_VCA_SAMAV3_NONX_ID:
#endif /* NXPBUILD__PHAL_VCA_SAMAV3_NONX */
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
        case PHAL_VCA_SAM_NONX_ID:
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */
            status = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_VCA);
            break;
#endif /* NXPBUILD__PHAL_VCA_SAM_NONX */

#ifdef NXPBUILD__PHAL_VCA_SAM_X
        case PHAL_VCA_SAMAV2_X_ID:
#ifdef NXPBUILD__PHAL_VCA_SAMAV3_X
        case PHAL_VCA_SAMAV3_X_ID:
#endif /* NXPBUILD__PHAL_VCA_SAMAV3_X */
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
        case PHAL_VCA_SAM_X_ID:
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */
            status = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_VCA);
            break;
#endif /* NXPBUILD__PHAL_VCA_SAM_X */

        default:
            status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);
            break;
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
#ifdef NXPBUILD__PH_LOG
    if((status & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pCumRndRC_log, pCumRndRC, 7);
    }
#endif
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
    return status;
}

phStatus_t phalVca_VerifyProximityCheckNew(void * pDataParams, uint8_t bOption, uint8_t * pPubRespTime, uint8_t * pResponse, uint16_t wRespLen,
    uint16_t wKeyNumber, uint16_t wKeyVersion, uint8_t * pRndCmdResp)
{
    phStatus_t PH_MEMLOC_REM status;
    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalVca_VerifyProximityCheckNew");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bOption);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pPubRespTime);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pResponse);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wKeyNumber);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wKeyVersion);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pRndCmdResp);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bOption_log, &bOption);
    PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pPubRespTime_log, pPubRespTime, 2);
    PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pResponse_log, pResponse, wRespLen);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wKeyNumber_log, &wKeyNumber);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wKeyVersion_log, &wKeyVersion);
    PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pRndCmdResp_log, pRndCmdResp, 8);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* Validate the parameters */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_VCA);

    /* Check data parameters */
    if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_VCA)
    {
        status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
        return status;
    }

    /* Perform operation on active layer */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_VCA_SW
        case PHAL_VCA_SW_ID:
        status = phalVca_Sw_Cmd_VerifyProximityCheckNew((phalVca_Sw_DataParams_t *)pDataParams, bOption, pPubRespTime, pResponse, wRespLen,
                wKeyNumber, wKeyVersion, pRndCmdResp);
            break;
#endif /* NXPBUILD__PHAL_VCA_SW */

#ifdef NXPBUILD__PHAL_VCA_SAM_NONX
        case PHAL_VCA_SAMAV2_NONX_ID:
#ifdef NXPBUILD__PHAL_VCA_SAMAV3_NONX
        case PHAL_VCA_SAMAV3_NONX_ID:
#endif /* NXPBUILD__PHAL_VCA_SAMAV3_NONX */
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
        case PHAL_VCA_SAM_NONX_ID:
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */
            status = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_VCA);
            break;
#endif /* NXPBUILD__PHAL_VCA_SAM_NONX */

#ifdef NXPBUILD__PHAL_VCA_SAM_X
        case PHAL_VCA_SAMAV2_X_ID:
#ifdef NXPBUILD__PHAL_VCA_SAMAV3_X
        case PHAL_VCA_SAMAV3_X_ID:
#endif /* NXPBUILD__PHAL_VCA_SAMAV3_X */
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
        case PHAL_VCA_SAM_X_ID:
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */
            status = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_VCA);
            break;
#endif /* NXPBUILD__PHAL_VCA_SAM_X */

        default:
            status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);
            break;
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
    return status;
}
#endif /* NXPBUILD__PH_CRYPTOSYM */

phStatus_t phalVca_VerifyProximityCheckUtility(void * pDataParams, uint8_t * pCmdMac, uint8_t * pCmdResp)
{
    phStatus_t PH_MEMLOC_REM status;
    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalVca_VerifyProximityCheckUtility");

    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pCmdMac);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pCmdResp);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pCmdMac_log, pCmdMac, 8);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_VCA);
    PH_ASSERT_NULL_PARAM(pCmdMac, PH_COMP_AL_VCA);
    PH_ASSERT_NULL_PARAM(pCmdResp, PH_COMP_AL_VCA);

    /* Check data parameters */
    if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_VCA)
    {
        status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
        return status;
    }

    /* Perform operation on active layer */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_VCA_SW
        case PHAL_VCA_SW_ID:
            status = phalVca_Sw_Cmd_VerifyProximityCheckUtility((phalVca_Sw_DataParams_t *) pDataParams, pCmdMac, pCmdResp);
            break;
#endif /* NXPBUILD__PHAL_VCA_SW */

#ifdef NXPBUILD__PHAL_VCA_SAM_X
        case PHAL_VCA_SAMAV2_X_ID:
#ifdef NXPBUILD__PHAL_VCA_SAMAV3_X
        case PHAL_VCA_SAMAV3_X_ID:
#endif /* NXPBUILD__PHAL_VCA_SAMAV3_X */
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
        case PHAL_VCA_SAM_X_ID:
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */
            status = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_VCA);
            break;
#endif /* NXPBUILD__PHAL_VCA_SAM_X */

        default:
            status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);
            break;
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pCmdResp_log, pCmdResp, 8);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
    return status;
}




phStatus_t phalVca_SetConfig(void * pDataParams, uint16_t wConfig, uint16_t wValue)
{
    phStatus_t PH_MEMLOC_REM status;
    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalVca_SetConfig");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wConfig);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wValue);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wConfig_log, &wConfig);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wValue_log, &wValue);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* Validate the parameters */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_VCA);

    /* Check data parameters */
    if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_VCA)
    {
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);
    }

    /* Perform operation on active layer */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_VCA_SW
        case PHAL_VCA_SW_ID:
            status = phalVca_Sw_SetConfig((phalVca_Sw_DataParams_t *) pDataParams, wConfig, wValue);
            break;
#endif /* NXPBUILD__PHAL_VCA_SW */

#ifdef NXPBUILD__PHAL_VCA_SAM_NONX
        case PHAL_VCA_SAMAV2_NONX_ID:
            status = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_VCA);
            break;

#ifdef NXPBUILD__PHAL_VCA_SAMAV3_NONX
        case PHAL_VCA_SAMAV3_NONX_ID:
#endif /* NXPBUILD__PHAL_VCA_SAMAV3_NONX */
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
        case PHAL_VCA_SAM_NONX_ID:
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */
            status = phalVca_Sam_NonX_SetConfig(pDataParams, wConfig, wValue);
            break;
#endif /* NXPBUILD__PHAL_VCA_SAM_NONX */

#ifdef NXPBUILD__PHAL_VCA_SAM_X
        case PHAL_VCA_SAMAV2_X_ID:
            status = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_VCA);
            break;

#ifdef NXPBUILD__PHAL_VCA_SAMAV3_X
        case PHAL_VCA_SAMAV3_X_ID:
#endif /* NXPBUILD__PHAL_VCA_SAMAV3_X */
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
        case PHAL_VCA_SAM_X_ID:
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */
            status = phalVca_Sam_X_SetConfig(pDataParams, wConfig, wValue);
            break;
#endif /* NXPBUILD__PHAL_VCA_SAM_X */

        default:
            status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);
            break;
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
    return status;
}

phStatus_t phalVca_GetConfig(void * pDataParams, uint16_t wConfig, uint16_t * pValue)
{
    phStatus_t PH_MEMLOC_REM status;
    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalVca_GetConfig");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wConfig);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pValue);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);
    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wConfig_log, &wConfig);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, pValue_log, pValue);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* Validate the parameters */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_VCA);
    PH_ASSERT_NULL_PARAM(pValue, PH_COMP_AL_VCA);

    /* Check data parameters */
    if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_VCA)
    {
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);
    }

    /* Perform operation on active layer */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_VCA_SW
        case PHAL_VCA_SW_ID:
            status = phalVca_Sw_GetConfig((phalVca_Sw_DataParams_t *) pDataParams, wConfig, pValue);
            break;
#endif /* NXPBUILD__PHAL_VCA_SW */

#ifdef NXPBUILD__PHAL_VCA_SAM_NONX
        case PHAL_VCA_SAMAV2_NONX_ID:
            status = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_VCA);
            break;

#ifdef NXPBUILD__PHAL_VCA_SAMAV3_NONX
        case PHAL_VCA_SAMAV3_NONX_ID:
#endif /* NXPBUILD__PHAL_VCA_SAMAV3_NONX */
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
        case PHAL_VCA_SAM_NONX_ID:
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */
            status = phalVca_Sam_NonX_GetConfig(pDataParams, wConfig, pValue);
            break;
#endif /* NXPBUILD__PHAL_VCA_SAM_NONX */

#ifdef NXPBUILD__PHAL_VCA_SAM_X
        case PHAL_VCA_SAMAV2_X_ID:
            status = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_VCA);
            break;

#ifdef NXPBUILD__PHAL_VCA_SAMAV3_X
        case PHAL_VCA_SAMAV3_X_ID:
            status = phalVca_Sam_X_GetConfig((phalVca_SamAV3_X_DataParams_t *) pDataParams, wConfig, pValue);
            break;
#endif /* NXPBUILD__PHAL_VCA_SAMAV3_X */
#endif /* NXPBUILD__PHAL_VCA_SAM_X */

        default:
            status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);
            break;
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
#ifdef NXPBUILD__PH_LOG
    if((status & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, pValue_log, pValue);
    }
#endif
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
    return status;
}

phStatus_t phalVca_SetSessionKeyUtility(void * pDataParams, uint8_t * pSessionKey, uint8_t bAuthMode)
{
    phStatus_t PH_MEMLOC_REM status;
    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalVca_SetSessionKeyUtility");

    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pSessionKey);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(bAuthMode);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT8(PH_LOG_LOGTYPE_DEBUG, bAuthMode_log, &bAuthMode);
    PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pSessionKey_log, pSessionKey, 16);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* Validate the parameters */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_VCA);
    PH_ASSERT_NULL_PARAM(pSessionKey, PH_COMP_AL_VCA);

    /* Check data parameters */
    if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_VCA)
    {
        status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
        return status;
    }

    /* Perform operation on active layer */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_VCA_SW
        case PHAL_VCA_SW_ID:
            status = phalVca_Sw_SetSessionKeyUtility((phalVca_Sw_DataParams_t *) pDataParams, pSessionKey, bAuthMode);
            break;
#endif /* NXPBUILD__PHAL_VCA_SW */

#ifdef NXPBUILD__PHAL_VCA_SAM_X
        case PHAL_VCA_SAMAV2_X_ID:
#ifdef NXPBUILD__PHAL_VCA_SAMAV3_X
        case PHAL_VCA_SAMAV3_X_ID:
#endif /* NXPBUILD__PHAL_VCA_SAMAV3_X */
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
        case PHAL_VCA_SAM_X_ID:
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */
            status = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_VCA);
            break;
#endif /* NXPBUILD__PHAL_VCA_SAM_X */

        default:
            status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);
            break;
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
    return status;
}

phStatus_t phalVca_SetApplicationType(void * pDataParams, void * pAlDataParams)
{
    phStatus_t PH_MEMLOC_REM status;
    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalVca_SetApplicationType");
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    /* Validate the parameters */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_VCA);
    PH_ASSERT_NULL_PARAM(pAlDataParams, PH_COMP_AL_VCA);

    /* Check data parameters */
    if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_VCA)
    {
        status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
        return status;
    }

    /* Perform operation on active layer */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_VCA_SW
        case PHAL_VCA_SW_ID:
            status = phalVca_Sw_SetApplicationType((phalVca_Sw_DataParams_t *) pDataParams, pAlDataParams);
            break;
#endif /* NXPBUILD__PHAL_VCA_SW */

#ifdef NXPBUILD__PHAL_VCA_SAM_X
        case PHAL_VCA_SAMAV2_X_ID:
#ifdef NXPBUILD__PHAL_VCA_SAMAV3_X
        case PHAL_VCA_SAMAV3_X_ID:
#endif /* NXPBUILD__PHAL_VCA_SAMAV3_X */
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
        case PHAL_VCA_SAM_X_ID:
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */
            status = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_VCA);
            break;
#endif /* NXPBUILD__PHAL_VCA_SAM_X */

        default:
            status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);
            break;
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
    return status;
}

#ifdef NXPBUILD__PH_CRYPTOSYM
phStatus_t phalVca_DecryptResponse(void * pDataParams, uint16_t wKeyNo, uint16_t wKeyVersion, uint8_t * pInData, uint8_t * pRandChal,
    uint8_t * pVCData)
{
    phStatus_t PH_MEMLOC_REM status;
    PH_LOG_HELPER_ALLOCATE_TEXT(bFunctionName, "phalVca_DecryptResponse");

    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wKeyNo);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(wKeyVersion);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pInData);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pRandChal);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(pVCData);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(status);

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wKeyNo_log, &wKeyNo);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_DEBUG, wKeyVersion_log, &wKeyVersion);
    PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pInData_log, pInData, 32);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_ENTER);

    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_VCA);
    PH_ASSERT_NULL_PARAM(pInData, PH_COMP_AL_VCA);

    /* Check data parameters */
    if(PH_GET_COMPCODE(pDataParams) != PH_COMP_AL_VCA)
    {
        status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);
        PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
        PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
        PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
        return status;
    }

    /* Perform operation on active layer */
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_VCA_SW
        case PHAL_VCA_SW_ID:
            status = phalVca_Sw_DecryptResponse((phalVca_Sw_DataParams_t *) pDataParams, wKeyNo, wKeyVersion, pInData, pRandChal, pVCData);
            break;
#endif /* NXPBUILD__PHAL_VCA_SW */

#ifdef NXPBUILD__PHAL_VCA_SAM_X
        case PHAL_VCA_SAMAV2_X_ID:
#ifdef NXPBUILD__PHAL_VCA_SAMAV3_X
        case PHAL_VCA_SAMAV3_X_ID:
#endif /* NXPBUILD__PHAL_VCA_SAMAV3_X */
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
        case PHAL_VCA_SAM_X_ID:
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */
            status = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_VCA);
            break;
#endif /* NXPBUILD__PHAL_VCA_SAM_X */

        default:
            status = PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);
            break;
    }

    PH_LOG_HELPER_ADDSTRING(PH_LOG_LOGTYPE_INFO, bFunctionName);
    PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pRandChal_log, pRandChal, 16);
    PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, pVCData_log, pVCData, 16);
    PH_LOG_HELPER_ADDPARAM_UINT16(PH_LOG_LOGTYPE_INFO, status_log, &status);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_LEAVE);
    return status;
}
#endif /* NXPBUILD__PH_CRYPTOSYM */

#endif /* NXPBUILD__PHAL_VCA */
