/*
 * Copyright 2016 - 2018, 2020, 2024 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Sam S Virtual Card Architecture(R) Application Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phhalHw.h>
#include <phalVca.h>
#include <phpalMifare.h>
#include <phKeyStore.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHAL_VCA_SAM_NONX

#include <phhalHw_SamAV2_Cmd.h>
#include "../phalVca_Int.h"
#include "phalVca_Sam_NonX.h"

#ifdef NXPBUILD__PHAL_VCA_INTERNAL
#include "../comps/phhalHw/src/Sam/Commands/07_VirtualCard/phhalHw_Sam_Cmd_VC.h"
#include "../comps/phhalHw/src/Sam/Commands/02_SecurityConfiguration/phhalHw_Sam_Cmd_SC.h"
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */

#ifdef NXPBUILD__PHAL_VCA_SAMAV3_NONX
#include <phhalHw_SamAV3_Cmd.h>
#endif /* NXPBUILD__PHAL_VCA_SAMAV3_NONX */

phStatus_t phalVca_SamAV2_NonX_Init(phalVca_SamAV2_DataParams_t * pDataParams, uint16_t wSizeOfDataParams, phhalHw_SamAV2_DataParams_t * pSamHal,
    void * pPalMifareDataParams, uint8_t * pRndq, phalVca_SamAV2_KeyDuos_t * pKeyDuos, uint8_t bNumKeyDuos,
    phalVca_SamAV2_CardTableEntry_t * pCardTableStorage, uint16_t wNumCardTableStorageEntries)
{
    if(sizeof(phalVca_SamAV2_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);
    }

    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_VCA);
    PH_ASSERT_NULL_PARAM(pPalMifareDataParams, PH_COMP_AL_VCA);
    PH_ASSERT_NULL_PARAM(pSamHal, PH_COMP_AL_VCA);
    PH_ASSERT_NULL_PARAM(pRndq, PH_COMP_AL_VCA);
    PH_ASSERT_NULL_PARAM(pKeyDuos, PH_COMP_AL_VCA);
    if(wNumCardTableStorageEntries)
        PH_ASSERT_NULL_PARAM(pCardTableStorage, PH_COMP_AL_VCA);

    /* init private data */
    pDataParams->wId = PH_COMP_AL_VCA | PHAL_VCA_SAMAV2_NONX_ID;
    pDataParams->pPalMifareDataParams = pPalMifareDataParams;
    pDataParams->pSamHal = pSamHal;
    pDataParams->pRndq = pRndq;
    pDataParams->pKeyDuos = pKeyDuos;
    pDataParams->bNumKeyDuos = bNumKeyDuos;
    pDataParams->wCurrentCardTablePos = 0U;
    pDataParams->pCardTable = pCardTableStorage;
    pDataParams->wNumCardTableEntries = wNumCardTableStorageEntries;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_VCA);
}

#ifdef NXPBUILD__PHAL_VCA_SAMAV3_NONX
phStatus_t phalVca_SamAV3_NonX_Init(phalVca_SamAV3_NonX_DataParams_t * pDataParams, uint16_t wSizeOfDataParams, phhalHw_SamAV3_DataParams_t * pSamHal,
    void * pPalMifareDataParams)
{
    if(sizeof(phalVca_SamAV3_NonX_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);
    }

    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_VCA);
    PH_ASSERT_NULL_PARAM(pPalMifareDataParams, PH_COMP_AL_VCA);
    PH_ASSERT_NULL_PARAM(pSamHal, PH_COMP_AL_VCA);

    /* init private data */
    pDataParams->wId = PH_COMP_AL_VCA | PHAL_VCA_SAMAV3_NONX_ID;
    pDataParams->pPalMifareDataParams = pPalMifareDataParams;
    pDataParams->pSamHal = pSamHal;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_VCA);
}
#endif /* NXPBUILD__PHAL_VCA_SAMAV3_NONX */

#ifdef NXPBUILD__PHAL_VCA_INTERNAL
phStatus_t phalVca_Sam_NonX_Init(phalVca_Sam_NonX_DataParams_t * pDataParams, uint16_t wSizeOfDataParams, void * pSamHal,
    void * pPalMifareDataParams)
{
    if(sizeof(phalVca_Sam_NonX_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_VCA);
    }

    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_VCA);
    PH_ASSERT_NULL_PARAM(pPalMifareDataParams, PH_COMP_AL_VCA);
    PH_ASSERT_NULL_PARAM(pSamHal, PH_COMP_AL_VCA);

    /* init private data */
    pDataParams->wId = PH_COMP_AL_VCA | PHAL_VCA_SAM_NONX_ID;
    pDataParams->pPalMifareDataParams = pPalMifareDataParams;
    pDataParams->pSamHal = pSamHal;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_VCA);
}
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */




phStatus_t phalVca_Sam_NonX_StartCardSelection(phalVca_SamAV2_DataParams_t * pDataParams)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM wIndex = 0;

    /* Obtain Random Numbers */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV2_Cmd_SAM_GetChallenge(pDataParams->pSamHal,
        (uint8_t) (12 * pDataParams->wNumCardTableEntries), pDataParams->pRndq));

    /* Send the complete stream to the SAM */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV2_Cmd_SAM_VirtualCardSupportMfp(
        pDataParams->pSamHal,
        PH_EXCHANGE_BUFFER_FIRST,
        NULL,
        0U,
        NULL,
        0U,
        NULL,
        NULL));

    pDataParams->wCurrentCardTablePos = 0U;
    for(wIndex = 0; wIndex < pDataParams->wNumCardTableEntries; ++wIndex)
    {
        pDataParams->pCardTable[wIndex].bNumKeyDuos = 0U;
        pDataParams->pCardTable[wIndex].wIidIndex = 0xFFU;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_VCA);
}

phStatus_t phalVca_Sam_NonX_FinalizeCardSelection(phalVca_SamAV2_DataParams_t * pDataParams, uint16_t* pNumValidIids)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;
    uint16_t    PH_MEMLOC_REM wIndex = 0;
    uint16_t    PH_MEMLOC_REM wIidIndex = 0;
    uint8_t     PH_MEMLOC_REM bValidIids = 0;
    uint8_t     PH_MEMLOC_REM bDummyIidIndex = 0;
    uint8_t     PH_MEMLOC_REM bDummyValidIids = 0;

    /* Check if we have something in the table */
    if(!pDataParams->wCurrentCardTablePos)
    {
        *pNumValidIids = 0U;
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_VCA);
    }

    /* Send the complete stream to the SAM */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV2_Cmd_SAM_VirtualCardSupportMfp(
        pDataParams->pSamHal,
        PH_EXCHANGE_BUFFER_LAST,
        NULL,
        0U,
        &pDataParams->pPcdCaps[1U],
        pDataParams->pPcdCaps[0U],
        &pResponse,
        &wRespLen));

    if(wRespLen != pDataParams->wCurrentCardTablePos * 11U)
    {
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_AL_VCA);
    }

    wIidIndex = 0U;

    /* Now we need to store the card data */
    for(wIndex = 0U; wIndex < pDataParams->wCurrentCardTablePos; ++wIndex)
    {
        /* We always copy the data into the card table */
        (void) memcpy(pDataParams->pCardTable[wIndex].pCardData, &pResponse[1U], 10U);

        /* Was it a valid entry? */
        if(pResponse[0U] != 0xFFU)
        {
            /* Copy the found IID index */
            ++bValidIids;
            pDataParams->pCardTable[wIndex].wIidIndex = wIidIndex + pResponse[0U];
        }
        /* Invalidate the found IID index */
        else
        {
            ++bDummyValidIids;
            pDataParams->pCardTable[wIndex].wIidIndex = bDummyIidIndex + pResponse[0U];
        }

        /* Update the IidIndex */
        wIidIndex = wIidIndex + pDataParams->pCardTable[wIndex].bNumKeyDuos;
        pResponse += (10U /* Card Data */ + 1U /* Success index */);
    }

    *pNumValidIids = bValidIids;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_VCA);
}

phStatus_t phalVca_Sam_NonX_SelectVc(phalVca_SamAV2_DataParams_t * pDataParams, uint16_t wValidIidIndex, uint16_t wKeyNo,
    uint16_t wKeyVer)
{
    /* local variables */
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[9];
    uint8_t *   PH_MEMLOC_REM pResponse;
    uint16_t    PH_MEMLOC_REM wRxLength;

    wStatus = phalVca_Sam_NonX_ResolveValidIndex(pDataParams, wValidIidIndex, &wValidIidIndex);

    /* for the case of an overflow we always send the first random number to the SAM */
    if((wStatus  & PH_ERR_MASK) == PH_ERR_INVALID_PARAMETER)
    {
        wValidIidIndex = 0U;
    }
    else
    {
        PH_CHECK_SUCCESS(wStatus);
    }

    aCmdBuff[0] = PHAL_VCA_CMD_SVC;

    /* Prepare Buffer */
    PH_CHECK_SUCCESS_FCT(wStatus1, phhalHw_SamAV2_Cmd_SAM_SelectVirtualCardMfp(
        pDataParams->pSamHal,
        (uint8_t) wKeyNo,
        (uint8_t) wKeyVer,
        &pDataParams->pRndq[wValidIidIndex * 12U],
        NULL,
        0U,
        &aCmdBuff[1]));

    /* command exchange */
    PH_CHECK_SUCCESS_FCT(wStatus1, phpalMifare_ExchangeL4(
        pDataParams->pPalMifareDataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        9U,
        &pResponse,
        &wRxLength));

    PH_CHECK_SUCCESS_FCT(wStatus1, phalVca_Int_ComputeErrorResponse(wRxLength, pResponse[0]));

    /* check response length */
    if(wRxLength != 1U /* STATUS */)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_VCA);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_VCA);
}

phStatus_t phalVca_Sam_NonX_DeselectVc(phalVca_SamAV2_DataParams_t * pDataParams)
{
    /* local variables */
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[1 /* command code */];
    uint8_t *   PH_MEMLOC_REM pResponse;
    uint16_t    PH_MEMLOC_REM wRxLength;

    /* command code */
    aCmdBuff[0] = PHAL_VCA_CMD_DVC;

    /* command exchange */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
        pDataParams->pPalMifareDataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        1U,
        &pResponse,
        &wRxLength));

    /* check response */
    PH_CHECK_SUCCESS_FCT(wStatus, phalVca_Int_ComputeErrorResponse(wRxLength, pResponse[0]));

    /* check response length */
    if(wRxLength != 1U /* STATUS */)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_VCA);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_VCA);
}

phStatus_t phalVca_Sam_NonX_VcSupport(phalVca_SamAV2_DataParams_t * pDataParams, uint8_t * pIid, uint16_t wKeyEncNo, uint16_t wKeyEncVer,
    uint16_t wKeyMacNo, uint16_t wKeyMacVer)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[1 /* command code */];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;
    uint16_t    PH_MEMLOC_REM wIndex = 0;

    /* send buffer */
    aCmdBuff[0U] = PHAL_VCA_CMD_VCS;

    /*Check available space in key duos list */
    if(pDataParams->pCardTable[pDataParams->wCurrentCardTablePos].bNumKeyDuos >= pDataParams->bNumKeyDuos)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_AL_VCA);
    }

    /* Add keys to key list */
    wIndex = pDataParams->pCardTable[pDataParams->wCurrentCardTablePos].bNumKeyDuos;
    pDataParams->pKeyDuos[wIndex].bKeyEncNumber = (uint8_t) wKeyEncNo;
    pDataParams->pKeyDuos[wIndex].bKeyEncVersion = (uint8_t) wKeyEncVer;
    pDataParams->pKeyDuos[wIndex].bKeyMacNumber = (uint8_t) wKeyMacNo;
    pDataParams->pKeyDuos[wIndex].bKeyMacVersion = (uint8_t) wKeyMacVer;
    ++pDataParams->pCardTable[pDataParams->wCurrentCardTablePos].bNumKeyDuos;

    /* buffer the command frame */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
        pDataParams->pPalMifareDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        aCmdBuff,
        1U,
        &pResponse,
        &wRespLen));

    /* Append IID and exchange the command */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
        pDataParams->pPalMifareDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        pIid,
        PHAL_VCA_IID_SIZE,
        &pResponse,
        &wRespLen));

    /* check response */
    PH_CHECK_SUCCESS_FCT(wStatus, phalVca_Int_ComputeErrorResponse(wRespLen, pResponse[0]));

    /* check response length */
    if(wRespLen != 1U /* STATUS */)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_VCA);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_VCA);
}

phStatus_t phalVca_Sam_NonX_VcSupportLast(phalVca_SamAV2_DataParams_t * pDataParams, uint8_t * pIid, uint8_t bLenCap, uint8_t * pPcdCapabilities,
    uint16_t wKeyEncNo, uint16_t wKeyEncVer, uint16_t wKeyMacNo, uint16_t wKeyMacVer)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[1 /* CMD */];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;
    uint16_t    PH_MEMLOC_REM wIndex;

    /* Now we create the message to be sent */
    aCmdBuff[0] = PHAL_VCA_CMD_VCSL;
    (void) memset(pDataParams->pPcdCaps, 0x00U, 4U);

    /* Copy PCD Caps */
    if(bLenCap)
    {
        if(bLenCap > 3U)
        {
            bLenCap = 3U;
        }
        pDataParams->pPcdCaps[0U] = bLenCap;

        (void) memcpy(&pDataParams->pPcdCaps[1U], pPcdCapabilities, bLenCap);
    }

    /* Check available space in key duos list */
    if(pDataParams->pCardTable[pDataParams->wCurrentCardTablePos].bNumKeyDuos >= pDataParams->bNumKeyDuos)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_AL_VCA);
    }

    /* Add keys to key list */
    wIndex = pDataParams->pCardTable[pDataParams->wCurrentCardTablePos].bNumKeyDuos;
    pDataParams->pKeyDuos[wIndex].bKeyEncNumber = (uint8_t) wKeyEncNo;
    pDataParams->pKeyDuos[wIndex].bKeyEncVersion = (uint8_t) wKeyEncVer;
    pDataParams->pKeyDuos[wIndex].bKeyMacNumber = (uint8_t) wKeyMacNo;
    pDataParams->pKeyDuos[wIndex].bKeyMacVersion = (uint8_t) wKeyMacVer;
    ++pDataParams->pCardTable[pDataParams->wCurrentCardTablePos].bNumKeyDuos;

    /* buffer command frame */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
        pDataParams->pPalMifareDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        aCmdBuff,
        1U,
        &pResponse,
        &wRespLen));

    /* buffer installation identifier */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
        pDataParams->pPalMifareDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pIid,
        PHAL_VCA_IID_SIZE,
        &pResponse,
        &wRespLen));

    /* buffer RNDQ identifier */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
        pDataParams->pPalMifareDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        &pDataParams->pRndq[pDataParams->wCurrentCardTablePos * 12U],
        12U,
        &pResponse,
        &wRespLen));

    /* append PCDCaps and transmit the command */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
        pDataParams->pPalMifareDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        pDataParams->pPcdCaps,
        1U + bLenCap,
        &pResponse,
        &wRespLen));

    PH_CHECK_SUCCESS_FCT(wStatus, phalVca_Int_ComputeErrorResponse(wRespLen, pResponse[0]));

    /* check response length */
    if(wRespLen != 1U /*Status */ + 16U /* Cryptogram */ + 8U /*MAC */)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_VCA);
    }

    /* Add Number key Duos */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV2_Cmd_SAM_VirtualCardSupportMfp(
        pDataParams->pSamHal,
        PH_EXCHANGE_BUFFER_CONT,
        &pDataParams->pCardTable[pDataParams->wCurrentCardTablePos].bNumKeyDuos,
        1U,
        NULL,
        0U,
        NULL,
        NULL));

    /* Add key Duos */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV2_Cmd_SAM_VirtualCardSupportMfp(
        pDataParams->pSamHal,
        PH_EXCHANGE_BUFFER_CONT,
        (uint8_t*) pDataParams->pKeyDuos,
        pDataParams->pCardTable[pDataParams->wCurrentCardTablePos].bNumKeyDuos << 2U,
        NULL,
        0U,
        NULL,
        NULL));

    /* Add RNDQ */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV2_Cmd_SAM_VirtualCardSupportMfp(
        pDataParams->pSamHal,
        PH_EXCHANGE_BUFFER_CONT,
        &pDataParams->pRndq[pDataParams->wCurrentCardTablePos * 12U],
        12U,
        NULL,
        0U,
        NULL,
        NULL));

    /* Add Card Data */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV2_Cmd_SAM_VirtualCardSupportMfp(
        pDataParams->pSamHal,
        PH_EXCHANGE_BUFFER_CONT,
        &pResponse[1U],
        24U,
        NULL,
        0U,
        NULL,
        NULL));

    pDataParams->wCurrentCardTablePos++;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_VCA);
}

phStatus_t phalVca_Sam_NonX_GetIidInfo(phalVca_SamAV2_DataParams_t * pDataParams, uint16_t wValidIidIndex, uint16_t * pIidIndex, uint8_t * pVcUidSize,
    uint8_t * pVcUid, uint8_t * pInfo, uint8_t * pPdCapabilities)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phalVca_Sam_NonX_ResolveValidIndex(pDataParams, wValidIidIndex, &wValidIidIndex));

    *pIidIndex = pDataParams->pCardTable[wValidIidIndex].wIidIndex;
    *pInfo = pDataParams->pCardTable[wValidIidIndex].pCardData[0U];
    pPdCapabilities[0U] = pDataParams->pCardTable[wValidIidIndex].pCardData[1U];
    pPdCapabilities[1U] = pDataParams->pCardTable[wValidIidIndex].pCardData[2U];

    if(*pInfo & 0x80U)
    {
        *pVcUidSize = 4U;
        (void) memcpy(pVcUid, &pDataParams->pCardTable[wValidIidIndex].pCardData[3U], 4U);
    }
    else
    {
        *pVcUidSize = 7U;
        (void) memcpy(pVcUid, &pDataParams->pCardTable[wValidIidIndex].pCardData[3U], 7U);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_VCA);
}

#ifdef NXPBUILD__PHAL_VCA_SAMAV3_NONX
phStatus_t phalVca_Sam_NonX_IsoSelect(void * pDataParams, uint8_t bSelectionControl, uint8_t bOption, uint8_t bDFnameLen,
    uint8_t * pDFname, uint8_t * pDivInput, uint8_t bDivInputLen, uint8_t bEncKeyNo, uint8_t bEncKeyVer, uint8_t bMacKeyNo, uint8_t bMacKeyVer,
    uint8_t * pResponse, uint16_t * pRespLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[PHAL_VCA_CMD_SIZE];
    uint16_t    PH_MEMLOC_REM wCmdLen = 0;
    uint8_t     PH_MEMLOC_REM aLE[2];
    uint8_t *   PH_MEMLOC_REM pPICC_Response = NULL;
    uint16_t    PH_MEMLOC_REM wPICC_RespLen = 0;
    uint8_t *   PH_MEMLOC_REM pSAM_Response = NULL;
    uint16_t    PH_MEMLOC_REM wSAM_RespLen = 0;

    if((pDFname == NULL) || (bDFnameLen > 16U))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_VCA);
    }
    if(bSelectionControl != 0x04U)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_VCA);
    }

    /* Clear the buffers. */
    (void) memset(aCmdBuff, 0x00U, sizeof(aCmdBuff));
    (void) memset(aLE, 0x00U, sizeof(aLE));

    /* Frame the command buffer */
    aCmdBuff[wCmdLen++] = 0x00U;
    aCmdBuff[wCmdLen++] = PHAL_VCA_CMD_ISOSVC;
    aCmdBuff[wCmdLen++] = bSelectionControl;
    aCmdBuff[wCmdLen++] = 0x00U;

    if (PHAL_VCA_RESOLVE_EXTENDED_LENGTH_APDU(pDataParams))
    {
        aCmdBuff[wCmdLen++] = 0x00U;
        aCmdBuff[wCmdLen++] = 0x00U;
    }

    aCmdBuff[wCmdLen++] = bDFnameLen;

    /* Buffer the command information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
        PHAL_VCA_RESOLVE_PAL_MIFARE_DATAPARAMS(pDataParams),
        PH_EXCHANGE_BUFFER_FIRST,
        aCmdBuff,
        wCmdLen,
        NULL,
        NULL));

    /* Buffer DFName and exchange the buffered information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
        PHAL_VCA_RESOLVE_PAL_MIFARE_DATAPARAMS(pDataParams),
        PH_EXCHANGE_BUFFER_CONT,
        pDFname,
        bDFnameLen,
        NULL,
        NULL));

    /* Command exchange with Le. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
        PHAL_VCA_RESOLVE_PAL_MIFARE_DATAPARAMS(pDataParams),
        PH_EXCHANGE_BUFFER_LAST,
        aLE,
        (uint16_t) (PHAL_VCA_RESOLVE_EXTENDED_LENGTH_APDU(pDataParams) ? 2U : 1U),
        &pPICC_Response,
        &wPICC_RespLen));

    /* Compute the status from PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalVca_Sam_NonX_ComputeErrorResponse_Extended(pDataParams,
        (uint16_t) ((pPICC_Response[wPICC_RespLen - 2U] << 8U) | pPICC_Response[wPICC_RespLen - 1U])));

    /* AuthVCMandatory is set. */
    if(wPICC_RespLen == (PHAL_VCA_AUTH_RND_LEN + 6 /* TLV Header + Status*/))
    {
        /* Remove the status code. */
        wPICC_RespLen -= 2U;

        if((pPICC_Response[0U] == 0x6FU) && (pPICC_Response[1U] == 0x22U) &&
            (pPICC_Response[2U] == 0x85U) && (pPICC_Response[3U] == 0x20U))
        {
            PH_CHECK_SUCCESS_FCT(wStatus, phalVca_Sam_NonX_SelectVC(
                pDataParams,
                bOption,
                bEncKeyNo,
                bEncKeyVer,
                bMacKeyNo,
                bMacKeyVer,
                &pPICC_Response[4U /* TLV Header */],
                (uint8_t) (wPICC_RespLen - 4U /* TLV Header */),
                pDivInput,
                bDivInputLen,
                &pSAM_Response,
                &wSAM_RespLen));

            /* Copy the VCData to the response buffer. */
            (void) memcpy(pResponse, pSAM_Response, wSAM_RespLen);
            *pRespLen = wSAM_RespLen;
        }

        /* Case-3: [if TargetVC != NULL AND TargetVC.AuthVCMandatory == false AND (IID is DESFire application DF name)]
        * FCI[36] bytes shall be stored in file ID 31 of the DF */
        else
        {
            (void) memcpy(pResponse, pPICC_Response, wPICC_RespLen);
            *pRespLen = wPICC_RespLen;
        }
    }

    /* AuthVCMandatory flag is not set and IsoSelect is success */
    else if(wPICC_RespLen == 2U)
    {
        *pRespLen = (uint16_t) (wPICC_RespLen - 2U);
    }

    /* AuthVCMandatory flag is not set and FileID is returned */
    else
    {
        (void) memcpy(pResponse, pPICC_Response, (wPICC_RespLen - 2U));
        *pRespLen = (wPICC_RespLen - 2U);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_VCA);
}

phStatus_t phalVca_Sam_NonX_IsoExternalAuthenticate(void * pDataParams, uint8_t * pInData)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[PHAL_VCA_CMD_SIZE];
    uint16_t    PH_MEMLOC_REM wCmdLen = 0;
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    /* Validate the parameter. */
    if(pInData == NULL)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_VCA);
    }

    /* Prepare "IsoExternlAuthenticate" command. */
    aCmdBuff[wCmdLen++] = 0x00;                     /* Class is always 0x00 */
    aCmdBuff[wCmdLen++] = PHAL_VCA_CMD_ISOEXT_AUTH; /* INS */
    aCmdBuff[wCmdLen++] = 0x00;                     /* P1 */
    aCmdBuff[wCmdLen++] = 0x00;                     /* P2 */

    if(PHAL_VCA_RESOLVE_EXTENDED_LENGTH_APDU(pDataParams))
    {
        aCmdBuff[wCmdLen++] = 0x00;
        aCmdBuff[wCmdLen++] = 0x00;
    }

    aCmdBuff[wCmdLen++] = 0x08; /* MAC length. */

    /* Copy the Input information to Command buffer. */
    (void) memcpy(&aCmdBuff[wCmdLen], pInData, 8);
    wCmdLen += 8;

    /* Exchange the buffered information to PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
        PHAL_VCA_RESOLVE_PAL_MIFARE_DATAPARAMS(pDataParams),
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        wCmdLen,
        &pResponse,
        &wRespLen));

    /* Compute the status from PICC. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalVca_Sam_NonX_ComputeErrorResponse_Extended(pDataParams,
        (uint16_t) ((pResponse[wRespLen - 2] << 8) | pResponse[wRespLen - 1])));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_VCA);
}
#endif /* NXPBUILD__PHAL_VCA_SAMAV3_NONX */






phStatus_t phalVca_Sam_NonX_ProximityCheck(phalVca_SamAV2_DataParams_t * pDataParams, uint8_t bGenerateRndC, uint8_t * pRndC, uint8_t bPps1,
    uint8_t bNumSteps, uint16_t wKeyNo, uint16_t wKeyVer, uint8_t * pUsedRndC)
{
    /* local variables */
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[9];
    uint8_t     PH_MEMLOC_REM bRndRC[14];
    uint8_t     PH_MEMLOC_REM bRndC[7];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    /* parameter checking */
    if((bGenerateRndC == 0U) && (pRndC == NULL))
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_AL_VCA);
    }

    /* send "Prepare Proximity Check" command */
    PH_CHECK_SUCCESS_FCT(wStatus1, phalVca_Int_PrepareProximityCheck(pDataParams->pPalMifareDataParams));

    /* check whether to generate RndC or not */
    if(bGenerateRndC)
    {
        pRndC = bRndC;
        PH_CHECK_SUCCESS_FCT(wStatus1, phhalHw_SamAV2_Cmd_SAM_GetChallenge(pDataParams->pSamHal, PHAL_VCA_PC_RND_LEN, pRndC));
    }

    /* send "Proximity Check" command */
    PH_CHECK_SUCCESS_FCT(wStatus1, phalVca_Int_ProximityCheck(
        pDataParams->pPalMifareDataParams,
        bNumSteps,
        pRndC,
        bRndRC));

    /* send "Verify Proximity Check" command */
    wStatus = phhalHw_SamAV2_Cmd_SAM_ProximityCheckMfp_Part1(pDataParams->pSamHal,
        (uint8_t) wKeyNo,
        (uint8_t) wKeyVer,
        bPps1,
        bRndRC,
        NULL,
        0U,
        &aCmdBuff[1]);

    if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING)
    {
        return wStatus;
    }

    aCmdBuff[0] = PHAL_VCA_CMD_VPC;

    /* append the MAC and exchange frame */
    PH_CHECK_SUCCESS_FCT(wStatus1, phpalMifare_ExchangeL4(
        pDataParams->pPalMifareDataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        9U,
        &pResponse,
        &wRespLen));

    /* Check the status Code */
    wStatus = phalVca_Int_ComputeErrorResponse(wRespLen, pResponse[0U]);
    if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS)
    {
        /* finish SAM chaining with KillAuthenticate command */
        /* Kill only card Auth */
        wStatus1 = phhalHw_SamAV2_Cmd_SAM_KillAuthentication(pDataParams->pSamHal, 0x01U);
        return wStatus;
    }

    /* check response length */
    if(wRespLen != 1U /* Status */ + 8U /* MAC */)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_VCA);
    }

    /* send "Verify Proximity Check" command */
    wStatus = phhalHw_SamAV2_Cmd_SAM_ProximityCheckMfp_Part2(pDataParams->pSamHal, &pResponse[1]);

    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAMAV2_ERR_CRYPTO)
    {
        return PH_ADD_COMPCODE(PHAL_VCA_ERR_AUTH, PH_COMP_AL_VCA);
    }

    PH_CHECK_SUCCESS(wStatus);
    /* Copy RndC if requested */
    if(pUsedRndC != NULL)
    {
        (void) memcpy(pUsedRndC, pRndC, PHAL_VCA_PC_RND_LEN);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_VCA);
}

#ifdef NXPBUILD__PHAL_VCA_SAMAV3_NONX
phStatus_t phalVca_Sam_NonX_ProximityCheckNew(void * pDataParams, uint8_t bGenerateRndC, uint8_t * pPrndC,
    uint8_t bNumSteps, uint16_t wKeyNo, uint16_t wKeyVer, uint8_t * pDivInput, uint8_t bDivInputLen, uint8_t * pOption,
    uint8_t * pPubRespTime, uint8_t * pResponse, uint16_t * pRespLen, uint8_t * pCumRndRC)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aPPCData[10];
    uint8_t     PH_MEMLOC_REM bPPCDataLen = 0;
    uint8_t     PH_MEMLOC_REM aResponse[10];
    uint16_t    PH_MEMLOC_REM wRespLen = 0;
    uint8_t *   PH_MEMLOC_REM pMac = NULL;
    uint8_t     PH_MEMLOC_REM bPiccRetCode = 0;

    /* Perform Prepare PC command execution with PICC */
    PH_CHECK_SUCCESS_FCT(wStatus, phalVca_Sam_NonX_PrepareProximityCheckNew(
        pDataParams,
        pOption,
        pPubRespTime,
        pResponse,
        pRespLen));

    /* Frame PPCData buffer. */
    aPPCData[bPPCDataLen++] = *pOption;
    aPPCData[bPPCDataLen++] = pPubRespTime[0U];
    aPPCData[bPPCDataLen++] = pPubRespTime[1U];

    /* Add [PPS1] || [ActBitRate] information to PPCData buffer if set in Option. */
    (void) memcpy(&aPPCData[bPPCDataLen], pResponse, *pRespLen);
    bPPCDataLen += (uint8_t) *pRespLen;

    /* Perform Proximity Check with PICC */
    PH_CHECK_SUCCESS_FCT(wStatus, phalVca_Sam_NonX_ExecuteProximityCheckNew(
        pDataParams,
        bGenerateRndC,
        pPrndC,
        pPubRespTime,
        bNumSteps,
        pCumRndRC));

    /* Perform Proximity check part 1 exchange with SAM */
    wStatus = phalVca_Sam_NonX_ProximityCheck_Part1(
        pDataParams,
        (uint8_t) (bDivInputLen ? PHHAL_HW_SAMAV3_CMD_PROXIMITY_CHECK_DIV_ON : PHHAL_HW_SAMAV3_CMD_PROXIMITY_CHECK_DIV_OFF),
        (uint8_t) wKeyNo,
        (uint8_t) wKeyVer,
        aPPCData,
        bPPCDataLen,
        pCumRndRC,
        16U,
        pDivInput,
        bDivInputLen,
        &pMac,
        &wRespLen);

    /* Check if chaining status is returned from HAL. */
    if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING)
    {
        /* Return the status. */
        return wStatus;
    }

    /* Perform Verify PC with PICC */
    wStatus = phalVca_Sam_NonX_VerifyProximityCheckNew(
        pDataParams,
        pMac,
        aResponse,
        &wRespLen);

    /* Perform Proximity check part 2 exchange with SAM */
    if(wRespLen > 0U)
    {
        wStatus = phalVca_Sam_NonX_ProximityCheck_Part2(
            pDataParams,
            aResponse,
            (uint8_t) wRespLen,
            &bPiccRetCode);

        /* Validate the response. */
        if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAMAV3_ERR_MIFARE_GEN)
        {
            PH_CHECK_SUCCESS_FCT(wStatus, phalVca_Int_ComputeErrorResponse_Extended(pDataParams, bPiccRetCode));
        }
        else
        {
            PH_CHECK_SUCCESS(wStatus);
        }
    }

    /*
    * Kill PICC Authentication for next SAM call to proceed further
    * This code update is based on information mentioned in MIFARE SAM AV3 known deviations from specification
    * section 5.2, to overcome the issue where if there is no payload for PART-2 exchange.
    */
    else
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phalVca_Sam_NonX_KillAuthentication(
            pDataParams,
            PHHAL_HW_SAMAV3_CMD_SAM_KILL_AUTHENTICATION_PARTIAL));
    }

    return wStatus;
}

phStatus_t phalVca_Sam_NonX_PrepareProximityCheckNew(void * pDataParams, uint8_t * pOption,
    uint8_t * pPubRespTime, uint8_t * pResponse, uint16_t * pRespLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t *   PH_MEMLOC_REM pResponse_Tmp = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen_Tmp = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[] = { PHAL_VCA_CMD_PPC };
    uint8_t     PH_MEMLOC_REM bOffset = 0;

    /* Exchange the data in ISO7816 format. */
    if(PHAL_VCA_RESOLVE_WRAPPED_MODE(pDataParams))
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phalVca_Sam_NonX_SendISOWrappedCmd(
            pDataParams,
            aCmdBuff,
            0x00U,  /* LC Value */
            &pResponse_Tmp,
            &wRespLen_Tmp));

        /* Validate the status. */
        PH_CHECK_SUCCESS_FCT(wStatus, phalVca_Int_ComputeErrorResponse_Extended(pDataParams,
            pResponse_Tmp[wRespLen_Tmp - 1U]));

        /* Adjusting the response length i.e. removing the status code. */
        wRespLen_Tmp -= 2U;
    }

    /* Exchange the data in Native format. */
    else
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
            PHAL_VCA_RESOLVE_PAL_MIFARE_DATAPARAMS(pDataParams),
            PH_EXCHANGE_DEFAULT,
            aCmdBuff,
            1U,
            &pResponse_Tmp,
            &wRespLen_Tmp));

        /* Validate the status. */
        PH_CHECK_SUCCESS_FCT(wStatus, phalVca_Int_ComputeErrorResponse_Extended(pDataParams, pResponse_Tmp[0U]));

        /* Incrementing the Index to point the response data */
        pResponse_Tmp++;

        /* Adjusting the response length i.e. removing the status code. */
        wRespLen_Tmp--;
    }

    /* Save Option from response data. */
    *pOption = pResponse_Tmp[bOffset++];

    /* Save Published Response Time from response data. */
    pPubRespTime[0U] = pResponse_Tmp[bOffset++];
    pPubRespTime[1U] = pResponse_Tmp[bOffset++];

    /* Save PPS from response data */
    if(*pOption & 0x01U)
    {
        *pResponse = pResponse_Tmp[bOffset];
        *pRespLen = 1U;
    }

    /* Save ActBitRate from response data */
    if(*pOption & 0x02U)
    {
        memcpy(pResponse, &pResponse_Tmp[bOffset], (wRespLen_Tmp - bOffset));
        *pRespLen = (uint8_t) (wRespLen_Tmp - bOffset);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_VCA);
}

phStatus_t phalVca_Sam_NonX_ExecuteProximityCheckNew(void * pDataParams, uint8_t bGenerateRndC,
    uint8_t * pPrndC, uint8_t * pPubRespTime, uint8_t bNumSteps, uint8_t * pCumRndRC)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bRndC[PHAL_VCA_SAMAV3_PC_RND_LEN];
    uint8_t     PH_MEMLOC_REM aCmdBuff[1 /* Command */ + 1 /* RndCLen */ + 8 /* RndC */];
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;
    uint8_t     PH_MEMLOC_REM bPayloadLen = 0;
    uint8_t     PH_MEMLOC_REM bRndCLen = 0;
    uint8_t     PH_MEMLOC_REM bRndRCLen = 0;
    uint16_t    PH_MEMLOC_REM wValue = 0;
    uint16_t    PH_MEMLOC_REM wThresholdTimeUpperLimit = 0;
    uint16_t    PH_MEMLOC_REM wThresholdTimeLowerLimit = 0;

    /* Validate the parameters. */
    if(((bGenerateRndC == 0U) && (pPrndC == NULL)) || (pCumRndRC == NULL))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_VCA);
    }

    /* Frame the command. */
    aCmdBuff[0U] = PHAL_VCA_CMD_PC;

    /* Get the random number from SAM. */
    if(bGenerateRndC)
    {
        pPrndC = bRndC;

        /* Get the Random Number from SAM */
        PH_CHECK_SUCCESS_FCT(wStatus, phalVca_Sam_NonX_GetRandom(pDataParams, 0x08U, pPrndC));
    }

    /* Exchange the ProximityCheck information.
     * Steps = 1: Only one iteration is made, All 8 random bytes in one Cmd.ProxmityCheck.
     * Steps = 2: Sends the first 1 random bytes in one Cmd.ProxmityCheck and the remaining 7 random byte in another one.
     * Steps = 3: 1 in one Cmd.ProxmityCheck, 1 in Second and remaining 6 in Third.
     * Steps = 4: 1 in one Cmd.ProxmityCheck, 1 in Second, 1 in third and remaining 6 in Fourth.
     * Steps = 5: 1 in one Cmd.ProxmityCheck, 1 in Second, 1 in third, 1 in fourth and remaining 4 in Fifth.
     * Steps = 6: 1 in one Cmd.ProxmityCheck, 1 in Second, 1 in third, 1 in fourth, 1 in Fifth and remaining 4 in Fifth.
     * Steps = 7: 1 in one Cmd.ProxmityCheck, 1 in Second, 1 in third, 1 in fourth, 1 in Fifth, 1 in Sixth and remaining 2 in Fifth.
     * Steps = 8: Sends 8 Cmd.ProxmityCheck with one random byte for each Exchange.
     */
    while(bNumSteps--)
    {
        /* RndC length */
        if(bNumSteps)
        {
            bPayloadLen = 1;
        }
        else
        {
            bPayloadLen = PHAL_VCA_SAMAV3_PC_RND_LEN - bRndCLen;
        }

        /* Length */
        aCmdBuff[1U] = bPayloadLen;

        /* RndC */
        (void) memcpy(&aCmdBuff[2], &pPrndC[bRndCLen], bPayloadLen);

        /* Get the bOption value for the checking the timing measurement ON/OFF */
        PH_CHECK_SUCCESS_FCT(wStatus, phalVca_GetConfig(pDataParams, PHAL_VCA_TIMING_MODE, &wValue));

        /* Start collecting the RC timeout. */
        if(wValue & 0x01U)
        {
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SetConfig(PHAL_VCA_RESOLVE_HAL_DATAPARAMS(pDataParams), PHHAL_HW_CONFIG_TIMING_MODE, PHHAL_HW_TIMING_MODE_FDT));
        }

        /* Exchange the data in ISO7816 format. */
        if(PHAL_VCA_RESOLVE_WRAPPED_MODE(pDataParams))
        {
            PH_CHECK_SUCCESS_FCT(wStatus, phalVca_Sam_NonX_SendISOWrappedCmd(
                pDataParams,
                aCmdBuff,
                (uint16_t) (1U + bPayloadLen),  /* bPayloadLen + RndC */
                &pResponse,
                &wRespLen));

            PH_CHECK_SUCCESS_FCT(wStatus, phalVca_Int_ComputeErrorResponse_Extended(pDataParams, pResponse[wRespLen - 1U]));

            /* Adjusting the response length i.e. removing the status code. */
            wRespLen -= 2U;
        }

        /* Exchange the command in Native format. */
        else
        {
            PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
                PHAL_VCA_RESOLVE_PAL_MIFARE_DATAPARAMS(pDataParams),
                PH_EXCHANGE_DEFAULT,
                aCmdBuff,
                2U + bPayloadLen,   /* (INS + bPayloadLen) + RndC */
                &pResponse,
                &wRespLen));

            /*
             * Response validation should not be performed in case if the length is
             *      1 byte  : One byte can be either a valid response or a error code which is difficult to identify
             *      0 byte  : If there is response, the passed value will be any number from the pointer which will
             *                result in false errors.
             */
            if((wRespLen != bPayloadLen) && (bPayloadLen != 0U) && (bPayloadLen != 1U))
            {
                PH_CHECK_SUCCESS_FCT(wStatus, phalVca_Int_ComputeErrorResponse_Extended(pDataParams, pResponse[wRespLen - 1U]));
            }
        }

        /* Copy RndR */
        (void) memcpy(&pCumRndRC[bRndRCLen], pResponse, wRespLen);
        bRndRCLen = bRndRCLen + (uint8_t) wRespLen;

        /* Copy RndC */
        (void) memcpy(&pCumRndRC[bRndRCLen], &pPrndC[bRndCLen], wRespLen);
        bRndRCLen = bRndRCLen + (uint8_t) wRespLen;
        bRndCLen = bRndCLen + (uint8_t) wRespLen;

        /* Get the bOption value for the checking the timing measurement ON/OFF */
        PH_CHECK_SUCCESS_FCT(wStatus, phalVca_GetConfig(pDataParams, PHAL_VCA_TIMING_MODE, &wValue));
        if(wValue & 0x01U)
        {
            /* Compute threshold time from PubRespTime. Threshold time = pubRespTime + 10% of pubRespTime */
            wThresholdTimeUpperLimit = pPubRespTime[0U];
            wThresholdTimeUpperLimit <<= 8U;
            wThresholdTimeUpperLimit |= pPubRespTime[1U];

            /* As per the ref arch V0.17, the threshold time should not be 20% beyond the Lower bound of PubResp Time. */
            wThresholdTimeLowerLimit = (wThresholdTimeUpperLimit * 80U) / 100U;

            /* Get the last command execution time */
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_GetConfig(PHAL_VCA_RESOLVE_HAL_DATAPARAMS(pDataParams), PHHAL_HW_CONFIG_TIMING_US, &wValue));

            /* If the response is not received within the threshold time, return internal error */
            if(wValue > wThresholdTimeUpperLimit || wValue < wThresholdTimeLowerLimit)
            {
                return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_AL_VCA);
            }
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_VCA);
}

phStatus_t phalVca_Sam_NonX_VerifyProximityCheckNew(void * pDataParams, uint8_t * pMac,
    uint8_t * pResponse, uint16_t * pRespLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[9];
    uint8_t     PH_MEMLOC_REM bCmdLen = 0;
    uint8_t *   PH_MEMLOC_REM pResponse_Tmp = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    /* Frame the command. */
    aCmdBuff[bCmdLen++] = PHAL_VCA_CMD_VPC;

    /* Append MAC to command buffer. */
    memcpy(&aCmdBuff[bCmdLen], pMac, PHAL_VCA_SAMAV3_TRUNCATED_MAC_SIZE);
    bCmdLen += PHAL_VCA_SAMAV3_TRUNCATED_MAC_SIZE;

    /* Exchange the data in ISO7816 format. */
    if(PHAL_VCA_RESOLVE_WRAPPED_MODE(pDataParams))
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phalVca_Sam_NonX_SendISOWrappedCmd(
            pDataParams,
            aCmdBuff,
            (uint8_t) (bCmdLen - 1U), /* Command Code excluded. */
            &pResponse_Tmp,
            &wRespLen));

        /* Adjusting the response length i.e. removing the status code. */
        if(wRespLen > 2U)
        {
            wRespLen -= 2U;
        }
        else
        {
            wRespLen -= 1U;
            pResponse_Tmp++;
        }
    }

    /* Exchange the command in Native format. */
    else
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
            PHAL_VCA_RESOLVE_PAL_MIFARE_DATAPARAMS(pDataParams),
            PH_EXCHANGE_DEFAULT,
            aCmdBuff,
            bCmdLen,
            &pResponse_Tmp,
            &wRespLen));

        /* Adjusting the response length i.e. removing the status code. */
        if(wRespLen > 1U)
        {
            wRespLen -= 1U;
            pResponse_Tmp++;
        }
    }

    /* Copy the response to the parameter. */
    (void) memcpy(pResponse, pResponse_Tmp, wRespLen);
    *pRespLen = wRespLen;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_VCA);
}


phStatus_t phalVca_Sam_NonX_SetConfig(void* pDataParams, uint16_t wConfig, uint16_t wValue)
{
    switch(wConfig)
    {
        case PHAL_VCA_ADDITIONAL_INFO:
            phalVca_Sam_NonX_SetAdditionalInfo(pDataParams, wValue);
            break;

        case PHAL_VCA_WRAPPED_MODE:
            phalVca_Sam_NonX_SetWrappedMode(pDataParams, (uint8_t) wValue);
            break;

        case PHAL_VCA_PC_EXTENDED_APDU:
            phalVca_Sam_NonX_SetExtendedAPDU(pDataParams, (uint8_t) wValue);
            break;

        case PHAL_VCA_TIMING_MODE:
            phalVca_Sam_NonX_SetTimingMode(pDataParams, (uint8_t) wValue);
            break;
        default:
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_AL_VCA);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_VCA);
}

phStatus_t phalVca_Sam_NonX_GetConfig(void* pDataParams, uint16_t wConfig, uint16_t* pValue)
{
    switch(wConfig)
    {
        case PHAL_VCA_ADDITIONAL_INFO:
            *pValue = phalVca_Sam_NonX_GetAdditionalInfo(pDataParams);
            break;

        case PHAL_VCA_WRAPPED_MODE:
            *pValue = phalVca_Sam_NonX_GetWrappedMode(pDataParams);
            break;

        case PHAL_VCA_PC_EXTENDED_APDU:
            *pValue = phalVca_Sam_NonX_GetExtendedAPDU(pDataParams);
            break;

        case PHAL_VCA_TIMING_MODE:
            *pValue = phalVca_Sam_NonX_GetTimingMode(pDataParams);
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_AL_VCA);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_VCA);
}

uint16_t phalVca_Sam_NonX_GetAdditionalInfo(void* pDataParams)
{
    uint16_t PH_MEMLOC_REM wAddInfo = 0;
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
        case PHAL_VCA_SAM_NONX_ID:
            wAddInfo = ((phalVca_Sam_NonX_DataParams_t*) pDataParams)->wAdditionalInfo;
            break;
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        default:
            wAddInfo = ((phalVca_SamAV3_NonX_DataParams_t*) pDataParams)->wAdditionalInfo;
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */
    }

    return wAddInfo;
}

uint8_t phalVca_Sam_NonX_GetWrappedMode(void* pDataParams)
{
    uint8_t PH_MEMLOC_REM bWrappedMode = 0;
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
        case PHAL_VCA_SAM_NONX_ID:
            bWrappedMode = ((phalVca_Sam_NonX_DataParams_t*) pDataParams)->bWrappedMode;
            break;
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        default:
            bWrappedMode = ((phalVca_SamAV3_NonX_DataParams_t*) pDataParams)->bWrappedMode;
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */
    }

    return bWrappedMode;
}

uint8_t phalVca_Sam_NonX_GetExtendedAPDU(void* pDataParams)
{
    uint8_t PH_MEMLOC_REM extendedApdu = 0;
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
        case PHAL_VCA_SAM_NONX_ID:
            extendedApdu = ((phalVca_Sam_NonX_DataParams_t*) pDataParams)->bExtendedLenApdu;
            break;
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        default:
            extendedApdu = ((phalVca_SamAV3_NonX_DataParams_t*) pDataParams)->bExtendedLenApdu;
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */
    }

    return extendedApdu;
}

uint8_t phalVca_Sam_NonX_GetTimingMode(void* pDataParams)
{
    uint8_t PH_MEMLOC_REM option = 0;
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
        case PHAL_VCA_SAM_NONX_ID:
            option = ((phalVca_Sam_NonX_DataParams_t*) pDataParams)->bOption;
            break;
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        default:
            option = ((phalVca_SamAV3_NonX_DataParams_t*) pDataParams)->bOption;
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */
    }

    return option;
}

phStatus_t phalVca_Sam_NonX_SendISOWrappedCmd(void * pDataParams, uint8_t * pCmdBuff, uint8_t bLc,
    uint8_t ** pResponse, uint16_t * pRespLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bApduLen = 4;  /* Initializing with 4 since Length of the Data(LC) starts from 4th element of pApdu[] */
    uint8_t    PH_MEMLOC_REM pApdu[8] = { 0x90 /* CLS */, 0x00 /* INS */, 0x00 /* P1 */, 0x00 /* P2 */, 0x00 , 0x00, 0x00 /* LC */, 0x00 /* Le */ };

    /* Check for permissible CmdBuff size */
    if(bLc > PHAL_VCA_MAXWRAPPEDAPDU_SIZE)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_AL_VCA);
    }

    pApdu[1U] = pCmdBuff[0U];  /* Proximity Check Command Code. */

    switch(pApdu[1U])
    {
        case PHAL_VCA_CMD_PPC:
            pApdu[4U] = 0x00U;  /* These bytes will be treated as Le */
            pApdu[5U] = 0x00U;  /* For extended length APDU support */

            /* Exchange the information to PICC. */
            PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
                PHAL_VCA_RESOLVE_PAL_MIFARE_DATAPARAMS(pDataParams),
                PH_EXCHANGE_DEFAULT,
                pApdu,
                PHAL_VCA_RESOLVE_EXTENDED_LENGTH_APDU(pDataParams) ? 7 : 5, /* 2 bytes Le should be passed in case of Extended Length APDU since LC field is not present */
                pResponse,
                pRespLen));
            break;

        case PHAL_VCA_CMD_PC:
        case PHAL_VCA_CMD_VPC:
            /* To Note: Extended APDU will be used,
             *  When user forces the 'length' to be sent as Extended length APDU. */
            if(!PHAL_VCA_RESOLVE_EXTENDED_LENGTH_APDU(pDataParams))
            {
                /* Encode 'Length' in Short APDU format */
                pApdu[bApduLen++] = (uint8_t) bLc; /* Set Data Length. */
            }
            else
            {
                /* Encode 'Length' in extended Length format */
                pApdu[bApduLen++] = 0x00U;
                pApdu[bApduLen++] = 0x00U;
                pApdu[bApduLen++] = (uint8_t) bLc; /* Set Data Length. */
            }

            /* Exchange the information to PICC. */
            PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
                PHAL_VCA_RESOLVE_PAL_MIFARE_DATAPARAMS(pDataParams),
                PH_EXCHANGE_BUFFER_FIRST,
                pApdu,
                bApduLen,
                pResponse,
                pRespLen));

            /* Check for LC value */
            if(bLc > 0U)
            {
                /* Transmit data as continued buffer */
                PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
                    PHAL_VCA_RESOLVE_PAL_MIFARE_DATAPARAMS(pDataParams),
                    PH_EXCHANGE_BUFFER_CONT,
                    &pCmdBuff[1U],
                    bLc,
                    pResponse,
                    pRespLen));
            }

            /* Resetting bApduLen for further use in case of Le */
            bApduLen = 0U;
            if(!PHAL_VCA_RESOLVE_EXTENDED_LENGTH_APDU(pDataParams))
            {
                /* Encode 'Length' in Short APDU format */
                pApdu[bApduLen++] = 0x00U; /* Set the expected data length as full. */
            }
            else
            {
                /* Encode 'Length' in extended Length format */
                pApdu[bApduLen++] = 0x00U;
                pApdu[bApduLen++] = 0x00U; /* Set the expected data length as full. */
            }
            /* Transmit Le as buffer Last */
            PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
                PHAL_VCA_RESOLVE_PAL_MIFARE_DATAPARAMS(pDataParams),
                PH_EXCHANGE_BUFFER_LAST,
                pApdu,
                bApduLen,
                pResponse,
                pRespLen));
            break;
        default:
            return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_VCA);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_VCA);
}

phStatus_t phalVca_Sam_NonX_ComputeErrorResponse_Extended(void * pDataParams, uint16_t wStatus)
{
    phStatus_t PH_MEMLOC_REM status = PH_ERR_SUCCESS;
    phStatus_t  PH_MEMLOC_REM statusTmp;
    switch(wStatus)
    {
        case PHAL_VCA_RESP_ACK_ISO4:
        case PHAL_VCA_ISO7816_SUCCESS:
        case PHAL_VCA_ISO7816_PC_SUCCESS:
            status = PH_ERR_SUCCESS;
            break;
        case PHAL_VCA_RESP_ERR_CMD_INVALID:
            status = PHAL_VCA_ERR_CMD_INVALID;
            break;
        case PHAL_VCA_RESP_ERR_FORMAT:
            status = PHAL_VCA_ERR_FORMAT;
            break;
        case PHAL_VCA_RESP_ERR_GEN:
            status = PHAL_VCA_ERR_GEN;
            break;
        case PHAL_VCA_RESP_ERR_CMD_OVERFLOW:
            status = PHAL_VCA_ERR_CMD_OVERFLOW;
            break;
        case PHAL_VCA_ISO7816_ERR_WRONG_LENGTH:
        case PHAL_VCA_ISO7816_ERR_WRONG_LE:
        case PHAL_VCA_ISO7816_ERR_FILE_NOT_FOUND:
        case PHAL_VCA_ISO7816_ERR_WRONG_PARAMS:
        case PHAL_VCA_ISO7816_ERR_WRONG_LC:
        case PHAL_VCA_ISO7816_ERR_NO_PRECISE_DIAGNOSTICS:
        case PHAL_VCA_ISO7816_ERR_EOF_REACHED:
        case PHAL_VCA_ISO7816_ERR_FILE_ACCESS:
        case PHAL_VCA_ISO7816_ERR_FILE_EMPTY:
        case PHAL_VCA_ISO7816_ERR_MEMORY_FAILURE:
        case PHAL_VCA_ISO7816_ERR_INCORRECT_PARAMS:
        case PHAL_VCA_ISO7816_ERR_WRONG_CLA:
        case PHAL_VCA_ISO7816_ERR_UNSUPPORTED_INS:
            status = PHAL_VCA_ERR_7816_GEN_ERROR;
            /* Set the error code to VC param structure*/
            PH_CHECK_SUCCESS_FCT(statusTmp, phalVca_SetConfig(pDataParams, PHAL_VCA_ADDITIONAL_INFO, wStatus));
            break;
        default:
            status = PH_ERR_PROTOCOL_ERROR;
            break;
    }
    return PH_ADD_COMPCODE(status, PH_COMP_AL_VCA);
}

phStatus_t phalVca_Sam_NonX_ResolveValidIndex(phalVca_SamAV2_DataParams_t * pDataParams, uint16_t wIidIndex,
    uint16_t * pValidIndex)
{
    uint8_t     PH_MEMLOC_REM bCurrentValidIndex = 0;
    uint8_t     PH_MEMLOC_REM bDummyValidIndex = 0;
    uint8_t     PH_MEMLOC_REM bDummyFoundIndex = 0;
    uint8_t     PH_MEMLOC_REM bIndex = 0;
    uint16_t    PH_MEMLOC_REM bFoundIndex = pDataParams->wCurrentCardTablePos;

    for(bIndex = 0; bIndex < pDataParams->wCurrentCardTablePos; ++bIndex)
    {
        if(pDataParams->pCardTable[bIndex].wIidIndex == 0xFFU)
        {
            if(bCurrentValidIndex == wIidIndex)
            {
                bDummyFoundIndex = bIndex;
            }
            else
            {
                bDummyFoundIndex = bIndex;
            }
            bDummyValidIndex++;
        }
        else
        {
            if(bCurrentValidIndex == wIidIndex)
            {
                bFoundIndex = bIndex;
            }
            else
            {
                bDummyFoundIndex = bIndex;
            }
            bCurrentValidIndex++;
        }
    }

    /* Check if the index is valid */
    if(bFoundIndex >= pDataParams->wCurrentCardTablePos)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_VCA);
    }

    *pValidIndex = bFoundIndex;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_VCA);
}

phStatus_t phalVca_Sam_NonX_SelectVC(void* pDataParams, uint8_t bOption, uint8_t bEncKeyNo, uint8_t bEncKeyVer,
    uint8_t bMacKeyNo, uint8_t bMacKeyVer, uint8_t* pData, uint8_t bDataLen, uint8_t* pDivInput, uint8_t bDivInputLen,
    uint8_t** ppResponse, uint16_t* pRespLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
        case PHAL_VCA_SAM_NONX_ID:
            wStatus = phhalHw_Sam_Cmd_SAM_SelectVC(
                PHAL_VCA_RESOLVE_HAL_DATAPARAMS(pDataParams),
                bOption,
                bEncKeyNo,
                bEncKeyVer,
                bMacKeyNo,
                bMacKeyVer,
                pData,
                bDataLen,
                pDivInput,
                bDivInputLen,
                ppResponse,
                pRespLen);
            break;
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        case PHAL_VCA_SAMAV3_NONX_ID:
            wStatus = phhalHw_SamAV3_Cmd_SAM_SelectVC(
                PHAL_VCA_RESOLVE_HAL_DATAPARAMS(pDataParams),
                bOption,
                bEncKeyNo,
                bEncKeyVer,
                bMacKeyNo,
                bMacKeyVer,
                pData,
                bDataLen,
                pDivInput,
                bDivInputLen,
                ppResponse,
                pRespLen);
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_VCA);
            break;
    }

    return wStatus;
}

void phalVca_Sam_NonX_SetAdditionalInfo(void* pDataParams, uint16_t wValue)
{
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
        case PHAL_VCA_SAM_NONX_ID:
            ((phalVca_Sam_NonX_DataParams_t*) pDataParams)->wAdditionalInfo = wValue;
            break;
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        default:
            ((phalVca_SamAV3_NonX_DataParams_t*) pDataParams)->wAdditionalInfo = wValue;
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */
    }
}

void phalVca_Sam_NonX_SetWrappedMode(void* pDataParams, uint16_t wValue)
{
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
        case PHAL_VCA_SAM_NONX_ID:
            ((phalVca_Sam_NonX_DataParams_t*) pDataParams)->wAdditionalInfo = wValue;
            break;
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        default:
            ((phalVca_SamAV3_NonX_DataParams_t*) pDataParams)->wAdditionalInfo = wValue;
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */
    }
}

void phalVca_Sam_NonX_SetExtendedAPDU(void* pDataParams, uint16_t wValue)
{
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
        case PHAL_VCA_SAM_NONX_ID:
            ((phalVca_Sam_NonX_DataParams_t*) pDataParams)->wAdditionalInfo = wValue;
            break;
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        default:
            ((phalVca_SamAV3_NonX_DataParams_t*) pDataParams)->wAdditionalInfo = wValue;
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */
    }
}

void phalVca_Sam_NonX_SetTimingMode(void* pDataParams, uint16_t wValue)
{
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
        case PHAL_VCA_SAM_NONX_ID:
            ((phalVca_Sam_NonX_DataParams_t*) pDataParams)->wAdditionalInfo = wValue;
            break;
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        default:
            ((phalVca_SamAV3_NonX_DataParams_t*) pDataParams)->wAdditionalInfo = wValue;
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */
    }
}

phStatus_t phalVca_Sam_NonX_ProximityCheck_Part1(void* pDataParams, uint8_t  bOption,
    uint8_t bKeyNo, uint8_t  bKeyVer, uint8_t* pPPCData, uint8_t  bPPCDataLen, uint8_t* pPCData,
    uint8_t bPCDataLen, uint8_t* pDivInput, uint8_t bDivInputLen, uint8_t** ppMac,
    uint16_t* pMacLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
        case PHAL_VCA_SAM_NONX_ID:
            wStatus = phhalHw_Sam_Cmd_SAM_ProximityCheck_Part1(
                PHAL_VCA_RESOLVE_HAL_DATAPARAMS(pDataParams),
                bOption,
                bKeyNo,
                bKeyVer,
                pPPCData,
                bPPCDataLen,
                pPCData,
                bPCDataLen,
                pDivInput,
                bDivInputLen,
                ppMac,
                pMacLen);
            break;
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        case PHAL_VCA_SAMAV3_NONX_ID:
            wStatus = phhalHw_SamAV3_Cmd_SAM_ProximityCheck_Part1(
                PHAL_VCA_RESOLVE_HAL_DATAPARAMS(pDataParams),
                bOption,
                bKeyNo,
                bKeyVer,
                pPPCData,
                bPPCDataLen,
                pPCData,
                bPCDataLen,
                pDivInput,
                bDivInputLen,
                ppMac,
                pMacLen);
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_VCA);
            break;
    }
    return wStatus;
}

phStatus_t phalVca_Sam_NonX_ProximityCheck_Part2(void* pDataParams, uint8_t* pData,
    uint8_t bDataLen, uint8_t* pPiccRetCode)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
        case PHAL_VCA_SAM_NONX_ID:
            wStatus = phhalHw_Sam_Cmd_SAM_ProximityCheck_Part2(
                PHAL_VCA_RESOLVE_HAL_DATAPARAMS(pDataParams),
                pData,
                bDataLen,
                pPiccRetCode);
            break;
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        case PHAL_VCA_SAMAV3_NONX_ID:
            wStatus = phhalHw_SamAV3_Cmd_SAM_ProximityCheck_Part2(
                PHAL_VCA_RESOLVE_HAL_DATAPARAMS(pDataParams),
                pData,
                bDataLen,
                pPiccRetCode);
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_VCA);
            break;
    }
    return wStatus;
}

phStatus_t phalVca_Sam_NonX_KillAuthentication(void* pDataParams, uint8_t bOption)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
        case PHAL_VCA_SAM_NONX_ID:
            wStatus = phhalHw_Sam_Cmd_SAM_KillAuthentication(
                PHAL_VCA_RESOLVE_HAL_DATAPARAMS(pDataParams),
                bOption);
            break;
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        case PHAL_VCA_SAMAV3_NONX_ID:
            wStatus = phhalHw_SamAV3_Cmd_SAM_KillAuthentication(
                PHAL_VCA_RESOLVE_HAL_DATAPARAMS(pDataParams),
                bOption);
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_VCA);
            break;
    }

    return wStatus;
}

phStatus_t phalVca_Sam_NonX_GetRandom(void* pDataParams, uint8_t bExpLen, uint8_t* pRnd)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    switch(PH_GET_COMPID(pDataParams))
    {
#ifdef NXPBUILD__PHAL_VCA_INTERNAL
        case PHAL_VCA_SAM_NONX_ID:
            wStatus = phhalHw_Sam_Cmd_SAM_GetRandom(
                PHAL_VCA_RESOLVE_HAL_DATAPARAMS(pDataParams),
                bExpLen,
                pRnd);
            break;
#endif /* NXPBUILD__PHAL_VCA_INTERNAL */

#ifdef NXPBUILD__PHHAL_HW_SAMAV3
        case PHAL_VCA_SAMAV3_NONX_ID:
            wStatus = phhalHw_SamAV3_Cmd_SAM_GetRandom(
                PHAL_VCA_RESOLVE_HAL_DATAPARAMS(pDataParams),
                bExpLen,
                pRnd);
            break;
#endif /* NXPBUILD__PHHAL_HW_SAMAV3 */

        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_VCA);
            break;
    }

    return wStatus;
}
#endif /* NXPBUILD__PHAL_VCA_SAMAV3_NONX */

#endif /* NXPBUILD__PHAL_VCA_SAM_NONX */
