/*
 * Copyright 2018, 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/**
 * \file
 * MIFARE PRIME NTAG application SamAV3 NonX component of Reader Library framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phhalHw.h>
#include <phCryptoSym.h>
#include <phCryptoRng.h>
#include <phKeyStore.h>
#include <ph_RefDefs.h>
#include <string.h>
#include <ph_TypeDefs.h>

#ifdef NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX
#include "../phalMfPrimeNtag_Int.h"
#include "phalMfPrimeNtag_Sam_NonX_Int.h"

phStatus_t phalMfPrimeNtag_Sam_NonX_Int_ValidateResponse(void * pDataParams, uint16_t wStatus, uint16_t wPiccRetCode)
{
	uint16_t	PH_MEMLOC_REM wPiccStat = 0;

	/* Evaluate the response. */
	if ((wStatus == PH_ERR_SUCCESS) ||
		((wStatus & PH_ERR_MASK) == PHHAL_HW_SAMAV3_ERR_DESFIRE_GEN))
	{
		/* Evaluate the PICC Status. */
		wPiccStat = (uint16_t) (((wPiccRetCode & 0xFF00) == 0x9100) ? (wPiccRetCode & 0x00FF) : wPiccRetCode);

		/* Validate the PICC Status. */
		PH_CHECK_SUCCESS_FCT(wStatus, phalMfprimeNtag_Int_ComputeErrorResponse(pDataParams, wPiccStat));
	}
	else
	{
		if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING)
		{
			wStatus = PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_AL_MFPRIMENTAG);
		}

		PH_CHECK_SUCCESS(wStatus);
	}

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}

phStatus_t phalMfPrimeNtag_Sam_NonX_Int_CardExchange(void * pDataParams, uint16_t bWrapCmd, uint16_t wBufferOption, uint16_t wTotDataLen, uint8_t bExchangeLE,
	uint8_t * pData, uint16_t wDataLen, uint8_t ** ppResponse, uint16_t * pRespLen, uint8_t * pPiccErrCode)
{
	phStatus_t		PH_MEMLOC_REM wStatus = 0;
	phStatus_t		PH_MEMLOC_REM wPICCStatus = 0;
	uint16_t		PH_MEMLOC_REM wLc = 0;
	uint16_t		PH_MEMLOC_REM wRespLen = 0;
	uint8_t*		PH_MEMLOC_REM pResponse = NULL;

	uint8_t			PH_MEMLOC_REM aLc[1] = {0x00};
	uint8_t			PH_MEMLOC_REM aLe[1] = {0x00};
	uint8_t			PH_MEMLOC_REM aISO7816Header[8] = {PHAL_MFPRIMENTAG_WRAPPEDAPDU_CLA, 0x00, PHAL_MFPRIMENTAG_WRAPPEDAPDU_P1, PHAL_MFPRIMENTAG_WRAPPEDAPDU_P2};
	uint8_t			PH_MEMLOC_REM bISO7816HeaderLen = 4;

	/* Exchange the command in Iso7816 wrapped formmat. ----------------------------------------------------------------- */
	if((wBufferOption == PH_EXCHANGE_BUFFER_FIRST) || (wBufferOption == PH_EXCHANGE_DEFAULT))
	{
		if(bWrapCmd)
		{
			/* Set the LC information. */
			wLc = (uint16_t) (wTotDataLen - 1 /* Excluding the command code. */);

			/* Update the command code to Iso7816 header */
			aISO7816Header[1] = pData[0];

			/* Add the ISO 7816 header to layer 4 buffer. */
			PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
				PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->pPalMifareDataParams,
				PH_EXCHANGE_BUFFER_FIRST,
				&aISO7816Header[0],
				bISO7816HeaderLen,
				NULL,
				NULL));

			/* Add Lc if available */
			if(wLc)
			{
				aLc[0] = (uint8_t) (wLc & 0x00FF);

				/* Add the Lc to layer 4 buffer. */
				PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
					PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->pPalMifareDataParams,
					PH_EXCHANGE_BUFFER_CONT,
					aLc,
					1,
					NULL,
					NULL));

				/* Add the data to layer 4 buffer. */
				PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
					PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->pPalMifareDataParams,
					PH_EXCHANGE_BUFFER_CONT,
					&pData[1],	/* Exclude the command code because it is added to INS. */
					(uint16_t) (wDataLen - 1),
					NULL,
					NULL));
			}
		}
	}

	if((wBufferOption == PH_EXCHANGE_BUFFER_CONT))
	{
		/* Add the data to layer 4 buffer. */
		PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
			PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->pPalMifareDataParams,
			PH_EXCHANGE_BUFFER_CONT,
			pData,
			wDataLen,
			NULL,
			NULL));
	}

    if((wBufferOption == PH_EXCHANGE_BUFFER_LAST) || (wBufferOption == PH_EXCHANGE_DEFAULT))
    {
        if((wBufferOption == PH_EXCHANGE_BUFFER_LAST) || !bWrapCmd)
        {
            /* Add the data to layer 4 buffer. */
            PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
                PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->pPalMifareDataParams,
                PH_EXCHANGE_BUFFER_CONT,
                pData,
                wDataLen,
                NULL,
                NULL));
        }

        /* Add Le to L4 buffer and exchange the command. */
        wStatus = phpalMifare_ExchangeL4(
            PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->pPalMifareDataParams,
            PH_EXCHANGE_BUFFER_LAST,
            aLe,
            (uint8_t) (bExchangeLE ? 1 : 0),
            &pResponse,
            &wRespLen);

        /* Create memory for updating the response of ISO 14443 format. */
        *ppResponse = pResponse;

        if(wRespLen >= 2U)
        {
            /* Combine Sw1 and Sw2 status codes. */
            wPICCStatus = (uint16_t) ((pResponse[wRespLen - 2] << 8) | pResponse[wRespLen - 1]);

            /* Evaluate the response. */
            wStatus = phalMfPrimeNtag_Sam_NonX_Int_ValidateResponse(pDataParams, wStatus, wPICCStatus);

            /* Update the response buffer length excluding SW1SW2. */
            *pRespLen = wRespLen - 2;

            /* Copy the second byte of response (SW2) to RxBuffer */
            *pPiccErrCode = pResponse[wRespLen - 1];
        }
        else
        {
            wStatus = phalMfPrimeNtag_Sam_NonX_Int_ValidateResponse(pDataParams, wStatus, 0x901E);
        }
	}

	return wStatus;
}

phStatus_t phalMfPrimeNtag_Sam_NonX_Int_AuthenticatePICC(void * pDataParams, uint8_t bAuthType, uint8_t bAuthMode, uint16_t wOption, uint16_t wKeyNo, uint16_t wKeyVer,
	uint8_t bKeyNoCard, uint8_t * pDivInput, uint8_t bDivInputLen, uint8_t * pPcdCapsIn, uint8_t bPcdCapsInLen, uint8_t * pPCDCap2, uint8_t * pPDCap2)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	phStatus_t	PH_MEMLOC_REM wStatus1 = 0;
	uint8_t     PH_MEMLOC_REM bP1 = 0;
	uint16_t	PH_MEMLOC_REM wCmdLen = 0;
	uint8_t		PH_MEMLOC_REM bIsPcdCapsInAvailable = PH_OFF;
	uint8_t *	PH_MEMLOC_REM pCardResponse = NULL;
	uint16_t	PH_MEMLOC_REM wCardRespLen = 0;
	uint8_t*	PH_MEMLOC_REM pSamResponse = NULL;
	uint16_t	PH_MEMLOC_REM wSamRespLen = 0;
	uint8_t		PH_MEMLOC_REM bPiccErrCode = 0;
	uint8_t		PH_MEMLOC_REM aCmdBuff[34];
	uint8_t		PH_MEMLOC_REM aAppId[3] = {0x00, 0x00, 0x00};

#ifdef RDR_LIB_PARAM_CHECK
	if(memcmp(PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->aAid, aAppId, 3) == 0x00)
	{
		/* At PICC level valid key numbes are 0 - 4. */
		if(bKeyNoCard > PHAL_MFPRIMENTAG_ORIGINALITY_KEY_LAST )
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
		}
	}
	else
	{
		/* At App level valid key numbers are 0 - 2. */
		if((bKeyNoCard > PHAL_MFPRIMENTAG_APP_KEY_LAST))
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
		}
	}

	/* Change for valid diversification options. */
	if ((wOption != PHAL_MFPRIMENTAG_NO_DIVERSIFICATION) &&
		(wOption != PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
	}

	/* Validate diversification input length. */
	if ((wOption != PHAL_MFPRIMENTAG_NO_DIVERSIFICATION) && (bDivInputLen > 31 ))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
	}
#endif

    /* Check for valid SAM keystore number and version. */
    if ((wKeyNo > 0x7f) || (wKeyVer > 0xff))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
    }

	/* Clear the command buffer and length. */
	wCmdLen = 0;
	memset(aCmdBuff, 0x00, sizeof(aCmdBuff));	/* PRQA S 3200 */

/* Frame the command buffer to be exchanged with PICC---------------------------------------------------------------------------------- */

	/* Add the Auth code to Command Buffer . */
	aCmdBuff[wCmdLen++] = bAuthType;
	aCmdBuff[wCmdLen++] = bKeyNoCard;

	/* Append PCD imput capabilitites in case of EV2 First auth. */
	if (bAuthType == PHAL_MFPRIMENTAG_CMD_AUTHENTICATE_EV2_FIRST)
	{
		/* Append LenCap Byte. */
		aCmdBuff[wCmdLen++] = bPcdCapsInLen;

		/* Set the PCD input availability flag. */
		bIsPcdCapsInAvailable = PH_ON;

		/* Reset the Authentication state. */
		PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_NonX_Int_ResetAuthStatus(pDataParams));
	}

	/* Exchange the command with the card. */
	wStatus = phalMfPrimeNtag_Sam_NonX_Int_CardExchange(
		pDataParams,
		PH_ON,
		PH_EXCHANGE_BUFFER_FIRST,
		(uint16_t) (wCmdLen + (bIsPcdCapsInAvailable ? bPcdCapsInLen : 0)),
		PH_OFF,
		aCmdBuff,
		wCmdLen,
		&pCardResponse,
		&wCardRespLen,
		&bPiccErrCode);

	/* Buffer PCD Input capabilities and exchange the bufferred information to card. */
	wStatus = phalMfPrimeNtag_Sam_NonX_Int_CardExchange(
		pDataParams,
		PH_ON,
		PH_EXCHANGE_BUFFER_LAST,
		0,
		PH_ON,
		pPcdCapsIn,
		(uint16_t) (bIsPcdCapsInAvailable ? bPcdCapsInLen : 0),
		&pCardResponse,
		&wCardRespLen,
		&bPiccErrCode);

	/* Validate the response for chaining. */
	if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING)
	{
		return wStatus;
	}

/* First part of Exchange with Sam hardware. ------------------------------------------------------------------------------------------- */

	/* Set Auth Type. */
	bP1 = (uint8_t) ((bAuthType == PHAL_MFPRIMENTAG_CMD_AUTHENTICATE_EV2_FIRST) ? PHHAL_HW_CMD_SAMAV3_AUTH_MODE_EV2_FIRST_AUTH :
						   PHHAL_HW_CMD_SAMAV3_AUTH_MODE_EV2_NON_FIRST_AUTH);

	/* Set Auth mode with diversification enabled. */
	bP1 |= (uint8_t) ((wOption != PHAL_MFPRIMENTAG_NO_DIVERSIFICATION) ? PHHAL_HW_CMD_SAMAV3_KEY_DIVERSIFICATION_ON :
		PHHAL_HW_CMD_SAMAV3_KEY_DIVERSIFICATION_OFF);

	/* Set Diversification flags.
	 * AV2 compatibility mode key diversification methods, 3TDEA, AES key
	 */
    if (wOption == PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS)
    {
        bP1 |= (uint8_t) PHHAL_HW_CMD_SAMAV3_KDF_AV2;
	}

	/* Set the secure messaging. */
	if((bKeyNoCard >= PHAL_MFPRIMENTAG_ORIGINALITY_KEY_FIRST) && (bKeyNoCard <= PHAL_MFPRIMENTAG_ORIGINALITY_KEY_LAST) &&
		(memcmp(PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->aAid, aAppId, 3) == 0x00))
	{
		bP1 = (uint8_t) (bP1 | PHHAL_HW_CMD_SAMAV3_SUPPRESS_SECURE_MESSAGING);
	}

	wStatus1 = phhalHw_SamAV3_Cmd_SAM_AuthenticatePICC_Part1(
		PHAL_MFPRIMENTAG_RESOLVE_HAL_DATAPARAMS(pDataParams),
		bP1,
		(uint8_t) wKeyNo,
		(uint8_t) wKeyVer,
		bAuthMode,
		pDivInput,
		bDivInputLen,
		pCardResponse,
		(uint8_t) wCardRespLen,
		&pSamResponse,
		&wSamRespLen);

	/* Check for the Chaining active */
	if ((wStatus1 & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING)
	{
		return wStatus1;
	}




/* Second part of Exchange with card. -------------------------------------------------------------------------------------------------- */
	wCmdLen = 0;
	bPiccErrCode = 0;
	memset(aCmdBuff, 0x00, sizeof(aCmdBuff));	/* PRQA S 3200 */

	/* Frame the command for Exchange to card. */
	aCmdBuff[wCmdLen++] = 0xAF;

	/* Copy the response received from SAM to Command buffer. */
	memcpy(&aCmdBuff[wCmdLen], pSamResponse, wSamRespLen);	/* PRQA S 3200 */
	wCmdLen += wSamRespLen;

	/* Exchange the command with the card. */
	wStatus = phalMfPrimeNtag_Sam_NonX_Int_CardExchange(
		pDataParams,
		PH_ON,
		PH_EXCHANGE_DEFAULT,
		wCmdLen,
		PH_ON,
		aCmdBuff,
		wCmdLen,
		&pCardResponse,
		&wCardRespLen,
		&bPiccErrCode);

	/* Second part of Exchange with Sam hardware. ----------------------------------------------------------------- */
	wStatus = phhalHw_SamAV3_Cmd_SAM_AuthenticatePICC_Part2(
		PHAL_MFPRIMENTAG_RESOLVE_HAL_DATAPARAMS(pDataParams),
		bPiccErrCode,
		pCardResponse,
		(uint8_t) wCardRespLen,
		pPDCap2,
		pPCDCap2,
		&bPiccErrCode);

	/* Validate the PICC error. */
	wStatus = phalMfPrimeNtag_Sam_NonX_Int_ValidateResponse(pDataParams, wStatus, bPiccErrCode);

	if(wStatus == PH_ERR_SUCCESS)
	{
		/* Do not update the auth state if originality keys are used. */
		if((bKeyNoCard >= PHAL_MFPRIMENTAG_ORIGINALITY_KEY_FIRST) && (bKeyNoCard <= PHAL_MFPRIMENTAG_ORIGINALITY_KEY_LAST) &&
			(memcmp(PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->aAid, aAppId, 3) == 0x00))
		{
			PH_CHECK_SUCCESS_FCT(wStatus1, phalMfPrimeNtag_Sam_NonX_Int_ResetAuthStatus(pDataParams));
		}
		else
		{
			PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode = PHAL_MFPRIMENTAG_AUTHENTICATEEV2;
			PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->bKeyNo = bKeyNoCard;
		}
	}

	return wStatus;
}

phStatus_t phalMfPrimeNtag_Sam_NonX_Int_ChangeKeyPICC(void * pDataParams, uint16_t wOption, uint8_t bKeyNoCard, uint16_t wCurrKeyNo, uint16_t wCurrKeyVer, uint16_t wNewKeyNo,
	uint16_t wNewKeyVer, uint8_t * pDivInput, uint8_t bDivInputLen)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	phStatus_t	PH_MEMLOC_REM wStatus1 = 0;
    uint8_t     PH_MEMLOC_REM bKeyCompMeth = 0;
    uint8_t     PH_MEMLOC_REM bCfg = 0;
	uint8_t		PH_MEMLOC_REM aCmdBuff[43];
	uint16_t	PH_MEMLOC_REM wCmdLen = 0;
	uint8_t *	PH_MEMLOC_REM pCardResponse = NULL;
	uint16_t	PH_MEMLOC_REM wCardRespLen = 0;
	uint16_t	PH_MEMLOC_REM wSamRespLen = 0;
	uint8_t *	PH_MEMLOC_REM pSamResponse = NULL;
	uint8_t		PH_MEMLOC_REM bPiccErrCode = 0;

#ifdef RDR_LIB_PARAM_CHECK
	uint8_t		PH_MEMLOC_REM aAppId[3] = {0x00, 0x00, 0x00};

	/* Only if seleted Aid is 0x000000. */
	if (memcmp(PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->aAid, aAppId, 3) == 0x00)
    {
		/* At PICC level ChangeKey cannot be performed. */
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
    }
    else
    {
		/* At App level, 0,1,2 are valid ones. */
		if(bKeyNoCard > PHAL_MFPRIMENTAG_APP_KEY_LAST)
		{
			/* Invalid application key specified */
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
		}
    }

	if (PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFPRIMENTAG_NOT_AUTHENTICATED)
	{
		return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_AL_MFPRIMENTAG);
	}
#endif

    /* Check for valid SAM key number and version. */
    if ((wCurrKeyNo > 0x7f) || (wCurrKeyVer > 0xff) || (wNewKeyNo > 0x7f) || (wNewKeyVer > 0xff))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
    }


/* Command Exchange with SAM. ---------------------------------------------------------------------------------------------------------- */
    /* Set the key compilation method. */
	if(wOption == PHAL_MFPRIMENTAG_NO_DIVERSIFICATION)
	{
		bKeyCompMeth = 0x00;
	}
	else
	{
#ifdef RDR_LIB_PARAM_CHECK
		/* Change for valid diversification options. */
		if ((wOption & PHHAL_HW_CMD_SAMAV3_DIV_METHOD_AV2) != PHHAL_HW_CMD_SAMAV3_DIV_METHOD_AV2)
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
		}

		/* Validate diversification input length. */
		if ( bDivInputLen > 31 )
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPRIMENTAG);
		}
#endif

		/* Assin the option to local variable. */
		bKeyCompMeth = (uint8_t) wOption;
	}

    /* Set if PICC targeted key equal to PICC authenticated key. */
    if ((PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->bKeyNo) == bKeyNoCard)
    {
        bKeyCompMeth = (uint8_t) (bKeyCompMeth | PHHAL_HW_CMD_SAMAV3_CRYPTO_MODE_SAME_KEY);
    }

	PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_ChangeKeyPICC(
		PHAL_MFPRIMENTAG_RESOLVE_HAL_DATAPARAMS(pDataParams),
		bKeyCompMeth,
		bCfg,
		0x00,
		(uint8_t) bKeyNoCard,
		(uint8_t) wCurrKeyNo,
		(uint8_t) wCurrKeyVer,
		(uint8_t) wNewKeyNo,
		(uint8_t) wNewKeyVer,
		pDivInput,
		bDivInputLen,
		&pSamResponse,
		&wSamRespLen));

	/* Command Exchange with Card. -------------------------------------------------------------------------------- */
	wCmdLen = 0;
	memset(aCmdBuff, 0x00, sizeof(aCmdBuff));	/* PRQA S 3200 */

	/* Frame the command information with command type. */
	aCmdBuff[wCmdLen++] = PHAL_MFPRIMENTAG_CMD_CHANGE_KEY;

	/* Add CardKey number to command buffer. */
	aCmdBuff[wCmdLen++] = bKeyNoCard;

	/* Copy the response received from SAM to Command buffer. */
	memcpy(&aCmdBuff[wCmdLen], pSamResponse, wSamRespLen);	/* PRQA S 3200 */
	wCmdLen += wSamRespLen;

	/* Exchange the command with the card. */
	wStatus = phalMfPrimeNtag_Sam_NonX_Int_CardExchange(
		pDataParams,
		PH_ON,
		PH_EXCHANGE_DEFAULT,
		wCmdLen,
		PH_ON,
		aCmdBuff,
		wCmdLen,
		&pCardResponse,
		&wCardRespLen,
		&bPiccErrCode);

    /* Evaluate the response. */
	wStatus = phalMfPrimeNtag_Sam_NonX_Int_ValidateResponse(pDataParams, wStatus, bPiccErrCode);

	/* Reset the Auth state. */
	if(wStatus != PH_ERR_SUCCESS)
	{
		/* Kill the PICC Authentication. */
		PH_CHECK_SUCCESS_FCT(wStatus1, phhalHw_SamAV3_Cmd_SAM_KillAuthentication(
			PHAL_MFPRIMENTAG_RESOLVE_HAL_DATAPARAMS(pDataParams),
			PHHAL_HW_SAMAV3_CMD_SAM_KILL_AUTHENTICATION_PARTIAL));

		return wStatus;
	}
	else
	{
		/* Verify the MAC. */
		if(wCardRespLen)
		{
			PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_NonX_Int_VerifySM(
				pDataParams,
				(uint16_t) (PH_EXCHANGE_DEFAULT | PHALMFPRIMENTAG_SAM_NONX_EXCHANGE_PICC_STATUS),
				PHAL_MFPRIMENTAG_COMMUNICATION_MACD,
				NULL,
				0,
				bPiccErrCode,
				pCardResponse,
				wCardRespLen,
				&pSamResponse,
				&wSamRespLen));
		}

		/* Reset authentication status only if the key authenticated with is changed. */
		if ((PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->bKeyNo) == bKeyNoCard)
		{
			PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_NonX_Int_ResetAuthStatus(pDataParams));
		}
	}

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}

phStatus_t phalMfPrimeNtag_Sam_NonX_Int_GenerateSM(void * pDataParams, uint8_t bCommMode, uint8_t * pCmdBuff, uint16_t wCmdLen, uint8_t * pData,
	uint16_t wDataLen, uint8_t ** ppOutBuffer, uint16_t * pOutBufLen)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM bOffset = 0;
	uint16_t	PH_MEMLOC_REM wBuffOption = 0;

	/* Exchange the information to Sam hardware to get the MAC information.
	 * This computed MAC might not be exchanged. This is computed to initial crypto informationin SAM which will be used for MAC verification.
	 */
	if(PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode != PHAL_MFPRIMENTAG_NOT_AUTHENTICATED)
	{
		if(bCommMode == PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN)
		{
			/* Buffer command counter information to SAM. */
			PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_ApplySM(
				PHAL_MFPRIMENTAG_RESOLVE_HAL_DATAPARAMS(pDataParams),
				PH_EXCHANGE_DEFAULT,
				PHHAL_HW_SAMAV3_CMD_APPLY_SM_COMM_MODE_PLAIN,
				0x00,
				0x01,
				NULL,
				0,
				ppOutBuffer,
				pOutBufLen));
		}
		else
		{
			/* Set the bufferring option. */
			wBuffOption = PH_EXCHANGE_BUFFER_FIRST;

			/* Compute the Offset to be exchagned. */
			if(bCommMode == PHAL_MFPRIMENTAG_COMMUNICATION_ENC)
			{
				bOffset = (uint8_t) wCmdLen;
				wBuffOption |= PHHAL_HW_SAMAV3_CMD_APPLY_SM_INCLUDE_OFFSET;
			}

			/* Buffer command information to SAM. */
			PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_ApplySM(
				PHAL_MFPRIMENTAG_RESOLVE_HAL_DATAPARAMS(pDataParams),
				wBuffOption,
				bCommMode,
				bOffset,
				0x00,
				pCmdBuff,
				(uint8_t) ((pCmdBuff[0] != 0xAF) ? wCmdLen : 0),
				ppOutBuffer,
				pOutBufLen));

			/* Buffer data information to SAM and exchange the infomration to received the SM information. */
			PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_ApplySM(
				PHAL_MFPRIMENTAG_RESOLVE_HAL_DATAPARAMS(pDataParams),
				PH_EXCHANGE_BUFFER_LAST,
				bCommMode,
				0,
				0,
				pData,
				(uint8_t) wDataLen,
				ppOutBuffer,
				pOutBufLen));
		}
	}

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}

phStatus_t phalMfPrimeNtag_Sam_NonX_Int_VerifySM(void * pDataParams, uint16_t wOption, uint8_t bCommMode, uint8_t * pResponse, uint16_t wRespLen, uint8_t bPiccStat,
	uint8_t * pRespMac, uint16_t wRespMacLen, uint8_t ** ppOutBuffer, uint16_t * pOutBufLen)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint16_t	PH_MEMLOC_REM wBuffOption = 0;

	/* Exchange the information to Sam hardware to get the MAC information. */
	if((PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode != PHAL_MFPRIMENTAG_NOT_AUTHENTICATED) &&
		(bCommMode != PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN))
	{
		/* Set the buffering flag to Default. */
		wBuffOption = PH_EXCHANGE_BUFFER_FIRST;
		wBuffOption |= (uint16_t) (((wOption & 0xFF0F) == PH_EXCHANGE_DEFAULT) ? wBuffOption : PH_EXCHANGE_TXCHAINING);

		/* Set PICC Status to be exchanged for the first frame. */
		if(((wOption & PH_EXCHANGE_CUSTOM_BITS_MASK) == PHALMFPRIMENTAG_SAM_NONX_EXCHANGE_PICC_STATUS) &&
			(bPiccStat == PHAL_MFPRIMENTAG_RESP_ADDITIONAL_FRAME))
		{
			bPiccStat = PHAL_MFPRIMENTAG_RESP_OPERATION_OK;
		}

		/* Buffer the PICC status information to Sam buffer. */
		PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_RemoveSM(
			PHAL_MFPRIMENTAG_RESOLVE_HAL_DATAPARAMS(pDataParams),
			wBuffOption,
			bCommMode,
			&bPiccStat,
			(uint8_t) (((wOption & PH_EXCHANGE_CUSTOM_BITS_MASK) == PHALMFPRIMENTAG_SAM_NONX_EXCHANGE_PICC_STATUS)  ? 1 : 0),
			NULL,
			NULL));

		/* Buffer the Plain response information to Sam buffer. */
		PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_RemoveSM(
			PHAL_MFPRIMENTAG_RESOLVE_HAL_DATAPARAMS(pDataParams),
			PH_EXCHANGE_BUFFER_CONT,
			bCommMode,
			pResponse,
			(uint8_t) wRespLen,
			NULL,
			NULL));

		/* Buffer Mac and Exchagne the bufferred information to Sam hardware. */
		PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_RemoveSM(
			PHAL_MFPRIMENTAG_RESOLVE_HAL_DATAPARAMS(pDataParams),
			PH_EXCHANGE_BUFFER_LAST,
			0x00,
			pRespMac,
			(uint8_t) wRespMacLen,
			ppOutBuffer,
			pOutBufLen));
	}

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}

phStatus_t phalMfPrimeNtag_Sam_NonX_Int_ReadData(void * pDataParams, uint8_t bCmd_ComMode, uint8_t bResp_ComMode, uint8_t * pCmdBuff, uint16_t wCmdLen,
	uint8_t ** ppResponse, uint16_t * pRespLen)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	phStatus_t	PH_MEMLOC_REM wStatus1 = 0;
	uint16_t	PH_MEMLOC_REM wBuffOption = 0;
	uint16_t	PH_MEMLOC_REM wBuffOption1 = 0;
	uint8_t		PH_MEMLOC_REM bFirstFrame = 0;
	uint8_t		PH_MEMLOC_REM bLastFrame = 0;
	uint8_t		PH_MEMLOC_REM bFinished = 0;
	uint8_t	*	PH_MEMLOC_REM pMac = NULL;
	uint16_t	PH_MEMLOC_REM wMacLen = 0;
	uint16_t	PH_MEMLOC_REM wOffset = 0;
	uint16_t	PH_MEMLOC_REM wTotLen = 0;
	uint8_t	*	PH_MEMLOC_REM pCardResponse = NULL;
	uint8_t	*	PH_MEMLOC_REM pResponse = NULL;
	uint16_t	PH_MEMLOC_REM wCardRespLen = 0;
	uint16_t	PH_MEMLOC_REM wRespLen = 0;
	uint8_t		PH_MEMLOC_REM bPiccErrCode = 0;

	/* Secure the information to be exchanged. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_NonX_Int_GenerateSM(
		pDataParams,
		bCmd_ComMode,
		pCmdBuff,
		wCmdLen,
		NULL,
		0,
		&pMac,
		&wMacLen));

	/* Frame the total length. */
	wTotLen = (uint16_t) (wCmdLen + wMacLen);

	/* Set exchange optionto First. */
	wBuffOption = PH_EXCHANGE_BUFFER_FIRST;

	do
	{
		/* Buffer the command information. */
		wStatus1 = phalMfPrimeNtag_Sam_NonX_Int_CardExchange(
			pDataParams,
			PH_ON,
			wBuffOption,
			wTotLen,
			PH_ON,
			pCmdBuff,
			wCmdLen,
			&pCardResponse,
			&wCardRespLen,
			&bPiccErrCode);

		/* Buffer the Mac information and exchange the complete information to PICC. */
		if(wBuffOption != PH_EXCHANGE_DEFAULT)
		{
			wStatus1 = phalMfPrimeNtag_Sam_NonX_Int_CardExchange(
				pDataParams,
				PH_ON,
				PH_EXCHANGE_BUFFER_LAST,
				0,
				PH_ON,
				pMac,
				wMacLen,
				&pCardResponse,
				&wCardRespLen,
				&bPiccErrCode);

			/* Set First Frame. */
			bFirstFrame = PH_ON;

			/* Subtract the total length with MAC. */
			wTotLen -= wMacLen;
		}

		/* Evaluate the response. */
		wStatus1 = phalMfPrimeNtag_Sam_NonX_Int_ValidateResponse(pDataParams, wStatus1, bPiccErrCode);

		/* Set the last frame to end the looping. */
		bLastFrame = (uint8_t) ((wStatus1 == PH_ERR_SUCCESS) ? PH_ON : PH_OFF);

		/* Reset the command length. */
		wCmdLen = 0;

		/* Update command information. */
		if(!bLastFrame)
		{
			pCmdBuff[0] = PHAL_MFPRIMENTAG_RESP_ADDITIONAL_FRAME;
			wCmdLen = 1;

			/* Update the Total length. */
			wTotLen = 1;
		}

		/* Set the exchange option to RxChaining if there is still more information to be exchanged. */
		wBuffOption = PH_EXCHANGE_DEFAULT;

		/* Update the variables and parameters. */
		if(ppResponse[0] != NULL)
		{
			memcpy(&ppResponse[0][wOffset], pCardResponse, (bLastFrame ? (wCardRespLen - wMacLen ) : wCardRespLen));	/* PRQA S 3200 */
		}
		else
		{
			ppResponse[0] = pCardResponse;
		}

		*pRespLen += wCardRespLen;
		wOffset += (uint16_t) wCardRespLen;

		/* Reset the Auth state of PICC only. */
		if((wStatus1 != PH_ERR_SUCCESS) && ((wStatus1 & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
		{
			PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_NonX_Int_ResetAuthStatus(pDataParams));
			bFinished = PH_ON;
		}
		else
		{
			/* Set the buffering options. */
			wBuffOption1 = (uint16_t) (bLastFrame ? PH_EXCHANGE_DEFAULT : PH_EXCHANGE_RXCHAINING);

			/* Set the PICC status utilization. */
			if(bFirstFrame)
			{
				wBuffOption1 |= (uint16_t) PHALMFPRIMENTAG_SAM_NONX_EXCHANGE_PICC_STATUS;
			}

			/* Set the Mac Length. */
			if(bResp_ComMode != PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN)
			{
				wMacLen = (uint16_t) (((PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFPRIMENTAG_NOT_AUTHENTICATED) ||
					!bLastFrame) ? 0 : 8);
			}

			/* Verify the security of the received information. */
		 	wStatus = phalMfPrimeNtag_Sam_NonX_Int_VerifySM(
				pDataParams,
				wBuffOption1,
				bResp_ComMode,
				pCardResponse,
				(uint16_t) (wCardRespLen - (bLastFrame ? wMacLen : 0)),
				bPiccErrCode,
				&pCardResponse[wCardRespLen - (bLastFrame ? wMacLen : 0)],
				(uint16_t) (bLastFrame ? wMacLen : 0),
				&pResponse,
				&wRespLen);

			/* Copy the response to the buffer. */
			if(bResp_ComMode == PHAL_MFPRIMENTAG_COMMUNICATION_ENC)
			{
				memcpy(ppResponse[0], pResponse, wRespLen);
				*pRespLen = wRespLen;
			}

			/* Subtract if Mac is available. */
			*pRespLen -= (uint16_t) ((bLastFrame && (bResp_ComMode != PHAL_MFPRIMENTAG_COMMUNICATION_ENC)) ? wMacLen : 0);

			/* Validate the status. */
			if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING)
			{
				PH_CHECK_SUCCESS(wStatus);
			}

			/* Clear First Frame. */
			bFirstFrame = PH_OFF;
		}

		/* Set finished flag. */
		if(wStatus1 == PH_ERR_SUCCESS)
		{
			bFinished = PH_ON;
		}
	}while(!bFinished);

	return wStatus1;
}

phStatus_t phalMfPrimeNtag_Sam_NonX_Int_WriteData(void * pDataParams, uint8_t bIsDataCmd, uint8_t bCmd_ComMode, uint8_t bResp_ComMode,
	uint8_t * pCmdBuff, uint16_t wCmdLen, uint8_t * pData, uint32_t dwDataLen, uint8_t ** ppResponse, uint16_t * pRespLen)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	phStatus_t	PH_MEMLOC_REM wStatus1 = 0;
	uint16_t	PH_MEMLOC_REM wTotLen = 0;
	uint8_t		PH_MEMLOC_REM aCmdBuf_Tmp[50];
	uint16_t	PH_MEMLOC_REM wCmdLen_Tmp = 0;
	uint8_t		PH_MEMLOC_REM bExchangeMac = 0;
	uint8_t	*	PH_MEMLOC_REM pSMData = NULL;
	uint16_t	PH_MEMLOC_REM wSMDataLen = 0;
	uint8_t	*	PH_MEMLOC_REM pCardResponse = NULL;
	uint16_t	PH_MEMLOC_REM wCardRespLen = 0;
	uint8_t		PH_MEMLOC_REM bPiccErrCode = 0;
	uint8_t		PH_MEMLOC_REM bDataLen = 0;
	uint8_t		PH_MEMLOC_REM bIns = 0;
	uint32_t	PH_MEMLOC_REM dwRemLen = 0;

#ifdef RDR_LIB_PARAM_CHECK
	if ((bCmd_ComMode == PHAL_MFPRIMENTAG_COMMUNICATION_ENC) && (PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode == PHAL_MFPRIMENTAG_NOT_AUTHENTICATED))
	{
		return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_AL_MFPRIMENTAG);
	}
#endif

	/* Save the command information and it length because in course . */
	memcpy(aCmdBuf_Tmp, pCmdBuff, wCmdLen);
	wCmdLen_Tmp = wCmdLen;

	/* Set INS flag if this function is called for Write command.*/
	bIns = (uint8_t) ((pCmdBuff[0] == PHAL_MFPRIMENTAG_CMD_WRITE_DATA_ISO) ? PH_ON : PH_OFF);

	/* Encrypt the information. */
	if(PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode != PHAL_MFPRIMENTAG_NOT_AUTHENTICATED)
	{
		wStatus = phalMfPrimeNtag_Sam_NonX_Int_GenerateSM(
			pDataParams,
			bCmd_ComMode,
			aCmdBuf_Tmp,
			wCmdLen_Tmp,
			pData,
			(uint8_t) dwDataLen,
			&pSMData,
			&wSMDataLen);

		/* Set if Mac on command is required. */
		bExchangeMac = (uint8_t) ((bCmd_ComMode == PHAL_MFPRIMENTAG_COMMUNICATION_MACD) ? PHALMFPRIMENTAG_SAM_NONX_MAC_ON_CMD :
			PHALMFPRIMENTAG_SAM_NONX_NO_MAC_ON_CMD);
	}
	else
	{
		/* Do Nothing */
	}

	if(bIsDataCmd)
	{
		/* Set the lengths. */
		dwRemLen = (bCmd_ComMode == PHAL_MFPRIMENTAG_COMMUNICATION_ENC) ? wSMDataLen : dwDataLen;

		/* Performing chunk exchange if large data flag is set. */
		bDataLen = (uint8_t) dwRemLen;
	}
	else
	{
		bDataLen = (uint8_t) dwRemLen;

		if(bCmd_ComMode == PHAL_MFPRIMENTAG_COMMUNICATION_ENC)
		{
			bDataLen = (uint8_t) wSMDataLen;
		}
	}

	/* Frame the total length. */
	wTotLen = 0;
	wTotLen = (uint16_t) (wCmdLen_Tmp + ((bCmd_ComMode == PHAL_MFPRIMENTAG_COMMUNICATION_ENC) ? bDataLen : dwDataLen));
	wTotLen = (uint16_t) (bExchangeMac ? ( wTotLen + wSMDataLen) : wTotLen);

	/* Update the total length. This will be used to update the actual LC information of ISO7816 format. */
	if(bIns)
	{
		if(bCmd_ComMode == PHAL_MFPRIMENTAG_COMMUNICATION_MACD)
		{
			wTotLen = (uint16_t) (wCmdLen_Tmp + dwRemLen);
			wTotLen = (uint16_t) (wTotLen + 8);
		}
	}

	/* Buffer the command information. */
	PH_CHECK_SUCCESS_FCT(wStatus1, phalMfPrimeNtag_Sam_NonX_Int_CardExchange(
		pDataParams,
		PH_ON,
		PH_EXCHANGE_BUFFER_FIRST,
		wTotLen,
		PH_OFF,
		aCmdBuf_Tmp,
		wCmdLen_Tmp,
		NULL,
		NULL,
		NULL));

	/* Buffer the data information. */
	if(bCmd_ComMode != PHAL_MFPRIMENTAG_COMMUNICATION_ENC)
	{
		PH_CHECK_SUCCESS_FCT(wStatus1, phalMfPrimeNtag_Sam_NonX_Int_CardExchange(
			pDataParams,
			PH_ON,
			PH_EXCHANGE_BUFFER_CONT,
			0,
			PH_OFF,
			pData,
			(uint16_t) dwRemLen,
			NULL,
			NULL,
			NULL));
	}

	/* Buffer the Mac information exchange the complete information to PICC. */
	PH_CHECK_SUCCESS_FCT(wStatus1, phalMfPrimeNtag_Sam_NonX_Int_CardExchange(
		pDataParams,
		PH_ON,
		PH_EXCHANGE_BUFFER_LAST,
		0,
		PH_ON,
		pSMData,
		(uint16_t) ((bCmd_ComMode == PHAL_MFPRIMENTAG_COMMUNICATION_ENC) ? bDataLen : (bExchangeMac ? wSMDataLen : 0)),
		&pCardResponse,
		&wCardRespLen,
		&bPiccErrCode));

	/* Verify the security of the received information. */
	if(bResp_ComMode != PHAL_MFPRIMENTAG_COMMUNICATION_PLAIN)
	{
		PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_NonX_Int_VerifySM(
			pDataParams,
			(uint16_t) (PH_EXCHANGE_DEFAULT | PHALMFPRIMENTAG_SAM_NONX_EXCHANGE_PICC_STATUS),
			bResp_ComMode,
			NULL,
			0,
			bPiccErrCode,
			pCardResponse,
			wCardRespLen,
			ppResponse,
			pRespLen));
	}

	/* Evaluate the response. */
	PH_CHECK_SUCCESS_FCT(wStatus, phalMfPrimeNtag_Sam_NonX_Int_ValidateResponse(pDataParams, wStatus1, bPiccErrCode));

	return wStatus1;
}

phStatus_t phalMfPrimeNtag_Sam_NonX_Int_ResetAuthStatus(void * pDataParams)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;

	/* Reset the Authmode and Key number */
	PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->bAuthMode	= 0xFF;
    PHAL_MFPRIMENTAG_RESOLVE_DATAPARAMS(pDataParams)->bKeyNo	= 0xFF;

	/* Kill the PICC Authentication. */
	PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_KillAuthentication(
		PHAL_MFPRIMENTAG_RESOLVE_HAL_DATAPARAMS(pDataParams),
		PHHAL_HW_SAMAV3_CMD_SAM_KILL_AUTHENTICATION_PARTIAL));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}

phStatus_t phalMfPrimeNtag_Sam_NonX_Int_ComputeSDMSessionVectors(void * pDataParams, uint8_t bSdmOption, uint16_t wSrcKeyNo, uint16_t wSrcKeyVer,
	uint16_t wDstKeyNo, uint8_t * pUid, uint8_t bUidLen, uint8_t * pSDMReadCtr)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM aSV[16];
	uint8_t		PH_MEMLOC_REM bSvLen = 0;
	uint32_t	PH_MEMLOC_REM dwSDMReadCtr = 0;

	/* Validate the Counter value. */
	if(pSDMReadCtr != NULL)
	{
		dwSDMReadCtr = (uint32_t) (pSDMReadCtr[0] | (pSDMReadCtr[1] << 8) | (pSDMReadCtr[2] << 16) | (pSDMReadCtr[3] << 24));
		if (dwSDMReadCtr == 0xFFFFFF)
		{
			return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_AL_MFPRIMENTAG);
		}
	}

	/* Clear the session vector SV. */
	memset(aSV, 0, 16);		/* PRQA S 3200 */

	/* Frame the default values in session vector. */
	aSV[bSvLen++] = 0x3C;
	aSV[bSvLen++] = 0xC3;
	aSV[bSvLen++] = 0x00;
	aSV[bSvLen++] = 0x01;
	aSV[bSvLen++] = 0x00;
	aSV[bSvLen++] = 0x80;

	/* Append the UID */
	if(bSdmOption & PHAL_MFPRIMENTAG_VCUID_PRESENT)
	{
		memcpy(&aSV[bSvLen], pUid, bUidLen);	/* PRQA S 3200 */
		bSvLen += bUidLen;
	}

	/* Append the SDM ReadCtr information. */
	if(bSdmOption & PHAL_MFPRIMENTAG_RDCTR_PRESENT)
	{
		aSV[bSvLen++] = (uint8_t) (dwSDMReadCtr & 0xFF);
		aSV[bSvLen++] = (uint8_t) ((dwSDMReadCtr & 0xFF00) >> 8);
		aSV[bSvLen++] = (uint8_t) ((dwSDMReadCtr & 0xFF0000) >> 16);
	}

	/* Update the SV length to 16. */
	bSvLen = 16;

	/* Exchange the session vector information to SAM. */
	PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_DeriveKey(
		PHAL_MFPRIMENTAG_RESOLVE_HAL_DATAPARAMS(pDataParams),
		(uint8_t) wSrcKeyNo,
		(uint8_t) wSrcKeyVer,
		(uint8_t) wDstKeyNo,
		aSV,
		bSvLen));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPRIMENTAG);
}
#endif /* NXPBUILD__PHAL_MFPRIMENTAG_SAM_NONX */
