/*
 * Copyright 2020, 2024 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal functions of Sam X implementation of MIFARE (R) application layer.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phpalMifare.h>
#include <phalMfpEVx.h>
#include "phalMfpEVx_Sam_X_Int.h"
#include "../phalMfpEVx_Int.h"
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHAL_MFPEVX

phStatus_t phalMfpEVx_Sam_X_Int_CardExchange(void * pDataParams, uint16_t wOption, uint8_t bIsoLayer, uint8_t * pPayload,
    uint16_t wPayloadLen, uint8_t ** ppResponse, uint16_t * pRespLen, uint8_t * pPiccErrCode)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;

    /* Exchange the command to card based on activation level. */
    if(bIsoLayer)
    {
        /* Exchange the command in ISO14443 L4 activated state. */
        PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
            PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->pPalMifareDataParams,
            wOption,
            pPayload,
            wPayloadLen,
            &pResponse,
            pRespLen));

    }
    else
    {
        /* Exchange the command in ISO14443 L3 activated state. */
        PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL3(
            PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->pPalMifareDataParams,
            wOption,
            pPayload,
            wPayloadLen,
            &pResponse,
            pRespLen));
    }

    if((wOption == PH_EXCHANGE_DEFAULT) || (wOption == PH_EXCHANGE_BUFFER_LAST))
    {
        /* Add the status code. */
        *pPiccErrCode = pResponse[0U];

        /* Check the response for ISO14443 L4 state. */
        if(bIsoLayer || (pPayload[0U] == PHAL_MFPEVX_CMD_READ_SIG))
        {
            if(*pPiccErrCode != PHAL_MFPEVX_RESP_ADDITIONAL_FRAME)
            {
                PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Int_ComputeErrorResponse(1U, pResponse[0U], bIsoLayer));
            }

            /* Decrement the response length to discard status code. */
            --pRespLen[0U];

            /* Add the Response data excluding the status code. */
            *ppResponse = &pResponse[1U];
        }
        /* Check the response for ISO14443 L3 state. */
        else
        {
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Int_ComputeErrorResponseMfc(1, pResponse[0]));

            /* Copy the response for Read command only. Other MIFARE Classic command will return ACK / NACK / NoResponse. */
            if(*pRespLen == PHAL_MFPEVX_DATA_BLOCK_SIZE)
            {
                /* Add the Response data excluding the status code. */
                *ppResponse = &pResponse[0U];

                /* Update the PICC status code to zero. */
                *pPiccErrCode = 0U;
            }
            else
            {
                /* Decrement the response length to discard status code. */
                --pRespLen[0U];

                /* Add the Response data excluding the status code. */
                *ppResponse = &pResponse[1];
            }
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPEVX);
}

phStatus_t phalMfpEVx_Sam_X_Int_MFP_Authenticate(void * pDataParams, uint8_t bLayer4Comm, uint8_t bFirstAuth, uint8_t bKdf, uint16_t wBlockNr,
    uint16_t wKeyNo, uint16_t wKeyVer, uint8_t * pDivInput, uint8_t bDivInputLen, uint8_t * pPcdCap2In, uint8_t bPcdCap2InLen,
    uint8_t * pPcdCap2Out, uint8_t * pPdCap2)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bPiccErrCode = 0;
    uint8_t     PH_MEMLOC_REM aPcdCap2In[6];
    uint8_t     PH_MEMLOC_REM bOption = 0;

    /* Reset the PCDCap2In buffer to default data if user has not passed any information.  */
    if(pPcdCap2In == NULL)
    {
        (void) memset(aPcdCap2In, 0x00U, sizeof(aPcdCap2In));
        pPcdCap2In = &aPcdCap2In[0U];
    }

    /* Check if PCDCap2[0] consists of value for EV0 and EV1 secure message.
     * If PcdCap[0] = 0x00, then EV0 secure messaging applies.
     * If PcdCap[0] = 0x01, then EV1 secure messaging applies.
     */
    if(pPcdCap2In[0U] > PHAL_MFPEVX_SECURE_MESSAGE_EV1)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPEVX);
    }

    /* Update the option variable with proper authentication type and input PCD Caps. */
    if(!bFirstAuth)
    {
        bOption |= PHHAL_HW_SAMAV3_MFP_AUTHENTICATE_NON_FIRST;
        bPcdCap2InLen = 0U;
    }

    /* Set PCDCaps Len in case of L3 Activate state. */
    if(!bLayer4Comm)
    {
        bPcdCap2InLen = 0U;
    }

    /* Update the option variable with proper diversification information. */
    if(bDivInputLen)
        bOption |= PHHAL_HW_SAMAV3_MFP_AUTHENTICATE_DIVERSIFICATION_ON;

    /* Update the option variable with key derivation information. */
    bOption |= bKdf;

    /* Exchange the information to Hal Sam layer. */
    wStatus = phhalHw_SamAV3_Cmd_MFP_Authenticate(
        PHAL_MFPEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
        bOption,
        (uint8_t) wKeyNo,
        (uint8_t) wKeyVer,
        wBlockNr,
        pPcdCap2In,
        bPcdCap2InLen,
        pDivInput,
        bDivInputLen,
        pPcdCap2Out,
        pPdCap2,
        &bPiccErrCode);

    /* Return the error code. */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAMAV3_ERR_MIFARE_PLUS_GEN)
    {
        /* Compute the response code. */
        PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Int_ComputeErrorResponse(1, bPiccErrCode, bLayer4Comm));
    }
    else
    {
        /* Return the error code other than success.*/
        if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS)
        {
            PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Sam_X_ConvertNak(wStatus));
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPEVX);
}

phStatus_t phalMfpEVx_Sam_X_Int_CombinedReadMFP(void * pDataParams, uint8_t bCmdCode, uint16_t wBlockNr,
    uint8_t bNumBlocks, uint8_t * pBlocks)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[4];
    uint8_t     PH_MEMLOC_REM bCmdLen = 0;
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;
    uint8_t     PH_MEMLOC_REM bPiccErrCode = 0;
    uint16_t    PH_MEMLOC_REM wBufferOffset = 0;
    uint16_t    PH_MEMLOC_REM wOption = 0;
    uint32_t    PH_MEMLOC_REM dwTMIStatus = 0;

    /* Update the buffering option to DEFAULT. */
    wOption = PH_EXCHANGE_DEFAULT;

    /* Build the command frame. */
    aCmdBuff[bCmdLen++] = bCmdCode;
    aCmdBuff[bCmdLen++] = (uint8_t) (wBlockNr & 0x00FFU); /* LSB */
    aCmdBuff[bCmdLen++] = (uint8_t) ((wBlockNr & 0xFF00U) >> 8U);   /* MSB */
    aCmdBuff[bCmdLen++] = bNumBlocks;

    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
        PH_TMIUTILS_TMI_STATUS, &dwTMIStatus));

    /* Check TMI Collection Status */
    if(dwTMIStatus)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
            PH_TMIUTILS_NO_PADDING, aCmdBuff, bCmdLen, NULL, 0U, PHAL_MFPEVX_DATA_BLOCK_SIZE));
    }

    /* Iterate until SUCCESS is received. */
    do
    {
        /* Exchange the ReadCmd information to HAL layer. */
        wStatus = phhalHw_SamAV3_Cmd_MFP_CombinedRead(
            PHAL_MFPEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
            wOption,
            aCmdBuff,
            bCmdLen,
            &pResponse,
            &wRespLen,
            &bPiccErrCode);

        /* Extract and validate the response code. */
        if(((wStatus & PH_ERR_MASK) == PHHAL_HW_SAMAV3_ERR_MIFARE_PLUS_GEN) ||
            ((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING) ||
            ((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS))
        {
            /* Check for response if its first frame. */
            if((wOption != PH_EXCHANGE_RXCHAINING) || ((wStatus & PH_ERR_MASK) == PHHAL_HW_SAMAV3_ERR_MIFARE_PLUS_GEN))
            {
                /* Compute the response code. */
                PH_CHECK_SUCCESS_FCT(wStatus1, phalMfpEVx_Int_ComputeErrorResponse(1U, bPiccErrCode, PH_ON));
            }
        }
        else
        {
            return wStatus;
        }

        /* Copy the received data to the response buffer. */
        (void) memcpy(&pBlocks[wBufferOffset], pResponse, wRespLen);
        wBufferOffset += wRespLen;

        /* Update the buffering option to RX_CHAINING */
        wOption = PH_EXCHANGE_RXCHAINING;

        /* Reset Command buffer length as it not required for next frames. */
        bCmdLen = 0U;

    } while((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS);

    /* Check TMI Collection Status */
    if(dwTMIStatus)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
            PH_TMIUTILS_NO_PADDING, NULL, 0U, pBlocks, (bNumBlocks * 16U), PHAL_MFPEVX_DATA_BLOCK_SIZE));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPEVX);
}

phStatus_t phalMfpEVx_Sam_X_Int_MFP_CombinedWrite(void * pDataParams, uint8_t bCmdCode, uint16_t wSrcBlockNr, uint16_t wDstBlockNr, uint8_t * pData,
    uint8_t bDataLen, uint8_t * pTMC, uint8_t * pTMV)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bPiccErrCode = 0;
    uint8_t     PH_MEMLOC_REM aHeader[5 + 1 /* Allocate Ext */];
    uint8_t     PH_MEMLOC_REM bHeaderLen = 0;

    uint8_t     PH_MEMLOC_REM bTMIBufLen = 0;
    uint32_t    PH_MEMLOC_REM dwTMIStatus = 0;

    /* Reset the data buffer and its length variable. */
    bHeaderLen = 0U;
    (void) memset(aHeader, 0x00U, sizeof(aHeader));

    /* Frame the payload information. */
    aHeader[bHeaderLen++] = bCmdCode;
    aHeader[bHeaderLen++] = (uint8_t) (wSrcBlockNr & 0x00FFU);
    aHeader[bHeaderLen++] = (uint8_t) ((wSrcBlockNr & 0xFF00U) >> 8U);

    /* Add DesBlockNr to data buffer. */
    if((bCmdCode == PHAL_MFPEVX_CMD_INCRTR) || (bCmdCode == PHAL_MFPEVX_CMD_INCRTR_M) ||
        (bCmdCode == PHAL_MFPEVX_CMD_DECRTR) || (bCmdCode == PHAL_MFPEVX_CMD_DECRTR_M))
    {
        aHeader[bHeaderLen++] = (uint8_t) (wDstBlockNr & 0x00FFU);
        aHeader[bHeaderLen++] = (uint8_t) ((wDstBlockNr & 0xFF00U) >> 8U);
    }

    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
        PH_TMIUTILS_TMI_STATUS, &dwTMIStatus));

    /* Check TMI Collection Status */
    if(dwTMIStatus)
    {
        /* For Write commands. */
        if((bCmdCode == PHAL_MFPEVX_CMD_WRITE_EM) || (bCmdCode == PHAL_MFPEVX_CMD_WRITE_EN) ||
            (bCmdCode == PHAL_MFPEVX_CMD_WRITE_PM) || (bCmdCode == PHAL_MFPEVX_CMD_WRITE_PN))
        {
            /* For TMI buffering. */
            aHeader[bHeaderLen] = (uint8_t) (bDataLen / PHAL_MFPEVX_DATA_BLOCK_SIZE);

            /* Update the TMI Buffer length. */
            bTMIBufLen = (uint8_t) (bHeaderLen + 1U);
        }

        /* For Value commands. */
        else
        {
            /* Update the TMI Buffer length. */
            bTMIBufLen = bHeaderLen;
        }

        PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
            PH_TMIUTILS_NO_PADDING, aHeader, bTMIBufLen, pData, bDataLen, PHAL_MFPEVX_DATA_BLOCK_SIZE));
    }

    /* Buffer the payload information to Hal exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_MFP_CombinedWrite(
        PHAL_MFPEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
        PH_EXCHANGE_BUFFER_FIRST,
        aHeader,
        bHeaderLen,
        NULL,
        NULL,
        NULL));

    /* Buffer the data / value to Hal exchange buffer and exchange the buffered information to Sam hardware. */
    wStatus = phhalHw_SamAV3_Cmd_MFP_CombinedWrite(
        PHAL_MFPEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
        PH_EXCHANGE_BUFFER_LAST,
        pData,
        bDataLen,
        pTMC,
        pTMV,
        &bPiccErrCode);

    /* Return the error code. */
    if(((wStatus & PH_ERR_MASK) == PHHAL_HW_SAMAV3_ERR_MIFARE_PLUS_GEN) ||
        ((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS))
    {
        /* Compute the response code. */
        PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Int_ComputeErrorResponse(1, bPiccErrCode, PH_ON));
    }
    else
    {
        return wStatus;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPEVX);
}

phStatus_t phalMfpEVx_Sam_X_Int_WriteExtMfc(void * pDataParams, uint8_t bCmdCode, uint8_t bBlockNo, uint8_t * pValue, uint8_t wValueLen,
    uint8_t * pTMC, uint8_t * pTMV)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aCmdBuff[2];
    uint8_t     PH_MEMLOC_REM aTmp[4] = { 0x00, 0x00, 0x00, 0x00 };
    uint8_t     PH_MEMLOC_REM bCmdLen = 0;
    uint8_t *   PH_MEMLOC_REM pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    /* Frame the command. */
    aCmdBuff[bCmdLen++] = bCmdCode;
    aCmdBuff[bCmdLen++] = bBlockNo;

    /* Perform TMICollection. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Sam_X_Int_CollectTMI(pDataParams, aCmdBuff, bCmdLen, pValue, wValueLen));

    /* Exchange the command frame (first part) */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL3(
        PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->pPalMifareDataParams,
        PH_EXCHANGE_DEFAULT,
        aCmdBuff,
        bCmdLen,
        &pResponse,
        &wRespLen));

    if(wRespLen == 1)
    {
        /* Evaluate the response for any errors. */
        PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Int_ComputeErrorResponseMfc(wRespLen, pResponse[0]));
    }
    else
    {
        /* Do not check for Error code if its a Transfer command because Transfer command will return the
         * TMC and TMV data if TM Protected block for first part of command exchange.
         */
        if(bCmdCode != PHAL_MFPEVX_CMD_MFC_TRANSFER)
        {
            /* If Response byte is not equal to 1, then it's a PH_ERR_PROTOCOL_ERROR */
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_MIFARE);
        }
    }

    if((bCmdCode == PHAL_MFPEVX_CMD_MFC_INCREMENT) || (bCmdCode == PHAL_MFPEVX_CMD_MFC_DECREMENT) ||
        (bCmdCode == PHAL_MFPEVX_CMD_MFC_RESTORE))
    {
        if(bCmdCode == PHAL_MFPEVX_CMD_MFC_RESTORE)
        {
            pValue = aTmp;
            wValueLen = 4U;
        }

        /* Exchange the data (second part) */
        wStatus = phpalMifare_ExchangeL3(
            PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->pPalMifareDataParams,
            PH_EXCHANGE_DEFAULT,
            pValue,
            wValueLen,
            &pResponse,
            &wRespLen
        );
    }

    /* Check for success in the response.
     * The error handling will be performed as follows.
     *    1. If TMC and TMV is not returned, the wRxLength will be one and the error handling will be processed.
     *    2. If TMC and TMV is returned, the wRxLenth will be greater than one. So there will be no error handling
     *       processed rather it will just return.
     */
    PH_CHECK_SUCCESS_FCT(wStatus, phalMfpEVx_Int_ComputeErrorResponseMfc(wRespLen, pResponse[0U]));

    /* Check if TMV and TMC is returned for Transfer command. */
    if(bCmdCode == PHAL_MFPEVX_CMD_MFC_TRANSFER)
    {
        /* If TMC and TMV is returned the RxLength will be greater than 1. */
        if(wRespLen > 1U)
        {
            /* Check if response equals to sum of TMC and TMV size. */
            if(wRespLen != (PHAL_MFPEVX_SIZE_TMC + PHAL_MFPEVX_SIZE_TMV))
            {
                return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFPEVX);
            }

            /* Check if NULL is passed for TMC parameter. */
            if(pTMC == NULL)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPEVX);
            }

            /* Check if NULL is passed for TMV parameter. */
            if(pTMV == NULL)
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFPEVX);
            }

            (void) memcpy(pTMC, pResponse, PHAL_MFPEVX_SIZE_TMC);
            (void) memcpy(pTMV, &pResponse[PHAL_MFPEVX_SIZE_TMC], PHAL_MFPEVX_SIZE_TMV);
        }
        else
        {
            if((pTMC != NULL) && (pTMV != NULL))
            {
                (void) memset(pTMC, 0x00U, PHAL_MFPEVX_SIZE_TMC);
                (void) memset(pTMV, 0x00U, PHAL_MFPEVX_SIZE_TMV);
            }
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPEVX);
}

phStatus_t phalMfpEVx_Sam_X_ConvertNak(phStatus_t wStatus)
{
    phStatus_t  PH_MEMLOC_REM wStatusTmp = 0;

    /* Remove the error code with respect to SAM HAL. */
    switch(wStatus & PH_ERR_MASK)
    {
        case PHHAL_HW_SAMAV3_ERR_MIFARE_NAK0:
            wStatusTmp = PH_ADD_COMPCODE(PHPAL_MIFARE_ERR_NAK0, PH_COMP_PAL_MIFARE);
            break;

        case PHHAL_HW_SAMAV3_ERR_MIFARE_NAK1:
            wStatusTmp = PH_ADD_COMPCODE(PHPAL_MIFARE_ERR_NAK1, PH_COMP_PAL_MIFARE);
            break;

        case PHHAL_HW_SAMAV3_ERR_MIFARE_NAK4:
            wStatusTmp = PH_ADD_COMPCODE(PHPAL_MIFARE_ERR_NAK4, PH_COMP_PAL_MIFARE);
            break;

        case PHHAL_HW_SAMAV3_ERR_MIFARE_NAK5:
            wStatusTmp = PH_ADD_COMPCODE(PHPAL_MIFARE_ERR_NAK5, PH_COMP_PAL_MIFARE);
            break;

        case PH_ERR_SUCCESS_INCOMPLETE_BYTE:
            wStatusTmp = PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_MIFARE);
            break;

        default:
            wStatusTmp = wStatus;
            break;
    }

    return wStatusTmp;
}

phStatus_t phalMfpEVx_Sam_X_Int_ComputeTMACSessionVectors(void * pDataParams, uint8_t bOption, uint16_t wSrcKeyNo, uint16_t wSrcKeyVer, uint16_t wDstKeyNo,
    uint8_t * pTMC, uint8_t * pUid, uint8_t bUidLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aSV[16];
    uint8_t     PH_MEMLOC_REM aIV[16];
    uint8_t     PH_MEMLOC_REM bSvLen = 0;
    uint32_t    PH_MEMLOC_REM dwTMC = 0;

    /* Convert TMC to Uint32. */
    dwTMC = (uint32_t) (pTMC[0U] | (pTMC[1U] << 8U) | (pTMC[2U] << 16U) | (pTMC[3U] << 24U));

    /* If TMC is 0xFFFFFFFF, then return error */
    if(dwTMC == 0xFFFFFFFFU)
    {
        return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_AL_MFPEVX);
    }

    /* Increment the TMC by 1. */
    dwTMC++;

    /* Clear the session vector SV. */
    (void) memset(aSV, 0U, 16U);
    (void) memset(aIV, 0U, 16U);

    /* Compute the session vector. */
    aSV[bSvLen++] = (uint8_t) ((bOption == PHAL_MFPEVX_SAM_X_SESSION_TMAC_ENC) ? 0xA5U : 0x5AU);
    aSV[bSvLen++] = 0x00U;
    aSV[bSvLen++] = 0x01U;
    aSV[bSvLen++] = 0x00U;
    aSV[bSvLen++] = 0x80U;

    /* Append the TMC information. */
    aSV[bSvLen++] = (uint8_t) (dwTMC & 0xFFU);
    aSV[bSvLen++] = (uint8_t) ((dwTMC & 0xFF00U) >> 8U);
    aSV[bSvLen++] = (uint8_t) ((dwTMC & 0xFF0000U) >> 16U);
    aSV[bSvLen++] = (uint8_t) ((dwTMC & 0xFF000000U) >> 24U);

    /* Append the UID information. */
    (void) memcpy(&aSV[bSvLen], pUid, bUidLen);
    bSvLen = 16U;

    /* Load zero IV. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_LoadInitVector(
        PHAL_MFPEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
        PHHAL_HW_SAMAV3_CMD_SAM_LOAD_IV_MODE_SET_IV,
        aIV,
        16));

    /* Exchange the session vector information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SamAV3_Cmd_SAM_DeriveKey(
        PHAL_MFPEVX_RESOLVE_HAL_DATAPARAMS(pDataParams),
        (uint8_t) wSrcKeyNo,
        (uint8_t) wSrcKeyVer,
        (uint8_t) wDstKeyNo,
        aSV,
        16U));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPEVX);
}

phStatus_t phalMfpEVx_Sam_X_Int_CollectTMI(void * pDataParams, uint8_t * pCmdBuf, uint16_t wCmdLen, uint8_t * pData, uint16_t wDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint32_t    PH_MEMLOC_REM dwTMIStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_GetConfig(PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
        PH_TMIUTILS_TMI_STATUS, &dwTMIStatus));

    /* Check TMI Collection Status */
    if(dwTMIStatus)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phTMIUtils_CollectTMI(PHAL_MFPEVX_RESOLVE_DATAPARAMS(pDataParams)->pTMIDataParams,
            PH_TMIUTILS_NO_PADDING, pCmdBuf, wCmdLen, pData, wDataLen, PHAL_MFPEVX_DATA_BLOCK_SIZE));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFPEVX);
}

#endif /* NXPBUILD__PHAL_MFPEVX */
