/*
 * Copyright 2016 - 2018, 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Generic MIFARE Plus(R) Application Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phpalMifare.h>
#include <phalMfp.h>
#include "phalMfp_Sam_X_Int.h"
#include "../phalMfp_Int.h"
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHAL_MFP

phStatus_t phalMfp_Sam_X_Int_MFP_Authenticate(void * pDataParams, uint8_t bLayer4Comm, uint8_t bFirstAuth, uint8_t bKdf, uint16_t wBlockNr,
	uint16_t wKeyNo, uint16_t wKeyVer, uint8_t * pDivInput, uint8_t bDivInputLen, uint8_t * pPcdCap2In, uint8_t bPcdCap2InLen,
	uint8_t * pPcdCap2Out, uint8_t * pPdCap2)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM bPiccErrCode = 0;
	uint8_t		PH_MEMLOC_REM bOption = 0;

	/* Check if bPcdCap2InLen data is not greater than 6. */
    if (bPcdCap2InLen > 6)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFP);
    }

	/* Update Option based on the hardware. */
	if(PH_GET_COMPID(pDataParams) == PHAL_MFP_SAMAV2_X_ID)
	{
		bOption |= bFirstAuth;

		if(bKdf == PHAL_MFP_AUTHENTICATE_SL2_KDF)
		{
			bOption |= 0x06;
		}

		if(bKdf == PHAL_MFP_AUTHENTICATE_SL3_KDF)
		{
			bOption |= 0x08;
		}
	}
	else
	{
		/* Update the option variable with proper authentication type and input PCD Caps. */
		if (!bFirstAuth)
		{
			bOption |= 0x02;
			bPcdCap2InLen = 0;
		}

		/* Update the option variable with key derivation information. */
		bOption |= bKdf;

		/* Update the option variable with proper diversification information. */
		if (bDivInputLen)
		{
			bOption |= 0x01;
		}
	}

	/* Append PCD Capibilites to command buffer in case of First authentication and Layer 3 / 4 Activated state. */
	if(bLayer4Comm)
	{
		if(!bFirstAuth)
		{
			bPcdCap2InLen = 0;
		}
	}
	else
	{
		bPcdCap2InLen = 0;
	}

	/* Exchange the information to Hal Sam layer. */
	wStatus = PHAL_MFP_AUTHENTICATE(
				pDataParams,
				bOption,
				(uint8_t) wKeyNo,
				(uint8_t) wKeyVer,
				wBlockNr,
				pPcdCap2In,
				bPcdCap2InLen,
				pDivInput,
				bDivInputLen,
				pPcdCap2Out,
				pPdCap2,
				&bPiccErrCode);

		/* Return the error code. */
		if ((wStatus & PH_ERR_MASK) == PHHAL_HW_SAMAV2_ERR_MIFARE_PLUS_GEN)
		{
			/* Compute the response code. */
			PH_CHECK_SUCCESS_FCT(wStatus, phalMfp_Int_ComputeErrorResponse(1, bPiccErrCode, bLayer4Comm));
		}
		else
		{
			/* Return the error code other than success.*/
			if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS)
			{
				return wStatus;
			}
		else
		{
			/* In case of Originality Key - reset secure messaging */
			if ((wBlockNr >= PHAL_MFP_SAM_X_ORIGINALITY_KEY_FIRST) && (wBlockNr <= PHAL_MFP_SAM_X_ORIGINALITY_KEY_LAST))
			{
				/* Kill only card Auth */
				PH_CHECK_SUCCESS_FCT(wStatus, PHHAL_HW_SAM_KILL_AUTHENTICATION(pDataParams, 0x01));
			}
		}
		}

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFP);
}

phStatus_t phalMfp_Sam_X_Int_MFP_CombinedWrite(void * pDataParams, uint8_t bCmdCode, uint16_t wSrcBlockNr, uint16_t wDstBlockNr, uint8_t * pData,
	uint8_t bDataLen)
{
	phStatus_t	PH_MEMLOC_REM wStatus = 0;
	uint8_t		PH_MEMLOC_REM bPiccErrCode = 0;
	uint8_t		PH_MEMLOC_REM aHeader[5];
	uint8_t		PH_MEMLOC_REM bHeaderLen = 0;

	/* Reset the data buffer and its length variable. */
	bHeaderLen = 0;
	memset(aHeader, 0x00, sizeof(aHeader));		/* PRQA S 3200 */

	/* Frame the payload information. */
	aHeader[bHeaderLen++] = bCmdCode;
	aHeader[bHeaderLen++] = (uint8_t) (wSrcBlockNr & 0x00FF);
	aHeader[bHeaderLen++] = (uint8_t) ((wSrcBlockNr & 0xFF00) >> 8);

	/* Add DesBlockNr to data buffer. */
	if((bCmdCode == PHAL_MFP_CMD_INCRTR) || (bCmdCode == PHAL_MFP_CMD_INCRTR_M) ||
	   (bCmdCode == PHAL_MFP_CMD_DECRTR) || (bCmdCode == PHAL_MFP_CMD_DECRTR_M))
	{
		aHeader[bHeaderLen++] = (uint8_t) (wDstBlockNr & 0x00FF);
		aHeader[bHeaderLen++] = (uint8_t) ((wDstBlockNr & 0xFF00) >> 8);
	}

	/* Buffer the payload information to Hal exchange buffer. */
	PH_CHECK_SUCCESS_FCT(wStatus, PHAL_MFP_COMBINED_WRITE(
		pDataParams,
		PH_EXCHANGE_BUFFER_FIRST,
		aHeader,
		bHeaderLen,
		NULL));

	/* Buffer the data / value to Hal exchange buffer and exchagne the bufferred information to Sam hardware. */
	wStatus = PHAL_MFP_COMBINED_WRITE(
		pDataParams,
		PH_EXCHANGE_BUFFER_LAST,
		pData,
		bDataLen,
		&bPiccErrCode);

		/* Return the error code. */
		if (((wStatus & PH_ERR_MASK) == PHHAL_HW_SAMAV2_ERR_MIFARE_PLUS_GEN) ||
			((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS))
		{
			/* Compute the response code. */
			PH_CHECK_SUCCESS_FCT(wStatus, phalMfp_Int_ComputeErrorResponse(1, bPiccErrCode, PH_ON));
		}
		else
		{
			return wStatus;
		}

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFP);
}

phStatus_t phalMfp_Sam_X_Int_CombinedReadMFP(void * pDataParams, uint8_t bCmdCode, uint16_t wBlockNr, uint8_t bNumBlocks, uint8_t * pBlocks)
{
	phStatus_t  PH_MEMLOC_REM wStatus = 0;
	phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
	uint8_t     PH_MEMLOC_REM aCmdBuff[4];
	uint8_t     PH_MEMLOC_REM bCmdLen = 0;
	uint8_t *   PH_MEMLOC_REM pResponse = NULL;
	uint16_t    PH_MEMLOC_REM wRespLen = 0;
	uint8_t     PH_MEMLOC_REM bPiccErrCode = 0;
	uint8_t     PH_MEMLOC_REM bBufferOffset = 0;
	uint16_t    PH_MEMLOC_REM wOption = 0;

	/* Update the bufferring option to DEFAULT. */
	wOption = PH_EXCHANGE_DEFAULT;

	/* Build the command frame. */
	aCmdBuff[bCmdLen++] = bCmdCode;
	aCmdBuff[bCmdLen++] = (uint8_t) (wBlockNr & 0x00FF); /* LSB */
	aCmdBuff[bCmdLen++] = (uint8_t) ((wBlockNr & 0xFF00) >> 8);   /* MSB */
	aCmdBuff[bCmdLen++] = bNumBlocks;

	/* Iterate untill SUCCESS is received. */
	do
	{
		/* Excahgne the ReadCmd information to HAL layer. */
		wStatus = PHAL_MFP_COMBINED_READ(
			pDataParams,
			wOption,
			aCmdBuff,
			bCmdLen,
			&pResponse,
			&wRespLen,
			&bPiccErrCode);

		/* Extract and validate the response code. */
		if (((wStatus & PH_ERR_MASK) == PHHAL_HW_SAMAV2_ERR_MIFARE_PLUS_GEN) ||
			((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING) ||
			((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS))
		{
			/* Check for response if its first frame. */
			if((wOption != PH_EXCHANGE_RXCHAINING) || ((wStatus & PH_ERR_MASK) == PHHAL_HW_SAMAV2_ERR_MIFARE_PLUS_GEN))
			{
				/* Compute the response code. */
				PH_CHECK_SUCCESS_FCT(wStatus1, phalMfp_Int_ComputeErrorResponse(1, bPiccErrCode, PH_ON));
			}
		}
		else
		{
			return wStatus;
		}

		/* Copy the received data to the response buffer. */
		memcpy(&pBlocks[bBufferOffset], pResponse, wRespLen);/* PRQA S 3200 */
		bBufferOffset = (uint8_t) wRespLen;

		/* Update the buffering option to RX_CHAINING */
		wOption = PH_EXCHANGE_RXCHAINING;

		/* Reset Command buffer length as it not required for next frames. */
		bCmdLen = 0;

	} while((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS);

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFP);
}

#endif /* NXPBUILD__PHAL_MFP */
