/*
*         Copyright (c), NXP Semiconductors Bangalore / India
*
*                     (C)NXP Semiconductors
*       All rights are reserved. Reproduction in whole or in part is
*      prohibited without the written consent of the copyright owner.
*  NXP reserves the right to make changes without notice at any time.
* NXP makes no warranty, expressed, implied or statutory, including but
* not limited to any implied warranty of merchantability or fitness for any
* particular purpose, or that the use will not infringe any third party patent,
* copyright or trademark. NXP must not be liable for any loss or damage
*                          arising from its use.
*/

#include <ph_Status.h>
#include <ph_RefDefs.h>
#include <ph_TypeDefs.h>
#include <string.h>
#include <phTools.h>
#include <phKeyStore.h>
#ifdef NXPBUILD__PH_CRYPTOSYM
#include <phCryptoSym.h>
#endif /* NXPBUILD__PH_CRYPTOSYM */
#ifdef NXPBUILD__PH_CRYPTORNG
#include <phCryptoRng.h>
#endif /* NXPBUILD__PH_CRYPTORNG */
#include <phTMIUtils.h>
#include <phalVca.h>

#ifdef NXPBUILD__PHAL_MFIDENTITY_SW

#include "../phalMfIdentity_Int.h"
#include "phalMfIdentity_Sw.h"
#include "phalMfIdentity_Sw_Int.h"

/* APP level keys are invalid between 0x0D to 0x21. */
 #define IS_INVALID_APP_KEY(keyNo) ((((keyNo) & 0x7f) > 0x0D) 	&& (((keyNo) & 0x7f) < 0x21))

/* VC keys are invalid after 0x23. */
 #define IS_INVALID_VC_KEY(keyNo)	(((keyNo) & 0x7f) > 0x23)

/* #define LEGACY_MFID_CHGFILESETTINGS_IMPL  0x01 */

phStatus_t phalMfIdentity_Sw_Init(phalMfIdentity_Sw_DataParams_t * pDataParams, uint16_t wSizeOfDataParams, void * pPalMifareDataParams,
	void * pKeyStoreDataParams, void * pCryptoDataParamsEnc, void * pCryptoDataParamsMac, void * pCryptoRngDataParams, void * pTMIDataParams,
	void * pHalDataParams)
{
	/* data param check */
	if(sizeof(phalMfIdentity_Sw_DataParams_t) != wSizeOfDataParams)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_AL_MFIDENTITY);
	}

	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pPalMifareDataParams, PH_COMP_AL_MFIDENTITY);

#ifdef NXPBUILD__PH_CRYPTOSYM
	PH_ASSERT_NULL_PARAM(pKeyStoreDataParams, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pCryptoDataParamsEnc, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pCryptoDataParamsMac, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pCryptoRngDataParams, PH_COMP_AL_MFIDENTITY);
#endif /* NXPBUILD__PH_CRYPTOSYM */

	PH_ASSERT_NULL_PARAM(pTMIDataParams, PH_COMP_AL_MFIDENTITY);
	PH_ASSERT_NULL_PARAM(pHalDataParams, PH_COMP_AL_MFIDENTITY);

	/* init private data */
	pDataParams->wId = PH_COMP_AL_MFIDENTITY | PHAL_MFIDENTITY_SW_ID;
	pDataParams->pPalMifareDataParams = pPalMifareDataParams;
	pDataParams->pKeyStoreDataParams = pKeyStoreDataParams;
	pDataParams->pCryptoDataParamsEnc = pCryptoDataParamsEnc;
	pDataParams->pCryptoDataParamsMac = pCryptoDataParamsMac;
	pDataParams->pCryptoRngDataParams = pCryptoRngDataParams;
	pDataParams->pTMIDataParams = pTMIDataParams;
	pDataParams->pHalDataParams = pHalDataParams;
	/* 2 Byte CRC initial value in Authenticate mode. */
	pDataParams->wCrc = PH_TOOLS_CRC16_PRESET_ISO14443A;

	/* 4 Byte CRC initial value in 0x1A, 0xAA mode. */
	pDataParams->dwCrc = PH_TOOLS_CRC32_PRESET_DF8;

	memset(pDataParams->bSesAuthENCKey, 0x00, 24);  /* PRQA S 3200 */
	pDataParams->bKeyNo = 0xFF; /* Set to invalid */
	memset(pDataParams->bIv, 0x00, 16); /* PRQA S 3200 */
	memset(pDataParams->pAid, 0x00, 3);  /* PRQA S 3200 */
	pDataParams->bAuthMode = PHAL_MFIDENTITY_NOT_AUTHENTICATED; /* Set to invalid */
	pDataParams->bWrappedMode = 0x01; /* Set to true */
	pDataParams->bCryptoMethod = 0xFF; /* No crypto just after init */
	pDataParams->wAdditionalInfo = 0x0000;
	pDataParams->bShortLenApdu = 0x01; /* Since Prime variant supports only Short Length APDU, setting this to 0x01 always */
	pDataParams->dwPayLoadLen = 0;
	pDataParams->wCmdCtr = 0;
	memset(pDataParams->bTi, 0x00, PHAL_MFIDENTITY_SIZE_TI);  /* PRQA S 3200 */
	memset(pDataParams->bSesAuthMACKey, 0x00, 16);  /* PRQA S 3200 */
	memset(pDataParams->pUnprocByteBuff, 0x00, PHAL_MFIDENTITY_SIZE_MAC);  /* PRQA S 3200 */
	pDataParams->bNoUnprocBytes = 0;
	memset(pDataParams->bLastBlockBuffer, 0x00, 16);  /* PRQA S 3200 */
	pDataParams->bLastBlockIndex = 0;
	
	pDataParams->bFileOption			= 0x00;
	pDataParams->bSDMOption				= 0x00;
	pDataParams->bUidLength				= 0x00;
	pDataParams->bKeyType				= 0x00;	
	pDataParams->bSDMKeyNo				= 0xFF;	
	pDataParams->bSDMKeyVer				= 0xFF;	

	memset(pDataParams->bUid, 0x00, 3);				/* PRQA S 3200 */
	memset(pDataParams->aSDMReadCTR, 0x00, 3);		/* PRQA S 3200 */
	memset(pDataParams->aSDMENCLength, 0x00, 3);		/* PRQA S 3200 */
	memset(pDataParams->aSDMENCOffset, 0x00, 3);	/* PRQA S 3200 */

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}

#ifdef NXPBUILD__PH_CRYPTOSYM
/* MIFARE Identity Secure Messaging commands. ------------------------------------------------------------------------------------------ */
phStatus_t phalMfIdentity_Sw_AuthenticateEv2(phalMfIdentity_Sw_DataParams_t * pDataParams, uint8_t bAuthOption, uint16_t wOption, uint16_t wKeyNo,
	uint16_t wKeyVer, uint8_t bKeyNoCard, uint8_t * pDivInput, uint8_t bDivLen, uint8_t bLenPcdCapsIn, uint8_t *pPcdCapsIn, uint8_t *pPcdCapsOut,
	uint8_t *pPdCapsOut)
{
	phStatus_t  PH_MEMLOC_REM statusTmp;

	/* Decide whether the Auth option is Non LRP or Regular EV2 */
	if((bAuthOption == PHAL_MFIDENTITY_AUTHFIRST_NON_LRP) || (bAuthOption == PHAL_MFIDENTITY_AUTHNONFIRST_NON_LRP))
	{
		PH_CHECK_SUCCESS_FCT(statusTmp, phalMfIdentity_Sw_Int_AuthenticateEv2(
			pDataParams,
			bAuthOption,
			wOption,
			wKeyNo,
			wKeyVer,
			bKeyNoCard,
			pDivInput,
			bDivLen,
			bLenPcdCapsIn,
			pPcdCapsIn,
			pPcdCapsOut,
			pPdCapsOut
		));
	}
#ifdef NXPBUILD__PH_CRYPTOSYM_LRP
	else if((bAuthOption == PHAL_MFIDENTITY_AUTHFIRST_LRP) || (bAuthOption == PHAL_MFIDENTITY_AUTHNONFIRST_LRP))
	{
		PH_CHECK_SUCCESS_FCT(statusTmp, phalMfIdentity_Sw_Int_AuthenticateLrp(
			pDataParams,
			bAuthOption,
			wOption,
			wKeyNo,
			wKeyVer,
			bKeyNoCard,
			pDivInput,
			bDivLen,
			bLenPcdCapsIn,
			pPcdCapsIn,
			pPcdCapsOut,
			pPdCapsOut
		));
	}
#endif /* NXPBUILD__PH_CRYPTOSYM_LRP */
	else
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
	}

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}
#endif /*NXPBUILD__PH_CRYPTOSYM */




/* MIFARE Identity Memory and Configuration commands. ---------------------------------------------------------------------------------- */
#ifdef NXPBUILD__PH_CRYPTOSYM
phStatus_t phalMfIdentity_Sw_SetConfiguration(phalMfIdentity_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t * pData, uint8_t bDataLen)
{
	uint8_t  PH_MEMLOC_REM bCmdBuff[8];
	uint16_t PH_MEMLOC_REM wCmdLen = 0;
	uint8_t  PH_MEMLOC_REM bPaddingMethod = PH_CRYPTOSYM_PADDING_MODE_1;
	phStatus_t PH_MEMLOC_REM statusTmp;
	uint8_t PH_MEMLOC_REM bCryptoMethodBkup = 0xFF;
	uint8_t PH_MEMLOC_REM bEnableLrp = 0x00;

	/* form the command */
	bCmdBuff[wCmdLen++] = PHAL_MFIDENTITY_CMD_SET_CONFIG;
	bCmdBuff[wCmdLen++] = bOption;
	switch(bOption)
	{
		/* Data = 1B configuration data */
		case PHAL_MFIDENTITY_SET_CONFIG_OPTION0:
			/* User defined SAK */
		case PHAL_MFIDENTITY_SET_CONFIG_OPTION3:
			/* Secure Messaging Configuration */
		case PHAL_MFIDENTITY_SET_CONFIG_OPTION4:
			/* Capability data, consisting of VCTID Override, PDCap1 and PDCap2 */
		case PHAL_MFIDENTITY_SET_CONFIG_OPTION5:
			/*Failed Authentication configuration */
		case PHAL_MFIDENTITY_SET_CONFIG_OPTION10:
			break;
			/*Hardware configuration*/
		case PHAL_MFIDENTITY_SET_CONFIG_OPTION11:
			break;

			/* User defined ATS */
		case PHAL_MFIDENTITY_SET_CONFIG_OPTION2:
			bPaddingMethod = PH_CRYPTOSYM_PADDING_MODE_2;
			break;

		default:
			/* Do not check for Invalid parameter here. */
			break;
	}

	/* In case of Option-5, where LRP to be enabled(i.e pData[4] = 01), Data is Sent in FULL and the response is expected in PLAIN.
	* Hence the followiung interface should know that it should not process the Response data in FULL mode. Hence
	* using the below flag variable */
	if(bOption == PHAL_MFIDENTITY_SET_CONFIG_OPTION5 && (pData[4] == 0x02))
	{
		bCryptoMethodBkup = pDataParams->bCryptoMethod;
		pDataParams->bCryptoMethod = PH_CRYPTOSYM_CONFIG_LRP; /*  This is just a value. Same to be checked inside phalMfIdentity_Sw_Int_Write_Enc API */
		bEnableLrp = 0x01;
	}

	statusTmp = phalMfIdentity_Sw_Int_Write_Enc(
		pDataParams,
		PHAL_MFIDENTITY_DEFAULT_MODE,
		bCmdBuff,
		wCmdLen,
		bPaddingMethod,
		0x00,
		pData,
		(uint16_t) bDataLen
	);

	/* Store back the original Crypto method the pDataParams->bCryptoMethod once the data is complete */
	if(bEnableLrp)
	{
		pDataParams->bCryptoMethod = bCryptoMethodBkup;
	}

	if((statusTmp == PH_ERR_SUCCESS) && (bEnableLrp))
	{
		phalMfIdentity_Sw_Int_ResetAuthStatus(pDataParams);
		bEnableLrp = 0x00;
	}

	return statusTmp;
}
#endif /* NXPBUILD__PH_CRYPTOSYM */

phStatus_t phalMfIdentity_Sw_GetVersion(phalMfIdentity_Sw_DataParams_t * pDataParams, uint8_t * pVerInfo, uint8_t * pVerLength)
{
	phStatus_t  PH_MEMLOC_REM statusTmp;
	uint8_t     PH_MEMLOC_REM bCmdBuff[8];
	uint16_t    PH_MEMLOC_REM wRxlen;
	uint8_t     PH_MEMLOC_REM *pRecv;
	uint8_t		PH_MEMLOC_REM aConfigValue[2];

	/* form the command */
	bCmdBuff[0] = PHAL_MFIDENTITY_CMD_GET_VERSION;

	PH_CHECK_SUCCESS_FCT(statusTmp, phalMfIdentity_Sw_Int_ReadData_Plain(
		pDataParams,
		(pDataParams->bAuthMode == PHAL_MFIDENTITY_AUTHENTICATEEV2) ? PHAL_MFIDENTITY_COMMUNICATION_MACD : PHAL_MFIDENTITY_COMMUNICATION_PLAIN,
		bCmdBuff,
		1,
		&pRecv,
		&wRxlen
	));
#ifdef RDR_LIB_PARAM_CHECK
	/* If received Data length is not equal to 28B,29B(In case of 7BUID) or 30B(In case of 10B UID), 27B(In case of 4B UID)
	* then its a Protocol Error
	*/
	if(((wRxlen != PHAL_MFIDENTITY_DEF_VERSION_LENGTH) && (wRxlen != PHAL_MFIDENTITY_VERSION_LENGTH_WITH_FABKEY_ID)) &&
		(wRxlen != PHAL_MFIDENTITY_10B_VERSION_LENGTH) &&
		(wRxlen != PHAL_MFIDENTITY_4B_VERSION_LENGTH))
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFIDENTITY);
	}
#endif
	if(wRxlen > 0)
		memcpy(pVerInfo, pRecv, wRxlen); /* PRQA S 3200 */

	*pVerLength = (uint8_t) (wRxlen);

	/* Do a Set Config of ADDITIONAL_INFO to set  the length(wLength) of the Version string */
	aConfigValue[0] = (uint8_t) (wRxlen & 0xFF);
	aConfigValue[1] = (uint8_t) ((wRxlen & 0xFF) >> 8);
	phalMfIdentity_Sw_SetConfig((phalMfIdentity_Sw_DataParams_t *) pDataParams, PHAL_MFIDENTITY_ADDITIONAL_INFO, aConfigValue);

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}

#ifdef NXPBUILD__PH_CRYPTOSYM
phStatus_t phalMfIdentity_Sw_GetCardUID(phalMfIdentity_Sw_DataParams_t * pDataParams, uint8_t * pUid, uint8_t * pUidLength)
{
	phStatus_t  PH_MEMLOC_REM statusTmp;
	uint8_t     PH_MEMLOC_REM bCmdBuff[20];
	uint16_t    PH_MEMLOC_REM wRxlen;
	uint16_t    PH_MEMLOC_REM wCmdLen = 0;
	uint8_t     PH_MEMLOC_REM *pRecv;

	/* form the command */
	bCmdBuff[wCmdLen++] = PHAL_MFIDENTITY_CMD_GET_CARD_UID;

	PH_CHECK_SUCCESS_FCT(statusTmp, phalMfIdentity_Sw_Int_ReadData_Enc(
		pDataParams,
		PHAL_MFIDENTITY_COMMUNICATION_ENC,
		bCmdBuff,
		wCmdLen,
		&pRecv,
		&wRxlen
	));

	/* Response will be received as
	* 1. 7 byte UID
	* 2. [1 Byte UID Format] + [1 byte UID Length(0x04)] + 4 byte UID
	* 3. [1 Byte UID Format] + [1 byte UID Length(0x0A)] + 10 byte UID
	*/
	if((wRxlen != PHAL_MFIDENTITY_DEFAULT_UID_LENGTH) &&
		(wRxlen != (PHAL_MFIDENTITY_10B_UID_LENGTH) &&
		(wRxlen != PHAL_MFIDENTITY_4B_UID_LENGTH)))
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFIDENTITY);
	}

	if((wRxlen == PHAL_MFIDENTITY_10B_UID_LENGTH) ||
		(wRxlen == PHAL_MFIDENTITY_4B_UID_LENGTH))
	{
		/* In case of 4B/10B UID, strip out first 2 bytes as it contains UID format and UID length */
		*pUidLength = (uint8_t) (wRxlen - 2);
		memcpy(pUid, &pRecv[2], *pUidLength);
	}
	else
	{
		*pUidLength = (uint8_t) wRxlen;
		memcpy(pUid, pRecv, *pUidLength);
	}

	pDataParams->bUidLength = (uint8_t) wRxlen;
	memcpy(pDataParams->bUid, pUid, *pUidLength);

	/* Set the card Length in wAdditionalInfo. This is done to assist C# wrapper as it will not be able
	to recognize the card UID Legth */
	PH_CHECK_SUCCESS_FCT(statusTmp, phalMfIdentity_SetConfig(pDataParams, PHAL_MFIDENTITY_ADDITIONAL_INFO, pUidLength));

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}
#endif /* NXPBUILD__PH_CRYPTOSYM */




#ifdef NXPBUILD__PH_CRYPTOSYM
/* MIFARE Identity Key Management commands. -------------------------------------------------------------------------------------------- */
phStatus_t phalMfIdentity_Sw_ChangeKey(phalMfIdentity_Sw_DataParams_t * pDataParams, uint16_t wOption, uint16_t wOldKeyNo, uint16_t wOldKeyVer,
	uint16_t wNewKeyNo, uint16_t wNewKeyVer, uint8_t bKeyNoCard, uint8_t * pDivInput, uint8_t bDivLen)
{
	uint16_t    PH_MEMLOC_REM statusTmp;
	uint8_t     PH_MEMLOC_REM bCmdBuff[42];
	uint8_t     PH_MEMLOC_REM bWorkBuffer[42];
	uint8_t     PH_MEMLOC_REM bOldKey[32];
	uint8_t     PH_MEMLOC_REM bNewKey[32];
	uint8_t     PH_MEMLOC_REM bNewKeyLen = 0;
	uint8_t     PH_MEMLOC_REM bIndex;
	uint16_t    PH_MEMLOC_REM wRxlen;
	uint16_t    PH_MEMLOC_REM wOldKeyType;
	uint16_t    PH_MEMLOC_REM wNewKeyType;
	uint16_t    PH_MEMLOC_REM wCmdLen = 0;
	uint8_t     PH_MEMLOC_REM bCMAC[PH_CRYPTOSYM_AES_BLOCK_SIZE];
	uint8_t     PH_MEMLOC_REM bCMacCard[8];
	uint8_t     PH_MEMLOC_REM bMacLen;
	uint8_t     PH_MEMLOC_REM bIvLen = 0;
	uint32_t    PH_MEMLOC_REM dwCrc;
	uint16_t    PH_MEMLOC_REM wTmp;
	uint16_t    PH_MEMLOC_REM wTmpOption = 0x0000;
	uint8_t     PH_MEMLOC_REM bAppId[3] = {0x00, 0x00, 0x00};
	uint8_t *   PH_MEMLOC_REM pRecv;
	uint16_t    PH_MEMLOC_REM wWorkBufferLen = 0;

#ifdef RDR_LIB_PARAM_CHECK
	if((wOption == 0x0000) || (bDivLen > 31))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
	}
	/* Validate Card Key Number */
	if(memcmp(pDataParams->pAid, bAppId, 3) == 0x00)
	{
		/* At PICC level, no changeKey can be performed */
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
	}
	else
	{
		/* At App level, 0,1,2,3,4 are valid ones. */
		if((bKeyNoCard > PHAL_MFIDENTITY_ORIGINALITY_KEY_LAST))
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
		}
	}
	if(pDataParams->bAuthMode == PHAL_MFIDENTITY_NOT_AUTHENTICATED)
	{
		return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_AL_MFIDENTITY);
	}
#endif
	memset(bWorkBuffer, 0x00, 42); /* PRQA S 3200 */
	memset(bCmdBuff, 0x00, 42); /* PRQA S 3200 */
								/* form the command */
	bCmdBuff[wCmdLen++] = PHAL_MFIDENTITY_CMD_CHANGE_KEY;
	bCmdBuff[wCmdLen++] = bKeyNoCard;

	/* The IV will be different if AuthMode is AV2. Here the
	* ENC IV has to be computed and used for encryption.
	* The MAC IV is required to generate the MAC and append this to
	* the command before sending to the card.
	*/

	if(pDataParams->bAuthMode == PHAL_MFIDENTITY_AUTHENTICATEEV2)
	{
		bIvLen = PH_CRYPTOSYM_AES_BLOCK_SIZE;
		/* the IV is constructed by encrypting with KeyID.SesAuthENCKey according to the ECB mode
		* As ECB encription doesnot use IV during the encription so we need not backup/ update with zero IV */
		PH_CHECK_SUCCESS_FCT(statusTmp, phalMfIdentity_Sw_Int_ComputeIv(
			PH_OFF,
			pDataParams->bTi,
			pDataParams->wCmdCtr,
			pDataParams->bIv
		));

		/* Encrypt IV */
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_Encrypt(
			pDataParams->pCryptoDataParamsEnc,
			PH_CRYPTOSYM_CIPHER_MODE_ECB,
			pDataParams->bIv,
			bIvLen,
			pDataParams->bIv
		));
	}
	/* Load Iv */
	PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadIv(
		pDataParams->pCryptoDataParamsEnc,
		pDataParams->bIv,
		bIvLen
	));

	PH_CHECK_SUCCESS_FCT(statusTmp, phKeyStore_GetKey(
		pDataParams->pKeyStoreDataParams,
		wOldKeyNo,
		wOldKeyVer,
		sizeof(bOldKey),
		bOldKey,
		&wOldKeyType
	));

	PH_CHECK_SUCCESS_FCT(statusTmp, phKeyStore_GetKey(
		pDataParams->pKeyStoreDataParams,
		wNewKeyNo,
		wNewKeyVer,
		sizeof(bNewKey),
		bNewKey,
		&wNewKeyType
	));

	if(wOldKeyType == PH_KEYSTORE_KEY_TYPE_DES)
	{
		wOldKeyType = PH_KEYSTORE_KEY_TYPE_2K3DES;
		memcpy(&bOldKey[8], bOldKey, 8); /* PRQA S 3200 */
	}
	if(wNewKeyType == PH_KEYSTORE_KEY_TYPE_DES)
	{
		wNewKeyType = PH_KEYSTORE_KEY_TYPE_2K3DES;
		memcpy(&bNewKey[8], bNewKey, 8); /* PRQA S 3200 */
	}
	/*
	It is allowed to change a key type for PICC master key.
	Old key may not be diversified but new key can be.
	Old key may be diversified with one round but new key can
	be diversified with two rounds.

	Key diversification method (DESFire or MFPlus) cannot be changed
	between old and new key.

	It is assumed that the diversification input specified is the same
	for both old key and new key
	*/

	if((wOption != PHAL_MFIDENTITY_NO_DIVERSIFICATION) && (bDivLen != 0x00))
	{
		if(wOption & PHAL_MFIDENTITY_CHGKEY_DIV_NEW_KEY)
		{
			if(wOption & PHAL_MFIDENTITY_CHGKEY_DIV_METHOD_CMAC)
			{
				wTmpOption = PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS;
			}
			else
			{
				wTmpOption = PH_CRYPTOSYM_DIV_MODE_DESFIRE;
				if(wOption & PHAL_MFIDENTITY_CHGKEY_DIV_NEW_KEY_ONERND)
				{
					wTmpOption |= PH_CRYPTOSYM_DIV_OPTION_2K3DES_HALF;
				}
			}
			PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_DiversifyDirectKey(
				pDataParams->pCryptoDataParamsEnc,
				wTmpOption,
				bNewKey,
				wNewKeyType,
				pDivInput,
				bDivLen,
				bNewKey
			));
		}
		if(wOption & PHAL_MFIDENTITY_CHGKEY_DIV_OLD_KEY)
		{
			if(wOption & PHAL_MFIDENTITY_CHGKEY_DIV_METHOD_CMAC)
			{
				wTmpOption |= PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS;
			}
			else
			{
				wTmpOption |= PH_CRYPTOSYM_DIV_MODE_DESFIRE;
				if(wOption & PHAL_MFIDENTITY_CHGKEY_DIV_OLD_KEY_ONERND)
				{
					wTmpOption |= PH_CRYPTOSYM_DIV_OPTION_2K3DES_HALF;
				}
			}
			PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_DiversifyDirectKey(
				pDataParams->pCryptoDataParamsEnc,
				wTmpOption,
				bOldKey,
				wOldKeyType,
				pDivInput,
				bDivLen,
				bOldKey
			));
		}

		/* Reload the IV and key since the diversify function has invalidated the key */
		/* Load the Session key which is valid for this authentication */
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadKeyDirect(
			pDataParams->pCryptoDataParamsEnc,
			pDataParams->bSesAuthENCKey,
			pDataParams->bCryptoMethod
		));
		/* The IV will be different if AuthMode is AV2. Here the
		* ENC IV has to be computed and used for encryption.
		* The MAC IV is required to generate the MAC and append this to
		* the command before sending to the card.
		*/
		if(pDataParams->bAuthMode == PHAL_MFIDENTITY_AUTHENTICATEEV2)
		{
			/* the IV is constructed by encrypting with KeyID.SesAuthENCKey according to the ECB mode
			* As ECB encription doesnot use IV during the encription so we need not backup/ update with zero IV*/
			PH_CHECK_SUCCESS_FCT(statusTmp, phalMfIdentity_Sw_Int_ComputeIv(
				PH_OFF,
				pDataParams->bTi,
				pDataParams->wCmdCtr,
				pDataParams->bIv
			));

			/* Encrypt IV */
			PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_Encrypt(
				pDataParams->pCryptoDataParamsEnc,
				PH_CRYPTOSYM_CIPHER_MODE_ECB,
				pDataParams->bIv,
				bIvLen,
				pDataParams->bIv
			));
		}
		/* Load Iv */
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadIv(
			pDataParams->pCryptoDataParamsEnc,
			pDataParams->bIv,
			bIvLen
		));

		/* Need to check whether this is required for 0x0A mode also*/
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_SetConfig(
			pDataParams->pCryptoDataParamsEnc,
			PH_CRYPTOSYM_CONFIG_KEEP_IV,
			PH_CRYPTOSYM_VALUE_KEEP_IV_ON
		));
	}

	switch(wNewKeyType)
	{
		case PH_CRYPTOSYM_KEY_TYPE_AES128:
			bNewKeyLen = PH_CRYPTOSYM_AES128_KEY_SIZE;
			break;

		case PH_CRYPTOSYM_KEY_TYPE_2K3DES:
			bNewKeyLen = PH_CRYPTOSYM_2K3DES_KEY_SIZE;
			break;

		case PH_CRYPTOSYM_KEY_TYPE_3K3DES:
			bNewKeyLen = PH_CRYPTOSYM_3K3DES_KEY_SIZE;
			break;

		default:
			return PH_ADD_COMPCODE(PH_ERR_KEY, PH_COMP_AL_MFIDENTITY);
	}

	/* Need new else if statement for AuthEV2
	* In AuthEV2, it is sent in FULL mode, meaning the Data is encrypted
	* and a MAC is calculated and attached at the end.
	* ENC Session key is used for encryption and MAC session key is
	* used for MAC calculation
	*/
	if(pDataParams->bAuthMode == PHAL_MFIDENTITY_AUTHENTICATEEV2)
	{
		if((pDataParams->bKeyNo & 0x3FU) != (bKeyNoCard & 0x3FU))
		{
			/* Copy bNewKey to the bWorkBuffer */
			for(bIndex = 0; bIndex < bNewKeyLen; bIndex++)
			{
				bWorkBuffer[bIndex] = bNewKey[bIndex];
			}

			/* Append Key Version */
			bWorkBuffer[bIndex++] = (uint8_t) wNewKeyVer;

			/* Calculate CRC32 over the new key  ie CRC32NK shall be the 4-byte CRC value taken over NewKey */
			PH_CHECK_SUCCESS_FCT(statusTmp, phTools_CalculateCrc32(
				PH_TOOLS_CRC_OPTION_DEFAULT,
				PH_TOOLS_CRC32_PRESET_DF8,
				PH_TOOLS_CRC32_POLY_DF8,
				bWorkBuffer,
				(uint16_t) bIndex - 1,
				&dwCrc
			));

			/* Key number authenticated with is not the key to be
			* changed
			* xored_Data = pNewKey ^ wKey
			* bWorkBuffer contains pNewKey
			*/
			for(bIndex = 0; bIndex < bNewKeyLen; bIndex++)
			{
				bWorkBuffer[bIndex] = bOldKey[bIndex] ^ bWorkBuffer[bIndex];
			}
			/* xored_Data+ [AES key version] + CRC32 (all prev. data) + CRC32(new key)+padding */
			/* Adding key version should always be true because we are only dealing with
			* AES128 keys here
			*/
			bIndex++;   /* Just increment bIndex because it already contains wNewKeyVer */
			memcpy(&bCmdBuff[2], bWorkBuffer, bIndex); /* PRQA S 3200 */
			wCmdLen = wCmdLen + bIndex;

			memcpy(&bCmdBuff[wCmdLen], &dwCrc, 4); /* PRQA S 3200 */
			wCmdLen += 4;

			wTmp = wCmdLen - 2;

			if(wTmp % bIvLen)
			{
				/* Apply padding */
				PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_ApplyPadding(
					PH_CRYPTOSYM_PADDING_MODE_2,
					&bCmdBuff[2],
					wTmp,
					bIvLen,
					sizeof(bCmdBuff) - 2,
					&bCmdBuff[2],
					&wTmp
				));
			}

			/* Encrypt */
			PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_Encrypt(
				pDataParams->pCryptoDataParamsEnc,
				PH_CRYPTOSYM_CIPHER_MODE_CBC | PH_EXCHANGE_BUFFER_CONT,
				&bCmdBuff[2],
				wTmp,
				&bCmdBuff[2]
			));
			wCmdLen = wTmp + 2;

		}
		else
		{
			memcpy(&bCmdBuff[wCmdLen], bNewKey, bNewKeyLen); /* PRQA S 3200 */
			wCmdLen = wCmdLen + bNewKeyLen;

			/* Also check if it is PICC master key and is an AES key is to be written.
			If so then key version also needs to be appended to the new key before
			calculating CRC */
			if(memcmp(pDataParams->pAid, bAppId, 3) == 0x00)
			{
				if((bKeyNoCard & 0xC0) == 0x80)
				{
					/* PICC master key is being changed to AES key. Version is relevant */
					bCmdBuff[wCmdLen++] = (uint8_t) wNewKeyVer;
				}
			}
			else
			{
				bCmdBuff[wCmdLen++] = (uint8_t) wNewKeyVer;
			}

			wTmp = wCmdLen - 2;
			/* Apply padding */
			PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_ApplyPadding(
				PH_CRYPTOSYM_PADDING_MODE_2,
				&bCmdBuff[2],
				wTmp,
				bIvLen,
				sizeof(bCmdBuff) - 2,
				&bCmdBuff[2],
				&wTmp
			));

			/* Encrypt */
			PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_Encrypt(
				pDataParams->pCryptoDataParamsEnc,
				PH_CRYPTOSYM_CIPHER_MODE_CBC | PH_EXCHANGE_BUFFER_CONT,
				&bCmdBuff[2],
				wTmp,
				&bCmdBuff[2]
			));
			/* Update Cmd len */
			wCmdLen = wTmp + 2;
		}

		memset(pDataParams->bIv, 0x00, bIvLen);  /* PRQA S 3200 */
		wWorkBufferLen = 0;
		bWorkBuffer[wWorkBufferLen++] = bCmdBuff[0];
		/* Add CmdCtr and TI for MAC calculation */
		bWorkBuffer[wWorkBufferLen++] = (uint8_t) (pDataParams->wCmdCtr);
		bWorkBuffer[wWorkBufferLen++] = (uint8_t) (pDataParams->wCmdCtr >> 8);
		memcpy(&bWorkBuffer[wWorkBufferLen], pDataParams->bTi, PHAL_MFIDENTITY_SIZE_TI); /* PRQA S 3200 */
		wWorkBufferLen += PHAL_MFIDENTITY_SIZE_TI;

		memcpy(&bWorkBuffer[wWorkBufferLen], &bCmdBuff[1], (wCmdLen - 1)); /* PRQA S 3200 */
		wWorkBufferLen += (wCmdLen - 1);

		/* Load Iv */
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadIv(
			pDataParams->pCryptoDataParamsMac,
			pDataParams->bIv,
			bIvLen
		));

		/* Append MAC */
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_CalculateMac(
			pDataParams->pCryptoDataParamsMac,
			(PH_CRYPTOSYM_MAC_MODE_CMAC),
			bWorkBuffer,
			wWorkBufferLen,
			bCMAC,
			&bMacLen
		));

		/* Truncate the MAC generated */
		phalMfIdentity_Sw_Int_TruncateMac(bCMAC);

		memcpy(&bCmdBuff[wCmdLen], bCMAC, 8); /* PRQA S 3200 */
		wCmdLen += 8;
	}
	else
	{
		/* ERROR: NOT_AUTHENTICATED */
		return PH_ADD_COMPCODE(PH_ERR_AUTH_ERROR, PH_COMP_AL_MFIDENTITY);
	}

	/* Send the command */
	statusTmp = phalMfIdentity_ExchangeCmd(
		pDataParams,
		pDataParams->pPalMifareDataParams,
		pDataParams->bWrappedMode,
		bCmdBuff,
		wCmdLen,
		&pRecv,
		&wRxlen
	);
	if(statusTmp != PH_ERR_SUCCESS)
	{
		if(pDataParams->bAuthMode != PHAL_MFIDENTITY_AUTHENTICATE)
		{
			phalMfIdentity_Sw_Int_ResetAuthStatus(pDataParams);
		}
		return statusTmp;
	}

	/* Max 8 byte CMAC is expected nothing more. */
	if(wRxlen > 8)
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFIDENTITY);
	}

	memcpy(bWorkBuffer, pRecv, wRxlen);  /* PRQA S 3200 */

										 /* Verification of MAC also required for AuthEV2
										 */
	if(pDataParams->bAuthMode == PHAL_MFIDENTITY_AUTHENTICATEEV2)
	{
		/* reset authentication status if the key authenticated with is changed in
		* this case the card does not return a MAC because authentication is lost
		*/
		if((pDataParams->bKeyNo & 0x3FU) == (bKeyNoCard & 0x3FU))
		{
			phalMfIdentity_Sw_Int_ResetAuthStatus(pDataParams);
		}
		else
		{
			if(wRxlen < 8) /* If no CMAC received */
			{
				return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFIDENTITY);
			}

			/* Increment the command counter. */
			pDataParams->wCmdCtr++;

			/* Load IV */
			PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadIv(
				pDataParams->pCryptoDataParamsMac,
				pDataParams->bIv,
				bIvLen
			));

			/* copy CMAC received from card*/
			memcpy(bCMacCard, &bWorkBuffer[wRxlen - 8], 8);  /* PRQA S 3200 */
			wRxlen -= 8;

			/*
			* Calculate MAC on RC || wCmdCtr || TI || RespData
			* bWorkBuffer is used as receive buffer so pDataParams->pUnprocByteBuff is used
			*/
			pDataParams->bNoUnprocBytes = 0x00;
			pDataParams->pUnprocByteBuff[pDataParams->bNoUnprocBytes++] = 0x00;
			pDataParams->pUnprocByteBuff[pDataParams->bNoUnprocBytes++] = (uint8_t) (pDataParams->wCmdCtr);
			pDataParams->pUnprocByteBuff[pDataParams->bNoUnprocBytes++] = (uint8_t) (pDataParams->wCmdCtr >> 8);
			memcpy(&pDataParams->pUnprocByteBuff[pDataParams->bNoUnprocBytes], pDataParams->bTi, PHAL_MFIDENTITY_SIZE_TI); /* PRQA S 3200 */
			pDataParams->bNoUnprocBytes += PHAL_MFIDENTITY_SIZE_TI;

			/* verify the MAC */
			PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_CalculateMac(
				pDataParams->pCryptoDataParamsMac,
				(PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_DEFAULT),
				pDataParams->pUnprocByteBuff,
				pDataParams->bNoUnprocBytes,
				bCMAC,
				&bMacLen
			));

			/* Truncate the MAC generated */
			phalMfIdentity_Sw_Int_TruncateMac(bCMAC);

			/* Compare the CMAC from card and CMAC calculated */
			if(memcmp(bCMacCard, bCMAC, 8) != 0)
			{
				phalMfIdentity_Sw_Int_ResetAuthStatus(pDataParams);
				return PH_ADD_COMPCODE(PH_ERR_INTEGRITY_ERROR, PH_COMP_AL_MFIDENTITY);
			}
		}
	}
	else
	{
		/*Do Nothing. This is for PRQA compliance */
	}
	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}

phStatus_t phalMfIdentity_Sw_GetKeyVersion(phalMfIdentity_Sw_DataParams_t * pDataParams, uint8_t bKeyNo, uint8_t bKeySetNo, uint8_t * pKeyVersion,
	uint8_t * bRxLen)
{
	/**
	* This command can be issued without valid authentication
	*/
	uint16_t    PH_MEMLOC_REM statusTmp;
	uint8_t     PH_MEMLOC_REM bCmdBuff[20];
	uint16_t    PH_MEMLOC_REM wCmdLen = 0;
	uint16_t    PH_MEMLOC_REM wRxlen;
	uint8_t     PH_MEMLOC_REM *pRecv;

#ifdef RDR_LIB_PARAM_CHECK
	uint8_t     PH_MEMLOC_REM bAppId[3] = {0x00, 0x00, 0x00};
	if(memcmp(pDataParams->pAid, bAppId, 3) == 0x00)
	{
		/* At PICC level, 1,2,3,4 are valid ones. 0 is excluded */
		if((bKeyNo > PHAL_MFIDENTITY_ORIGINALITY_KEY_LAST) || (bKeyNo < PHAL_MFIDENTITY_ORIGINALITY_KEY_FIRST))
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
		}
	}
	else
	{
		/* At App level, 0,1,2,3,4 are valid ones. */
		if((bKeyNo > PHAL_MFIDENTITY_ORIGINALITY_KEY_LAST))
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
		}
	}
#endif

	bKeySetNo = 0x00; /* To avoid warnings */

					  /* form the command */
	bCmdBuff[wCmdLen++] = PHAL_MFIDENTITY_CMD_GET_KEY_VERSION;
	bCmdBuff[wCmdLen++] = bKeyNo;

	PH_CHECK_SUCCESS_FCT(statusTmp, phalMfIdentity_Sw_Int_ReadData_Plain(
		pDataParams,
		(pDataParams->bAuthMode == PHAL_MFIDENTITY_AUTHENTICATEEV2) ? PHAL_MFIDENTITY_COMMUNICATION_MACD : PHAL_MFIDENTITY_COMMUNICATION_PLAIN,
		bCmdBuff,
		wCmdLen,
		&pRecv,
		&wRxlen
	));

	/*
	* If length of return bytes is not 1B
	*/
	if(wRxlen != 0x01)
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFIDENTITY);
	}

	if(memcpy(pKeyVersion, pRecv, wRxlen) == NULL) /* PRQA S 3200 */
	{
		return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_AL_MFIDENTITY);
	}

	*bRxLen = (uint8_t) wRxlen;

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}
#endif /* NXPBUILD__PH_CRYPTOSYM */




/* MIFARE Identity File Management commands. ------------------------------------------------------------------------------------------- */
#ifdef NXPBUILD__PH_CRYPTOSYM
phStatus_t phalMfIdentity_Sw_CreateTransactionMacFile(phalMfIdentity_Sw_DataParams_t * pDataParams, uint8_t bFileNo, uint8_t bCommMode,
	uint8_t * pAccessRights, uint8_t bKeyType, uint8_t * bTMKey, uint8_t bTMKeyVer)
{
	uint8_t     PH_MEMLOC_REM bCmdBuff[32];
	uint16_t    PH_MEMLOC_REM wCmdLen = 0;

#ifdef RDR_LIB_PARAM_CHECK
	/* Check for valid KeyType == AES128 */
	if(bKeyType != PHAL_MFIDENTITY_KEY_TYPE_AES128)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
	}
	if((bCommMode != PHAL_MFIDENTITY_COMMUNICATION_PLAIN) &&
		(bCommMode != PHAL_MFIDENTITY_COMMUNICATION_PLAIN_1) &&
		(bCommMode != PHAL_MFIDENTITY_COMMUNICATION_ENC) &&
		(bCommMode != PHAL_MFIDENTITY_COMMUNICATION_MACD))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
	}
#endif
	/* form the command */
	bCmdBuff[wCmdLen++] = PHAL_MFIDENTITY_CMD_CREATE_TRANSACTION_MACFILE;
	bCmdBuff[wCmdLen++] = bFileNo;

	/* Copy communication settings. communication settings in the first nibble so right shifting */
	bCmdBuff[wCmdLen++] = bCommMode >> 4;

	/* Copy Access rights */
	/* Consider checking KEY TYPE passed here and accordingly
	* update the bits and store in cmd buff. */
	bCmdBuff[wCmdLen++] = pAccessRights[0];
	bCmdBuff[wCmdLen++] = pAccessRights[1];
	bCmdBuff[wCmdLen++] = bKeyType;

	memcpy(&bCmdBuff[wCmdLen], bTMKey, 16); /* PRQA S 3200 */
	wCmdLen += 16;

	bCmdBuff[wCmdLen++] = bTMKeyVer;

	/* When authenticated, CommMode.Full is applied (this is different compared to
	* the other filetypes which require CommMode.MAC as no data needs to be encrypted).
	* If not authenticated, the command is send in plain of course. */

	if(pDataParams->bAuthMode == PHAL_MFIDENTITY_NOT_AUTHENTICATED)
	{
		/* COMMUNICATION IS PLAIN */
		return phalMfIdentity_Sw_Int_Write_Plain(pDataParams,
			PHAL_MFIDENTITY_DEFAULT_MODE,
			bCmdBuff,
			wCmdLen,
			PHAL_MFIDENTITY_COMMUNICATION_PLAIN,
			NULL,
			0x0000
		);
	}
	else
	{
		return phalMfIdentity_Sw_Int_Write_Enc(
			pDataParams,
			PHAL_MFIDENTITY_DEFAULT_MODE,
			bCmdBuff,
			6,
			PH_CRYPTOSYM_PADDING_MODE_1,
			0x00,
			&bCmdBuff[6],
			(wCmdLen - 6)
		);
	}
}
#endif /* NXPBUILD__PH_CRYPTOSYM */

phStatus_t phalMfIdentity_Sw_DeleteFile(phalMfIdentity_Sw_DataParams_t * pDataParams, uint8_t bFileNo)
{
	uint8_t     PH_MEMLOC_REM bCmdBuff[8];

	/* form the command */
	bCmdBuff[0] = PHAL_MFIDENTITY_CMD_DELETE_FILE;
	bCmdBuff[1] = bFileNo;

	/* COMMUNICATION IS PLAIN */
	return phalMfIdentity_Sw_Int_Write_Plain(pDataParams,
		PHAL_MFIDENTITY_DEFAULT_MODE,
		bCmdBuff,
		2,
		(pDataParams->bAuthMode == PHAL_MFIDENTITY_AUTHENTICATEEV2) ? PHAL_MFIDENTITY_COMMUNICATION_MACD : PHAL_MFIDENTITY_COMMUNICATION_PLAIN,
		NULL,
		0x0000
	);
}

phStatus_t phalMfIdentity_Sw_GetFileSettings(phalMfIdentity_Sw_DataParams_t * pDataParams, uint8_t bFileNo, uint8_t * pFSBuffer,
	uint8_t * pBufferLen)
{
    uint16_t    PH_MEMLOC_REM statusTmp;
    uint8_t     PH_MEMLOC_REM bCmdBuff[8];
    uint16_t    PH_MEMLOC_REM wRxlen;
    uint8_t     PH_MEMLOC_REM *pRecv;

    /* form the command */
    bCmdBuff[0] = PHAL_MFIDENTITY_CMD_GET_FILE_SETTINGS;
    bCmdBuff[1] = bFileNo;

    PH_CHECK_SUCCESS_FCT(statusTmp, phalMfIdentity_Sw_Int_ReadData_Plain(
        pDataParams,
        (pDataParams->bAuthMode == PHAL_MFIDENTITY_AUTHENTICATEEV2)? PHAL_MFIDENTITY_COMMUNICATION_MACD : PHAL_MFIDENTITY_COMMUNICATION_PLAIN,
        bCmdBuff,
        2,
        &pRecv,
        &wRxlen
        ));

    /* Update pBufferLen and return  */
	memcpy(pFSBuffer, pRecv, wRxlen);
    *pBufferLen = (uint8_t) wRxlen;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}

#ifdef NXPBUILD__PH_CRYPTOSYM
phStatus_t phalMfIdentity_Sw_GetFileCounters(phalMfIdentity_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bFileNo,
	uint8_t * pFileCounters, uint8_t * pRxLen)
{
	uint16_t    PH_MEMLOC_REM statusTmp;
	uint8_t     PH_MEMLOC_REM bCmdBuff[20];
    uint16_t    PH_MEMLOC_REM wRxlen;
    uint16_t    PH_MEMLOC_REM wCmdLen = 0;
    uint8_t     PH_MEMLOC_REM *pRecv;

	bCmdBuff[wCmdLen++] = PHAL_MFIDENTITY_CMD_GET_FILE_COUNTERS;
	bCmdBuff[wCmdLen++] = bFileNo;

	if(bOption == PHAL_MFIDENTITY_COMMUNICATION_PLAIN)
	{
		PH_CHECK_SUCCESS_FCT(statusTmp, phalMfIdentity_Sw_Int_ReadData_Plain(
		pDataParams,
		PHAL_MFIDENTITY_COMMUNICATION_PLAIN,
		bCmdBuff,
		wCmdLen,
		&pRecv,
		&wRxlen
		));
	}
	else
	{
		PH_CHECK_SUCCESS_FCT(statusTmp, phalMfIdentity_Sw_Int_ReadData_Enc(
			pDataParams,
			PHAL_MFIDENTITY_COMMUNICATION_ENC,
			bCmdBuff,
			wCmdLen,
			&pRecv,
			&wRxlen
			));
	}

	/* Response will be received as
	 * 1. 3 Byte SDMReadCounter
	 * 2. 2 Bytes Reserved
	 */
    if (wRxlen != PHAL_MFIDENTITY_SDM_FILE_CTR_LENGTH)
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFIDENTITY);
    }

	*pRxLen = (uint8_t)wRxlen;
	memcpy(pFileCounters, pRecv, *pRxLen);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}

phStatus_t phalMfIdentity_Sw_ChangeFileSettings(phalMfIdentity_Sw_DataParams_t * pDataParams, uint8_t bCommMode, uint8_t bFileNo,
	uint8_t bFileOption, uint8_t *pAccessRights, uint8_t bTmcLimitLen, uint8_t * pTmcLimit)
{
    uint8_t     PH_MEMLOC_REM bCmdBuff[24];
    uint16_t    PH_MEMLOC_REM wCmdLen = 0;

	/* form the command */
    bCmdBuff[wCmdLen++] = PHAL_MFIDENTITY_CMD_CHANGE_FILE_SETTINGS;
    bCmdBuff[wCmdLen++] = bFileNo;

	/* Check if Parameter passed as per Ref arch or as per Desfire Ev2. */
	if(bCommMode & PHAL_MFIDENTITY_MFID_SPECIFICS_ENABLED)
	{
		/* MFID Specific Case, it is clear that the bFileOption parameter sent by the user is prepared as per
		 * reference architecture.
		 */

		/*  bCommMode is ORed with 0x01 from C# wrapper. Clear the bCommMode[bit-0] */
		bCommMode = bCommMode & 0xFE;

		/* Copy File Option byte */
		bCmdBuff[wCmdLen++] = bFileOption;

		/* Copy Access rights */
		bCmdBuff[wCmdLen++] = pAccessRights[0];
		bCmdBuff[wCmdLen++] = pAccessRights[1];

		/* Copy optional Data If targetted TMAC File and TMC Limit flag is enabled, bAdditionalInfo contains the TMCLImit */
		if(bFileOption & PHAL_MFIDENTITY_TMCLIMITCONFIG)
		{
			if(pTmcLimit != NULL)
			{
				memcpy(&bCmdBuff[wCmdLen], pTmcLimit, bTmcLimitLen);
				wCmdLen +=  bTmcLimitLen;
			}
		}
	}
	else
	{
		/* In this Case, here the bFileOption parameter is prepared as per Desfire Ev2 Convention.
		 * Note that in this case, bit-4 and bit-5 is not enabled.
		 */

		 /* Copy communication settings. Communication settings in the first nibble so right shifting */
		if (bFileOption & 0x80)
		{
			bCmdBuff[wCmdLen++] = (uint8_t)(((bFileOption >> 4) & 0x03) | 0x80);
		}
		else
		{
			bCmdBuff[wCmdLen++] = (uint8_t)(bFileOption >> 4);
		}

		/* Copy Access rights */
		bCmdBuff[wCmdLen++] = pAccessRights[0];
		bCmdBuff[wCmdLen++] = pAccessRights[1];
	}


	/* Based on the Mode of communication selected by the caller/Application, transmit the data */
	if(((bCommMode & 0xF0U) == PHAL_MFIDENTITY_COMMUNICATION_PLAIN))
	{
		/* COMMUNICATION IS PLAIN */
		return phalMfIdentity_Sw_Int_Write_Plain(pDataParams,
		PHAL_MFIDENTITY_DEFAULT_MODE,
        bCmdBuff,
        wCmdLen,
        PHAL_MFIDENTITY_COMMUNICATION_PLAIN,
        NULL,
        0x0000
        );
	}
	#ifdef NXPBUILD__PH_CRYPTOSYM
	else if((bCommMode & 0xF0U) == PHAL_MFIDENTITY_COMMUNICATION_MACD)
	{
		/* COMMUNICATION IS Mac */
		return phalMfIdentity_Sw_Int_Write_Plain(pDataParams,
		PHAL_MFIDENTITY_DEFAULT_MODE,
        bCmdBuff,
        wCmdLen,
        PHAL_MFIDENTITY_COMMUNICATION_MACD,
        NULL,
        0x0000
        );
	}
	else if((bCommMode & 0xF0U) == PHAL_MFIDENTITY_COMMUNICATION_ENC)
	{
		return phalMfIdentity_Sw_Int_Write_Enc(pDataParams,
			PHAL_MFIDENTITY_DEFAULT_MODE,
			bCmdBuff,
			0x0002,
			PH_CRYPTOSYM_PADDING_MODE_1,
			0x00,
			&bCmdBuff[2],
			wCmdLen - 2
			);
    }
	#endif /*NXPBUILD__PH_CRYPTOSYM */
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }
}

phStatus_t phalMfIdentity_Sw_ChangeFileSettingsSDM(phalMfIdentity_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bFileNo,
	uint8_t bFileOption, uint8_t *pAccessRights, uint8_t bSdmOptions, uint8_t *pSdmAccessRights, uint8_t *pPICCDataOffset,
	uint8_t *pSDMMACInputOffset, uint8_t *pSDMENCOffset, uint8_t *pSDMENCLen, uint8_t *pSDMMACOffset, uint8_t *pSDMReadCtrOffset,
	uint8_t *pTMCLimitValue)
{
	uint8_t     PH_MEMLOC_REM bCmdBuff[60];
    uint16_t    PH_MEMLOC_REM wCmdLen = 0;
	phStatus_t  PH_MEMLOC_REM statusTmp;

	bCmdBuff[wCmdLen++] = PHAL_MFIDENTITY_CMD_CHANGE_FILE_SETTINGS;
	bCmdBuff[wCmdLen++] = bFileNo;
	bCmdBuff[wCmdLen++] = bFileOption;
	if(pAccessRights != NULL)
	{
		bCmdBuff[wCmdLen++] = pAccessRights[0];
		bCmdBuff[wCmdLen++] = pAccessRights[1];
	}
	bCmdBuff[wCmdLen++] = bSdmOptions;
	if( pSdmAccessRights != NULL)
	{
		bCmdBuff[wCmdLen++] = pSdmAccessRights[0];
		bCmdBuff[wCmdLen++] = pSdmAccessRights[1];
	}

	if(pPICCDataOffset != NULL)
	{
		bCmdBuff[wCmdLen++] = pPICCDataOffset[0];
		bCmdBuff[wCmdLen++] = pPICCDataOffset[1];
		bCmdBuff[wCmdLen++] = pPICCDataOffset[2];
	}

	if(pSDMMACInputOffset != NULL)
	{
		bCmdBuff[wCmdLen++] = pSDMMACInputOffset[0];
		bCmdBuff[wCmdLen++] = pSDMMACInputOffset[1];
		bCmdBuff[wCmdLen++] = pSDMMACInputOffset[2];
	}

	if(pSDMENCOffset != NULL)
	{
		bCmdBuff[wCmdLen++] = pSDMENCOffset[0];
		bCmdBuff[wCmdLen++] = pSDMENCOffset[1];
		bCmdBuff[wCmdLen++] = pSDMENCOffset[2];
	}

	if(pSDMENCLen > 0)
	{
		bCmdBuff[wCmdLen++] = pSDMENCLen[0];
		bCmdBuff[wCmdLen++] = pSDMENCLen[1];
		bCmdBuff[wCmdLen++] = pSDMENCLen[2];
	}

	if(pSDMMACOffset != NULL)
	{
		bCmdBuff[wCmdLen++] = pSDMMACOffset[0];
		bCmdBuff[wCmdLen++] = pSDMMACOffset[1];
		bCmdBuff[wCmdLen++] = pSDMMACOffset[2];
	}

	if(pSDMReadCtrOffset != NULL)
	{
		bCmdBuff[wCmdLen++] = pSDMReadCtrOffset[0];
		bCmdBuff[wCmdLen++] = pSDMReadCtrOffset[1];
		bCmdBuff[wCmdLen++] = pSDMReadCtrOffset[2];
	}

	if(pTMCLimitValue != NULL)
	{
		if((((phCryptoSym_Sw_DataParams_t *)(pDataParams->pCryptoDataParamsMac))->bLrpEnabled != 0x01))
		{
			bCmdBuff[wCmdLen++] = pTMCLimitValue[0];
			bCmdBuff[wCmdLen++] = pTMCLimitValue[1];
			bCmdBuff[wCmdLen++] = pTMCLimitValue[2];
			bCmdBuff[wCmdLen++] = pTMCLimitValue[3];
		}
		else
		{
			bCmdBuff[wCmdLen++] = pTMCLimitValue[0];
			bCmdBuff[wCmdLen++] = pTMCLimitValue[1];
		}
	}

	if(((bOption & 0xF0U) == PHAL_MFIDENTITY_COMMUNICATION_PLAIN))
	{
		/* COMMUNICATION IS PLAIN */
		PH_CHECK_SUCCESS_FCT(statusTmp, phalMfIdentity_Sw_Int_Write_Plain(pDataParams,
		PHAL_MFIDENTITY_DEFAULT_MODE,
        bCmdBuff,
        wCmdLen,
        PHAL_MFIDENTITY_COMMUNICATION_PLAIN,
        NULL,
        0x0000
        ));
	}
#ifdef NXPBUILD__PH_CRYPTOSYM
	else if((bOption & 0xF0U) == PHAL_MFIDENTITY_COMMUNICATION_MACD)
	{
		/* COMMUNICATION IS Mac */
		PH_CHECK_SUCCESS_FCT(statusTmp,phalMfIdentity_Sw_Int_Write_Plain(pDataParams,
		PHAL_MFIDENTITY_DEFAULT_MODE,
        bCmdBuff,
        wCmdLen,
        PHAL_MFIDENTITY_COMMUNICATION_MACD,
        NULL,
        0x0000
        ));
	}
	else if((bOption & 0xF0U) == PHAL_MFIDENTITY_COMMUNICATION_ENC)
	{
		PH_CHECK_SUCCESS_FCT(statusTmp, phalMfIdentity_Sw_Int_Write_Enc(pDataParams,
			PHAL_MFIDENTITY_DEFAULT_MODE,
			bCmdBuff,
			0x0002,
			PH_CRYPTOSYM_PADDING_MODE_1,
			0x00,
			&bCmdBuff[2],
			wCmdLen - 2
			));
    }
#endif /* NXPBUILD__PH_CRYPTOSYM */
    else
    {
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }
	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}
#endif /* NXPBUILD__PH_CRYPTOSYM */




/* MIFARE Identity Data Management commands. ------------------------------------------------------------------------------------------- */
phStatus_t phalMfIdentity_Sw_ReadData(phalMfIdentity_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bIns, uint8_t bFileNo,
	uint8_t * pOffset, uint8_t * pLength, uint8_t ** ppRxdata, uint16_t * pRxdataLen)
{
    /* The signature of this is changed. We include
    * the bIns as third parameter that will differentiate
    * between application chaining and ISO chaining modes
    */
    phStatus_t  PH_MEMLOC_REM status = 0;
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint8_t     PH_MEMLOC_REM bCmdBuff[16];
    uint8_t     PH_MEMLOC_REM bTmp = 0;
    uint16_t    PH_MEMLOC_REM wCmdLen = 0;
    uint32_t    PH_MEMLOC_REM dwTMIStatus = 0;
    uint32_t    PH_MEMLOC_REM dwDataLen = 0;

	uint16_t    PH_MEMLOC_REM wRxlen;
	uint8_t     PH_MEMLOC_REM  bBuf[256];
	uint8_t     PH_MEMLOC_REM *pRecv = bBuf;

	uint32_t    PH_MEMLOC_REM dwEncFileData = 0;
    uint32_t    PH_MEMLOC_REM dwEncFileDataLength = 0;
	uint32_t    PH_MEMLOC_REM dwOffsetValue = 0;
	uint8_t     PH_MEMLOC_BUF  bEncbuf[256];
	uint8_t     PH_MEMLOC_BUF  bEncRespData[256];
	uint16_t    PH_MEMLOC_REM wAuthMode = 0x00;
	uint16_t    PH_MEMLOC_REM wEncFileMode = 0x00;

#ifdef RDR_LIB_PARAM_CHECK
	/* bit[1] of bIns will also be used. Hence bIns should be checked for above 0x03*/
    if (bIns > 0x03)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }
#endif
    if ((bOption & 0x0FU) == PH_EXCHANGE_RXCHAINING)
    {
        bCmdBuff[wCmdLen++] = PHAL_MFIDENTITY_RESP_ADDITIONAL_FRAME;
    }
    else if ((bOption & 0x0FU) == PH_EXCHANGE_DEFAULT)
    {
		/* copy data length */
		dwDataLen = pLength[2];
		dwDataLen = dwDataLen << 8 | pLength[1];
		dwDataLen = dwDataLen << 8 | pLength[0];

        if (bIns == 0x00)
        {
            bCmdBuff[wCmdLen++] = PHAL_MFIDENTITY_CMD_READ_DATA;
        }
        else
        {
        bCmdBuff[wCmdLen++] = PHAL_MFIDENTITY_CMD_READ_DATA_ISO;
        }

        bCmdBuff[wCmdLen++] = bFileNo;
        memcpy(&bCmdBuff[wCmdLen], pOffset, 3); /* PRQA S 3200 */
        wCmdLen += 3;
        memcpy(&bCmdBuff[wCmdLen], pLength, 3); /* PRQA S 3200 */
        wCmdLen += 3;

        PH_CHECK_SUCCESS_FCT(statusTmp, phTMIUtils_GetConfig((phTMIUtils_t *)pDataParams->pTMIDataParams,
            PH_TMIUTILS_TMI_STATUS,
            &dwTMIStatus));

        /* Check TMI Collection Status */
        if (dwTMIStatus == PH_ON)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phTMIUtils_CollectTMI((phTMIUtils_t *)pDataParams->pTMIDataParams,
                (uint8_t)((dwDataLen == 0) ? (PH_TMIUTILS_READ_INS | PH_TMIUTILS_ZEROPAD_CMDBUFF) : PH_TMIUTILS_ZEROPAD_CMDBUFF),
                bCmdBuff,
                wCmdLen,
                NULL,
                0,
                PHAL_MFIDENTITY_BLOCK_SIZE
                ));
        }
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }

    if ((bOption & 0xF0U) == PHAL_MFIDENTITY_COMMUNICATION_ENC)
    {
#ifdef NXPBUILD__PH_CRYPTOSYM
        /* Upload Payload size for proper CRC calculation */
        if ((bOption & 0x0FU) !=  PH_EXCHANGE_RXCHAINING)
        {
            pDataParams->dwPayLoadLen = dwDataLen;
        }

        status = phalMfIdentity_Sw_Int_ReadData_Enc(
            pDataParams,
            bOption | ((bIns == 0x00) ? PHAL_MFIDENTITY_DEFAULT_MODE : PHAL_MFIDENTITY_ISO_CHAINING_MODE),
            bCmdBuff,
            wCmdLen,
            ppRxdata,
            pRxdataLen
            );
#endif /* NXPBUILD__PH_CRYPTOSYM */
    }
    else if (((bOption & 0xF0U) == PHAL_MFIDENTITY_COMMUNICATION_MACD) ||
        ((bOption & 0xF0U) == PHAL_MFIDENTITY_COMMUNICATION_PLAIN))
    {
        status = phalMfIdentity_Sw_Int_ReadData_Plain(
            pDataParams,
            bOption | ((bIns == 0x00) ? PHAL_MFIDENTITY_DEFAULT_MODE : PHAL_MFIDENTITY_ISO_CHAINING_MODE),
            bCmdBuff,
            wCmdLen,
            ppRxdata,
            pRxdataLen
            );
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }

    /* Check TMI Collection Status */
    if (dwTMIStatus == PH_ON)
    {
        if (dwDataLen == 0)
        {
            bTmp = PH_TMIUTILS_READ_INS;
        }
        if (status == PH_ERR_SUCCESS)
        {
            bTmp |= PH_TMIUTILS_ZEROPAD_DATABUFF;
        }

		/* Copy Original Data */
		memcpy(pRecv,*ppRxdata,*pRxdataLen);
		wRxlen = *pRxdataLen;

		if(wRxlen > 0x00)
		{
			/* Extract SDMENCFILEDATA to check enabled or disabled */
			if(pDataParams->bSDMOption & 0x10)
				wEncFileMode = 0x01;
			else
				wEncFileMode = 0x00;

			/* To get the AuthMode whether EV2 or LRP*/
			PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_GetConfig(pDataParams->pCryptoDataParamsMac,PH_CRYPTOSYM_CONFIG_LRP, &wAuthMode));

			if (bFileNo == PHAL_MFIDENTITY_SDM_FILE_ID)
			{
				if( pDataParams->bFileOption & 0x40 )
				{
					if((pDataParams->bSDMOption == 0x00) || (wEncFileMode == 0x00) )
					{
						if(wAuthMode == 0x00)
						{
							/* EV2 Data */
							memcpy(pRecv,pRecv+16,wRxlen-24);	/* PRQA S 3200 */
							wRxlen = wRxlen -24;
						}
						else
						{
							/* LRP Data */
							memcpy(pRecv,pRecv+24,wRxlen-24-8);	/* PRQA S 3200 */
							wRxlen = wRxlen -24-8;
						}
					}
					else
					{
						/* copy SDMENCfile offset and Length */
						dwEncFileData = pDataParams->aSDMENCOffset[2];
						dwEncFileData = dwEncFileData << 8 | pDataParams->aSDMENCOffset[1];
						dwEncFileData = dwEncFileData << 8 | pDataParams->aSDMENCOffset[0];

						dwEncFileDataLength = pDataParams->aSDMENCLength[2];
						dwEncFileDataLength = dwEncFileDataLength << 8 | pDataParams->aSDMENCLength[1];
						dwEncFileDataLength = dwEncFileDataLength << 8 | pDataParams->aSDMENCLength[0];

						dwOffsetValue = pOffset[2];
						dwOffsetValue = dwOffsetValue << 8 | pOffset[1];
						dwOffsetValue = dwOffsetValue << 8 | pOffset[0];

						/* Restore back Original Data */
						memcpy(*ppRxdata,pRecv,wRxlen);		/* PRQA S 3200 */
						*pRxdataLen = wRxlen;

						if(dwOffsetValue == 0x000000)
						{
							if(wAuthMode == 0x00)
								memcpy(bEncbuf,pRecv+dwEncFileData+PHAL_MFIDENTITY_PICC_DATA_EV2,dwEncFileDataLength);	/* PRQA S 3200 */
							else
								memcpy(bEncbuf,pRecv+dwEncFileData+PHAL_MFIDENTITY_PICC_DATA_LRP,dwEncFileDataLength);	/* PRQA S 3200 */

							/* Decrypt the encrypted file data. */
							phalMfIdentity_Sw_DecryptSDMENCFileData (
								pDataParams,
								PHAL_MFIDENTITY_NO_DIVERSIFICATION,
								pDataParams->bSDMKeyNo,
								pDataParams->bSDMKeyVer,
								NULL,
								0,
								pDataParams->bUid,
								pDataParams->bUidLength,
								pDataParams->aSDMReadCTR,
								bEncbuf,
								(uint16_t) dwEncFileDataLength,
								bEncRespData);

							if(wAuthMode == 0x00)
							{
								memcpy(pRecv+dwEncFileData+PHAL_MFIDENTITY_PICC_DATA_EV2,bEncRespData,dwEncFileDataLength);	/* PRQA S 3200 */
								wRxlen = wRxlen - PHAL_MFIDENTITY_PICC_DATA_EV2 - PHAL_MFIDENTITY_MAC_DATA;
								memcpy(pRecv,pRecv+PHAL_MFIDENTITY_PICC_DATA_EV2,wRxlen);	/* PRQA S 3200 */
							}
							else
							{
								memcpy(pRecv+dwEncFileData+PHAL_MFIDENTITY_PICC_DATA_LRP,bEncRespData,dwEncFileDataLength);	/* PRQA S 3200 */
								wRxlen = wRxlen - PHAL_MFIDENTITY_PICC_DATA_LRP - PHAL_MFIDENTITY_MAC_DATA;
								memcpy(pRecv,pRecv+PHAL_MFIDENTITY_PICC_DATA_LRP,wRxlen);	/* PRQA S 3200 */
							}
						}
						else if(dwOffsetValue < dwEncFileData)
						{
							if(wAuthMode == 0x00)
								memcpy(bEncbuf,pRecv+dwEncFileData+PHAL_MFIDENTITY_PICC_DATA_EV2-dwOffsetValue,dwEncFileDataLength);	/* PRQA S 3200 */
							else
								memcpy(bEncbuf,pRecv+dwEncFileData+PHAL_MFIDENTITY_PICC_DATA_LRP-dwOffsetValue,dwEncFileDataLength);	/* PRQA S 3200 */

							phalMfIdentity_Sw_DecryptSDMENCFileData (
								pDataParams,
								PHAL_MFIDENTITY_NO_DIVERSIFICATION,
								pDataParams->bSDMKeyNo,
								pDataParams->bSDMKeyVer,
								NULL,
								0,
								pDataParams->bUid,
								pDataParams->bUidLength,
								pDataParams->aSDMReadCTR,
								bEncbuf,
								(uint16_t) dwEncFileDataLength,
								bEncRespData);

							if(wAuthMode == 0x00)
							{
								memcpy(pRecv+dwEncFileData+PHAL_MFIDENTITY_PICC_DATA_EV2-dwOffsetValue,bEncRespData,dwEncFileDataLength);
								wRxlen = wRxlen - PHAL_MFIDENTITY_PICC_DATA_EV2 - PHAL_MFIDENTITY_MAC_DATA;
								memcpy(pRecv,pRecv+PHAL_MFIDENTITY_PICC_DATA_EV2,wRxlen);	/* PRQA S 3200 */
							}
							else
							{
								memcpy(pRecv+dwEncFileData+PHAL_MFIDENTITY_PICC_DATA_LRP-dwOffsetValue,bEncRespData,dwEncFileDataLength);
								wRxlen = wRxlen - PHAL_MFIDENTITY_PICC_DATA_LRP - PHAL_MFIDENTITY_MAC_DATA;
								memcpy(pRecv,pRecv+PHAL_MFIDENTITY_PICC_DATA_LRP,wRxlen);	/* PRQA S 3200 */
							}
						}
					}
				}
			}
		}

		PH_CHECK_SUCCESS_FCT(statusTmp, phTMIUtils_CollectTMI((phTMIUtils_t *)pDataParams->pTMIDataParams,
            bTmp,
            NULL,
            0,
            pRecv,
            wRxlen,
            PHAL_MFIDENTITY_BLOCK_SIZE
            ));

        if ((status == PH_ERR_SUCCESS) && (dwDataLen == 0))
        {
            /* Reset wOffsetInTMI to 0 */
            PH_CHECK_SUCCESS_FCT(statusTmp, phTMIUtils_SetConfig((phTMIUtils_t *)pDataParams->pTMIDataParams,
                PH_TMIUTILS_TMI_OFFSET_LENGTH,
                0
                ));
        }
    }
    return status;
}

phStatus_t phalMfIdentity_Sw_WriteData(phalMfIdentity_Sw_DataParams_t * pDataParams, uint8_t bCommOption, uint8_t bIns, uint8_t bFileNo,
	uint8_t * pOffset, uint8_t * pData, uint8_t * pDataLen)
{
    /* The signature of this is changed. We include
    * the bIns as third parameter that will differentiate
    * between application chaining and ISO chaining modes
    */
    phStatus_t  PH_MEMLOC_REM statusTmp;
	uint8_t     PH_MEMLOC_REM bCommOptionTemp = bCommOption;
    uint8_t     PH_MEMLOC_REM bLoopData = 1;
#ifdef NXPBUILD__PH_CRYPTOSYM
    uint8_t     PH_MEMLOC_REM bIvLen = PH_CRYPTOSYM_DES_BLOCK_SIZE;
#endif /*NXPBUILD__PH_CRYPTOSYM */
    uint8_t     PH_MEMLOC_REM bCmdBuff[16];
    uint16_t    PH_MEMLOC_REM wCmdLen = 0;
    uint16_t    PH_MEMLOC_REM wDataLenTemp;
    uint32_t    PH_MEMLOC_REM dwDataLen;
    uint32_t    PH_MEMLOC_REM dwTMIStatus = 0;
    uint32_t    PH_MEMLOC_REM dwDataWritten = 0;

#ifdef RDR_LIB_PARAM_CHECK
    if (bIns > 0x01)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }
    if ((bCommOption != PHAL_MFIDENTITY_COMMUNICATION_PLAIN) &&
        (bCommOption != PHAL_MFIDENTITY_COMMUNICATION_ENC) &&
        (bCommOption != PHAL_MFIDENTITY_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }
#endif
    /* form the command depending on bIns */
    if (bIns == 0x00)
    {
        bCmdBuff[wCmdLen++] = PHAL_MFIDENTITY_CMD_WRITE_DATA;
    }
    else
    {
    bCmdBuff[wCmdLen++] = PHAL_MFIDENTITY_CMD_WRITE_DATA_ISO;
    }
    bCmdBuff[wCmdLen++] = bFileNo;
    memcpy(&bCmdBuff[wCmdLen], pOffset, 3); /* PRQA S 3200 */
    wCmdLen += 3;
    memcpy(&bCmdBuff[wCmdLen], pDataLen, 3); /* PRQA S 3200 */
    wCmdLen += 3;

	/* copy data length */
	dwDataLen = pDataLen[2];
	dwDataLen = dwDataLen << 8 | pDataLen[1];
	dwDataLen = dwDataLen << 8 | pDataLen[0];

	/* to handle 2 MB of data update maximum of data bytes that can be sent in a single loop */
	if(dwDataLen > PHAL_MFIDENTITY_MAX_WRITE_SIZE)
	{
		wDataLenTemp =(uint16_t) PHAL_MFIDENTITY_MAX_WRITE_SIZE;
		bLoopData = (uint8_t)(dwDataLen/PHAL_MFIDENTITY_MAX_WRITE_SIZE);
		if(dwDataLen % PHAL_MFIDENTITY_MAX_WRITE_SIZE)
		{
			bLoopData++;
		}
		bCommOptionTemp = PHAL_MFIDENTITY_MAC_DATA_INCOMPLETE |bCommOption;
	}
	else
	{
		wDataLenTemp = (uint16_t)dwDataLen;
	}

	/* update the total number of data to be written
	 * when INS and wrapped mode is enable, we need to send the total number of bytes with Le */
	pDataParams->dwPayLoadLen = dwDataLen;

    if (pDataParams->bAuthMode == PHAL_MFIDENTITY_NOT_AUTHENTICATED)
    {
        bCommOption = PHAL_MFIDENTITY_COMMUNICATION_PLAIN;
    }
	#ifdef NXPBUILD__PH_CRYPTOSYM
	else
	{
		if((bCommOption & 0xF0U) == PHAL_MFIDENTITY_COMMUNICATION_MACD)
		{
			pDataParams->dwPayLoadLen = pDataParams->dwPayLoadLen + ((pDataParams->bAuthMode == PHAL_MFIDENTITY_AUTHENTICATE) ? 0x04 : 0x08);
		}
		if((bCommOption & 0xF0U) == PHAL_MFIDENTITY_COMMUNICATION_ENC)
		{
			if(pDataParams->bAuthMode == PHAL_MFIDENTITY_AUTHENTICATEEV2)
			{
				pDataParams->dwPayLoadLen = 8 + pDataParams->dwPayLoadLen + ((pDataParams->dwPayLoadLen % PH_CRYPTOSYM_AES_BLOCK_SIZE)?
																		(PH_CRYPTOSYM_AES_BLOCK_SIZE - (pDataParams->dwPayLoadLen % PH_CRYPTOSYM_AES_BLOCK_SIZE)) : PH_CRYPTOSYM_AES_BLOCK_SIZE) ;
			}
			else
			{
				pDataParams->dwPayLoadLen  = pDataParams->dwPayLoadLen  + ((pDataParams->bAuthMode == PHAL_MFIDENTITY_AUTHENTICATE) ? 0x02 : 0x04);
				bIvLen = (pDataParams->bAuthMode == PHAL_MFIDENTITY_AUTHENTICATEAES) ? PH_CRYPTOSYM_AES_BLOCK_SIZE : PH_CRYPTOSYM_DES_BLOCK_SIZE;

				if(pDataParams->dwPayLoadLen % bIvLen)
				{
					pDataParams->dwPayLoadLen = pDataParams->dwPayLoadLen +  (bIvLen - (pDataParams->dwPayLoadLen % bIvLen));
				}
			}
		}
	}
	#endif /*NXPBUILD__PH_CRYPTOSYM */
    PH_CHECK_SUCCESS_FCT(statusTmp, phTMIUtils_GetConfig((phTMIUtils_t *)pDataParams->pTMIDataParams,
        PH_TMIUTILS_TMI_STATUS,
        &dwTMIStatus));

    /* Check TMI Collection Status */
    if (dwTMIStatus == PH_ON)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phTMIUtils_CollectTMI((phTMIUtils_t *)pDataParams->pTMIDataParams,
            (PH_TMIUTILS_ZEROPAD_CMDBUFF | PH_TMIUTILS_ZEROPAD_DATABUFF),
            bCmdBuff,
            wCmdLen,
            pData,
            dwDataLen,
            PHAL_MFIDENTITY_BLOCK_SIZE
            ));
    }

	do
	{
		if (bLoopData == 1)
		{
			bCommOptionTemp = bCommOption;
			wDataLenTemp = (uint16_t)(dwDataLen - dwDataWritten);
		}
		if ((bCommOption & 0xF0U) == PHAL_MFIDENTITY_COMMUNICATION_ENC)
		{
#ifdef NXPBUILD__PH_CRYPTOSYM
			statusTmp =  phalMfIdentity_Sw_Int_Write_Enc(
				pDataParams,
				((bIns == 0x00) ? PHAL_MFIDENTITY_DEFAULT_MODE : PHAL_MFIDENTITY_ISO_CHAINING_MODE),
				bCmdBuff,
				wCmdLen,
				PH_CRYPTOSYM_PADDING_MODE_1,
				bCommOptionTemp,
				&pData[dwDataWritten],
				wDataLenTemp
				);
#endif /*NXPBUILD__PH_CRYPTOSYM */
		}
		else
		{
			statusTmp =  phalMfIdentity_Sw_Int_Write_Plain(
				pDataParams,
				((bIns == 0x00) ? PHAL_MFIDENTITY_DEFAULT_MODE : PHAL_MFIDENTITY_ISO_CHAINING_MODE),
				bCmdBuff,
				wCmdLen,
				bCommOptionTemp,
				&pData[dwDataWritten],
				wDataLenTemp
				);
		}

		if((statusTmp & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING)
		{
			return statusTmp;
		}

		bLoopData--;
		dwDataWritten +=wDataLenTemp;

		bCmdBuff[0] = 0xAF;
		wCmdLen = 1;

	}while(bLoopData > 0);

	return statusTmp;

}

phStatus_t phalMfIdentity_Sw_ReadRecords(phalMfIdentity_Sw_DataParams_t * pDataParams, uint8_t bCommOption, uint8_t bIns, uint8_t bFileNo,
	uint8_t * pRecNo, uint8_t * pRecCount, uint8_t * pRecSize, uint8_t ** ppRxdata, uint16_t * pRxdataLen)
{
	uint8_t     PH_MEMLOC_REM bCmdBuff[8];
	uint16_t    PH_MEMLOC_REM wCmdLen = 0;
	uint32_t    PH_MEMLOC_REM dwRecLen = 0;
	uint32_t    PH_MEMLOC_REM dwNumRec = 0;
	phStatus_t  PH_MEMLOC_REM status = 0;
	uint32_t    PH_MEMLOC_REM dwTMIStatus = 0;
	phStatus_t  PH_MEMLOC_REM statusTmp;
	uint8_t     PH_MEMLOC_REM bOption = 0;
	uint32_t    PH_MEMLOC_REM dwDataLen = 0;
	uint32_t    PH_MEMLOC_REM dwOffsetTMI = 0;
	uint32_t    PH_MEMLOC_REM dwTMIBufInd = 0;
	uint32_t    PH_MEMLOC_REM dwTotalRecLen = 0;
	uint32_t    PH_MEMLOC_REM dwNumRecCal = 0;

#ifdef RDR_LIB_PARAM_CHECK
	if(bIns > 0x03)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
	}
#endif
	/* copy size of each record */
	dwRecLen = pRecSize[2];/* MSB */
	dwRecLen = dwRecLen << 8 | pRecSize[1];
	dwRecLen = dwRecLen << 8 | pRecSize[0];/* LSB */

										   /* copy number of records to be read */
	dwNumRec = pRecCount[2];/* MSB */
	dwNumRec = dwNumRec << 8 | pRecCount[1];
	dwNumRec = dwNumRec << 8 | pRecCount[0];/* LSB */

	if((bCommOption & 0x0FU) == PH_EXCHANGE_RXCHAINING)
	{
		bCmdBuff[wCmdLen++] = PHAL_MFIDENTITY_RESP_ADDITIONAL_FRAME;
	}
	else if((bCommOption & 0x0FU) == PH_EXCHANGE_DEFAULT)
	{
		/* form the command depending upon bIns */
		if(bIns == 0x00)
		{
			bCmdBuff[wCmdLen++] = PHAL_MFIDENTITY_CMD_READ_RECORDS;
		}
		else
		{
			bCmdBuff[wCmdLen++] = PHAL_MFIDENTITY_CMD_READ_RECORDS_ISO;
		}
		bCmdBuff[wCmdLen++] = bFileNo;

		/* Record No */
		memcpy(&bCmdBuff[wCmdLen], pRecNo, 3); /* PRQA S 3200 */
		wCmdLen += 3;

		/* Record Count */
		memcpy(&bCmdBuff[wCmdLen], pRecCount, 3); /* PRQA S 3200 */
		wCmdLen += 3;

		/* Total number of bytes to read */
		dwTotalRecLen = (uint32_t) dwRecLen * dwNumRec;

		PH_CHECK_SUCCESS_FCT(statusTmp, phTMIUtils_GetConfig((phTMIUtils_t *) pDataParams->pTMIDataParams,
			PH_TMIUTILS_TMI_STATUS,
			&dwTMIStatus));

		/* Check TMI Collection Status */
		if(dwTMIStatus == PH_ON)
		{
			/* Should should provide atleast wRecLen / wNumRec to update in TIM collection */
			if((0 == dwRecLen) && (0 == dwNumRec))
			{
				return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
			}
			PH_CHECK_SUCCESS_FCT(statusTmp, phTMIUtils_CollectTMI((phTMIUtils_t *) pDataParams->pTMIDataParams,
				(uint8_t) ((dwNumRec == 0) ? (PH_TMIUTILS_READ_INS | PH_TMIUTILS_ZEROPAD_CMDBUFF) : PH_TMIUTILS_ZEROPAD_CMDBUFF),
				bCmdBuff,
				wCmdLen,
				NULL,
				0,
				PHAL_MFIDENTITY_BLOCK_SIZE
			));
		}
	}
	else
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
	}

	if((bCommOption & 0xF0U) == PHAL_MFIDENTITY_COMMUNICATION_ENC)
	{
#ifdef NXPBUILD__PH_CRYPTOSYM
		/* Upload Payload size for proper CRC calculation */
		if((bCommOption & 0x0FU) != PH_EXCHANGE_RXCHAINING)
		{
			pDataParams->dwPayLoadLen = dwTotalRecLen;
		}

		status = phalMfIdentity_Sw_Int_ReadData_Enc(
			pDataParams,
			bCommOption | ((bIns == 0x00) ? PHAL_MFIDENTITY_DEFAULT_MODE : PHAL_MFIDENTITY_ISO_CHAINING_MODE),
			bCmdBuff,
			wCmdLen,
			ppRxdata,
			pRxdataLen
		);
#endif /* NXPBUILD__PH_CRYPTOSYM */
	}
	else if(((bCommOption & 0xF0U) == PHAL_MFIDENTITY_COMMUNICATION_PLAIN) ||
		((bCommOption & 0xF0U) == PHAL_MFIDENTITY_COMMUNICATION_MACD))
	{

		if((pDataParams->bAuthMode == PHAL_MFIDENTITY_AUTHENTICATE) && ((bCommOption & 0xF0U) == PHAL_MFIDENTITY_COMMUNICATION_MACD))
		{
			bCommOption = bCommOption | PHAL_MFIDENTITY_COMMUNICATION_MAC_ON_RC;
		}

		status = phalMfIdentity_Sw_Int_ReadData_Plain(
			pDataParams,
			bCommOption | ((bIns == 0x00) ? PHAL_MFIDENTITY_DEFAULT_MODE : PHAL_MFIDENTITY_ISO_CHAINING_MODE),
			bCmdBuff,
			wCmdLen,
			ppRxdata,
			pRxdataLen
		);
	}
	else
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
	}

	if(((status & PH_ERR_MASK) != PH_ERR_SUCCESS) &&
		((status & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
	{
		return status;
	}

	if((status == PH_ERR_SUCCESS) && (bCmdBuff[0] != PHAL_MFIDENTITY_RESP_ADDITIONAL_FRAME))
	{
		/* Can check this case if user has not given num of records
		as 0x000000. If 0x000000, then all records are read */
		if((dwTotalRecLen != *pRxdataLen) && (dwTotalRecLen != 0))
		{
			/* Reset authentication status */
			if((pDataParams->bAuthMode == PHAL_MFIDENTITY_AUTHENTICATEISO) ||
				(pDataParams->bAuthMode == PHAL_MFIDENTITY_AUTHENTICATEAES))
			{
				phalMfIdentity_Sw_Int_ResetAuthStatus(pDataParams);
			}
			return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFIDENTITY);
		}
	}

	/* if function called with PH_EXCHANGE_RXCHAINING */
	PH_CHECK_SUCCESS_FCT(statusTmp, phTMIUtils_GetConfig((phTMIUtils_t *) pDataParams->pTMIDataParams,
		PH_TMIUTILS_TMI_STATUS,
		&dwTMIStatus));

	/* Check TMI Collection Status */
	if(dwTMIStatus == PH_ON)
	{
		if((dwNumRec == 0) && (status == PH_ERR_SUCCESS))
		{
			PH_CHECK_SUCCESS_FCT(statusTmp, phTMIUtils_GetConfig((phTMIUtils_t *) pDataParams->pTMIDataParams,
				PH_TMIUTILS_TMI_OFFSET_LENGTH,
				&dwOffsetTMI
			));
			PH_CHECK_SUCCESS_FCT(statusTmp, phTMIUtils_GetConfig((phTMIUtils_t *) pDataParams->pTMIDataParams,
				PH_TMIUTILS_TMI_BUFFER_INDEX,
				&dwTMIBufInd
			));

			/* calculate Rx length in case of chaining */
			dwDataLen = *pRxdataLen + dwTMIBufInd - (dwOffsetTMI + 11);

			/* for Qmore compliance below code is added check is done before itself  */
			if(dwRecLen == 0)
			{
				return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFIDENTITY);
			}

			/* if user update worng RecSize, we cant calculate recCnt */
			if(dwDataLen % dwRecLen)
			{
				return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFIDENTITY);
			}

			/* calculate number of records */
			dwNumRecCal = dwDataLen / dwRecLen;

			/* update record count */
			PH_CHECK_SUCCESS_FCT(statusTmp, phTMIUtils_SetConfig((phTMIUtils_t *) pDataParams->pTMIDataParams,
				PH_TMIUTILS_TMI_OFFSET_VALUE,
				dwNumRecCal
			));

		}
		if(status == PH_ERR_SUCCESS)
		{
			bOption = PH_TMIUTILS_ZEROPAD_DATABUFF;
		}

		PH_CHECK_SUCCESS_FCT(statusTmp, phTMIUtils_CollectTMI((phTMIUtils_t *) pDataParams->pTMIDataParams,
			bOption,
			NULL,
			0,
			*ppRxdata,
			*pRxdataLen,
			PHAL_MFIDENTITY_BLOCK_SIZE
		));

		if((status == PH_ERR_SUCCESS) && (dwNumRec == 0))
		{
			/* Reset wOffsetInTMI to 0 */
			PH_CHECK_SUCCESS_FCT(statusTmp, phTMIUtils_SetConfig((phTMIUtils_t *) pDataParams->pTMIDataParams,
				PH_TMIUTILS_TMI_OFFSET_LENGTH,
				0
			));
		}
	}
	return status;
}

phStatus_t phalMfIdentity_Sw_WriteRecord(phalMfIdentity_Sw_DataParams_t * pDataParams, uint8_t bCommOption, uint8_t bIns, uint8_t bFileNo,
	uint8_t * pOffset, uint8_t * pData, uint8_t * pDataLen)
{
    phStatus_t  PH_MEMLOC_REM statusTmp;
	uint8_t     PH_MEMLOC_REM bCmdBuff[8];
	uint8_t     PH_MEMLOC_REM bCommOptionTemp = bCommOption;
    uint8_t     PH_MEMLOC_REM bLoopData = 1;
#ifdef NXPBUILD__PH_CRYPTOSYM
    uint8_t     PH_MEMLOC_REM bIvLen = PH_CRYPTOSYM_DES_BLOCK_SIZE;
#endif /*NXPBUILD__PH_CRYPTOSYM */
    uint16_t    PH_MEMLOC_REM wDataLenTemp;
    uint16_t    PH_MEMLOC_REM wCmdLen = 0;
    uint32_t    PH_MEMLOC_REM dwDataLen;
    uint32_t    PH_MEMLOC_REM dwTMIStatus = 0;
    uint32_t    PH_MEMLOC_REM dwDataWritten = 0;

#ifdef RDR_LIB_PARAM_CHECK
    if (bIns > 0x01)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }
    if ((bCommOption != PHAL_MFIDENTITY_COMMUNICATION_PLAIN) &&
        (bCommOption != PHAL_MFIDENTITY_COMMUNICATION_ENC) &&
        (bCommOption != PHAL_MFIDENTITY_COMMUNICATION_MACD))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }
#endif
    /* form the command depending on bIns */
    if (bIns == 0x00)
    {
        bCmdBuff[wCmdLen++] = PHAL_MFIDENTITY_CMD_WRITE_RECORD;
    }
    else
    {
    bCmdBuff[wCmdLen++] = PHAL_MFIDENTITY_CMD_WRITE_RECORD_ISO;
    }
    bCmdBuff[wCmdLen++] = bFileNo;
    memcpy(&bCmdBuff[wCmdLen], pOffset, 3); /* PRQA S 3200 */
    wCmdLen += 3;
    memcpy(&bCmdBuff[wCmdLen], pDataLen, 3); /* PRQA S 3200 */
    wCmdLen += 3;

    /* Assuming here that the size can never go beyond FFFF. */
	dwDataLen = pDataLen[2];
	dwDataLen = dwDataLen << 8 | pDataLen[1];
	dwDataLen = dwDataLen << 8 | pDataLen[0];

	/* to handle 2 MB of data update maximum of data bytes that can be sent in a single loop */
	if(dwDataLen > PHAL_MFIDENTITY_MAX_WRITE_SIZE)
	{
		wDataLenTemp = (uint16_t)PHAL_MFIDENTITY_MAX_WRITE_SIZE;
		bLoopData = (uint8_t)(dwDataLen/PHAL_MFIDENTITY_MAX_WRITE_SIZE);
		if(dwDataLen % PHAL_MFIDENTITY_MAX_WRITE_SIZE)
		{
			bLoopData++;
		}
		bCommOptionTemp = PHAL_MFIDENTITY_MAC_DATA_INCOMPLETE |bCommOption;
	}
	else
	{
		wDataLenTemp = (uint16_t)dwDataLen;
	}


	/* update the total number of data to be written
	 * when INS and wrapped mode is enable, we need to send the total number of bytes with Le */
	pDataParams->dwPayLoadLen = dwDataLen;

    if (pDataParams->bAuthMode == PHAL_MFIDENTITY_NOT_AUTHENTICATED)
    {
        bCommOption = PHAL_MFIDENTITY_COMMUNICATION_PLAIN;
    }
#ifdef NXPBUILD__PH_CRYPTOSYM
	else
	{
		if((bCommOption & 0xF0U) == PHAL_MFIDENTITY_COMMUNICATION_MACD)
		{
			pDataParams->dwPayLoadLen = pDataParams->dwPayLoadLen + ((pDataParams->bAuthMode == PHAL_MFIDENTITY_AUTHENTICATE) ? 0x04 : 0x08);
		}
		if((bCommOption & 0xF0U) == PHAL_MFIDENTITY_COMMUNICATION_ENC)
		{
			if(pDataParams->bAuthMode == PHAL_MFIDENTITY_AUTHENTICATEEV2)
			{
				pDataParams->dwPayLoadLen = 8 + pDataParams->dwPayLoadLen + ((pDataParams->dwPayLoadLen % PH_CRYPTOSYM_AES_BLOCK_SIZE)?
																		(PH_CRYPTOSYM_AES_BLOCK_SIZE - (pDataParams->dwPayLoadLen % PH_CRYPTOSYM_AES_BLOCK_SIZE)) : PH_CRYPTOSYM_AES_BLOCK_SIZE) ;
			}
			else
			{
				pDataParams->dwPayLoadLen  = pDataParams->dwPayLoadLen  + ((pDataParams->bAuthMode == PHAL_MFIDENTITY_AUTHENTICATE) ? 0x02 : 0x04);
				bIvLen = (pDataParams->bAuthMode == PHAL_MFIDENTITY_AUTHENTICATEAES) ? PH_CRYPTOSYM_AES_BLOCK_SIZE : PH_CRYPTOSYM_DES_BLOCK_SIZE;

				if(pDataParams->dwPayLoadLen % bIvLen)
				{
					pDataParams->dwPayLoadLen = pDataParams->dwPayLoadLen +  (bIvLen - (pDataParams->dwPayLoadLen % bIvLen));
				}
			}
		}
	}
#endif /*NXPBUILD__PH_CRYPTOSYM */
    PH_CHECK_SUCCESS_FCT(statusTmp, phTMIUtils_GetConfig((phTMIUtils_t *)pDataParams->pTMIDataParams,
        PH_TMIUTILS_TMI_STATUS,
        &dwTMIStatus));

    /* Check TMI Collection Status */
    if (dwTMIStatus == PH_ON)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phTMIUtils_CollectTMI((phTMIUtils_t *)pDataParams->pTMIDataParams,
            (PH_TMIUTILS_ZEROPAD_CMDBUFF | PH_TMIUTILS_ZEROPAD_DATABUFF),
            bCmdBuff,
            wCmdLen,
            pData,
            dwDataLen,
            PHAL_MFIDENTITY_BLOCK_SIZE
            ));
    }

	do
	{
		if (bLoopData == 1)
		{
			bCommOptionTemp = bCommOption;
			wDataLenTemp = (uint16_t)(dwDataLen - dwDataWritten);
		}

		if ((pDataParams->bAuthMode == PHAL_MFIDENTITY_NOT_AUTHENTICATED)||
			(bCommOption == PHAL_MFIDENTITY_COMMUNICATION_MACD) ||
			(bCommOption == PHAL_MFIDENTITY_COMMUNICATION_PLAIN))
		{
			statusTmp =  phalMfIdentity_Sw_Int_Write_Plain(
				pDataParams,
				((bIns == 0x00) ? PHAL_MFIDENTITY_DEFAULT_MODE : PHAL_MFIDENTITY_ISO_CHAINING_MODE),
				bCmdBuff,
				wCmdLen,
				bCommOptionTemp,
				&pData[dwDataWritten],
				wDataLenTemp
				);

		}
		#ifdef NXPBUILD__PH_CRYPTOSYM
		else
		{
			statusTmp =  phalMfIdentity_Sw_Int_Write_Enc(
				pDataParams,
				((bIns == 0x00) ? PHAL_MFIDENTITY_DEFAULT_MODE : PHAL_MFIDENTITY_ISO_CHAINING_MODE),
				bCmdBuff,
				wCmdLen,
				PH_CRYPTOSYM_PADDING_MODE_1,
				bCommOptionTemp,
				&pData[dwDataWritten],
				wDataLenTemp
				);
		}
		#endif /*NXPBUILD__PH_CRYPTOSYM */
		if((statusTmp & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING)
		{
			return statusTmp;
		}

		bLoopData--;
		dwDataWritten +=wDataLenTemp;

		bCmdBuff[0] = 0xAF;
		wCmdLen = 1;

	}while(bLoopData > 0);

	return statusTmp;
}

phStatus_t phalMfIdentity_Sw_ClearRecordFile(phalMfIdentity_Sw_DataParams_t * pDataParams, uint8_t bFileNo)
{
    uint8_t PH_MEMLOC_REM bCmdBuff[8];
    uint32_t    PH_MEMLOC_REM dwTMIStatus = 0;
    phStatus_t  PH_MEMLOC_REM statusTmp;

    /* form the command */
    bCmdBuff[0] = PHAL_MFIDENTITY_CMD_CLEAR_RECORD_FILE;
    bCmdBuff[1] = bFileNo;

    PH_CHECK_SUCCESS_FCT(statusTmp, phTMIUtils_GetConfig((phTMIUtils_t *)pDataParams->pTMIDataParams,
        PH_TMIUTILS_TMI_STATUS,
        &dwTMIStatus));

    /* Check TMI Collection Status */
    if (dwTMIStatus == PH_ON)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phTMIUtils_CollectTMI((phTMIUtils_t *)pDataParams->pTMIDataParams,
            PH_TMIUTILS_ZEROPAD_CMDBUFF,
            bCmdBuff,
            2,
            NULL,
            0,
            PHAL_MFIDENTITY_BLOCK_SIZE
            ));
    }

    return phalMfIdentity_Sw_Int_Write_Plain(pDataParams,
		PHAL_MFIDENTITY_DEFAULT_MODE,
        bCmdBuff,
        2,
        (pDataParams->bAuthMode == PHAL_MFIDENTITY_AUTHENTICATEEV2)? PHAL_MFIDENTITY_COMMUNICATION_MACD : PHAL_MFIDENTITY_COMMUNICATION_PLAIN,
        NULL,
        0x0000
        );
}




/* MIFARE Identity Transaction Management commands. ------------------------------------------------------------------------------------ */
phStatus_t phalMfIdentity_Sw_CommitTransaction(phalMfIdentity_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t * pTMC, uint8_t * pTMAC)
{
    uint16_t    PH_MEMLOC_REM statusTmp;
    uint8_t     PH_MEMLOC_REM bCmdBuff[8];
    uint16_t    PH_MEMLOC_REM wCmdLen = 0;
    uint8_t     PH_MEMLOC_REM *pRecv = NULL;
    uint16_t    PH_MEMLOC_REM wRxlen = 0;

#ifdef RDR_LIB_PARAM_CHECK
    /* As per ref_arch 0.04 for Cmd.CommitTransaction: simplified to always
	 * use CommMode.MAC, so communication mode of response does not depend
	 * on File-Type.TransactionMAC anymore.
     */
    if ((bOption & 0x0F) > 0x01)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }
#endif
    /* form the command */
    bCmdBuff[wCmdLen++] = PHAL_MFIDENTITY_CMD_COMMIT_TXN;

    /*
	 * 0x01 and 0x00 are values to support the legacy implementation.
	 *		Here if it 0x00 the Option is not exchanged to the PICC. Its only exchanged for 0x01
	 *
	 * 0x80 and 0x81 are the values to exchange the Option by to PICC.
	 *		If its 0x80, Option byte will be exchanged by the Option information will be zero.
	 *		If its 0x81, Option byte will be exchanged by the Option information will be one.
	 */
	if((bOption & 0x80) || (bOption & 0x0F))
    {
        bCmdBuff[wCmdLen++] = (bOption & 0x0F);
    }

    PH_CHECK_SUCCESS_FCT(statusTmp, phalMfIdentity_Sw_Int_ReadData_Plain(
        pDataParams,
        (pDataParams->bAuthMode == PHAL_MFIDENTITY_AUTHENTICATEEV2)? PHAL_MFIDENTITY_COMMUNICATION_MACD : PHAL_MFIDENTITY_COMMUNICATION_PLAIN,
        bCmdBuff,
        wCmdLen,
        &pRecv,
        &wRxlen
        ));


    if (((bOption & 0x0F) == 0x01) && (wRxlen != 0x0C))
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFIDENTITY);
    }
	else
	{
		if(wRxlen != 0)
		{
			memcpy(pTMC, pRecv, 4); /* PRQA S 3200 */
			memcpy(pTMAC, &pRecv[4],8); /* PRQA S 3200 */
		}
	}

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}

phStatus_t phalMfIdentity_Sw_AbortTransaction(phalMfIdentity_Sw_DataParams_t * pDataParams)
{
    uint8_t     PH_MEMLOC_REM bCmdBuff[8];

    /* form the command */
    bCmdBuff[0] = PHAL_MFIDENTITY_CMD_ABORT_TXN;

    /* COMMUNICATION IS PLAIN */
    return phalMfIdentity_Sw_Int_Write_Plain(pDataParams,
		PHAL_MFIDENTITY_DEFAULT_MODE,
        bCmdBuff,
        1,
        (pDataParams->bAuthMode == PHAL_MFIDENTITY_AUTHENTICATEEV2)? PHAL_MFIDENTITY_COMMUNICATION_MACD : PHAL_MFIDENTITY_COMMUNICATION_PLAIN,
        NULL,
        0x0000
        );
}

phStatus_t phalMfIdentity_Sw_CommitReaderID(phalMfIdentity_Sw_DataParams_t * pDataParams, uint8_t * pTMRI, uint8_t * pEncTMRI)
{
	uint16_t    PH_MEMLOC_REM statusTmp;
	uint8_t     PH_MEMLOC_REM bCmdBuff[24];
	uint16_t    PH_MEMLOC_REM wRxlen;
	uint8_t     PH_MEMLOC_REM *pRecv;
	uint32_t    PH_MEMLOC_REM dwTMIStatus = 0;
	uint8_t		PH_MEMLOC_REM bOption = PHAL_MFIDENTITY_COMMUNICATION_PLAIN;
	uint8_t     PH_MEMLOC_REM bEncTMRILocal[16];
	uint8_t		PH_MEMLOC_REM aConfigValue[2];

	/* form the command */
	bCmdBuff[0] = PHAL_MFIDENTITY_CMD_COMMIT_READER_ID;
	memcpy(&bCmdBuff[1], pTMRI, 16); /* PRQA S 3200 */

	if(pDataParams->bAuthMode == PHAL_MFIDENTITY_AUTHENTICATEEV2)
	{
		bOption = PHAL_MFIDENTITY_COMMUNICATION_MACD;
	}

	/*
	* For Ev2 SM: PCD->PICC: cmd + cmdData ( CMAC (Kses, Cmd||cmdData) Mac is used as IV for next operation)
	*			   PICC->PCD: RC + respData + CMAC (Kses, cmdData||RC)
	*/
	PH_CHECK_SUCCESS_FCT(statusTmp, phalMfIdentity_Sw_Int_ReadData_Plain(
		pDataParams,
		(pDataParams->bAuthMode == PHAL_MFIDENTITY_AUTHENTICATEEV2) ? PHAL_MFIDENTITY_COMMUNICATION_MACD : PHAL_MFIDENTITY_COMMUNICATION_PLAIN,
		bCmdBuff,
		17,
		&pRecv,
		&wRxlen
	));

	if((bOption == PHAL_MFIDENTITY_COMMUNICATION_PLAIN) && (wRxlen != 0))
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFIDENTITY);
	}
	else if((bOption == PHAL_MFIDENTITY_COMMUNICATION_MACD) && (wRxlen != 16)) 		/* Response should be 16 bytes long */
	{
		return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_MFIDENTITY);
	}
	if(wRxlen > 0)
	{
		memcpy(bEncTMRILocal, pRecv, wRxlen);
		memcpy(pEncTMRI, pRecv, wRxlen);
	}
	else
	{
		pEncTMRI = '\0';
	}

	/* Do a Set Config of ADDITIONAL_INFO to set  the length(wLength) of the recieved TMRI */
	aConfigValue[0] = (uint8_t) (wRxlen & 0xFF);
	aConfigValue[1] = (uint8_t) ((wRxlen & 0xFF) >> 8);
	phalMfIdentity_Sw_SetConfig((phalMfIdentity_Sw_DataParams_t *) pDataParams, PHAL_MFIDENTITY_ADDITIONAL_INFO, aConfigValue);
	PH_CHECK_SUCCESS_FCT(statusTmp, phTMIUtils_GetConfig((phTMIUtils_t *) pDataParams->pTMIDataParams,
		PH_TMIUTILS_TMI_STATUS,
		&dwTMIStatus));

	/* Check TMI Collection Status */
	if(dwTMIStatus == PH_ON)
	{
		/* As per Ref arch(0.13),
		* If authenticated, Cmd.CommitReaderID shall update the Transaction MAC Input TMI as follows:
		* TMI = TMI || Cmd || TMRICur||EncTMRI||ZeroPadding
		*/
		if(pDataParams->bAuthMode == PHAL_MFIDENTITY_AUTHENTICATEEV2)
		{
			PH_CHECK_SUCCESS_FCT(statusTmp, phTMIUtils_CollectTMI((phTMIUtils_t *) pDataParams->pTMIDataParams,
				PH_TMIUTILS_ZEROPAD_DATABUFF,
				bCmdBuff,
				17,
				bEncTMRILocal,
				16,
				PHAL_MFIDENTITY_BLOCK_SIZE
			));
			memcpy(pEncTMRI, bEncTMRILocal, 16);
		}
		/* As per Ref arch(0.13),
		* If Not authenticated, Cmd.CommitReaderID shall update the Transaction MAC Input TMI as follows:
		* TMI = TMI || Cmd || TMRICur||ZeroPadding
		*/
		else
		{
			PH_CHECK_SUCCESS_FCT(statusTmp, phTMIUtils_CollectTMI((phTMIUtils_t *) pDataParams->pTMIDataParams,
				PH_TMIUTILS_ZEROPAD_CMDBUFF,
				bCmdBuff,
				17,
				NULL,
				0x00,
				PHAL_MFIDENTITY_BLOCK_SIZE
			));
			pEncTMRI = '\0';
		}
	}
	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}




/* MIFARE Identity ISO7816-4 commands. ------------------------------------------------------------------------------------------------- */
phStatus_t phalMfIdentity_Sw_IsoSelectFile(phalMfIdentity_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bSelector, uint8_t * pFid,
	uint8_t * pDFname, uint8_t bDFnameLen, uint8_t	bExtendedLenApdu, uint8_t ** ppFCI, uint16_t * pwFCILen)
{
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint8_t     PH_MEMLOC_REM bData[24];
    uint32_t     PH_MEMLOC_REM bLc = 0;
	uint32_t     PH_MEMLOC_REM bLe = 0;
	uint8_t		PH_MEMLOC_REM bFileId[3] = {'\0'};
	uint8_t     PH_MEMLOC_REM bPICCDefaultDFName[8] = {0xD2,0x76,0x00,0x00,0x85,0x01,0x00}; /*D2760000850100 PICC DFNAME*/
	phStatus_t  PH_MEMLOC_REM status;
	uint16_t    wVal = 0;

#ifdef RDR_LIB_PARAM_CHECK
    if (bDFnameLen > 16)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }
	/* If User intends to select through DFName, check for validity of DFName passed */
	if((bSelector ==  0x04) && (bDFnameLen == 0x00))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
	}
    if ((bOption != 0x00) && (bOption != 0x0C))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }
	if((bSelector !=  0x04) && (bSelector !=  0x03) && (pFid == NULL))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
	}
#endif
    if((bSelector ==  0x00) || (bSelector == 0x01) || (bSelector == 0x02))
	{
        /* Selection by EF Id*/
        /* Send MSB first to card */
#ifndef RDR_LIB_PARAM_CHECK
		if(pFid != NULL)
		{
#endif
			 bFileId[1] = bData[0] = pFid[1];
			 bFileId[0] = bData[1] = pFid[0];
			 bFileId[2] = 0x00;
			 bLc = 2;
#ifndef RDR_LIB_PARAM_CHECK
		}
#endif
	}
	else if((bSelector ==  0x04))
	{
        /* Selection by DF Name */
        memcpy(bData, pDFname, bDFnameLen); /* PRQA S 3200 */
        bLc = bDFnameLen;
	}
	else
	{
		/* Do Nothing, Pass it to PICC. Let Card throw error */
	}

	statusTmp = phalMfIdentity_Int_Send7816Apdu(
        pDataParams,
        pDataParams->pPalMifareDataParams,
        0x03,
        PHAL_MFIDENTITY_CMD_ISO7816_SELECT_FILE,
        bSelector,
        bOption,
        bLc,
		bExtendedLenApdu,
        bData,
        bLe,
        ppFCI,
        pwFCILen);

	if((statusTmp & PH_ERR_MASK) == PHAL_MFIDENTITY_ERR_DF_7816_GEN_ERROR)
	{
		status = phalMfIdentity_GetConfig(pDataParams, PHAL_MFIDENTITY_ADDITIONAL_INFO, (uint8_t *) &wVal);
	}

	if((statusTmp == PH_ERR_SUCCESS) || (wVal == PHAL_MFIDENTITY_ISO7816_ERR_LIMITED_FUNCTIONALITY_INS))
	{
		/* Reset Authentication should not be targeted for elementary file selection using file ID */
		if(bSelector !=  0x02)
		{
			/* Reset Authentication Status here */
			phalMfIdentity_Sw_Int_ResetAuthStatus(pDataParams);
		}

		/* ISO wrapped mode is on */
		pDataParams->bWrappedMode = 1;

		/* once the selection Success, update the File Id to master data structure if the selection is done through AID*/
		if((bSelector ==  0x00) || (bSelector == 0x01) || (bSelector == 0x02))
		{
			memcpy(pDataParams->pAid, bFileId, sizeof(bFileId));
		}
		else if((bSelector ==  0x04))
		{
			/* Check for PICC Level default DF Name */
			if (memcmp(bPICCDefaultDFName, pDFname, bDFnameLen) == 0)
			{
				bFileId[0] = 0x00;
				bFileId[1] = 0x00;
				bFileId[2] = 0x00;
			}
			else
			{
				/* APP Level ISOSelect Done. This is done to avoid session key generation at the time of authenticate API */
				bFileId[0] = 0xff;
				bFileId[1] = 0xff;
				bFileId[2] = 0xff;
			}
			memcpy(pDataParams->pAid, bFileId, sizeof(bFileId));
		}
	}
    return PH_ADD_COMPCODE(statusTmp, PH_COMP_AL_MFIDENTITY);
}




/* MIFARE Identity OriginalityCheck commands. ------------------------------------------------------------------------------------------ */
phStatus_t phalMfIdentity_Sw_ReadSign(phalMfIdentity_Sw_DataParams_t * pDataParams, uint8_t bAddr, uint8_t ** pSignature)
{

	phStatus_t  PH_MEMLOC_REM statusTmp;
	uint8_t     PH_MEMLOC_REM bCmdBuff[2];
	uint16_t    PH_MEMLOC_REM wCmdLen = 0;
	uint16_t    PH_MEMLOC_REM wRxLength = 0;

	/* build command frame */
	bCmdBuff[wCmdLen++] = PHAL_MFIDENTITY_CMD_READ_SIG;
	bCmdBuff[wCmdLen++] = bAddr;

	/* Req spec(ver 0.14 says),
	* 1. Cmd.Read_Sig shall return the NXPOriginalitySignature as written during wafer test in plain if not authenticated
	* 2. Cmd.Read_Sig shall require MACed command if authenticated.
	*/
	if(pDataParams->bAuthMode == PHAL_MFIDENTITY_NOT_AUTHENTICATED)
	{
		PH_CHECK_SUCCESS_FCT(statusTmp, phalMfIdentity_Sw_Int_ReadData_Plain(
			pDataParams,
			PHAL_MFIDENTITY_COMMUNICATION_PLAIN,
			bCmdBuff,
			wCmdLen,
			pSignature,
			&wRxLength
		));
	}
#ifdef NXPBUILD__PH_CRYPTOSYM
	else
	{
		/* Set the expected data length as 56 bytes */
		pDataParams->dwPayLoadLen = PHAL_MFIDENTITY_SIG_LENGTH;

		PH_CHECK_SUCCESS_FCT(statusTmp, phalMfIdentity_Sw_Int_ReadData_Enc(
			pDataParams,
			PHAL_MFIDENTITY_COMMUNICATION_ENC,
			bCmdBuff,
			wCmdLen,
			pSignature,
			&wRxLength
		));
	}
#endif /* NXPBUILD__PH_CRYPTOSYM */
	/* check received length :- 56 byte signature */
	if(wRxLength != PHAL_MFIDENTITY_SIG_LENGTH)
	{
		return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_AL_MFIDENTITY);
	}

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);

}




/* MIFARE DESFire Light Miscellaneous commands. ---------------------------------------------------------------------------------------- */
phStatus_t phalMfIdentity_Sw_GetConfig(phalMfIdentity_Sw_DataParams_t *pDataParams, uint16_t wConfig, uint8_t * pValue)
{
    switch (wConfig)
    {
		case PHAL_MFIDENTITY_ADDITIONAL_INFO:
			pValue[0] = (uint8_t ) (pDataParams->wAdditionalInfo & 0x00FF);
			pValue[1] = (uint8_t ) ((pDataParams->wAdditionalInfo & 0xFF00) >> 8);
			break;

		case PHAL_MFIDENTITY_WRAPPED_MODE:
			*pValue = pDataParams->bWrappedMode;
			break;

		case PHAL_MFIDENTITY_SHORT_LENGTH_APDU:
			*pValue = pDataParams->bShortLenApdu;
			break;

		case PHAL_MFIDENTITY_FILE_OPTION:
			*pValue = pDataParams->bFileOption;
			break;

		case PHAL_MFIDENTITY_SDM_OPTION:
			*pValue = pDataParams->bSDMOption;
			break;

		case PHAL_MFIDENTITY_KEY_INFO:
			pValue[0] = pDataParams->bSDMKeyNo;
			pValue[1] = pDataParams->bSDMKeyVer;
			break;

		case PHAL_MFIDENTITY_SDM_ENC_OFFSET:
			memcpy(pValue, pDataParams->aSDMENCOffset, 3); /* PRQA S 3200 */
			break;

		case PHAL_MFIDENTITY_SDM_ENC_LENGTH:
			memcpy(pValue, pDataParams->aSDMENCLength, 3); /* PRQA S 3200 */
			break;

		case PHAL_MFIDENTITY_SDM_READ_CTR:
			memcpy(pValue, pDataParams->aSDMReadCTR, 3); /* PRQA S 3200 */
			break;

		case PHAL_MFIDENTITY_SDM_KEY_TYPE:
			*pValue = pDataParams->bKeyType;
			break;

		default:
			return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}

phStatus_t phalMfIdentity_Sw_SetConfig(phalMfIdentity_Sw_DataParams_t * pDataParams, uint16_t wConfig, uint8_t *pValue)
{
    switch (wConfig)
    {
		case PHAL_MFIDENTITY_ADDITIONAL_INFO:
			pDataParams->wAdditionalInfo = (uint16_t) (pValue[0] | (pValue[1] << 8));
			break;

		case PHAL_MFIDENTITY_WRAPPED_MODE:
			pDataParams->bWrappedMode = pValue[0];
			break;

		case PHAL_MFIDENTITY_SHORT_LENGTH_APDU:
			pDataParams->bShortLenApdu = pValue[0];
			break;

		case PHAL_MFIDENTITY_FILE_OPTION:
			pDataParams->bFileOption = pValue[0];
			break;

		case PHAL_MFIDENTITY_SDM_OPTION:
			pDataParams->bSDMOption = pValue[0];
			break;

		case PHAL_MFIDENTITY_KEY_INFO:
			pDataParams->bSDMKeyNo = pValue[0];
			pDataParams->bSDMKeyVer = pValue[1];
			break;

		case PHAL_MFIDENTITY_SDM_ENC_OFFSET:
			memcpy(pDataParams->aSDMENCOffset, pValue, 3); /* PRQA S 3200 */
			break;

		case PHAL_MFIDENTITY_SDM_ENC_LENGTH:
			memcpy(pDataParams->aSDMENCLength, pValue, 3); /* PRQA S 3200 */
			break;

		case PHAL_MFIDENTITY_SDM_READ_CTR:
			memcpy(pDataParams->aSDMReadCTR, pValue, 3); /* PRQA S 3200 */
			break;

		case PHAL_MFIDENTITY_SDM_KEY_TYPE:
			pDataParams->bKeyType = *pValue;
			break;

		default:
			return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_AL_MFIDENTITY);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}

phStatus_t phalMfIdentity_Sw_ResetAuthentication(phalMfIdentity_Sw_DataParams_t * pDataParams)
{
    phalMfIdentity_Sw_Int_ResetAuthStatus(pDataParams);

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}

#ifdef NXPBUILD__PH_CRYPTOSYM
phStatus_t phalMfIdentity_Sw_CalculateTMV(phalMfIdentity_Sw_DataParams_t *pDataParams, uint16_t wOption, uint16_t wKeyNoTMACKey,
	uint16_t wKeyVerTMACKey, uint8_t * pDivInput, uint8_t bDivInputLen, uint8_t * pTMC, uint8_t * pUid, uint8_t bUidLen, uint8_t * pTMI,
	uint32_t dwTMILen, uint8_t * pTMV)
{
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint8_t     PH_MEMLOC_REM bKey[PH_CRYPTOSYM_AES128_KEY_SIZE];
    uint8_t     PH_MEMLOC_REM bMacLen = 0;
	uint8_t     PH_MEMLOC_REM bKeyLen = 0;
    uint8_t     PH_MEMLOC_REM bSVMacLen = 0;
    uint16_t    PH_MEMLOC_REM wKeyType;
    uint16_t    PH_MEMLOC_REM wTmpTMILen;
    uint16_t    PH_MEMLOC_REM bLoopData = 1;
    uint8_t     PH_MEMLOC_REM bTmpIV[PH_CRYPTOSYM_AES_BLOCK_SIZE];
    uint8_t     PH_MEMLOC_REM bSV[PH_CRYPTOSYM_AES128_KEY_SIZE];
    uint8_t     PH_MEMLOC_REM bCMAC[PH_CRYPTOSYM_AES_BLOCK_SIZE];
    uint32_t    PH_MEMLOC_REM dwTMC = 0;
	uint32_t    PH_MEMLOC_REM dwActTMC = 0;
	uint32_t    PH_MEMLOC_REM dwSessTMC = 0;
    uint32_t    PH_MEMLOC_REM dwTMCtemp = 0;
    uint32_t    PH_MEMLOC_REM dwTMILenWritten = 0;
	uint16_t    PH_MEMLOC_REM wCommMode;
	uint16_t    PH_MEMLOC_REM wAuthMode;

	PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_GetConfig(
		pDataParams->pCryptoDataParamsMac,
		PH_CRYPTOSYM_CONFIG_LRP,
		&wAuthMode
		));

	/* If LRP Enabled, Session keys are generated according to LRP Algorithm */
	if(wAuthMode == 0x01)
	{

		/* Get Key out of the key store object */
		PH_CHECK_SUCCESS_FCT(statusTmp, phKeyStore_GetKey(
			pDataParams->pKeyStoreDataParams,
			wKeyNoTMACKey,
			wKeyVerTMACKey,
			PH_CRYPTOSYM_AES128_KEY_SIZE,
			bKey,
			&wKeyType
			));

		/* Invalid key type at wKeyNoTMACKey and wKeyVerTMACKey */
		if (wKeyType != PH_CRYPTOSYM_KEY_TYPE_AES128)
		{
			return PH_ADD_COMPCODE(PH_ERR_KEY, PH_COMP_AL_MFIDENTITY);
		}

		/* Formation of Actual TMC . TMC  communicated LSB first. */
		dwActTMC = pTMC[0];
		dwTMCtemp = pTMC[1];
		dwActTMC |= (dwTMCtemp << 8);

		/* Formation of Session TMC . TMC communicated LSB first. */
		dwSessTMC = pTMC[2];
		dwTMCtemp = pTMC[3];
		dwSessTMC |= (dwTMCtemp << 8);

		/* Increment Both Actual and session TMC */
		dwActTMC++;
		dwSessTMC++;

		/* Update bSV buffer SV = 0x00 || 0x01 || 0x00 || 0x80 ||(actTMC + 1) || (sessTMC + 1) || UID [||0x00::0x00] || 0x5A */
		bSV[bSVMacLen++] =  0x00;
		bSV[bSVMacLen++] =  0x01;
		bSV[bSVMacLen++] =  0x00;
		bSV[bSVMacLen++] =  0x80;
		bSV[bSVMacLen++] = (uint8_t)(dwActTMC & 0xFF);
		bSV[bSVMacLen++] = (uint8_t)((dwActTMC >> 8) & 0xFF);
		bSV[bSVMacLen++] = (uint8_t)((dwSessTMC) & 0xFF);
		bSV[bSVMacLen++] = (uint8_t)((dwSessTMC >> 8) & 0xFF);

		/* Copy UID into SV - UID should not more then 7 byte in
		* this case else bSV array size needs to be changed
		*/
		memcpy(&bSV[bSVMacLen], pUid, bUidLen); /* PRQA S 3200 */

		bSVMacLen += bUidLen;

		/* SV padded with the zero bytes up to a length of 16 bytes(INCLUSIVE OF 0x5A) (if needed)*/
		if(bSVMacLen < PH_CRYPTOSYM_AES128_KEY_SIZE-1)
		{
			memset(&bSV[bSVMacLen], 0x00, (PH_CRYPTOSYM_AES128_KEY_SIZE - bSVMacLen)); /* PRQA S 3200 */
		}

		/* At Last add footer byte */
		bSV[bSVMacLen++] = 0x5A;

		/* Now compute Session mac Keys */

		/* load Master key */
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadKeyDirect(
			pDataParams->pCryptoDataParamsMac,
			bKey,
			wKeyType));

		/* Calculate Mac to obtain SesTMMACKey */
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_CalculateMac(
			pDataParams->pCryptoDataParamsMac,
			(PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_DEFAULT),
			bSV,
			PH_CRYPTOSYM_AES_BLOCK_SIZE,
			bKey,
			&bKeyLen
			));

		/* Now calculate TMV as TMV = MACtTM(KSesTMMAC; TMI) */
		/* load key -SesTMMACKey*/
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadKeyDirect(
			pDataParams->pCryptoDataParamsMac,
			bKey,
			PH_CRYPTOSYM_KEY_TYPE_AES128));

		/* to handle 2 MB of data update maximum of data bytes that can be sent in a single loop */
		if(dwTMILen > PHAL_MFIDENTITY_MAX_WRITE_SIZE)
		{
			wTmpTMILen = (uint16_t) PHAL_MFIDENTITY_MAX_WRITE_SIZE;
			bLoopData = (uint8_t)(dwTMILen/PHAL_MFIDENTITY_MAX_WRITE_SIZE);
			if(dwTMILen % PHAL_MFIDENTITY_MAX_WRITE_SIZE)
			{
				bLoopData++;
			}
			wCommMode = PH_EXCHANGE_BUFFER_FIRST;
		}
		else
		{
			wTmpTMILen = (uint16_t)dwTMILen;
			wCommMode = PH_EXCHANGE_DEFAULT;
		}

        /* Disable LRP since Decrypt Reader id shall happen with session key of LRP and SMEV2 Decrypt  */
        PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_SetConfig(pDataParams->pCryptoDataParamsMac, PH_CRYPTOSYM_CONFIG_LRP, 0x00));
		do{

			if (bLoopData == 1)
			{
				wTmpTMILen = (uint16_t)(dwTMILen - dwTMILenWritten);
			}

			/* Encrypt TMI to obtain TMV */
			PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_CalculateMac(
				pDataParams->pCryptoDataParamsMac,
				(PH_CRYPTOSYM_MAC_MODE_CMAC | wCommMode),
				&pTMI[dwTMILenWritten],
				wTmpTMILen,
				bCMAC,
				&bMacLen
				));
			bLoopData--;
			wCommMode = (bLoopData == 1)?PH_EXCHANGE_BUFFER_LAST:PH_EXCHANGE_BUFFER_CONT;
			dwTMILenWritten +=wTmpTMILen;
		}while(bLoopData > 0);

		/* Restore back LRP enabled mode */
        PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_SetConfig(pDataParams->pCryptoDataParamsMac, PH_CRYPTOSYM_CONFIG_LRP, 0x01));

		/* ReLoad the Key session master stored in Master Data structure */
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadKeyDirect(
			pDataParams->pCryptoDataParamsMac,
			pDataParams->bKeySessionAuthMaster,
			PH_CRYPTOSYM_KEY_TYPE_AES128));

		/* ReLoad the Key session master stored in Master Data structure */
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadKeyDirect(
			pDataParams->pCryptoDataParamsEnc,
			pDataParams->bKeySessionAuthMaster,
			PH_CRYPTOSYM_KEY_TYPE_AES128));
	}
	else
	{
		/*OLD this block is replaced by below for Qmore compliance
		dwTMC |= (uint32_t)pTMC[3];
		dwTMC |= (uint32_t)(pTMC[2] << 8);
		dwTMC |= (uint32_t)(pTMC[1] << 16);
		dwTMC |= (uint32_t)(pTMC[0] << 24); */

		/* Formation of TMC as double word value- TMC shall be communicated LSB first. */
		dwTMC = pTMC[0];
		dwTMCtemp = pTMC[1];
		dwTMC |= (dwTMCtemp << 8);
		dwTMCtemp = pTMC[2];
		dwTMC |= (dwTMCtemp << 16);
		dwTMCtemp = pTMC[3];
		dwTMC |= (dwTMCtemp << 24);

#ifdef RDR_LIB_PARAM_CHECK
		/* If TMC is 0xFFFFFFFF, then return error */
		if (dwTMC == 0xFFFFFFFF)
		{
			return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_AL_MFIDENTITY);
		}

		if ((wOption != PHAL_MFIDENTITY_NO_DIVERSIFICATION) && (bDivInputLen > 31))
		{
			return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_MFIDENTITY);
		}
#endif
		/* Get Key out of the key store object */
		PH_CHECK_SUCCESS_FCT(statusTmp, phKeyStore_GetKey(
			pDataParams->pKeyStoreDataParams,
			wKeyNoTMACKey,
			wKeyVerTMACKey,
			PH_CRYPTOSYM_AES128_KEY_SIZE,
			bKey,
			&wKeyType
			));

		/* Invalid key type at wKeyNoTMACKey and wKeyVerTMACKey */
		if (wKeyType != PH_CRYPTOSYM_KEY_TYPE_AES128)
		{
			return PH_ADD_COMPCODE(PH_ERR_KEY, PH_COMP_AL_MFIDENTITY);
		}

		if ((wOption != PHAL_MFIDENTITY_NO_DIVERSIFICATION) && (bDivInputLen != 0x00))
		{
			/* Key is diversified and put back in bKey */
			PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_DiversifyDirectKey(
				pDataParams->pCryptoDataParamsEnc,
				wOption,
				bKey,
				wKeyType,
				pDivInput,
				bDivInputLen,
				bKey
				));
		}

		/* Increment dwTMC */
		dwTMC++;

		/* Update bSV buffer SV = 0x5A || 0x00 || 0x01 || 0x00 || 0x80 ||(TMC + 1) || UID [||0x00::0x00] */
		bSV[bSVMacLen++] = 0x5A;
		bSV[bSVMacLen++] = 0x00;
		bSV[bSVMacLen++] = 0x01;
		bSV[bSVMacLen++] = 0x00;
		bSV[bSVMacLen++] = 0x80;
		bSV[bSVMacLen++] = (uint8_t)(dwTMC & 0xFF);
		bSV[bSVMacLen++] = (uint8_t)((dwTMC >> 8) & 0xFF);
		bSV[bSVMacLen++] = (uint8_t)((dwTMC >> 16) & 0xFF);
		bSV[bSVMacLen++] = (uint8_t)((dwTMC >> 24) & 0xFF);

		/* Copy UID into SV - UID should not more then 7 byte in
		* this case else bSV array size needs to be changed
		*/
		memcpy(&bSV[bSVMacLen], pUid, bUidLen); /* PRQA S 3200 */

		bSVMacLen += bUidLen;

		/* SV padded with the zero bytes up to a length of 16 bytes (if needed)*/
		if(bSVMacLen < PH_CRYPTOSYM_AES128_KEY_SIZE)
		{
			memset(&bSV[bSVMacLen], 0x00, (PH_CRYPTOSYM_AES128_KEY_SIZE - bSVMacLen)); /* PRQA S 3200 */
		}

		/* load key */
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadKeyDirect(
			pDataParams->pCryptoDataParamsEnc,
			bKey,
			wKeyType));

		/* Create a Back up of the current IV */
		memcpy(bTmpIV, pDataParams->bIv, PH_CRYPTOSYM_AES_BLOCK_SIZE); /* PRQA S 3200 */

		/* Load zero to IV */
		memset(pDataParams->bIv, 0x00, PH_CRYPTOSYM_AES_BLOCK_SIZE);  /* PRQA S 3200 */

		/* Load Iv */
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadIv(
			pDataParams->pCryptoDataParamsEnc,
			pDataParams->bIv,
			PH_CRYPTOSYM_AES_BLOCK_SIZE
			));

		/* Encrypt SV to obtain SesTMMACKey */
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_CalculateMac(
			pDataParams->pCryptoDataParamsEnc,
			(PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_DEFAULT),
			bSV,
			PH_CRYPTOSYM_AES_BLOCK_SIZE,
			bKey,
			&bKeyLen
			));

		/* Now calculate TMV as TMV = MACtTM(KSesTMMAC; TMI) */
		/* load key -SesTMMACKey*/
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadKeyDirect(
			pDataParams->pCryptoDataParamsMac,
			bKey,
			PH_CRYPTOSYM_KEY_TYPE_AES128));

		/* Load Iv */
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadIv(
			pDataParams->pCryptoDataParamsMac,
			pDataParams->bIv,
			PH_CRYPTOSYM_AES_BLOCK_SIZE
			));

		/* to handle 2 MB of data update maximum of data bytes that can be sent in a single loop */
		if(dwTMILen > PHAL_MFIDENTITY_MAX_WRITE_SIZE)
		{
			wTmpTMILen = (uint16_t) PHAL_MFIDENTITY_MAX_WRITE_SIZE;
			bLoopData = (uint8_t)(dwTMILen/PHAL_MFIDENTITY_MAX_WRITE_SIZE);
			if(dwTMILen % PHAL_MFIDENTITY_MAX_WRITE_SIZE)
			{
				bLoopData++;
			}
			wCommMode = PH_EXCHANGE_BUFFER_FIRST;
		}
		else
		{
			wTmpTMILen = (uint16_t)dwTMILen;
			wCommMode = PH_EXCHANGE_DEFAULT;
		}

		do{

			if (bLoopData == 1)
			{
				wTmpTMILen = (uint16_t)(dwTMILen - dwTMILenWritten);
			}
			/* Encrypt TMI to obtain TMV */
			PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_CalculateMac(
				pDataParams->pCryptoDataParamsMac,
				(PH_CRYPTOSYM_MAC_MODE_CMAC | wCommMode),
				&pTMI[dwTMILenWritten],
				wTmpTMILen,
				bCMAC,
				&bMacLen
				));
			bLoopData--;
			wCommMode = (bLoopData == 1)?PH_EXCHANGE_BUFFER_LAST:PH_EXCHANGE_BUFFER_CONT;
			dwTMILenWritten +=wTmpTMILen;
		}while(bLoopData > 0);

		if(pDataParams->bAuthMode != PHAL_MFIDENTITY_NOT_AUTHENTICATED)
		{
			/* Load the session key */
			PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadKeyDirect(
				pDataParams->pCryptoDataParamsEnc,
				pDataParams->bSesAuthENCKey,
				pDataParams->bCryptoMethod
				));

			/* Load the session key */
			PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadKeyDirect(
				pDataParams->pCryptoDataParamsMac,
				pDataParams->bSesAuthMACKey,
				pDataParams->bCryptoMethod
				));
		}
	}
	/* Truncate the Calculated CMAC */
	phalMfIdentity_Sw_Int_TruncateMac(bCMAC);

	/* Copy the Truncated CMAC into the return buffer */
	memcpy(pTMV, bCMAC, PHAL_MFIDENTITY_TRUNCATED_MAC_SIZE); /* PRQA S 3200 */

	/* Restore back the IV */
	memcpy(pDataParams->bIv, bTmpIV, PH_CRYPTOSYM_AES_BLOCK_SIZE); /* PRQA S 3200 */

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}


phStatus_t phalMfIdentity_Sw_DecryptReaderID(phalMfIdentity_Sw_DataParams_t *pDataParams, uint16_t wOption, uint16_t wKeyNoTMACKey,
	uint16_t wKeyVerTMACKey, uint8_t * pDivInput, uint8_t bDivInputLen, uint8_t * pTMC, uint8_t * pUid, uint8_t bUidLen, uint8_t * pEncTMRI,
	uint8_t * pTMRIPrev)
{
	phStatus_t  PH_MEMLOC_REM statusTmp;
	uint8_t     PH_MEMLOC_REM bKey[PH_CRYPTOSYM_AES128_KEY_SIZE];
	uint8_t     PH_MEMLOC_REM bSVMacLen = 0;
	uint16_t    PH_MEMLOC_REM wKeyType;
	uint8_t     PH_MEMLOC_REM bTmpIV[PH_CRYPTOSYM_AES_BLOCK_SIZE];
	uint8_t     PH_MEMLOC_REM bSV[PH_CRYPTOSYM_AES128_KEY_SIZE];
	uint32_t    PH_MEMLOC_REM dwTMC = 0;
	uint16_t    PH_MEMLOC_REM wActTMC = 0;
	uint16_t    PH_MEMLOC_REM wSessTMC = 0;
	uint32_t    PH_MEMLOC_REM dwTMCtemp = 0;
	uint16_t    PH_MEMLOC_REM wAuthMode;
	uint16_t    PH_MEMLOC_REM wBkupIV;

	PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_GetConfig(
		pDataParams->pCryptoDataParamsEnc,
		PH_CRYPTOSYM_CONFIG_LRP,
		&wAuthMode
		));

	/* If LRP Enabled, Session keys are generated according to LRP Algorithm */
	if(wAuthMode == 0x01)
	{

		/* Get Key out of the key store object */
		PH_CHECK_SUCCESS_FCT(statusTmp, phKeyStore_GetKey(
			pDataParams->pKeyStoreDataParams,
			wKeyNoTMACKey,
			wKeyVerTMACKey,
			PH_CRYPTOSYM_AES128_KEY_SIZE,
			bKey,
			&wKeyType
			));

		/* Invalid key type at wKeyNoTMACKey and wKeyVerTMACKey */
		if (wKeyType != PH_CRYPTOSYM_KEY_TYPE_AES128)
		{
			return PH_ADD_COMPCODE(PH_ERR_KEY, PH_COMP_AL_MFIDENTITY);
		}

		/* Formation of Actual TMC . TMC  communicated LSB first. */
		wActTMC = pTMC[0];
		dwTMCtemp = pTMC[1];
		wActTMC |= (dwTMCtemp << 8);

		/* Formation of Session TMC . TMC communicated LSB first. */
		wSessTMC = pTMC[2];
		dwTMCtemp = pTMC[3];
		wSessTMC |= (dwTMCtemp << 8);

		/* Increment Both Actual and session TMC */
		wActTMC++;
		wSessTMC++;

		/* Update bSV buffer SV = 0x00 || 0x01 || 0x00 || 0x80 ||(actTMC + 1) || (sessTMC + 1) || UID [||0x00::0x00] || 0xA5 */
		bSV[bSVMacLen++] =  0x00;
		bSV[bSVMacLen++] =  0x01;
		bSV[bSVMacLen++] =  0x00;
		bSV[bSVMacLen++] =  0x80;
		bSV[bSVMacLen++] = (uint8_t)(wActTMC & 0xFF);
		bSV[bSVMacLen++] = (uint8_t)((wActTMC >> 8) & 0xFF);
		bSV[bSVMacLen++] = (uint8_t)((wSessTMC) & 0xFF);
		bSV[bSVMacLen++] = (uint8_t)((wSessTMC >> 8) & 0xFF);

		/* Copy UID into SV - UID should not more then 7 byte in
		* this case else bSV array size needs to be changed
		*/
		memcpy(&bSV[bSVMacLen], pUid, bUidLen); /* PRQA S 3200 */

		bSVMacLen += bUidLen;

		/* SV padded with the zero bytes up to a length of 16 bytes (if needed) -- 16 Bytes inclusive of 0xA5*/
		if(bSVMacLen < PH_CRYPTOSYM_AES128_KEY_SIZE-1)
		{
			memset(&bSV[bSVMacLen], 0x00, (PH_CRYPTOSYM_AES128_KEY_SIZE - bSVMacLen)); /* PRQA S 3200 */
		}

		/* At Last add footer byte */
		bSV[bSVMacLen++] = 0xA5;

		/* Now compute Session mac Keys */

		/* load Master key */
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadKeyDirect(
			pDataParams->pCryptoDataParamsEnc,
			bKey,
			wKeyType));

		/* Create a Back up of the current IV */
		memcpy(bTmpIV, pDataParams->bIv, PH_CRYPTOSYM_AES_BLOCK_SIZE); /* PRQA S 3200 */

		/* Load zero to IV */
		memset(pDataParams->bIv, 0x00, PH_CRYPTOSYM_AES_BLOCK_SIZE);  /* PRQA S 3200 */

		/* Load Iv */
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadIv(
			pDataParams->pCryptoDataParamsEnc,
			pDataParams->bIv,
			PH_CRYPTOSYM_AES_BLOCK_SIZE
			));

		/* Get the IV Status */
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_GetConfig(pDataParams->pCryptoDataParamsEnc,
			PH_CRYPTOSYM_CONFIG_KEEP_IV,
			&wBkupIV));

		/* Set Keep IV Off */
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_SetConfig(pDataParams->pCryptoDataParamsEnc,
			PH_CRYPTOSYM_CONFIG_KEEP_IV,
			PH_CRYPTOSYM_VALUE_KEEP_IV_OFF));

		/* Calculate Mac to obtain KSesTMENC */
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_CalculateMac(
			pDataParams->pCryptoDataParamsEnc,
			(PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_DEFAULT),
			bSV,
			PH_CRYPTOSYM_AES_BLOCK_SIZE,
			bKey,
			&bSVMacLen
			));

		/* load KSesTMENC */
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadKeyDirect(
			pDataParams->pCryptoDataParamsEnc,
			bKey,
			PH_CRYPTOSYM_KEY_TYPE_AES128));

		/* Load Iv */
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadIv(
			pDataParams->pCryptoDataParamsEnc,
			pDataParams->bIv,
			PH_CRYPTOSYM_AES_BLOCK_SIZE
			));

		/* Disable LRP since Decrypt Reader id shall happen with session key of LRP and SMEV2 Decrypt  */
        PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_SetConfig(pDataParams->pCryptoDataParamsEnc, PH_CRYPTOSYM_CONFIG_LRP, 0x00));

		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_Decrypt(
		    pDataParams->pCryptoDataParamsEnc,
			PH_CRYPTOSYM_CIPHER_MODE_CBC | PH_EXCHANGE_DEFAULT,
			pEncTMRI,
			PH_CRYPTOSYM_AES_BLOCK_SIZE,
			pTMRIPrev
			));

		/* Set the original IV Status back */
		phCryptoSym_SetConfig(pDataParams->pCryptoDataParamsEnc,
			PH_CRYPTOSYM_CONFIG_KEEP_IV,
			wBkupIV);

        /* Restore back LRP enabled mode */
        PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_SetConfig(pDataParams->pCryptoDataParamsEnc, PH_CRYPTOSYM_CONFIG_LRP, 0x01));

		/* Restore back the IV */
		memcpy(pDataParams->bIv, bTmpIV, PH_CRYPTOSYM_AES_BLOCK_SIZE); /* PRQA S 3200 */

		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadIv(
			pDataParams->pCryptoDataParamsEnc,
			pDataParams->bIv,
			PH_CRYPTOSYM_AES_BLOCK_SIZE
			));

		if(pDataParams->bAuthMode != PHAL_MFIDENTITY_NOT_AUTHENTICATED)
		{
 			/* Load the session ENC key */
			PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadKeyDirect(
				pDataParams->pCryptoDataParamsEnc,
				pDataParams->bKeySessionAuthMaster,
				pDataParams->bCryptoMethod
				));

			/* Load the session MAC key */
			PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadKeyDirect(
				pDataParams->pCryptoDataParamsMac,
				pDataParams->bKeySessionAuthMaster,
				pDataParams->bCryptoMethod
				));
        }
		return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
	}
	else
	{

		/* Formation of TMC as double word value- TMC shall be communicated LSB first. */
		dwTMC = pTMC[0];
		dwTMCtemp = pTMC[1];
		dwTMC |= (dwTMCtemp << 8);
		dwTMCtemp = pTMC[2];
		dwTMC |= (dwTMCtemp << 16);
		dwTMCtemp = pTMC[3];
		dwTMC |= (dwTMCtemp << 24);


		/* Get Key out of the key store object */
		PH_CHECK_SUCCESS_FCT(statusTmp, phKeyStore_GetKey(
			pDataParams->pKeyStoreDataParams,
			wKeyNoTMACKey,
			wKeyVerTMACKey,
			PH_CRYPTOSYM_AES128_KEY_SIZE,
			bKey,
			&wKeyType
			));

		/* Invalid key type at wKeyNoTMACKey and wKeyVerTMACKey */
		if (wKeyType != PH_CRYPTOSYM_KEY_TYPE_AES128)
		{
			return PH_ADD_COMPCODE(PH_ERR_KEY, PH_COMP_AL_MFIDENTITY);
		}

		if ((wOption != PHAL_MFIDENTITY_NO_DIVERSIFICATION) && (bDivInputLen != 0x00))
		{
			/* Key is diversified and put back in bKey */
			PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_DiversifyDirectKey(
				pDataParams->pCryptoDataParamsEnc,
				wOption,
				bKey,
				wKeyType,
				pDivInput,
				bDivInputLen,
				bKey
				));
		}

		/* Increment dwTMC */
		dwTMC++;

		/* Update bSV buffer SV = 0xA5 || 0x00 || 0x01 || 0x00 || 0x80 ||(TMC + 1) || UID [||0x00::0x00] */
		bSV[bSVMacLen++] = 0xA5;
		bSV[bSVMacLen++] = 0x00;
		bSV[bSVMacLen++] = 0x01;
		bSV[bSVMacLen++] = 0x00;
		bSV[bSVMacLen++] = 0x80;
		bSV[bSVMacLen++] = (uint8_t)(dwTMC & 0xFF);
		bSV[bSVMacLen++] = (uint8_t)((dwTMC >> 8) & 0xFF);
		bSV[bSVMacLen++] = (uint8_t)((dwTMC >> 16) & 0xFF);
		bSV[bSVMacLen++] = (uint8_t)((dwTMC >> 24) & 0xFF);

		/* Copy UID into SV - UID should not more then 7 byte in
		* this case else bSV array size needs to be changed
		*/
		memcpy(&bSV[bSVMacLen], pUid, bUidLen); /* PRQA S 3200 */

		bSVMacLen += bUidLen;

		/* SV padded with the zero bytes up to a length of 16 bytes (if needed)*/
		if(bSVMacLen < PH_CRYPTOSYM_AES128_KEY_SIZE)
		{
			memset(&bSV[bSVMacLen], 0x00, (PH_CRYPTOSYM_AES128_KEY_SIZE - bSVMacLen)); /* PRQA S 3200 */
		}

		/* load key */
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadKeyDirect(
			pDataParams->pCryptoDataParamsEnc,
			bKey,
			wKeyType));

		/* Create a Back up of the current IV */
		memcpy(bTmpIV, pDataParams->bIv, PH_CRYPTOSYM_AES_BLOCK_SIZE); /* PRQA S 3200 */

		/* Load zero to IV */
		memset(pDataParams->bIv, 0x00, PH_CRYPTOSYM_AES_BLOCK_SIZE);  /* PRQA S 3200 */

		/* Load Iv */
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadIv(
			pDataParams->pCryptoDataParamsEnc,
			pDataParams->bIv,
			PH_CRYPTOSYM_AES_BLOCK_SIZE
			));

		/* Encrypt SV to obtain SesTMMACKey */
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_CalculateMac(
			pDataParams->pCryptoDataParamsEnc,
			(PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_DEFAULT),
			bSV,
			PH_CRYPTOSYM_AES_BLOCK_SIZE,
			bKey,
			&bSVMacLen
			));

		/* Now calculate TMV as TMV = MACtTM(KSesTMMAC; TMI) */
		/* load key -SesTMMACKey*/
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadKeyDirect(
			pDataParams->pCryptoDataParamsEnc,
			bKey,
			PH_CRYPTOSYM_KEY_TYPE_AES128));

		/* Load Iv */
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadIv(
			pDataParams->pCryptoDataParamsEnc,
			pDataParams->bIv,
			PH_CRYPTOSYM_AES_BLOCK_SIZE
			));

		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_Decrypt(
			pDataParams->pCryptoDataParamsEnc,
			PH_CRYPTOSYM_CIPHER_MODE_CBC | PH_EXCHANGE_DEFAULT,
			pEncTMRI,
			PH_CRYPTOSYM_AES_BLOCK_SIZE,
			pTMRIPrev
			));

		/* Restore back the IV */
		memcpy(pDataParams->bIv, bTmpIV, PH_CRYPTOSYM_AES_BLOCK_SIZE); /* PRQA S 3200 */

		if(pDataParams->bAuthMode != PHAL_MFIDENTITY_NOT_AUTHENTICATED)
		{
			/* Load the session key */
			PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadKeyDirect(
				pDataParams->pCryptoDataParamsEnc,
				pDataParams->bSesAuthENCKey,
				pDataParams->bCryptoMethod
				));
			/* Load the session key */
			PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadKeyDirect(
				pDataParams->pCryptoDataParamsMac,
				pDataParams->bSesAuthMACKey,
				pDataParams->bCryptoMethod
				));
		}

		return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
	}
}

phStatus_t phalMfIdentity_Sw_CalculateMACSDM(phalMfIdentity_Sw_DataParams_t * pDataParams, uint16_t wOption, uint8_t bSdmOption,
	uint16_t wSDMMacKeyNo, uint16_t wSDMMacKeyVer, uint8_t * pDivInput, uint8_t bDivInputLen, uint8_t * pUid, uint8_t bUidLen,
	uint8_t * pSDMReadCtr, uint8_t * pInData, uint16_t wInDataLen, uint8_t * pRespMac)
{
	phStatus_t  PH_MEMLOC_REM statusTmp =  PH_ERR_SUCCESS;
    uint8_t     PH_MEMLOC_REM bKey[PH_CRYPTOSYM_AES128_KEY_SIZE];
    uint16_t    PH_MEMLOC_REM wKeyType;
    uint8_t     PH_MEMLOC_REM bTmpIV[PH_CRYPTOSYM_AES_BLOCK_SIZE];
    uint8_t     PH_MEMLOC_REM bSV[PH_CRYPTOSYM_AES128_KEY_SIZE];
    uint8_t     PH_MEMLOC_REM bCMAC[PH_CRYPTOSYM_AES_BLOCK_SIZE];
    uint32_t    PH_MEMLOC_REM dwSDMctr = 0;
	uint32_t    PH_MEMLOC_REM dwSDMctrtemp = 0;
    uint8_t     PH_MEMLOC_REM bSVMacLen = 0;
	uint16_t    PH_MEMLOC_REM wAuthMode = 0x00;
	uint8_t		PH_MEMLOC_REM bKeySessionAuthMaster[PH_CRYPTOSYM_AES_BLOCK_SIZE];
	uint8_t     PH_MEMLOC_REM bMacLen = 0x00;

	pDivInput =  NULL; /* Avoid Warnings */
	bDivInputLen = 0x00; /* Avoid Warnings */
	wOption =  PHAL_MFIDENTITY_NO_DIVERSIFICATION; /* Avoid Warnings */

	PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_GetConfig(
		pDataParams->pCryptoDataParamsMac,
		PH_CRYPTOSYM_CONFIG_LRP,
		&wAuthMode
		));

	if((wAuthMode == 0x01) || pDataParams->bKeyType)
	{
		/* SV = 0x00 || 0x01 || 0x00 || 0x80 [|| VCUID ][|| SDMReadCtr][||ZeroPadding] || 0x1E || 0xE1 */
		/* Padding not required as we have 16 Bytes block */;
		bSV[bSVMacLen++] = 0x00;
		bSV[bSVMacLen++] = 0x01;
		bSV[bSVMacLen++] = 0x00;
		bSV[bSVMacLen++] = 0x80;

		/* Generate Session keys as per LRP Mode */
		/* Formation of SDM Ctr as double word value- SDM Ctr shall be communicated LSB first. */
		dwSDMctr = pSDMReadCtr[0];
		dwSDMctrtemp = pSDMReadCtr[1];
		dwSDMctr |= (dwSDMctrtemp << 8);
		dwSDMctrtemp = pSDMReadCtr[2];
		dwSDMctr |= (dwSDMctrtemp << 16);
		dwSDMctrtemp = pSDMReadCtr[3];
		dwSDMctr |= (dwSDMctrtemp << 24);

		/* If SDM Ctr is 0xFFFFFF, then return error */
		if (dwSDMctr == 0xFFFFFF)
		{
			return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_AL_MFIDENTITY);
		}

		/* Copy UID into SV - UID should not more then 7 byte in
			* this case else bSV array size needs to be changed
			*/
		if(bSdmOption & PHAL_MFIDENTITY_VCUID_PRESENT)
		{
			/* If VCUID should be considered for MAC calcluation, and pUID is passed as NULL, throw error */
			memcpy(&bSV[bSVMacLen], pUid, bUidLen); /* PRQA S 3200 */
			bSVMacLen += bUidLen;
		}

		if(bSdmOption & PHAL_MFIDENTITY_RDCTR_PRESENT)
		{
			/* If SDMReadCounter should be considered for MAC calcluation, and pUID is passed as NULL, throw error */
			bSV[bSVMacLen++] = (uint8_t)(dwSDMctr & 0xFF);
			bSV[bSVMacLen++] = (uint8_t)((dwSDMctr >> 8) & 0xFF);
			bSV[bSVMacLen++] = (uint8_t)((dwSDMctr >> 16) & 0xFF);
		}

		if(bUidLen != 0x07)
		{
			/* Pad the remaining 3 bytes with 0x00 to make SV1 AES_BLOCK_SIZE */
			memcpy(&bSV[bSVMacLen], 0x00, 0x03);
		}

		/* At Last add footer bytes */
		bSV[bSVMacLen++] = 0x1E;
		bSV[bSVMacLen++] = 0xE1;

		/* Session MAC/ENC key generation follow following steps
			Step-1 : SDMFileReadSPT = generatePlaintexts(4;KSDMFileRead)
			Step-2 : fKSDMFileReadUpdateg = generateUpdatedKeys(1;KSDMFileRead)
			Step-3 : KSesSDMFileReadMaster = MACLRP (KSDMFileRead; SV )
			Step-4 : SesSDMFileReadSPT = generateP laintexts(4;KSesSDMFileReadMaster)
			Step-5 : fKSesSDMFileReadMACUpdate;KSesSDMFileReadENCUpdateg = generateUpdatedKeys(2;KSesSDMFileReadMaster)
		*/

		/* Get the key Kx from Software key store  */
		PH_CHECK_SUCCESS_FCT(statusTmp, phKeyStore_GetKey(
			pDataParams->pKeyStoreDataParams,
			wSDMMacKeyNo,
			wSDMMacKeyVer,
			PH_CRYPTOSYM_AES128_KEY_SIZE,
			bKey,
			&wKeyType
			));

		/* Invalid key type at wKeyNo and wKeyVer */
		if (wKeyType != PH_CRYPTOSYM_KEY_TYPE_AES128)
		{
			return PH_ADD_COMPCODE(PH_ERR_KEY, PH_COMP_AL_MFIDENTITY);
		}

		/* load key */
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadKeyDirect(
			pDataParams->pCryptoDataParamsMac,
			bKey,
			wKeyType));

		/* Generate Session Master Key using Session Vector  */
		memset(bKeySessionAuthMaster, 0x00, PH_CRYPTOSYM_AES_BLOCK_SIZE);
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_CalculateMac(pDataParams->pCryptoDataParamsMac,
            PH_CRYPTOSYM_MAC_MODE_LRP_SKM,
            bSV,
			PH_CRYPTOSYM_AES_BLOCK_SIZE,
            bKeySessionAuthMaster,
            &bMacLen));

		/* Create a Back up of the current IV */
		memcpy(bTmpIV, pDataParams->bIv, PH_CRYPTOSYM_AES_BLOCK_SIZE); /* PRQA S 3200 */

		/* Load zero to IV */
		memset(pDataParams->bIv, 0x00, PH_CRYPTOSYM_AES_BLOCK_SIZE);  /* PRQA S 3200 */

		/* Load Iv */
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadIv(
			pDataParams->pCryptoDataParamsMac,
			pDataParams->bIv,
			PH_CRYPTOSYM_AES_BLOCK_SIZE
			));

		/* Now Load the Session Master Key to software keystore */
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadKeyDirect(
			pDataParams->pCryptoDataParamsMac,
			bKeySessionAuthMaster,
			PH_CRYPTOSYM_KEY_TYPE_AES128));


		/* Mac The Input Data using K(sessionSDMMaster) to obtain SDMMac.  */
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_CalculateMac(
			pDataParams->pCryptoDataParamsMac,
			PH_CRYPTOSYM_MAC_MODE_LRP | PH_EXCHANGE_DEFAULT,
			pInData,
			wInDataLen,
			bCMAC,
			&bSVMacLen
			));

		/* Truncate the MAC generated */
		phalMfIdentity_Sw_Int_TruncateMac(bCMAC);

		/* Copy the Truncated CMAC into the return buffer */
		memcpy(pRespMac, bCMAC, PHAL_MFIDENTITY_TRUNCATED_MAC_SIZE); /* PRQA S 3200 */

		/* Restore the Original  IV */
		memcpy(pDataParams->bIv, bTmpIV, PH_CRYPTOSYM_AES_BLOCK_SIZE);  /* PRQA S 3200 */

		/* Restore the Original Auth Session Master Key */
		if(pDataParams->bAuthMode != PHAL_MFIDENTITY_NOT_AUTHENTICATED)
		{
 			/* Load the session MAC key */
			PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadKeyDirect(
				pDataParams->pCryptoDataParamsMac,
				pDataParams->bKeySessionAuthMaster,
				pDataParams->bCryptoMethod
				));
        }
		return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
	}
	else
	{
		/* Generate Session keys as per AES Mode */
		/* Formation of SDM Ctr as double word value- SDM Ctr shall be communicated LSB first. */
		dwSDMctr = pSDMReadCtr[0];
		dwSDMctrtemp = pSDMReadCtr[1];
		dwSDMctr |= (dwSDMctrtemp << 8);
		dwSDMctrtemp = pSDMReadCtr[2];
		dwSDMctr |= (dwSDMctrtemp << 16);
		dwSDMctrtemp = pSDMReadCtr[3];
		dwSDMctr |= (dwSDMctrtemp << 24);

		/* If SDM Ctr is 0xFFFFFF, then return error */
		if (dwSDMctr == 0xFFFFFF)
		{
			return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_AL_MFIDENTITY);
		}

		/* Get Key out of the key store object */
		PH_CHECK_SUCCESS_FCT(statusTmp, phKeyStore_GetKey(
			pDataParams->pKeyStoreDataParams,
			wSDMMacKeyNo,
			wSDMMacKeyVer,
			PH_CRYPTOSYM_AES128_KEY_SIZE,
			bKey,
			&wKeyType
			));

		/* Invalid key type at wKeyNoTMACKey and wKeyVerTMACKey */
		if (wKeyType != PH_CRYPTOSYM_KEY_TYPE_AES128)
		{
			return PH_ADD_COMPCODE(PH_ERR_KEY, PH_COMP_AL_MFIDENTITY);
		}

		/* SV2 = 0x3C || 0xC3 || 0x00 || 0x01 || 0x00 || 0x80 [|| VCUID ][|| SDMReadCtr][||ZeroPadding] */
		/* Padding not required as we have 16 Bytes block */
		bSV[bSVMacLen++] = 0x3C;
		bSV[bSVMacLen++] = 0xC3;
		bSV[bSVMacLen++] = 0x00;
		bSV[bSVMacLen++] = 0x01;
		bSV[bSVMacLen++] = 0x00;
		bSV[bSVMacLen++] = 0x80;

		/* Copy UID into SV - UID should not more then 7 byte in
			* this case else bSV array size needs to be changed
			*/
		if(bSdmOption & PHAL_MFIDENTITY_VCUID_PRESENT)
		{
			/* If VCUID should be considered for MAC calcluation, and pUID is passed as NULL, throw error */
			memcpy(&bSV[bSVMacLen], pUid, bUidLen); /* PRQA S 3200 */
			bSVMacLen += bUidLen;
		}

		if(bSdmOption & PHAL_MFIDENTITY_RDCTR_PRESENT)
		{
			/* If SDMReadCounter should be considered for MAC calcluation, and pUID is passed as NULL, throw error */
			bSV[bSVMacLen++] = (uint8_t)(dwSDMctr & 0xFF);
			bSV[bSVMacLen++] = (uint8_t)((dwSDMctr >> 8) & 0xFF);
			bSV[bSVMacLen++] = (uint8_t)((dwSDMctr >> 16) & 0xFF);
		}

		/* SV padded with the zero bytes up to a length of 16 bytes (if needed)*/
		if(bSVMacLen < PH_CRYPTOSYM_AES128_KEY_SIZE)
		{
			memset(&bSV[bSVMacLen], 0x00, (PH_CRYPTOSYM_AES128_KEY_SIZE - bSVMacLen)); /* PRQA S 3200 */
		}

		/* load key */
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadKeyDirect(
			pDataParams->pCryptoDataParamsMac,
			bKey,
			wKeyType));

		/* Create a Back up of the current IV */
		memcpy(bTmpIV, pDataParams->bIv, PH_CRYPTOSYM_AES_BLOCK_SIZE); /* PRQA S 3200 */

		/* Load zero to IV */
		memset(pDataParams->bIv, 0x00, PH_CRYPTOSYM_AES_BLOCK_SIZE);  /* PRQA S 3200 */

		/* Load Iv */
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadIv(
			pDataParams->pCryptoDataParamsMac,
			pDataParams->bIv,
			PH_CRYPTOSYM_AES_BLOCK_SIZE
			));

		/* MAC SV to obtain KSesSDMFileReadMAC */
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_CalculateMac(
			pDataParams->pCryptoDataParamsMac,
			(PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_DEFAULT),
			bSV,
			PH_CRYPTOSYM_AES_BLOCK_SIZE,
			bKey,
			&bSVMacLen
			));

		/* load KSesSDMFileReadMAC */
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadKeyDirect(
			pDataParams->pCryptoDataParamsMac,
			bKey,
			PH_CRYPTOSYM_KEY_TYPE_AES128));

		/* Load zero to IV */
		memset(pDataParams->bIv, 0x00, PH_CRYPTOSYM_AES_BLOCK_SIZE);  /* PRQA S 3200 */

		/* Load Iv */
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadIv(
			pDataParams->pCryptoDataParamsMac,
			pDataParams->bIv,
			PH_CRYPTOSYM_AES_BLOCK_SIZE
			));

		/* Mac The Input Data using K(sessionSDMMacKey) to obtain SDMMac.  */
		PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_CalculateMac(
			pDataParams->pCryptoDataParamsMac,
			(PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_DEFAULT),
			pInData,
			wInDataLen,
			bCMAC,
			&bSVMacLen
			));

		/* Truncate the MAC generated */
		phalMfIdentity_Sw_Int_TruncateMac(bCMAC);

		/* Copy the Truncated CMAC into the return buffer */
		memcpy(pRespMac, bCMAC, PHAL_MFIDENTITY_TRUNCATED_MAC_SIZE); /* PRQA S 3200 */

		/* Restore back the IV */
		memcpy(pDataParams->bIv, bTmpIV, PH_CRYPTOSYM_AES_BLOCK_SIZE); /* PRQA S 3200 */

		if(pDataParams->bAuthMode != PHAL_MFIDENTITY_NOT_AUTHENTICATED)
		{
			/* Load the session key */
			PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_LoadKeyDirect(
				pDataParams->pCryptoDataParamsMac,
				pDataParams->bSesAuthMACKey,
				pDataParams->bCryptoMethod
				));
		}
	}
	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFIDENTITY);
}

phStatus_t phalMfIdentity_Sw_DecryptSDMENCFileData(phalMfIdentity_Sw_DataParams_t * pDataParams, uint16_t wOption, uint16_t wEncKeyNo,
	uint16_t wEncKeyVer, uint8_t * pDivInput, uint8_t bDivInputLen, uint8_t * pUid, uint8_t bUidLen, uint8_t * pSDMReadCtr, uint8_t * pEncdata,
	uint16_t wEncDataLen, uint8_t * pPlainData )
{
	phStatus_t  PH_MEMLOC_REM statusTmp = PH_ERR_SUCCESS;
	uint16_t    PH_MEMLOC_REM wAuthMode = 0x00;
	uint8_t		PH_MEMLOC_REM bSdmSessMacKey[16] = {'\0'};
	uint8_t		PH_MEMLOC_REM bSdmSessEncKey[16] = {'\0'};
	uint8_t		PH_MEMLOC_REM bIV[16] = {'\0'};
	uint8_t     wIvSize = 16;
	PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_GetConfig(
		pDataParams->pCryptoDataParamsMac,
		PH_CRYPTOSYM_CONFIG_LRP,
		&wAuthMode
	));
	pDivInput = NULL; /* To Avoid warning */
	bDivInputLen = 0x00; /* to avoid warning */
	wOption = 0x0000;

	if((wAuthMode == 0x01) || pDataParams->bKeyType)
	{
		/* Decrypt data in LRP Mode */
		/* Step-1: Generate Session Keys */
		PH_CHECK_SUCCESS_FCT(statusTmp, phalMfIdentity_Sw_Int_GenerateSDMSessionKeysLRP(pDataParams,
			wEncKeyNo,
			wEncKeyVer,
			pSDMReadCtr,
			pUid,
			bUidLen,
			bSdmSessEncKey,
			bSdmSessMacKey));

		/* Load zero to IV */
		memset(pDataParams->bIv, 0x00, PH_CRYPTOSYM_AES_BLOCK_SIZE);
		memcpy(bIV, pSDMReadCtr, 3);
		wIvSize = 6; /* SDMReadCtr || 0x000000*/
					 /* Step-3 : Decrypyt data */
		PH_CHECK_SUCCESS_FCT(statusTmp, phalMfIdentity_Sw_Int_DecryptSDMData(pDataParams,
			bSdmSessEncKey,
			bIV,
			wIvSize,
			pEncdata,
			wEncDataLen));

		/* Copy the OutPut Buffer to Plain data */
		memcpy(pPlainData, pEncdata, wEncDataLen);
	}
	else
	{
		/* Decrypt Data in AES Mode  */
		/* Step-1: Generate Session Keys */
		PH_CHECK_SUCCESS_FCT(statusTmp, phalMfIdentity_Sw_Int_GenerateSDMSessionKeysAES(pDataParams,
			wEncKeyNo,
			wEncKeyVer,
			pSDMReadCtr,
			pUid,
			bUidLen,
			bSdmSessEncKey,
			bSdmSessMacKey));

		/* Step-2 : Compute IV */
		PH_CHECK_SUCCESS_FCT(statusTmp, phalMfIdentity_Sw_Int_ComputeSDMIV(pDataParams,
			bSdmSessEncKey,
			pSDMReadCtr,
			bIV));

		/* Step-3 : Decrypyt data */
		PH_CHECK_SUCCESS_FCT(statusTmp, phalMfIdentity_Sw_Int_DecryptSDMData(pDataParams,
			bSdmSessEncKey,
			bIV,
			wIvSize,
			pEncdata,
			wEncDataLen));

		/* Copy the OutPut Buffer to Plain data */
		memcpy(pPlainData, pEncdata, wEncDataLen);
	}
	return PH_ADD_COMPCODE(statusTmp, PH_COMP_AL_MFIDENTITY);
}

phStatus_t phalMfIdentity_Sw_DecryptSDMPICCData(phalMfIdentity_Sw_DataParams_t * pDataParams, uint16_t wOption, uint16_t wKeyNo, uint16_t wKeyVer,
	uint8_t * pDivInput, uint8_t bDivInputLen, uint8_t * pEncdata, uint16_t wEncDataLen, uint8_t * pPlainData)
{
	phStatus_t  PH_MEMLOC_REM statusTmp =  PH_ERR_SUCCESS;
	uint8_t		PH_MEMLOC_REM bIV[PH_CRYPTOSYM_AES128_KEY_SIZE] = {'\0'};
	uint16_t    PH_MEMLOC_REM wAuthMode = 0x00;

	uint8_t   PH_MEMLOC_REM bKey[PH_CRYPTOSYM_AES128_KEY_SIZE];
	uint16_t  PH_MEMLOC_REM wKeyType;
	uint8_t wIvSize = 16;
	pDivInput =  NULL ; /* To Avoid warning */
	bDivInputLen =  0x00; /* to avoid warning */
	wOption =  PHAL_MFIDENTITY_NO_DIVERSIFICATION; /* Avoid Warnings */

	PH_CHECK_SUCCESS_FCT(statusTmp, phCryptoSym_GetConfig(
	pDataParams->pCryptoDataParamsMac,
	PH_CRYPTOSYM_CONFIG_LRP,
	&wAuthMode
	));

	/*  Get the Keys from SW Key Store */
	PH_CHECK_SUCCESS_FCT(statusTmp, phKeyStore_GetKey(
		pDataParams->pKeyStoreDataParams,
		wKeyNo,
		wKeyVer,
		PH_CRYPTOSYM_AES128_KEY_SIZE,
		bKey,
		&wKeyType
		));

	if (wKeyType != PH_CRYPTOSYM_KEY_TYPE_AES128)
	{
		return PH_ADD_COMPCODE(PH_ERR_KEY, PH_COMP_AL_MFIDENTITY);
	}

	if((wAuthMode == 0x01) || pDataParams->bKeyType)
	{
		/* Load zero to IV */
		memset(pDataParams->bIv, 0x00, PH_CRYPTOSYM_AES_BLOCK_SIZE);

		/* Copy First 8 bytes i.e PICCRand data to IV*/
        memcpy(bIV, pEncdata, 8);
		wIvSize = 8;
		/* Decrypt Data */
		PH_CHECK_SUCCESS_FCT(statusTmp, phalMfIdentity_Sw_Int_DecryptSDMData(pDataParams,
			bKey,
			bIV,
			wIvSize,
			pEncdata + 8,
			wEncDataLen - 8));

		/* Copy the OutPut Buffer to Plain data */
		memcpy(pPlainData, pEncdata+8, 16);
	}
	else {
		/* Load zero to IV */
        memset(pDataParams->bIv, 0x00, PH_CRYPTOSYM_AES_BLOCK_SIZE);

        /* Decrypt Data */
        PH_CHECK_SUCCESS_FCT(statusTmp, phalMfIdentity_Sw_Int_DecryptSDMData(pDataParams,
             bKey,
             bIV,
			 wIvSize,
             pEncdata,
             wEncDataLen));

		/* Copy the OutPut Buffer to Plain data */
		memcpy(pPlainData, pEncdata, wEncDataLen);

	}

	return PH_ADD_COMPCODE(statusTmp, PH_COMP_AL_MFIDENTITY);
}

#endif /*NXPBUILD__PH_CRYPTOSYM */

#endif /* NXPBUILD__PHAL_MFIDENTITY_SW */
