/*
 * Copyright 2022 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Generic Tools Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include "phToolsAtrParser.h"
#include <math.h>

uint8_t  Atr[PHHAL_HW_CONTACT_MAX_ATR_SIZE];
uint32_t AtrSize = 0;

phToolsAtrParser_GlobalParameters globalParams;
phToolsAtrParser_T0Parameters     t0Params;
phToolsAtrParser_T1Parameters     t1Params;

phStatus_t phToolsAtrParser_SetInternalAtr(
                                           uint8_t * pAtrBuffer,
                                           uint32_t  wAtrSize
                                           )
{
    phStatus_t statusTmp;

    /* Check size and nullptr */
    if(wAtrSize < 2 || wAtrSize > PHHAL_HW_CONTACT_MAX_ATR_SIZE || pAtrBuffer == NULL)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_TOOLS);
    }

    /* Copy to internal buffer */
    memcpy(&Atr[0], pAtrBuffer, wAtrSize);
    /* Check ATR and set default values */
    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_CheckInternalAtr());
    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_SetDefaultValues());

    /* Parse specified Parameters from ATR */
    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_ParseGlobalParameters());
    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_ParseT0Parameters());
    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_ParseT1Parameters());

    /* Everything passed so we set AtrSize */
    AtrSize = wAtrSize;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TOOLS);
}

phStatus_t phToolsAtrParser_CheckInternalAtr()
{
    /*TODOGL: implement this*/
    /* Check if values are in range according to ISO7816 */
    /* Check if length matches available bytes in ATR */
    /* Check TCK if available and correct */
    /* Check if protocols in numerical order */

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TOOLS);
}

phStatus_t phToolsAtrParser_ParseGlobalParameters()
{
    phStatus_t statusTmp;
    phStatus_t status;
    uint8_t bByteOffset = 0;
    uint8_t bTdByteOffset;
    uint8_t i = 0;
    uint8_t bVccFound = 0;

    /* First byte which always has to be present is TS */
    globalParams.bTS = Atr[PH_TOOLS_ATRPARSER_TS_OFFSET];

    /* Check if TA1 is available and get Fi & Di */
    if((Atr[PH_TOOLS_ATRPARSER_T0_OFFSET] & 0x10) == 0x10)
    {
        /* In TA2 If bit 5 is set to 0, then the integers Fi and Di defined above by TA1 shall apply. */
        /* In TA2 If bit 5 is set to 1, then implicit values (not defined by the interface bytes) shall apply. */
        globalParams.bFdValue = Atr[PH_TOOLS_ATRPARSER_T0_OFFSET + 1];
    }

    /* TB1 should be ignored if it is available */

    /* Parse GT for PPS which is the same as for T=0 */
    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_ParseGT(0, &(globalParams.bGTetu)));

    /* Get offset for T=15 to parse supported VCCs */
    status = phToolsAtrParser_GetProtocolTypeOffset(15, &bByteOffset);
    bTdByteOffset = bByteOffset;

    if((status & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        while(bTdByteOffset != 0)
        {
            /* Check if TA is available */
            if(((Atr[bTdByteOffset] >> 4) & 0x01) == 1)
            {
                bByteOffset++;
                if(!bVccFound)
                {
                    /* Bits 6 to 1 -> Supported VCCs and Bits 8 and 7 -> ClockStop */
                    globalParams.bSupportedVCCs = Atr[bByteOffset] & 0x07;
                    bVccFound = 1;
                }
            }
            if(((Atr[bTdByteOffset] >> 4) & 0x02) == 0x02)
            {
                bByteOffset++;
            }
            if(((Atr[bTdByteOffset] >> 4) & 0x04) == 0x04)
            {
                bByteOffset++;
            }
            if(((Atr[bTdByteOffset] >> 4) & 0x08) == 0x08)
            {
                bByteOffset++;

                /* Check if another TD is available */
                if((Atr[bByteOffset] & 0x0F) == 15)
                {
                    bTdByteOffset = bByteOffset;
                }
                else
                {
                    bTdByteOffset = 0;
                }
            }
            else
            {
                bTdByteOffset = 0;
            }
        }
    }

    /* Parse bSupportedProtocols */
    for(i = 1; i < PH_TOOLS_ATRPARSER_MAX_NUMBER_OF_PROTOCOLS; i++)
    {
        /* Try to get offsets for all protocol types */
        status = phToolsAtrParser_GetProtocolTypeOffset(i, &bByteOffset);

        if((status & PH_ERR_MASK) == PH_ERR_SUCCESS)
        {
            globalParams.bSupportedProtocols[i] = 1;
        }
        else
        {
            globalParams.bSupportedProtocols[i] = 0;
        }
    }

    /* check if next TD1 is available */
    bByteOffset = 0;
    if((Atr[PH_TOOLS_ATRPARSER_T0_OFFSET] >> 7) & 0x01)
    {
        /* calculate offset to TD1 */
        bByteOffset += 1 +
            ((Atr[PH_TOOLS_ATRPARSER_T0_OFFSET + bByteOffset] >> 4) & 0x01) +
            ((Atr[PH_TOOLS_ATRPARSER_T0_OFFSET + bByteOffset] >> 5) & 0x01) +
            ((Atr[PH_TOOLS_ATRPARSER_T0_OFFSET + bByteOffset] >> 6) & 0x01);

        /* Set Protocol of TD1 to supported */
        globalParams.bSupportedProtocols[(Atr[PH_TOOLS_ATRPARSER_T0_OFFSET + bByteOffset] & 0x0F)] = 1;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TOOLS);
}

phStatus_t phToolsAtrParser_ParseT0Parameters()
{
    phStatus_t statusTmp;
    uint8_t bByteOffset = 0;

    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_ParseGT(0, &(t0Params.bGTetu)));

    /* check if next TD1 is available */
    if((Atr[PH_TOOLS_ATRPARSER_T0_OFFSET] >> 7) & 0x01)
    {
        /* calculate offset to TD1 */
        bByteOffset += 1 + PH_TOOLS_ATRPARSER_T0_OFFSET +
            ((Atr[PH_TOOLS_ATRPARSER_T0_OFFSET + bByteOffset] >> 4) & 0x01) +
            ((Atr[PH_TOOLS_ATRPARSER_T0_OFFSET + bByteOffset] >> 5) & 0x01) +
            ((Atr[PH_TOOLS_ATRPARSER_T0_OFFSET + bByteOffset] >> 6) & 0x01);

        /* Check if TC is available and parse WI */
        if((Atr[bByteOffset] & 0x40) == 0x40)
        {
            bByteOffset += 1 +
                ((Atr[bByteOffset] >> 4) & 0x01) +
                ((Atr[bByteOffset] >> 5) & 0x01);

            t0Params.bWI = Atr[bByteOffset];
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TOOLS);
}

phStatus_t phToolsAtrParser_ParseT1Parameters()
{
    phStatus_t statusTmp;
    phStatus_t status;
    uint8_t bByteOffset;
    uint8_t bTdByteOffset;
    uint8_t bIfscFound = 0, bCwiBwiFound = 0, bRedCodeFound = 0;

    /* Parse CGT (CGT=GT for T=1) */
    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_ParseGT(1, &(t1Params.bCGTetu)));

    status = phToolsAtrParser_GetProtocolTypeOffset(1, &bByteOffset);
    bTdByteOffset = bByteOffset;

    if((status & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        while(bTdByteOffset != 0)
        {
            if(((Atr[bTdByteOffset] >> 4) & 0x01) == 0x01)
            {
                bByteOffset++;
                if(!bIfscFound)
                {
                    t1Params.bIFSC = Atr[bByteOffset];
                    bIfscFound = 1;
                }
            }
            if(((Atr[bTdByteOffset] >> 4) & 0x02) == 0x02)
            {
                bByteOffset++;
                if(!bCwiBwiFound)
                {
                    t1Params.bCWI = Atr[bByteOffset] & 0x0F;
                    t1Params.bBWI = Atr[bByteOffset] >> 4;
                    bCwiBwiFound = 1;
                }
            }
            if(((Atr[bTdByteOffset] >> 4) & 0x04) == 0x04)
            {
                bByteOffset++;
                if(!bRedCodeFound)
                {
                    t1Params.bRedundancyCode = (uint8_t)(Atr[bByteOffset]&0x01);
                    bRedCodeFound = 1;
                }
            }
            if(((Atr[bTdByteOffset] >> 4) & 0x08) == 0x08)
            {
                bByteOffset++;

                /* Check if another TD is available */
                if((Atr[bByteOffset] & 0x0F) == 1)
                {
                    bTdByteOffset = bByteOffset;
                }
                else
                {
                    bTdByteOffset = 0;
                }
            }
            else
            {
                bTdByteOffset = 0;
            }
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TOOLS);
}

phStatus_t phToolsAtrParser_GetProtocolTypeOffset(
                                                  uint8_t bProtocolType,
                                                  uint8_t * pOffsetTD
                                                  )
{
    uint8_t bTD = 1;
    uint8_t bByteOffset = 0;
    uint8_t bTdCount = 0;

    /* Start the search from TD2 on because former ones are global parameters */
    while(bTD)
    {
        if((Atr[PH_TOOLS_ATRPARSER_T0_OFFSET + bByteOffset] & 0x0F) == bProtocolType && bTdCount >= 2)
        {
            *pOffsetTD = PH_TOOLS_ATRPARSER_T0_OFFSET + bByteOffset;
            return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TOOLS);
        }

        /* check if next TD is available */
        bTD = (Atr[PH_TOOLS_ATRPARSER_T0_OFFSET + bByteOffset] >> 7) & 0x01;

        /* calculate offset to next TD */
        bByteOffset += bTD +
        ((Atr[PH_TOOLS_ATRPARSER_T0_OFFSET + bByteOffset] >> 4) & 0x01) +
        ((Atr[PH_TOOLS_ATRPARSER_T0_OFFSET + bByteOffset] >> 5) & 0x01) +
        ((Atr[PH_TOOLS_ATRPARSER_T0_OFFSET + bByteOffset] >> 6) & 0x01);

        bTdCount++;

        if(bByteOffset > PHHAL_HW_CONTACT_MAX_ATR_SIZE - PH_TOOLS_ATRPARSER_T0_OFFSET)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_TOOLS);
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_TOOLS);
}

phStatus_t phToolsAtrParser_ParseGT(
                                    uint8_t bProtocolType,
                                    uint8_t * pGuardTimeEtu
                                    )
{
    uint8_t bTC1Offset = 0;

    /* Check if TC1 is available and get GT */
    if((Atr[PH_TOOLS_ATRPARSER_T0_OFFSET] & 0x40) == 0x40)
    {
        bTC1Offset += ((Atr[PH_TOOLS_ATRPARSER_T0_OFFSET] >> 4) & 0x01) +
                      ((Atr[PH_TOOLS_ATRPARSER_T0_OFFSET] >> 5) & 0x01) + 1;

        if(Atr[PH_TOOLS_ATRPARSER_T0_OFFSET + bTC1Offset] >= 0 && Atr[PH_TOOLS_ATRPARSER_T0_OFFSET + bTC1Offset] <= 254)
        {
            /* formula according to ISO is 12 + R * Atr[PH_TOOLS_ATRPARSER_T0_OFFSET + bTC1Offset] / f . 12 + x etu */
            /* If T=15 is absent in the Answer-to-Reset, then R = F / D, i.e., the integers used for computing the etu. */
            /* If T=15 is present in the Answer-to-Reset, then R = Fi / Di, i.e., the integers defined above by TA1. */
            *pGuardTimeEtu = 12 + Atr[PH_TOOLS_ATRPARSER_T0_OFFSET + bTC1Offset];
        }
        else
        {
            /* Protocol dependent */
            switch(bProtocolType)
            {
            case 0:
                *pGuardTimeEtu = 12;
                break;

            case 1:
                *pGuardTimeEtu = 11;
                break;

            default:
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_TOOLS);
            }
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TOOLS);
}

phStatus_t phToolsAtrParser_SetDefaultValues()
{
    uint8_t i = 0;

    /* Reset ATR length as just default values are applied */
    AtrSize = 0;

    /* Set default Values (if available in ATR they are overwritten later on) */
    globalParams.bTS = 0x3B;
    globalParams.bFdValue = 0x11;
    globalParams.bGTetu = 12;
    globalParams.bSupportedVCCs = 1;

    t0Params.bGTetu = 12;
    t0Params.bWI = 10;

    t1Params.bCGTetu = 12;
    t1Params.bIFSC = 32;
    t1Params.bCWI = 13;
    t1Params.bBWI = 4;
    t1Params.bRedundancyCode = 0;

    for(i = 0; i < PH_TOOLS_ATRPARSER_MAX_NUMBER_OF_PROTOCOLS; i++)
    {
        globalParams.bSupportedProtocols[i] = 0;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TOOLS);
}

phStatus_t phToolsAtrParser_GetT0_GTclk(
                                        uint32_t * pGTclk,
                                        uint32_t bFdIndexValue
                                        )
{
    phStatus_t statusTmp;
    uint16_t wFValue;
    uint16_t wDValue;

    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_ConvertFdIndexesToValues(bFdIndexValue, &wFValue, &wDValue));
    *pGTclk = t0Params.bGTetu * wFValue / wDValue;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TOOLS);
}

phStatus_t phToolsAtrParser_GetT0_WI(
                                     uint8_t * pWI
                                     )
{
    *pWI = t0Params.bWI;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TOOLS);
}

phStatus_t phToolsAtrParser_GetT0_WTclk(
                                        uint32_t * pWTclk,
                                        uint32_t bFdIndexValue
                                        )
{
    phStatus_t statusTmp;
    uint8_t bConfig;
    uint16_t wFValue;
    uint16_t wDValue;

    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_GetT0_WI(&bConfig));
    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_ConvertFdIndexesToValues(bFdIndexValue, &wFValue, &wDValue));
    *pWTclk = bConfig * 960 * wFValue;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TOOLS);
}

phStatus_t phToolsAtrParser_GetT1_CGTclk(
                                         uint32_t * pCGTetu,
                                         uint32_t bFdIndexValue
                                         )
{
    phStatus_t statusTmp;
    uint16_t wFValue;
    uint16_t wDValue;

    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_ConvertFdIndexesToValues(bFdIndexValue, &wFValue, &wDValue));
    *pCGTetu = t1Params.bCGTetu * wFValue / wDValue;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TOOLS);
}

phStatus_t phToolsAtrParser_GetT1_IFSC(
                                       uint8_t * pIFSC
                                       )
{
    *pIFSC = t1Params.bIFSC;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TOOLS);
}

phStatus_t phToolsAtrParser_GetT1_CWI(
                                      uint8_t * pCWI
                                      )
{
    *pCWI = t1Params.bCWI;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TOOLS);
}

phStatus_t phToolsAtrParser_GetT1_BWI(
                                      uint8_t * pBWI
                                      )
{
    *pBWI = t1Params.bBWI;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TOOLS);
}

phStatus_t phToolsAtrParser_GetT1_RedundancyMode(
                                                 uint8_t * pRedundancyMode
                                                 )
{
    switch(t1Params.bRedundancyCode)
    {
    case 0:
        *pRedundancyMode = PHHAL_HW_CONTACT_MODE_LRC;
        break;

    case 1:
        *pRedundancyMode = PHHAL_HW_CONTACT_MODE_CRC;
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_TOOLS);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TOOLS);
}

phStatus_t phToolsAtrParser_GetT1_CWTclk(
                                         uint32_t * pCWTclk,
                                         uint32_t bFdIndexValue
                                         )
{
    phStatus_t statusTmp;
    uint16_t wFValue;
    uint16_t wDValue;
    uint8_t bConfig;

    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_ConvertFdIndexesToValues(bFdIndexValue, &wFValue, &wDValue));
    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_GetT1_CWI(&bConfig));

    *pCWTclk = (uint32_t)(11 * (wFValue / wDValue) + pow(2, bConfig) * wFValue / wDValue);
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TOOLS);
}

phStatus_t phToolsAtrParser_GetT1_BWTclk(
                                         uint32_t * pBWTclk,
                                         uint32_t bFdIndexValue
                                         )
{
    phStatus_t statusTmp;
    uint16_t wFValue;
    uint16_t wDValue;
    uint8_t bConfig;

    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_ConvertFdIndexesToValues(bFdIndexValue, &wFValue, &wDValue));
    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_GetT1_BWI(&bConfig));

    *pBWTclk = (uint32_t)(11 * (wFValue / wDValue) + pow(2, bConfig) * 960 * PH_TOOLS_ATRPARSER_DEFAULT_F_VALUE);
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TOOLS);
}

phStatus_t phToolsAtrParser_GetGlobal_TS(
                                         uint8_t * pTS
                                         )
{
    *pTS = globalParams.bTS;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TOOLS);
}

phStatus_t phToolsAtrParser_GetGlobal_FD(
                                         uint8_t * pFD
                                         )
{
    *pFD = globalParams.bFdValue;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TOOLS);
}

phStatus_t phToolsAtrParser_GetGlobal_GTclk(
                                            uint32_t * pGTclk,
                                            uint32_t bFdIndexValue
                                            )
{
    phStatus_t statusTmp;
    uint16_t wFValue;
    uint16_t wDValue;

    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_ConvertFdIndexesToValues(bFdIndexValue, &wFValue, &wDValue));

    *pGTclk = globalParams.bGTetu * wFValue / wDValue;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TOOLS);
}

phStatus_t phToolsAtrParser_GetGlobal_SupportedVCCs(
                                                    uint8_t * pSupportedVCCs
                                                    )
{
    *pSupportedVCCs = globalParams.bSupportedVCCs;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TOOLS);
}

phStatus_t phToolsAtrParser_GetGlobal_CheckIfProtocolSupported(
                                                               uint8_t bProtocolType,
                                                               uint8_t * pSupported
                                                               )
{
    if(bProtocolType == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_TOOLS);
    }

    /* To be compatible with HalContact definitions as they start with T=0 -> 1 */
    bProtocolType--;

    if(bProtocolType < 0 || bProtocolType > PH_TOOLS_ATRPARSER_MAX_NUMBER_OF_PROTOCOLS)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_TOOLS);
    }

    *pSupported = globalParams.bSupportedProtocols[bProtocolType];
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TOOLS);
}

phStatus_t phToolsAtrParser_GetGlobal_Convention(
                                                 uint8_t * pConvention
                                                 )
{
    phStatus_t statusTmp;
    uint8_t wConfig;

    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_GetGlobal_TS(&wConfig));

    switch(wConfig)
    {
    case 0x3F:
        *pConvention = PHHAL_HW_CONTACT_INVERSE_CONVENTION;
        break;

    case 0x3B:
        *pConvention = PHHAL_HW_CONTACT_DIRECT_CONVENTION;
        break;

    /* No ATR recieved yet which is fine but return 0x00 */
    case 0x00:
        *pConvention = 0x00;
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_TOOLS);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TOOLS);
}

phStatus_t phToolsAtrParser_ConvertFdIndexesToValues(
                                                     uint32_t bFdIndexes,
                                                     uint16_t * fValue,
                                                     uint16_t * dValue
                                                     )
{
    switch(bFdIndexes & 0x0F)
    {
    case 0:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_TOOLS);

    case 1:
        *dValue = 1;
        break;

    case 2:
        *dValue = 2;
        break;

    case 3:
        *dValue = 4;
        break;

    case 4:
        *dValue = 8;
        break;

    case 5:
        *dValue = 16;
        break;

    case 6:
        *dValue = 32;
        break;

    case 7:
        *dValue = 64;
        break;

    case 8:
        *dValue = 12;
        break;

    case 9:
        *dValue = 20;
        break;

    case 10:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_TOOLS);

    case 11:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_TOOLS);

    case 12:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_TOOLS);

    case 13:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_TOOLS);

    case 14:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_TOOLS);

    case 15:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_TOOLS);
    }

    switch(bFdIndexes >> 4)
    {
    case 0:
        *fValue = 372;
        break;

    case 1:
        *fValue = 372;
        break;

    case 2:
        *fValue = 558;
        break;

    case 3:
        *fValue = 744;
        break;

    case 4:
        *fValue = 1116;
        break;

    case 5:
        *fValue = 1488;
        break;

    case 6:
        *fValue = 1860;
        break;

    case 7:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_TOOLS);

    case 8:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_TOOLS);

    case 9:
        *fValue = 512;
        break;

    case 10:
        *fValue = 768;
        break;

    case 11:
        *fValue = 1024;
        break;

    case 12:
        *fValue = 1536;
        break;

    case 13:
        *fValue = 2048;
        break;

    case 14:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_TOOLS);

    case 15:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_TOOLS);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TOOLS);
}

phStatus_t phToolsAtrParser_GetAtrLength(
                                         uint8_t * bAtrLength
                                         )
{
    *bAtrLength = (uint8_t)AtrSize;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_TOOLS);
}