/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal logic file of SAM (AV4 and future SAM's) KeyStore component.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <ph_RefDefs.h>
#include <phhalHw.h>

#ifdef NXPBUILD__PH_KEYSTORE_SAM

#include "phKeyStore_Sam.h"
#include "phKeyStore_Sam_Int.h"

#include "../../comps/phhalHw/src/Sam/phhalHw_Sam.h"
#include "../../comps/phhalHw/src/Sam/Commands/03_KeyManagement/phhalHw_Sam_Cmd_KM.h"

phStatus_t phKeyStore_Sam_Int_GetKeyEntry(phKeyStore_Sam_DataParams_t * pDataParams, uint8_t bKeyNo,
    uint8_t bIsRamKey, phKeyStore_Sam_KeyEntry_t * pKeyEntry)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bKeyEntryOffset = 0;

    uint8_t *  PH_MEMLOC_REM pKeyEntryBuff = NULL;
    uint16_t   PH_MEMLOC_REM wKeyEntryLen = 0;

    /* Get the key entry information from Sam.  */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_SAM_GetKeyEntry(
        pDataParams->pHalDataParams,
        bKeyNo,
        (uint8_t) !bIsRamKey,
        &pKeyEntryBuff,
        &wKeyEntryLen));

    /* Clear the Key data member. */
    memset(pKeyEntry->aKeyData, 0x00, sizeof(pKeyEntry->aKeyData));

    /* Update the version members. */
    switch(wKeyEntryLen)
    {
        case PH_KEYSTORE_SAM_KEY_ENTRY_LEN_SAMAV2_FORMAT_VER_ABC:
        case PH_KEYSTORE_SAM_KEY_ENTRY_LEN_SAM_FORMAT_VER_ABC:
            pKeyEntry->bVersionKeyA = pKeyEntryBuff[bKeyEntryOffset++];
            pKeyEntry->bVersionKeyB = pKeyEntryBuff[bKeyEntryOffset++];
            pKeyEntry->bVersionKeyC = pKeyEntryBuff[bKeyEntryOffset++];

            pKeyEntry->bVersionKeyBValid = PH_ON;
            pKeyEntry->bVersionKeyCValid = PH_ON;
            break;

        case PH_KEYSTORE_SAM_KEY_ENTRY_LEN_SAMAV2_FORMAT_VER_AB:
        case PH_KEYSTORE_SAM_KEY_ENTRY_LEN_SAM_FORMAT_VER_AB:
            pKeyEntry->bVersionKeyA = pKeyEntryBuff[bKeyEntryOffset++];
            pKeyEntry->bVersionKeyB = pKeyEntryBuff[bKeyEntryOffset++];
            pKeyEntry->bVersionKeyC = 0x00;

            pKeyEntry->bVersionKeyBValid = PH_ON;
            pKeyEntry->bVersionKeyCValid = PH_OFF;
            break;

        case PH_KEYSTORE_SAM_KEY_ENTRY_LEN_SAM_FORMAT_VER_A:
            pKeyEntry->bVersionKeyA = pKeyEntryBuff[bKeyEntryOffset++];
            pKeyEntry->bVersionKeyB = 0x00;
            pKeyEntry->bVersionKeyC = 0x00;

            pKeyEntry->bVersionKeyBValid = PH_OFF;
            pKeyEntry->bVersionKeyCValid = PH_OFF;
            break;

        case PH_KEYSTORE_SAM_KEY_ENTRY_LEN_SAM_RAM_KEY:
            pKeyEntry->bVersionKeyA = 0x00;
            pKeyEntry->bVersionKeyB = 0x00;
            pKeyEntry->bVersionKeyC = 0x00;

            pKeyEntry->bVersionKeyBValid = PH_OFF;
            pKeyEntry->bVersionKeyCValid = PH_OFF;
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_KEYSTORE);
    }

    /* Update DF_AID and DF_KeyNo members. */
    if(bIsRamKey == (uint8_t) PH_OFF)
    {
        memcpy(pKeyEntry->aDFAid, &pKeyEntryBuff[bKeyEntryOffset], sizeof(pKeyEntry->aDFAid));
        bKeyEntryOffset += (uint8_t) sizeof(pKeyEntry->aDFAid);

        pKeyEntry->bDFKeyNo = pKeyEntryBuff[bKeyEntryOffset++];
    }

    /* Update KeyNo and KeyV members for Change Entry Key. */
    pKeyEntry->bKeyNoCEK = pKeyEntryBuff[bKeyEntryOffset++];
    pKeyEntry->bKeyVCEK = pKeyEntryBuff[bKeyEntryOffset++];

    /* Update member for Reference number of Key Usage Counter. */
    pKeyEntry->bRefNoKUC = pKeyEntryBuff[bKeyEntryOffset++];

    /* Update SET configuration member. */
    memcpy(pKeyEntry->aSet, &pKeyEntryBuff[bKeyEntryOffset], sizeof(pKeyEntry->aSet));
    bKeyEntryOffset += (uint8_t) sizeof(pKeyEntry->aSet);

    /* Update ExtSET configuration member. */
    pKeyEntry->aExtSet[0U] = pKeyEntryBuff[bKeyEntryOffset++];
    pKeyEntry->aExtSet[1U] = 0x00;

    /* Update ExtSET configuration member. */
    pKeyEntry->aExtSet[1U] = pKeyEntryBuff[bKeyEntryOffset++];

    /* Update KeyNo and KeyV members of Access Entry Key. */
    pKeyEntry->bKeyNoAEK = pKeyEntryBuff[bKeyEntryOffset++];
    pKeyEntry->bKeyVAEK = pKeyEntryBuff[bKeyEntryOffset++];

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_KEYSTORE);
}

phStatus_t phKeyStore_Sam_Int_ConvertKeyEntryToBuffer(phKeyStore_Sam_KeyEntry_t * pKeyEntry,
    uint8_t * pKey, uint8_t bKeyLen, uint8_t * pKeyEntryBuff, uint8_t * pKeyEntryBuffLen)
{
    uint8_t    PH_MEMLOC_REM bKeyEntryBuffLen = 0;

    /* Reset the Key Entry buffer. */
    memset(pKeyEntryBuff, 0x00, PH_KEYSTORE_KEY_ENTRY_SIZE_MAX);

    /* Copy the Key information to KeyEntry buffer. */
    memcpy(pKeyEntryBuff, pKey, bKeyLen);
    bKeyEntryBuffLen += (uint8_t) (PH_KEYSTORE_KEY_TYPE_AES128_SIZE * 3U);

    /* Copy DF_AID to KeyEntry buffer. */
    memcpy(&pKeyEntryBuff[bKeyEntryBuffLen], pKeyEntry->aDFAid, sizeof(pKeyEntry->aDFAid));
    bKeyEntryBuffLen += (uint8_t) sizeof(pKeyEntry->aDFAid);

    /* Copy DF_KeyNo to KeyEntry buffer. */
    pKeyEntryBuff[bKeyEntryBuffLen++] = pKeyEntry->bDFKeyNo;

    /* Copy KeyNo and KeyVer of Change Entry Key to KeyEntry buffer. */
    pKeyEntryBuff[bKeyEntryBuffLen++] = pKeyEntry->bKeyNoCEK;
    pKeyEntryBuff[bKeyEntryBuffLen++] = pKeyEntry->bKeyVCEK;

    /* Copy Reference number of Key Usage Counter to KeyEntry buffer. */
    pKeyEntryBuff[bKeyEntryBuffLen++] = pKeyEntry->bRefNoKUC;

    /* Copy SET configuration to KeyEntry buffer. */
    memcpy(&pKeyEntryBuff[bKeyEntryBuffLen], pKeyEntry->aSet, sizeof(pKeyEntry->aSet));
    bKeyEntryBuffLen += (uint8_t) sizeof(pKeyEntry->aSet);

    /* Copy Versions to KeyEntry buffer. */
    pKeyEntryBuff[bKeyEntryBuffLen++] = pKeyEntry->bVersionKeyA;
    pKeyEntryBuff[bKeyEntryBuffLen++] = pKeyEntry->bVersionKeyB;
    pKeyEntryBuff[bKeyEntryBuffLen++] = pKeyEntry->bVersionKeyC;

    /* Copy ExtSET configuration to KeyEntry buffer. */
    pKeyEntryBuff[bKeyEntryBuffLen++] = pKeyEntry->aExtSet[0];

    /* Copy ExtSET configuration */
    pKeyEntryBuff[bKeyEntryBuffLen++] = pKeyEntry->aExtSet[1];

    /* Copy AccessKey Number and Version */
    pKeyEntryBuff[bKeyEntryBuffLen++] = pKeyEntry->bKeyNoAEK;
    pKeyEntryBuff[bKeyEntryBuffLen++] = pKeyEntry->bKeyVAEK;

    /* Update the Buffer length parameter. */
    *pKeyEntryBuffLen = bKeyEntryBuffLen;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_KEYSTORE);
}

phStatus_t phKeyStore_Sam_Int_SetKeyType(phKeyStore_Sam_KeyEntry_t * pKeyEntry, uint16_t wKeyType,
    uint8_t b2K3DESOption, uint8_t bIsLRPKey)
{
    switch(wKeyType)
    {
        case PH_KEYSTORE_KEY_TYPE_AES128:
            if(bIsLRPKey)
            {
                pKeyEntry->aSet[0] |= (uint8_t) (PH_KEYSTORE_SAM_KEYTYPE_LRP_AES128_MASK << 3U);
            }
            else
            {
                pKeyEntry->aSet[0] |= (uint8_t) (PH_KEYSTORE_SAM_KEYTYPE_AES128_MASK << 3U);
            }

            pKeyEntry->bVersionKeyBValid = 0x01U;
            pKeyEntry->bVersionKeyCValid = 0x01U;
            break;

        case PH_KEYSTORE_KEY_TYPE_AES192:
            pKeyEntry->aSet[0] |= (uint8_t) (PH_KEYSTORE_SAM_KEYTYPE_AES192_MASK << 3U);
            pKeyEntry->bVersionKeyBValid = 0x01U;
            pKeyEntry->bVersionKeyCValid = 0x00U;
            break;

        case PH_KEYSTORE_KEY_TYPE_AES256:
            pKeyEntry->aSet[0] |= (uint8_t) (PH_KEYSTORE_SAM_KEYTYPE_AES256_MASK << 3U);
            pKeyEntry->bVersionKeyBValid = 0x00U;
            pKeyEntry->bVersionKeyCValid = 0x00U;
            break;

        case PH_KEYSTORE_KEY_TYPE_DES:
        case PH_KEYSTORE_KEY_TYPE_2K3DES:
            if(b2K3DESOption == PH_KEYSTORE_SAM_DES_OPTION_DESFIRE4)
            {
                pKeyEntry->aSet[0] |= (uint8_t) (PH_KEYSTORE_SAM_KEYTYPE_3DESDF4_MASK << 3U);
            }
            else if(b2K3DESOption == PH_KEYSTORE_SAM_DES_OPTION_ISO_CRC16)
            {
                pKeyEntry->aSet[0] |= (uint8_t) (PH_KEYSTORE_SAM_KEYTYPE_2K3DES_MASK << 3U);
            }
            else if(b2K3DESOption == PH_KEYSTORE_SAM_DES_OPTION_ISO_CRC32)
            {
                pKeyEntry->aSet[0] |= (uint8_t) (PH_KEYSTORE_SAM_KEYTYPE_2K3DESDF8_MASK << 3U);
            }
            else
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_KEYSTORE);
            }

            pKeyEntry->bVersionKeyBValid = 0x01U;
            pKeyEntry->bVersionKeyCValid = 0x01U;
            break;

        case PH_KEYSTORE_KEY_TYPE_3K3DES:
            pKeyEntry->aSet[0] |= (uint8_t) (PH_KEYSTORE_SAM_KETYPE_3K3DES_MASK << 3U);
            pKeyEntry->bVersionKeyBValid = 0x01U;
            pKeyEntry->bVersionKeyCValid = 0x00U;
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_KEYSTORE);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_KEYSTORE);
}

phStatus_t phKeyStore_Sam_Int_GetKeyType(phKeyStore_Sam_KeyEntry_t * pKeyEntry, uint16_t * pKeyType,
    uint8_t * pIsLRPKey)
{
    /* Get the KeyType loaded to KeyStore. */
    switch(((pKeyEntry->aSet[0] & PH_KEYSTORE_SAM_KEYTYPE_MASK) >> 3U))
    {
        case PH_KEYSTORE_SAM_KEYTYPE_AES128_MASK:
            *pKeyType = PH_KEYSTORE_KEY_TYPE_AES128;
            *pIsLRPKey = PH_OFF;
            break;

        case PH_KEYSTORE_SAM_KEYTYPE_LRP_AES128_MASK:
            *pKeyType = PH_KEYSTORE_KEY_TYPE_AES128;
            *pIsLRPKey = PH_ON;
            break;

        case PH_KEYSTORE_SAM_KEYTYPE_AES192_MASK:
            *pKeyType = PH_KEYSTORE_KEY_TYPE_AES192;
            break;

        case PH_KEYSTORE_SAM_KEYTYPE_AES256_MASK:
            *pKeyType = PH_KEYSTORE_KEY_TYPE_AES256;
            break;

        case PH_KEYSTORE_SAM_KEYTYPE_2K3DES_MASK:
            *pKeyType = PH_KEYSTORE_KEY_TYPE_2K3DES;
            break;

        case PH_KEYSTORE_SAM_KETYPE_3K3DES_MASK:
            *pKeyType = PH_KEYSTORE_KEY_TYPE_3K3DES;
            break;

        case PH_KEYSTORE_SAM_KEYTYPE_3DESDF4_MASK:
            *pKeyType = PH_KEYSTORE_KEY_TYPE_2K3DES;
            break;

        case PH_KEYSTORE_SAM_KEYTYPE_2K3DESDF8_MASK:
            *pKeyType = PH_KEYSTORE_KEY_TYPE_2K3DES;
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_KEYSTORE);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_KEYSTORE);
}

phStatus_t phKeyStore_Sam_Int_GetKeySize(uint16_t wKeyType, uint8_t * pKeySize)
{
    switch(wKeyType)
    {
        case PH_KEYSTORE_KEY_TYPE_AES128:
        case PH_KEYSTORE_KEY_TYPE_DES:
        case PH_KEYSTORE_KEY_TYPE_2K3DES:
            *pKeySize = 16U;
            break;

        case PH_KEYSTORE_KEY_TYPE_AES192:
        case PH_KEYSTORE_KEY_TYPE_3K3DES:
            *pKeySize = 24U;
            break;

        case PH_KEYSTORE_KEY_TYPE_AES256:
            *pKeySize = 32U;
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_KEYSTORE);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_KEYSTORE);
}

#endif /* NXPBUILD__PH_KEYSTORE_SAM */
