/*
 * Copyright 2013, 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * DES implementation of the Symmetric Cryptography Library.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHCRYPTOSYM_SW_DES_H
#define PHCRYPTOSYM_SW_DES_H

#include <ph_Status.h>
#include <ph_RefDefs.h>
#include <phCryptoSym.h>

/** \defgroup phCryptoSym_Sw_DES Cipher : DES
* \brief Software implementation of the DES algorithm
* @{
*/

/**
* \brief Performs a decryption of a DES Block (8 byte) using a DES Key.
* The key needs to be provided in the pDataParams structure. By specifying the bKeyNumber, either the
* first, the second or the third key is used.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phCryptoSym_Sw_Des_DecryptBlock(
    phCryptoSym_Sw_DataParams_t * pDataParams,  /**< [In] Pointer to this layers parameter structure. */
    uint8_t PH_CRYTOSYM_SW_FAST_RAM * pBlock,   /**< [InOut] Block to be enciphered in place. */
    uint8_t bKeyNumber                          /**< [In] Number of the key to be used (0, 1 or 2) */
    );

/**
* \brief Performs an encryption of a DES Block (8 byte) using a DES Key.
* The key needs to be provided in the pDataParams structure. By specifying the bKeyNumber, either the
* first, the second or the third key is used.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phCryptoSym_Sw_Des_EncryptBlock(
    phCryptoSym_Sw_DataParams_t * pDataParams,  /**< [In] Pointer to this layers parameter structure. */
    uint8_t PH_CRYTOSYM_SW_FAST_RAM * pBlock,   /**< [InOut] Block to be enciphered in place. */
    uint8_t bKeyNumber                          /**< [In] Number of the key to be used (0, 1 or 2) */
    );

/**
* \brief Performs the key Expansion of the DES key(s)
* According to the number of keys specified, the pDataParams key array is filled up in the following sequence:
* Expanded KEY1 [128] || Expanded KEY2 [128] || Expanded KEY3 [128]
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phCryptoSym_Sw_Des_KeyInit(
                                      phCryptoSym_Sw_DataParams_t * pDataParams,    /**< [In] Pointer to this layers parameter structure. */
                                      const uint8_t * pKey,                         /**< [In] Key array to be loaded into the DES engine. */
                                      uint8_t bNumKeys                              /**< [In] Amount of keys provided (1, 2 or 3) */
                                      );

/**
* \brief Decode the KeyVersion of a DES key.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phCryptoSym_Sw_Des_DecodeVersion(
    uint8_t * pKey,         /**< [In] Key to decode. */
    uint16_t * pKeyVersion  /**< [Out] Version of the key. */
    );

/**
* \brief Encode the KeyVersion in a DES key.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_INVALID_PARAMETER Operation aborted.
* \retval Other Depending on implementation and underlaying component.
*/
phStatus_t phCryptoSym_Sw_Des_EncodeVersion(
    uint8_t * pKey,         /**< [In] Key to encode. */
    uint16_t wKeyVersion,   /**< [In] Version of the key. */
    uint16_t wKeyType,      /**< [In] Type of Key. */
    uint8_t * pEncodedKey   /**< [Out] Encoded key. */
    );

/** @}
* end of phCryptoSym_Sw_DES group
*/

#endif /* PHCRYPTOSYM_SW_DES_H */
