/*
 * Copyright 2022 - 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * DUT specific Symmetric Cryptography Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PH_CRYPTOSYM_DUT

#include <phCryptoRng.h>
#include <phhalHw_DUT_Cmd.h>
#include "phCryptoSym_DUT.h"

phStatus_t phCryptoSym_DUT_Init(phCryptoSym_DUT_DataParams_t * pDataParams, uint16_t wSizeOfDataParams, void * pHalDataParams, uint8_t * pTxBuffer,
    uint16_t wTxBuffSize, uint8_t * pRxBuffer, uint16_t wRxBuffSize)
{
    if(sizeof(phCryptoSym_DUT_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_CRYPTOSYM);
    }
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_CRYPTOSYM);
    PH_ASSERT_NULL_DATA_PARAM(pTxBuffer, PH_COMP_CRYPTOSYM);
    PH_ASSERT_NULL_DATA_PARAM(pRxBuffer, PH_COMP_CRYPTOSYM);

    /* Validate the buffer size. */
    if(wTxBuffSize < PH_CRYPTOSYM_INTERNAL_BUFFER_SIZE)
        return PH_ADD_COMPCODE(PH_ERR_PARAMETER_SIZE, PH_COMP_CRYPTOASYM);

    if(wRxBuffSize < PH_CRYPTOSYM_INTERNAL_BUFFER_SIZE)
        return PH_ADD_COMPCODE(PH_ERR_PARAMETER_SIZE, PH_COMP_CRYPTOASYM);

    /* Init. private data */
    pDataParams->wId = PH_COMP_CRYPTOSYM | PH_CRYPTOSYM_DUT_ID;
    pDataParams->pHalDataParams = pHalDataParams;

    pDataParams->pTxBuffer = pTxBuffer;
    pDataParams->wTxBuffSize = wTxBuffSize;

    pDataParams->pRxBuffer = pRxBuffer;
    pDataParams->wRxBuffSize = wRxBuffSize;

    return phhalHw_DUT_Cmd_CryptoSym_Init(pHalDataParams);
}

phStatus_t phCryptoSym_DUT_DeInit(phCryptoSym_DUT_DataParams_t * pDataParams)
{
    phStatus_t  PH_MEMLOC_REM wStatus = PH_ERR_SUCCESS;

    wStatus = phCryptoSym_DUT_InvalidateKey(pDataParams);

    wStatus = phhalHw_DUT_Cmd_CryptoSym_DeInit(pDataParams->pHalDataParams);

    pDataParams->wId = 0xFFFFU;
    pDataParams->pHalDataParams = NULL;

    pDataParams->pTxBuffer = NULL;
    pDataParams->wTxBuffSize = 0U;

    pDataParams->pRxBuffer = NULL;
    pDataParams->wRxBuffSize = 0U;

    return PH_ERR_SUCCESS;
}

phStatus_t phCryptoSym_DUT_GetLastStatus(phCryptoSym_DUT_DataParams_t * pDataParams, uint16_t wStatusMsgLen, int8_t * pStatusMsg,
    int32_t * pStatusCode)
{
    return phhalHw_DUT_Cmd_CryptoSym_GetLastStatus(
        pDataParams->pHalDataParams,
        wStatusMsgLen,
        pStatusMsg,
        pStatusCode);
}

phStatus_t phCryptoSym_DUT_InvalidateKey(phCryptoSym_DUT_DataParams_t * pDataParams)
{
    /* Reset all the key storage */
    (void) memset(pDataParams->aKey, 0x00, sizeof(pDataParams->aKey));
    (void) memset(pDataParams->aIV, 0x00, sizeof(pDataParams->aIV));

    pDataParams->wKeyType = PH_CRYPTOSYM_KEY_TYPE_INVALID;
    pDataParams->wKeepIV = PH_CRYPTOSYM_VALUE_KEEP_IV_OFF;

    return phhalHw_DUT_Cmd_CryptoSym_InvalidateKey(pDataParams->pHalDataParams);
}

phStatus_t phCryptoSym_DUT_SetConfig(phCryptoSym_DUT_DataParams_t * pDataParams, uint16_t wConfig, uint16_t wValue)
{
    switch(wConfig)
    {
        case PH_CRYPTOSYM_CONFIG_KEEP_IV:
            pDataParams->wKeepIV = wValue;

        default:
            return phhalHw_DUT_Cmd_CryptoSym_SetConfig(
                pDataParams->pHalDataParams,
                wConfig,
                wValue);
    }
}

phStatus_t phCryptoSym_DUT_GetConfig(phCryptoSym_DUT_DataParams_t * pDataParams, uint16_t wConfig, uint16_t * pValue)
{
    switch(wConfig)
    {
        case PH_CRYPTOSYM_CONFIG_KEEP_IV:
            *pValue = pDataParams->wKeepIV;
            break;

        case PH_CRYPTOSYM_CONFIG_KEY_SIZE:
            switch(pDataParams->wKeyType)
            {
                case PH_CRYPTOSYM_KEY_TYPE_AES128:
                    *pValue = PH_CRYPTOSYM_AES128_KEY_SIZE;
                    break;

                case PH_CRYPTOSYM_KEY_TYPE_AES192:
                    *pValue = PH_CRYPTOSYM_AES192_KEY_SIZE;
                    break;

                case PH_CRYPTOSYM_KEY_TYPE_AES256:
                    *pValue = PH_CRYPTOSYM_AES256_KEY_SIZE;
                    break;

                case PH_CRYPTOSYM_KEY_TYPE_DES:
                    *pValue = PH_CRYPTOSYM_DES_KEY_SIZE;
                    break;

                case PH_CRYPTOSYM_KEY_TYPE_2K3DES:
                    *pValue = PH_CRYPTOSYM_2K3DES_KEY_SIZE;
                    break;

                case PH_CRYPTOSYM_KEY_TYPE_3K3DES:
                    *pValue = PH_CRYPTOSYM_3K3DES_KEY_SIZE;
                    break;

                default:
                    return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_CRYPTOSYM);
            }
            break;

        case PH_CRYPTOSYM_CONFIG_BLOCK_SIZE:
            switch(pDataParams->wKeyType)
            {
                case PH_CRYPTOSYM_KEY_TYPE_AES128:
                case PH_CRYPTOSYM_KEY_TYPE_AES192:
                case PH_CRYPTOSYM_KEY_TYPE_AES256:
                    *pValue = PH_CRYPTOSYM_AES_BLOCK_SIZE;
                    break;

                case PH_CRYPTOSYM_KEY_TYPE_DES:
                case PH_CRYPTOSYM_KEY_TYPE_2K3DES:
                case PH_CRYPTOSYM_KEY_TYPE_3K3DES:
                    *pValue = PH_CRYPTOSYM_DES_BLOCK_SIZE;
                    break;

                default:
                    return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_CRYPTOSYM);
            }
            break;

        default:
            return phhalHw_DUT_Cmd_CryptoSym_GetConfig(
                pDataParams->pHalDataParams,
                wConfig,
                pValue);
    }

    return PH_ERR_SUCCESS;
}





phStatus_t phCryptoSym_DUT_Encrypt(phCryptoSym_DUT_DataParams_t * pDataParams, uint16_t wOption, const uint8_t * pPlainBuff,
    uint16_t wBuffLen, uint8_t * pEncBuff)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wBlockSize = 0;
    uint16_t    PH_MEMLOC_REM wBufOption = 0;

    uint8_t     PH_MEMLOC_REM *pTxBuff_Hal = NULL;
    uint16_t    PH_MEMLOC_REM wTxBuffSize_Hal = 0U;
    uint16_t    PH_MEMLOC_REM wTxBuffLen_Hal = 0U;
    uint16_t    PH_MEMLOC_REM wTxBuffStart_Hal = 0U;
    uint16_t    PH_MEMLOC_REM wTxBuffStartPos_Hal = 0U;

    uint8_t     PH_MEMLOC_REM *pRxBuff_Hal = NULL;
    uint16_t    PH_MEMLOC_REM wRxBuffSize_Hal = 0U;
    uint16_t    PH_MEMLOC_REM wRxBuffLen_Hal = 0U;
    uint16_t    PH_MEMLOC_REM wRxBuffStartPos_Hal = 0U;

    /* Backup HAL DataParams information. */
    phCryptoSym_DUT_BackupHalInfo(pDataParams, &pTxBuff_Hal, &wTxBuffSize_Hal, &wTxBuffLen_Hal, &wTxBuffStart_Hal, &wTxBuffStartPos_Hal,
        &pRxBuff_Hal, &wRxBuffSize_Hal, &wRxBuffLen_Hal, &wRxBuffStartPos_Hal);

    /* Get the Buffering option. */
    wBufOption = (uint16_t) (wOption & PH_EXCHANGE_BUFFER_MASK);

    TRY
    {
        /* Get the block size of the currently loaded key */
        PH_CHECK_SUCCESS_FCT_EXT(wStatus, phCryptoSym_DUT_GetConfig(pDataParams, PHHAL_HW_DUT_CMD_CRYPTOSYM_CONFIG_BLOCK_SIZE, &wBlockSize));

        /* Load the Key. */
        if(pDataParams->bIsDirectKey)
        {
            PH_CHECK_SUCCESS_FCT_EXT(wStatus, phhalHw_DUT_Cmd_CryptoSym_LoadKeyDirect(pDataParams->pHalDataParams, pDataParams->aKey,
                pDataParams->wKeyType));
        }

        /* Load the Current IV. */
        PH_CHECK_SUCCESS_FCT_EXT(wStatus, phhalHw_DUT_Cmd_CryptoSym_LoadIv(pDataParams->pHalDataParams, pDataParams->aIV, (uint8_t) wBlockSize));

        /* Perform Data Encryption */
        wStatus = phhalHw_DUT_Cmd_CryptoSym_Encrypt(
            pDataParams->pHalDataParams,
            wOption,
            pPlainBuff,
            wBuffLen,
            pEncBuff);

        /* Do the final update of the IV according to the keep IV setting. */
        if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
        {
            if((pDataParams->wKeepIV == PH_CRYPTOSYM_VALUE_KEEP_IV_ON) || (wBufOption & PH_EXCHANGE_BUFFERED_BIT))
            {
                (void) memcpy(pDataParams->aIV, &pEncBuff[wBuffLen - wBlockSize], wBlockSize);
            }
        }
    }
    FINALLY
    {
        /* Revert DataParams info. */
        phCryptoSym_DUT_RestoreHalInfo(pDataParams, &pTxBuff_Hal, &wTxBuffSize_Hal, &wTxBuffLen_Hal, &wTxBuffStart_Hal, &wTxBuffStartPos_Hal,
            &pRxBuff_Hal, &wRxBuffSize_Hal, &wRxBuffLen_Hal, &wRxBuffStartPos_Hal);
    }
    END

    return wStatus;
}

phStatus_t phCryptoSym_DUT_Decrypt(phCryptoSym_DUT_DataParams_t * pDataParams, uint16_t wOption, const uint8_t * pEncBuff,
    uint16_t wBuffLen, uint8_t * pPlainBuff)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wBlockSize = 0;

    uint8_t     PH_MEMLOC_REM *pTxBuff_Hal = NULL;
    uint16_t    PH_MEMLOC_REM wTxBuffSize_Hal = 0U;
    uint16_t    PH_MEMLOC_REM wTxBuffLen_Hal = 0U;
    uint16_t    PH_MEMLOC_REM wTxBuffStart_Hal = 0U;
    uint16_t    PH_MEMLOC_REM wTxBuffStartPos_Hal = 0U;

    uint8_t     PH_MEMLOC_REM *pRxBuff_Hal = NULL;
    uint16_t    PH_MEMLOC_REM wRxBuffSize_Hal = 0U;
    uint16_t    PH_MEMLOC_REM wRxBuffLen_Hal = 0U;
    uint16_t    PH_MEMLOC_REM wRxBuffStartPos_Hal = 0U;

    /* Backup HAL DataParams information. */
    phCryptoSym_DUT_BackupHalInfo(pDataParams, &pTxBuff_Hal, &wTxBuffSize_Hal, &wTxBuffLen_Hal, &wTxBuffStart_Hal, &wTxBuffStartPos_Hal,
        &pRxBuff_Hal, &wRxBuffSize_Hal, &wRxBuffLen_Hal, &wRxBuffStartPos_Hal);

    TRY
    {
        /* Get the block size of the currently loaded key */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_DUT_GetConfig(pDataParams, PHHAL_HW_DUT_CMD_CRYPTOSYM_CONFIG_BLOCK_SIZE, &wBlockSize));

        /* Load the Key. */
        if(pDataParams->bIsDirectKey)
        {
            PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_DUT_Cmd_CryptoSym_LoadKeyDirect(pDataParams->pHalDataParams, pDataParams->aKey,
                pDataParams->wKeyType));
        }

        /* Load the Current IV. */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_DUT_Cmd_CryptoSym_LoadIv(pDataParams->pHalDataParams, pDataParams->aIV, (uint8_t) wBlockSize));

        wStatus = phhalHw_DUT_Cmd_CryptoSym_Decrypt(
            pDataParams->pHalDataParams,
            wOption,
            pEncBuff,
            wBuffLen,
            pPlainBuff);

        /* Do the final update of the IV according to the keep IV setting. */
        if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
        {
            if((pDataParams->wKeepIV == PH_CRYPTOSYM_VALUE_KEEP_IV_ON) || (wOption & PH_EXCHANGE_BUFFERED_BIT))
            {
                (void) memcpy(pDataParams->aIV, &pEncBuff[wBuffLen - wBlockSize], wBlockSize);
            }
        }
    }
    FINALLY
    {
        /* Revert DataParams info. */
        phCryptoSym_DUT_RestoreHalInfo(pDataParams, &pTxBuff_Hal, &wTxBuffSize_Hal, &wTxBuffLen_Hal, &wTxBuffStart_Hal, &wTxBuffStartPos_Hal,
            &pRxBuff_Hal, &wRxBuffSize_Hal, &wRxBuffLen_Hal, &wRxBuffStartPos_Hal);
    }
    END

    return wStatus;
}

phStatus_t phCryptoSym_DUT_CalculateMac(phCryptoSym_DUT_DataParams_t * pDataParams, uint16_t wOption, const uint8_t* pData,
    uint16_t wDataLen, uint8_t * pMac, uint8_t * pMacLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wBlockSize = 0;

    uint8_t     PH_MEMLOC_REM *pTxBuff_Hal = NULL;
    uint16_t    PH_MEMLOC_REM wTxBuffSize_Hal = 0U;
    uint16_t    PH_MEMLOC_REM wTxBuffLen_Hal = 0U;
    uint16_t    PH_MEMLOC_REM wTxBuffStart_Hal = 0U;
    uint16_t    PH_MEMLOC_REM wTxBuffStartPos_Hal = 0U;

    uint8_t     PH_MEMLOC_REM *pRxBuff_Hal = NULL;
    uint16_t    PH_MEMLOC_REM wRxBuffSize_Hal = 0U;
    uint16_t    PH_MEMLOC_REM wRxBuffLen_Hal = 0U;
    uint16_t    PH_MEMLOC_REM wRxBuffStartPos_Hal = 0U;

    /* Backup HAL DataParams information. */
    phCryptoSym_DUT_BackupHalInfo(pDataParams, &pTxBuff_Hal, &wTxBuffSize_Hal, &wTxBuffLen_Hal, &wTxBuffStart_Hal, &wTxBuffStartPos_Hal,
        &pRxBuff_Hal, &wRxBuffSize_Hal, &wRxBuffLen_Hal, &wRxBuffStartPos_Hal);

    TRY
    {
        /* Get the block size of the currently loaded key */
        PH_CHECK_SUCCESS_FCT_EXT(wStatus, phCryptoSym_DUT_GetConfig(pDataParams, PHHAL_HW_DUT_CMD_CRYPTOSYM_CONFIG_BLOCK_SIZE, &wBlockSize));

        /* Load the Key. */
        if(pDataParams->bIsDirectKey)
        {
            PH_CHECK_SUCCESS_FCT_EXT(wStatus, phhalHw_DUT_Cmd_CryptoSym_LoadKeyDirect(pDataParams->pHalDataParams, pDataParams->aKey,
                pDataParams->wKeyType));
        }

        /* Load the Current IV. */
        PH_CHECK_SUCCESS_FCT_EXT(wStatus, phhalHw_DUT_Cmd_CryptoSym_LoadIv(pDataParams->pHalDataParams, pDataParams->aIV, (uint8_t) wBlockSize));

        wStatus = phhalHw_DUT_Cmd_CryptoSym_CalculateMac(
            pDataParams->pHalDataParams,
            wOption,
            pData,
            wDataLen,
            pMac,
            pMacLen);

        /* Do the final update of the IV according to the keep IV setting. */
        if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
        {
            if((pDataParams->wKeepIV == PH_CRYPTOSYM_VALUE_KEEP_IV_ON) || (wOption & PH_EXCHANGE_BUFFERED_BIT))
            {
                (void) memcpy(pDataParams->aIV, pMac, wBlockSize);
            }
        }
        else
        {
            /* Clear the IV for security reasons */
            (void) memset(pDataParams->aIV, 0, wBlockSize);
        }
    }
    FINALLY
    {
        /* Revert DataParams info. */
        phCryptoSym_DUT_RestoreHalInfo(pDataParams, &pTxBuff_Hal, &wTxBuffSize_Hal, &wTxBuffLen_Hal, &wTxBuffStart_Hal, &wTxBuffStartPos_Hal,
            &pRxBuff_Hal, &wRxBuffSize_Hal, &wRxBuffLen_Hal, &wRxBuffStartPos_Hal);
    }
    END

    return wStatus;
}

phStatus_t phCryptoSym_DUT_LoadIv(phCryptoSym_DUT_DataParams_t * pDataParams, const uint8_t * pIV, uint8_t bIVLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Copy the IV to DataParams. */
    (void) memcpy(pDataParams->aIV, pIV, bIVLen);

    return wStatus;
}

phStatus_t phCryptoSym_DUT_LoadKey(phCryptoSym_DUT_DataParams_t * pDataParams, uint16_t wKeyNo, uint16_t wKeyVer, uint16_t wKeyType)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Reset LoadKey Mode. */
    pDataParams->bIsDirectKey = PH_OFF;

    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_DUT_Cmd_CryptoSym_LoadKey(
        pDataParams->pHalDataParams,
        wKeyNo,
        wKeyVer,
        wKeyType));

    /* Update LoadKey Mode. */
    pDataParams->bIsDirectKey = PH_OFF;

    return wStatus;
}

phStatus_t phCryptoSym_DUT_LoadKeyDirect(phCryptoSym_DUT_DataParams_t * pDataParams, const uint8_t * pKey, uint16_t wKeyType)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Reset LoadKey Mode. */
    pDataParams->bIsDirectKey = PH_OFF;

    /* Update the KeyType to DataParams. */
    pDataParams->wKeyType = wKeyType;


    switch(wKeyType)
    {
        case PH_CRYPTOSYM_KEY_TYPE_DES:
            /* Copy the key to DataParams. */
            (void) memcpy(pDataParams->aKey, pKey, PH_CRYPTOSYM_DES_KEY_SIZE);
            break;

        case PH_CRYPTOSYM_KEY_TYPE_2K3DES:
            /* Copy the key to DataParams. */
            (void) memcpy(pDataParams->aKey, pKey, PH_CRYPTOSYM_2K3DES_KEY_SIZE);
            break;

        case PH_CRYPTOSYM_KEY_TYPE_3K3DES:
            /* Copy the key to DataParams. */
            (void) memcpy(pDataParams->aKey, pKey, PH_CRYPTOSYM_3K3DES_KEY_SIZE);
            break;

        case PH_CRYPTOSYM_KEY_TYPE_AES128:
            /* Copy the key to DataParams. */
            (void) memcpy(pDataParams->aKey, pKey, PH_CRYPTOSYM_AES128_KEY_SIZE);
            break;

        case PH_CRYPTOSYM_KEY_TYPE_AES192:
            /* Copy the key to DataParams. */
            (void) memcpy(pDataParams->aKey, pKey, PH_CRYPTOSYM_AES192_KEY_SIZE);
            break;

        case PH_CRYPTOSYM_KEY_TYPE_AES256:
            /* Copy the key to DataParams. */
            (void) memcpy(pDataParams->aKey, pKey, PH_CRYPTOSYM_AES256_KEY_SIZE);
            break;

        default:
            PH_UNUSED_VARIABLE(pDataParams);
            PH_UNUSED_VARIABLE(wKeyType);
            PH_UNUSED_VARIABLE(pKey);
            wStatus = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_CRYPTOSYM);
            break;
    }

    /* Update LoadKey Mode. */
    pDataParams->bIsDirectKey = PH_ON;

    return wStatus;
}

phStatus_t phCryptoSym_DUT_DiversifyKey(phCryptoSym_DUT_DataParams_t * pDataParams, uint16_t wOption, uint16_t wKeyNo, uint16_t wKeyVer,
    uint8_t * pDivInput, uint8_t bDivInLen, uint8_t * pDivKey, uint8_t * pDivKeyLen)
{
    return phhalHw_DUT_Cmd_CryptoSym_DiversifyKey(
        pDataParams->pHalDataParams,
        wOption,
        wKeyNo,
        wKeyVer,
        pDivInput,
        bDivInLen,
        pDivKey,
        pDivKeyLen);
}

phStatus_t phCryptoSym_DUT_DiversifyDirectKey(phCryptoSym_DUT_DataParams_t * pDataParams, uint16_t wOption, uint8_t * pKey, uint16_t wKeyType,
    uint8_t * pDivInput, uint8_t bDivInLen, uint8_t * pDivKey, uint8_t * pDivKeyLen)
{
    return phhalHw_DUT_Cmd_CryptoSym_DiversifyDirectKey(
        pDataParams->pHalDataParams,
        wOption,
        pKey,
        wKeyType,
        pDivInput,
        bDivInLen,
        pDivKey,
        pDivKeyLen);
}





void phCryptoSym_DUT_BackupHalInfo(phCryptoSym_DUT_DataParams_t * pDataParams, uint8_t **ppTxBuff_Hal, uint16_t *pTxBuffSize_Hal,
    uint16_t *pTxBuffLen_Hal, uint16_t *pTxBuffStart_Hal, uint16_t *pTxBuffStartPos_Hal, uint8_t **ppRxBuff_Hal, uint16_t *pRxBuffSize_Hal,
    uint16_t *pRxBuffLen_Hal, uint16_t *pRxBuffStartPos_Hal)
{
    /* Backup HAL DataParams information. */
    *ppTxBuff_Hal = ((phhalHw_DUT_DataParams_t *) pDataParams->pHalDataParams)->pTxBuffer;
    *pTxBuffSize_Hal = ((phhalHw_DUT_DataParams_t *) pDataParams->pHalDataParams)->wTxBufSize;
    *pTxBuffLen_Hal = ((phhalHw_DUT_DataParams_t *) pDataParams->pHalDataParams)->wTxBufLen;
    *pTxBuffStart_Hal = ((phhalHw_DUT_DataParams_t *) pDataParams->pHalDataParams)->wTxBufStart;
    *pTxBuffStartPos_Hal = ((phhalHw_DUT_DataParams_t *) pDataParams->pHalDataParams)->wTxBufStartPos;

    *ppRxBuff_Hal = ((phhalHw_DUT_DataParams_t *) pDataParams->pHalDataParams)->pRxBuffer;
    *pRxBuffSize_Hal = ((phhalHw_DUT_DataParams_t *) pDataParams->pHalDataParams)->wRxBufSize;
    *pRxBuffLen_Hal = ((phhalHw_DUT_DataParams_t *) pDataParams->pHalDataParams)->wRxBufLen;
    *pRxBuffStartPos_Hal = ((phhalHw_DUT_DataParams_t *) pDataParams->pHalDataParams)->wRxBufStartPos;

    /* Update Hal DataParams info. */
    ((phhalHw_DUT_DataParams_t *) pDataParams->pHalDataParams)->pTxBuffer = pDataParams->pTxBuffer;
    ((phhalHw_DUT_DataParams_t *) pDataParams->pHalDataParams)->wTxBufSize = pDataParams->wTxBuffSize;
    ((phhalHw_DUT_DataParams_t *) pDataParams->pHalDataParams)->wTxBufLen = 0U;
    ((phhalHw_DUT_DataParams_t *) pDataParams->pHalDataParams)->wTxBufStart = 0U;
    ((phhalHw_DUT_DataParams_t *) pDataParams->pHalDataParams)->wTxBufStartPos = 0U;

    ((phhalHw_DUT_DataParams_t *) pDataParams->pHalDataParams)->pRxBuffer = pDataParams->pRxBuffer;
    ((phhalHw_DUT_DataParams_t *) pDataParams->pHalDataParams)->wRxBufSize = pDataParams->wTxBuffSize;
    ((phhalHw_DUT_DataParams_t *) pDataParams->pHalDataParams)->wRxBufLen = 0U;
    ((phhalHw_DUT_DataParams_t *) pDataParams->pHalDataParams)->wRxBufStartPos = 0U;
}

void phCryptoSym_DUT_RestoreHalInfo(phCryptoSym_DUT_DataParams_t * pDataParams, uint8_t **ppTxBuff_Hal, uint16_t *pTxBuffSize_Hal,
    uint16_t *pTxBuffLen_Hal, uint16_t *pTxBuffStart_Hal, uint16_t *pTxBuffStartPos_Hal, uint8_t **ppRxBuff_Hal, uint16_t *pRxBuffSize_Hal,
    uint16_t *pRxBuffLen_Hal, uint16_t *pRxBuffStartPos_Hal)
{
    /* Revert DataParams info. */
    ((phhalHw_DUT_DataParams_t *) pDataParams->pHalDataParams)->pTxBuffer = *ppTxBuff_Hal;
    ((phhalHw_DUT_DataParams_t *) pDataParams->pHalDataParams)->wTxBufSize = *pTxBuffSize_Hal;
    ((phhalHw_DUT_DataParams_t *) pDataParams->pHalDataParams)->wTxBufLen = *pTxBuffLen_Hal;
    ((phhalHw_DUT_DataParams_t *) pDataParams->pHalDataParams)->wTxBufStart = *pTxBuffStart_Hal;
    ((phhalHw_DUT_DataParams_t *) pDataParams->pHalDataParams)->wTxBufStartPos = *pTxBuffStartPos_Hal;

    ((phhalHw_DUT_DataParams_t *) pDataParams->pHalDataParams)->pRxBuffer = *ppRxBuff_Hal;
    ((phhalHw_DUT_DataParams_t *) pDataParams->pHalDataParams)->wRxBufSize = *pRxBuffSize_Hal;
    ((phhalHw_DUT_DataParams_t *) pDataParams->pHalDataParams)->wRxBufLen = *pRxBuffLen_Hal;
    ((phhalHw_DUT_DataParams_t *) pDataParams->pHalDataParams)->wRxBufStartPos = *pRxBuffStartPos_Hal;
}

#endif /* NXPBUILD__PH_CRYPTOSYM_DUT */
