/*
 * Copyright 2013, 2017, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Example Source for RD710 SAM non X-MODE with two cards.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

/**
 * Header for this file
 */
#include "Example-Rd710_SamNonX_TwoCards.h"
#include <crtdbg.h>

/**
 * Reader Library Headers
 */
#include <phbalReg.h>
#include <phhalHw.h>
#include <phpalI14443p3a.h>
#include <phpalI14443p4a.h>
#include <phpalI14443p4.h>
#include <phCidManager.h>
#include <phpalMifare.h>
#include <phalMful.h>
#include <phKeyStore.h>
#include <phCryptoRng.h>
#include <phCryptoSym.h>
#include <phalMfc.h>
#include <phalMfp.h>
#include <phhalHw_Rd710_Cmd.h>

/** Reader Names */
#define CONTACTLESS_READER_NAME	"NXP Pegoda S CL 0 0"
#define CONTACT_READER_NAME		"NXP Pegoda S 0 0"

/**
 * Key store
 */
#define KEYCOUNT				0xFFU	/**< number of keys */
#define KEYVERSIONS				0x01U	/**< number of key versions */

#define SAM_KEY					0x00U	/**< SAM Key address */

#define SAM_AV1					0x01U	/**< SAM AV1 */
#define SAM_AV2					0x02U	/**< SAM AV2 */

#define SAM_LC0					0x00U	/**< SAM logical channel 0 */
#define SAM_LC1					0x01U	/**< SAM logical channel 1 */
#define SAM_LC2					0x02U	/**< SAM logical channel 1 */

#define PLAIN					0x00U	/**< plain mode */
#define MAC						0x01U	/**< CMAC mode */
#define FULL					0x02U	/**< full mode */

#define GLOBAL_BUFFER_SIZE		0x012CU	/**< global size of transmit and recieve buffer */

#define RETRYCOUNT				0x05U	/**< retry count */

#define CHECK_SUCCESS(x)                                                \
if ((x) != PH_ERR_SUCCESS)                                              \
{                                                                       \
    printf("An error occured (0x%04X), press any key to exit...", (x)); \
    _getch();															\
    return 0;                                                           \
}

void * pKeyStore = NULL;
void * pBal = NULL;

uint8_t aDESKey[] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
uint8_t aAESKey[] = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };

#ifdef NXPBUILD__PH_LOG
phLog_RegisterEntry_t logRegisterEntries[5];
phLog_LogEntry_t logEntries[32];
#endif /* NXPBUILD__PH_LOG */

#ifdef NXPBUILD__PH_LOG
void Log_Callback(void * pDataParams, uint8_t bOption, phLog_LogEntry_t * pEntries, uint16_t wEntryCount)
{
    phStatus_t status;
    uint8_t bIsFirstString;
    uint8_t bIsFirstParam;
    uint16_t wIndex;
    uint16_t wByteIndex;
    uint16_t wOption;
    char sOutput[512];
    char sTmp[7];

    /* satisfy compiler */
    if (pDataParams);

    /* Ignore some calls */
    if ((strncmp((char*)pEntries[0].pString, "phhalHw_SetConfig", 17) == 0) ||
        (strncmp((char*)pEntries[0].pString, "phhalHw_GetConfig", 17) == 0) ||
        (strncmp((char*)pEntries[0].pString, "phhalHw_ReadReg", 15) == 0) ||
        (strncmp((char*)pEntries[0].pString, "phhalHw_WriteReg", 16) == 0))
    {
        return;
    }

    /* Clear output string */
    sprintf(sOutput, "");

    /* Init. states */
    bIsFirstString = 1;
    bIsFirstParam = 1;

    if (bOption != PH_LOG_OPTION_CATEGORY_GEN)
    {
        strcat_s(sOutput, sizeof(sOutput), "\n");
    }

    for (wIndex = 0; wIndex < wEntryCount; ++wIndex)
    {
        if ((pEntries[wIndex].bLogType == PH_LOG_LOGTYPE_INFO) && pEntries[wIndex].wDataLen == 0)
        {
            strcat_s(sOutput, sizeof(sOutput), (char*)pEntries[wIndex].pString);
            if (bIsFirstString)
            {
                switch (bOption)
                {
                case PH_LOG_OPTION_CATEGORY_ENTER:
                    strcat_s(sOutput, sizeof(sOutput), " [ENTRY] ");
                    break;
                case PH_LOG_OPTION_CATEGORY_GEN:
                    strcat_s(sOutput, sizeof(sOutput), " [GEN] ");
                    break;
                case PH_LOG_OPTION_CATEGORY_LEAVE:
                    strcat_s(sOutput, sizeof(sOutput), " [LEAVE] ");
                    break;
                }
                bIsFirstString = 0;
            }
        }
        else
        {
            if (!bIsFirstParam)
            {
                if ((strcmp((char*)pEntries[wIndex].pString, "status") != 0) && (pEntries[wIndex].wDataLen > 0))
                {
                    strcat_s(sOutput, sizeof(sOutput), ", ");
                }
            }
            else
            {
                bIsFirstParam = 0;
            }
        }

        if ((strcmp((char*)pEntries[wIndex].pString, "wOption") == 0) && (pEntries[wIndex].wDataLen == 2))
        {
            wOption = (((uint8_t*)(pEntries[wIndex].pData))[0] << 8) | ((uint8_t*)(pEntries[wIndex].pData))[1];
            strcat_s(sOutput, sizeof(sOutput), "wOption = ");
            switch (wOption & PH_EXCHANGE_MODE_MASK)
            {
            case PH_EXCHANGE_DEFAULT:
                strcat_s(sOutput, sizeof(sOutput), "DEFAULT");
                break;
            case PH_EXCHANGE_TXCHAINING:
                strcat_s(sOutput, sizeof(sOutput), "TXCHAINING");
                break;
            case PH_EXCHANGE_RXCHAINING:
                strcat_s(sOutput, sizeof(sOutput), "RXCHAINING");
                break;
            case PH_EXCHANGE_RXCHAINING_BUFSIZE:
                strcat_s(sOutput, sizeof(sOutput), "RXCHAINING_BUFSIZE");
                break;
            }
            if ((wOption & PH_EXCHANGE_BUFFERED_BIT) != 0)
            {
                strcat_s(sOutput, sizeof(sOutput), " | BUFFERED_BIT");
            }
            if ((wOption & PH_EXCHANGE_LEAVE_BUFFER_BIT) != 0)
            {
                strcat_s(sOutput, sizeof(sOutput), " | LEAVE_BUFFER_BIT");
            }
        }
        else if ((strcmp((char*)pEntries[wIndex].pString, "status") == 0) && (pEntries[wIndex].wDataLen == 2))
        {
			strcat_s(sOutput, sizeof(sOutput), (char*)pEntries[wIndex].pString);
            strcat_s(sOutput, sizeof(sOutput), " = ");
            status = (((uint8_t*)(pEntries[wIndex].pData))[0] << 8) | ((uint8_t*)(pEntries[wIndex].pData))[1];
            sprintf_s(sTmp, sizeof(sTmp), "0x%04X", status);
            strcat_s(sOutput, sizeof(sOutput), sTmp);
        }
        else if (pEntries[wIndex].wDataLen > 0)
        {
            strcat_s(sOutput, sizeof(sOutput), (char*)pEntries[wIndex].pString);
            strcat_s(sOutput, sizeof(sOutput), "(");
            sprintf_s(sTmp, sizeof(sTmp), "%d", pEntries[wIndex].wDataLen);
            strcat_s(sOutput, sizeof(sOutput), sTmp);
            strcat_s(sOutput, sizeof(sOutput), ") = ");

            for (wByteIndex = 0; wByteIndex < pEntries[wIndex].wDataLen; ++wByteIndex)
            {
                sprintf_s(sTmp, sizeof(sTmp), "%02X", ((uint8_t*)pEntries[wIndex].pData)[wByteIndex]);
                strcat_s(sOutput, sizeof(sOutput), sTmp);
                if ((wByteIndex + 1) < pEntries[wIndex].wDataLen)
                {
                    strcat_s(sOutput, sizeof(sOutput), " ");
                }
            }
        }
    }

    if (bOption == PH_LOG_OPTION_CATEGORY_LEAVE)
    {
        strcat_s(sOutput, sizeof(sOutput), "\n");
    }
    else
    {
		if(bOption == PH_LOG_OPTION_CATEGORY_GEN)
			strcat_s(sOutput, sizeof(sOutput), "\n");
    }
    printf(sOutput);
}
#endif /* NXPBUILD__PH_LOG */

void PrintData ( uint8_t* pBuffer, uint32_t dwLength )
{
    uint32_t dwIndex;

    for ( dwIndex = 0; dwIndex < dwLength; ++dwIndex )
        printf ( "%2X", pBuffer[dwIndex] );
}

int __cdecl main()
{
    uint8_t aHalBufferReaderTx[GLOBAL_BUFFER_SIZE];  /**< HAL Reader transmit buffer */
    uint8_t aHalBufferReaderRx[GLOBAL_BUFFER_SIZE];  /**< HAL Reader receive buffer */
    uint8_t aBalBufferTx[GLOBAL_BUFFER_SIZE];        /**< BAL Reader transmit buffer */
    uint8_t aBalBufferRx[GLOBAL_BUFFER_SIZE];        /**< BAL Reader receive buffer */
    uint8_t aAtr[256];                /**< Atr */
    uint8_t aSeed[8];                 /**< Seed */
	uint16_t aVersion[0x1F];          /**< SAM Version */
    uint8_t aHalBufferSam[261];       /**< Hal SAM Buffer array */
    uint8_t aUid[10];                  /**< complete Uid; uint8_t[4/7/10]. */
    uint8_t bLength;                   /**< length */
    uint8_t aSak[1];                   /**< Select Acknowledge; uint8_t. */
    uint8_t bMoreCardsAvaliable;       /**< more card available */
    uint8_t aData[0x10];           /**< Response data */

    /**
     * keystore
     */
    phKeyStore_Sw_KeyEntry_t aKeyEntry[KEYCOUNT];
    phKeyStore_Sw_KeyVersionPair_t aKeyVersion[KEYCOUNT * KEYVERSIONS];
    phKeyStore_Sw_KUCEntry_t aKeyUsage[KEYCOUNT];

	/* Variables for Common data parameter. */
    phCryptoSym_Sw_DataParams_t stCryptoEnc;
    phCryptoSym_Sw_DataParams_t stCryptoMAC;
    phCryptoSym_Sw_DataParams_t stCryptoSymRnd;
    phCryptoRng_Sw_DataParams_t stCryptoRng;
    phKeyStore_Sw_DataParams_t stKeystoreSw;
    phKeyStore_SamAV2_DataParams_t stKeystoreSAM;

    /* Variables for BAL data parameter. */
    phbalReg_PcscWin_DataParams_t stBal_Contactless;
	phbalReg_Rd710Sam_DataParams_t stBal_Contact;

	/* Variables for HAL data parameter. */
	phhalHw_Rd710_DataParams_t stHAL_Contactless;
    phhalHw_SamAV2_DataParams_t stHAL_ContactLC0;
    phhalHw_SamAV2_DataParams_t stHAL_ContactLC1;

	/* Variables for PAL data parameter. */
    phpalI14443p3a_Rd710_DataParams_t stI14443p3a;
    phpalI14443p4a_Rd710_DataParams_t stI14443p4a;
    phpalI14443p4_Rd710_DataParams_t stI14443p4;
	phpalMifare_Rd710_DataParams_t stPalMifare;

	/* Variables for AL data parameter. */
    phalMful_Sw_DataParams_t stMful;

    /* Initialize logging */
#ifdef NXPBUILD__PH_LOG
	CHECK_SUCCESS ( phLog_Init ( Log_Callback, logRegisterEntries, (uint16_t) ( sizeof ( logRegisterEntries ) / sizeof ( phLog_RegisterEntry_t ) ) ) );
    CHECK_SUCCESS ( phLog_Register ( &stHAL_Contactless, logEntries, (uint16_t) ( sizeof ( logEntries ) / sizeof ( phLog_LogEntry_t ) ) ) );
    CHECK_SUCCESS ( phLog_Register ( &stHAL_ContactLC0, logEntries, (uint16_t) ( sizeof ( logEntries ) / sizeof ( phLog_LogEntry_t ) ) ) );
    CHECK_SUCCESS ( phLog_Register ( &stHAL_ContactLC1, logEntries, (uint16_t) ( sizeof ( logEntries ) / sizeof ( phLog_LogEntry_t ) ) ) );
    CHECK_SUCCESS ( phLog_Register ( &stKeystoreSAM, logEntries, (uint16_t) ( sizeof ( logEntries ) / sizeof ( phLog_LogEntry_t ) ) ) );
    CHECK_SUCCESS ( phLog_Register ( &stMful, logEntries, (uint16_t) ( sizeof ( logEntries ) / sizeof ( phLog_LogEntry_t ) ) ) );
#endif /* NXPBUILD__PH_LOG */

    printf ( "\nNxpRdLib ANSI-C Example Program V1.0a\n\n" );
    printf ( "Please ensure that a Pegoda 2 reader in S - Mode is connected and in working condition.\n\n" );
    printf ( "Performing startup...\n\n" );

    /* Initialize Software Keystore component. */
    CHECK_SUCCESS ( phKeyStore_Sw_Init ( &stKeystoreSw, sizeof ( phKeyStore_Sw_DataParams_t ), aKeyEntry, KEYCOUNT, aKeyVersion, KEYVERSIONS, aKeyUsage, KEYCOUNT ) );
	pKeyStore = &stKeystoreSw;

    /** Initialize Crypto components. */
    CHECK_SUCCESS ( phCryptoSym_Sw_Init ( &stCryptoEnc, sizeof ( phCryptoSym_Sw_DataParams_t ), pKeyStore ) );
    CHECK_SUCCESS ( phCryptoSym_Sw_Init ( &stCryptoMAC, sizeof ( phCryptoSym_Sw_DataParams_t ), pKeyStore ) );
    CHECK_SUCCESS ( phCryptoSym_Sw_Init ( &stCryptoSymRnd, sizeof ( phCryptoSym_Sw_DataParams_t ), pKeyStore ) );
    CHECK_SUCCESS ( phCryptoRng_Sw_Init ( &stCryptoRng, sizeof ( phCryptoRng_Sw_DataParams_t ), &stCryptoSymRnd ) );

    /* Seed the random number generator with the given seed. */
    memset(aSeed, 0x00, 0x08);
    CHECK_SUCCESS ( phCryptoRng_Seed ( &stCryptoRng, aSeed, sizeof ( aSeed ) ) );

    printf ( "------------------------------------------------------------------------------------------------\n" );
    printf ( "Make sure Pegoda reader is connected in S mode. \n" );
	printf ( "\n\n" );

 	/* Initialize contactless BAL layer. */
	CHECK_SUCCESS ( phbalReg_PcscWin_Init ( &stBal_Contactless, sizeof ( phbalReg_PcscWin_DataParams_t ), aAtr, sizeof ( aAtr ) ) );
	pBal = &stBal_Contactless;

	/* Set the BAL Communication Configuration values */
	CHECK_SUCCESS ( phbalReg_SetConfig ( &stBal_Contactless, PHBAL_REG_PCSCWIN_CONFIG_PROTOCOL, PHBAL_REG_PCSCWIN_VALUE_PROTOCOL_UNDEFINED ) );
	CHECK_SUCCESS ( phbalReg_SetConfig ( &stBal_Contactless, PHBAL_REG_PCSCWIN_CONFIG_SHARE, PHBAL_REG_PCSCWIN_VALUE_SHARE_DIRECT ) );

	/* Select and Open contactless reader. */
    CHECK_SUCCESS ( phbalReg_SetPort ( &stBal_Contactless, (uint8_t *) CONTACTLESS_READER_NAME ) );
	CHECK_SUCCESS ( phbalReg_OpenPort ( &stBal_Contactless ) );

	/* Initialize contact BAL layer. */
	CHECK_SUCCESS ( phbalReg_Rd710Sam_Init ( &stBal_Contact, sizeof ( phbalReg_Rd710Sam_DataParams_t ), &stBal_Contactless, aAtr, sizeof ( aAtr ),
		aBalBufferTx, sizeof ( aBalBufferTx ), aBalBufferRx, sizeof ( aBalBufferRx ) ) );
	pBal = &stBal_Contact;

	/* Open the BAL Rd710 SAM Port. */
	CHECK_SUCCESS ( phbalReg_OpenPort ( &stBal_Contact ) );

	/* Initialise the Rd710 HAL component. */
	CHECK_SUCCESS ( phhalHw_Rd710_Init ( &stHAL_Contactless, sizeof ( phhalHw_Rd710_DataParams_t ), &stBal_Contactless, 0, aHalBufferReaderTx,
		sizeof ( aHalBufferReaderTx ), aHalBufferReaderRx, sizeof ( aHalBufferReaderRx ) ) );
	CHECK_SUCCESS ( phhalHw_Rd710_Cmd_InitReader ( &stHAL_Contactless ) );

    /* Initialize Sam AV2 HAL component. */
    CHECK_SUCCESS ( phhalHw_SamAV2_Init ( &stHAL_ContactLC0, sizeof ( phhalHw_SamAV2_DataParams_t ), pBal, &stHAL_Contactless, pKeyStore, &stCryptoEnc,
		&stCryptoMAC, &stCryptoRng, PHHAL_HW_SAMAV2_OPMODE_NON_X, SAM_LC0, aHalBufferSam, sizeof ( aHalBufferSam ), aHalBufferSam, sizeof ( aHalBufferSam ) ) );
    CHECK_SUCCESS ( phhalHw_SamAV2_Init ( &stHAL_ContactLC1, sizeof ( phhalHw_SamAV2_DataParams_t ), pBal, &stHAL_Contactless, pKeyStore, &stCryptoEnc,
		&stCryptoMAC, &stCryptoRng, PHHAL_HW_SAMAV2_OPMODE_NON_X, SAM_LC1, aHalBufferSam, sizeof ( aHalBufferSam ), aHalBufferSam, sizeof ( aHalBufferSam ) ) );

    /* Initialize SAM Keystore component. */
    CHECK_SUCCESS ( phKeyStore_SamAV2_Init ( &stKeystoreSAM, sizeof ( phKeyStore_SamAV2_DataParams_t ), &stHAL_ContactLC1 ) );
	pKeyStore = &stKeystoreSAM;

	/* Detect SAM settings LC0 and LC1. */
	CHECK_SUCCESS ( phhalHw_SamAV2_DetectMode ( &stHAL_ContactLC0 ) );
	CHECK_SUCCESS ( phhalHw_SamAV2_DetectMode(&stHAL_ContactLC1 ) );

	/* Get SAM hostmode. */
    CHECK_SUCCESS ( phhalHw_GetConfig ( &stHAL_ContactLC0, PHHAL_HW_SAMAV2_CONFIG_HOSTMODE, aVersion ) );

    /* Authenticate with SAM on LC0 which is the Management channel. */
    if (aVersion[0] == SAM_AV1)
    {
		/* Load a the key. */
		CHECK_SUCCESS ( phKeyStore_FormatKeyEntry ( &stKeystoreSw, SAM_KEY, PH_KEYSTORE_KEY_TYPE_DES ) );
		CHECK_SUCCESS ( phKeyStore_SetKeyAtPos ( &stKeystoreSw, SAM_KEY, 0, PH_KEYSTORE_KEY_TYPE_DES, aDESKey, 0) );

        /* Perform HOST AUTHENTICATION in PLAIN mode. */
		CHECK_SUCCESS ( phhalHw_SamAV2_Cmd_SAM_AuthenticateHost ( &stHAL_ContactLC1, PLAIN, 0, 0x00, SAM_KEY, 0x00, NULL, 0 ) );
    }
    else
    {
		/* Load a the key. */
		CHECK_SUCCESS ( phKeyStore_FormatKeyEntry ( &stKeystoreSw, SAM_KEY, PH_KEYSTORE_KEY_TYPE_AES128 ) );
		CHECK_SUCCESS ( phKeyStore_SetKeyAtPos ( &stKeystoreSw, SAM_KEY, 0, PH_KEYSTORE_KEY_TYPE_AES128, aAESKey, 0) );

	    /* Perform HOST AUTHENTICATION in PLAIN mode. */
		CHECK_SUCCESS ( phhalHw_SamAV2_Cmd_SAM_AuthenticateHost ( &stHAL_ContactLC1, PLAIN, 0, 0x00, SAM_KEY, 0x00, NULL, 0 ) );
    }

    /* Initialize the PAL components. */
    CHECK_SUCCESS ( phpalI14443p3a_Rd710_Init ( &stI14443p3a, sizeof ( phpalI14443p3a_Rd710_DataParams_t ), &stHAL_Contactless ) );
    CHECK_SUCCESS ( phpalI14443p4a_Rd710_Init ( &stI14443p4a, sizeof ( phpalI14443p4a_Rd710_DataParams_t ), &stHAL_Contactless ) );
    CHECK_SUCCESS ( phpalI14443p4_Rd710_Init ( &stI14443p4, sizeof ( phpalI14443p4_Rd710_DataParams_t ), &stHAL_Contactless ) );
    CHECK_SUCCESS ( phpalMifare_Rd710_Init ( &stPalMifare, sizeof ( phpalMifare_Rd710_DataParams_t ), &stHAL_Contactless ) );

    /* Initialize AL component. */
    CHECK_SUCCESS ( phalMful_Sw_Init ( &stMful, sizeof ( phalMful_Sw_DataParams_t ), &stPalMifare, pKeyStore, &stCryptoEnc, &stCryptoRng ) );

	/* Set ISO 14443p4 configuration parameter. */
    CHECK_SUCCESS ( phpalI14443p4_SetConfig ( &stI14443p4, RETRYCOUNT, 5 ) );

	/* Configure Hardware for Type-A cards */
	CHECK_SUCCESS ( phhalHw_ApplyProtocolSettings ( &stHAL_ContactLC0, PHHAL_HW_CARDTYPE_ISO14443A ) );

    /* Perform Field Reset. */
    CHECK_SUCCESS ( phhalHw_FieldReset ( &stHAL_ContactLC0 ) );
	CHECK_SUCCESS ( phhalHw_Wait ( &stHAL_ContactLC0, 1, 5 ) );

	printf ( "\n\n" );
	printf ( "Card 1 #########################################################################################\n\n" );

    /* Activate Layer 3 card. */
    CHECK_SUCCESS ( phpalI14443p3a_ActivateCard ( &stI14443p3a, NULL, 0x00, aUid, &bLength, aSak, &bMoreCardsAvaliable ) );
	printf ( "UID: " );
	PrintData ( aUid, bLength );

    CHECK_SUCCESS ( phalMful_Read ( &stMful, 0x08, aData ) );

	aData[0] = 2;
    CHECK_SUCCESS ( phalMful_Write ( &stMful, 0x08, aData ) );
    CHECK_SUCCESS ( phalMful_Read ( &stMful, 0x08, aData ) );
	CHECK_SUCCESS ( phpalI14443p3a_HaltA ( &stI14443p3a ) );

	printf ( "\n\n" );
	printf ( "Card 2 #########################################################################################\n\n" );

    /* Activate Layer 3 card. */
    CHECK_SUCCESS ( phpalI14443p3a_ActivateCard ( &stI14443p3a, NULL, 0x00, aUid, &bLength, aSak, &bMoreCardsAvaliable ) );
	printf ( "UID: " );
	PrintData ( aUid, bLength );

    CHECK_SUCCESS ( phalMful_Read ( &stMful, 0x08, aData ) );

	aData[0] = 2;
    CHECK_SUCCESS ( phalMful_Write ( &stMful, 0x08, aData ) );
    CHECK_SUCCESS ( phalMful_Read ( &stMful, 0x08, aData ) );
	CHECK_SUCCESS ( phpalI14443p3a_HaltA ( &stI14443p3a ) );

    /* Close the reader ports. */
	CHECK_SUCCESS ( phbalReg_ClosePort ( &stBal_Contact ) );
    CHECK_SUCCESS ( phbalReg_ClosePort ( &stBal_Contactless ) );

	printf ( "\n\n" );
    printf ( "------------------------------------------------------------------------------------------------\n" );
    printf ( "No more tests available.\n" );
    printf ( "Press any key to continue...\n\n" );
    _getch ();

    /**
	 * Returns zero if the procedure was successful.
	 */
    return 0;
}
