/*
 * Copyright 2013, 2016, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Example Source for Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

/**
 * Header for this file
 */
#include "Example-Rd70x_Mful_Mfulc.h"

/**
 * Reader Library Headers
 */
#include <phbalReg.h>
#include <phhalHw.h>
#include <phpalI14443p3a.h>
#include <phpalI14443p4a.h>
#include <phpalI14443p4.h>
#include <phCidManager.h>
#include <phpalMifare.h>
#include <phalMful.h>
#include <phKeyStore.h>
#include <phCryptoRng.h>
#include <phCryptoSym.h>

/**
 * Standard Header
 */
#ifdef _WIN32
#pragma warning(push)           /* PRQA S 3116 */
#pragma warning(disable:4001)   /* PRQA S 3116 */
#endif
#include <memory.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <conio.h>
#ifdef _WIN32
#pragma warning(pop)            /* PRQA S 3116 */
#endif

#define KEYCOUNT                 0xFFU  /**< number of keys */
#define KEYVERSIONS              0x01U  /**< number of key versions */

#define PAGE_ADDRESS_2           0x02U  /**< Page 2 used for BCC1, internal, Lock Byte 0 and Lock Byte 1 */

/**
 * Page 4 address used in:
 *   UL/ULC as first user page
 *   T2T as LockControl TLV0, LockControl TLV1, LockControl TLV2, LockControl TLV3
 */
#define PAGE_ADDRESS_4           0x04U

/**
 * Page 5 address used in:
 *   UL/ULC as second user page
 *   T2T as LockControl TLV4, NDEFMessage TLV0, NDEFMessage TLV1, Terminator TLV0
 */
#define PAGE_ADDRESS_5           0x05U

#define OTP_ADDRESS              0x03U  /**< One time programmable (OTP) page address */
#define OTP_EXAMPLE_DATA_ADDRESS 0x08U  /**< OTP example data address */
#define ULC_KEY_PAGE_44_ADDRESS  0x2CU  /**< ULC 3DES key page address 44 */
#define ULC_KEY_PAGE_45_ADDRESS  0x2DU  /**< ULC 3DES key page address 45 */
#define ULC_KEY_PAGE_46_ADDRESS  0x2EU  /**< ULC 3DES key page address 46 */
#define ULC_KEY_PAGE_47_ADDRESS  0x2FU  /**< ULC 3DES key page address 47 */

#define UL_C_AUTH_0                  0x2AU  /**< Ultralight C AUTH0 page address */
#define UL_C_AUTH_1                  0x2BU  /**< Ultralight C AUTH1 page address */
#define NO_AUTH                      0x30U  /**< no Authentication needed */
#define PAGE_ACCESS_EXAMPLE_ADDRESS  0x27U  /**< Example page address to which the authentication is required */
#define READ_ONLY                    0x01U  /**< write access restricted, read access allowed without authentication */
#define READ_WRITE_RESTRICTED        0x00U  /**< read and write access restricted without authentication */

#define NAME_PAGE_ADDRESS        0x08U    /**< UL/ULC example: 'name' page address */
#define ADDRESS_PAGE_ADDRESS     0x09U    /**< UL/ULC example: 'address' page address */
#define LOCK_BYTE_1_LOCK_PAGE_8  0x01U    /**< UL/ULC example: Lock byte 1 should lock page 8 */

#define READ_RESPONSE_SIZE   0x10U  /**< UL/ULC read command response size */

#define PAGE_SIZE          0x04U  /**< page size */
#define CRYPTO_DATA_SIZE   0x08U  /**< crypto data size */
#define KEY_2K3DES_SIZE    0x10U  /**< 2K3DES key size */

/**
 * Key store
 */
#define RAND_KEY_2K3DES_ADDRESS  0x01U    /**< Random 2K3DES key address in keystore */
#define RAND_KEY_2K3DES_VERSION  0x00U    /**< Random 2K3DES key version in keystore */
#define UL_C_KEY_ADDRESS         0x02U    /**< Ultralight C key address in keystore */
#define UL_C_KEY_VERSION         0x00U    /**< Ultralight C key version in keystore */
#define KEY_POSITION             0x00U    /**< Key position */

/**
 * NFC Forum Type 2 Tag
 */
#define NFCT2T_MAGIC_NUMBER         0xE1U  /**< NFC T2T magic number */
#define NFCT2T_OPERATIONAL_VERSION  0x10U  /**< NFC T2T Operational version number */
#define NFCT2T_DATA_SIZE            0x12U  /**< NFC T2T 144 Bytes of data area */
#define NFCT2T_LOCK_CONTROL_TLV0    0x01U  /**< NFC T2T Lock Control TLV0 */
#define NFCT2T_LOCK_CONTROL_TLV1    0x03U  /**< NFC T2T Lock Control TLV1 */
#define NFCT2T_LOCK_CONTROL_TLV2    0xA0U  /**< NFC T2T Lock Control TLV2 */
#define NFCT2T_LOCK_CONTROL_TLV3    0x10U  /**< NFC T2T Lock Control TLV3 */
#define NFCT2T_LOCK_CONTROL_TLV4    0x44U  /**< NFC T2T Lock Control TLV4 */
#define NFCT2T_NDEF_MESSAGE_TLV0    0x03U  /**< NFC T2T NDEF Message TLV0 */
#define NFCT2T_NDEF_MESSAGE_TLV1    0x00U  /**< NFC T2T NDEF Message TLV1 */
#define NFCT2T_TERMINATOR_TLV0      0xFEU  /**< NFC T2T Terminator TLV0 */

#define LOCK_BYTE_0_LOCK_PAGE_4    0x10U  /**< Lock Byte set to lock page 4 */

/**
 *   OTP Example bytes: 0x1C05140A
 */
#define OTP_EXAMPLE_BYTE_0     0x1CU
#define OTP_EXAMPLE_BYTE_1     0x05U
#define OTP_EXAMPLE_BYTE_2     0x14U
#define OTP_EXAMPLE_BYTE_3     0x0AU

/** New ULC Key for personalization */
static uint8_t gaUlcKey[] = {0x49, 0x45, 0x4D, 0x4B, 0x41, 0x45, 0x52, 0x42, 0x21, 0x4E, 0x41, 0x43, 0x55, 0x4F, 0x59, 0x46};

static uint8_t gaUlcKeyPart2r[] = {0x4B, 0x4D, 0x45, 0x49};    /**< ULC key page 44 */
static uint8_t gaUlcKeyPart1r[] = {0x42, 0x52, 0x45, 0x41};    /**< ULC key page 45 */
static uint8_t gaUlcKeyPart4r[] = {0x43, 0x41, 0x4E, 0x21};    /**< ULC key page 46 */
static uint8_t gaUlcKeyPart3r[] = {0x46, 0x59, 0x4F, 0x55};    /**< ULC key page 47 */

static uint8_t gaName[] = { 0x01, 0x02, 0x03, 0x04};     /**< UL/ULC example: 'name' page data */
static uint8_t gaAddress[] = { 0x05, 0x06, 0x07, 0x08};  /**< UL/ULC example: 'address' page data */

#define CHECK_SUCCESS(x)                                                \
if ((x) != PH_ERR_SUCCESS)                                              \
{                                                                       \
    printf("An error occurred (0x%04X), press any key to exit...", (x)); \
    _getch();                                              \
    return 0;                                                           \
}

void PRINT_BUFFER(uint8_t * pBuffer, uint8_t bLength)
{
    uint8_t bIndex;

    for (bIndex = 0; bIndex < bLength; ++bIndex)
    {
        printf("%02X ", pBuffer[bIndex]);
    }
    printf("\n");
}

uint8_t resetCard(phalMful_Sw_DataParams_t * palMful,             /**< Pointer to MIFARE Ultralight parameter structure. */
                  phpalMifare_Sw_DataParams_t * palMifare,        /**< Pointer to MIFARE parameter structure. */
                  phKeyStore_Sw_DataParams_t * pKeyStore,         /**< Pointer to Keystore parameter structure. */
                  phCryptoSym_Sw_DataParams_t * pCryptoEnc,       /**< Pointer to Crypto ENC parameter structure. */
                  phCryptoRng_Sw_DataParams_t * pCryptoRng,       /**< CryptoRng parameter structure for Rng */
                  phhalHw_Rd70x_DataParams_t * pHal,              /**< Pointer to HAL parameter structure */
                  phpalI14443p3a_Sw_DataParams_t * palI14443p3a)  /**< PAL-ISO14443P3A parameter structure */
{
    phStatus_t status;
    uint8_t aSak[1];    /**< Select Acknowledge; uint8_t. */
    uint8_t aUid[10];   /**< complete Uid; uint8_t[4/7/10]. */
    uint8_t bMoreCardsAvaliable; /**< more card available */
    uint8_t bLength;    /**< Length */

    /**
     * init. Ultralight-C component
     */
    status = phalMful_Sw_Init(palMful,                          /**< [In] Pointer to this layer's parameter structure. */
                              sizeof(phalMful_Sw_DataParams_t), /**< [In] Specifies the size of the data parameter structure */
                              palMifare,                        /**< [In] Pointer to palMifare parameter structure. */
                              pKeyStore,                        /**< [In] Pointer to phKeystore parameter structure. */
                              pCryptoEnc,                       /**< [In] Pointer to phCrypto data parameters structure. */
                              pCryptoRng);                      /**< [In] Pointer to the parameter structure of the CryptoRng layer. */
    CHECK_SUCCESS(status);

    /**
     * Field Reset
     */
    status = phhalHw_FieldReset(pHal);  /**< [In] Pointer to this layer's parameter structure. */
    CHECK_SUCCESS(status);

    /**
     * Activate Layer 3 card
     */
    status = phpalI14443p3a_ActivateCard(palI14443p3a,  /**< [In] PAL-ISO14443P3A parameter structure */
                                         NULL,          /**< [In] unknown UID */
                                         0x00,          /**< [In] UID length 0 */
                                         aUid,          /**< [Out] complete Uid; uint8_t[4/7/10]. */
                                         &bLength,      /**< [Out] Length of Uid; 4/7/10. */
                                         aSak,          /**< [Out] Select Acknowledge; uint8_t. */
                                         &bMoreCardsAvaliable);  /**< [Out] Whether there are more cards in the field or not; uint8_t. */
    CHECK_SUCCESS(status);

    return PH_ERR_SUCCESS;
}


/**
 * Ultralight and Ultralight C examples: read, write, lock, OTP usecase
 */
uint8_t exampleUL_ULC(phalMful_Sw_DataParams_t * palMful,          /**< Pointer to MIFARE Ultralight parameter structure. */
                      phpalMifare_Sw_DataParams_t * palMifare,     /**< Pointer to MIFARE parameter structure. */
                      phKeyStore_Sw_DataParams_t * pKeyStore,      /**< Pointer to Keystore parameter structure. */
                      phCryptoSym_Sw_DataParams_t * pCryptoEnc,    /**< Pointer to Crypto ENC parameter structure. */
                      phCryptoRng_Sw_DataParams_t * pCryptoRng,    /**< Pointer to Crypto Rng parameter structure. */
                      phhalHw_Rd70x_DataParams_t * pHal,           /**< Pointer to HAL parameter structure */
                      phpalI14443p3a_Sw_DataParams_t * palI14443p3a) /**< Pointer to PAL-ISO14443P3A parameter structure */
{
    phStatus_t status;
    uint8_t aData[READ_RESPONSE_SIZE];           /**< Response data */
    uint8_t aEncryptedData[CRYPTO_DATA_SIZE];    /**< Encrypted data */
    uint8_t aDecryptedData[CRYPTO_DATA_SIZE];    /**< decrypted data */
    uint8_t aKey2K3DES[KEY_2K3DES_SIZE];         /**< 2K3DES Key */
    uint8_t aEncryptedDataFirstPart[PAGE_SIZE];  /**< First part of encrypted data */
    uint8_t aEncryptedDataSecondPart[PAGE_SIZE]; /**< Second part of encrypted data */
    uint8_t aDataFirstPart[READ_RESPONSE_SIZE];  /**< First part of data */
    uint8_t aDataSecondPart[READ_RESPONSE_SIZE]; /**< Second part of data */
    uint8_t aDataToDecrypt[CRYPTO_DATA_SIZE];    /**< Data to be decrypted */

    /**
     * a.) Read, write two parts: One is 'name' and the other 'address'
     *     than only lock 'name'.
     * Perform Ultralight Read at NAME_PAGE_ADDRESS
     */
    status = phalMful_Read(palMful,             /**< [In] Pointer to this layer's parameter structure. */
                           NAME_PAGE_ADDRESS,   /**< [In] Address on Picc to read from. */
                           aData);              /**< [Out] pData[#PHAL_MFUL_READ_BLOCK_LENGTH] containing data returned from the Picc. */
    CHECK_SUCCESS(status);

    /**
     * Perform Ultralight Read at ADDRESS_PAGE_ADDRESS
     */
    status = phalMful_Read(palMful,                /**< [In] Pointer to this layer's parameter structure. */
                           ADDRESS_PAGE_ADDRESS,   /**< [In] Address on Picc to read from. */
                           aData);                 /**< [Out] pData[#PHAL_MFUL_READ_BLOCK_LENGTH] containing data returned from the Picc. */
    CHECK_SUCCESS(status);

    /**
     * Perform Ultralight Write at NAME_PAGE_ADDRESS
     */
    status = phalMful_Write(palMful,            /**< [In] Pointer to this layer's parameter structure. */
                            NAME_PAGE_ADDRESS,  /**< [In] Address on Picc to write to. */
                            gaName);            /**< [In] pData[#PHAL_MFUL_WRITE_BLOCK_LENGTH] containing block to be written to the Picc. */
    CHECK_SUCCESS(status);

    /**
     * Perform Ultralight Write at ADDRESS_PAGE_ADDRESS
     */
    status = phalMful_Write(palMful,               /**< [In] Pointer to this layer's parameter structure. */
                            ADDRESS_PAGE_ADDRESS,  /**< [In] Address on Picc to write to. */
                            gaAddress);            /**< [In] pData[#PHAL_MFUL_WRITE_BLOCK_LENGTH] containing block to be written to the Picc. */
    CHECK_SUCCESS(status);

    /**
     * Perform Ultralight Write: lock page 8
     * aData byte 0: BCC1 = 0x00
     * aData byte 1: internal  = 0x00
     * aData byte 2: Lock Byte 0 = 0x00
     * aData byte 3: Lock Byte 1 = LOCK_BYTE_1_LOCK_PAGE_8
     */
    aData[0] = aData[1] = aData[2] = 0x00;
    aData[3] = 0x00; /* LOCK_BYTE_1_LOCK_PAGE_8; */
    status = phalMful_Write(palMful, PAGE_ADDRESS_2, aData);
    CHECK_SUCCESS(status);

    /**
     * read
     */
    status = phalMful_Read(palMful, NAME_PAGE_ADDRESS, aData);
    CHECK_SUCCESS(status);

    /**
     * write impossible
     */
    status = phalMful_Write(palMful, NAME_PAGE_ADDRESS, aData);
    if ((status) != PH_ERR_SUCCESS)
    {
        /**
         * write was impossible
         */
        status = PH_ERR_SUCCESS;
    }
    else
    {
        printf("An error occurred: write should be impossible, press any key to exit...");
        _getch();
        return 0;
    }

    /**
     * reset card
     */
    status = resetCard(palMful,        /**< Pointer to MIFARE Ultralight parameter structure. */
                       palMifare,      /**< Pointer to MIFARE parameter structure. */
                       pKeyStore,      /**< Pointer to Keystore parameter structure. */
                       pCryptoEnc,     /**< Pointer to Crypto ENC parameter structure. */
                       pCryptoRng,     /**< CryptoRng parameter structure for Rng */
                       pHal,           /**< Pointer to HAL parameter structure */
                       palI14443p3a);  /**< PAL-ISO14443P3A parameter structure */
    CHECK_SUCCESS(status);

    /**
     * b.) OTP bits example
     * set data to OTP_EXAMPLE_BYTE_x
     */
    memset(aData, 0x00, sizeof(aData));
    aData[0] = OTP_EXAMPLE_BYTE_0;
    aData[1] = OTP_EXAMPLE_BYTE_1;
    aData[2] = OTP_EXAMPLE_BYTE_2;
    aData[3] = OTP_EXAMPLE_BYTE_3;

    /**
     * obtain random bytes
     */
    status = phCryptoRng_Rnd(pCryptoRng,       /**< [In] CryptoRng parameter structure for Rng */
                             KEY_2K3DES_SIZE,  /**< [In] number of random bytes to generate */
                             aKey2K3DES);      /**< [Out] generated bytes; uint8_t[dwNumBytes] */
    CHECK_SUCCESS(status);

    /**
     * load a random 2K3DES key into the key store at position 0x01
     */
    status = phKeyStore_FormatKeyEntry(pKeyStore,                    /**< [In] Pointer to this layer's parameter structure. */
                                       RAND_KEY_2K3DES_ADDRESS,      /**< [In] KeyEntry number to be Formatted (0x00 to F0). */
                                       PH_KEYSTORE_KEY_TYPE_2K3DES); /**< [In] New Key type of the KeyEntry (predefined type of KeyType).*/
    CHECK_SUCCESS(status);

    status = phKeyStore_SetKeyAtPos(pKeyStore,                /**< [In] Pointer to this layer's parameter structure. */
                                    RAND_KEY_2K3DES_ADDRESS,  /**< [In] Key number of the key to be loaded. */
                                    KEY_POSITION,             /**< [In] Key position to be updated. */
                                    PH_KEYSTORE_KEY_TYPE_2K3DES,  /**< [In] Key type of the key to be loaded.*/
                                    aKey2K3DES,               /**< [In] Pointer to the key itself.*/
                                    RAND_KEY_2K3DES_VERSION); /**< [In] New Key version of the key to be updated. */
    CHECK_SUCCESS(status);

    /**
     * load key
     */
    status = phCryptoSym_LoadKey(pCryptoEnc,                    /**< [In] CryptoSym parameter structure for ENC */
                                 RAND_KEY_2K3DES_ADDRESS,       /**< [In] Number of the Key */
                                 RAND_KEY_2K3DES_VERSION,       /**< [In] Version of the key */
                                 PH_KEYSTORE_KEY_TYPE_2K3DES);  /**< [In] Type of Key */
    CHECK_SUCCESS(status);

    /**
     * encrypt data
     */
    status = phCryptoSym_Encrypt(pCryptoEnc,        /**< [In] CryptoSym parameter structure for ENC */
                                 PH_CRYPTOSYM_CIPHER_MODE_CBC,  /**< [In] Option byte specifying the cipher mode and the update behavior of the IV */
                                 aData,             /**< [In] plain data buffer */
                                 CRYPTO_DATA_SIZE,  /**< [In] length of plain and encrypted data buffer - needs to be a multiple of the current block size */
                                 aEncryptedData);   /**< [Out] encrypted data buffer */
    CHECK_SUCCESS(status);

    /**
     * write first part of the encrypted data to the OTP bytes
     */
    memcpy(aEncryptedDataFirstPart, aEncryptedData, PAGE_SIZE);
    memcpy(aEncryptedDataSecondPart, &aEncryptedData[PAGE_SIZE], PAGE_SIZE);
    status = phalMful_Write(palMful, OTP_ADDRESS, aEncryptedDataFirstPart);
    CHECK_SUCCESS(status);

    /**
     * write second part of the encrypted data to a user block
     */
    status = phalMful_Write(palMful, OTP_EXAMPLE_DATA_ADDRESS, aEncryptedDataSecondPart);
    CHECK_SUCCESS(status);

    /**
     * read first part of the encrypted data of the OTP bytes
     */
    /* set OTP_ADDRESS for OTP */
    status = phalMful_Read(palMful, OTP_ADDRESS, aDataFirstPart);
    CHECK_SUCCESS(status);

    /**
     * read second part of the encrypted data of a user block
     */
    status = phalMful_Read(palMful, OTP_EXAMPLE_DATA_ADDRESS, aDataSecondPart);
    CHECK_SUCCESS(status);

    /**
     * merge both decrypted data blocks
     */
    memcpy(aDataToDecrypt, aDataFirstPart, PAGE_SIZE);
    memcpy(&aDataToDecrypt[PAGE_SIZE], aDataSecondPart, PAGE_SIZE);

    /**
     * decrypt data
     */
    status = phCryptoSym_Decrypt(pCryptoEnc,        /**< [In] Pointer to this layer's parameter structure. */
                                 PH_CRYPTOSYM_CIPHER_MODE_CBC,  /**< [In] Option byte specifying the cipher mode and the update behavior of the IV */
                                 aDataToDecrypt,    /**< [In] encrypted data buffer */
                                 CRYPTO_DATA_SIZE,  /**< [In] length of plain and encrypted data buffer - needs to be a multiple of the current block size */
                                 aDecryptedData);   /**< [Out] plain data buffer */
    CHECK_SUCCESS(status);

    /**
     * compare the data
     */
    if (memcmp(aDecryptedData, aData, PAGE_SIZE))
    {
        printf("An error occurred: decrypted data should be equal to plain data, press any key to exit...");
        _getch();
        return 0;
    }
    else
    {
        /**
         * <success>
         */
    }

    return PH_ERR_SUCCESS;
}


/**
 * Ultralight C Examples
 */
uint8_t exampleULC(phalMful_Sw_DataParams_t * palMful,          /**< Pointer to MIFARE Ultralight parameter structure. */
                   phpalMifare_Sw_DataParams_t * palMifare,     /**< Pointer to MIFARE parameter structure. */
                   phKeyStore_Sw_DataParams_t * pKeyStore,      /**< Pointer to Keystore parameter structure. */
                   phCryptoSym_Sw_DataParams_t * pCryptoEnc,    /**< Pointer to Crypto ENC parameter structure. */
                   phCryptoRng_Sw_DataParams_t * pCryptoRng,    /**< Pointer to Crypto Rng parameter structure. */
                   phhalHw_Rd70x_DataParams_t * pHal,           /**< Pointer to HAL parameter structure */
                   phpalI14443p3a_Sw_DataParams_t * palI14443p3a) /**< Pointer to PAL-ISO14443P3A parameter structure */
{
    phStatus_t status;
    uint8_t aData[READ_RESPONSE_SIZE];
    uint8_t aAuth0Backup[READ_RESPONSE_SIZE];
    uint8_t aAuth1Backup[READ_RESPONSE_SIZE];

    /**
     * Prepare KeyStore for ULC key
     */
    status = phKeyStore_FormatKeyEntry(pKeyStore, UL_C_KEY_ADDRESS, PH_KEYSTORE_KEY_TYPE_2K3DES);
    CHECK_SUCCESS(status);

    /* Load default ULC key into keystore */
    status = phKeyStore_SetKeyAtPos(pKeyStore, UL_C_KEY_ADDRESS, KEY_POSITION, PH_KEYSTORE_KEY_TYPE_2K3DES, gaUlcKey, UL_C_KEY_VERSION);
    CHECK_SUCCESS(status);

    /**
    * write UL key
    */
    printf("Checking Card state...\n");

    status = phalMful_Write(palMful, ULC_KEY_PAGE_44_ADDRESS, gaUlcKeyPart1r);
    if ((status & PH_ERR_MASK) == PH_ERR_SUCCESS)
    {
        printf("Card personalization possible, writing default key...\n");
        status = phalMful_Write(palMful, ULC_KEY_PAGE_45_ADDRESS, gaUlcKeyPart2r);
        CHECK_SUCCESS(status);
        status = phalMful_Write(palMful, ULC_KEY_PAGE_46_ADDRESS, gaUlcKeyPart3r);
        CHECK_SUCCESS(status);
        status = phalMful_Write(palMful, ULC_KEY_PAGE_47_ADDRESS, gaUlcKeyPart4r);
        CHECK_SUCCESS(status);
    }
    else
    {
        printf("Card already personalized, trying default key...\n");
    }

    /**
     * reset Card
     */
    status = resetCard(palMful,        /**< Pointer to MIFARE Ultralight parameter structure. */
                       palMifare,      /**< Pointer to MIFARE parameter structure. */
                       pKeyStore,      /**< Pointer to Keystore parameter structure. */
                       pCryptoEnc,     /**< Pointer to Crypto ENC parameter structure. */
                       pCryptoRng,     /**< CryptoRng parameter structure for Rng */
                       pHal,           /**< Pointer to HAL parameter structure */
                       palI14443p3a);  /**< PAL-ISO14443P3A parameter structure */
    CHECK_SUCCESS(status);

    /**
     * a.) Perform Ultralight-C Authentication
     */
    status = phalMful_UlcAuthenticate(palMful,            /**< [In] Pointer to MIFARE Ultralight parameter structure. */
									  0x00,				  /**< [In] /**< [In] Option for specifying the diversification input. Only valid for Sam AV3
														   *			\arg 0x00: PHHAL_HW_SAMAV3_CMD_ULC_AUTHENTICATE_DIV_OFF
														   *			\arg 0x01: PHHAL_HW_SAMAV3_CMD_ULC_AUTHENTICATE_DIV_ON
								 						   */
                                      UL_C_KEY_ADDRESS,   /**< [In] Key number to be used in authentication. */
                                      UL_C_KEY_VERSION,   /**< [In] Key version to be used in authentication. */
									  NULL,				  /**< [In] Diversification input for key diversification. (1 to 31 byte(s) input). */
									  0);				  /**< [In] Length of diversification input. */
    CHECK_SUCCESS(status);

    /**
     * b.) Backup AUTH0 register
     */
    status = phalMful_Read(palMful, UL_C_AUTH_0, aAuth0Backup);
    CHECK_SUCCESS(status);

    /**
     *c.) Ultralight C r/w limit example
     *     set auth0 at page PAGE_ACCESS_EXAMPLE_ADDRESS
     */
    memcpy(aData, aAuth0Backup, sizeof(aAuth0Backup)); /* PRQA S 3200 */
    aData[0] = PAGE_ACCESS_EXAMPLE_ADDRESS;
    aData[1] = aData[2] = aData[3] = 0x00;
    status = phalMful_Write(palMful, UL_C_AUTH_0, aData);
    CHECK_SUCCESS(status);

    /**
     * d.) Backup AUTH1 register
     */
    status = phalMful_Read(palMful, UL_C_AUTH_1, aAuth1Backup);
    CHECK_SUCCESS(status);

    /**
     *e.) Ultralight C r/w limit example
     *     set auth1 to r/w access restricted
     */
    memcpy(aData, aAuth1Backup, sizeof(aAuth1Backup)); /* PRQA S 3200 */
    aData[0] = READ_WRITE_RESTRICTED;
    aData[1] = aData[2] = aData[3] = 0x00;
    status = phalMful_Write(palMful, UL_C_AUTH_1, aData);
    CHECK_SUCCESS(status);

    /**
     * reset Card
     */
    status = resetCard(palMful,        /**< Pointer to MIFARE Ultralight parameter structure. */
                       palMifare,      /**< Pointer to MIFARE parameter structure. */
                       pKeyStore,      /**< Pointer to Keystore parameter structure. */
                       pCryptoEnc,     /**< Pointer to Crypto ENC parameter structure. */
                       pCryptoRng,     /**< CryptoRng parameter structure for Rng */
                       pHal,           /**< Pointer to HAL parameter structure */
                       palI14443p3a);  /**< PAL-ISO14443P3A parameter structure */
    CHECK_SUCCESS(status);

    /**
    * read should not be possible without authentication
    */
    status = phalMful_Read(palMful, PAGE_ACCESS_EXAMPLE_ADDRESS, aData);
    if ((status) == PH_ERR_SUCCESS)
    {
        printf("An error occurred: read should not be possible, press any key to exit...");
        _getch();
        return 1;
    }

    /**
     * reset Card
     */
    status = resetCard(palMful,        /**< Pointer to MIFARE Ultralight parameter structure. */
                       palMifare,      /**< Pointer to MIFARE parameter structure. */
                       pKeyStore,      /**< Pointer to Keystore parameter structure. */
                       pCryptoEnc,     /**< Pointer to Crypto ENC parameter structure. */
                       pCryptoRng,     /**< CryptoRng parameter structure for Rng */
                       pHal,           /**< Pointer to HAL parameter structure */
                       palI14443p3a);  /**< PAL-ISO14443P3A parameter structure */
    CHECK_SUCCESS(status);

    /**
     * Perform Ultralight-C Authentication
     */
    status = phalMful_UlcAuthenticate(palMful, 0x00, UL_C_KEY_ADDRESS, UL_C_KEY_VERSION, NULL, 0);
    CHECK_SUCCESS(status);

    /**
    * read should be possible after authentication
    */
    status = phalMful_Read(palMful, PAGE_ACCESS_EXAMPLE_ADDRESS, aData);
    CHECK_SUCCESS(status);

    /**
     * reset AUTH0 register
     */
    status = phalMful_Write(palMful, UL_C_AUTH_0, aAuth0Backup);
    CHECK_SUCCESS(status);

    /**
     * reset AUTH1 register
     */
    status = phalMful_Write(palMful, UL_C_AUTH_1, aAuth1Backup);
    CHECK_SUCCESS(status);

    return PH_ERR_SUCCESS;
}

/**
 * MIFARE ULC NFC Forum Type 2 Tag example
 * MIFARE ULC Blank Card is needed
 */
uint8_t exampleNFCT2T(phalMful_Sw_DataParams_t * palMful,          /**< Pointer to MIFARE Ultralight parameter structure. */
                      phpalMifare_Sw_DataParams_t * palMifare,     /**< Pointer to MIFARE parameter structure. */
                      phKeyStore_Sw_DataParams_t * pKeyStore,      /**< Pointer to Keystore parameter structure. */
                      phCryptoSym_Sw_DataParams_t * pCryptoEnc,    /**< Pointer to Crypto ENC parameter structure. */
                      phCryptoRng_Sw_DataParams_t * pCryptoRng,    /**< Pointer to Crypto Rng parameter structure. */
                      phhalHw_Rd70x_DataParams_t * pHal,           /**< Pointer to HAL parameter structure */
                      phpalI14443p3a_Sw_DataParams_t * palI14443p3a) /**< Pointer to PAL-ISO14443P3A parameter structure */
{
    phStatus_t status;
    uint8_t aWriteData[PAGE_SIZE];
    uint8_t aData[READ_RESPONSE_SIZE];

    /**
     * a.) Format ULC as NFC Forum Type 2 Tag
     *     write CC-OTP page 3: NFCT2T magic number, operational version, data size, 0x00
     */
    aWriteData[0] = NFCT2T_MAGIC_NUMBER;          /**< NFCT2T magic number */
    aWriteData[1] = NFCT2T_OPERATIONAL_VERSION;   /**< NFCT2T operational version */
    aWriteData[2] = NFCT2T_DATA_SIZE;             /**< NFCT2T data area size divided by 8 */
    aWriteData[3] = 0x00;
    status = phalMful_Write(palMful, OTP_ADDRESS, aWriteData);
    CHECK_SUCCESS(status);

    /**
     * write page 4: LockControl TLV0, LockControl TLV1, LockControl TLV2, LockControl TLV3
     */
    aWriteData[0] = NFCT2T_LOCK_CONTROL_TLV0;   /**< NFCT2T LockControl TLV0 */
    aWriteData[1] = NFCT2T_LOCK_CONTROL_TLV1;   /**< NFCT2T LockControl TLV1 */
    aWriteData[2] = NFCT2T_LOCK_CONTROL_TLV2;   /**< NFCT2T LockControl TLV2 */
    aWriteData[3] = NFCT2T_LOCK_CONTROL_TLV3;   /**< NFCT2T LockControl TLV3 */
    status = phalMful_Write(palMful, PAGE_ADDRESS_4, aWriteData);
    CHECK_SUCCESS(status);

    /**
     * write page 5: LockControl TLV4, NDEFMessage TLV0, NDEFMessage TLV1, Terminator TLV0
     */
    aWriteData[0] = NFCT2T_LOCK_CONTROL_TLV4;   /**< NFCT2T LockControl TLV0 */
    aWriteData[1] = NFCT2T_NDEF_MESSAGE_TLV0;   /**< NFCT2T NDEF Message TLV0 */
    aWriteData[2] = NFCT2T_NDEF_MESSAGE_TLV1;   /**< NFCT2T NDEF Message TLV1 */
    aWriteData[3] = NFCT2T_TERMINATOR_TLV0;     /**< NFCT2T Terminator TLV0 */
    status = phalMful_Write(palMful, PAGE_ADDRESS_5, aWriteData);
    CHECK_SUCCESS(status);

    /**
     * Load Mifare Ultralight key at position 2 in KeyStore
     */
    status = phKeyStore_FormatKeyEntry(pKeyStore, UL_C_KEY_ADDRESS, PH_KEYSTORE_KEY_TYPE_2K3DES);
    CHECK_SUCCESS(status);
    status = phKeyStore_SetKeyAtPos(pKeyStore, UL_C_KEY_ADDRESS, KEY_POSITION, PH_KEYSTORE_KEY_TYPE_2K3DES, gaUlcKey, UL_C_KEY_VERSION);
    CHECK_SUCCESS(status);

    /**
     * write UL key
     */
    status = phalMful_Write(palMful, ULC_KEY_PAGE_44_ADDRESS, gaUlcKeyPart1r);
    CHECK_SUCCESS(status);
    status = phalMful_Write(palMful, ULC_KEY_PAGE_45_ADDRESS, gaUlcKeyPart2r);
    CHECK_SUCCESS(status);
    status = phalMful_Write(palMful, ULC_KEY_PAGE_46_ADDRESS, gaUlcKeyPart3r);
    CHECK_SUCCESS(status);
    status = phalMful_Write(palMful, ULC_KEY_PAGE_47_ADDRESS, gaUlcKeyPart4r);
    CHECK_SUCCESS(status);

    /**
     * reset
     */
    status = resetCard(palMful, palMifare, pKeyStore, pCryptoEnc, pCryptoRng, pHal, palI14443p3a);
    CHECK_SUCCESS(status);

    /**
     * Ultralight C NFCT2T write limit example
     * set auth0 at page 0x27
     */
    aWriteData[0] = PAGE_ACCESS_EXAMPLE_ADDRESS;
    aWriteData[1] = aWriteData[2] = aWriteData[3] = 0x00;
    status = phalMful_Write(palMful, UL_C_AUTH_0, aWriteData);
    CHECK_SUCCESS(status);

    /**
     * set auth1 to write access restricted
     */
    aWriteData[0] = READ_ONLY;
    aWriteData[1] = aWriteData[2] = aWriteData[3] = 0x00;
    status = phalMful_Write(palMful, UL_C_AUTH_1, aWriteData);
    CHECK_SUCCESS(status);

    /**
     * reset
     */
    status = resetCard(palMful, palMifare, pKeyStore, pCryptoEnc, pCryptoRng, pHal, palI14443p3a);
    CHECK_SUCCESS(status);

    /**
     * read
     */
    status = phalMful_Read(palMful, PAGE_ACCESS_EXAMPLE_ADDRESS, aData);
    CHECK_SUCCESS(status);

    /**
     * write impossible
     */
    status = phalMful_Write(palMful, PAGE_ACCESS_EXAMPLE_ADDRESS, aWriteData);
    if ((status) != PH_ERR_SUCCESS)
    {
        /**
         * write is impossible, as it should be
         */
        status = PH_ERR_SUCCESS;
    }
    else
    {
        printf("An error occurred: read should not be possible, press any key to exit...\n");
        _getch();
        return 0;
    }

    /**
     * reset auth status
     */
    status = resetCard(palMful, palMifare, pKeyStore, pCryptoEnc, pCryptoRng, pHal, palI14443p3a);
    CHECK_SUCCESS(status);

    /**
     * Perform Ultralight-C Authentication
     */
    status = phalMful_UlcAuthenticate(palMful, 0x00, UL_C_KEY_ADDRESS, UL_C_KEY_VERSION, NULL, 0);
    CHECK_SUCCESS(status);

    /**
     * write
     */
    status = phalMful_Write(palMful, PAGE_ACCESS_EXAMPLE_ADDRESS, aWriteData);
    CHECK_SUCCESS(status);

    /**
     * read
     */
    status = phalMful_Read(palMful, PAGE_ACCESS_EXAMPLE_ADDRESS, aData);
    CHECK_SUCCESS(status);

    /**
     * reset r/w limits
     * set auth0 at page NO_AUTH
     */
    aWriteData[0] = NO_AUTH;
    aWriteData[1] = aWriteData[2] = aWriteData[3] = 0x00;
    status = phalMful_Write(palMful, UL_C_AUTH_0, aWriteData);
    CHECK_SUCCESS(status);

    /**
     * set auth1 to r/w access restricted
     */
    aWriteData[0] = READ_WRITE_RESTRICTED;
    aWriteData[1] = aWriteData[2] = aWriteData[3] = 0x00;
    status = phalMful_Write(palMful, UL_C_AUTH_1, aWriteData);
    CHECK_SUCCESS(status);

    /**
     * write possible after reset r/w limits
     */
    status = phalMful_Write(palMful, PAGE_ACCESS_EXAMPLE_ADDRESS, aWriteData);
    CHECK_SUCCESS(status);

    /**
     * read
     */
    status = phalMful_Read(palMful, PAGE_ACCESS_EXAMPLE_ADDRESS, aData);
    CHECK_SUCCESS(status);

    return PH_ERR_SUCCESS;

}

int __cdecl main()
{
    phStatus_t status;
    uint8_t aReaderListPegoda[512];    /**< Pegoda reader list */
    uint8_t aHalBufferReaderTx[0xFF];  /**< HAL Reader transmit buffer */
    uint8_t aHalBufferReaderRx[0xFF];  /**< HAL Reader receive buffer */
    uint8_t aSak[1];                   /**< Select Acknowledge; uint8_t. */
    uint8_t aUid[10];                  /**< complete Uid; uint8_t[4/7/10]. */
    uint8_t aSeed[16];                 /**< Seed */
    uint8_t bMoreCardsAvaliable;       /**< more card available */
    uint8_t bLength;                   /**< length */
    uint16_t wCount;                   /**< counter */
    int iChoice;                       /**< choice */

    /**
     * KeyStore
     */
    phKeyStore_Sw_KeyEntry_t aKeyEntry[KEYCOUNT];
    phKeyStore_Sw_KeyVersionPair_t aKeyVersion[KEYCOUNT * KEYVERSIONS];
    phKeyStore_Sw_KUCEntry_t aKeyUsageCounter[KEYCOUNT];

    /**
     * Bfl data parameter storage
     */
    phbalReg_Rd70xUsbWin_DataParams_t balReader;    /**< BAL parameter structure */
    phhalHw_Rd70x_DataParams_t halReader;           /**< HAL parameter structure */
    phpalI14443p3a_Sw_DataParams_t palI14443p3a;    /**< PAL-ISO14443P3A parameter structure */
    phpalI14443p4a_Sw_DataParams_t palI14443p4a;    /**< PAL-ISO14443P4A parameter structure */
    phpalI14443p4_Sw_DataParams_t palI14443p4;      /**< ISO14443-4 parameter structure */
    phpalMifare_Sw_DataParams_t palMifare;          /**< PAL-MIFARE parameter structure */
    phCryptoSym_Sw_DataParams_t cryptoEnc;          /**< CryptoSym parameter structure for ENC */
    phCryptoSym_Sw_DataParams_t cryptoMac;          /**< CryptoSym parameter structure for MAC */
    phCryptoSym_Sw_DataParams_t cryptoSymRng;       /**< CryptoSym parameter structure for SymRng */
    phCryptoRng_Sw_DataParams_t cryptoRng;          /**< CryptoRng parameter structure for Rng */
    phKeyStore_Sw_DataParams_t keyStore;            /**< KeyStore parameter structure */
    phalMful_Sw_DataParams_t alMful;                /**< Pointer to MIFARE Ultralight parameter structure. */
    void * pHal;                                    /**< Pointer to the parameter structure of the HAL layer. */

    printf("\nNxpRdLib ANSI-C Example Program V1.0a\n\n");
    printf("Please ensure that a Pegoda reader is connected and in working condition.\n\n");
    printf("Performing startup...\n\n");

    /**
     * Initialize the Reader BAL component
     */
    status = phbalReg_Rd70xUsbWin_Init(&balReader,          /**< [In] BAL parameter structure */
                                       sizeof(balReader));  /**< [In] Specifies the size of the data parameter structure */
    CHECK_SUCCESS(status);

    /**
     * Get list of connected pegoda readers
     */
    status = phbalReg_GetPortList(&balReader,         /**< [In] BAL parameter structure */
                                  sizeof(aReaderListPegoda),  /**< [In] Buffer Size of Port Name String */
                                  aReaderListPegoda,  /**< [Out] Port Name as Multi-Unicode String */
                                  &wCount);           /**< [Out] Number of found port strings. */
    CHECK_SUCCESS(status);

    /**
     * Connect to the first reader out of the list
     */
    status = phbalReg_SetPort(&balReader,          /**< [In] BAL parameter structure */
                              aReaderListPegoda);  /**< [In] Port Name as String. */
    CHECK_SUCCESS(status);

    /**
     * open the reader port
     */
    status = phbalReg_OpenPort(&balReader);        /**< [In] BAL parameter structure */
    CHECK_SUCCESS(status);

    /**
     * init. the Reader HAL component
    */
    status = phhalHw_Rd70x_Init(&halReader,          /**< [In] HAL parameter structure */
                                sizeof(halReader),   /**< [In] Specifies the size of the data parameter structure */
                                &balReader,          /**< [In] BAL parameter structure */
                                aHalBufferReaderTx,  /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
                                sizeof(aHalBufferReaderTx),  /**< [In] Size of the global transmit buffer. */
                                aHalBufferReaderRx,  /**< [In] Pointer to global receive buffer used by the Exchange() function. */
                                sizeof(aHalBufferReaderRx));  /**< [In] Size of the global receive buffer. */
    CHECK_SUCCESS(status);

    /**
     * The default reader is the Pegoda
     */
    pHal = &halReader;

    /**
     * init. keystore
     */
    status = phKeyStore_Sw_Init(&keyStore,         /**< [In] KeyStore parameter structure */
                                sizeof(keyStore),  /**< [In] Specifies the size of the data parameter structure */
                                aKeyEntry,         /**< [In] Pointer to a storage containing the key entries. */
                                KEYCOUNT,          /**< [In] number of pKeyEntries. */
                                aKeyVersion,       /**< [In] Pointer to a storage containing the key version pairs. */
                                KEYVERSIONS,       /**< [In] amount of key versions available in each key entry. */
                                aKeyUsageCounter,  /**< [In] Key usage counter entry storage, size = sizeof(phKeyStore_Sw_KUCEntry_t) * wNumKUCEntries */
                                KEYCOUNT);         /**< [In] Number of Key usage counter entries. */
    CHECK_SUCCESS(status);

    /**
     * init. crypto
     */
    status = phCryptoSym_Sw_Init(&cryptoEnc,    /**< [In] CryptoSym parameter structure for ENC */
                                 sizeof(phCryptoSym_Sw_DataParams_t),  /**< [In] Specifies the size of the data parameter structure */
                                 &keyStore);        /**< [In] Pointer to a key store structure (can be null).*/
    CHECK_SUCCESS(status);
    status = phCryptoSym_Sw_Init(&cryptoMac,    /**< [In] CryptoSym parameter structure for MAC */
                                 sizeof(phCryptoSym_Sw_DataParams_t),    /**< [In] Specifies the size of the data parameter structure */
                                 &keyStore);        /**< [In] Pointer to a key store structure (can be null).*/
    CHECK_SUCCESS(status);
    status = phCryptoSym_Sw_Init(&cryptoSymRng,    /**< [In] CryptoSym parameter structure for symRng */
                                 sizeof(phCryptoSym_Sw_DataParams_t),  /**< [In] Specifies the size of the data parameter structure */
                                 &keyStore);        /**< [In] Pointer to a key store structure (can be null).*/
    CHECK_SUCCESS(status);
    status = phCryptoRng_Sw_Init(&cryptoRng,    /**< [In] CryptoRng parameter structure for Rng */
                                 sizeof(phCryptoRng_Sw_DataParams_t),  /**< [In] Specifies the size of the data parameter structure */
                                 &cryptoSymRng);  /**< [In] Pointer to a key store structure (can be null).*/
    CHECK_SUCCESS(status);

    /**
     * clear aSeed
     */
    memset(aSeed, 0x00, sizeof(aSeed));

    /**
     * seed the random number
     */
    status = phCryptoRng_Seed(&cryptoRng,    /**< [In] CryptoRng parameter structure for Rng */
                              aSeed,         /**< [In] Seed */
                              sizeof(aSeed));  /**< [In] Size of the seed. */
    CHECK_SUCCESS(status);

    /**
     * init the 14443-3A component
     */
    status = phpalI14443p3a_Sw_Init(&palI14443p3a,  /**< [In] PAL-ISO14443P3A parameter structure */
                                    sizeof(palI14443p3a),  /**< [In] Specifies the size of the data parameter structure */
                                    pHal);          /**< [In] Pointer to the parameter structure of the underlying layer.*/
    CHECK_SUCCESS(status);

    /**
     * init the 14443-4A component
     */
    status = phpalI14443p4a_Sw_Init(&palI14443p4a,  /**< [In]  PAL-ISO14443P4A parameter structure */
                                    sizeof(palI14443p4a),  /**< [In] Specifies the size of the data parameter structure */
                                    pHal);          /**< [In] Pointer to the parameter structure of the underlying layer.*/
    CHECK_SUCCESS(status);

    /**
     * init the 14443-4 component
     */
    status = phpalI14443p4_Sw_Init(&palI14443p4,  /**< [In] ISO14443-4 parameter structure */
                                   sizeof(palI14443p4),  /**< [In] Specifies the size of the data parameter structure */
                                   pHal);         /**< [In] Pointer to the parameter structure of the underlying HAL layer.*/
    CHECK_SUCCESS(status);

    /**
     * init. MIFARE
     */
    status = phpalMifare_Sw_Init(&palMifare,     /**< [In] PAL-MIFARE parameter structure */
                                 sizeof(palMifare),  /**< [In] Specifies the size of the data parameter structure */
                                 pHal,           /**< [In] Pointer to the parameter structure of the underlying HAL layer. */
                                 &palI14443p4);  /**< [In] Pointer to the parameter structure of the underlying ISO14443-4 layer. */
    CHECK_SUCCESS(status);

    /**
     * init. Ultralight-C component
     */
    status = phalMful_Sw_Init(&alMful,           /**< [In] Pointer to MIFARE Ultralight parameter structure. */
                              sizeof(phalMful_Sw_DataParams_t), /**< [In] Specifies the size of the data parameter structure */
                              &palMifare,        /**< [In] PAL-MIFARE parameter structure */
                              &keyStore,         /**< [In] Pointer to a key store structure */
                              &cryptoEnc,        /**< [In] CryptoSym parameter structure for ENC */
                              &cryptoRng);       /**< [In] CryptoRng parameter structure for Rng */
    CHECK_SUCCESS(status);

    /**
     * Field Reset
     */
    status = phhalHw_FieldReset(pHal);  /**< [In] Pointer to the parameter structure of the HAL layer. */
    CHECK_SUCCESS(status);

    /**
     * Configure HAL for Type-A cards
     */
    status = phhalHw_ApplyProtocolSettings(pHal,  /**< [In] Pointer to this layer's parameter structure. */
                                           PHHAL_HW_CARDTYPE_ISO14443A);  /**< [In] Type of card for which the hal should be configured for. */
    CHECK_SUCCESS(status);

    printf("Please insert a UL/ULC card for the examples!\n");
    printf("Press any key to continue...\n\n");
    /* _getch(); */

    /**
     * Activate Layer 3 card
     */
    status = phpalI14443p3a_ActivateCard(&palI14443p3a,  /**< [In] PAL-ISO14443P3A parameter structure */
                                         NULL,           /**< [In] unknown UID */
                                         0x00,           /**< [In] UID length 0 */
                                         aUid,           /**< [Out] complete Uid; uint8_t[4/7/10]. */
                                         &bLength,       /**< [Out] Length of Uid; 4/7/10. */
                                         aSak,           /**< [Out] Select Acknowledge; uint8_t. */
                                         &bMoreCardsAvaliable);  /**< [Out] Whether there are more cards in the field or not; uint8_t. */
    CHECK_SUCCESS(status);

    printf("Perform the examples: \n");

    printf("---------------------------\n");
    printf("MIFARE Ultralight Example\n");
    printf("[1] UL//ULC Example\n");
    printf("[2] ULC Example\n");
    printf("[3] ULC NFC Forum Type 2 Tag Example\n");
    printf("[x] Exit\n");
    printf("----------------\n");
    printf("Choice:");
    iChoice = _getch();
    printf(" %c", iChoice);
    printf("\n\n");
    printf("---------------------------\n");
    printf("\n\n");

    switch (iChoice)
    {
        case '1':
        {
            printf("Ultralight and Ultralight C Examples.\n");
            exampleUL_ULC(&alMful,         /**< MIFARE Ultralight parameter structure. */
                          &palMifare,      /**< PAL-MIFARE parameter structure */
                          &keyStore,       /**< Key store structure */
                          &cryptoEnc,      /**< CryptoSym parameter structure for ENC */
                          &cryptoRng,      /**< CryptoRng parameter structure for Rng */
                          pHal,            /**< Pointer to the parameter structure of the HAL layer. */
                          &palI14443p3a);  /**< PAL-ISO14443P3A parameter structure */
            break;
        }
        case '2':
        {
            printf("Ultralight C Examples.\n");
            exampleULC(&alMful,         /**< MIFARE Ultralight parameter structure. */
                       &palMifare,      /**< PAL-MIFARE parameter structure */
                       &keyStore,       /**< Key store structure */
                       &cryptoEnc,      /**< CryptoSym parameter structure for ENC */
                       &cryptoRng,      /**< CryptoRng parameter structure for Rng */
                       pHal,            /**< Pointer to the parameter structure of the HAL layer. */
                       &palI14443p3a);  /**< PAL-ISO14443P3A parameter structure */
            break;
        }
        case '3':
        {
            printf("Ultralight C NFC Forum Type 2 Tag Examples.\n");
            exampleNFCT2T(&alMful,         /**< MIFARE Ultralight parameter structure. */
                          &palMifare,      /**< PAL-MIFARE parameter structure */
                          &keyStore,       /**< Key store structure */
                          &cryptoEnc,      /**< CryptoSym parameter structure for ENC */
                          &cryptoRng,      /**< CryptoRng parameter structure for Rng */
                          pHal,            /**< Pointer to the parameter structure of the HAL layer. */
                          &palI14443p3a);  /**< PAL-ISO14443P3A parameter structure */
            break;
        }
        default:
        {
            break;
        }
    }

    /**
     * Switch off the field
     */
    status = phhalHw_FieldOff(pHal);  /**< [In] Pointer to the parameter structure of the HAL layer. */
    CHECK_SUCCESS(status);

    /**
     * we are finished, release bal handle
     */
    status = phbalReg_ClosePort(&balReader);  /**< [In] Pointer to the parameter structure of the BAL layer. */
    CHECK_SUCCESS(status);

    printf("-------------------------\n");
    printf("No more tests available.\n");
    printf("Press any key to continue...\n\n");
    _getch();

    /**
     * Returns zero if the procedure was successful.
     */
    return 0;
}
