/*
 * Copyright 2013, 2019, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Example Source for Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

/**
* Header
*/
#include "Example-Rd70x_NonX_Mfc.h"
/**
* Key store
*/
#define KEYCOUNT                 0xFFU /**< number of keys */
#define KEYVERSIONS              0x01U /**< number of key versions */

#define SAM_KEY                  0x00U /**< SAM Key address */
#define KEY_VERSION_SAM          0x00U /**< SAM Key version */
#define MIFARE_CLASSIC_KEY       0x2FU /**< MIFARE classic key address */
#define KEY_VERSION_MFC          0x00U /**< MIFARE classic key version */
#define MIFARE_PLUS_AES_KEY      0x02U /**< MIFARE Plus AES Key address */
#define KEY_VERSION_MFP_AES      0x00U /**< MIFARE Plus AES key version */
#define KEY_POS                  0x00U

#define SAM_AV1                  0x01U  /**< SAM AV1 */
#define SAM_AV2                  0x02U  /**< SAM AV2 */

#define SAM_LC0         0x00U /**< SAM logical channel 0 */
#define SAM_LC1         0x01U /**< SAM logical channel 1 */

#define PLAIN       0x00U    /**< plain mode */
#define MAC         0x01U    /**< CMAC mode */
#define FULL        0x02U    /**< full mode */

#define GLOBAL_BUFFER_SIZE     0x012CU  /**< global size of transmit and recieve buffer */

#define KEY_VERSION_BUFFER_SIZE 0x06 /**< Key version buffer size */

#define RETRYCOUNT 0x05U  /**< retry count */

/**
* Pegoda 2 configurations
*/
#define PCSC_READER_P2_NON_X_NAME   "NXP Pegoda S 0 0"        /**< Pegoda 2: SAM in non X-Mode */

static const uint8_t gaMifareAKey[] =    /**< MIFARE classic key 0xFF */
{ 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF };

int __cdecl main()
{
	phStatus_t status;
	uint8_t aHalBufferReaderTx[GLOBAL_BUFFER_SIZE];  /**< HAL Reader transmit buffer */
	uint8_t aHalBufferReaderRx[GLOBAL_BUFFER_SIZE];  /**< HAL Reader receive buffer */
	uint8_t aAtr[256];                /**< Atr */
	uint8_t aSeed[8];                 /**< Seed */
	uint16_t aVersion[0x1F];          /**< SAM Version */
	uint8_t aHalBufferSam[261];       /**< Hal SAM Buffer array */
	uint8_t bSamHostMode = 0;         /**< SAM host mode */
	uint8_t div = 0;                  /**< diversivication */
	uint8_t authtype = PLAIN;         /**< authentication type: PLAIN, MAC, FULL */

	uint8_t bReaderListPegoda[256];
	uint16_t wCount = 0;
	uint8_t bGenBuff[255];
	uint8_t * pRxPtr;
#ifdef EXAMPLE__LOWLEVEL
	uint16_t wCrc, wParity;
#endif
	uint8_t bValidBits;
    uint8_t bCount = 0;
	phStatus_t statusTmp;

	/**
	* keystore
	*/
	phKeyStore_Sw_KeyEntry_t aKeyEntry[KEYCOUNT];
	phKeyStore_Sw_KeyVersionPair_t aKeyVersion[KEYCOUNT * KEYVERSIONS];
	phKeyStore_Sw_KUCEntry_t aKeyUsage[KEYCOUNT];

	/**
	* Bfl data parameter storage
	*/
	phbalReg_PcscWin_DataParams_t balPcsc;            /**< PCSC (Windows) BAL parameter structure */
	phbalReg_Rd70xUsbWin_DataParams_t balRd70x;            /**< PCSC (Windows) BAL parameter structure */

	phhalHw_Rd70x_DataParams_t halRd70x;              /**< Rd710 HAL parameter structure */
	phpalI14443p3a_Sw_DataParams_t I14443p3a_Sw;      /**< Software PAL-ISO14443P3A parameter structure */
	phpalI14443p4a_Sw_DataParams_t I14443p4a_Sw;      /**< Software PAL-ISO14443P4A parameter structure */
	phpalI14443p4_Sw_DataParams_t I14443p4_Sw;        /**< Software PAL-ISO14443P4 parameter structure */
	phpalMifare_Sw_DataParams_t palMifare_Sw;         /**< Software PAL-MIFARE parameter structure */
	phCryptoSym_Sw_DataParams_t cryptoEnc;            /**< CryptoSym parameter structure for ENC */
	phCryptoSym_Sw_DataParams_t cryptoMac;            /**< CryptoSym parameter structure for MAC */
	phCryptoSym_Sw_DataParams_t cryptoSymRnd;         /**< CryptoSym parameter structure for SymRng */
	phCryptoRng_Sw_DataParams_t cryptoRng;            /**< CryptoRng parameter structure for Rng */
	phKeyStore_Sw_DataParams_t keyStoreSw;            /**< Software KeyStore parameter structure */
	phalMfc_Sw_DataParams_t alMfc;                    /**< MIFARE Classic parameter structure. */
	phhalHw_SamAV2_DataParams_t halSamLC0;            /**< SamAV2 HAL parameter structure LC0 */
	phhalHw_SamAV2_DataParams_t halSamLC1;            /**< SamAV2 HAL parameter structure LC1 */
	phKeyStore_SamAV2_DataParams_t keyStoreSAM;       /**< SAM KeyStore parameter structure */
	void * pHal = NULL;                               /**< Pointer to the parameter structure of the HAL layer. */
	void * pKeyStore;                                 /**< Pointer to a KeyStore parameter structure */
	void * pI14443p3a;                                /**< Pointer to a PAL-ISO14443P3A parameter structure */
	void * pI14443p4a;                                /**< Pointer to a PAL-ISO14443P4A parameter structure */
	void * pI14443p4;                                 /**< Pointer to a PAL-ISO14443P4 parameter structure */
	void * pBal;                                      /**< Pointer to a BAL parameter structure */
	void * pAlMfc;                                    /**< Pointer to a MIFARE Classic parameter structure */
	/* common variables */

	/* variables used by the BAL component */

	/* variables used by the PAL (ISO14443-3) component */
	uint8_t pUidIn[10], bLenUidIn, pUidOut[10], bLenUidOut;
	uint8_t bSak, bMoreCardsAvaliable;

	/* variables used by the PAL (Mifare) component */

	/* variables used by the AL (Mifare) component */
	uint8_t  bBlockNo, bKeyType;
	uint16_t wKeyNumber, wKeyVersion;
	uint8_t pOriginalBlockData[PHAL_MFC_DATA_BLOCK_LENGTH];
	uint8_t pNewBlockData[PHAL_MFC_DATA_BLOCK_LENGTH];
	uint8_t pReadBlockData[PHAL_MFC_DATA_BLOCK_LENGTH];

	printf("\nNxpRdLib ANSI-C Example Program V1.0a\n\n");
	printf("Please ensure that a Pegoda reader is connected and in working condition.\n\n");
	printf("Performing startup...\n\n");

	/* Initialize logging */

	/**
	* init. Sw keystore
	*/
	status = phKeyStore_Sw_Init(&keyStoreSw,   /**< [In] Pointer to this layer's parameter structure. */
		sizeof(phKeyStore_Sw_DataParams_t),  /**< [In] Specifies the size of the data parameter structure */
		aKeyEntry,     /**< [In] Pointer to a storage containing the key entries. */
		KEYCOUNT,      /**< [In] size of pKeyEntries. */
		aKeyVersion,   /**< [In] Pointer to a storage containing the key version pairs. */
		KEYVERSIONS,   /**< [In] amount of key versions available in each key entry. */
		aKeyUsage,     /**< [In] Key usage counter entry storage, size = sizeof(phKeyStore_Sw_KUCEntry_t) * wNumKUCEntries */
		KEYCOUNT);     /**< [In] Number of Key usage counter entries. */
	CHECK_SUCCESS(status);
	pKeyStore = &keyStoreSw;  /**< set the software keystore */

	/**
	* init. crypto
	*/
	status = phCryptoSym_Sw_Init(&cryptoEnc,    /**< [In] Pointer to this layer's parameter structure. */
		sizeof(phCryptoSym_Sw_DataParams_t),   /**< [In] Specifies the size of the data parameter structure */
		pKeyStore);  /**< [In] Pointer to a key store structure (can be null).*/
	CHECK_SUCCESS(status);
	status = phCryptoSym_Sw_Init(&cryptoMac,    /**< [In] Pointer to this layer's parameter structure. */
		sizeof(phCryptoSym_Sw_DataParams_t),   /**< [In] Specifies the size of the data parameter structure */
		pKeyStore);    /**< [In] Pointer to a key store structure (can be null).*/
	CHECK_SUCCESS(status);
	status = phCryptoSym_Sw_Init(&cryptoSymRnd, /**< [In] Pointer to this layer's parameter structure. */
		sizeof(phCryptoSym_Sw_DataParams_t),   /**< [In] Specifies the size of the data parameter structure */
		pKeyStore);    /**< [In] Pointer to a key store structure (can be null).*/
	CHECK_SUCCESS(status);
	status = phCryptoRng_Sw_Init(&cryptoRng,    /**< [In] Pointer to this layer's parameter structure. */
		sizeof(phCryptoRng_Sw_DataParams_t),   /**< [In] Specifies the size of the data parameter structure */
		&cryptoSymRnd);  /**< [In] Pointer to the parameter structure of the symmetric crypto layer. */
	CHECK_SUCCESS(status);

	/**
	* clear seed
	*/
	memset(aSeed, 0x00, 0x08);
	/**
	* seed the random number generator with the given seed
	*/
	status = phCryptoRng_Seed(&cryptoRng,      /**< [In] CryptoSym parameter structure for Rng */
		aSeed,           /**< [In] Seed */
		sizeof(aSeed));  /**< [In] Size of the seed. */
	CHECK_SUCCESS(status);

	printf("----------------\n");
	printf("Connected Pegoda\n");
	printf("----------------\n");
	printf("[2] Pegoda. SAM in non X-mode\n");
	printf("\n\n");

	/**
	* Init USB BAL
	*/
	status = phbalReg_Rd70xUsbWin_Init(&balRd70x,
		sizeof(phbalReg_Rd70xUsbWin_DataParams_t));
	CHECK_SUCCESS(status);

	/**
	* Select Port to be used
	* ReaderList: \\?\usb#vid_0742&pid_ff01#5&329a137b&0&1#{de939260-03ee-11d4-a130-00105accce85};
	*/
	/* Get list of connected pegoda readers */
	status = phbalReg_GetPortList(&balRd70x, sizeof(bReaderListPegoda), bReaderListPegoda, &wCount);
	CHECK_SUCCESS(status);

	/* Connect to the first reader */
	status = phbalReg_SetPort(&balRd70x, bReaderListPegoda);
	CHECK_SUCCESS(status);



	/**
	* Init. PCSC BAL
	*/
	status = phbalReg_PcscWin_Init(&balPcsc,	/**< [In] Pointer to this layer's parameter structure. */
		sizeof(phbalReg_PcscWin_DataParams_t),	/**< [In] Specifies the size of the data parameter structure. */
		aAtr,                                   /**< [In] ATR buffer. */
		sizeof(aAtr));                          /**< [In] ATR buffer size. */
	CHECK_SUCCESS(status);

	pBal = &balPcsc;  /**< set the BAL PCSC */
#if 0
	/**
	* set the BAL Communication Configuration values
	*/
	status = phbalReg_SetConfig(&balPcsc,    /**< [In] Pointer to this layer's parameter structure. */
		PHBAL_REG_PCSCWIN_CONFIG_PROTOCOL,  /**< BAL Communication Configs */
		PHBAL_REG_PCSCWIN_VALUE_PROTOCOL_UNDEFINED); /**< BAL Communication Configuration values */
	CHECK_SUCCESS(status);
	/**
	* set the BAL Communication Configs
	*/
	status = phbalReg_SetConfig(&balPcsc,    /**< [In] Pointer to this layer's parameter structure. */
		PHBAL_REG_PCSCWIN_CONFIG_SHARE,  /**< BAL Communication Configs */
		PHBAL_REG_PCSCWIN_VALUE_SHARE_SHARED);  /**< BAL Communication Configuration values */
	CHECK_SUCCESS(status);
#endif
	/**
	* Select Port to be used (SAM in non X-Mode)
	*/
	status = phbalReg_SetPort(&balPcsc,  /**< [In] Pointer to this layer's parameter structure. */
		(uint8_t*)"Broadcom Corp Contacted SmartCard 0");  /**< [In] Port Name as String. */
	CHECK_SUCCESS(status);

	/**
	* Initialise the Rd70x HAL component:
	*
	*/

	status = phhalHw_Rd70x_Init(&halRd70x,
		sizeof(phhalHw_Rd70x_DataParams_t),
		&balRd70x,
		aHalBufferReaderTx,
		sizeof(aHalBufferReaderTx),
		aHalBufferReaderRx,
		sizeof(aHalBufferReaderRx));
	CHECK_SUCCESS(status);


	pHal = &halRd70x;  /**< Set HAL pointer */


	pBal = &balPcsc;  /**< Set BAL pointer */

	/**
	* init the SAM HAL LC0 component (SAM in non X-Mode)
	*/
	status = phhalHw_SamAV2_Init(&halSamLC0,                /**< SamAV2 HAL parameter structure LC0 */
		sizeof(phhalHw_SamAV2_DataParams_t),   /**< [In] Specifies the size of the data parameter structure */
		&balPcsc,                      /**< PCSC (Windows) BAL parameter structure */
		pHal,                      /**< Pointer to the hal data params of a reader. NULL in case of X-Mode */
		pKeyStore,                 /**< Pointer to KeyStore parameter structure */
		&cryptoEnc,                /**< CryptoSym parameter structure for ENC */
		&cryptoMac,                /**< CryptoSym parameter structure for MAC */
		&cryptoRng,                /**< Pointer to the parameter structure of the CryptoRng layer. */
		PHHAL_HW_SAMAV2_OPMODE_NON_X, /**< Operation mode */
		SAM_LC0,                   /**< Logical channel */
		aHalBufferSam,             /**< Hal SAM Buffer array TX */
		sizeof(aHalBufferSam),     /**< Hal SAM Buffer array length TX */
		aHalBufferSam,             /**< Hal SAM Buffer array RX */
		sizeof(aHalBufferSam));    /**< Hal SAM Buffer array length RX */
	CHECK_SUCCESS(status);

	/**
	* Init hal of sam with HComm AV1 object
	* init the SAM HAL LC1 component
	*/
	status = phhalHw_SamAV2_Init(&halSamLC1,                    /**< SamAV2 HAL parameter structure LC1 */
		sizeof(phhalHw_SamAV2_DataParams_t),   /**< [In] Specifies the size of the data parameter structure */
		&balPcsc,                          /**< PCSC (Windows) BAL parameter structure */
		NULL,                          /**< Pointer to the hal data params of a reader. NULL in case of X-Mode */
		pKeyStore,                     /**< Pointer to KeyStore parameter structure */
		&cryptoEnc,                    /**< CryptoSym parameter structure for ENC */
		&cryptoMac,                    /**< CryptoSym parameter structure for MAC */
		&cryptoRng,                    /**< Pointer to the parameter structure of the CryptoRng layer. */
		PHHAL_HW_SAMAV2_OPMODE_X_RC523,   /**< Operation mode */
		SAM_LC1,                       /**< Logical channel */
		aHalBufferSam,                 /**< Hal SAM Buffer array TX */
		sizeof(aHalBufferSam),         /**< Hal SAM Buffer array length TX */
		aHalBufferSam,                 /**< Hal SAM Buffer array RX */
		sizeof(aHalBufferSam));        /**< Hal SAM Buffer array length RX */
	CHECK_SUCCESS(status);

	/**
	* Init SAM as keyStore object
	*/
	status = phKeyStore_SamAV2_Init(&keyStoreSAM,  /**< [In] Pointer to this layer's parameter structure. */
		sizeof(phKeyStore_SamAV2_DataParams_t),  /**< [In] Specifies the size of the data parameter structure */
		&halSamLC1);  /**< [In] Pointer to the parameter structure of the underlying layer.*/
	CHECK_SUCCESS(status);

	/**
	* Open the PCSC port
	*/
	status = phbalReg_OpenPort(&balPcsc);  /**< [In] Pointer to this layer's parameter structure. */
	CHECK_SUCCESS(status);

	/**
	* Open the BAL Rd70x Port
	*/
	status = phbalReg_OpenPort(&balRd70x);  /**< [In] Pointer to this layer's parameter structure. */
	CHECK_SUCCESS(status);

	/**
	* Detect SAM settings LC0
	*/
	status = phhalHw_SamAV2_DetectMode(&halSamLC0);  /**< [In] Pointer to this layer's parameter structure. */
	CHECK_SUCCESS(status);

	/**
	* Detect SAM settings LC1
	*/
	status = phhalHw_SamAV2_DetectMode(&halSamLC1);  /**< [In] Pointer to this layer's parameter structure. */
	CHECK_SUCCESS(status);

	/**
	* Get SAM hostmode
	*/
	status = phhalHw_GetConfig(&halSamLC0, PHHAL_HW_SAMAV2_CONFIG_HOSTMODE, aVersion);
	CHECK_SUCCESS(status);

	if (aVersion[0] == SAM_AV1)
	{
		/**
		* AV1:
		* Load SAM DES key at position 0 in KeyStore
		* set the host mode
		*/
		bSamHostMode = SAM_AV1;
		/**
		* format the key entry
		*/
		status = phKeyStore_FormatKeyEntry(pKeyStore,  /**< [In] Pointer to the parameter structure of the KeyStore component. */
			SAM_KEY,       /**< [In] KeyEntry number to be Formatted. */
			PH_KEYSTORE_KEY_TYPE_DES);  /**< [In] New Key type of the KeyEntry (predefined type of KeyType).*/
		CHECK_SUCCESS(status);
	}
	else
	{
		/**
		* AV2:
		* Load SAM AES128 key at position 0 in KeyStore
		* set the host mode
		*/
		bSamHostMode = SAM_AV2;
		/**
		* format the key entry
		*/
		status = phKeyStore_FormatKeyEntry(pKeyStore,  /**< [In] Pointer to the parameter structure of the KeyStore component. */
			SAM_KEY,    /**< [In] KeyEntry number to be Formatted. */
			PH_KEYSTORE_KEY_TYPE_AES128);  /**< [In] New Key type of the KeyEntry (predefined type of KeyType).*/
		CHECK_SUCCESS(status);
	}

	pKeyStore = &keyStoreSAM;  /**< set the sam keystore */
	pHal = &halSamLC0; /** Change the reader to SAM */

	/**
	* init the 14443-3A component
	*/
	status = phpalI14443p3a_Sw_Init(&I14443p3a_Sw,  /**< [In] PAL-ISO14443P3A parameter structure */
		sizeof(phpalI14443p3a_Sw_DataParams_t),  /**< [In] Specifies the size of the data parameter structure */
		pHal);          /**< [In] Pointer to the parameter structure of the underlying layer.*/
	CHECK_SUCCESS(status);

	/**
	* init the 14443-4A component
	*/
	status = phpalI14443p4a_Sw_Init(&I14443p4a_Sw,  /**< [In] PAL-ISO14443P4A parameter structure */
		sizeof(phpalI14443p4a_Sw_DataParams_t),  /**< [In] Specifies the size of the data parameter structure */
		pHal);          /**< [In] Pointer to the parameter structure of the underlying layer.*/
	CHECK_SUCCESS(status);

	/**
	* init the 14443-4 component
	*/
	status = phpalI14443p4_Sw_Init(&I14443p4_Sw,  /**< [In] PAL-ISO14443P4 parameter structure */
		sizeof(phpalI14443p4_Sw_DataParams_t),  /**< [In] Specifies the size of the data parameter structure */
		pHal);         /**< [In] Pointer to the parameter structure of the underlying layer.*/
	CHECK_SUCCESS(status);

	/**
	* init. mifare pal
	*/
	status = phpalMifare_Sw_Init(&palMifare_Sw,   /**< [In] PAL-MIFARE parameter structure */
		sizeof(phpalMifare_Sw_DataParams_t),  /**< [In] Specifies the size of the data parameter structure */
		pHal,            /**< [In] Pointer to the parameter structure of the underlying layer.*/
		&I14443p4_Sw);   /**< [In] Pointer to the parameter structure of the underlying ISO14443-4 layer. */
	CHECK_SUCCESS(status);

	status = phalMfc_Sw_Init(&alMfc,        /**< [In] MIFARE Classic parameter structure. */
		sizeof(phalMfc_Sw_DataParams_t), /**< [In] Specifies the size of the data parameter structure */
		&palMifare_Sw, /**< [In] Pointer to a palMifare component context. */
		NULL);    /**< [In] Pointer to the parameter structure of the KeyStore component. */
	CHECK_SUCCESS(status);

	/**
	* Set the generics
	*/
	pI14443p3a = &I14443p3a_Sw;
	pI14443p4a = &I14443p4a_Sw;
	pI14443p4 = &I14443p4_Sw;
	pAlMfc = &alMfc;

	/**
	* Set ISO 14443p4 configuration parameter
	*/
	status = phpalI14443p4_SetConfig(pI14443p4,  /**< [In] Pointer to this layer's parameter structure. */
		RETRYCOUNT, /**< [In] Configuration Identifier */
		0);         /**< [In] Configuration Value */
	CHECK_SUCCESS(status);


	/**
	* Authenticate with SAM on LC1 which is the Management channel
	*/
	/**
	* AV1 or AV2?
	*/
	if (bSamHostMode == SAM_AV1)
	{
		/**
		* Plain Mode
		* Perform HOST AUTHENTICATION
		*/
		status = phhalHw_SamAV2_Cmd_SAM_AuthenticateHost(&halSamLC1,  /**< Pointer to this layer's parameter structure. */
			0x00,       /**< AuthType: Default */
			0,          /**< Host Key Number */
			0x00,       /**< Host Key Version */
			0x00,       /**< Sam Key Number */
			0x00,       /**< Sam Key Version */
			&div,       /**< No Diversification input */
			0);         /**< Diversification length */
		CHECK_SUCCESS(status);
	}
	else
	{
		/**
		* Perform HOST AUTHENTICATION
		*/
		status = phhalHw_SamAV2_Cmd_SAM_AuthenticateHost(&halSamLC1,  /**< Pointer to this layer's parameter structure. */
			authtype,   /**< AuthType: Default */
			0,          /**< Host Key Number */
			0x00,       /**< Host Key Version */
			0x00,       /**< Sam Key Number */
			0x00,       /**< Sam Key Version */
			&div,       /**< No Diversification input */
			0);         /**< Diversification length */
		CHECK_SUCCESS(status);
	}


	printf("----------------\n");
	printf("Card Example:\n");
	printf("----------------\n");
	printf("\n\n");

	/**
	* Load MIFARE key at position 1 in KeyStore

	status = phKeyStore_FormatKeyEntry(pKeyStore,
	MIFARE_CLASSIC_KEY,
	PH_KEYSTORE_KEY_TYPE_MIFARE);
	CHECK_SUCCESS(status);
	*/
	status = phKeyStore_SetKeyAtPos(pKeyStore,
		MIFARE_CLASSIC_KEY,
		KEY_POS,
		PH_KEYSTORE_KEY_TYPE_MIFARE,
		(uint8_t*)gaMifareAKey,
		KEY_VERSION_MFC);
	CHECK_SUCCESS(status);

#if 0
	status = phKeyStore_GetKeyEntry(pKeyStore,
		MIFARE_CLASSIC_KEY,
		KEY_VERSION_BUFFER_SIZE,
		wKeyVersion,
		wKeyVersionLength,
		pKeyType);
#endif

	/**
	* Perform RF-Reset
	*/
	status = phhalHw_FieldReset(pHal);    /**< [In] Pointer to the parameter structure of the underlying HAL layer. */
	CHECK_SUCCESS(status);

	/* ******************************************************************************* */
	/* CARD COMMUNICATION */
	/* ******************************************************************************* */

	/* Activate card */
	bLenUidIn = 0;
	status = phpalI14443p3a_ActivateCard(pI14443p3a, pUidIn, bLenUidIn,
		pUidOut, &bLenUidOut,
		&bSak, &bMoreCardsAvaliable);
	printf("phpalI14443p3a_ActivateCard:  %04X\n", status);

	if(PH_ERR_SUCCESS == status)
	{
		status = phpalI14443p3a_GetSerialNo(pI14443p3a, pUidOut, &bLenUidOut);
		printf("phpalI14443p3a_GetSerialNo:   %04X\nUID: ", status);
		if(PH_ERR_SUCCESS == status)
			printBufferHex(pUidOut, bLenUidOut);

		/* Authenticate */
		bBlockNo = 0xAF;
		bKeyType = PHPAL_MIFARE_KEYA;
		wKeyNumber = 0x2F;
		wKeyVersion = 0;
		status = phalMfc_Authenticate(pAlMfc, bBlockNo, bKeyType, wKeyNumber, wKeyVersion,
			pUidOut, bLenUidOut);
		printf("phalMfc_Authenticate:         %04X\n", status);

		if(PH_ERR_SUCCESS == status)
		{
			/* Read one MIFARE(R) Block, write one Block and restore the written Block. */
			memcpy(pNewBlockData, "\x01\x02\x03\x04\x05\x06\x07\x08\x09\x0A\x0B\x0C\x0D\x0E\x0F\x10",
				sizeof(pNewBlockData));

			/* Read one block */
			bBlockNo = 0xAD;
			status = phalMfc_Read(pAlMfc, bBlockNo, pOriginalBlockData);
			printf("phalMfc_Read:                 %04X\n", status);
			if(PH_ERR_SUCCESS == status)
			{
				printf("Block %d Data: ", bBlockNo);
				printBufferHex(pOriginalBlockData, PHAL_MFC_DATA_BLOCK_LENGTH);
			}

			/* Write one block */
			status = phalMfc_Write(pAlMfc, bBlockNo, pNewBlockData);
			printf("phalMfc_Write:                %04X\n", status);

			status = phalMfc_Read(pAlMfc, bBlockNo, pReadBlockData);
			printf("phalMfc_Read:                 %04X\n", status);
			if(PH_ERR_SUCCESS == status)
			{
				printf("Block %d Data: ", bBlockNo);
				printBufferHex(pReadBlockData, PHAL_MFC_DATA_BLOCK_LENGTH);
			}

			/* Restore old Block Data */
			status = phalMfc_Write(pAlMfc, bBlockNo, pOriginalBlockData);
			printf("phalMfc_Write:                %04X\n", status);

			status = phalMfc_Read(pAlMfc, bBlockNo, pReadBlockData);
			printf("phalMfc_Read:                 %04X\n", status);
			if(PH_ERR_SUCCESS == status)
			{
				printf("Block %d Data: ", bBlockNo);
				printBufferHex(pReadBlockData, PHAL_MFC_DATA_BLOCK_LENGTH);
			}

			/* Change key on block 0x2F */
			/* First send the write command to Classic encrypted */
			bGenBuff[0] = 0xA0;
			bGenBuff[1] = 0xAF;
			wCount = 0x02;

			status = phhalHw_SamAV2_Cmd_SAM_EncipherData(&halSamLC0,
				PH_EXCHANGE_DEFAULT,
				bGenBuff,
				2,
				0,
				&pRxPtr,
				&wCount);

			CHECK_SUCCESS(status);

			memcpy(bGenBuff, pRxPtr, wCount);


			/* Raw exchange without enciphering and without deciphering */
			status = phpalMifare_ExchangeRaw(&palMifare_Sw,
				PHHAL_HW_SAMAV2_EXCHANGE_NO_ENCIPHERING_BIT | PHHAL_HW_SAMAV2_EXCHANGE_NO_DECIPHERING_BIT,
				bGenBuff,
				wCount,
				0,
				&pRxPtr,
				&wCount,
				&bValidBits);
			CHECK_SUCCESS(status);

			/* Decrypt the response from the card */
			/* Perform actual deciphering */
			memcpy(bGenBuff, pRxPtr, wCount);

			statusTmp = phhalHw_SamAV2_Cmd_SAM_DecipherData(
				&halSamLC0,
				PH_EXCHANGE_DEFAULT,
				bGenBuff,
				(uint8_t)wCount,
				NULL,
				&pRxPtr,
				&wCount);

			/* Bail out on Error */
			if ((statusTmp & PH_ERR_MASK) != PH_ERR_SUCCESS)
			{
				return statusTmp;
			}

			if (*pRxPtr != 0x0A)
			{
				return *pRxPtr;
			}

			/* Now send the changekeymifare command to SAM and receive the encrypted
			cryptogram */
			bGenBuff[0] = 0xFF;
			bGenBuff[1] = 0x07;
			bGenBuff[2] = 0x80;
			bGenBuff[3] = 0x69;
			PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SamAV2_Cmd_SAM_ChangeKeyMifare(
				&halSamLC0,
				PH_EXCHANGE_DEFAULT,
				(uint8_t)wKeyNumber,
				0x01,
				0x01,
				bGenBuff,
				NULL,
				0,
				bGenBuff,
				&bCount));

			/* Again exchange raw */
			status = phpalMifare_ExchangeRaw(&palMifare_Sw,
				PHHAL_HW_SAMAV2_EXCHANGE_NO_ENCIPHERING_BIT | PHHAL_HW_SAMAV2_EXCHANGE_NO_DECIPHERING_BIT,
				bGenBuff,
				wCount,
				0,
				&pRxPtr,
				&wCount,
				&bValidBits);
			CHECK_SUCCESS(status);

			/* Decrypt the response */
			memcpy(bGenBuff, pRxPtr, wCount);

			statusTmp = phhalHw_SamAV2_Cmd_SAM_DecipherData(
				&halSamLC0,
				PH_EXCHANGE_DEFAULT,
				bGenBuff,
				(uint8_t)wCount,
				NULL,
				&pRxPtr,
				&wCount);

			/* Bail out on Error */
			if ((statusTmp & PH_ERR_MASK) != PH_ERR_SUCCESS)
			{
				return statusTmp;
			}

			if (*pRxPtr != 0x0A)
			{
				return *pRxPtr;
			}

#ifdef EXAMPLE__LOWLEVEL
			/* Get amount of complete bytes */
			halSamLC0.wCfgShadow[PHHAL_HW_CONFIG_TXLASTBITS] = (uint8_t)(wCount % 9);
			if (halSamLC0.wCfgShadow[PHHAL_HW_CONFIG_TXLASTBITS] != 0x00)
			{
				--halSamLC0.wCfgShadow[PHHAL_HW_CONFIG_TXLASTBITS];
			}


			/* Retrieve Parity-setting */
			PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_GetConfig(halSamLC0.pReaderHalDataParams, PHHAL_HW_CONFIG_PARITY, &wParity));

			/* Disable Parity */
			if (wParity != PH_OFF)
			{
				PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(halSamLC0.pReaderHalDataParams, PHHAL_HW_CONFIG_PARITY, PH_OFF));
			}

			/* Disable TxCrc */
			PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(halSamLC0.pReaderHalDataParams, PHHAL_HW_CONFIG_TXCRC, PH_OFF));

			/* Retrieve RxCrc-setting */
			PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_GetConfig(halSamLC0.pReaderHalDataParams, PHHAL_HW_CONFIG_RXCRC, &wCrc));

			/* Disable RxCrc */
			if (wCrc != PH_OFF)
			{
				PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(halSamLC0.pReaderHalDataParams, PHHAL_HW_CONFIG_RXCRC, PH_OFF));
			}

			/* Set TxLastBits */
			PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(halSamLC0.pReaderHalDataParams,
				PHHAL_HW_CONFIG_TXLASTBITS,
				(uint16_t)halSamLC0.wCfgShadow[PHHAL_HW_CONFIG_TXLASTBITS]));


			status = phhalHw_Exchange(halSamLC0.pReaderHalDataParams,
				PH_EXCHANGE_DEFAULT,
				bGenBuff,
				5,
				&pRxPtr,
				&wCount);
			/* Restore Parity-setting again since many PAL layers expect it */

			PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(halSamLC0.pReaderHalDataParams, PHHAL_HW_CONFIG_PARITY, PH_ON));

			/* Restore RxCRC-setting again */

			PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(halSamLC0.pReaderHalDataParams, PHHAL_HW_CONFIG_RXCRC, PH_ON));

			/* Retrieve RxLastBits */
			PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_GetConfig(halSamLC0.pReaderHalDataParams, PHHAL_HW_CONFIG_RXLASTBITS, &bValidBits));
			halSamLC0.wAdditionalInfo = bValidBits;

			/* Clear TxLastBits */
			halSamLC0.wCfgShadow[PHHAL_HW_CONFIG_TXLASTBITS] = 0;

			/* status = phpalMifare_ExchangeRaw(
			&palMifare_Sw,
			PH_EXCHANGE_DEFAULT,
			bGenBuff,
			wCount,
			0,
			&pRxPtr,
			&wCount,
			&bLastBits);
			*/
			/* status check */
			if ((status & PH_ERR_MASK) != PH_ERR_SUCCESS_INCOMPLETE_BYTE)
			{
				PH_CHECK_SUCCESS(status);
			}

			/* Now decrypt the received data from the card */
			/* Perform actual deciphering */
			memcpy(bGenBuff, pRxPtr, wCount);

			statusTmp = phhalHw_SamAV2_Cmd_SAM_DecipherData(
				&halSamLC0,
				PH_EXCHANGE_DEFAULT,
				bGenBuff,
				(uint8_t)wCount,
				NULL,
				&pRxPtr,
				&wCount);

			/* Bail out on Error */
			if ((statusTmp & PH_ERR_MASK) != PH_ERR_SUCCESS)
			{
				/* In NonX-Mode, reset RxStartPos */
				if (halSamLC0.bOpMode == PHHAL_HW_SAMAV2_OPMODE_NON_X)
				{
					halSamLC0.wRxBufStartPos = 0;
					halSamLC0.wRxBufLen = 0;
				}
				return statusTmp;
			}

			if (*pRxPtr != 0x0A)
			{
				return *pRxPtr;
			}

			/* Now send the changekeymifare command to SAM and receive the encrypted
			cryptogram */
			bGenBuff[0] = 0xFF;
			bGenBuff[1] = 0x07;
			bGenBuff[2] = 0x80;
			bGenBuff[3] = 0x69;
			PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SamAV2_Cmd_SAM_ChangeKeyMifare(
				&halSamLC0,
				PH_EXCHANGE_DEFAULT,
				wKeyNumber,
				0x01,
				0x01,
				bGenBuff,
				NULL,
				0,
				bGenBuff,
				&wCount));

			/* Now send this data again to the card. Disable Tx, Rx CRC and parity */

			/* Get amount of complete bytes */
			halSamLC0.wCfgShadow[PHHAL_HW_CONFIG_TXLASTBITS] = (uint8_t)(wCount % 9);
			if (halSamLC0.wCfgShadow[PHHAL_HW_CONFIG_TXLASTBITS] != 0x00)
			{
				--halSamLC0.wCfgShadow[PHHAL_HW_CONFIG_TXLASTBITS];
			}


			/* Retrieve Parity-setting */
			PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_GetConfig(halSamLC0.pReaderHalDataParams,
				PHHAL_HW_CONFIG_PARITY,
				&wParity));

			/* Disable Parity */
			if (wParity != PH_OFF)
			{
				PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(halSamLC0.pReaderHalDataParams,
					PHHAL_HW_CONFIG_PARITY,
					PH_OFF));
			}

			/* Disable TxCrc */
			PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(halSamLC0.pReaderHalDataParams,
				PHHAL_HW_CONFIG_TXCRC, PH_OFF));

			/* Retrieve RxCrc-setting */
			PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_GetConfig(halSamLC0.pReaderHalDataParams,
				PHHAL_HW_CONFIG_RXCRC, &wCrc));

			/* Disable RxCrc */
			if (wCrc != PH_OFF)
			{
				PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(halSamLC0.pReaderHalDataParams,
					PHHAL_HW_CONFIG_RXCRC, PH_OFF));
			}

			/* Set TxLastBits */
			PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(halSamLC0.pReaderHalDataParams,
				PHHAL_HW_CONFIG_TXLASTBITS,
				(uint16_t)halSamLC0.wCfgShadow[PHHAL_HW_CONFIG_TXLASTBITS]));

			status = phhalHw_Exchange(halSamLC0.pReaderHalDataParams,
				PH_EXCHANGE_DEFAULT,
				bGenBuff,
				wCount,
				&pRxPtr,
				&wCount);
			/* Restore Parity-setting again since many PAL layers expect it */

			PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(halSamLC0.pReaderHalDataParams, PHHAL_HW_CONFIG_PARITY, PH_ON));

			/* Restore RxCRC-setting again */

			PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SetConfig(halSamLC0.pReaderHalDataParams, PHHAL_HW_CONFIG_RXCRC, PH_ON));

			/* Retrieve RxLastBits */
			PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_GetConfig(halSamLC0.pReaderHalDataParams, PHHAL_HW_CONFIG_RXLASTBITS, &bValidBits));
			halSamLC0.wAdditionalInfo = bValidBits;

			/* Clear TxLastBits */
			halSamLC0.wCfgShadow[PHHAL_HW_CONFIG_TXLASTBITS] = 0;

			if ((status & PH_ERR_MASK) != PH_ERR_SUCCESS_INCOMPLETE_BYTE)
			{
				PH_CHECK_SUCCESS(status);
			}

			/* Now decrypt the received data from the card */
			/* Perform actual deciphering */
			memcpy(bGenBuff, pRxPtr, wCount);

			/* If this is successful, decrypt the card response and ensure that it
			is correct */
			statusTmp = phhalHw_SamAV2_Cmd_SAM_DecipherData(
				&halSamLC0,
				PH_EXCHANGE_DEFAULT,
				bGenBuff,
				(uint8_t)wCount,
				NULL,
				&pRxPtr,
				&wCount);

			/* Bail out on Error */
			if ((statusTmp & PH_ERR_MASK) != PH_ERR_SUCCESS)
			{
				/* In NonX-Mode, reset RxStartPos */
				if (halSamLC0.bOpMode == PHHAL_HW_SAMAV2_OPMODE_NON_X)
				{
					halSamLC0.wRxBufStartPos = 0;
					halSamLC0.wRxBufLen = 0;
				}
				return statusTmp;
			}

			if (*pRxPtr != 0x0A)
			{
				return *pRxPtr;
			}

		}
#endif
		}
	}

	status = phbalReg_ClosePort(&balPcsc);
	status = phbalReg_ClosePort(&balRd70x);
	fprintf(stdout, "\n press any key to exit\n");
	_getch();
	return 0;
}

/**
* Helper functions
*/
void printBufferHex(uint8_t * pBuffer, uint8_t bLength)
{
	uint8_t bIndex;

	for (bIndex = 0; bIndex < bLength; ++bIndex)
	{
		printf("%02X ", pBuffer[bIndex]);
	}
	printf("\n");
}