/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Example application for MIFARE DUOX.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7230 $
 * $Date: 2025-03-14 10:27:23 +0530 (Fri, 14 Mar 2025) $
 */

#ifndef EXAMPLE_MIFAREDUOX_H
#define EXAMPLE_MIFAREDUOX_H

/* Reader Library Headers. */
#include <phbalReg.h>

#include <phhalHw.h>
#include <phhalHw_DUT_Cmd.h>
#include <phhalHw_Rd710_Cmd.h>

#include <phCidManager.h>
#include <phpalI14443p3a.h>
#include <phpalI14443p4a.h>
#include <phpalI14443p4.h>
#include <phpalMifare.h>

#include <phKeyStore.h>
#include <phCryptoRng.h>
#include <phCryptoSym.h>
#include <phCryptoASym.h>
#include <phTMIUtils.h>

#include <phalMfDuoX.h>
#include <phalVca.h>

#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */

/* Standard Header. */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <conio.h>

/* mBedTLS Crypto Library Includes */
#include <mbedtls/pk.h>
#include <mBedTLS/error.h>
#include <mbedtls/x509_crt.h>

/* Local Header Files includes */
#include "Example-MfDUOX_Keys.h"
#include "Example-MfDUOX_Cert.h"

#undef NXPBUILD__PH_LOG                                 /* Disabling Log. To enable comment this line. */

#define PCSC_READER_NO_SAM_NAME "NXP Pegoda N CL 0 0"   /* Pegoda 2 reader name with no SAM support. */

/* KeyStore Entries */
#define NUM_KEY_ENTRY                               20
#define NUM_KEY_VERSION_PAIR                        2
#define NUM_KUC_ENTRY                               255

/* MIFARE DUOX PICC Key Numbers */
#define PICC_MASTER_KEY                             0   /* PICC Master Key */
#define APP_MASTER_KEY                              0   /* Application Master Key */
#define APP_KEY_1                                   1   /* Application Key 1 */
#define APP_KEY_2                                   2   /* Application Key 2 */
#define APP_KEY_3                                   3   /* Application Key 3 */
#define VC_CONFIGURATION_KEY                    0x20    /* VC Configuration Key */
#define VC_PROXIMITY_KEY                        0x21    /* VC Proximity Key */

#define ECC_EV_CHARGING_KEY                         0   /* ECC based Key number for VDE ECDSA Sign */
#define ECC_AUTHENTICATION_KEY                      0   /* ECC based Key number for ISO General Authentication */
#define ECC_ORIGINALITY_CHECK_KEY                   1   /* ECC based Key number for ISO Internal Authentication */
#define ECC_TSIG_KEY                                2   /* ECC based Key number for Transaction Signature Operation */
#define ECC_SDM_KEY                                 3   /* ECC based Key number for Secure Dynamic Operation */
#define ECC_ORIGINALITY_SIG_KEY                     4   /* ECC based Key number for Originality Check Operation */

/* MIFARE DUOX Application Type */
#define APPLICATION_TYPE_NORMAL                     1
#define APPLICATION_TYPE_ECC                        2
#define APPLICATION_TYPE_EV_CHARGING                3

/* MIFARE DUOX PICC File Numbers */
#define STANDARD_DATA_FILE_ORIGINALITY_CHECK        1   /* File for Originality Signature verification. */
#define STANDARD_DATA_FILE_ECC                      0   /* File for storing the Device certificate that wil be used for authentication. */
#define STANDARD_DATA_FILE_EVC                      1   /* File for EV Charging Feature's Data operation. */
#define STANDARD_DATA_FILE                          2   /* File for Data Operation post Data Operation with Plain as File options. */
#define VALUE_FILE                                  3   /* File for Value Operation. */
#define RECORD_FILE                                 4   /* File for Record Operation. */
#define TRANSACTION_MAC_FILE                        5   /* File for Transaction MAC Operation. */

/* Flags for example application */
#define DO_NOT_DISPLAY_COMMAND_INFO             PH_OFF  /* Do not Displays the printf that is implemented in the function(s) */
#define DISPLAY_COMMAND_INFO                    PH_ON   /* Displays the printf that is implemented in the function(s) */
#define SELECT_PICC                             0       /* Option to indicate the selection is performed for PICC. */
#define SELECT_APPLICATION                      1       /* Option to indicate the selection is performed for Application. */
#define SELECT_FILE                             2       /* Option to indicate the selection is performed for File. */
#define DO_NOT_ACTIVATE_TAG                     PH_OFF  /* Option to indicate Tag should not be activated. */
#define ACTIVATE_TAG                            PH_ON   /* Option to indicate Tag should be activated. */

/* Flags for Length */
#define REMOVE_LENGTH_CERTIFICATE               3

#define PRINT_DATA(Data, DataLen)               PrintData((Data), (DataLen), PH_OFF)
#define PRINT_DATA_NEWLINE(Data, DataLen)       PrintData((Data), (DataLen), PH_ON)

#define CHECK_SUCCESS(Status)                                                                   \
if ((Status) != PH_ERR_SUCCESS)                                                                 \
{                                                                                               \
    printf("\n\n");                                                                             \
    printf("An error occurred: (0x%04X)\n", (Status));                                          \
    printf("Pressing any key will exit the demo.\n");                                           \
    _getch();                                                                                   \
    return 0;                                                                                   \
}

#define CHECK_SUCCESS_BREAK(Status)                                                             \
    if ( (Status) != PH_ERR_SUCCESS )                                                           \
    {                                                                                           \
        printf ( "\n\n" );                                                                      \
        printf ( "An error occurred: (0x%04X)\n", (Status) );                                   \
        ErrorMessage ( Status );                                                                \
                                                                                                \
        break;                                                                                  \
    }

#define CHECK_SUCCESS_MBEDTLS(Status)                                                           \
    if ( (Status) < 0 )                                                                         \
    {                                                                                           \
        printf ( "\n\nAn error occurred in mBedTLS Library while parsing the certificate\n" );  \
        printf ( "Error Code   : -0x%04x\n", (unsigned) -(Status) );                            \
        printf ( "Error Message: (%s)\n", aErrorMsg );                                          \
                                                                                                \
        break;                                                                                  \
    }

#define EXIT_IF_REQUESTED(Option) if ( ( (Option) == 'x' ) || ( (Option) == 'X' ) )  exit ( 0 );

#define UNUSED_VARIABLE(x)  {for( ( x ) = ( x ) ; ( x ) != ( x ) ; );}
#define UNUSED_ARRAY(x)  {for( ( (x)[0] ) = ( (x)[0] ) ; ( (x)[0] ) != ( (x)[0] ) ; );}

#define AUTH_STATE(AuthState)     ((AuthState == PHAL_MFDUOX_EV2_AUTHENTICATED) ?               \
    "Authentication using Application Master key (0)     " :                                    \
    "ECC Mutual Authentication (ISOGeneralAuthenticate)  ")

#define AUTH_STATE_MSG(AuthState, Msg)     ((AuthState == PHAL_MFDUOX_EV2_AUTHENTICATED) ? (Msg) \
    : "ECC Mutual Authentication (ISOGeneralAuthenticate)  ")

#define AUTH_KEY_TYPE(IsAES256)     ((IsAES256) ? "AES 256-Bit" : "AES 128-Bit")

#define COMM_OPTION(CommOption)                                                                 \
    (((CommOption) == PHAL_MFDUOX_COMMUNICATION_FULL) ? "Full" :                                \
     ((CommOption) == PHAL_MFDUOX_COMMUNICATION_MAC)  ? "Mac"  :                                \
     "Plain")

#define FILE_OPTION(FileOption)                                                                 \
    (((FileOption) == PHAL_MFDUOX_FILE_OPTION_FULL) ? "Full" :                                  \
     ((FileOption) == PHAL_MFDUOX_FILE_OPTION_MAC)  ? "Mac"  :                                  \
     "Plain")


enum Reader { NONE = 0, RD710, RD730, SIMULATOR };
enum Demo
{
    ISO_GENERAL_AUTHENTICATION = 1, ISO_GENERAL_AUTHENTICATION_FINAL, AES128_AUTHENTICATION,
    AES256_AUTHENTICATION, STANDARD_FILE_OPERATION, VALUE_FILE_OPERATION, RECORD_FILE_OPERATION,
    TRANSACTION_MAC, TRANSACTION_SIGNATURE, SECURE_DYNAMIC_MESSAGING_MAC, SECURE_DYNAMIC_MESSAGING_SIG,
    ORIGINALITY_CHECK, GET_VERSION, PROXIMITY_CHECK, EV_CHARGING
};

/* Context for RdLib components. -------------------------------------------------------------------------------*/
void* pHal_Generic;                                 /* Pointer for Hal component to be assigned. */

phKeyStore_Sw_DataParams_t stKeyStore;
phTMIUtils_t stTmiUtils;                            /* TMIUtils parameter structure. */

phpalI14443p3a_Sw_DataParams_t stI14443p3a;         /* PAL-ISO14443P3A parameter structure */
phpalI14443p4a_Sw_DataParams_t stI14443p4a;         /* PAL-ISO14443P4A parameter structure */
phpalI14443p4_Sw_DataParams_t stI14443p4;           /* ISO14443-4 parameter structure */
phpalMifare_Sw_DataParams_t stMifare;               /* PAL-MIFARE parameter structure */

phalMfDuoX_Sw_DataParams_t stMfDuoX;                /* MIFAER DUOX parameter structure */
phalVca_Sw_DataParams_t stVca;                      /* VCA / PC parameter structure */

uint16_t wIsSimulator = PH_OFF;
uint16_t wIsAES256KeyType = PH_OFF;
uint8_t bAuth_State = PHAL_MFDUOX_NOT_AUTHENTICATED;

uint8_t bSak;                                       /* Select Acknowledge data */

uint8_t aPcdCap2In[6];                              /* PCDCapsIn[0], 0x00: EV0 Secure messaging, 0x01: EV1 secure messaging. */
uint8_t aPcdCap2Out[6];                             /* Pointer to PCDCaps sent from the card (6 bytes). */
uint8_t aPdCap2[6];                                 /* Pointer to PDCaps. (6 bytes) */

uint8_t aAppID_Master[3] = { 0x00, 0x00, 0x00 };    /* Application Identifier for Master Application */
uint8_t aAppID_Normal[3] = { 0x01, 0x00, 0x00 };    /* Application Identifier for normal application */
uint8_t aAppID_ECC[3] = { 0x02, 0x00, 0x00 };       /* Application Identifier for ECC Application */
uint8_t aAppID_EVC[3] = { 0x03, 0x00, 0x00 };       /* Application Identifier for EV Charging Application */
uint8_t aISODfName_EccApp[16] = { 0xA0, 0x00, 0x00, 0x03, 0x96, 0x56, 0x43, 0x41,
                                  0x03, 0xF1, 0x23, 0x40, 0x00, 0x00, 0x00, 0x00 };
uint8_t aISODfName_EvcApp[16] = { 0xA0, 0x00, 0x00, 0x08, 0x45, 0x00, 0x00, 0x00,
                                  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01 };

uint8_t aUid[10];                                   /* Complete UID, 4, 7, 10 byte . */
uint8_t bUid_Len = 0;                               /* Length of the UID received. */

uint8_t * pAID = aAppID_Normal;
uint8_t aData[16] = { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
                      0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10 };

uint8_t aSDMInfo[] =
{
    0xC1,               /* VCUID, ReadCtr Enabled. Encoding Mode as ASCII */
    0xFE, 0xE0,         /* SDMFileRead2 = 0x0F, SDMCtrRet = 0x0E, SDMMetaRead = 0xE0, SDMFileRead = 0x00 */
    0x00, 0x00, 0x00,   /* VCUID / PICC Data Offset */
    0x1A, 0x00, 0x00,   /* SDM Read Counter Offset */
    0x00, 0x00, 0x00,   /* SDM MAC Input Offset */
    0x40, 0x00, 0x00,   /* SDM ENC Offset */
    0x20, 0x00, 0x00,   /* SDM ENC Length */
    0x80, 0x00, 0x00    /* SDM MAC Offset */
};

phStatus_t UpdateKeystore();
void ErrorMessage(phStatus_t wStatus);
void ToBinary(uint8_t *pData, uint16_t *pDataLen);
void PrintData(unsigned char* pBuffer, unsigned int dwLength, uint16_t wAddNewLine);

phStatus_t ActivateCard();
phStatus_t Prepare_ISOGeneralAuthenticate(uint8_t bAuthType, uint8_t bPICC_KeyNo, uint16_t wDisplayInfo);
phStatus_t ExtractPublicKey(uint8_t * pCertificate, uint16_t wCertLen, uint8_t * pPubKey, uint16_t * pPubKeyLen);
phStatus_t Authenticate(uint8_t bAuthState, uint8_t bPICC_KeyNo);
phStatus_t CreateSelectApplication(uint8_t bAppType, uint16_t wDisplayInfo);
phStatus_t DeleteFile(uint8_t bFileID);
phStatus_t CreateStandardFile(uint8_t bFileNo, uint16_t wIsISOFileID_Present, uint8_t bFileOption,
    uint8_t *pAccessRights, uint16_t wDisplayInfo);
phStatus_t CreateValueFile(uint8_t bFileNo, uint8_t bFileOption, uint8_t *pAccessRights);
phStatus_t CreateRecordFile(uint8_t bFileNo, uint8_t bFileOption, uint8_t *pAccessRights);
phStatus_t CreateTransactionMACFile(uint8_t bTMKeyOption);
phStatus_t Write(uint8_t bFileNo, uint8_t bCommOption, uint8_t bChaining, uint32_t dwOffset,
    uint8_t * pData, uint32_t dwDataLen, uint16_t wDisplayInfo);
phStatus_t ReadData(uint8_t bFileNo, uint8_t bCommOption, uint8_t bChaining, uint32_t dwOffset,
    uint32_t dwDataLen, uint8_t ** ppResponse, uint16_t * pRespLen, uint16_t wDisplayInfo);
phStatus_t ChangeFileSettings(uint8_t bFileOption, uint8_t *pAccessRights);
phStatus_t EnableVCProximityKey();





phStatus_t Demo_ECCAuthentication(uint8_t bAuthType);
phStatus_t Demo_AESAuthentication(uint16_t wIsKeyTypeAES256);
phStatus_t Demo_StandardFileOperation();
phStatus_t Demo_ValueFileOperation();
phStatus_t Demo_RecordFileOperation();
phStatus_t Demo_TrasactionMAC();
phStatus_t Demo_TrasactionSIG();
phStatus_t Demo_SecureDynamicMessaging_MAC();
phStatus_t Demo_SecureDynamicMessaging_SIG();
phStatus_t Demo_OriginalityCheck();
phStatus_t Demo_GetVersion();
phStatus_t Demo_ProximityCheck();
phStatus_t Demo_EVCharging();

#ifdef NXPBUILD__PH_LOG
void MfDuoX_Log(void* pDataParams, uint8_t bOption, phLog_LogEntry_t* pEntries, uint16_t wEntryCount);
#endif /* NXPBUILD__PH_LOG */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* EXAMPLE_MIFAREDUOX_H */
