/*
 * Copyright 2013, 2015, 2017, 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * FPGA-Box Reader specific HAL-Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phbalReg.h>
#include <phhalHw.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PH_KEYSTORE
#include <phKeyStore.h>
#endif /* NXPBUILD__PH_KEYSTORE */

#ifdef NXPBUILD__PHHAL_HW_RDFPGAV6

#include "phhalHw_RdFpgaV6.h"
#include "phhalHw_RdFpgaV6_Int.h"

#pragma warning(push)                   /* PRQA S 3116 */
#pragma warning(disable:4001)           /* PRQA S 3116 */
#include "external/phDefines.h"
#pragma warning(pop)                    /* PRQA S 3116 */

/* Default shadow for ISO14443-3A Mode */
static const uint16_t PH_MEMLOC_CONST_ROM wRdFpgaV6_DefaultShadow_14443a[][2] =
{
    {PHHAL_HW_CONFIG_PARITY,                PH_ON},
    {PHHAL_HW_CONFIG_TXCRC,                 PH_OFF},
    {PHHAL_HW_CONFIG_RXCRC,                 PH_OFF},
    {PHHAL_HW_CONFIG_RXDEAFBITS,            PH_OFF},
    {PHHAL_HW_CONFIG_TXDATARATE,            PHHAL_HW_RF_DATARATE_106},
    {PHHAL_HW_CONFIG_RXDATARATE,            PHHAL_HW_RF_DATARATE_106}
};

/* Default shadow for ISO14443-3B Mode */
static const uint16_t PH_MEMLOC_CONST_ROM wRdFpgaV6_DefaultShadow_14443b[][2] =
{
    {PHHAL_HW_CONFIG_PARITY,                PH_OFF},
    {PHHAL_HW_CONFIG_TXCRC,                 PH_ON},
    {PHHAL_HW_CONFIG_RXCRC,                 PH_ON},
    {PHHAL_HW_CONFIG_RXDEAFBITS,            PH_OFF},
    {PHHAL_HW_CONFIG_TXDATARATE,            PHHAL_HW_RF_DATARATE_106},
    {PHHAL_HW_CONFIG_RXDATARATE,            PHHAL_HW_RF_DATARATE_106},
};

/* Default shadow for ISO14443-3B Mode */
static const uint16_t PH_MEMLOC_CONST_ROM wRdFpgaV6_DefaultShadow_14443f[][2] =
{
    {PHHAL_HW_CONFIG_PARITY,                PH_OFF},
    {PHHAL_HW_CONFIG_TXCRC,                 PH_ON},
    {PHHAL_HW_CONFIG_RXCRC,                 PH_ON},
    {PHHAL_HW_CONFIG_RXDEAFBITS,            PH_OFF},
    {PHHAL_HW_CONFIG_TXDATARATE,            PHHAL_HW_RF_DATARATE_212},
    {PHHAL_HW_CONFIG_RXDATARATE,            PHHAL_HW_RF_DATARATE_212},
};

phStatus_t phhalHw_RdFpgaV6_Init(
                                 phhalHw_RdFpgaV6_DataParams_t * pDataParams,
                                 uint16_t wSizeOfDataParams,
                                 phbalReg_RdFpgaV6_DataParams_t * pBalDataParams,
                                 uint8_t * pTxBuffer,
                                 uint16_t wTxBufSize,
                                 uint8_t * pRxBuffer,
                                 uint16_t wRxBufSize
                                 )
{
    uint32_t dwTemp;
    phStatus_t statusTmp;

    if (sizeof(phhalHw_RdFpgaV6_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_HAL);
    }
    PH_ASSERT_NULL (pDataParams);
    PH_ASSERT_NULL (pTxBuffer);
    PH_ASSERT_NULL (pRxBuffer);

    /* Init. private data */
    memset(pDataParams, 0x00, sizeof(phhalHw_RdFpgaV6_DataParams_t));  /* PRQA S 3200 */
    pDataParams->wId                    = PH_COMP_HAL | PHHAL_HW_RDFPGAV6_ID;
    pDataParams->pBalDataParams         = pBalDataParams;
    pDataParams->pTxBuffer              = pTxBuffer;
    pDataParams->wTxBufSize             = wTxBufSize;
    pDataParams->wTxBufLen              = 0;
    pDataParams->pRxBuffer              = pRxBuffer;
    pDataParams->wRxBufSize             = wRxBufSize;
    pDataParams->wRxBufLen              = 0;
    pDataParams->wRxBufStartPos         = 0;
    pDataParams->wTxBufStartPos         = 0;
    pDataParams->wJoiner				= PHHAL_HW_RDFPGAV6_CMD_JOINER_BOTH;
    pDataParams->bCardType              = PHHAL_HW_CARDTYPE_ISO14443A;
    pDataParams->wTimingMode            = PHHAL_HW_TIMING_MODE_OFF;
    pDataParams->dwTimeoutUs            = PHHAL_HW_RDFPGAV6_DEFAULT_TIMEOUT_MS * 1000;
    /* get the digital delay */
    dwTemp = (uint32_t) pDataParams->dwTimeoutUs;
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_Int_GetDigDelay(pDataParams, &dwTemp));
    pDataParams->dwTimeoutETU			= (uint32_t)((float32_t)dwTemp * 13.56 / 128.0);
    pDataParams->wAdditionalInfo        = 0;
    pDataParams->wTxRxStartBackup       = 0;
    pDataParams->dwFieldOffTimeCycles   = (uint32_t)((float32_t) PHHAL_HW_FIELD_OFF_DEFAULT * 13.56 * 1000);
    pDataParams->wFieldRecoveryTime     = PHHAL_HW_FIELD_RECOVERY_DEFAULT;
    pDataParams->bSymbolStart           = PH_OFF;
    pDataParams->bSymbolEnd             = PH_OFF;
    pDataParams->bRfResetAfterTo        = PH_OFF;
    pDataParams->bFieldState			= PH_ON;
    pDataParams->bTypeBModeActive       = PH_OFF;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_RdFpgaV6_Exchange(
                                     phhalHw_RdFpgaV6_DataParams_t * pDataParams,
                                     uint16_t wOption,
                                     uint8_t * pTxBuffer,
                                     uint16_t wTxLength,
                                     uint8_t ** ppRxBuffer,
                                     uint16_t * pRxLength
                                     )
{
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint16_t    PH_MEMLOC_REM wPosition;
    uint8_t     PH_MEMLOC_REM bRxLastBits = 0;
    uint16_t	PH_MEMLOC_REM wRxLength;
    uint8_t     PH_MEMLOC_REM bTempFieldState;

    /* FPGA only reacts on command if field is on */
    if (pDataParams->bFieldState == PH_OFF)
    {
        /* save current state */
        bTempFieldState = PH_OFF;

        /* switch on field to be able to perform command */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_FieldOn(pDataParams));
    }
    else
    {
        /* save current state */
        bTempFieldState = PH_ON;
    }

    /* Check options */
    if ((wOption & (uint16_t)~(uint16_t)(PH_EXCHANGE_BUFFERED_BIT | PH_EXCHANGE_LEAVE_BUFFER_BIT)) != 0x0000)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    /* clear internal buffer if requested */
    if (!(wOption & PH_EXCHANGE_LEAVE_BUFFER_BIT))
    {
        pDataParams->wTxBufLen = 0;
    }

    /* set the receive length */
    if (pRxLength != NULL)
    {
        *pRxLength = 0;
    }

     /* Fill the global TxBuffer */
     if (wOption & PH_EXCHANGE_BUFFERED_BIT || wOption & PH_EXCHANGE_LEAVE_BUFFER_BIT)
     {
         if (wTxLength != 0)
         {
             /* Calculate start position */
             wPosition = pDataParams->wTxBufStartPos + pDataParams->wTxBufLen;

             /* Buffer overflow check */
             if ((wPosition + wTxLength) > pDataParams->wTxBufSize)
             {
                 pDataParams->wTxBufLen = 0;
                return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
             }

             /* Copy transmit buffer*/
             memcpy(&pDataParams->pTxBuffer[wPosition], pTxBuffer, wTxLength);  /* PRQA S 3200 */
             pDataParams->wTxBufLen = pDataParams->wTxBufLen + wTxLength;
         }
    }

    /* Shall we already perform the Exchange? */
    if (wOption & PH_EXCHANGE_BUFFERED_BIT)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    }

    /* Perform the actual exchange */
    switch (pDataParams->bCardType)
    {
    case PHHAL_HW_CARDTYPE_ISO14443A:
    case PHHAL_HW_CARDTYPE_ISO14443B:
    case PHHAL_HW_CARDTYPE_FELICA:
        /* Perform actual exchange */
        if (wOption & PH_EXCHANGE_LEAVE_BUFFER_BIT)
        {
            /* perform buffered exchange */
            statusTmp = phhalHw_RdFpgaV6_Int_ConvertError(pDataParams, (pDataParams->pBalDataParams->pfRdFpga_ExchangeBits)((uint8_t *) (pDataParams->pTxBuffer + pDataParams->wTxBufStartPos),
                (uint16_t)pDataParams->wTxBufLen,
                (uint8_t)pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXLASTBITS],
                (uint8_t *)(pDataParams->pRxBuffer + pDataParams->wRxBufStartPos),
                (uint16_t)(pDataParams->wRxBufSize -pDataParams->wRxBufStartPos),
                (uint16_t *)&wRxLength,
                (uint8_t *)&bRxLastBits));
        }
        else
        {
             /* perform unbuffered exchange */
             statusTmp = phhalHw_RdFpgaV6_Int_ConvertError(pDataParams, (pDataParams->pBalDataParams->pfRdFpga_ExchangeBits) (	(uint8_t *) pTxBuffer,
                 (uint16_t)wTxLength,
                 (uint8_t)pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXLASTBITS],
                 (uint8_t *)(pDataParams->pRxBuffer + pDataParams->wRxBufStartPos),
                 (uint16_t)(pDataParams->wRxBufSize -pDataParams->wRxBufStartPos),
                (uint16_t *)&wRxLength,
                 (uint8_t *)&bRxLastBits));
        }

        /* Save RxLastBits */
        pDataParams->wAdditionalInfo = (uint16_t)bRxLastBits;
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    }

    if (ppRxBuffer != NULL)
    {
        /* assign Pointer */
        *ppRxBuffer = pDataParams->pRxBuffer;
    }
    if (pRxLength != NULL)
    {
        /* Return correct data length */
        *pRxLength = wRxLength + pDataParams->wRxBufStartPos;
    }

    /* Store RxBufLen in DataParams */
    pDataParams->wRxBufLen = wRxLength + pDataParams->wRxBufStartPos;

    /* Reset our transmit buffer contents */
    pDataParams->wTxBufLen = 0;

    /* Reset TxLastBits */
    pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXLASTBITS] = 0;

    /* Incomplete byte check */
    if (((statusTmp & PH_ERR_MASK) == PH_ERR_SUCCESS) && (bRxLastBits != 0))
    {
        statusTmp = (uint16_t)PH_ADD_COMPCODE(PH_ERR_SUCCESS_INCOMPLETE_BYTE, PH_COMP_HAL);
    }

    /* Buffer Overflow mapping */
    if ((statusTmp & PH_ERR_MASK) == PH_ERR_BUFFER_OVERFLOW)
    {
        statusTmp = (uint16_t)PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    return statusTmp;
}

phStatus_t phhalHw_RdFpgaV6_WriteRegister(
    phhalHw_RdFpgaV6_DataParams_t * pDataParams,
    uint8_t bAddress,
    uint8_t bValue
    )
{
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint8_t     PH_MEMLOC_REM bTempFieldState;
    phbalReg_RdFpgaV6_DataParams_t * pDataParamsBal = pDataParams->pBalDataParams;

    /* check if right BAL */
    if (PH_GET_COMPID(pDataParamsBal) != PHBAL_REG_RDFPGAV6_ID)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_BAL);
    }

    /* check if BAL is opened otherwise throw BAL error */
    if (pDataParamsBal->bPortOpen == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }

    /* FPGA only reacts on command if field is on */
    if (pDataParams->bFieldState == PH_OFF)
    {
        /* save current state */
        bTempFieldState = PH_OFF;

        /* switch on field to be able to perform command */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_FieldOn(pDataParams));
    }
    else
    {
        /* save current state */
        bTempFieldState = PH_ON;
    }

    /* call reader function */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_Int_ConvertError( pDataParams, (pDataParams->pBalDataParams->pfRdFpga_SetJoinerReg)((uint8_t)bAddress, (uint8_t)bValue)));

    /* restore Field state */
    if ( bTempFieldState == PH_OFF )
    {
        /* switch off field again */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_FieldOff(pDataParams));
    }

    return statusTmp;
}

phStatus_t phhalHw_RdFpgaV6_ReadRegister(
    phhalHw_RdFpgaV6_DataParams_t * pDataParams,
    uint8_t bAddress,
    uint8_t * pValue
    )
{
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint8_t     PH_MEMLOC_REM bTempFieldState;

    phbalReg_RdFpgaV6_DataParams_t * pDataParamsBal = pDataParams->pBalDataParams;

    /* check if right BAL */
    if (PH_GET_COMPID(pDataParamsBal) != PHBAL_REG_RDFPGAV6_ID)
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_BAL);
    }

    /* check if BAL is opened otherwise throw BAL error */
    if (pDataParamsBal->bPortOpen == 0)
    {
        return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_BAL);
    }

    /* FPGA only reacts on command if field is on */
    if (pDataParams->bFieldState == PH_OFF)
    {
        /* save current state */
        bTempFieldState = PH_OFF;

        /* switch on field to be able to perform command */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_FieldOn(pDataParams));
    }
    else
    {
        /* save current state */
        bTempFieldState = PH_ON;
    }

    /* call reader function */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_Int_ConvertError(pDataParams, (pDataParams->pBalDataParams->pfRdFpga_GetJoinerReg)( (uint8_t) bAddress, (uint8_t *) pValue)));

    /* restore Field state */
    if ( bTempFieldState == PH_OFF )
    {
        /* switch off field again */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_FieldOff(pDataParams));
    }

    return statusTmp;
}

phStatus_t phhalHw_RdFpgaV6_ApplyProtocolSettings(
    phhalHw_RdFpgaV6_DataParams_t * pDataParams,
    uint8_t bCardType
    )
{
    phStatus_t  PH_MEMLOC_REM statusTmp;
    phStatus_t  PH_MEMLOC_REM statusTmp2;
    uint16_t    PH_MEMLOC_COUNT wIndex;
    uint16_t *  PH_MEMLOC_REM pShadowDefault;
    uint16_t    PH_MEMLOC_REM wShadowCount;
    uint8_t     PH_MEMLOC_REM bUseDefaultShadow;
    uint16_t    PH_MEMLOC_REM wConfig;
    uint8_t     PH_MEMLOC_REM bTempFieldState;

    /* init Status Temp */
    statusTmp = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);

    /* FPGA only reacts on command if field is on */
    if (pDataParams->bFieldState == PH_OFF)
    {
        /* save current state */
        bTempFieldState = PH_OFF;

        /* switch on field to be able to perform command */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_FieldOn(pDataParams));
    }
    else
    {
        /* save current state */
        bTempFieldState = PH_ON;
    }

    /* Store new card type */
    if (bCardType != PHHAL_HW_CARDTYPE_CURRENT)
    {
        if((bCardType == PHHAL_HW_CARDTYPE_ISO14443B || bCardType == PHHAL_HW_CARDTYPE_FELICA) && !pDataParams->bTypeBModeActive)
        {
            /* activate type b mode */
            PH_CHECK_SUCCESS_FCT(statusTmp, pDataParams->pBalDataParams->pfRdFpga_ActivateTypeBMode());
            pDataParams->bTypeBModeActive = PH_ON;
        }

        if(bCardType == PHHAL_HW_CARDTYPE_ISO14443A && pDataParams->bTypeBModeActive)
        {
            /* deactivate type b mode */
            PH_CHECK_SUCCESS_FCT(statusTmp, pDataParams->pBalDataParams->pfRdFpga_ActivateTypeBMode());
            pDataParams->bTypeBModeActive = PH_OFF;
        }

        pDataParams->bCardType = bCardType;
        bUseDefaultShadow = 1;
    }
    else
    {
        bUseDefaultShadow = 0;
    }

    /* configure reader IC for current card */
    switch (pDataParams->bCardType)
    {
        /* configure hardware for ISO 14443A */
    case PHHAL_HW_CARDTYPE_ISO14443A:
        /* Use 14443a default shadow */
        pShadowDefault = (uint16_t*)wRdFpgaV6_DefaultShadow_14443a;
        wShadowCount = sizeof(wRdFpgaV6_DefaultShadow_14443a) / (sizeof(uint16_t) * 2);

        /* switch to card type a */
        PH_CHECK_SUCCESS_FCT(statusTmp, pDataParams->pBalDataParams->pfRdFpga_SetParam
            (PH_RDL_READER_CARD_TYPE, PH_RDL_READER_CARD_TYPE_ISO14443A , PH_RDL_PARAM_SETMODE_NORMAL));
        break;

        /* configure hardware for ISO 14443B */
    case PHHAL_HW_CARDTYPE_ISO14443B:
        /* Use 14443b default shadow */
        pShadowDefault = (uint16_t*)wRdFpgaV6_DefaultShadow_14443b;
        wShadowCount = sizeof(wRdFpgaV6_DefaultShadow_14443b) / (sizeof(uint16_t) * 2);

        /* switch to card type b */
        PH_CHECK_SUCCESS_FCT(statusTmp, pDataParams->pBalDataParams->pfRdFpga_SetParam
            (PH_RDL_READER_CARD_TYPE, PH_RDL_READER_CARD_TYPE_ISO14443B , PH_RDL_PARAM_SETMODE_NORMAL));
        break;

    case PHHAL_HW_CARDTYPE_FELICA:
        /* Use 14443b default shadow */
        pShadowDefault = (uint16_t*)wRdFpgaV6_DefaultShadow_14443f;
        wShadowCount = sizeof(wRdFpgaV6_DefaultShadow_14443f) / (sizeof(uint16_t) * 2);

        /* swtich to card type felica */
        PH_CHECK_SUCCESS_FCT(statusTmp, pDataParams->pBalDataParams->pfRdFpga_SetParam
            (PH_RDL_READER_CARD_TYPE, PH_RDL_READER_CARD_TYPE_FELICA , PH_RDL_PARAM_SETMODE_NORMAL));
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    }

    /* Apply shadowed registers */
    for (wIndex = 0; wIndex < wShadowCount; ++wIndex)
    {
        /* Get wConfig */
        wConfig = pShadowDefault[wIndex << 1];

        /* Default shadow: */
        if (bUseDefaultShadow)
        {
            statusTmp = phhalHw_RdFpgaV6_SetConfig(pDataParams, wConfig, pShadowDefault[(wIndex << 1) + 1]);
        }
        /* Current shadow: */
        else
        {
            statusTmp = phhalHw_RdFpgaV6_SetConfig(pDataParams, wConfig, pDataParams->wCfgShadow[wConfig]);
        }
    }

    /* Apply default timeout if neccessary */
    if (bUseDefaultShadow)
    {
        statusTmp = phhalHw_RdFpgaV6_SetConfig(pDataParams, PHHAL_HW_CONFIG_TIMEOUT_VALUE_US, PHHAL_HW_RDFPGAV6_DEFAULT_TIMEOUT_MS * 1000);
    }

    /* restore Field state */
    if (bTempFieldState == PH_OFF)
    {
        /*switch off field again*/
        PH_CHECK_SUCCESS_FCT(statusTmp2, phhalHw_RdFpgaV6_FieldOff(pDataParams));
    }

    return PH_ADD_COMPCODE(statusTmp, PH_COMP_HAL);
}

phStatus_t phhalHw_RdFpgaV6_SetConfig(
                                      phhalHw_RdFpgaV6_DataParams_t * pDataParams,
                                      uint16_t wConfig,
                                      uint16_t wValue
                                      )
{

    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint32_t PH_MEMLOC_REM dwTemp;
    uint8_t PH_MEMLOC_REM bTemp;
    uint8_t PH_MEMLOC_REM bTempFieldState;

    /* FPGA only reacts on command if field is on */
    if (pDataParams->bFieldState == PH_OFF)
    {
        /* save current state */
        bTempFieldState = PH_OFF;

        /* switch on field to be able to perform command */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_FieldOn(pDataParams));

        /* store the current field state in case we exit with error */
        pDataParams->bFieldState = PH_ON;
    }
    else
    {
        /* save current state */
        bTempFieldState = PH_ON;
    }

    switch (wConfig)
    {
    case PHHAL_HW_CONFIG_PARITY:
        /* only allow on and off */
        if (wValue != PH_OFF && wValue != PH_ON)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }

        /* select both target joiners */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_SetConfig(pDataParams, PHHAL_HW_RDFPGAV6_CONFIG_TARGET_JOINER, PHHAL_HW_RDFPGAV6_CMD_JOINER_BOTH));

        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_Int_ConvertError(pDataParams, (pDataParams->pBalDataParams->pfRdFpga_SetParam)((uint32_t)PH_RDL_READER_PARITY_ENABLED, (uint32_t)wValue, PH_RDL_PARAM_SETMODE_NORMAL)));

        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        break;

    case PHHAL_HW_CONFIG_TXCRC:
        /* only allow on and off */
        if (wValue != PH_OFF && wValue != PH_ON)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        /* read out parameter */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_Int_ConvertError(pDataParams, (pDataParams->pBalDataParams->pfRdFpga_GetParam)((uint32_t)PH_RDL_READER_CRC_MODE, &dwTemp)));

        /* deactivate Mifare CRC */
        dwTemp &= ~ (uint32_t)0x04;

        /* modify parameter */
        if (wValue == PH_ON)
        {
            dwTemp |= (uint32_t)PH_RDL_CRC_UPLINK; /* set the value */
        }
        else
        {
            dwTemp &= ~ (uint32_t)PH_RDL_CRC_UPLINK; /* clear the value */
        }

        /* write the parameter */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_Int_ConvertError(pDataParams, (pDataParams->pBalDataParams->pfRdFpga_SetParam) ((uint32_t)PH_RDL_READER_CRC_MODE, dwTemp, PH_RDL_PARAM_SETMODE_NORMAL)));

        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        break;

    case PHHAL_HW_CONFIG_RXCRC:
        /* only allow on and off */
        if (wValue != PH_OFF && wValue != PH_ON)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        /* read out parameter */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_Int_ConvertError(pDataParams, (pDataParams->pBalDataParams->pfRdFpga_GetParam)((uint32_t)PH_RDL_READER_CRC_MODE, &dwTemp)));

        /* deactivate Mifare CRC */
        dwTemp &= ~ (uint32_t)0x04;

        /* modify parameter */
        if (wValue == PH_ON)
        {
            dwTemp |= (uint32_t)PH_RDL_CRC_DOWNLINK; /* set the value */
        }
        else
        {
            dwTemp &= ~(uint32_t)PH_RDL_CRC_DOWNLINK; /* clear the value */
        }

        /* write the parameter */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_Int_ConvertError(pDataParams, (pDataParams->pBalDataParams->pfRdFpga_SetParam)((uint32_t)PH_RDL_READER_CRC_MODE, dwTemp, PH_RDL_PARAM_SETMODE_NORMAL)));

        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        break;

    case PHHAL_HW_CONFIG_TXLASTBITS:
        if (wValue > 7)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }

        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        break;

    case PHHAL_HW_CONFIG_RXALIGN:
        /* write parameter to reader */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_Int_ConvertError(pDataParams, (pDataParams->pBalDataParams->pfRdFpga_SetParam)((uint32_t)PH_RDL_READER_RXALIGN, (uint32_t)wValue, PH_RDL_PARAM_SETMODE_NORMAL)));

        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        break;

    case PHHAL_HW_CONFIG_CLEARBITSAFTERCOLL:

        /* set joinerI */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_SetConfig(pDataParams, PHHAL_HW_RDFPGAV6_CONFIG_TARGET_JOINER, PHHAL_HW_RDFPGAV6_CMD_JOINER_I));

        /* read value from joiner I */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_ReadRegister(pDataParams, 0x0E, &bTemp));

        /* select right value */
        if (wValue == PH_ON)
        {
            bTemp &= (uint8_t)~0x80; /* Clear Bits ofter collision (set to 0) */
        }
        else
        {
            bTemp |= (uint8_t)0x80; /* Leave Bits ofter collision (set to 1) */
        }

        /* write to joiner I */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_WriteRegister(pDataParams, 0x0E, bTemp));

        /* set joinerQ */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_SetConfig(pDataParams, PHHAL_HW_RDFPGAV6_CONFIG_TARGET_JOINER, PHHAL_HW_RDFPGAV6_CMD_JOINER_Q));

        /* read value from joiner Q */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_ReadRegister(pDataParams, 0x0E, &bTemp));

        /* select right value */
        if (wValue == PH_ON)
        {
            bTemp &= (uint8_t)~0x80; /* Clear Bits ofter collision (set to 0) */
        }
        else
        {
            bTemp |= (uint8_t)0x80; /* Leave Bits ofter collision (set to 1) */
        }

        /* write to joiner Q */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_WriteRegister(pDataParams, 0x0E, bTemp));

        /* we have to set RxNoErr here as well */

        /* set joinerI */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_SetConfig(pDataParams, PHHAL_HW_RDFPGAV6_CONFIG_TARGET_JOINER, PHHAL_HW_RDFPGAV6_CMD_JOINER_I));

        /* read value from joiner I */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_ReadRegister(pDataParams, 0x13, &bTemp));

        /* check if value is below 4 bit, then switch off */
        if (wValue == PH_ON)
        {
            bTemp &= (uint8_t)~0x08; 	/* switch RxNoErr off */
        }
        else
        {
            bTemp |= (uint8_t)0x08;	/* switch RxNoErr on */
        }

        /* write to joiner I */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_WriteRegister(pDataParams, 0x13, bTemp));

        /* set joinerQ */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_SetConfig(pDataParams, PHHAL_HW_RDFPGAV6_CONFIG_TARGET_JOINER, PHHAL_HW_RDFPGAV6_CMD_JOINER_Q));

        /* read value from joiner Q */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_ReadRegister(pDataParams, 0x13, &bTemp));

        /* check if value is below 4 bit, then switch off */
        if (wValue == PH_ON)
        {
            bTemp &= (uint8_t)~0x08; 	/* switch RxNoErr off */
        }
        else
        {
            bTemp |= (uint8_t)0x08;	    /* switch RxNoErr on */
        }

        /* write to joiner Q */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_WriteRegister(pDataParams, 0x13, bTemp));

        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        break;

    case PHHAL_HW_CONFIG_TXDATARATE:

        /* check data rates */
        if (wValue != PHHAL_HW_RF_DATARATE_106 && \
            wValue != PHHAL_HW_RF_DATARATE_212 && \
            wValue != PHHAL_HW_RF_DATARATE_424 && \
            wValue != PHHAL_HW_RF_DATARATE_848)
        {
            /* if not type b or one of the vhbr data rate */
            if((pDataParams->bCardType != PHHAL_HW_CARDTYPE_ISO14443B) ||
                (wValue != PHHAL_HW_RF_DATARATE_1695 &&
                wValue != PHHAL_HW_RF_DATARATE_3390 &&
                wValue != PHHAL_HW_RF_DATARATE_6780))
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }

            /* activate vhbr mode */
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_Int_ConvertError(pDataParams, (pDataParams->pBalDataParams->pfRdFpga_SetParam)((uint32_t)PH_RDL_FPGA_VHBR_ASK_MODE, (uint32_t)1, PH_RDL_PARAM_SETMODE_NORMAL)));
        }

        /* select both joiners */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_SetConfig(pDataParams, PHHAL_HW_RDFPGAV6_CONFIG_TARGET_JOINER, PHHAL_HW_RDFPGAV6_CMD_JOINER_BOTH));

        /* write parameter */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_Int_ConvertError(pDataParams, (pDataParams->pBalDataParams->pfRdFpga_SetParam)((uint32_t)PH_RDL_READER_COMM_UPLINK_SPEED, (uint32_t)wValue, PH_RDL_PARAM_SETMODE_NORMAL)));

        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        break;

    case PHHAL_HW_CONFIG_RXDATARATE:
        /* check data rates */
        if (wValue != PHHAL_HW_RF_DATARATE_106 && \
            wValue != PHHAL_HW_RF_DATARATE_212 && \
            wValue != PHHAL_HW_RF_DATARATE_424 && \
            wValue != PHHAL_HW_RF_DATARATE_848)
        {
            /* if not type b or one of the vhbr data rate */
            if((pDataParams->bCardType != PHHAL_HW_CARDTYPE_ISO14443B) ||
                (wValue != PHHAL_HW_RF_DATARATE_1695 &&
                wValue != PHHAL_HW_RF_DATARATE_3390 &&
                wValue != PHHAL_HW_RF_DATARATE_6780))
            {
                return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
            }

            /* activate vhbr mode */
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_Int_ConvertError(pDataParams, (pDataParams->pBalDataParams->pfRdFpga_SetParam)((uint32_t)PH_RDL_FPGA_VHBR_ASK_MODE, (uint32_t)1, PH_RDL_PARAM_SETMODE_NORMAL)));
        }

        /* write parameter */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_Int_ConvertError(pDataParams, (pDataParams->pBalDataParams->pfRdFpga_SetParam)((uint32_t)PH_RDL_READER_COMM_DOWNLINK_SPEED, (uint32_t)wValue, PH_RDL_PARAM_SETMODE_NORMAL)));

        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = wValue;
        break;

    case PHHAL_HW_CONFIG_TIMEOUT_VALUE_US:

        /* get the digital delay */
        dwTemp = (uint32_t)wValue;
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_Int_GetDigDelay(pDataParams, &dwTemp));

        /* calculate ETU (128/fc) */
        dwTemp = (uint32_t)((float32_t)dwTemp * 13.56 / 128);

        /* write timeout to reader */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_Int_ConvertError(pDataParams, (pDataParams->pBalDataParams->pfRdFpga_SetParam)((uint32_t)PH_RDL_READER_TIMEOUT_ETU, dwTemp, PH_RDL_PARAM_SETMODE_NORMAL)));

        pDataParams->dwTimeoutUs = (uint32_t)wValue;
        pDataParams->dwTimeoutETU = dwTemp;
        break;

    case PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS:

        /* get the digital delay */
        dwTemp = (uint32_t)wValue * 1000;
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_Int_GetDigDelay(pDataParams, &dwTemp));

        /* calculate ETU (128/fc) */
        dwTemp = (uint32_t)((float32_t)dwTemp * 13.56 / 128);
        /* write timeout to reader */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_Int_ConvertError(pDataParams, (pDataParams->pBalDataParams->pfRdFpga_SetParam)((uint32_t)PH_RDL_READER_TIMEOUT_ETU, dwTemp , PH_RDL_PARAM_SETMODE_NORMAL)));

        pDataParams->dwTimeoutUs = (uint32_t)(wValue * 1000);
        pDataParams->dwTimeoutETU = dwTemp;
        break;

    case PHHAL_HW_CONFIG_TIMING_MODE:
        /* Check supported option bits */
        switch (wValue & PHHAL_HW_TIMING_MODE_OPTION_MASK)
        {
        case PHHAL_HW_TIMING_MODE_OPTION_DEFAULT:
        case PHHAL_HW_TIMING_MODE_OPTION_AUTOCLEAR:
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }

        /* Check supported timing modes */
        switch (wValue & (uint16_t)~(uint16_t)PHHAL_HW_TIMING_MODE_OPTION_MASK)
        {
        case PHHAL_HW_TIMING_MODE_OFF:
            break;

        /* FDT not supported in this Layer use higher layer in testbench */
        default:
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }

        /* Store config data */
        pDataParams->wTimingMode = wValue;
        break;

    case PHHAL_HW_CONFIG_FIELD_OFF_TIME:
        if (wValue == 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }

        /* calculate field off time in FPGA-ETUs (1/fc) */
        pDataParams->dwFieldOffTimeCycles = (uint32_t) ((float32_t)wValue * 13.56 * 1000);
        break;

    case PHHAL_HW_RDFPGAV6_CONFIG_FIELD_OFF_TIME_US:
        if (wValue == 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }

        /* calculate field off time in FPGA-ETUs (1/fc) */
        pDataParams->dwFieldOffTimeCycles = (uint32_t) ((float32_t)wValue * 13.56 + 0.5);
        break;

    case PHHAL_HW_RDFPGAV6_CONFIG_FIELD_OFF_TIME_CYCLE:
        if (wValue == 0)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }

        pDataParams->dwFieldOffTimeCycles = wValue;
        break;

    case PHHAL_HW_CONFIG_FIELD_RECOVERY_TIME:

        /* Store config data */
        pDataParams->wFieldRecoveryTime = wValue;
        break;

    case PHHAL_HW_CONFIG_SYMBOL_START:

        /* set joinerI */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_SetConfig(pDataParams, PHHAL_HW_RDFPGAV6_CONFIG_TARGET_JOINER, PHHAL_HW_RDFPGAV6_CMD_JOINER_I));

        /* read value from joiner I */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_ReadRegister(pDataParams, 0x1E, &bTemp));

        /* check if data rate is 106 kBit otherwise SOF and and EOF required */
        if ( pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXDATARATE] == PHHAL_HW_RF_DATARATE_106 &&
            pDataParams->wCfgShadow[PHHAL_HW_CONFIG_RXDATARATE] == PHHAL_HW_RF_DATARATE_106 &&
            wValue == PH_OFF)
        {
            bTemp &= (uint8_t)~0x80; /* SOF not required */
        }
        else
        {
            bTemp |= (uint8_t)0x80; /* SOF required */
        }

        /* write to joiner I */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_WriteRegister(pDataParams, 0x1E, bTemp));

        /* set joinerQ */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_SetConfig(pDataParams, PHHAL_HW_RDFPGAV6_CONFIG_TARGET_JOINER, PHHAL_HW_RDFPGAV6_CMD_JOINER_Q));

        /* read value from joiner Q */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_ReadRegister(pDataParams, 0x1E, &bTemp));

        /* check if data rate is 106 kBit otherwise SOF and and EOF required */
        if ( pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXDATARATE] == PHHAL_HW_RF_DATARATE_106 &&
            pDataParams->wCfgShadow[PHHAL_HW_CONFIG_RXDATARATE] == PHHAL_HW_RF_DATARATE_106 &&
            wValue == PH_OFF)
        {
            bTemp &= (uint8_t)~0x80; /* SOF not required */
        }
        else
        {
            bTemp |= (uint8_t)0x80; /* SOF required */
        }

        /* write to joiner Q */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_WriteRegister(pDataParams, 0x1E, bTemp));

        /* Store config data */
        pDataParams->bSymbolStart = (uint8_t)wValue;
        break;

    case PHHAL_HW_CONFIG_SYMBOL_END:

        /* set joinerI */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_SetConfig(pDataParams, PHHAL_HW_RDFPGAV6_CONFIG_TARGET_JOINER, PHHAL_HW_RDFPGAV6_CMD_JOINER_I));

        /* read value from joiner I */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_ReadRegister(pDataParams, 0x1E, &bTemp));

        /* check if data rate is 106 kBit otherwise SOF and and EOF required */
        if ( pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXDATARATE] == PHHAL_HW_RF_DATARATE_106 &&
            pDataParams->wCfgShadow[PHHAL_HW_CONFIG_RXDATARATE] == PHHAL_HW_RF_DATARATE_106 &&
            wValue == PH_OFF)
        {
            bTemp &= (uint8_t)~0x40; /* EOF not required */
        }
        else
        {
            bTemp |= (uint8_t)0x40; /* EOF required */
        }

        /* write to joiner I */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_WriteRegister(pDataParams, 0x1E, bTemp));

        /* set joinerQ */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_SetConfig(pDataParams, PHHAL_HW_RDFPGAV6_CONFIG_TARGET_JOINER, PHHAL_HW_RDFPGAV6_CMD_JOINER_Q));

        /* read value from joiner Q */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_ReadRegister(pDataParams, 0x1E, &bTemp));

        /* check if data rate is 106 kBit otherwise SOF and and EOF required */
        if ( pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TXDATARATE] == PHHAL_HW_RF_DATARATE_106 &&
            pDataParams->wCfgShadow[PHHAL_HW_CONFIG_RXDATARATE] == PHHAL_HW_RF_DATARATE_106 &&
            wValue == PH_OFF)
        {
            bTemp &= (uint8_t)~0x40; /* EOF not required */
        }
        else
        {
            bTemp |= (uint8_t)0x40; /* EOF required */
        }

        /* write to joiner Q */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_WriteRegister(pDataParams, 0x1E, bTemp));

        /* Store config data */
        pDataParams->bSymbolEnd = (uint8_t)wValue;
        break;

    case PHHAL_HW_CONFIG_DISABLE_MF_CRYPTO1:
        /* do nothing implement for compatibility reasons */
        break;

    case PHHAL_HW_CONFIG_RXBUFFER_STARTPOS:

        /* Boundary check */
        if (wValue >= pDataParams->wRxBufSize)
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }

        /* Set start position */
        pDataParams->wRxBufStartPos = wValue;
        pDataParams->wRxBufLen = wValue;

        /* Preserve RxBuffer contents if needed */
        if (pDataParams->pTxBuffer == pDataParams->pRxBuffer)
        {
            pDataParams->wTxBufStartPos = pDataParams->wRxBufStartPos;
        }
        else
        {
            pDataParams->wTxBufStartPos = 0;
        }
        break;

    case PHHAL_HW_CONFIG_TXBUFFER_LENGTH:

        /* Check parameter */
        if ((pDataParams->wTxBufStartPos + wValue) > pDataParams->wTxBufSize)
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }

        /* set buffer length */
        pDataParams->wTxBufLen = wValue;
        break;

    case PHHAL_HW_CONFIG_TXBUFFER:

        /* Check additional info parameter */
        if ((pDataParams->wTxBufStartPos + pDataParams->wAdditionalInfo) >= pDataParams->wTxBufSize)
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }

        /* Modify TxBuffer byte */
        pDataParams->pTxBuffer[pDataParams->wTxBufStartPos  + pDataParams->wAdditionalInfo] = (uint8_t)wValue;
        break;

    case PHHAL_HW_CONFIG_ADDITIONAL_INFO:

        /* Modify additional info parameter */
        pDataParams->wAdditionalInfo = wValue;
        break;

    case PHHAL_HW_CONFIG_RXDEAFBITS:

        /* we do not support DEAF, write zero into shadow */
        /* Write config data into shadow */
        pDataParams->wCfgShadow[wConfig] = 0;
        break;

    case PHHAL_HW_CONFIG_RFRESET_ON_TIMEOUT:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);

    case PHHAL_HW_RDFPGAV6_CONFIG_TARGET_JOINER:

        if ( wValue != PHHAL_HW_RDFPGAV6_CMD_JOINER_I &&
            wValue != PHHAL_HW_RDFPGAV6_CMD_JOINER_Q &&
            wValue != PHHAL_HW_RDFPGAV6_CMD_JOINER_BOTH )
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }

        /* set target joiner */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_Int_ConvertError(pDataParams, (pDataParams->pBalDataParams->pfRdFpga_SetTargetJoiner)(wValue)));

        /* assign value */
        pDataParams->wJoiner = wValue;
        break;

    case PHHAL_HW_RDFPGAV6_CONFIG_LOGGING:
        /* only allow on and off */
        if (wValue != PH_OFF && wValue != PH_ON)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }
        /* switch logging on/off */
        if (wValue == PH_ON)
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_Int_ConvertError(pDataParams, (pDataParams->pBalDataParams->pfRdFpga_SetParam)((uint32_t)PH_RDL_FPGA_DEBUG_OUTPUT, 0x7FFF , PH_RDL_PARAM_SETMODE_NORMAL)));
        }
        else
        {
            PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_Int_ConvertError(pDataParams, (pDataParams->pBalDataParams->pfRdFpga_SetParam)((uint32_t)PH_RDL_FPGA_DEBUG_OUTPUT, 0x0 , PH_RDL_PARAM_SETMODE_NORMAL)));
        }
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    }

    /* restore Field state */
    if (bTempFieldState == PH_OFF)
    {
        /* switch off field again */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_FieldOff(pDataParams));
        /* restore field state */
        pDataParams->bFieldState = PH_OFF;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_RdFpgaV6_GetConfig(
                                      phhalHw_RdFpgaV6_DataParams_t * pDataParams,
                                      uint16_t wConfig,
                                      uint16_t * pValue
                                      )
{
    switch (wConfig)
    {
    case PHHAL_HW_CONFIG_PARITY:
        /* Read config from shadow */
        *pValue = pDataParams->wCfgShadow[wConfig];
        break;

    case PHHAL_HW_CONFIG_TXCRC:
        /* Read config from shadow */
        *pValue = pDataParams->wCfgShadow[wConfig];
        break;

    case PHHAL_HW_CONFIG_RXCRC:
        /* Read config from shadow */
        *pValue = pDataParams->wCfgShadow[wConfig];
        break;

    case PHHAL_HW_CONFIG_TXLASTBITS:
        /* Read config from shadow */
        *pValue = pDataParams->wCfgShadow[wConfig];
        break;

    case PHHAL_HW_CONFIG_ADDITIONAL_INFO:
    case PHHAL_HW_CONFIG_RXLASTBITS:
        *pValue = pDataParams->wAdditionalInfo;
        break;

    case PHHAL_HW_CONFIG_RXDEAFBITS:
        /* Read config from shadow */
        *pValue = pDataParams->wCfgShadow[wConfig];
        break;

    case PHHAL_HW_CONFIG_CLEARBITSAFTERCOLL:
        /* Read config from shadow */
        *pValue = pDataParams->wCfgShadow[wConfig];
        break;

    case PHHAL_HW_CONFIG_TXDATARATE:
        /* Read config from shadow */
        *pValue = pDataParams->wCfgShadow[wConfig];
        break;

    case PHHAL_HW_CONFIG_RXDATARATE:
        /* Read config from shadow */
        *pValue = pDataParams->wCfgShadow[wConfig];
        break;

    case PHHAL_HW_CONFIG_MODINDEX:
        /* Read config from shadow */
        *pValue = pDataParams->wCfgShadow[wConfig];
        break;

    case PHHAL_HW_CONFIG_ASK100:
        /* Read config from shadow */
        *pValue = pDataParams->wCfgShadow[wConfig];
        break;

    case PHHAL_HW_CONFIG_TIMEOUT_VALUE_US:
        /* Overflow check */
        if (pDataParams->dwTimeoutUs > 0xFFFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_HAL);
        }
        *pValue = (uint16_t)pDataParams->dwTimeoutUs;
        break;

    case PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS:
        /* Return parameter */
        *pValue = (uint16_t)(pDataParams->dwTimeoutUs / 1000);
        break;

    case PHHAL_HW_CONFIG_TIMING_MODE:
        /* Return parameter */
        *pValue = pDataParams->wTimingMode;
        break;

    case PHHAL_HW_CONFIG_FIELD_OFF_TIME:
        /* Overflow check */
        if ((pDataParams->dwFieldOffTimeCycles / 13.56 / 1000) > 0xFFFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_HAL);
        }
        /* Return parameter */
        *pValue = (uint16_t)(pDataParams->dwFieldOffTimeCycles / 13.56 / 1000);
        break;

    case PHHAL_HW_RDFPGAV6_CONFIG_FIELD_OFF_TIME_US:
        /* Overflow check */
        if ((pDataParams->dwFieldOffTimeCycles / 13.56) > 0xFFFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_HAL);
        }
        /* Return parameter */
        *pValue = (uint16_t)(pDataParams->dwFieldOffTimeCycles / 13.56);
        break;

    case PHHAL_HW_RDFPGAV6_CONFIG_FIELD_OFF_TIME_CYCLE:
        /* Overflow check */
        if (pDataParams->dwFieldOffTimeCycles > 0xFFFF)
        {
            return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_HAL);
        }
        /* Return parameter */
        *pValue = (uint16_t)pDataParams->dwFieldOffTimeCycles;
        break;

    case PHHAL_HW_CONFIG_FIELD_RECOVERY_TIME:
        /* Return parameter */
        *pValue = pDataParams->wFieldRecoveryTime;
        break;

    case PHHAL_HW_CONFIG_SYMBOL_START:
        /* Return parameter */
        *pValue = pDataParams->bSymbolStart;
        break;

    case PHHAL_HW_CONFIG_SYMBOL_END:
        /* Return parameter */
        *pValue = pDataParams->bSymbolEnd;
        break;

    case PHHAL_HW_CONFIG_DISABLE_MF_CRYPTO1:
        /* return 1 as crypto is not supported */
        *pValue = PH_ON;
        break;

    case PHHAL_HW_CONFIG_SUBCARRIER:
        /* Return parameter */
        *pValue = PHHAL_HW_SUBCARRIER_SINGLE;
        break;

    case PHHAL_HW_CONFIG_RXBUFFER_STARTPOS:
        /* Return parameter */
        *pValue = pDataParams->wRxBufStartPos;
        break;

    case PHHAL_HW_CONFIG_RXBUFFER_BUFSIZE:
        /* Return parameter */
        *pValue = pDataParams->wRxBufSize;
        break;

    case PHHAL_HW_CONFIG_TXBUFFER_BUFSIZE:
        /* Return parameter */
        *pValue = pDataParams->wTxBufSize - pDataParams->wTxBufStartPos;
        break;

    case PHHAL_HW_CONFIG_TXBUFFER_LENGTH:
        /* Return parameter */
        *pValue = pDataParams->wTxBufLen;
        break;

    case PHHAL_HW_CONFIG_TXBUFFER:
        /* Check additional info parameter */
        if ((pDataParams->wTxBufStartPos + pDataParams->wAdditionalInfo) >= pDataParams->wTxBufSize)
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }

        /* Retrieve TxBuffer byte */
        *pValue = (uint16_t)pDataParams->pTxBuffer[pDataParams->wTxBufStartPos + pDataParams->wAdditionalInfo];
        break;

    case PHHAL_HW_CONFIG_RFRESET_ON_TIMEOUT:
        *pValue = (uint16_t)pDataParams->bRfResetAfterTo;
        break;

    case PHHAL_HW_CONFIG_CARD_TYPE:
        /* Return parameter */
        *pValue = (uint16_t)pDataParams->bCardType;
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_RdFpgaV6_FieldOn(
                                    phhalHw_RdFpgaV6_DataParams_t * pDataParams
                                    )
{
    phStatus_t  PH_MEMLOC_REM statusTmp = PH_ERR_SUCCESS;

    if (pDataParams->bFieldState == PH_OFF)
    {
        /* call reader function */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_Int_ConvertError(pDataParams, (pDataParams->pBalDataParams->pfRdFpga_ActivateField)()));

        /* wait field recovery time */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_Wait(pDataParams, PHHAL_HW_TIME_MILLISECONDS, pDataParams->wFieldRecoveryTime));

        /* set status right */
        pDataParams->bFieldState = PH_ON;
    }

    return statusTmp;
}

phStatus_t phhalHw_RdFpgaV6_FieldOff(
                                     phhalHw_RdFpgaV6_DataParams_t * pDataParams
                                     )
{
    phStatus_t  PH_MEMLOC_REM statusTmp = PH_ERR_SUCCESS;

    if (pDataParams->bFieldState == PH_ON)
    {
        /* call reader function */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_Int_ConvertError(pDataParams, (pDataParams->pBalDataParams->pfRdFpga_DeactivateField)()));

        /* set status right */
        pDataParams->bFieldState = PH_OFF;
    }

    return statusTmp;
}

phStatus_t phhalHw_RdFpgaV6_FieldReset(
                                       phhalHw_RdFpgaV6_DataParams_t * pDataParams
                                       )
{
    phStatus_t  PH_MEMLOC_REM statusTmp;
    uint16_t PH_MEMLOC_REM wTxValue;
    uint16_t PH_MEMLOC_REM wRxValue;

    /* Get current Tx Datarate to reset later */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_GetConfig(pDataParams, PHHAL_HW_CONFIG_TXDATARATE, &wTxValue));

    /* Get current Rx Datarate to reset later */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_GetConfig(pDataParams, PHHAL_HW_CONFIG_RXDATARATE, &wRxValue));

    if (pDataParams->bFieldState == PH_ON)
    {
        /* call reader function */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_Int_ConvertError(pDataParams, (pDataParams->pBalDataParams->pfRdFpga_HFPause)(pDataParams->dwFieldOffTimeCycles)));

        /* wait field recovery time */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_Wait(pDataParams, PHHAL_HW_TIME_MILLISECONDS, pDataParams->wFieldRecoveryTime));
    }
    else
    {
        /* wait field reset time */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_Wait(pDataParams, PHHAL_HW_TIME_MILLISECONDS, (uint16_t)(pDataParams->dwFieldOffTimeCycles / 13.56  / 1000)));

        /* wait field recovery time */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_Wait(pDataParams, PHHAL_HW_TIME_MILLISECONDS, pDataParams->wFieldRecoveryTime));
    }

    /* Reset to Tx Datarate as it was before */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_SetConfig(pDataParams, PHHAL_HW_CONFIG_TXDATARATE, wTxValue));

    /* Reset to Rx Datarate as it was before */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_RdFpgaV6_SetConfig(pDataParams, PHHAL_HW_CONFIG_RXDATARATE, wRxValue));

    return statusTmp;
}

phStatus_t phhalHw_RdFpgaV6_Wait(
                                 phhalHw_RdFpgaV6_DataParams_t * pDataParams,
                                 uint8_t bUnit,
                                 uint16_t wTimeout
                                 )
{
    /* satisfy compiler */
    if (pDataParams);

    /* Parameter check */
    if ((bUnit != PHHAL_HW_TIME_MICROSECONDS) && (bUnit != PHHAL_HW_TIME_MILLISECONDS))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }
    if (bUnit == PHHAL_HW_TIME_MICROSECONDS)
    {
        /* calculate the timeout in ms */
        wTimeout /=1000;
        /* increase by one, to ensure timeout, even if result improper rounded */
        wTimeout++;
    }

    /* wait the time */
    return phhalHw_RdFpgaV6_Int_Wait(wTimeout);
}

phStatus_t phhalHw_RdFpgaV6_MfcAuthenticateKeyNo(
    phhalHw_RdFpgaV6_DataParams_t * pDataParams,
    uint8_t bBlockNo,
    uint8_t bKeyType,
    uint16_t wKeyNo,
    uint16_t wKeyVersion,
    uint8_t * pUid
    )
{
    /* satisfy compiler */
    bBlockNo = 0;
    bKeyType = 0;
    wKeyNo = 0;
    wKeyVersion = 0;
    *pUid = 0;
    if (pDataParams);

    /* return unsupported command */
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_HAL);
}

phStatus_t phhalHw_RdFpgaV6_MfcAuthenticate(
    phhalHw_RdFpgaV6_DataParams_t * pDataParams,
    uint8_t bBlockNo,
    uint8_t bKeyType,
    uint8_t * pKey,
    uint8_t * pUid
    )
{
    /* satisfy compiler */
    bBlockNo = 0;
    bKeyType = 0;
    *pKey = 0;
    *pUid = 0;
    if (pDataParams);

    /* return unsupported command */
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_HAL);
}

#endif /* NXPBUILD__PHHAL_HW_RDFPGAV6 */
